/*
 * wrapper.hxx
 *
 * The header file for the OpenH323 Wrapper Library
 *
 * Copyright (c) 2002-2005 InAccess Networks
 * Michalis Manousos <manousos@inaccessnetworks.com>
 * Dimitris Economou <decon@inaccessnetworks.com>
 *
 * This file is part of "H.323 support for ASTERISK"
 *
 * "H.323 support for ASTERISK" is free software;
 * you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation;
 * either version 2 of the License, or (at your option) any later version. 
 *
 * "H.323 support for ASTERISK" is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details. 
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 * $Id: wrapper.hxx,v 1.49 2005/09/09 14:35:27 manousos Exp $
 *
 */

/************************************************************************/

#ifndef	_WRAPPER_HXX
#define	_WRAPPER_HXX

/************************************************************************/
/* TYPES & VARIABLES ****************************************************/

/* Different types of exceptions generated by the OpenH323 wrapper Library */
#define	OH323EXC_NULL				0
#define	OH323EXC_USER_INPUT_TONE	1
#define	OH323EXC_USER_MESSAGE		2
#define	OH323EXC_CALL_ALERTED		3
#define	OH323EXC_CALL_TRANSFER		4
#define	OH323EXC_CALL_ESTABLISHED	5
#define	OH323EXC_CALL_CLEARED		6
#define	OH323EXC_CALL_PROGRESS		7
#define	OH323EXC_CTRL_ERROR			8

/* Gatekeeper check results */
#define	OH323GK_FAILED				(-1)
#define	OH323GK_NOTUSED				(-2)
#define	OH323GK_NOTREGISTERED		(-3)

/* Call end reason codes */
#define	OH323END_LOCAL_USER			1
#define	OH323END_NO_ACCEPT			2
#define	OH323END_ANSWER_DENIED		3
#define	OH323END_REMOTE_USER		4
#define	OH323END_REFUSAL			5
#define	OH323END_NO_ANSWER			6
#define	OH323END_CALLER_ABORT		7
#define	OH323END_TRANSPORT_FAIL		8
#define	OH323END_CONNECT_FAIL		9
#define	OH323END_GATEKEEPER			10
#define	OH323END_NO_USER			11
#define	OH323END_NO_BANDWIDTH		12
#define	OH323END_CAPABILITY			13
#define	OH323END_CALLFWD			14
#define	OH323END_SECURITY			15
#define	OH323END_LOCAL_BUSY			16
#define	OH323END_LOCAL_CONGESTION	17
#define	OH323END_REMOTE_BUSY		18
#define	OH323END_REMOTE_CONGESTION	19
#define	OH323END_UNREACHABLE		20
#define	OH323END_NO_ENDPOINT		21
#define	OH323END_HOST_OFFLINE		22
#define	OH323END_TEMP_FAILURE		23
#define	OH323END_Q931CAUSE			24
#define	OH323END_DURATION_LIMIT		25
#define	OH323END_INVALIDCID			26

/* Requests for outgoing calls can define up to this number of
   capabilities (codecs) to be negotiated */
#define	WRAP_MAX_CAP_SET			(20)

/* The boolean type */
typedef enum boolean {
	YES,
	NO
} boolean_t;

/* The codec capability enumerator */
typedef enum cap_type {
	G711U = 1,	/* G.711 u-Law */
	G711A,		/* G.711 A-Law */
	G7231,		/* G.723.1(6.3k) */
	G72316K3,	/* G.723.1(6.3k) */
	G72315K3,	/* G.723.1(5.3k) */
	G7231A6K3,	/* G.723.1A(6.3k) */
	G7231A5K3,	/* G.723.1A(5.3k) */
	G726,		/* G.726(32k) */
	G72616K,	/* G.726(16k) */
	G72624K,	/* G.726(24k) */
	G72632K,	/* G.726(32k) */
	G72640K,	/* G.726(40k) */
	G728,		/* G.728 */
	G729,		/* G.729 */
	G729A,		/* G.729A */
	G729B,		/* G.729B */
	G729AB,		/* G.729AB */
	GSM0610,	/* GSM 0610 */
	MSGSM,		/* Microsoft GSM Audio Capability */
	LPC10,		/* LPC-10 */
	LINEAR16,	/* 16 bit linear PCM */
	SPEEXN8K,	/* Speex Narrow (8k) */
	CAP_UNDEFINED
} cap_type_t;

typedef enum cap_ret_val {
	CAP_EP_ER,
	CAP_NSUP_ER,
	CAP_EXIST,
	CAP_INSERT_OK,
	CAP_REMOVE_OK,
	CAP_REMOVEALL_OK
} cap_ret_val_t;

/* The user input mode enumarator */
typedef enum ui_mode {
	UIMODE_Q931,		/* Q.931 Keypad Information Element */
	UIMODE_STRING,		/* H.245 string */
	UIMODE_TONE,		/* H.245 tone */
	UIMODE_RFC2833,		/* RFC2833 */
	UIMODE_INBAND,		/* Inband */
	UIMODE_UNDEFINED
} uimode_t;

typedef enum gk_mode {
	GKMODE_DISABLE = 0,
	GKMODE_DISCOVER,
	GKMODE_NAME,
	GKMODE_ID,
	GKMODE_NAME_ID
} gkmode_t;

/* The listener enumarator 
   This enumerator could hold all the
   possible types of interfaces (i.e. ATM) */
typedef enum lis_type {
	TCP,
	UDP,
	LIS_UNDEFINED
} lis_type_t;

/* These are the return values for
   the h323_start_listener and
   the h323_remove_listener functions. */
typedef enum lis_ret_val {
	LIS_EP_ER,			/* End Point error */
	LIS_NSUP_ER,		/* Listener not supported */
	LIS_EXIST,			/* Listener already exist */
	LIS_FAILOP_ER,		/* Fail to establish listener */
	LIS_START_OK,		/* Listener is started succesfully */
	LIS_REMOVE_OK,		/* Listener is removed succesfully */
	LIS_REMOVEALL_OK	/* All listeners were removed succesfully */
} lis_ret_val_t;

/* This enum shows the direction of a logical channel. */
typedef enum lchan_dir {
	NONE = 0,
	PLAYER,
	RECORDER,
	BOTH
} lchan_dir_t;

/* The return values of the h323_{make/clear/answer/indicate}_call functions */
typedef enum call_ret_val {
	CALL_EP_ER,
	CALL_START_OK,
	CALL_START_ER,
	CALL_END_OK,
	CALL_END_ER,
	CALL_ANS_OK,
	CALL_ANS_ER,
	CALL_IND_OK,
	CALL_IND_ER,
	CALL_CHG_OK,
	CALL_CHG_ER
} call_ret_val_t;

/* This enum contains the indications that can be send over a H.323 connection. */
typedef enum call_indication {
	IND_RINGING = 1,
	IND_BUSY,
	IND_CONGESTION,
	IND_PROGRESS
} indication_t;

/* The 'call_options' struct is filled by the 
   application and passed to the library to make the call */
typedef struct call_options {
	char			*cid;
	char			*cidname;
	char			*dnis;
	int				cap[WRAP_MAX_CAP_SET];
	unsigned int	connectPort;
} call_options_t;

/* The 'call_details' struct is filled by the library and passed to
   the application through the various callback functions */
typedef struct call_details {
	unsigned int app_id;/* Application assigned ID */
	char call_id[128];	/* Globaly unique call ID */
	char conf_id[128];	/* Globally unique conference ID */
	char call_token[256];
	unsigned int call_reference;
	char call_source_alias[256];
	char call_dest_alias[256];
	char call_source_e164[256];
	char call_dest_e164[256];
	char call_dnid[256];
	char call_rdnis[256];
	char remote_app[256];
	int call_end_reason;
	int call_q931_cause;
	int call_duration;
	char remote_addr[256];
	char local_addr[256];
} call_details_t;

/* The 'user_details' struct is partially filled by the library 
 * (username/ipaddr/calling_num/called_num/redirect_num) for incoming calls
 * and the rest fields are provided by the application */
typedef struct user_details {
	int  incoming_call;		/* non-zero for incoming, 0 for outgoing */
	unsigned int app_id;	/* application-provided ID */
	char username[128];
	char ipaddr[128];		/* signaling address */
	int  port;				/* signaling port */
	int  faststart;			/* negative means undefined */
	int  h245tunnel;		/* negative means undefined */
	int  h245insetup;		/* negative means undefined */
	int  prefcodec;			/* negative means undefined */
	int  prefdtmf;			/* negative means undefined */
	char calling_num[128];
	char called_num[128];
	char redirect_num[128];
	char connected_num[128];
	char dnid[128];
	int  num_plan;			/* Numbering plan */
	int  ton;				/* Type of number */
	int  presentation;
	int  screening;
} user_details_t;

typedef struct rtp_stats {
	long packets_sent;
	long octets_sent;
	long packets_recv;
	long octets_recv;
	long packets_lost;
	long packets_late;
	long packets_ooo;
	long average_send_time;
	long max_send_time;
	long min_send_time;
	long average_recv_time;
	long max_recv_time;
	long min_recv_time;
	long average_jitter;
	long max_jitter;
	long current_jitter_size;
	long with_rr;
	long reported_fraction_lost;
	long reported_total_lost;
	long reported_sequence;
	long reported_jitter;
} rtp_stats_t;

/* This is a callback prototype function, called when the Openh323 
   OnStartLogicalChannel is invoked. */
/*typedef char *(*start_logchan_cb)(call_details_t, lchan_dir_t, int, int, int *, char *, int);*/
typedef char *(*start_logchan_cb)(call_details_t, lchan_dir_t, int, int, int *);

/* This is the callback prototype function which has to be called when
   the OpenH323 OnConnectionCleared callback is invoked */
typedef char *(*clear_con_cb)(call_details_t);

/* This is the callback prototype function which has to be
   called when the OnAlerting callback is invoked */
typedef int (*alert_con_cb)(call_details_t);

/* This is the callback prototype function called when the Openh323
   library has an exception pending. */
typedef int (*h323_exception_cb)(call_details_t, int, char *);

/* This is the callback prototype function called when the
   OnAnswerCall callback is invoked. */
typedef int (*init_con_cb)(call_details_t);

/* This is the callback prototype function called when connection-specific
   parameters are needed at the beginning of a call (incoming/outgoing). */
typedef int (*user_data_cb)(user_details_t *);

/* This is the callback prototype function called when the
   OnRTPStatistics callback is invoked. */
typedef int (*stats_con_cb)(call_details_t, rtp_stats_t);

/* This is the callback prototype functions called when the 
   H323Connection objects need data to send/have data ready. */
typedef int (*read_con_cb)(int id, char *buf, int len);
typedef int (*write_con_cb)(int id, char *buf, int len);

/************************************************************************/
/* Wrapper library API **************************************************/

/* XXX Needs to be documented... */
#ifdef cplusplus
extern "C" {
#endif   

	/* See "cap_type" above */
	const int h323_codec_number = 22;

	void h323_appinfo_set(char *, int, int, int);

	void h323_end_point_create(char **, int, int, int, char *);
	void h323_end_point_destroy(void);
	boolean_t end_point_exist(void);
    
	/* Set the callback functions */
	int h323_callback_register(start_logchan_cb, clear_con_cb, alert_con_cb,
									h323_exception_cb, init_con_cb, user_data_cb,
									stats_con_cb);

	/* H.323 codec and user-input capability related functions */
	cap_ret_val_t	h323_add_capability(cap_type_t, int);
	cap_ret_val_t	h323_set_capability(cap_type_t, int);
	cap_ret_val_t	h323_find_capability(cap_type_t);
	cap_ret_val_t	h323_removeall_capabilities(void);
	cap_ret_val_t	h323_set_senduimode(uimode_t);

	/* H.323 options setup related functions */
	int				h323_set_options(int, int, int, int, int, int, 
#ifdef HAS_OH323MODS
									int, int, 
#endif
									int);
	int				h323_set_ports(unsigned, unsigned, unsigned, unsigned,
									unsigned, unsigned);

	/* H.323 listener related funcions */
	lis_ret_val_t 	h323_start_listener(lis_type_t lis, char *, int);
	lis_ret_val_t 	h323_remove_listener(lis_type_t lis);
	lis_ret_val_t 	h323_removeall_listeners(void);

	/* H.323 gatekeeper related funcions */
	int				h323_get_gk(char *, int);
	int				h323_set_gk(char *, char *, char *, int, char **, int);
	int				h323_reset_gk(char *, char *);

	int				h323_get_conn_info(const char *, char *, int);
	unsigned		h323_check_bandwidth(void);

	/* Send a DTMF tone to remote endpoint */
	void h323_send_tone(const char *call_token, char tone);
	void h323_send_text(const char *call_token, char *text);

	/* H.323 create, handle and destroy sessions */
	call_ret_val_t h323_make_call(char *host, call_details_t *pcd, 
									user_details_t *ud);
	call_ret_val_t h323_clear_call(const char *call_token);
	int h323_set_hangup_cause(const char *call_token, int cause);
	call_ret_val_t h323_answer_call(const char *call_token);
	call_ret_val_t h323_change_call(const char *call_token, const char *new_mode);
	call_ret_val_t h323_indicate_call(const char *call_token, indication_t type);

	/* Get the status of a call */
	int h323_is_call_connected(const char *call_token);
	int h323_get_reason_code(int reason);
	char *h323_get_reason_desc(int reason);
	int h323_get_cause_code(int cause);
	char *h323_get_cause_desc(int cause);

#ifdef cplusplus
}
#endif

#endif	/* _WRAPPER_HXX */

/* End of file *************************************************************/
