#include <gtk/gtk.h>
#include <math.h>
#include <audacious/plugin.h>

#include "plugin_icon.xpm"

#include "../config.h"

#include "plugin_main.h"
#include "plugin_dialogs.h"
#include "plugin_skin.h"
#include "plugin_helper.h"
#include "plugin_worker.h"

/****************************************************************************
                            Local variables
*****************************************************************************/
GdkPixbuf 	*pluginIcon		= NULL;

GThread		*vumeter_thread1	= NULL;

gint		num_of_windows		= -1,
		num_of_samples		= 3,
		data_source		= 1,
		target_fps		= 25,
		decay_pct		= 10,
		plugin_initialized	= 0,
		worker_can_quit		= 1,
		worker_state		= 0,
	
		devmode_enabled		= 0;

float 		devmode_left_value	= -100,
		devmode_right_value	= -100;

gint16		shared_data[2][512];

// Windows
vumeter_window	plugin_win[MAX_INSTANCES];

// List of skins that directory scanner found (name + dirnum)
GArray		*plugin_skin_list	= NULL;

// Loaded skins (settings, image pointers, etc..)
GArray		*plugin_skin_data	= NULL;

extern float	rms_values[3],
		peak_values[3];

/****************************************************************************
                            Function definitions
*****************************************************************************/
void      win_click_event (GtkWidget *, GdkEventButton *, gpointer);
void      win_focus_event (GtkWidget *, GdkEventFocus *, gpointer);
void	  win_close_event (GtkObject *, gpointer);
gboolean  expose_cb       (GtkWidget *, GdkEventExpose *, gpointer);

void      vumeter_render(gint16 data[2][512]);
void      vumeter_play(void);
void      vumeter_pause(void);
void      vumeter_init(void);
void      vumeter_cleanup(void);

void      vumeter_save_configuration(vumeter_window *);
void      vumeter_load_configuration(vumeter_window *);

gint vumeter_dockbug_timer( gpointer );
gint vumeter_error_timer  ( gpointer );

VisPlugin *get_vplugin_info(void);

/****************************************************************************
                            Plugin definition
*****************************************************************************/
VisPlugin vumeter_vp =
{
	.description = PACKAGE_NAME,

        .num_pcm_chs_wanted = 2,
	.num_freq_chs_wanted = 0,

	.about = vumeter_about,
	.configure = vumeter_config,

        .init = vumeter_init,
	.cleanup = vumeter_cleanup,
	.playback_start = vumeter_play,
	.playback_stop = vumeter_pause,
	.render_pcm = vumeter_render
};

VisPlugin *vumeter_vplist[] = { &vumeter_vp, NULL };
SIMPLE_VISUAL_PLUGIN(vumeter, vumeter_vplist);

/****************************************************************************
                            Render function
*****************************************************************************/
void vumeter_render(gint16 data[2][512])
{
	if(worker_state!=0) return;
	memcpy(shared_data,data,sizeof(gint16)*2*512);
	worker_state=1;
}

/****************************************************************************
                        Play / Pause functions
*****************************************************************************/
void vumeter_play(void)
{
}

void vumeter_pause(void)
{
}


/****************************************************************************
                      Function to create one plugin window
*****************************************************************************/
void vumeter_window_init(gint inum,GdkPixbuf *titleimg)
{
	vumeter_skin   	*skin;
	gint	 	snum=plugin_win[inum].skin_num-1,
			tbar_height=0;

	skin = &g_array_index(plugin_skin_data, vumeter_skin, snum);

	if(titleimg == NULL)
		titleimg = skin->img_titlebar_off;

	if(titleimg != NULL)
	{
		tbar_height = gdk_pixbuf_get_height(titleimg);
		gdk_draw_pixbuf(	plugin_win[inum].dbuf, plugin_win[inum].pen, titleimg,
					0,0,0,0,-1,-1,
					GDK_RGB_DITHER_NONE,0,0);
	}

	// Draw background image to double buffer
	if(skin->img_background != NULL)
	{
		gdk_draw_pixbuf(	plugin_win[inum].dbuf, plugin_win[inum].pen, skin->img_background,
					0,0,0,tbar_height,-1,-1,
					GDK_RGB_DITHER_NONE,0,0);
	}
}


GtkWidget *vumeter_create_window(gint inum, gint skin_num)
{
	gint		width=g_array_index(plugin_skin_data, vumeter_skin, skin_num-1).width,
			height=g_array_index(plugin_skin_data, vumeter_skin, skin_num-1).height;
	GtkWidget	*newWin=NULL;

	// Create new window, set title, and disable resize
	newWin = gtk_window_new  ( GTK_WINDOW_TOPLEVEL );
	gtk_window_set_title	 ( GTK_WINDOW(newWin), PACKAGE_STRING);
	gtk_window_set_resizable ( GTK_WINDOW(newWin), FALSE );

	// Realize widget
	gtk_widget_realize(newWin);

	// Disable decorations
	gtk_window_set_decorated( GTK_WINDOW(newWin), FALSE);

	// Add focus, move and button events to window
	gtk_widget_add_events	( GTK_WIDGET(newWin), GDK_BUTTON_PRESS_MASK | 
	                           GDK_BUTTON_RELEASE_MASK | GDK_FOCUS_CHANGE_MASK);

	// Add signal handlers
	
	gtk_signal_connect(	GTK_OBJECT(newWin), "destroy", 
				GTK_SIGNAL_FUNC(win_close_event), &plugin_win[inum] );

	gtk_signal_connect(	GTK_OBJECT(newWin), "button_press_event", 
				GTK_SIGNAL_FUNC(win_click_event), &plugin_win[inum] );

	gtk_signal_connect(	GTK_OBJECT(newWin), "button_release_event", 
				GTK_SIGNAL_FUNC(win_click_event), &plugin_win[inum] );

	gtk_signal_connect(	GTK_OBJECT(newWin), "focus_in_event", 
				GTK_SIGNAL_FUNC(win_focus_event), &plugin_win[inum] );
	gtk_signal_connect(	GTK_OBJECT(newWin), "focus_out_event", 
				GTK_SIGNAL_FUNC(win_focus_event), &plugin_win[inum] );
	
	gtk_signal_connect(	GTK_OBJECT (newWin), "expose_event", 
				GTK_SIGNAL_FUNC (expose_cb), &plugin_win[inum]);

	// Set widget size and add icon to window
	gtk_window_set_icon         ( GTK_WINDOW(newWin), pluginIcon );
	gtk_widget_set_size_request ( GTK_WIDGET(newWin) , width, height);

	// Move window to it's position
	if(plugin_win[inum].xpos>0 && plugin_win[inum].ypos>0)
	{
		gtk_window_move ( GTK_WINDOW(newWin) , plugin_win[inum].xpos, plugin_win[inum].ypos);
	}

	// Show widget, and all it's sub widgets
	gtk_widget_show_all ( GTK_WIDGET(newWin) );
	plugin_win[inum].win      = newWin;

	// Fill structure
	plugin_win[inum].skin_num = skin_num;
	plugin_win[inum].width    = width;
	plugin_win[inum].height   = height;
	plugin_win[inum].slot_num = inum;

	// Create drawing area
	plugin_win[inum].pixmap	= gdk_pixmap_new( newWin->window,width,height,-1 );
	plugin_win[inum].dbuf	= gdk_pixmap_new( newWin->window,width,height,-1 );
	plugin_win[inum].pen    = gdk_gc_new ( newWin->window );

	// Clear pixmap & dbuf
	gdk_draw_rectangle(plugin_win[inum].pixmap,plugin_win[inum].pen,TRUE,0,0,width,height);
	gdk_draw_rectangle(plugin_win[inum].dbuf,plugin_win[inum].pen,TRUE,0,0,width,height);
	vumeter_window_init(inum,NULL);

	// Update config window
	vumeter_update_window_list();

	// Return pointer to window
	DEBUG( printf("VUMETER: window created (title: %s , size: %d x %d , slot: %d)\n",PACKAGE_STRING,width,height,inum); );
	return(newWin);
}


void vumeter_change_window_skin(gint inum,gint path_num,gchar *skin_name)
{
	gint		l1,l2,
			snum=0,
			found;
	vumeter_skin   	*skin;
	GdkPixmap	*old_pmap,
			*new_pmap;

	// Are given parameters valid?
	if(inum<0 || inum>=MAX_INSTANCES) return;
	if(path_num<0 || path_num>1) return;
	if(strlen(skin_name)==0) return;
	if(plugin_win[inum].win!=NULL)

	// Try to load specified skin
	snum = vumeter_load_skin( path_num, skin_name);
	if( snum == 0 ) return;

	skin = &g_array_index(plugin_skin_data, vumeter_skin, snum-1);

	// Resize window, if needed
	if(skin->width != plugin_win[inum].width || skin->height != plugin_win[inum].height)
	{
		DEBUG( printf("VUMETER: window resized (slot: %d  %dx%d -> %dx%d)\n",inum,
				plugin_win[inum].width,plugin_win[inum].height,skin->width,skin->height); );
		gtk_widget_set_size_request ( GTK_WIDGET(plugin_win[inum].win) , skin->width, skin->height);
	}

	// Change window parameters
	plugin_win[inum].skin_num = snum;
	plugin_win[inum].width    = skin->width;
	plugin_win[inum].height   = skin->height;

	// Resize pixmap
	old_pmap = plugin_win[inum].pixmap;
	new_pmap = gdk_pixmap_new( plugin_win[inum].win->window,skin->width,skin->height,-1 );
	gdk_draw_rectangle(new_pmap,plugin_win[inum].pen,TRUE,0,0,skin->width,skin->height);
	plugin_win[inum].pixmap = new_pmap;
	g_object_unref(old_pmap);

	// Resize dbuf
	old_pmap = plugin_win[inum].dbuf;
	new_pmap = gdk_pixmap_new( plugin_win[inum].win->window,skin->width,skin->height,-1 );
	gdk_draw_rectangle(new_pmap,plugin_win[inum].pen,TRUE,0,0,skin->width,skin->height);
	plugin_win[inum].dbuf = new_pmap;
	g_object_unref(old_pmap);

	// Queue redraw
	vumeter_window_init(inum,NULL);
	gtk_widget_queue_draw(plugin_win[inum].win);
	
	// Free unused skins (if any)
	for(l1=0; l1<plugin_skin_data->len; l1++)
	{
		skin = &g_array_index(plugin_skin_data, vumeter_skin, l1);
		if(skin->pathnum==-1) continue;

		found = 0;
		for(l2=0; l2<MAX_INSTANCES; l2++)
		if( plugin_win[l2].skin_num == (l1+1) )
		{
			found = 1; break;
		}

		if(found==0)
		{
			DEBUG( printf("VUMETER: Skin (snum: %d , pnum: %d) not in use, freeing memory.\n",
					l1,skin->pathnum););
			vumeter_deinit_skin( &g_array_index(plugin_skin_data, vumeter_skin, l1) );
		}
	}
}

/****************************************************************************
                        Initialization and cleanup
*****************************************************************************/
void vumeter_init(void)
{
	int 	i;

	// Init 
	devmode_enabled = 0;
	devmode_left_value = -100.0;
	devmode_right_value = -100.0;
	worker_can_quit = 0;
	worker_state    = 0;

	for(i=0; i<MAX_INSTANCES; i++)
		reset_win_structure(&plugin_win[i]);

	// Look for available skins
	if(vumeter_scan_skin_dirs()==0)
	{
		vumeter_error_dialog("VUMETER: No skins found! Please check you installation.\n");
		gtk_timeout_add(10,vumeter_error_timer,NULL);
		return;
	}

	// Load configuration
	vumeter_load_configuration(plugin_win);

	// Create worker thread
	if( ( vumeter_thread1 = g_thread_create((GThreadFunc)vumeter_worker,NULL,TRUE,NULL) ) == NULL )
	{
		vumeter_error_dialog("VUMETER: Unable to create worker thread :...(\n");
		gtk_timeout_add(10,vumeter_error_timer,NULL);
		return;
	}

	// Create icon for pluginwindow
	if(pluginIcon==NULL)
	{
		pluginIcon = gdk_pixbuf_new_from_xpm_data((const char **)plugin_icon_xpm);
	}

	// Create requested number of windows, and load skin configuration
	for(i=0; i<num_of_windows; i++)
	{	
		// Create window
		if(plugin_win[i].win==NULL)
		{
			if(vumeter_create_window(i,plugin_win[i].skin_num)==NULL)
			{
				printf("VUMETER: Critical error while creating windows!\n");
				gtk_timeout_add(10,vumeter_error_timer,NULL);
				return;
			}
		}
	}

	plugin_initialized = 1;
}

void vumeter_cleanup(void)
{
	gint i;

	DEBUG( printf("VUMETER: Cleanup...\n"); );

	// FIXME (lock!);
	worker_can_quit = 1;

	// Save configuration a
	vumeter_save_configuration(plugin_win);

	// Wait a little bit longer for worker
	usleep(100000);

	// Wait for worker thread to quit
	if(vumeter_thread1!=NULL)
		g_thread_join(vumeter_thread1);

	// Close windows
	for(i=0; i<MAX_INSTANCES; i++)
	if(plugin_win[i].win!=NULL)
		gtk_object_destroy( GTK_OBJECT(plugin_win[i].win) );

	// Free memory
	if(plugin_skin_data != NULL)
	{
		for(i=0; i<plugin_skin_data->len; i++)
			vumeter_deinit_skin( &g_array_index(plugin_skin_data, vumeter_skin, i) );

		g_array_free(plugin_skin_data,TRUE);
	}

	if(pluginIcon!=NULL)	
		g_object_unref( G_OBJECT(pluginIcon) );

	if(plugin_skin_list != NULL)
		g_array_free(plugin_skin_list,TRUE);

	// Reset values
	vumeter_thread1		= NULL;
	pluginIcon		= NULL;
	plugin_skin_data	= NULL;
	plugin_skin_list	= NULL;
	plugin_initialized	= 0;
}

/****************************************************************************
                          Timer callbacks
*****************************************************************************/
gint vumeter_error_timer( gpointer data )
{
	DEBUG( printf("VUMETER: Entered disable function!\n") );
	vumeter_vp.disable_plugin(&vumeter_vp);
	return(0);
}

/****************************************************************************
                          Area expose events
*****************************************************************************/
gboolean expose_cb (GtkWidget *widget, GdkEventExpose *event, gpointer data)
{

	vumeter_window *ptr = (vumeter_window *)data;
	vumeter_skin   *skin;
	vumeter_module *module;
	gint		snum,l1,l2,
			t_x,t_y;
	GdkPixbuf	*img_to_use;
	float		tmp_val=0.0,
			angle=0.0,
			rad_tmp = (M_PI/180.0);

	if(data==NULL || ptr==NULL)	return(FALSE);

	// 
	snum 		= ptr->skin_num-1;
	skin 		= &g_array_index(plugin_skin_data, vumeter_skin, snum);

	// Dbuf to pixmap
	gdk_draw_drawable(	ptr->pixmap, ptr->pen, ptr->dbuf,
				0,0,0,0,ptr->width,ptr->height);

	// Process all layers
	for(l1=1; l1<=5; l1++)
	for(l2=0; l2<skin->modules->len; l2++)
	{
		module = &g_array_index(skin->modules,vumeter_module,l2);
		if(module->enabled==0) continue;
		if(module->layer!=l1) continue;
		if(module->channel<0 || module->channel>2) continue;

		// Select data source (Peak / RMS)
		if(data_source==1)	tmp_val = rms_values[module->channel];
		else 			tmp_val = peak_values[module->channel];

	   	// Draw all vumeter needles
		if(module->type == 1)
		{
			if ( tmp_val <= module->db_min )	angle=module->angle_min; 
			else if ( tmp_val >= module->db_max )	angle=module->angle_max; 
			else {
				angle = log10(fabs(tmp_val-1.0))/log10(fabs(module->db_min)+1.0);
				angle = module->angle_range*(1.0-angle)+module->angle_min;
			}

			// Calculate end point in unit circle 
			// note: cos & sin  need angles in radians
			t_x = module->position[0] + floor( (float)module->radius * cos ( ( 90.0 + angle ) * rad_tmp ) );
			t_y = module->position[1] + floor( (float)module->radius * sin ( ( 90.0 + angle ) * rad_tmp ) );

			gdk_gc_set_line_attributes( ptr->pen, module->width, GDK_LINE_SOLID, GDK_CAP_BUTT, GDK_JOIN_MITER );
			gdk_gc_set_rgb_fg_color( ptr->pen , &module->color );
			gdk_draw_line(	ptr->pixmap, ptr->pen, 
					module->position[0], module->position[1], t_x, t_y );
		}

		// Draw all image layers
		if(module->type == 2) 
		{
			if(tmp_val >= module->db_min && tmp_val<= module->db_max)	img_to_use = module->on_img;
			else								img_to_use = module->off_img;

			// 
			if(img_to_use == NULL) continue;
			gdk_draw_pixbuf(ptr->pixmap, ptr->pen, img_to_use,
					0,0, module->position[0],module->position[1], -1,-1,
					GDK_RGB_DITHER_NONE,0,0);
		}
	}

	// pixmap to window
	gdk_draw_drawable(	ptr->win->window, ptr->pen, ptr->pixmap,
				0,0,0,0, ptr->width,ptr->height);

	return(TRUE);
}

/****************************************************************************
                            Mouse events
*****************************************************************************/
void win_close_event (GtkObject *aObject, gpointer aCallbackData)
{
	vumeter_window *ptr = (vumeter_window *)aCallbackData;
	gint inum = ptr->slot_num;

	// Free memory
	g_object_unref(plugin_win[inum].pixmap);
	g_object_unref(plugin_win[inum].dbuf);
	g_object_unref(plugin_win[inum].pen);

	// Reset structure
	reset_win_structure(&plugin_win[inum]);

	// ...
	num_of_windows--;
	vumeter_update_window_list();

	DEBUG( printf("VUMETER: window closed (slot: %d)\n",inum); );
}

void win_click_event(GtkWidget *aWidget, GdkEventButton *aEvent, gpointer aCallbackData)
{

	vumeter_window *ptr = (vumeter_window *)aCallbackData;
	vumeter_skin   *skin;
	gint		snum,tbar_height; 
	
	// We only care about left mouse button
	if( aEvent->button != 1 ) return;
	// 
	snum 		= ptr->skin_num-1;
	skin 		= &g_array_index(plugin_skin_data, vumeter_skin, snum);
	tbar_height	= gdk_pixbuf_get_height(skin->img_titlebar_on);

	// Button has been pressed
	if( aEvent->type == GDK_BUTTON_PRESS )
	{
		if(aEvent->x >= skin->exit_button_pos[0][0] && aEvent->y >= skin->exit_button_pos[0][1] &&
		   aEvent->x <= skin->exit_button_pos[1][0] && aEvent->y <= skin->exit_button_pos[1][1])
		{
			DEBUG( printf("VUMETER: Click on windows (id: %d ) exit button!\n",ptr->slot_num) );
			if(num_of_windows==1)
			{
				gtk_timeout_add(10,vumeter_error_timer,NULL);
			}
			gtk_object_destroy( GTK_OBJECT(ptr->win) );
			return;
		} 

		if(aEvent->x >= skin->conf_button_pos[0][0] && aEvent->y >= skin->conf_button_pos[0][1] &&
		   aEvent->x <= skin->conf_button_pos[1][0] && aEvent->y <= skin->conf_button_pos[1][1])
		{
			vumeter_config();
			return;
		} 

		if(aEvent->y >=0 && aEvent->y < tbar_height )
		{
			gtk_window_begin_move_drag ( GTK_WINDOW(aWidget), 1, aEvent->x_root, aEvent->y_root, aEvent->time);
		}
	}
	
}

void win_focus_event(GtkWidget *aWidget, GdkEventFocus *aEvent, gpointer aCallbackData)
{

	GdkPixbuf 	*tmp1;

	vumeter_window *ptr = (vumeter_window *)aCallbackData;
	gint		snum;
	snum = ptr->skin_num-1;

	// If (true) then window got focus :)
	if(aEvent->in)	tmp1 = g_array_index(plugin_skin_data, vumeter_skin, snum).img_titlebar_on;
	else		tmp1 = g_array_index(plugin_skin_data, vumeter_skin, snum).img_titlebar_off;

	// Redraw 
	vumeter_window_init(ptr->slot_num,tmp1);
	gtk_widget_queue_draw(ptr->win);
}

/****************************************************************************
                            One line miracles
*****************************************************************************/
VisPlugin *get_vplugin_info(void) { return &vumeter_vp; }
