/*================================================================
 * midi.c
 *	midi player program for AWE32 sound driver
 *
 * Copyright (C) 1996-1999 Takashi Iwai
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *================================================================*/

#include <stdio.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <stdlib.h>
#include "channel.h"
#include "seq.h"
#include "midievent.h"
#include "util.h"
#include "sighandle.h"
#include "controls.h"
#include "options.h"
#include "config.h"

/*#define DONT_TRUST_SYSTEM_SLEEP*/

/*
 * prototypes
 */
#ifdef RCFILE
static void parse_default_options(void);
static int parse_rcfile(char *path, char *base);
#endif
static void parse_options(int argc, char **argv);

static int wait_pending(MidiInfo *mp);
static int send_seq_events(MidiInfo *mp);

void print_usage(void);
static void clean_up(int sig);

static int search_event(MidiInfo *mp, int delta);
static void jump_to(MidiInfo *mp, int csec, int output_lyric);
static void jump_to_start(MidiInfo *mp);
static void reset_status(MidiInfo *mp);
static void find_start_pos(MidiInfo *mp);

static void push_clear(void);
static int pop_event(MidiInfo *mp);
static int check_same_csec(MidiEvent *ev, MidiInfo *mp);
static int do_control(MidiInfo *mp);

/* Sanity check */
#if !(defined(INCLUDE_TK_MODE) || defined(INCLUDE_NCURSES_MODE) || defined (INCLUDE_DUMB_MODE) || defined(INCLUDE_PIPE_MODE))
#error No control interfaces defined!
#endif

#ifdef INCLUDE_TK_MODE
extern ControlMode tk_control_mode;
extern char *tk_display, *tk_geometry, *tk_effect;
#endif
#ifdef INCLUDE_NCURSES_MODE
extern ControlMode ncurses_control_mode;
#endif
#ifdef INCLUDE_DUMB_MODE
extern ControlMode dumb_control_mode;
#endif
#ifdef INCLUDE_PIPE_MODE
extern ControlMode pipe_control_mode;
extern int pipe_in, pipe_out, gen_shmid;
#endif

ControlMode *ctl_list[] = {
#ifdef INCLUDE_TK_MODE
	&tk_control_mode,
#endif
#ifdef INCLUDE_NCURSES_MODE
	&ncurses_control_mode,
#endif
#ifdef INCLUDE_DUMB_MODE
	&dumb_control_mode,
#endif
#ifdef INCLUDE_PIPE_MODE
	&pipe_control_mode,
#endif
};

ControlMode *ctl;

MidiInfo glinfo;
int verbose = 0, debug = 0;
static int playing_mode = 0;

void main(int argc, char **argv)
{

	memcpy(&glinfo, &gl_default, sizeof(glinfo));

	if (!getenv("DISPLAY"))
#ifdef INCLUDE_NCURSES_MODE
		ctl = &ncurses_control_mode;
#elif defined(INCLUDE_DUMB_MODE)
		ctl = &dumb_control_mode;
#elif defined(INCLUDE_PIPE_MODE)
		ctl = &pipe_control_mode;
#else
		{
			printf("Can't run in text console!\n"
				"Please recompile with ncurses or dumb interface support.\n\n");
			print_usage();
			exit(1);
		}
#endif
	else
		ctl = ctl_list[0];

#ifdef RCFILE
	parse_default_options();
#endif
	parse_options(argc, argv);

	if (optind >= argc && ctl->need_args) {
		print_usage();
		exit(1);
	}

	ctl->verbosity = verbose;
	ctl->playing_mode = playing_mode;

	if (argc > optind && *argv[optind] == '-')
		ctl->open(1, 0);
	else
		ctl->open(0, 0);

	add_signal(SIGTERM, clean_up, 1);
	add_signal(SIGINT, clean_up, 1);
	add_signal(SIGQUIT, clean_up, 1);

	ctl->pass_playing_list(argc - optind, argv + optind);
	ctl->close();

	exit(0);
}

/*----------------------------------------------------------------
 * parse options
 *----------------------------------------------------------------*/

#ifdef RCFILE
static void parse_default_options(void)
{
	char *p, rcfile[256];

	rcfile[0] = 0;
	if ((p = getenv("HOME")) != NULL && *p) {
		sprintf(rcfile, "%s/%s", p, RCFILE);
		if (access(rcfile, R_OK) != 0)
			rcfile[0] = 0;
	}
	if (! *rcfile) {
#ifdef SYSTEM_RCFILE
		strcpy(rcfile, SYSTEM_RCFILE);
		if (access(rcfile, R_OK) != 0)
			return;
#else
		return;
#endif
	}

	parse_rcfile(rcfile, NULL);
}

#define MAX_ARGC	100

static int parse_rcfile(char *path, char *base)
{
	FILE *fp;
	char line[256];

	if ((fp = fopen(path, "r")) == NULL)
		return 0;

	while (fgets(line, sizeof(line), fp)) {
		char *argv[MAX_ARGC];
		int argc;

		argv[0] = strtoken(line);
		if (argv[0] == NULL || !*argv[0] || *argv[0] == '#')
			continue;

		for (argc = 1; argc < MAX_ARGC; argc++) {
			argv[argc] = strtoken(NULL);
			if (!argv[argc])
				break;
		}
		parse_options(argc, argv);
	}

	fclose (fp);
	return 1;
}

#endif /* RCFILE */

static awe_option_args long_options[] = {
	{"help", 1, 0, OPT_HELP},
	{"interface", 1, 0, OPT_INTERFACE},
	{"trace", 2, 0, OPT_TRACE},
	{"mode", 1, 0, OPT_MODE},
	{"drum", 1, 0, OPT_DRUM},
	{"drumflag", 1, 0, OPT_DRUMFLAG},
	{"chorus", 1, 0, OPT_CHORUS},
	{"reverb", 1, 0, OPT_REVERB},
	{"volume", 1, 0, OPT_VOLUME},
	{"volscale", 1, 0, OPT_VOLSCALE},
	{"acceptall", 2, 0, OPT_ACCEPTALL},
	{"realpan", 2, 0, OPT_REALTIME_PAN},
	{"tracks", 1, 0, OPT_TRACKS},
	{"multipart", 1, 0, OPT_MULTIPART},
	{"mt32", 1, 0, OPT_MT32},
	{"dynamic", 1, 0, OPT_DYNAMIC},
	{"xgload", 1, 0, OPT_XGLOAD},
	{"xgmap", 2, 0, OPT_XGMAP},
	{"increment", 2, 0, OPT_INCREMENTAL},
	{"samecsec", 2, 0, OPT_SAMECSEC},
	{"gsmacro", 2, 0, OPT_GSMACRO},
	{"xgmacro", 2, 0, OPT_XGMACRO},
	{"verbose", 2, 0, OPT_VERBOSE},
	{"autoskip", 2, 0, OPT_AUTOSKIP},
	{"parsetitle", 2, 0, OPT_PARSETITLE},
	{"chorusdepth", 1, 0, OPT_CHORUSDEPTH},
	{"reverbdepth", 1, 0, OPT_REVERBDEPTH},
	{"tuning", 2, 0, OPT_TUNING},
	{"chnprior", 2, 0, OPT_CHN_PRIOR},
	{"newvolume", 2, 0, OPT_NEWVOLUME},
	{"usefx", 2, 0, OPT_USEFX},
	{"fx_cutoff", 1, 0, OPT_FX_CUTOFF},
	{"fx_resonance", 1, 0, OPT_FX_RESONANCE},
	{"fx_attack", 1, 0, OPT_FX_ATTACK},
	{"fx_release", 1, 0, OPT_FX_RELEASE},
	{"fx_vibrate", 1, 0, OPT_FX_VIBRATE},
	{"fx_vibdepth", 1, 0, OPT_FX_VIBDEPTH},
	{"fx_vibdelay", 1, 0, OPT_FX_VIBDELAY},
	{"convert", 1, 0, OPT_CONVERT},
	{"seqbuf", 1, 0, OPT_SEQBUF},
	{"seqecho", 1, 0, OPT_SEQECHO},
#ifdef INCLUDE_TK_MODE
	{"effect", 1, 0, OPT_EFFECT},
	{"display", 1, 0, OPT_DISPLAY},
	{"geometry", 1, 0, OPT_GEOMETRY},
#endif
#ifdef INCLUDE_PIPE_MODE
	{"shmid", 1, 0, OPT_SHMID},
	{"pipein", 1, 0, OPT_PIPEIN},
	{"pipeout", 1, 0, OPT_PIPEOUT},
#endif
	{0, 0, 0, 0},
};
static int option_index;

#define BASE_OPTION_FLAGS	"hvc:r:D:i:V:S:sPOT:CGXM:tm:L:YN"
#ifdef INCLUDE_TK_MODE
#define OPTION_FLAGS	BASE_OPTION_FLAGS "d:g:"
#else
#define OPTION_FLAGS	BASE_OPTION_FLAGS
#endif

static void parse_options(int argc, char **argv)
{
	int cp, i, mode;
	char *p;

	while ((cp = awe_getopt(argc, argv, OPTION_FLAGS,
				long_options, &option_index)) != -1) {
		switch (cp) {
#ifdef INCLUDE_PIPE_MODE
		case OPT_PIPEIN:
			pipe_in = atoi(optarg); break;
		case OPT_PIPEOUT:
			pipe_out = atoi(optarg); break;
		case OPT_SHMID:
			gen_shmid = atoi(optarg); break;
#endif
		case OPT_XGLOAD:
			if (glinfo.xgload) free(glinfo.xgload);
			if (strcmp(optarg, "none") == 0)
				glinfo.xgload = NULL;
			else
				glinfo.xgload = safe_strdup(optarg);
			break;
		case OPT_DYNAMIC:
			if (glinfo.dynamicload) free(glinfo.dynamicload);
			if (optarg) {
				if (strcmp(optarg, "none") == 0)
					glinfo.dynamicload = NULL;
				else
					glinfo.dynamicload = safe_strdup(optarg);
			}
			break;
		case OPT_INTERFACE:
			mode = optarg[0];
			for (i = 0; i < numberof(ctl_list); i++) {
				if (ctl_list[i]->id_character == mode) {
					ctl = ctl_list[i];
					break;
				}
			}
			if (i >= numberof(ctl_list)) {
				printf("illegal interface option\n");
		                print_usage();
				exit(1);
			}  
			break;
		case OPT_TRACE:
			if (optarg) {
				if (bool_val(optarg))
					BITON(playing_mode, PLAY_TRACE);
				else
					BITOFF(playing_mode, ~PLAY_TRACE);
			} else
				BITSWT(playing_mode, PLAY_TRACE);
			break;
		case OPT_MODE:
			for (p = optarg; *p; p++) {
				switch (*p) {
				case 't':
					BITSWT(playing_mode, PLAY_TRACE); break;
				case 's':
					BITOFF(playing_mode, PLAY_NORMAL);
					BITON(playing_mode, PLAY_SHUFFLE); break;
				case 'n':
					BITOFF(playing_mode, PLAY_SHUFFLE);
					BITON(playing_mode, PLAY_NORMAL); break;
				case 'p':
					BITSWT(playing_mode, PLAY_AUTOSTART); break;
				case 'q':
					BITON(playing_mode, PLAY_AUTOEXIT);
					BITOFF(playing_mode, PLAY_REPEAT); break;
				case 'r':
					BITON(playing_mode, PLAY_REPEAT);
					BITOFF(playing_mode, PLAY_AUTOEXIT); break;
				default:
					fprintf(stderr, "illegal mode option '%c'\n", *p);
					break;
				}
			}
			break;
#ifdef INCLUDE_TK_MODE
		case OPT_EFFECT:
			tk_effect = optarg;
			break;
		case OPT_GEOMETRY:
			tk_geometry = optarg;
			break;
		case OPT_DISPLAY:
			tk_display = optarg;
			break;
#endif
		default:
			general_options(OPTION_FLAGS, cp);
			break;
		}
	}
}


void print_usage(void)
{
	int i;
	printf("drvmidi ver.%s  MIDI player for AWE sound driver\n", VERSION_STR);
	printf("     copyright (c) 1996,1997 by Takashi Iwai\n");
	printf("usage: drvmidi [-options] midifile[.gz] ...\n");
	printf("options are:\n");
	printf("  -i, --interface=type: available interfaces are\n");
	for (i = 0; i < numberof(ctl_list); i++) {
		printf("        %c : %s", ctl_list[i]->id_character,
		       ctl_list[i]->id_name);
		if (ctl == ctl_list[i])
			printf("  (default)");
		printf("\n");
	}
	print_general_options(stdout, OPTION_FLAGS, long_options);
	printf("  -t, --trace[=bool]: turn on tracing mode (equivalent with -mt)\n");
	printf("  -m, --mode=modifiers: set playing mode\n");
	printf("      n: normal play, s: shffule play, r: repeat play, ");
#if defined(INCLUDE_TK_MODE) || defined(INCLUDE_NCURSES_MODE)
	printf("t: trace on");
#endif
	printf("\n");
#ifdef INCLUDE_TK_MODE
	printf("      p: auto start, q: auto exit\n");
	printf("  --effect=config-file: set user defined chorus/reverb configuration\n");
	printf("  -d, --display=xdisp: set window display\n");
	printf("  -g, --geometry=xgeom: set window geometry\n");
#endif
}

/*----------------------------------------------------------------
 * open / close sequencer
 *----------------------------------------------------------------*/

int midi_open(MidiInfo *mp)
{
	if (! seq_opened()) {
		int no_block, do_echo;
		no_block = (mp->seq_buffered && ctl->need_sync);
		do_echo = (no_block && mp->use_echoback);
		seq_blocking_mode(!no_block);
		if (seq_init(do_echo)) {
			ctl->cmsg(CMSG_ERROR, -1, "can't open sequencer device");
			return 1;
		}
	}

	mp->master_volume = 100;
	seq_set_chorus(mp->chorus);
	seq_set_reverb(mp->reverb);
	seq_change_volume(mp->volume_base);
	seq_equalizer(mp->bass_level, mp->treble_level);
	seq_set_drumchannels(mp->drumflag);
	seq_channel_priority(mp->chn_prior);
	seq_set_realtime_pan(mp->realtime_pan);
	seq_new_volume_mode(mp->new_volume_mode);
	if (mp->midi_mode == MODE_XG && mp->xg_mapping)
		seq_set_def_drum(64);
	else
		seq_set_def_drum(0);
	seqbuf_dump();

	ctl->system_change(SY_CHORUS_MODE, mp->chorus);
	ctl->system_change(SY_REVERB_MODE, mp->reverb);
	ctl->master_volume(mp->volume_base);

	return 0;
}

void midi_close(MidiInfo *mp)
{
	if (seq_opened()) {
		seq_terminate_all();
		seq_end();
	}
}

static void clean_up(int sig)
{
	/*ctl->cmsg(CMSG_INFO, 0, "\nterminating..");*/
	if (seq_opened()) {
		seq_terminate_all();
		seq_end();
	}
	ctl->close();
}

/*================================================================*/

/* search the appropriate event for skip forward/back and returns its time */
static int search_event(MidiInfo *mp, int delta)
{
	int i, target;
	MidiEvent *ev;

	ev = mp->list + mp->curev;
	target = mp->curcs + delta;
	if (delta > 0) {
		for (i = mp->curev; i <= mp->endidx; i++, ev++) {
			if (ev->csec >= target)
				break;
		}
		return ev->csec;
	} else {
		for (i = mp->curev; i > 0; i--, ev--) {
			if (ev->csec <= target)
				break;
		}
		return ev->csec;
	}
}

/* store the channel status by the specified time, and jump there */
static void jump_to(MidiInfo *mp, int csec, int from_start)
{
	int i;
	MidiEvent *ev;
	int type;

	seq_terminate_all();
	push_clear();

	type = (from_start ? EV_STARTUP : EV_SETSTAT);
	ev = mp->list;
	if (!from_start && mp->curcs < csec) {
		i = mp->curev;
		ev += i;
	} else {
		i = 0;
		channel_init(mp);
	}
	for (; i <= mp->endidx; i++, ev++) {
		if (ev->csec >= csec)
			break;
		do_midi_event(ev, mp, type);
	}

	if (i > mp->endidx)
		i = mp->endidx;

	mp->curev = mp->prevev = i;
	mp->updatecs = mp->prevcs = mp->curcs = mp->list[i].csec;

	ctl->reset(mp);
	ctl->current_time(mp->curcs);

	channel_set(mp);
	seq_clear(mp->curcs);
	mp->echocs = mp->curcs;
}

/* jump to start event or the first note-on event */
static void jump_to_start(MidiInfo *mp)
{
	if (mp->startcs > 0)
		jump_to(mp, mp->startcs, TRUE);
	else
		reset_status(mp);
}

/* terminate sounds and reset all status */
static void reset_status(MidiInfo *mp)
{
	seq_terminate_all();
	channel_init(mp);
	push_clear();

	mp->curev = mp->prevev = 0;
	mp->updatecs = mp->prevcs = mp->curcs = 0;

	ctl->reset(mp);
	ctl->current_time(0);

	channel_set(mp);
	seq_clear(0);
	mp->echocs = 0;
}

#define TOO_LONG_DELTA	4000

/* find start position */
static void find_start_pos(MidiInfo *mp)
{
	MidiEvent *ev = mp->list;
	int i = 0;

	mp->startcs = 0;
	if (mp->skip_blank_head) {
		for (i = 0; i < mp->nlists; i++) {
			if (ev[i].type == ME_NOTEON) {
				mp->startcs = ev[i].csec;
				break;
			}
		}
	}
	mp->endidx = mp->nlists - 1;
	for (i++; i < mp->nlists; i++) {
		if (ev[i].csec - ev[i-1].csec >= TOO_LONG_DELTA) {
			mp->endidx = i - 1;
			break;
		}
	}
}

/*----------------------------------------------------------------*/

/* play a midi file */
int play_midi_file(MidiInfo *mp)
{
	int piped;
	MidiEvent *ev;
	FILE *fp;
	char *p;
	int cmd;

	if ((fp = CmpOpenFile(mp->filename, &piped)) == NULL) {
		ctl->cmsg(CMSG_ERROR, -1, "can't open MIDI file %s",
			  mp->filename);
		return RC_NEXT;
	}
	ctl->cmsg(CMSG_INFO, 0, "reading midi %s:", mp->filename);
	if ((p = strrchr(mp->filename, '/')) != NULL)
		ctl->file_name(p + 1);
	else
		ctl->file_name(mp->filename);

	init_preload(mp);

	/* read index file before reading midi file */
	find_midi_index(mp, TRUE);

	ev = ReadMidiFile(fp, mp);
	CmpCloseFile(fp, piped);

	if (ev == NULL)
		return RC_ERROR;
	else if (mp->nlists == 0) {
		FreeMidiFile(mp);
		return RC_ERROR;
	}

	/* read index file after reading midi file */
	find_midi_index(mp, FALSE);

	if (mp->midi_mode == MODE_XG && mp->xgload) {
		if (! mp->dynamic_loaded) {
			if (midi_open(mp)) return RC_NONE;
			ctl->cmsg(CMSG_INFO, 1, "loading xg map %s", mp->xgload);
			preload_sample(mp, mp->xgload);
		}
	} else if (mp->dynamicload) {
		if (! mp->dynamic_loaded) {
			if (midi_open(mp)) return RC_NONE;
			ctl->cmsg(CMSG_INFO, 1, "loading default font %s", mp->dynamicload);
			preload_sample(mp, mp->dynamicload);
		}
	}

	load_partial_fonts(mp);

	if (midi_open(mp))
		return RC_NONE;

	find_start_pos(mp);
	ctl->total_time(mp->list[mp->endidx].csec);

	for (;;) {
		jump_to_start(mp);
		while ((cmd = send_seq_events(mp)) == RC_NONE)
			;
		if (cmd != RC_TUNE_END || !ctl->repeated)
			break;
	}
	reset_status(mp);
	FreeMidiFile(mp); /* free event list */

	if (cmd == RC_TUNE_END)
		return RC_NEXT;
	if (cmd == RC_QUIT || cmd == RC_KILL)
		midi_close(mp);
	return cmd;
}


/*----------------------------------------------------------------
 * trick to avoid send on/off at the same time
 *----------------------------------------------------------------*/

/* temporary event-list handlers */

#define MAX_PUSHLIST		32

static int npushed;
static int ev_pushed[MAX_PUSHLIST];

static void push_clear(void)
{
	npushed = 0;
}

static int pop_event(MidiInfo *mp)
{
	int i;
	if (npushed <= 0)
		return RC_NONE;
	seq_wait(1);
	mp->prevcs++;
	for (i = 0; i < npushed; i++) {
		do_midi_event(mp->list + ev_pushed[i], mp, EV_PLAY);
	}
	npushed = 0;
	return RC_NONE;
}

/*----------------------------------------------------------------*/

/* check the note-on event was done in the same csec */
static int check_same_csec(MidiEvent *ev, MidiInfo *mp)
{
	int i;

	if (! mp->check_same_csec) return FALSE;

	if (ev->type != ME_NOTEOFF &&
	    (ev->type != ME_NOTEON || VelOf(ev) != 0))
		return FALSE;
	if (npushed >= MAX_PUSHLIST)
		return FALSE;

	/* search backward the note-on event;
	 * if found in the same csec, do this event later.
	 */
	for (i = mp->curev - 1; i >= 0; i--) {
		if (mp->list[i].csec < ev->csec)
			break;
		if (mp->list[i].type == ME_NOTEON &&
		    mp->list[i].channel == ev->channel &&
		    KeyOf(mp->list + i) == KeyOf(ev) &&
		    VelOf(mp->list + i) != 0) {
			ev_pushed[npushed++] = mp->curev;
			return TRUE;
		}
	}
	return FALSE;
}

/*----------------------------------------------------------------*/

/* send wait and do the event */
static int process_event(MidiInfo *mp)
{
	int rc;
	MidiEvent *ev;

	if (seq_pending())
		return RC_NONE;

	ev = mp->list + mp->curev;
	if (ev->csec > mp->prevcs) {
		pop_event(mp);
		if (ev->csec > mp->prevcs)
			seq_wait(ev->csec - mp->prevcs);
		if (ctl->need_sync && ev->csec >= mp->echocs + 100) {
			seq_echo(ev->csec);
			mp->echocs = ev->csec;
		}
		mp->prevcs = ev->csec;
	}

	/* check note on/off events */
	if (! check_same_csec(ev, mp)) {
		/* send a seq event */
		if ((rc = do_midi_event(ev, mp, EV_PLAY)) != RC_NONE)
			return rc;
	}

	if (mp->curev >= mp->endidx) {
		pop_event(mp);
		seq_echo(ev->csec+1);
		return RC_TUNE_END;
	}

	mp->curev++;
	return RC_NONE;
}


/*----------------------------------------------------------------
 * get input message from control panel
 *----------------------------------------------------------------*/

static int do_control(MidiInfo *mp)
{
	int cmd, val;

	val = 0;
	cmd = ctl->read(mp, &val);
	if (cmd == RC_NONE) return RC_NONE;

	switch (cmd) {
	case RC_PAUSE:
		seq_terminate_all();
		for (;;) {
			val = 1;
			cmd = ctl->blocking_read(mp, &val);
			if (cmd == RC_CONTINUE)
				break;
			else if (cmd == RC_QUIT || cmd == RC_KILL)
				return cmd;
		}
		/* no break; continued */
	case RC_CONTINUE:
		jump_to(mp, mp->curcs, FALSE);
		return RC_JUMP;
	case RC_CHANGE_CHORUS:
		mp->chorus = glinfo.chorus = val;
		seq_set_chorus(val);
		return RC_NONE;
	case RC_CHANGE_REVERB:
		mp->reverb = glinfo.reverb = val;
		seq_set_reverb(val);
		return RC_NONE;
	case RC_CHANGE_VOLUME:
		mp->volume_base = glinfo.volume_base = val;
		seq_change_volume(mp->volume_base * mp->master_volume / 127);
		ctl->master_volume(val);
		return RC_NONE;
	case RC_CHANGE_BASS:
		mp->bass_level = glinfo.bass_level = val;
		seq_equalizer(mp->bass_level, mp->treble_level);
		return RC_NONE;
	case RC_CHANGE_TREBLE:
		mp->treble_level = glinfo.treble_level = val;
		seq_equalizer(mp->bass_level, mp->treble_level);
		return RC_NONE;
	case RC_BASE_CHANGE:
		mp->base_offset += val;
		jump_to(mp, mp->curcs, FALSE);
		return RC_JUMP;
	case RC_RESTART:
		jump_to_start(mp);
		return RC_JUMP;
	case RC_JUMP:
		jump_to(mp, val, FALSE);
		return RC_JUMP;
	case RC_SEARCH:
		jump_to(mp, search_event(mp, val), FALSE);
		return RC_JUMP;
	case RC_PREVIOUS:
		if (!ctl->fuzzy_prev || mp->curcs > mp->startcs + 300) {
			jump_to_start(mp);
			return RC_JUMP;
		}
		cmd = RC_REALLY_PREVIOUS;
		/* continue */
	case RC_KILL:
	case RC_NEXT:
	case RC_REALLY_PREVIOUS:
	case RC_LOAD_FILE:
	case RC_QUIT:
		seq_terminate_all();
		return cmd;
	}
	return RC_NONE;
}


/*----------------------------------------------------------------
 * play event list
 *----------------------------------------------------------------*/

#define BUFFER_THRESHOLD	200
#define TIME_TO_UPDATE(mp) (seq_lasttime() > (mp)->curcs + BUFFER_THRESHOLD)

/* send seq events */
static int send_seq_events(MidiInfo *mp)
{
	int rc;
	do {
		if (seq_pending()) {
			ctl->cmsg(CMSG_INFO, 2, "pending..");
			rc = wait_pending(mp);
			continue;
		}
		if ((rc = process_event(mp)) != RC_NONE)
			break;
		if (ctl->need_sync &&
		    (!mp->seq_buffered || TIME_TO_UPDATE(mp))) {
			seqbuf_dump();
			rc = wait_pending(mp);
		}
	} while (rc == RC_NONE);

	if (rc == RC_TUNE_END) {
		seqbuf_dump();
		if (ctl->need_sync) {
			while ((rc = wait_pending(mp)) == RC_NONE)
				;
		} else
			seq_wait_time(mp->list[mp->endidx].csec);
	}
	return rc;
}

#define SLEEP_DELTA		5
#define UPDATE_DELTA		10

/* update interface and control status */
static void update_status(MidiInfo *mp)
{
	/* update interface status */
	for (; mp->prevev < mp->curev; mp->prevev++) {
		if (mp->list[mp->prevev].csec > mp->curcs)
			break;
		do_midi_event(mp->list + mp->prevev, mp, EV_CONTROL);
	}
	/* update control */
	if (mp->curcs >= mp->updatecs + UPDATE_DELTA) {
		ctl->update(mp);
		mp->updatecs = mp->curcs;
	}
	/* set current time */
	ctl->current_time(mp->curcs);
}


#ifdef DONT_TRUST_SYSTEM_SLEEP
#define CHECK_DELTA_TIME()	1
#else
#define CHECK_DELTA_TIME()	(seq_lasttime() - seq_curtime() <= SLEEP_DELTA*2)
#endif

/* wait and update controls */
static int wait_pending(MidiInfo *mp)
{
	int rc;
	
	do {
		if (seq_timer_started())
			update_status(mp);
		/* input from control */
		if ((rc = do_control(mp)) == RC_JUMP)
			break;
		else if (rc != RC_NONE)
			return rc;
		
		if (! mp->seq_buffered) {
			if (CHECK_DELTA_TIME()) {
				mp->curcs = seq_wait_time(seq_lasttime());
				if (mp->curcs >= mp->list[mp->endidx].csec)
					return RC_TUNE_END;
				return RC_NONE;
			}
		}
		mp->curcs = seq_sleep(SLEEP_DELTA);

		if (mp->curcs > mp->list[mp->endidx].csec)
			return RC_TUNE_END;

		if (mp->seq_buffered && !TIME_TO_UPDATE(mp))
			break;
	} while (seq_pending());

	return RC_NONE;
}
