/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef APPLICATION_H_
#define APPLICATION_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ConfigFile.h"
#include "UrlPatternsFile.h"
#include "ContentFilterGroup.h"
#include "IntrusivePtr.h"
#include "InterthreadCommandQueue.h"
#include "NetworkActivityHandler.h"
#include "FilterJsLogHandler.h"
#include "RequestLogHandler.h"
#include "types.h"
#include <sigc++/sigc++.h>
#include <wx/string.h>
#include <wx/app.h>
#include <wx/filename.h>
#include <string>
#include <iosfwd>
#include <memory>
#include <deque>
#include <time.h>

class Config;
class UrlPatterns;
class TrayIcon;
class TrayMenu;
class WorkerThreadPool;
class wxIdleEvent;
class ACE_FILE_IO;

class Application :
	public wxApp,
	private InterthreadCommandQueue::Notifier
{
public:
	typedef std::deque<IntrusivePtr<ContentFilterGroup> > ContentFilterList;
	
	Application();
	
	~Application();
	
	RequestLogHandler& getRequestLogHandler() { return m_requestLogHandler; }

	virtual bool OnInit();
	
	virtual int OnExit();

	void showTrayMenu();
	
	void showLogDialog();
	
	void showRequestLogWindow();
	
	void showBasicConfigDialog();
	
	void showAdvancedConfigWindow();
	
	void showFilterConfigWindow();

	void setFilteringEnabled(bool val);
	
	void setScriptFilteringEnabled(bool val);
	
	void requestAppExit();
	
	static bool fileToStream(ACE_FILE_IO& file, std::ostream& strm);
	
	static bool readFile(
		wxFileName const& fname, std::string& target,
		time_t* mtime = 0, bool log_errors = true);
	
	static bool writeFile(
		wxFileName const& fname, std::string const& data,
		time_t* mtime = 0, bool log_errors = true);
	
	static bool renameFile(
		wxFileName const& from, wxFileName const& to,
		bool log_errors = true);
	
	static bool deleteFile(wxFileName const& fname, bool log_errors = true);
	
	static bool processConfig(
		std::string const& text, Config& target,
		ConfigFile& file_structure);
	
	bool applyConfig(Config const& config);
	
	void applyBrokenConfig(Config const& broken_config);
	
	bool loadConfig();
	
	static bool processUrlPatterns(
		std::string const& text, wxString const& fname,
		UrlPatterns& target, UrlPatternsFile& file_structure);
	
	static void applyStandardUrlPatterns(UrlPatterns const& patterns);
	
	static void applyLocalUrlPatterns(UrlPatterns const& patterns);
	
	void loadStandardUrlPatterns();
	
	void loadLocalUrlPatterns();
	
	void loadContentFilters();
	
	bool loadFilterFnames(wxFileName const& dirname, std::deque<wxString>& fnames);
	
	void loadEnabledFilterList(
		ContentFilterGroup& group, wxFileName const& dirname);
	
	void appendContentFilterGroup(
		IntrusivePtr<ContentFilterGroup> const& group);
	
	bool removeContentFilterGroup(
		IntrusivePtr<ContentFilterGroup> const& group);
	
	void sortContentFiltersByFileName();
	
	void updateGlobalContentFilters();
	
	wxFileName getConfigFileName() const;
	
	wxFileName getConfigDefaultFileName() const;
	
	wxFileName getStandardUrlPatternsFileName() const;
	
	wxFileName getLocalUrlPatternsFileName() const;
	
	wxFileName getFiltersDir() const;
	
	ConfigFile& configFile() { return m_configFile; }
	
	ConfigFile const& configFile() const { return m_configFile; }
	
	UrlPatternsFile& standardUrlPatternsFile() {
		return m_standardUrlPatternsFile;
	}
	
	UrlPatternsFile const& standardUrlPatternsFile() const {
		return m_standardUrlPatternsFile;
	}
	
	UrlPatternsFile& localUrlPatternsFile() {
		return m_localUrlPatternsFile;
	}
	
	UrlPatternsFile const& localUrlPatternsFile() const {
		return m_localUrlPatternsFile;
	}
	
	ContentFilterList& contentFilters() {
		return m_contentFilters;
	}
	
	ContentFilterList const& contentFilters() const {
		return m_contentFilters;
	}
	
	bool isAcceptingConnections() const {
		return m_isAcceptingConnections;
	}
	
	sigc::signal1<void, bool>& acceptingConnectionsSignal() {
		return m_acceptingConnectionsSignal;
	}
	
	void handleNetworkActivity();
private:
	struct ContentFilterFnameComparator;
	
	enum { COMMAND_QUEUE_CAPACITY = 32 };
	
	virtual void notifyCommandsPending();
	
	virtual void notifyQueueEmpty();
	
	void onIdle(wxIdleEvent& evt);
	
	void processCommands();
	
	void setAcceptingConnections(bool val);
	
	bool m_isExiting;
	int m_commandNestLevel;
	volatile int32_t m_havePendingCommands;
	InterthreadCommandQueue m_commandQueue;
	NetworkActivityHandler m_networkActivityHandler;
	FilterJsLogHandler m_filterJsLogHandler;
	RequestLogHandler m_requestLogHandler;
	wxFileName m_appDir;
	wxFileName m_confDir;
	ConfigFile m_configFile;
	UrlPatternsFile m_standardUrlPatternsFile;
	UrlPatternsFile m_localUrlPatternsFile;
	ContentFilterList m_contentFilters;
	sigc::signal1<void, bool> m_acceptingConnectionsSignal;
	std::auto_ptr<TrayIcon> m_ptrTrayIcon;
	std::auto_ptr<TrayMenu> m_ptrTrayMenu;
	bool m_isAcceptingConnections;
	std::auto_ptr<WorkerThreadPool> m_ptrWorkerPool;
	// Must be the last thing to construct and first to destroy
	// because we don't want worker threads to act on destroyed objects.
	
	DECLARE_EVENT_TABLE()
};

DECLARE_APP(Application)

#endif
