/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ConnAcceptor.h"
#include "EventHandler.h"
#include "RefCounter.h"
#include "IntrusivePtr.h"
#include "ReactorFactory.h"
#include "SynchFactory.h"
#include "RefCountableSAP.h"
#include "ServerConnectionPool.h"
#include <ace/config-lite.h>
#include <ace/SOCK_Acceptor.h>
#include <ace/SOCK_Stream.h>
#include <ace/INET_Addr.h>
#include <ace/Time_Value.h>

using namespace std;

class ConnAcceptor::EvtHandler : public EventHandler<RefCounter<ACE_MT_SYNCH> >
{
public:
	EvtHandler(ConnAcceptor& owner, AcceptorPtr const& acceptor);
private:
	virtual void handleRead(ACE_HANDLE);
	
	ConnAcceptor& m_rOwner;
	AcceptorPtr m_ptrAcceptor;
};


class ConnAcceptor::AcceptorReadyException
{
public:
	AcceptorPtr acceptor;
	
	AcceptorReadyException(AcceptorPtr acceptor) : acceptor(acceptor) {}
};


ConnAcceptor::ConnAcceptor(StopOnSignals stop_on_signals)
{
	m_ptrReactor = ReactorFactory::createBestReactor(
		SynchFactory<ACE_MT_SYNCH>(),
		stop_on_signals == STOP_ON_SIGNALS
	);
	
	/*
	The reactor in ConnAcceptor turns out to be a perfect
	candidate for cleaning up ServerConnectionPool.
	*/
	ServerConnectionPool::instance()->connectToReactor(*m_ptrReactor);
}

ConnAcceptor::~ConnAcceptor()
{
}

bool
ConnAcceptor::add(AcceptorPtr const& acceptor)
{
	Reactor::EventHandlerPtr handler(new EvtHandler(*this, acceptor));
	try {
		m_ptrReactor->registerHandler(
			acceptor->get_handle(), handler, Reactor::READ
		);
		return true;
	} catch (Reactor::Exception&) {
		return false;
	}
}

void
ConnAcceptor::remove(AcceptorPtr const& acceptor)
{
	ReactorHandlerId id = m_ptrReactor->findHandler(acceptor->get_handle());
	if (id) {
		m_ptrReactor->unregisterHandler(id);
	}
}

void
ConnAcceptor::removeAll()
{
	m_ptrReactor->unregisterAllHandlers();
}

ConnAcceptor::Status
ConnAcceptor::accept(ACE_SOCK_Stream& peer, ACE_INET_Addr& peer_addr)
{
	Reactor::Status status = Reactor::SUCCESS;
	do {
		try {
			status = m_ptrReactor->runEventLoop();
		} catch (AcceptorReadyException const& e) {
			return acceptConnection(e.acceptor, peer, peer_addr);
		}
	} while (status == Reactor::SUCCESS);
	
	if (status == Reactor::STOPPED) {
		m_ptrReactor->restart();
		return ABORTED;
	} else if (status == Reactor::INTERRUPTED) {
		return STOPPED_BY_SIGNAL;
	} else {
		return FAILED;
	}
}

void
ConnAcceptor::abort()
{
	m_ptrReactor->stop();
}

ConnAcceptor::Status
ConnAcceptor::acceptConnection(
	AcceptorPtr const& acceptor,
	ACE_SOCK_Stream& peer, ACE_INET_Addr& peer_addr)
{
	ACE_Time_Value timeout;
	int const restart = 0; // don't retry on EINTR
	int const reset_new_handle = 1; // call WSAEventSelect(h, 0, 0);
	int const res = acceptor->accept(
		peer, &peer_addr, &timeout, restart, reset_new_handle
	);
	if (res == 0) {
		return ACCEPTED;
	} else {
		return FAILED;
	}
}


/*===================== ConnAcceptor::EvtHandler =======================*/

ConnAcceptor::EvtHandler::EvtHandler(
	ConnAcceptor& owner, AcceptorPtr const& acceptor)
:	m_rOwner(owner),
	m_ptrAcceptor(acceptor)
{
}

void
ConnAcceptor::EvtHandler::handleRead(ACE_HANDLE)
{
	throw AcceptorReadyException(m_ptrAcceptor);
	/*
	We just have to throw an exception here.
	Consider what happens in case of the edge-triggered WFMOReactor:
	If WaitForMultipleObjects() indicates that several handles are ready,
	several event handlers will be dispatched.  But we need only one
	of them!  If we ignore the rest, we won't get any more notifications
	on them (notice the edge-triggered behaviour).
	Also, we can't use Reactor::stop(), as it's asynchronous, so it
	won't stop the reactor immediately.
	Fortunately, Reactor was designed to be completely exception-safe.
	*/
}
