;*=====================================================================*/
;*    serrano/prgm/project/bigloo/bmacs/ude/ude-repl.el                */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Mon May 25 16:31:35 1998                          */
;*    Last change :  Sat Dec  5 08:20:47 1998 (serrano)                */
;*    -------------------------------------------------------------    */
;*    This file implements a REPL process embedded in emacs.           */
;*    To fetch region in the code it uses buffer local variables that  */
;*    are defined inside UDE-CONFIG.                                   */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(provide 'ude-repl)
(require 'ude-config)
(require 'ude-icon)
(require 'ude-toolbar)
(require 'ude-autoload)
(require 'comint)

;*---------------------------------------------------------------------*/
;*    ude-repl-buffer ...                                              */
;*---------------------------------------------------------------------*/
(defvar ude-repl-buffer nil)

;*---------------------------------------------------------------------*/
;*    ude-repl ...                                                     */
;*---------------------------------------------------------------------*/
(defun ude-repl ()
  (let ((bufname (concat "*" ude-repl-buffer-name "*")))
    (switch-to-buffer-other-frame bufname)
    (comint-run ude-repl)
    (setq ude-repl-buffer (current-buffer))
    (make-variable-buffer-local 'comint-prompt-regexp)
    (setq comint-prompt-regexp ude-repl-prompt)
    (process-kill-without-query (get-buffer-process ude-repl-buffer))
    (set-process-sentinel (get-buffer-process ude-repl-buffer)
			  'ude-repl-sentinel)
    (ude-repl-init-toolbar)
    (run-hooks 'ude-repl-hooks)))

;*---------------------------------------------------------------------*/
;*    ude-repl-other-frame ...                                         */
;*---------------------------------------------------------------------*/
(defun ude-repl-other-frame ()
  (interactive)
  (if (bufferp ude-repl-buffer)
      (let ((pop-up-frames t))
	(display-buffer ude-repl-buffer))
    (ude-repl)))

;*---------------------------------------------------------------------*/
;*    ude-repl-sentinel ...                                            */
;*    -------------------------------------------------------------    */
;*    This function is called when the ude running process changes     */
;*    of state. This means that the process has uden killed or         */
;*    stopped or anything else like that.                              */
;*---------------------------------------------------------------------*/
(defun ude-repl-sentinel (proc msg)
  (cond
   ((null (buffer-name (process-buffer proc)))
    ;; the buffer has uden killed. We stop displaying
    ;; arrow in the source files.
    (set-process-buffer proc nil))
   ((memq (process-status proc) '(signal exit))
    (let ((window (get-buffer-window ude-repl-buffer t)))
      (if (one-window-p window)
	  (let ((frame (window-frame window)))
	    (delete-frame frame))))
    (kill-buffer ude-repl-buffer)
    (setq ude-repl-buffer nil))))

;*---------------------------------------------------------------------*/
;*    ude-repl-send-region ...                                         */
;*---------------------------------------------------------------------*/
(defun ude-repl-send-region (beg end)
  (interactive "r")
  (if (not (bufferp ude-repl-buffer))
      (let ((buffer (current-buffer)))
	(ude-repl)
	(set-buffer buffer)))
  (let ((command (buffer-substring beg end))
	(proc    (get-buffer-process ude-repl-buffer)))
    (set-buffer ude-repl-buffer)
    (goto-char (process-mark proc))
    (delete-region (point) (point-max))
    (insert command)
    (let* ((pmark (process-mark proc))
	   (intxt (if (>= (point) (marker-position pmark))
		      (progn (if comint-eol-on-send (end-of-line))
			     (buffer-substring pmark (point)))
		    nil)))
      (comint-send-input)
      (goto-char (point-max)))))

;*---------------------------------------------------------------------*/
;*    ude-repl-find ...                                                */
;*---------------------------------------------------------------------*/
(defun ude-repl-find (var)
  "Find a variable definition."
  (interactive (ude-interactive-ident (point) "variable: "))
  (let ((search-str (format "(define[ \t\n]+[(]?%s" (regexp-quote var))))
    (if (not (re-search-backward search-str (point-min) t))
	(ude-error "Can't find REPL variable `%s'" var))))

;*---------------------------------------------------------------------*/
;*    ude-repl-quit ...                                                */
;*---------------------------------------------------------------------*/
(defun ude-repl-quit ()
  "Kill the comint subjob and repl buffer"
  (interactive)
  (condition-case ()
      (comint-kill-subjob)
    (error
     (if (bufferp ude-repl-buffer)
	 (ude-delete-buffer-window-frame ude-repl-buffer)))))

;*---------------------------------------------------------------------*/
;*    ude-repl-send-buffer ...                                         */
;*---------------------------------------------------------------------*/
(defun ude-repl-send-buffer ()
  "Send the whole buffer to the inferior Bigloo process."
  (interactive)
  (ude-repl-send-region (point-min) (point-max)))

;*---------------------------------------------------------------------*/
;*    Various compile toolbar button                                   */
;*---------------------------------------------------------------------*/
(defvar ude-repl-tag-button
  (toolbar-make-button-list ude-tag-icon))
(defvar ude-next-button
  (toolbar-make-button-list ude-repl-next-icon))
(defvar ude-prev-button
  (toolbar-make-button-list ude-repl-prev-icon))
(defvar ude-repl-help-button
  (toolbar-make-button-list ude-help-icon))
(defvar ude-repl-info-button
  (toolbar-make-button-list ude-info-icon))
(defvar ude-repl-quit-button
  (toolbar-make-button-list ude-quit-icon))

;*---------------------------------------------------------------------*/
;*    ude-repl-opened-toolbar ...                                      */
;*---------------------------------------------------------------------*/
(defvar ude-repl-opened-toolbar 
  '(;;close button
    [ude-close-toolbar-button ude-close-repl-toolbar t "Close toolbar"]
    [:style 2d :size 2]

    ;; the quit button
    [ude-repl-quit-button ude-repl-quit t "Quit Repl"]
    [:style 2d :size 2]
    
    ;; the tag button
    [ude-repl-tag-button ude-repl-find t "Find definition"]
    [:style 2d :size 2]
    
    ;; the next button
    [ude-next-button comint-next-input t "Next Input"]
    [:style 2d :size 2]
    
    ;; prev error
    [ude-prev-button comint-previous-input t "Previous Input"]
    [:style 2d :size 2]
    
    ;; flushing right
    nil
    [:style 2d :size 2]
    ;; the help action
    [ude-repl-help-button describe-mode t "Help"]
    ;; the info button
    [ude-repl-info-button ude-docline t
    "The online documentation for Bee"]))

;*---------------------------------------------------------------------*/
;*    ude-repl-closed-toolbar ...                                      */
;*---------------------------------------------------------------------*/
(defvar ude-repl-closed-toolbar
  '([ude-open-toolbar-button ude-open-repl-toolbar t "Open toolbar"]))

;*---------------------------------------------------------------------*/
;*    opening/closing toolbars ...                                     */
;*---------------------------------------------------------------------*/
(defun ude-close-repl-toolbar ()
  (ude-open-close-toolbar ude-repl-closed-toolbar))

(defun ude-open-repl-toolbar ()
  (ude-open-close-toolbar ude-repl-opened-toolbar))

;*---------------------------------------------------------------------*/
;*    ude-repl-init-toolbar ...                                        */
;*    -------------------------------------------------------------    */
;*    This hook simply set the UDE repl toolbar for the buffer         */
;*---------------------------------------------------------------------*/
(defun ude-repl-init-toolbar ()
  (set-specifier default-toolbar-visible-p t)
  (set-specifier default-toolbar ude-repl-opened-toolbar
		 (current-buffer)))
  
