//-*- Mode: C++; indent-tabs-mode: nil; -*-
//
//  BMPx - The Dumb Music Player
//  Copyright (C) 2005-2006 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non GPL-compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifndef BMP_FAM_MONITOR_HH
#define BMP_FAM_MONITOR_HH

#include <boost/shared_ptr.hpp>

#include <glibmm.h>
#include <list>
#include <fam.h>
#include <utility>

namespace Bmp
{
  /** This namespace contains classes and auxilliaries for online file monitoring
   *
   */
  namespace FileMonitor
  {
    /** Slot for notifications of added or removed files.
     *  Returns void, and takes an std::string as argument,
     *  which is the fully qualified path that was modified.
     */
    typedef sigc::slot<void, std::string const&> FamFunc;

    /** Monitor is the main class running watches and 
     *  processing FAM/GAMIN events
     */
    class Monitor
    {
      public:

        Monitor ();
        ~Monitor ();

        bool monitor_running;

        /** Adds a watch with a given name for a given path
        *
        * @param monitor_name Name of the monitor, i.e. "Skins"
        * @param path_name Path to watch for events
        * @param func_added @link Bmp::FamFunc@endlink to call when a file has been added
        * @param func_removed @link Bmp::FamFunc@endlink to call when a file has been removed
        *
        */
        bool
        add_watch  (std::string const&  monitor_name,
                    std::string const&  path_name,
                    FamFunc const&      func_added,
                    FamFunc const&      func_removed);

        /** Terminates a running watch and disconnects the slots 
        *
        * @param monitor_name The Name of the monitor that was supplied with @link  Bmp::FileMonitor::Monitor::add_watch()@endlink
        *
        */
        bool stop_watch (std::string const& monitor_name);

        private:

          /** This class is a watch of a particular path, running a
           *  collector thread and processing FAM/GAMIN events
           */
          class Watch
          {
            public:

              Watch (std::string const&  monitor_name,
                     std::string const&  path_name,
                     FamFunc const&      func_added,
                     FamFunc const&      func_removed);

              ~Watch ();

              void terminate_watch ();

            private:

              typedef std::pair<std::string, int> Event;
              typedef std::list<Event> EventList;

              std::string  monitor_name;
              std::string  path_name;
              FamFunc      func_added;
              FamFunc      func_removed;

              FAMConnection  fc;
              Glib::Thread  *thread;
              Glib::Mutex    termination;
              bool           terminate;
              EventList      events;

              void watch_thread ();
              void monitor_thread ();
          };

        Monitor (const Monitor &original);
        const Monitor & operator = (const Monitor &original);
        typedef boost::shared_ptr<Watch> TWatchPtr;
        std::map<std::string, TWatchPtr> watches;
    };
  } // namespace FileMonitor
} // namespace Bmp

#endif // !BMP_FAM_MONITOR_HH
