//  BMPx - The Dumb Music Player
//  Copyright (C) 2005 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License Version 2
//  as published by the Free Software Foundation.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <time.h>
#include <errno.h>

#include <glibmm.h>
#include <glibmm/i18n.h>
#include <gtkmm.h>
#include <libglademm.h>

#include <cstring>
#include <iostream>
#include <sstream>
#include <string>

#include <boost/algorithm/string.hpp>
#include <boost/format.hpp>
#include <boost/shared_ptr.hpp>

#include "main.hh"
#include "network.hh"
#include "paths.hh"
#include "stock.hh"
#include "debug.hh"
#include "util.hh"
#include "util_file.hh"
#include "ui_toolbox.hh"
#include "uri++.hh"

#include "ui-part-podcasts.hh"
#include "dialog-add-podcast.hh"

#include "neon++/session.hh"
#include "neon++/request.hh"

#include "x_play.hh"

namespace
{

#define POD_ACTION_ADD_PODCAST "pod-action-add-podcast"
#define POD_ACTION_DEL_PODCAST "pod-action-del-podcast"
#define POD_ACTION_UPD_PODCAST "pod-action-upd-podcast"

#define POD_ACTION_COPY_RSS_LINK   "pod-action-copy-rss-link"
#define POD_ACTION_COPY_WEB_LINK   "pod-action-copy-web-link"

#define POD_ACTION_DOWNLOAD_ITEM   "pod-action-download-item"

  const char *ui_menu_podcasts =
  "<ui>"
  ""
  "<menubar name='MenuBarMain'>"
  "   <menu action='MenuUiPartPodcasts'>"
  "     <menuitem action='" POD_ACTION_ADD_PODCAST "'/>"
  "     <menuitem action='" POD_ACTION_DEL_PODCAST "'/>"
  "     <menuitem action='" POD_ACTION_UPD_PODCAST "'/>"
  "       <separator name='podcasts-sep1'/>"
  "     <menuitem action='" POD_ACTION_COPY_RSS_LINK "'/>"
  "     <menuitem action='" POD_ACTION_COPY_WEB_LINK "'/>"
  "       <separator name='podcasts-sep2'/>"
  "     <menuitem action='" POD_ACTION_DOWNLOAD_ITEM "'/>"
  "   </menu>"
  "</menubar>"
  ""
  "</ui>";

  Glib::ustring
  get_timestr_from_time_t (time_t atime)
  {
    using namespace Glib;

    struct tm atm;
    localtime_r (&atime, &atm);   

    char bdate[64]; 
    strftime (bdate, 64, "%d %b %Y", &atm); 

    char btime[64]; 
    strftime (btime, 64, "%T", &atm); 

    ustring str;
    str . append ("<b>") . append (bdate) . append ("</b> ") . append (btime);

    return str;
  }
}

namespace Bmp
{
  namespace UiPart
  {
    guint 
    Podcasts::add_ui ()
    {
      if (!Network::check_connected())
        return 0;
      else
        return m_ui_manager->add_ui_from_string (ui_menu_podcasts);
    };

    void
    Podcasts::upd_cast_internal (Bmp::PodCastTools::PodCast const& cast,
                                 Glib::RefPtr<Gdk::Pixbuf> & cast_logo,
                                 Gtk::TreeModel::iterator const& m_iter)
    {
      using namespace Gtk;
      using namespace Gdk;
      using namespace Glib;

      bool defaultimage = true;

      if (cast_logo)
        {
          (*m_iter)[casts.image] = cast_logo; 
          defaultimage = false;
        }

      (*m_iter)[casts.defaultimage] = defaultimage;

      ustring title;
      title.append ("<big><b>");
      title.append (Markup::escape_text (cast.title.get()));
      title.append ("</b></big>");

      title.append ("\n\n<i><small>");
      title.append (_("Last Updated:\n"));
      title.append (get_timestr_from_time_t (cast.last_poll_time));
      title.append ("</small></i>");

      (*m_iter)[casts.title]      = title;
      (*m_iter)[casts.cast_uri]   = cast.uri;
      (*m_iter)[casts.key]        = title.casefold_collate_key();
  
      m_store_casts->row_changed (m_store_casts->get_path (m_iter), m_iter);
      m_view_casts->get_selection()->select (m_iter);
    }

    void
    Podcasts::add_cast_internal (Bmp::PodCastTools::PodCast const& cast,
                                 Glib::RefPtr<Gdk::Pixbuf> & cast_logo)
    {
      using namespace Gtk;
      using namespace Gdk;
      using namespace Glib;

      TreeModel::iterator m_iter = m_store_casts->append (); 
      bool defaultimage = true;

      if (cast_logo)
        {
          (*m_iter)[casts.image] = cast_logo; 
          defaultimage = false;
        }

      (*m_iter)[casts.defaultimage] = defaultimage;

      ustring title;
      title.append ("<big><b>");
      title.append (Markup::escape_text (cast.title.get()));
      title.append ("</b></big>");

      title.append ("\n\n<i><small>");
      title.append (_("Last Updated:\n"));
      title.append (get_timestr_from_time_t (cast.last_poll_time));
      title.append ("</small></i>");

      (*m_iter)[casts.title]      = title;
      (*m_iter)[casts.cast_uri]   = cast.uri;
      (*m_iter)[casts.key]        = title.casefold_collate_key();
      (*m_iter)[casts.guid]       = m_guid; 

      context_map.insert (std::make_pair (m_guid, m_iter));
      m_guid++;
    }

    void
    Podcasts::del_podcast ()
    {
      using namespace Gtk;
      using namespace Bmp::PodCastTools;
      
      TreeModel::iterator m_iter = m_view_casts->get_selection()->get_selected();

      guint64 guid = (*m_iter)[casts.guid];
      context_map.erase (guid);

      m_cast_manager->pod_cast_delete ((*m_iter)[casts.cast_uri]);

      m_view_casts->get_selection()->unselect (m_iter);
      m_store_casts->erase (m_iter);
    }

    void
    Podcasts::upd_podcast ()
    {
      using namespace Gtk;
      using namespace Bmp::PodCastTools;
      
      TreeModel::iterator m_iter = m_view_casts->get_selection()->get_selected();

      Glib::ustring cast_uri ((*m_iter)[casts.cast_uri]);
      m_cast_manager->pod_cast_update (cast_uri);

      try{
        Glib::RefPtr<Gdk::Pixbuf> cast_logo;
        PodCast const& cast (m_cast_manager->pod_cast_fetch (cast_uri, cast_logo));
        upd_cast_internal (cast, cast_logo, m_iter);
        }
      catch (PodCastTools::NetworkError& cxe)
        {
          g_warning ("Network Error during reading of podcast: %s", cxe.what());
          return;
        }
      catch (PodCastTools::ParsingError& cxe)
        {
          g_warning ("Parsing Error during reading of podcast: %s", cxe.what());
          return;
        }
      catch (PodCastTools::InvalidUriError& cxe)
        {
          g_warning ("URI Is Invalid: %s", cxe.what());
          return;
        }
    }

    void
    Podcasts::add_podcast ()
    {
      using namespace Gdk;
      using namespace Gtk;
      using namespace Glib;
      using namespace Bmp::PodCastTools;
      using namespace std;

      DialogAddPodcast * dialog (DialogAddPodcast::create());
      Glib::ustring uri;
      int response = dialog->run (uri);
      delete dialog;

      if (response == GTK_RESPONSE_OK)
        {
          try{
            Glib::RefPtr<Gdk::Pixbuf> cast_logo;
            m_cast_manager->pod_cast_cache (uri);
            PodCast const& cast (m_cast_manager->pod_cast_fetch (uri, cast_logo));
            add_cast_internal (cast, cast_logo);
            }
          catch (PodCastTools::NetworkError& cxe)
            {
              g_warning ("Network Error during reading of podcast: %s", cxe.what());
              return;
            }
          catch (PodCastTools::ParsingError& cxe)
            {
              g_warning ("Parsing Error during reading of podcast: %s", cxe.what());
              return;
            }
          catch (PodCastTools::InvalidUriError& cxe)
            {
              g_warning ("URI Is Invalid: %s", cxe.what());
              return;
            }
          catch (PodCastTools::PodCastInvalidError& cxe)
           {
              g_warning ("URI Is Invalid: %s", cxe.what());
              return;
            }
        }
    }

    void
    Podcasts::copy_podcast_link ()
    {
      using namespace Gtk;
      using namespace Bmp::PodCastTools;
      
      TreeModel::iterator m_iter = m_view_casts->get_selection()->get_selected();
      Gtk::Clipboard::get_for_display (Gdk::Display::get_default())->set_text ((*m_iter)[casts.cast_uri]); 
    }

    void
    Podcasts::copy_web_link ()
    {
      using namespace Gtk;
      using namespace Bmp::PodCastTools;
      
      TreeModel::iterator m_iter = m_view_casts->get_selection()->get_selected();
      PodCast const& cast (m_cast_manager->pod_cast_fetch ((*m_iter)[casts.cast_uri]));

      if (cast.link)
        {
          Gtk::Clipboard::get_for_display (Gdk::Display::get_default())->set_text (cast.link.get());
        }
    }


    Podcasts::Podcasts (Glib::RefPtr<Gnome::Glade::Xml> const& xml, Glib::RefPtr<Gtk::UIManager> ui_manager)

         :  
            PlaybackSource  (PlaybackSource::NONE),
            UiPart::Base    (xml, ui_manager),
            m_playing       (Gdk::Pixbuf::create_from_file (Glib::build_filename
                              (BMP_IMAGE_DIR_MAIN, "playing.png"))),
            m_feed_default  (Gdk::Pixbuf::create_from_file (Glib::build_filename
                              (BMP_IMAGE_DIR_PODCAST, "feed-default.png"))),
            m_guid          (0)
    {
        using namespace Gtk;
        using namespace Glib;
        using namespace Bmp::PodCastTools;
        using namespace std;

        if (!Network::check_connected())
          {
            m_ref_xml->get_widget ("podcasts-vpaned")->set_sensitive (false);
            return;
          }

        m_ref_xml->get_widget ("podcasts-textview", textview);
        textbuffer = Gtk::TextBuffer::create();
        textview->set_buffer (textbuffer);

        m_actions = Gtk::ActionGroup::create ("ActionsPodcasts"); 

        m_actions->add ( Gtk::Action::create ("dummy", "dummy"));
        m_actions->add ( Gtk::Action::create ("MenuUiPartPodcasts", _("Po_dcasts")));

        m_actions->add ( Gtk::Action::create (POD_ACTION_ADD_PODCAST,
                                  Gtk::StockID (BMP_STOCK_FEED_ADD),
                                  _("_Add Podcast")),
                                  (sigc::mem_fun (this, &Bmp::UiPart::Podcasts::add_podcast)));

        m_actions->add ( Gtk::Action::create (POD_ACTION_DEL_PODCAST,
                                  Gtk::StockID (BMP_STOCK_FEED_DELETE),
                                  _("_Remove Podcast")),
                                  (sigc::mem_fun (this, &Bmp::UiPart::Podcasts::del_podcast)));

        m_actions->add ( Gtk::Action::create (POD_ACTION_UPD_PODCAST,
                                  Gtk::StockID (GTK_STOCK_REFRESH),
                                  _("_Update Podcast")),
                                  (sigc::mem_fun (this, &Bmp::UiPart::Podcasts::upd_podcast)));

        m_actions->add ( Gtk::Action::create (POD_ACTION_COPY_RSS_LINK,
                                  Gtk::StockID (BMP_STOCK_HTML_LINK),
                                  _("_Copy Podcast RSS Link")),
                                  (sigc::mem_fun (this, &Bmp::UiPart::Podcasts::copy_podcast_link)));

        m_actions->add ( Gtk::Action::create (POD_ACTION_COPY_WEB_LINK,
                                  Gtk::StockID (BMP_STOCK_HTML_LINK),
                                  _("_Copy Podcast Web Link")),
                                  (sigc::mem_fun (this, &Bmp::UiPart::Podcasts::copy_web_link)));

        m_actions->add ( Gtk::Action::create (POD_ACTION_DOWNLOAD_ITEM,
                                  Gtk::StockID (BMP_STOCK_DOWNLOAD),
                                  _("_Download Cast Item")),
                                  (sigc::mem_fun (this, &Bmp::UiPart::Podcasts::download_cast_item))); 

        m_ui_manager->insert_action_group (m_actions);

        m_actions->get_action (POD_ACTION_COPY_RSS_LINK)->set_sensitive (false);
        m_actions->get_action (POD_ACTION_COPY_WEB_LINK)->set_sensitive (false);
        m_actions->get_action (POD_ACTION_DEL_PODCAST)->set_sensitive (false);
        m_actions->get_action (POD_ACTION_UPD_PODCAST)->set_sensitive (false);

        m_actions->get_action (POD_ACTION_DOWNLOAD_ITEM)->set_sensitive (false);

        try{
          m_cast_manager = 
            boost::shared_ptr
            <Bmp::PodCastTools::PodCastManager>
            (new Bmp::PodCastTools::PodCastManager());
          }
        catch (...)
          {
            m_actions->set_sensitive (false);
            m_ref_xml->get_widget ("podcasts-vpaned")->set_sensitive (false);
            Gtk::MessageDialog dialog (_("There was an error parsing the feedlist.opml file. Please check your "
                                         "Podcast cache directory for sanity, as well as the feedlist.opml file.\n\n"
                                         "Podcasts have to bedisabled until the error has been resolved, sorry for any"
                                         "inconveniences."),
                                          false,
                                          Gtk::MESSAGE_ERROR,
                                          Gtk::BUTTONS_OK,
                                          true);
            dialog.set_title (_("Podcast Manager: Parse Error - BMP"));
            dialog.run ();
            return;
          }

        // Casts Store
        m_ref_xml->get_widget ("podcasts-view-casts", m_view_casts);
        for (unsigned int n = 0; n < 2; ++n)
        {
          switch (n)
          {
            case 0:
            {
              Gtk::CellRendererPixbuf *cell = manage (new CellRendererPixbuf());
              cell->property_xalign() = 0.5; 
              cell->property_xpad() = 8; 
              cell->property_ypad() = 4; 
              TreeViewColumn *column = manage (new TreeViewColumn ("", *cell));
              column->set_cell_data_func (*cell, sigc::mem_fun (this, &Bmp::UiPart::Podcasts::cell_data_func_cast_logo));
              column->set_resizable (false);
              column->set_expand (false);
              column->set_sizing (TREE_VIEW_COLUMN_FIXED);
              column->set_fixed_width (80);
              m_view_casts->append_column (*column);
              break;
            }

            case 1:
            {
              Gtk::CellRendererText *cell = manage (new CellRendererText());
              cell->property_xalign() = 0.0; 
              cell->property_yalign() = 0.0; 
              cell->property_xpad() = 4; 
              cell->property_ypad() = 4; 
              cell->property_ellipsize() = Pango::ELLIPSIZE_END;

              TreeViewColumn *column  = manage (new TreeViewColumn ("", *cell));
              column->add_attribute (*cell, "markup", 1);
              column->set_sizing (TREE_VIEW_COLUMN_FIXED);
              m_view_casts->append_column (*column);
              break;
            }
          }
        }

        m_store_casts = ListStore::create (casts);
        m_view_casts->set_model (m_store_casts);
        m_view_casts->get_selection()->set_mode (SELECTION_SINGLE);
        m_view_casts->get_selection()->signal_changed().connect
            (sigc::mem_fun (this, &Bmp::UiPart::Podcasts::casts_selection_changed));
        m_store_casts->set_default_sort_func
              (sigc::mem_fun (this, &Bmp::UiPart::Podcasts::casts_default_sort_func));
        m_store_casts->set_sort_column (-1, Gtk::SORT_ASCENDING);

        // Items Store
        m_ref_xml->get_widget ("podcasts-view-items", m_view_items);
        for (unsigned int n = 0; n < 5; ++n)
        {
          switch (n)
          {
            case 0:
            {
              Gtk::CellRendererPixbuf *cell = manage (new CellRendererPixbuf());
              cell->property_xalign() = 0.5; 
              cell->property_xpad() = 2; 
              cell->property_ypad() = 4; 
              TreeViewColumn *column = manage (new TreeViewColumn ("", *cell));
              column->set_cell_data_func (*cell, sigc::mem_fun (this, &Bmp::UiPart::Podcasts::cell_data_func_playing));
              column->set_resizable (false);
              column->set_expand (false);
              column->set_sizing (TREE_VIEW_COLUMN_FIXED);
              column->set_fixed_width (30);
              Gtk::Image * image = Gtk::manage ( new Gtk::Image() );
              image->set (Glib::build_filename (BMP_IMAGE_DIR, "blue-speaker.png")); 
              column->set_widget (*image); 
              image->show();
              m_view_items->append_column (*column);
              break;
            }

            case 1:
            {
              Gtk::CellRendererPixbuf * cell = 0; 
              TreeViewColumn *column = manage (new TreeViewColumn ());

              cell = manage (new CellRendererPixbuf());
              cell->property_xalign () = 0.2;
              column->pack_start (*cell, true);
              column->set_cell_data_func (*cell, sigc::mem_fun (this, &Bmp::UiPart::Podcasts::cell_data_func_disk));

              cell = manage (new CellRendererPixbuf());
              cell->property_xalign () = 0.8; 
              column->pack_start (*cell, true);
              column->set_cell_data_func (*cell, sigc::mem_fun (this, &Bmp::UiPart::Podcasts::cell_data_func_new));

              column->set_resizable (false);
              column->set_expand (false);
              column->set_sizing (Gtk::TREE_VIEW_COLUMN_FIXED);
              column->set_fixed_width (56);
              column->set_sort_column (4); 
              m_view_items->append_column (*column);
              break;
            }

            case 2:
            {
              Gtk::CellRendererText *cell = manage (new CellRendererText());
              cell->property_xalign() = 0.0; 
              cell->property_yalign() = 0.5; 
              cell->property_xpad() = 4; 
              cell->property_ypad() = 4; 
              cell->property_ellipsize() = Pango::ELLIPSIZE_END;
              TreeViewColumn *column  = manage (new TreeViewColumn ("Title", *cell));
              column->add_attribute (*cell, "markup", 0);
              column->set_resizable (true);
              column->set_expand (false);
              column->set_sort_column (0);
              column->set_min_width (300);
              m_view_items->append_column (*column);
              break;
            }

            case 3:
            {
              Gtk::CellRendererText *cell = manage (new CellRendererText());
              cell->property_xalign() = 0.0; 
              cell->property_yalign() = 0.5; 
              cell->property_xpad() = 4; 
              cell->property_ypad() = 4; 
              TreeViewColumn *column  = manage (new TreeViewColumn ("Date", *cell));
              column->set_cell_data_func (*cell, sigc::mem_fun (this, &Bmp::UiPart::Podcasts::cell_data_func_date));
              column->set_resizable (true);
              column->set_expand (false);
              column->set_sort_column (2);
              m_view_items->append_column (*column);
              break;
            }

            case 4:
            {
              Gtk::CellRendererText *cell = manage (new CellRendererText());
              cell->property_xalign() = 0.0; 
              cell->property_yalign() = 0.5; 
              cell->property_xpad() = 4; 
              cell->property_ypad() = 4; 
              TreeViewColumn *column  = manage (new TreeViewColumn ("Feed", *cell));
              column->add_attribute (*cell, "markup", 1);
              column->set_resizable (true);
              column->set_expand (false);
              column->set_sort_column (1);
              m_view_items->append_column (*column);
              break;
            }
          }
        }

        m_store_items = ListStore::create (items);

        m_view_items->set_model (m_store_items);
        m_view_items->get_selection()->set_mode (SELECTION_SINGLE);
        m_view_items->get_selection()->signal_changed().connect
            (sigc::mem_fun (this, &Bmp::UiPart::Podcasts::items_selection_changed));
        m_view_items->signal_row_activated().connect
                (sigc::mem_fun (this, &Bmp::UiPart::Podcasts::activate_default));

#if 0
        m_store_filter = TreeModelFilter::create (m_store_base);
        m_store_filter->set_visible_func
              (sigc::mem_fun (*this, &Bmp::AlbumView::visible_func));
#endif

      try {
          PodCastList list;
          m_cast_manager->pod_cast_get_list (list);

          for (PodCastList::const_iterator c = list.begin() ; c != list.end () ; ++c)
            {
             try {
                  if (mcs->key_get <bool> ("podcasts", "update-on-startup"))
                    {
                      Glib::RefPtr<Gdk::Pixbuf> cast_logo;
                      PodCast cast_orig = m_cast_manager->pod_cast_fetch (*c, cast_logo);
                      m_cast_manager->pod_cast_update (cast_orig.uri);
                      PodCast const& cast (m_cast_manager->pod_cast_fetch (cast_orig.uri, cast_logo));
                      add_cast_internal (cast, cast_logo);
                    }
                  else
                    {
                      Glib::RefPtr<Gdk::Pixbuf> cast_logo;
                      PodCast const& cast (m_cast_manager->pod_cast_fetch (*c, cast_logo));
                      add_cast_internal (cast, cast_logo);
                    } 
                }
              catch (PodCastTools::ParsingError& cxe)
                {
                  g_warning ("Parsing Error during reading of podcast: %s", cxe.what());
                  break;
                }
              catch (PodCastTools::InvalidUriError& cxe)
                {
                  g_warning ("URI Is Invalid: %s", cxe.what());
                  break;
                }
            }
        }
      catch (...) {}

      ::play->signal_buffering().connect
        (sigc::mem_fun (this, &Bmp::UiPart::Podcasts::buffering));
    }

    void
    Podcasts::buffering (double buffering)
    {
      using namespace Gtk;
      using namespace Glib;

      if (buffering == 1.)
        {
          if (m_title && m_sti) s_track_info_.emit (m_title.get(), m_sti.get());
        }
    }

    Podcasts::~Podcasts ()
    {}

    ///////////////////////////////////////////////////////////////////////////////////////

    void
    Podcasts::download_cast_item ()
    {
      using namespace Gtk;
      using namespace Glib;
      using namespace Bmp::PodCastTools;

      m_ref_xml->get_widget ("podcast-progress-download")->show ();
      m_ref_xml->get_widget ("podcasts-vpaned")->set_sensitive (false);
      m_actions->set_sensitive (false);
      while (gtk_events_pending()) gtk_main_iteration ();

      TreeModel::iterator i_iter (m_view_items->get_selection ()->get_selected());
      PodCastItem item = ((*i_iter)[items.item]);      

      TreeModel::iterator c_iter (m_view_casts->get_selection ()->get_selected());
      PodCast const& cast (m_cast_manager->pod_cast_fetch ((*c_iter)[casts.cast_uri]));

      Bmp::URI u (item.enclosure_url.get());

      using namespace boost::algorithm;
      std::vector<std::string> subs;
      split (subs, u.path, is_any_of ("."));

      std::string _type = subs[subs.size()-1];

      std::string title = Glib::filename_from_utf8 (cast.title.get());
      std::string basepath = Glib::build_filename (mcs->key_get<std::string>("podcasts","download-dir"), title);

      if (g_mkdir_with_parents (basepath.c_str(), S_IRWXU) == 0)
        {
          std::string filename (Glib::build_filename (basepath, 
                                                      Glib::filename_from_utf8 (item.title.get())));

          filename += "." + _type;

          Neon::Request r ( u.hostname,
                            u.fullpath(), 
                            u.port,
                            Neon::Request::RESPONSE_READ,
                            Neon::Request::METHOD_GET,
                            u.scheme,
                            true,
                            filename);
          
          r.signal_request_progress().connect
            (sigc::mem_fun (this, &Bmp::UiPart::Podcasts::download_cast_item_progress));

          if (r.dispatch () == NE_OK)
            {
              PodCastOverlayItem overlay_item; 
              overlay_item.listened_to = item.listened_to;
              overlay_item.downloaded = true;
              overlay_item.localfilename = filename;
              m_cast_manager->pod_cast_item_change (cast.uri, item.guid_value.get(), overlay_item); 
              item.downloaded = true;
              item.localfilename = filename;
              (*i_iter)[items.item] = item;
              m_store_items->row_changed (m_store_items->get_path (i_iter), i_iter);
            }
          else
            {
              s_error_message_.emit (_("Network Error Occured during podcast item download"));
            }
        }
      else
        {
          int errsv = errno;
          Glib::ustring path (Glib::filename_to_utf8 (basepath));
          Glib::ustring message (_("Unable to create/write into directory"));
          message.append ("'").append (path).append ("': ");
          message.append (g_strerror (errsv));
          s_error_message_.emit (message);
        }

      m_ref_xml->get_widget ("podcast-progress-download")->hide ();
      m_ref_xml->get_widget ("podcasts-vpaned")->set_sensitive (true);
      m_actions->set_sensitive (true);
      while (gtk_events_pending()) gtk_main_iteration ();
    }

    void
    Podcasts::download_cast_item_progress (double fraction)
    {
      using namespace Gtk;
      using namespace Glib;
      using namespace Bmp::PodCastTools;

      static boost::format format_progress ("%d %%");
      Gtk::ProgressBar * progress = dynamic_cast<Gtk::ProgressBar *>(m_ref_xml->get_widget ("podcast-progress-download"));
  
      progress->set_fraction (fraction);
      progress->set_text ((format_progress % (int(fraction*100))).str());

      while (gtk_events_pending()) gtk_main_iteration ();
    }

    bool
    Podcasts::sel_guid_is_current_guid ()
    {
      using namespace Gtk;
      using namespace Bmp::PodCastTools;

      if (!m_current_guid) return false; 

      TreeModel::iterator c_iter (m_view_casts->get_selection ()->get_selected());
      return (guint64((*c_iter)[casts.guid]) == m_current_guid.get());
    }

    int
    Podcasts::casts_default_sort_func (Gtk::TreeModel::iterator const& iter_a, Gtk::TreeModel::iterator const& iter_b)
    {
      return std::string((*iter_a)[casts.key]).compare(std::string((*iter_b)[casts.key]));
    }

    void
    Podcasts::cell_data_func_cast_logo (Gtk::CellRenderer* cell, Gtk::TreeModel::iterator const& m_iter)
    {
      Gtk::CellRendererPixbuf * _cell = dynamic_cast<Gtk::CellRendererPixbuf *>(cell);

      if ((*m_iter)[casts.defaultimage]) 
          _cell->property_pixbuf() = m_feed_default->scale_simple (64, 64, Gdk::INTERP_BILINEAR); 
      else
          _cell->property_pixbuf() = Glib::RefPtr<Gdk::Pixbuf>((*m_iter)[casts.image])->scale_simple (64, 64, Gdk::INTERP_BILINEAR);
    }

    void
    Podcasts::cell_data_func_date (Gtk::CellRenderer * cell, Gtk::TreeModel::iterator const& m_iter)
    {
      using namespace Glib;  
      using namespace Gtk;  

      CellRendererText *_cell = dynamic_cast<CellRendererText *>(cell);

      time_t atime = ((*m_iter)[items.date]);
      _cell->property_markup() = get_timestr_from_time_t (atime);
    }

    void
    Podcasts::cell_data_func_playing (Gtk::CellRenderer * cell, Gtk::TreeModel::iterator const& m_iter)
    {
      using namespace Glib;  
      using namespace Gdk;  
      using namespace Gtk;  

      CellRendererPixbuf *_cell = dynamic_cast<CellRendererPixbuf *>(cell);

      if (sel_guid_is_current_guid() && m_current_iter &&
                  (m_store_items->get_path (m_current_iter.get()) == m_store_items->get_path (m_iter))) 
        {
          _cell->property_pixbuf() = m_playing;
        }
      else
        {
          _cell->property_pixbuf() = RefPtr<Pixbuf>(0); 
        }
    }

    void
    Podcasts::cell_data_func_new (Gtk::CellRenderer * cell, Gtk::TreeModel::iterator const& m_iter)
    {
      using namespace Glib;  
      using namespace Gdk;  
      using namespace Gtk;  
      using namespace Bmp::PodCastTools;  

      CellRendererPixbuf *_cell = dynamic_cast<CellRendererPixbuf *>(cell);

      if (!(*m_iter)[items.listenedto])
        {
          _cell->property_pixbuf() = m_view_items->render_icon
                  (Gtk::StockID (BMP_STOCK_NEW), Gtk::ICON_SIZE_SMALL_TOOLBAR); 
        }
      else
        {
          _cell->property_pixbuf() = RefPtr<Pixbuf>(0); 
        }
    }

    void
    Podcasts::cell_data_func_disk (Gtk::CellRenderer * cell, Gtk::TreeModel::iterator const& m_iter)
    {
      using namespace Glib;  
      using namespace Gdk;  
      using namespace Gtk;  
      using namespace Bmp::PodCastTools;  

      CellRendererPixbuf *_cell = dynamic_cast<CellRendererPixbuf *>(cell);

      PodCastItem const& item ((*m_iter)[items.item]);      

      if (item.downloaded)
        {
           _cell->property_pixbuf() = m_view_items->render_icon
                  (Gtk::StockID (BMP_STOCK_HARDDISK), Gtk::ICON_SIZE_SMALL_TOOLBAR); 
        }
      else
        {
          _cell->property_pixbuf() = RefPtr<Pixbuf>(0); 
        }
    }

    ///////////////////////////////////////////////////////////////////////////////////////

    void
    Podcasts::has_next_prev (bool & next, bool & prev)
    {
      using namespace Gtk;
      using namespace Glib;

      next = false;
      prev = false;

      if (sel_guid_is_current_guid() && m_current_iter) 
        {
          TreeModel::Children const& children = m_store_items->children();

          unsigned int position;
          position = m_store_items->get_path (m_current_iter.get()).get_indices().data()[0];

          prev = (position > 0);
          next = (position+1 != children.size());
        }
    }

    Glib::ustring
    Podcasts::get_uri ()
    {
      using namespace Gtk; 
      using namespace Bmp::PodCastTools;

      TreeModel::iterator const& m_iter (m_current_iter.get());
      PodCastItem const& item ((*m_iter)[items.item]);      

      if (item.downloaded)
        return Glib::filename_to_uri (item.localfilename);
      else
        return item.enclosure_url.get(); 
    }

    Glib::ustring
    Podcasts::get_type ()
    {
      using namespace Gtk; 
      using namespace Bmp::PodCastTools;

      TreeModel::iterator const& m_iter (m_current_iter.get());
      PodCastItem const& item ((*m_iter)[items.item]);      

      if (item.enclosure_type)
        return item.enclosure_type.get(); 
      else
        return Glib::ustring(); //Gotta guess here
    }

    GHashTable*
    Podcasts::get_metadata ()
    {
      return 0;
    }

    void
    Podcasts::go_next_prev (Direction direction)
    {
      using namespace Gtk;
      using namespace Glib;

      TreeModel::Path     m_path;
      TreeModel::iterator m_iter;
      unsigned int        position = 0;

      switch (direction)
      {
          case NEXT:
          case PREV:
            if (m_current_iter)
            {
              m_path = m_store_items->get_path (m_current_iter.get()); 
              m_iter = m_current_iter.get();
              position = m_path.get_indices().data()[0];
              m_current_iter.reset();
              m_store_items->row_changed (m_path, m_iter);
            }
            break;

          default: break;
      }

      TreeModel::Path path;

      switch (direction)
      {
          case NEXT:
            path.append_index (position + 1);
            m_current_iter = m_store_items->get_iter (path);
            break;
          
          case PREV:
            path.append_index (position - 1);
            m_current_iter = m_store_items->get_iter (path);
            break;

          case CHECKONLY: break;
      }

      bool next, prev;
      has_next_prev (next, prev);
 
      if (sel_guid_is_current_guid() && m_current_iter) 
        {
          if (next)
            m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_NEXT);
          else
            m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);

          if (prev)
            m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_PREV);
          else
            m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_PREV);

          TreeModel::Path path (m_store_items->get_path (m_current_iter.get()));
          m_store_items->row_changed (path, m_current_iter.get());
        }
      else
        {
            m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_PREV);
            m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);
        }

      s_caps_.emit (m_caps);

    }

    bool
    Podcasts::go_next ()
    {
      go_next_prev  (NEXT);
      return true;
    }

    bool
    Podcasts::go_prev ()
    {
      go_next_prev  (PREV);
      return true;
    }

    void
    Podcasts::stop ()
    {
      using namespace Gtk;

      m_sti.reset();
      m_title.reset();
 
      TreeModel::Path m_path; 
      TreeModel::iterator m_iter;

      if (sel_guid_is_current_guid() && m_current_iter) 
        {
          m_path = m_store_items->get_path(m_current_iter.get());
          m_iter = m_current_iter.get();
        }

      m_current_iter.reset();
      m_current_guid.reset();
      m_current_item_guid.reset();

      if (sel_guid_is_current_guid() && m_current_iter) 
        {
          m_store_items->row_changed (m_path, m_iter);
        }

      m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_PREV);
      m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);
      m_caps = Caps (m_caps & ~PlaybackSource::CAN_SEEK);
      s_caps_.emit (m_caps);
    }

    void
    Podcasts::play_requested ()
    {
    }

    void
    Podcasts::play ()
    {
      using namespace Gtk;
      using namespace Glib;
      using namespace Bmp::PodCastTools;
      
      TreeModel::iterator i_iter (m_view_items->get_selection ()->get_selected());
      PodCastItem item = ((*i_iter)[items.item]);      

      TreeModel::iterator c_iter (m_view_casts->get_selection ()->get_selected());
      PodCast const& cast (m_cast_manager->pod_cast_fetch ((*c_iter)[casts.cast_uri]));

      if (item.guid_value) 
        {
          PodCastOverlayItem overlay_item;
          overlay_from_item (item, overlay_item);

          item.listened_to = true;
          overlay_item.listened_to = true;

          (*i_iter)[items.listenedto]  = true; 
          (*i_iter)[items.item] = item; 
          m_cast_manager->pod_cast_item_change (cast.uri, item.guid_value.get(), overlay_item); 
        }

      SimpleTrackInfo sti;
      sti.artist    = cast.title.get(); 
      sti.title     = item.title.get(); 
      sti.album     = cast.title.get(); 
      sti.duration  = item.enclosure_length / 1000; 
      if ((*c_iter)[casts.defaultimage])
        {
          sti.image = m_feed_default; 
          sti.drawframe = false; 
        }
      else
        {
          sti.image = (*c_iter)[casts.image];
          sti.drawframe = true; 
        }

      m_sti = sti;

      ustring title; 
      title = item.title.get();
      title.append (" <i>(");
      title.append (Glib::Markup::escape_text(cast.title.get()));
      title.append (")</i>");

      m_title = title;

      m_current_iter = i_iter; 
      m_current_guid = (*c_iter)[casts.guid];
      m_current_item_guid = item.guid_value; 

      TreeModel::Path path (m_store_items->get_path (i_iter));
      m_store_items->row_changed (path, i_iter);

      go_next_prev (CHECKONLY);

      if (item.downloaded)
        {
          s_track_info_.emit (m_title.get(), m_sti.get());
          m_caps = Caps (m_caps |  PlaybackSource::CAN_SEEK);
          s_caps_.emit (m_caps);
        }
      else
        {
          m_caps = Caps (m_caps & ~PlaybackSource::CAN_SEEK);
          s_caps_.emit (m_caps);
        }
    }

    void
    Podcasts::restore_context ()
    {
      if (m_current_guid)
        {
          m_view_casts->get_selection()->select (context_map.find(m_current_guid.get())->second);
        }
    }

    ///////////////////////////////////////////////////////////////////////////////////////

    void
    Podcasts::activate_default (Gtk::TreeModel::Path const& path, Gtk::TreeViewColumn* column)
    {
      dynamic_cast<Gtk::Window *>(m_ref_xml->get_widget ("main-ui"))->activate_default ();
    }

    void
    Podcasts::casts_selection_changed()
    {
      using namespace Gtk;
      using namespace Bmp::PodCastTools;

      bool selected = (m_view_casts->get_selection ()->count_selected_rows() > 0);

      m_view_items->queue_draw();
      m_current_iter.reset();

      if (selected)
        {
          TreeModel::iterator m_iter = m_view_casts->get_selection()->get_selected();
          PodCast const& cast (m_cast_manager->pod_cast_fetch ((*m_iter)[casts.cast_uri]));

          if (!cast.link)
            m_actions->get_action (POD_ACTION_COPY_WEB_LINK)->set_sensitive (false);
          else
            m_actions->get_action (POD_ACTION_COPY_WEB_LINK)->set_sensitive (true);

          m_store_items->clear (); 
          for (VPodCastItems::const_iterator i = cast.items.begin(); i != cast.items.end() ; ++i)
            {
              TreeModel::iterator i_iter = m_store_items->append ();

              (*i_iter)[items.title]        = Glib::Markup::escape_text (i->title.get());
              (*i_iter)[items.feed]         = Glib::Markup::escape_text (cast.title.get());
              (*i_iter)[items.item]         = (*i); 
              (*i_iter)[items.listenedto]   = i->listened_to; 

              if (i->pub_date)
                {
                  (*i_iter)[items.date] = Util::parseRFC822Date (i->pub_date.get().c_str()); 
                }
      
              if ((i->guid_value) && (i->guid_value == m_current_item_guid))
                {
                  m_current_iter = i_iter;
                }
            }

          m_actions->get_action (POD_ACTION_COPY_RSS_LINK)->set_sensitive (true);

        }
      else
        {
          m_store_items->clear (); 
          m_actions->get_action (POD_ACTION_COPY_RSS_LINK)->set_sensitive (false);
        }

      m_actions->get_action (POD_ACTION_DEL_PODCAST)->set_sensitive (selected);
      m_actions->get_action (POD_ACTION_UPD_PODCAST)->set_sensitive (selected);

      go_next_prev (CHECKONLY);
    }

    void
    Podcasts::items_selection_changed()
    {
      using namespace Gtk;
      using namespace Bmp::PodCastTools;

      bool selected = (m_view_items->get_selection ()->count_selected_rows() > 0);

      if (selected)
        {
          m_caps = Caps (m_caps |  PlaybackSource::CAN_PLAY);
          TreeModel::iterator m_iter = m_view_items->get_selection()->get_selected();
          PodCastItem const& item ((*m_iter)[items.item]); 
          if (item.description)
            {
              textbuffer->set_text (Util::sanitize_html (item.description.get()));
            }
          else
            {
              textbuffer->set_text("");
            }
          m_actions->get_action (POD_ACTION_DOWNLOAD_ITEM)->set_sensitive (true);
        }
      else
        {
          m_actions->get_action (POD_ACTION_DOWNLOAD_ITEM)->set_sensitive (false);
          m_caps = Caps (m_caps & ~PlaybackSource::CAN_PLAY);
          textbuffer->set_text("");
        }
      s_caps_.emit (m_caps);
    }
  } // namespace UiPart
} // namespace Bmp
