/***************************************************************************
                 bknode.h  -  Bookmark Tree Node Definitions
                             -------------------
    begin                : Sun Sep 15 2002
    copyright            : (C) 2002 by Ken Schenke
    email                : kschenke at users dot sourceforge dot net
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU      *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the Free Software           *
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA         *
 *   02110-1301, USA                                                       *
 *                                                                         *
 ***************************************************************************/

#ifndef BKNODE_H
#define BKNODE_H

#include <QString>
#include <assert.h>

#include <vector>
#include <list>
#include <algorithm>

#include "bkexcept.h"

#ifdef _WIN32
#ifndef strcasecmp
#define strcasecmp stricmp
#endif
#endif

/***************************************************************************
 *                                                                         *
 *   Data types and defined constants.                                     *
 *                                                                         *
 ***************************************************************************/

class BkDiff;
class BkFolder;
class BkBookmark;

typedef unsigned int  BRWSNUM;
typedef   signed int  BRWSTYPE;
typedef unsigned long BKVALID;

#define BKVALID_TITLE		(1L << 0)
#define BKVALID_URL			(1L << 1)
#define BKVALID_MODIFIED	(1L << 2)
#define BKVALID_ADDED		(1L << 3)
#define BKVALID_LASTVISIT	(1L << 4)
#define BKVALID_DESC		(1L << 5)

enum BKNODET { NODETYPE_FOLDER=101, NODETYPE_BOOKMARK, NODETYPE_SEPARATOR };

/***************************************************************************
 *                                                                         *
 *   BkAttr class.                                                         *
 *                                                                         *
 *   This class holds an attribute associated with a BkNode class.  The    *
 *   class definition is presented here simply because it's used in a      *
 *   private vector in the BkNode class.  Browser attributes are private   *
 *   values found in a browser's bookmark database associated with a       *
 *   bookmark or folder.  Usually, the meaning of these attributes is not  *
 *   known or is of no concern to BookmarkBridge.  However, they may be of *
 *   importance to the browser so this class exists to assist the BkNode   *
 *   class and its derivations in keeping track of these attributes so     *
 *   they can be saved back to a browser's bookmark database when other    *
 *   changes are written.                                                  *
 *                                                                         *
 ***************************************************************************/

class BkAttr
{
public:
	BkAttr() { m_browser = 0; }

	BkAttr & operator=(const BkAttr &bkattr)
	{
		if(this != &bkattr)
		{
			m_name  = bkattr.m_name;
			m_value = bkattr.m_value;
			m_browser = bkattr.m_browser;
		}

		return *this;
	}

	bool operator==(const QString &str) const
		{ return (m_name == str); }

	QString m_name;
	QString m_value;
	BRWSNUM m_browser;
};

/***************************************************************************
 *                                                                         *
 *   BkOrder class.                                                        *
 *                                                                         *
 *   This class stores the order in which a bookmark or folder occured in  *
 *   while reading the browser's bookmarks.  The class definition is here  *
 *   because it's used in a private vector in the BkNode class.  The order *
 *   is important so that we can save the bookmarks and folders back to    *
 *   the browser in the same order in which we read them.                  *
 *                                                                         *
 ***************************************************************************/

class BkOrder
{
public:
	BkOrder(BRWSNUM browser, int order)  :
		m_Browser(browser),
		m_Order(order) { };
	BkOrder(const BkOrder &order) :
		m_Browser(order.m_Browser),
		m_Order(order.m_Order) { };
		
	BkOrder & operator=(const BkOrder &order)
	{
		if(this != &order)
		{
			m_Browser = order.m_Browser;
			m_Order = order.m_Order;
		}
		
		return *this;
	}
	
	bool operator==(BRWSNUM browser) const
		{ return browser == m_Browser; }

	BRWSNUM browser(void) const
		{ return m_Browser; }	
	int order(void) const
		{ return m_Order; }
	void setOrder(int order)
		{ m_Order = order; }

private:
	BRWSNUM	m_Browser;
	int	m_Order;
};

/***************************************************************************
 *                                                                         *
 *   BkNode class.                                                         *
 *                                                                         *
 *   This class serves as an abstract base for the bookmark tree built     *
 *   in memory at runtime.  This tree represents the hierarchical bookmark *
 *   structure for a browser or read from an XBEL XML file.  Derivatives   *
 *   include BkFolder and BkBookmark.                                      *
 *                                                                         *
 ***************************************************************************/

class BkNode
{
public:
	BkNode();
	BkNode(const BkNode &node);	// copy constructor
	virtual ~BkNode();          // virtual destructor

/***************************************************************************
 *                                                                         *
 *   Browser Attribute functions.  See the explanation for BkAttr above.   *
 *                                                                         *
 ***************************************************************************/

	void	attr(
		const QString &name, QString &value, BRWSNUM browser) const;
	void	attr(
		int iAttr, QString &name,
		QString &value, BRWSNUM &browser) const
		throw(BkException);
	int		nAttrs(void) const	{ return m_attrs.size(); }

/***************************************************************************
 *                                                                         *
 *   Browser Order functions.  See the explanation for BkOrder above.      *
 *                                                                         *
 ***************************************************************************/

 	void	copyOrder(const BkNode &node);
	int	order(BRWSNUM) const;
	void	setOrder(BRWSNUM browser, int order);

/***************************************************************************
 *                                                                         *
 *   Overloaded assignment, equality, and comparison operators.            *
 *                                                                         *
 ***************************************************************************/

	virtual bool operator==(const QString &str) const
			{ return (m_title.toLower() == str.toLower()); }
	virtual BkNode & operator=(const BkNode &node);
	virtual bool operator<(const BkNode &node) const
			{ return m_title.toLower() < node.m_title.toLower(); }

	virtual	BKNODET	type(void) const = 0;

/***************************************************************************
 *                                                                         *
 *   Miscellaneous node attributes.                                        *
 *                                                                         *
 ***************************************************************************/

	QString diffDesc(void) const	{ return m_DiffDesc; }
	QString	title(void) const	{ return m_title; }
	QString desc(void) const	{ return m_desc; }

	bool	ignore(void) const
			{ return m_ignore; }

	const std::vector<BRWSNUM> & browserFound(void) const
			{ return m_browserFound; }
	bool	browserFound(BRWSNUM browser) const
			{ return std::find(
				m_browserFound.begin(),
				m_browserFound.end(),
				browser) != m_browserFound.end();
			}

	const std::vector<BRWSNUM> & browserSaved(void) const
			{ return m_browserSaved; }
	bool	browserSaved(BRWSNUM browser) const
			{ return std::find(
				m_browserSaved.begin(),
				m_browserSaved.end(),
				browser) != m_browserSaved.end();
			}
	bool	isFieldValid(BKVALID field) const
			{ return m_ValidFields & field; }

	// The isDeleted() function is used after all browser bookmarks
	// have been loaded and compared against those currently stored
	// in each browser.  It returns a non-zero value if the bookmark / folder
	// was previously saved in any of the browers but is no longer.

	bool	isDeleted(void) const;

	QString path(void) const
			{ return m_path; }

	void	setAttr(const QString &name, const QString &value, BRWSNUM browser);
	void	setBrowserSaved(BRWSNUM browser)
			{
				if(std::find(
					m_browserSaved.begin(),
					m_browserSaved.end(),
					browser) == m_browserSaved.end())
						m_browserSaved.push_back(browser);
			}
	void	setBrowserSaved(const BkNode &node)
			{
				std::vector<BRWSNUM>::const_iterator it;

				for(it=node.m_browserSaved.begin();
					it!=node.m_browserSaved.end();
					++it)
				{
					if(std::find(
						m_browserSaved.begin(),
						m_browserSaved.end(),
						*it) == m_browserSaved.end())
						m_browserSaved.push_back(*it);
				}
			}
	void	setBrowserFound(BRWSNUM browser)
			{
				if(std::find(
					m_browserFound.begin(),
					m_browserFound.end(),
					browser) == m_browserFound.end())
						m_browserFound.push_back(browser);
			}
	void	setBrowserFound(const BkNode &node)
			{
				std::vector<BRWSNUM>::const_iterator it;

				for(it=node.m_browserFound.begin();
					it!=node.m_browserFound.end();
					++it)
				{
					if(std::find(
						m_browserFound.begin(),
						m_browserFound.end(),
						*it) == m_browserFound.end())
						m_browserFound.push_back(*it);
				}
			}
	void	setDiffDesc(const QString &diffDesc)
			{ m_DiffDesc = diffDesc; }
	void	setIgnore(bool ignore)
			{ m_ignore = ignore; }
	void	setParentPath(const QString &parentPath);
	void	setTitle(const QString &title);
	void	setDesc(const QString &desc);
	void	setValidField(BKVALID field)
			{ m_ValidFields |= field; }

	void	clearBrowserFound(void)
			{ m_browserFound.clear(); }
	void	clearBrowserFound(BRWSNUM browser)
			{
				std::vector<BRWSNUM>::iterator it;

				it = std::remove(m_browserFound.begin(), m_browserFound.end(), browser);
				if(it != m_browserFound.end())
					m_browserFound.erase(it, m_browserFound.end());
			}
	void	clearBrowserSaved(void)
			{ m_browserSaved.clear(); }
	void	clearBrowserSaved(BRWSNUM browser)
			{
				std::vector<BRWSNUM>::iterator it;

				it = std::remove(m_browserSaved.begin(), m_browserSaved.end(), browser);
				if(it != m_browserSaved.end())
					m_browserSaved.erase(it, m_browserSaved.end());
			}

protected:
	bool	m_ignore;
	QString	m_path;

private:
	QString	m_DiffDesc;
	QString	m_desc;
	QString	m_title;
	QString	m_parentPath;

	std::vector<BkAttr>	m_attrs;
	std::vector<BkOrder>	m_order;

	// The m_browserSaved field indicates, for each browser, if that
	// browser currently has the bookmark saved in it.  This field is
	// written to and read from the XBEL file.  This field is a vector
	// of BRWSNUM's.

	std::vector<BRWSNUM> m_browserSaved;

	// The m_browserFound field indicates, for each browser, if that
	// browser has the bookmark in its database.  If this field is not
	// set for a browser but is set in the m_browserSaved field, this
	// indicates that this bookmark was previously saved in the
	// browser but has been deleted.  If this field is set but the
	// m_browserSaved field is not set for this browser, this indicates
	// that this is a new bookmark that has not yet been saved to the
	// browser.  This field is not saved in the XBEL file.  This field
	// is a vector of BRWSNUM's.

	std::vector<BRWSNUM> m_browserFound;

	// The m_ValidFields field indicates which fields in this node are
	// valid.  As information for this bookmark is read from the XBEL
	// file or from a browser database, this field is used to indicate
	// which fields were available in the source database.  See the
	// BKVALID_* defines at the top of this header for valid values.

	BKVALID	m_ValidFields;
};

/***************************************************************************
 *                                                                         *
 *   BkBookmark class.                                                     *
 *                                                                         *
 *   This class derives from BkNode and represents a bookmark in the       *
 *   bookmark tree.                                                        *
 *                                                                         *
 ***************************************************************************/

class BkBookmark : public BkNode
{
public:
	BkBookmark();
	BkBookmark(const BkBookmark &bmark);	// copy constructor
	~BkBookmark() { };

	QString	diffTitle(void) const
			{ return m_DiffTitle; }
	virtual BKNODET type(void) const
			{ return NODETYPE_BOOKMARK; }
	QString	url(void) const
			{ return m_url; }

	virtual bool operator==(const QString &str) const
			{ return (m_url.toLower() == str.toLower()); }
	virtual bool operator==(const BkBookmark &bmark) const;
	virtual bool operator!=(const BkBookmark &bmark) const
			{ return !operator==(bmark); }

	BkBookmark & operator=(const BkBookmark & bookmark);

	void	setDiffTitle(const QString &diffTitle)
			{ m_DiffTitle = diffTitle; }
	void	setUrl(const QString &url)
			{ m_url = url; }

	bool	findUrl(const QString &url) const
			{ return url.toLower() == m_url.toLower(); }

	void	rollUpDifferences(void) throw(BkException);

private:
	QString	m_url;

	QString	m_DiffTitle;
};

/***************************************************************************
 *                                                                         *
 *   BkSeparator class.                                                    *
 *                                                                         *
 *   This class derives from BkNode and represents a separator in the      *
 *   bookmark tree.                                                        *
 *                                                                         *
 ***************************************************************************/

class BkSeparator : public BkNode
{
public:
	virtual BKNODET type(void) const
			{ return NODETYPE_SEPARATOR; }
};

/***************************************************************************
 *                                                                         *
 *   BkFolder class.                                                       *
 *                                                                         *
 *   This class derives from BkNode and represents a folder within the     *
 *   bookmark hierarchy.                                                   *
 *                                                                         *
 ***************************************************************************/

class BkFolder : public BkNode
{
public:
	BkFolder();
	~BkFolder() { };

	BkBookmark	& addBookmark(const BkBookmark &bookmark);
	BkFolder 	& addFolder(const BkFolder &folder);
	BkSeparator	& addSeparator(const BkSeparator &separator);

	std::vector<BkBookmark>::size_type bookmarks(void) const
			{ return m_bookmarks.size(); }
	std::vector<BkFolder>::size_type folders(void) const
			{ return m_folders.size(); }
	std::vector<BkSeparator>::size_type separators(void) const
			{ return m_separators.size(); }

	virtual 	BKNODET type(void) const
			{ return NODETYPE_FOLDER; }

	virtual BkFolder & operator=(const BkFolder & folder);

	std::vector<BkBookmark>::iterator BookmarksBegin(void)
			{ return m_bookmarks.begin(); }
	std::vector<BkBookmark>::iterator BookmarksEnd(void)
			{ return m_bookmarks.end(); }
	std::vector<BkBookmark>::const_iterator BookmarksBegin(void) const
			{ return m_bookmarks.begin(); }
	std::vector<BkBookmark>::const_iterator BookmarksEnd(void) const
			{ return m_bookmarks.end(); }
	std::vector<BkFolder>::iterator FoldersBegin(void)
			{ return m_folders.begin(); }
	std::vector<BkFolder>::iterator FoldersEnd(void)
			{ return m_folders.end(); }
	std::vector<BkFolder>::const_iterator FoldersBegin(void) const
			{ return m_folders.begin(); }
	std::vector<BkFolder>::const_iterator FoldersEnd(void) const
			{ return m_folders.end(); }
	std::vector<BkSeparator>::iterator SeparatorsBegin(void)
			{ return m_separators.begin(); }
	std::vector<BkSeparator>::iterator SeparatorsEnd(void)
			{ return m_separators.end(); }
	std::vector<BkSeparator>::const_iterator SeparatorsBegin(void) const
			{ return m_separators.begin(); }
	std::vector<BkSeparator>::const_iterator SeparatorsEnd(void) const
			{ return m_separators.end(); }

	void	rollUpDifferences(void) throw(BkException);
	void	sortContents(void);

private:

	void	rollUpDifferencesPvt(void) throw(BkException);

	std::vector<BkBookmark>	m_bookmarks;
	std::vector<BkFolder>	m_folders;
	std::vector<BkSeparator> m_separators;
};

typedef std::vector<BkFolder>	FolderLst;
typedef std::vector<BkBookmark>	BookmarkLst;
typedef std::vector<BkSeparator> SeparatorLst;
typedef std::vector<BRWSNUM>	BrowserOrdLst;

#endif // end of BKNODE_H
