/***************************************************************************
                 iexplore.cpp  -  IExplore Class Implementation
                             -------------------
    begin                : Sun Sep 15 2002
    copyright            : (C) 2002 by Ken Schenke
    email                : kschenke at users dot sourceforge dot net
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU      *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the Free Software           *
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA         *
 *   02110-1301, USA                                                       *
 *                                                                         *
 ***************************************************************************/

#include "iexplore.h"
#include "browserlist.h"

#include <QFileDialog>
#include <QTextStream>

#ifdef Q_WS_WIN
#include <shlobj.h>
#define strncasecmp strnicmp
#endif

using std::ofstream;

#define WIN32_MAX_PATH	   232	// maximum length of filenames in Windows

/***************************************************************************
 *                                                                         *
 *   Function Prototypes                                                   *
 *                                                                         *
 ***************************************************************************/

static	void	deleteExtraneousUrlFiles(const QString &path, const BkFolder &folder)
					throw(BkException);
static	void	PurgeDir(const QString &path) throw(BkException);
static	void	SaveBookmark(const QString & path, const BkBookmark & bookmark) throw(BkException);

/***************************************************************************
 *                                                                         *
 *   IExplore::classFactory()                                              *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      BrowserBk *                                                        *
 *   Description:                                                          *
 *      This function is called to allocate an instance of the IExplore    *
 *      class.  It is a static function.                                   *
 *                                                                         *
 ***************************************************************************/

BrowserBk *IExplore::classFactory(void)
{
	return new IExplore;
}

/***************************************************************************
 *                                                                         *
 *   PurgeDir()                                                            *
 *                                                                         *
 *   Parameters:                                                           *
 *      const QString &path                                                *
 *   Return:                                                               *
 *      None.  BkException thrown if error occurs                          *
 *   Description:                                                          *
 *      This function is called to delete a directory and all files and    *
 *      sub-directories.                                                   *
 *                                                                         *
 ***************************************************************************/

static void PurgeDir(const QString &path) throw(BkException)
{
	QDir dir(path);
	const QFileInfoList list(dir.entryInfoList());
	QFileInfoList::const_iterator fi = list.begin();

	// walk through the entries in this directory
	
	while(fi != list.end())
	{
		if(fi->isDir())
		{
			// skip the special directory entries representing
			// current and parent directory

			if( fi->fileName() == "."
			 || fi->fileName() == "..")
			{
				++fi;
				continue;
			}

			// this is a sub-directory.  delete it.
			
			QString subdir;
			subdir = path;
			subdir += "/";
			subdir += fi->fileName();
			PurgeDir(subdir);
			if(dir.rmdir(fi->fileName()) == false)
			{
				QString msg;

				msg = "Unable to Delete Directory \"";
				msg += path;
				msg += "/";
				msg += fi->fileName();
				msg += "\"";
				BKEXCEPT(msg);
			}
		}
		else if(dir.remove(fi->fileName()) == false)
		{
			QString msg;

			msg = "Unable to Delete Bookmark File \"";
			msg += path;
			msg += "/";
			msg += fi->fileName();
			msg += "\"";
			BKEXCEPT(msg);
		}

		++fi;
	}
}

/***************************************************************************
 *                                                                         *
 *   IExplore::AreBookmarksValid()                                         *
 *                                                                         *
 *   Parameters:                                                           *
 *      const QString &bookmarks                                           *
 *   Return:                                                               *
 *      true if the supplied location appears to be a Favorites folder.    *
 *   Description:                                                          *
 *      This function is called to verify the bookmark database location   *
 *      appears to be an Internet Explorer Favorites folder.               *
 *                                                                         *
 ***************************************************************************/

bool IExplore::AreBookmarksValid(const QString &bookmarks)
{
	// We want to do a real quick check of the favorites folder
	// given in the first parameter without scanning potentially
	// hundreds of files and directories.  We will require a .URL
	// file to be present in this folder or one of its child folders.

	return checkFolderForUrlFiles(bookmarks, true);
}

/***************************************************************************
 *                                                                         *
 *   IExplore::BrowseForBookmarks()                                        *
 *                                                                         *
 *   Parameters:                                                           *
 *      const BridgeCfg &cfg (WIN32 Only)                                  *
 *      QString &bookmarks                                                 *
 *      QWidget *parent                                                    *
 *   Return:                                                               *
 *      true if there was an error or the user hit Cancel                  *
 *   Description:                                                          *
 *      This function is called to present a directory browsing dialog box *
 *      to the user, allowing them to select their Internet Explorer       *
 *      Favorites folder.                                                  *
 *                                                                         *
 ***************************************************************************/

#if defined(Q_WS_WIN)
bool IExplore::BrowseForBookmarks(const BridgeCfg &cfg, QString &bookmarks, QWidget *parent)
{
	if(bookmarks==QString::null || bookmarks=="")
	{
		QString	favorites;
		QStringList	pathlist;

		// see if we can make a suggestion to the user.

		if(	DetectBrowser(cfg, pathlist) == false
		 ||	pathlist.count() < 1)
			return true;
		bookmarks = pathlist[0];
	}

	// present a dialog box to the user

	bookmarks =
		QFileDialog::getExistingDirectory(parent, "Select Your Favorites "
										  "Folder", bookmarks);

	return (bookmarks == QString::null);
}
#elif defined(Q_WS_X11)
bool IExplore::BrowseForBookmarks(const BridgeCfg &, QString &bookmarks, QWidget *parent)
{
	if(bookmarks==QString::null || bookmarks=="")
		bookmarks = QDir::homePath();

	// present a dialog box to the user

	bookmarks = QFileDialog::getExistingDirectory(parent, "Select Your Favorites "
		"Folder", bookmarks);

	return (bookmarks == QString::null);
}
#else
#error "Must Define platform-dependent IExplore::BrowseForBookmarks() Function"
#endif

/***************************************************************************
 *                                                                         *
 *   IExplore::checkFolderForUrlFiles()                                    *
 *                                                                         *
 *   Parameters:                                                           *
 *      QString &folder                                                    *
 *      bool recurse                                                       *
 *   Return:                                                               *
 *      true if any URL files were found                                   *
 *   Description:                                                          *
 *      This function is called to scan the folder (and possibly one more  *
 *      sub-directory deep for URL files.  It is used by the               *
 *      AreBookmarksValid() function in an attempt to verify the user      *
 *      picked a valid Favorites folder.                                   *
 *                                                                         *
 ***************************************************************************/

bool IExplore::checkFolderForUrlFiles(const QString &folder, bool recurse/*=false*/)
{
	QDir        favorites(folder);

	const QFileInfoList list(favorites.entryInfoList());
	QFileInfoList::const_iterator fi = list.begin();

	while(fi != list.end())
	{
		if(fi->isDir())
		{
			// skip the special directory entries representing
			// current and parent directory

			if( fi->fileName() == "."
			 || fi->fileName() == ".."
			 || recurse == false)
			{
				++fi;
				continue;
			}

			if(checkFolderForUrlFiles(folder+"/"+fi->fileName()))
				return true;
		}

		if(fi->fileName().toUpper().right(4) == ".URL")
			return true;

		++fi;
	}

	return false;
}

/***************************************************************************
 *                                                                         *
 *   IExplore::DetectBrowser()                                             *
 *                                                                         *
 *   Parameters:                                                           *
 *      const BridgeCfg &cfg                                               *
 *      QStringList &path                                                  *
 *   Return:                                                               *
 *      true if any the function was able to detect Internet Explorer      *
 *   Description:                                                          *
 *      This function is called in an attempt to detect if Internet        *
 *      Explorer is installed, and if so, where the Favorites folder is    *
 *      located.
 *                                                                         *
 ***************************************************************************/

#ifdef Q_WS_WIN
bool IExplore::DetectBrowser(const BridgeCfg &cfg, QStringList &paths)
{
	paths.clear();

	if(cfg.m_FavoritesDir.length())
		paths.append(cfg.m_FavoritesDir);

	return true;
}
#endif

/***************************************************************************
 *                                                                         *
 *   IExplore::readBookmarks()                                             *
 *                                                                         *
 *   Parameters:                                                           *
 *      const QString &path                                                *
 *      BRWSNUM browserOrd                                                 *
 *   Return:                                                               *
 *      None.  BkException thrown if error occurs                          *
 *   Description:                                                          *
 *      This function starts the process of reading Microsoft Internet     *
 *      Explorer bookmarks.  All it does is call readFolder().             *
 *                                                                         *
 ***************************************************************************/

void IExplore::readBookmarks(const QString &path, BRWSNUM browserOrd) throw(BkException)
{
	// clear the error message

	if((m_Root = new BkFolder) == NULL)
		BKEXCEPT("Unable to Allocate Memory");

	readFolder(path, *m_Root, browserOrd);
}

/***************************************************************************
 *                                                                         *
 *   IExplore::readFolder()                                                *
 *                                                                         *
 *   Parameters:                                                           *
 *      const QString &path                                                *
 *      BkFolder &folder                                                   *
 *      BRWSNUM browserOrd                                                 *
 *   Return:                                                               *
 *      None.  BkException thrown if error occurs                          *
 *   Description:                                                          *
 *      This function does the actual work of reading a bookmark folder,   *
 *      specified in the first parameter into the BkFolder tree, specified *
 *      in the second parameter.  If it encounters a sub-directory in the  *
 *      filesystem, it allocates a BkFolder class instance to server as    *
 *      sub-tree and calls itself for the child path.                      *
 *                                                                         *
 ***************************************************************************/

void IExplore::readFolder(const QString &path, BkFolder &folder, BRWSNUM browserOrd)
				throw(BkException)
{
	QDir		qdir(path);

	const QFileInfoList list(qdir.entryInfoList());
	QFileInfoList::const_iterator fi = list.begin();

	while(fi != list.end())
	{
		if(fi->isDir())
		{
			// skip the special directory entries representing
			// current and parent directory

			if(	fi->fileName() == "."
			 ||	fi->fileName() == "..")
			{
				++fi;
				continue;
			}

			// This is a sub-directory.  Create a child folder.

			BkFolder tmpF;
			BkFolder & child = folder.addFolder(tmpF);

			QString pathname = fi->fileName();
			child.setTitle(fi->fileName());
			child.setValidField(BKVALID_TITLE);

			// this folder belongs to IE

			child.setBrowserFound(browserOrd);
			child.setBrowserSaved(browserOrd);

			// Read the sub-directory.

			readFolder(fi->filePath(), child, browserOrd);
		}
		else
		{
			// does this file end in .URL?

			if(!fi->fileName().endsWith(".URL", Qt::CaseInsensitive))
			{
				++fi;
				continue;	// skip it
			}

			// read the url file

			BkBookmark	bookmark;
			BkBookmark	& child = folder.addBookmark(bookmark);

			// this bookmark belongs to IE

			child.setBrowserFound(browserOrd);
			child.setBrowserSaved(browserOrd);

			readUrlFile(*fi, child);
		}

		++fi;
	}
}

/***************************************************************************
 *                                                                         *
 *   IExplore::readUrlFile()                                               *
 *                                                                         *
 *   Parameters:                                                           *
 *      QFileInfo *fi                                                      *
 *      BkBookmark *bk                                                     *
 *   Return:                                                               *
 *      None.  BkException thrown if error occurs                          *
 *   Description:                                                          *
 *      This function reads a .URL file, specified in the first parameter  *
 *      into the BkBookmark node, specified in the second parameter.       *
 *                                                                         *
 ***************************************************************************/

void IExplore::readUrlFile(const QFileInfo& fi, BkBookmark &bk) throw(BkException)
{
	bool	InternetSection = false;
	QString	buffer;

	// set the bookmark's title (the filename)

	QString baseName = fi.fileName();
	if(baseName.endsWith(".URL", Qt::CaseInsensitive))
		baseName = baseName.left(baseName.length()-4);
	bk.setTitle(baseName);
	bk.setValidField(BKVALID_TITLE);

	// open the file

	QFile	fp;
	fp.setFileName(fi.filePath());
	if(fp.open(QIODevice::ReadOnly | QIODevice::Text) == false)
	{
		QString msg;

		msg = "Unable to Open Bookmark File \"";
		msg += fi.filePath();
		msg += "\"";
		BKEXCEPT(msg);
	}

	QTextStream stream(&fp);

	stream.setCodec("UTF-8");

	// loop through the lines in the file

	while(!stream.atEnd())
	{
		// read one line from the file

		buffer = stream.readLine().simplified();

		// skip blank lines

		if(buffer.length() < 1)
			continue;

		// see if we can recognize the line

		if(buffer.at(0) == '[')
		{
			if(buffer.toUpper() == "[INTERNETSHORTCUT]")
				InternetSection = true;
			else
				InternetSection = false;
		}
		else if(InternetSection &&
				buffer.startsWith("URL=", Qt::CaseInsensitive))
		{
			bk.setUrl(buffer.right(buffer.length()-4));
			bk.setValidField(BKVALID_URL);
		}
	}
}

/***************************************************************************
 *                                                                         *
 *   saveBookmark()                                                        *
 *                                                                         *
 *   Parameters:                                                           *
 *      const QString &path                                                *
 *      const BkBookmark &bookmark                                         *
 *   Return:                                                               *
 *      None.  BkException thrown if error occurs                          *
 *   Description:                                                          *
 *      This function saves the bookmark, specified in the second          *
 *      parameter as a URL file in the folder specified in the first       *
 *      parameter.
 *                                                                         *
 ***************************************************************************/

static void SaveBookmark(const QString & path, const BkBookmark & bookmark) throw(BkException)
{
	QFile	out;
	QString	filename(path);

	filename += "/";
	filename += bookmark.title();
	filename += ".url";

	if(filename.length() >= WIN32_MAX_PATH)
	{
		// truncate the name of the bookmark so that it's a valid Windows filename

		filename.replace(WIN32_MAX_PATH-4, filename.length()-WIN32_MAX_PATH, "");
	}

	out.setFileName(filename);

	if(out.open(QIODevice::WriteOnly | QIODevice::Text) == false)
	{
		QString	msg;

		msg = "Unable to Open Bookmark File for Writing\n";
		msg += filename;
		BKEXCEPT(msg);
	}
	QTextStream t(&out);

	t.setCodec("UTF-8");

	t
		<< "[InternetShortcut]"
		<< endl;
	t
		<< "URL="
		<< bookmark.url()
		<< endl;
}

/***************************************************************************
 *                                                                         *
 *   IExplore::saveBookmarksX()                                            *
 *                                                                         *
 *   Parameters:                                                           *
 *      const QString &path                                                *
 *      BkFolder &folder                                                   *
 *      BRWSNUM browserOrd                                                 *
 *   Return:                                                               *
 *      None.  BkException thrown if error occurs                          *
 *   Description:                                                          *
 *      This function is called by the IExplore::saveBookmarks() function, *
 *      overridden from the base class, BkBookmark.  This function does    *
 *      the real work of saving the bookmark changes back to Internet      *
 *      Explorer's bookmark .URL files.                                    *
 *                                                                         *
 ***************************************************************************/

void IExplore::saveBookmarksX(
	const QString &pathParm, BkFolder &folder, BRWSNUM browserOrd)
	throw(BkException)
{
	// the path passed as a parameter might have a trailing slash.
	// if so, we need to trim it off.

	QString temp(pathParm), path;
	if(temp.at(temp.length()-1) == '/')
		path = temp.left(temp.length()-1);
	else
		path = temp;

	std::vector<SortedNodeList> nodes;
	std::vector<SortedNodeList>::iterator it;

	SortNodes(folder, nodes, browserOrd);

	for(it=nodes.begin(); it!=nodes.end(); ++it)
	{
		if(it->m_type == NODETYPE_FOLDER)
		{
			QString	pathName;
			pathName = path;
			pathName += "/";
			pathName += it->m_fit->title();

			// has the folder been deleted?

			if(it->m_fit->isDeleted())
			{
				if(it->m_fit->browserFound(browserOrd))
				{
					PurgeDir(pathName);
					QDir dir(path);
					if(dir.rmdir(it->m_fit->title()) == false)
					{
						QString msg;

						msg = "Unabe to Delete \"";
						msg += path;
						msg += "/";
						msg += it->m_fit->title();
						msg += "\"";
						BKEXCEPT(msg);
					}
				}
				continue;
			}

			// has the user requested to ignore this folder?

			if(it->m_fit->ignore() && !it->m_fit->browserFound(browserOrd))
				continue;

			// is this a new folder for IE?

			if(it->m_fit->browserSaved(browserOrd) == false)
			{
				QDir dir(path);

				if( dir.exists(it->m_fit->title()) == false
				 && dir.mkdir(it->m_fit->title()) == false)
				{
					QString msg;

					msg = "Unable to Create \"";
					msg += path;
					msg += "/";
					msg += it->m_fit->title();
					msg += "\"";
					BKEXCEPT(msg);
				}
				it->m_fit->setBrowserFound(browserOrd);
				it->m_fit->setBrowserSaved(browserOrd);
			}

			// walk through the nodes in the sub-folder

			saveBookmarksX(pathName, *(it->m_fit), browserOrd);
		}
		else if(it->m_type == NODETYPE_BOOKMARK)
		{
			// has the bookmark been deleted?

			if(it->m_bit->isDeleted())
				continue;

			// has the user requested to ignore this bookmark?

			if(it->m_bit->ignore() && !it->m_bit->browserFound(browserOrd))
				continue;

			// Check to see if the bookmark is different from IE's copy,
			// or if IE doesn't have the bookmark at all?

			if(it->m_bit->browserSaved(browserOrd) == false)
			{
				SaveBookmark(path, *(it->m_bit));
				it->m_bit->setBrowserFound(browserOrd);
				it->m_bit->setBrowserSaved(browserOrd);
			}
		}
		else if(it->m_type == NODETYPE_SEPARATOR)
 			continue;	// Internet Explorer doesn't support separators
		else
			BKEXCEPT("Internal Error: Unrecognized Bookmark Type");
	}

	// Since IE's URL files are named for the bookmark's title,
	// if a bookmark's title was changed, we would have saved
	// the bookmark under a different URL file, thereby orphaning
	// the original URL file.  We need to delete these old URL files.

	deleteExtraneousUrlFiles(path, folder);
}

/***************************************************************************
 *                                                                         *
 *   deleteExtraneousUrlFiles()                                            *
 *                                                                         *
 *   Parameters:                                                           *
 *      const QString &path                                                *
 *      const BkFolder &folder                                             *
 *   Return:                                                               *
 *      None.  BkException thrown if error occurs                          *
 *   Description:                                                          *
 *      This function scans the bookmark folder given, both as a           *
 *      filesystem folder in the first parameter, and as a bookmark tree   *
 *      folder in the second folder, looking for URL files which are no    *
 *      longer referenced by the bookmark tree.  This can happen if a      *
 *      bookmark's title was changed.                                      *
 *                                                                         *
 ***************************************************************************/

static void deleteExtraneousUrlFiles(const QString &path, const BkFolder &folder)
			throw(BkException)
{
	BookmarkLst::const_iterator	bit;
	QString						filename;

	// scan the folder looking for files ending in .url
	
	QDir dir(path, "*.url", QDir::Name|QDir::IgnoreCase, QDir::Files);
	for(uint i=0; i<dir.count(); i++)
	{
		// figure out the filename without the ending .url

		if(!dir[i].endsWith(".URL", Qt::CaseInsensitive))
			continue;	// this should never occur because of the filter
		filename = dir[i].left(dir[i].length()-4);

		// loop through the bookmarks looking for match on the title

		for(	bit = folder.BookmarksBegin();
				bit != folder.BookmarksEnd();
				++bit)
		{
			if( bit->title().toUpper() == filename.toUpper()
			 && !bit->isDeleted())
				break;	// we have a match, stop searching
		}

		// if we found a title match, skip this one
		
		if(bit != folder.BookmarksEnd())
			continue;

		// there was no bookmark title matching this URL file
		// delete it
		
		if(dir.remove(dir[i]) == false)
		{
			QString	msg;

			msg = "Unable to Delete Bookmark File \"";
			msg += path;
			msg += "/";
			msg += dir[i];
			msg += "\"";
			BKEXCEPT(msg);
		}
	}
}
