#! /bin/sh
# initial network setup for s390 installation
#
# written by Gerhard Tonn and Stefan Gybas
# maintained by the boot-floppies team, debian-boot@lists.debian.org
# copyright (c) 2001
# This is free software under the GNU General Public License.

. /usr/share/netsetup/functions.sh
. /usr/share/netsetup/chandev.sh

initial_setup() {
  echo "127.0.0.1	localhost" > /etc/hosts
  echo "order hosts,bind" > /etc/host.conf
  echo "multi on" >> /etc/host.conf
  hostname localhost
  ifconfig lo 127.0.0.1 netmask 255.0.0.0
}


load_kernel_parms() {
  if [ -r /proc/cmdline ]; then
    for parm in `cat /proc/cmdline`; do
      parm=`echo $parm | sed -e 's/^ *//' | tr [A-Z] [a-z]`
      name=`echo $parm | cut -f1 -d=`
      value=`echo $parm | cut -f2 -d=`
      case $name in
        hostname)
	  ip_host=$value
	  ;;
        ip)
	  ip_addr=$value
	  ;;
        netmask)
	  ip_netmask=$value
	  ;;
        gateway)
	  ip_gateway=$value
	  ;;
        peer)
	  ip_peer=$value
	  ;;
        dns)
	  ip_dns=$value
	  ;;
	netinterface)
	  network_interface=$value
	  ;;
        netmodule)
	  network_module=$value
	  auto_module_setup=1
	  echo ""
	  echo "Performing automatic module setup!"
	  echo ""
	  ;;
        chandev)
	  chandev_parm=$value
	  ;;
      esac
    done
    if [ ! -z "$ip_host" -a ! -z "$ip_addr" ]; then
      if [ ! -z "$ip_netmask" -o ! -z "$ip_peer" ]; then
	echo ""
	echo "Performing automatic network setup!"
	echo ""
        auto_network_setup=1
      fi
    fi
  fi
}


symlink_extra_modules() {
  local mountpoint=$1
  local kver=`uname -r`
  rm -rf /lib/modules/$kver
  ln -s /lib/modules /lib/modules/$kver
  mount /dev/rd/1 $mountpoint 1>/dev/null 2>&1
  if [ $? -eq 0 ]; then
    if [ -d "$mountpoint/lib/modules/$kver" ]; then
      echo -n "Symlinking extra modules from second ram disk: "
      local pwd=`pwd`
      cd $mountpoint/lib/modules/$kver/kernel
      for module in `find . -name *.o`; do
        ln -s $mountpoint/lib/modules/$kver/kernel/$module /lib/modules/$module
        echo -n "${module##*/} "
      done
      cd $pwd
      echo ""
    else
      echo "No modules found that match kernel version $kver"
    fi
  else
    if [ ! -d "$mountpoint/lib/modules/$kver" ]; then
      echo "No second ram disk found. Additional modules are not avaliable!"
    fi
  fi
}


load_network_module() {
  if [ -z "$network_module_list" ]; then
    echo "No network modules have been found on this ram disk but you can still use it"
    echo "as a rescue system."
    exit 1
  fi

  while :; do
    if [ "$auto_module_setup" != "1" ]; then
      auto_setup=0
      echo ""
      echo "Select the type of your network device"
      echo "0) no network configuration (to use this as a rescue system)"
     
      local i=1
      for index in $network_module_list; do
        if [ -f /usr/share/netsetup/$index.sh ]; then
          echo -n "$i)"
          grep DESC: /usr/share/netsetup/$index.sh | cut -d: -f2-
        else
          echo "$i) $index"
        fi
        i=`expr $i + 1`
      done
      i=`expr $i - 1`
      readln "Enter your choice (0-$i): "
      echo ""
      if [ "$ans" = "0" ]; then
        exit 0
      fi

      element_from_list $ans $network_module_list
      if [ $? -ne 0 ]; then
        echo "Your answer is not valid. Please enter again!"
        continue;
      fi
      network_module=$element 
      if [ -f /usr/share/netsetup/$network_module.sh ]; then
        . /usr/share/netsetup/$network_module.sh
      else
        in_list $network_module $chandev_module_list
        if [ $? -eq 0 ]; then
          echo "Please enter the channel device general configuration options"
          echo "e.g. noauto or add_parms,0x10,0x0000,0xffff,portname:PORTNAME"
          echo "See the Device Drivers and Installation Commands red book for details."
          readln_mixed "chandev options: " $chandev_parm
          chandev_parm=$ans
          echo ""
          echo "Please enter channel device configuration options for $network_module"
          echo "e.g. qeth0,0xfd00,0xfd01,0xfd02,0,1 or lcs0,0x1234,0x1235"
          readln_mixed "$network_module options: " $chandev_module_parm
          chandev_module_parm=$ans
          module_parm=""
        else
          echo "Please enter the parameter(s) for the $network_module module"
          echo "e.g. iucv=router:linux2 for the netiucv module"
          echo "See the Device Drivers and Installation Commands red book for details."
          readln_mixed "$network_module parameter(s): " $module_parm
          chandev_parm=""
          chandev_module_parm=""
          module_parm=$ans
        fi
      fi
    fi

    echo ""
    echo "Module configuration will be:"
    echo "Module name              : $network_module"
    in_list $network_module $chandev_module_list
    if [ $? -eq 0 ]; then
      echo "Generic chandev parameter: $chandev_parm"
      echo "Chandev module parameter : $chandev_module_parm"
    else
      echo "Module parameter         : $module_parm"
    fi
    if [ "$auto_module_setup" != "1" ]; then
      yes_no "Is this correct (Yes/No) ? "
      if [ "$ans" != "yes" ]; then
        return 1
      fi
    fi

    # setup chandev configuration
    if [ ! -z "$chandev_parm" ]; then
      echo "$chandev_parm" > /proc/chandev
    fi
    if [ ! -z "$chandev_module_parm" ]; then
      echo "$chandev_module_parm" > /proc/chandev
    fi

    # try to load the module, unload first if it's already loaded
    echo ""
    depmod -a
    grep -q ^$network_module /proc/modules
    if [ $? -eq 0 ]; then
      rmmod -s $network_module
    fi
    modprobe $network_module $module_parm
    echo ""
    if [ $? -eq 0 ]; then
      # get a list of all network interfaces without the loopback interface
      interfaces_list=`grep : /proc/net/dev | cut -d: -f1 | sed -e "s/ //g" | grep -v ^lo$`
      if [ -z "$interfaces_list" ]; then
        echo "The $network_module has been loaded but no network interface was defined."
        echo "Please retry with different settings or try a manual setup!"
	rmmod $network_module
	auto_module_setup=0
	# TODO: undo chandev settings - how can we do this?
      else
	select_network_interface
	return 0
      fi
    else
      echo "Could not load the $network_module module. Please try again with"
      echo "different settings!"
      auto_module_setup=0
    fi
  done
}


select_network_interface() {
  list_length $interfaces_list
  local if_count=$?

  if [ $if_count -eq 1 ]; then
    network_interface=$interfaces_list
  else
    if [ "$auto_module_setup" = "1" -a ! -z "$network_interface" ]; then
      in_list $network_interface $interfaces_list
      if [ $? -eq 1 ]; then
        echo "Pre-selected interface $network_interface was not found!"
        echo ""
	network_interface=""
      fi
    else
      network_interface=""
    fi
    if [ -z "$network_interface" ]; then
      echo "More than one network interface was found. Which one do you want to configure?"
      local i=1
      for index in $interfaces_list; do
        case $index in
          eth*)
            echo "$i) $index (Ethernet)"
            ;;
          hsi*)
            echo "$i) $index (HiperSockets)"
            ;;
          tr*)
            echo "$i) $index (Token ring)"
            ;;
          iucv*)
            if [ -r /proc/net/iucv/$index/username ]; then
              echo "$i) IUCV connecton to `cat /proc/net/iucv/$index/username`"
            else
              echo "$i) $index"
            fi
            ;;
          *)
            echo "$i) $index"
  	  ;;
        esac
        i=`expr $i + 1`
      done
      while :; do
        readln "Enter your choice (1-$if_count): "
        echo ""
        element_from_list $ans $interfaces_list
        if [ $? -ne 0 ]; then
          echo "Your answer is not valid. Please enter again!"
          continue;
        fi
        break
      done
      network_interface=$element  
    fi
  fi

  echo "Configuring network interface $network_interface..."
  # make chandev and module params persistent
  mkdir -p /etc/modutils
  if [ "$chandev_parm" != "" ]; then
    echo "$chandev_parm" > /etc/modutils/0chandev.chandev
  fi
  if [ "$chandev_module_parm" != "" ]; then
    echo "$chandev_module_parm" > /etc/modutils/$network_module.chandev
  fi
  echo "alias $network_interface $network_module" > /etc/modutils/$network_module
  if [ "$module_parm" != "" ]; then
    echo "options $network_module $module_parm" >> /etc/modutils/$network_module
  fi
  /sbin/update-modules force
}


setup_network_interface() {
  ifconfig $network_interface | grep -q POINTOPOINT
  pointopoint=$?

  while :; do
    auto_setup=$auto_network_setup
    [ -z "$ip_search" ] || ip_host="$ip_host.$ip_search"
    readln "Please enter your host name with its full domain: " $ip_host
    ip_host=`echo $ans | cut -f1 -d.`
    ip_search=`echo $ans | cut -f2- -d.`
    if [ "$ip_host" = "$ip_search" ]; then
      ip_search=""
    fi

    if [ $pointopoint -ne 0 ]; then
      readln "Please enter your IP address: " $ip_addr
#      readln "Please enter your IP address or dhcp: " $ip_addr
    else
      readln "Please enter your IP address: " $ip_addr
    fi
    ip_addr=$ans
    if [ "$ip_addr" = "dhcp" ]; then
      # TODO: start DHCP client and check status
      echo "DHCP is not yet supported by this ram disk!"
      return
    fi

    if [ $pointopoint -ne 0 ]; then
      [ -z "$ip_netmask" ] && ip_netmask="255.255.255.0"
      readip4 "Please enter the net mask: " $ip_netmask
      ip_netmask=$ans
      eval `/usr/lib/netsetup/netsetupcalc $ip_addr $ip_netmask`
      ip_broadcast=$BROADCAST
      ip_network=$NETWORK
      [ -z "$ip_gateway" ] && ip_gateway=`echo $ip_network | sed -e "s/\.0$/.1/"`
      readip4 "Please enter the gateway address: " $ip_gateway
      ip_gateway=$ans
    else
      readip4 "Please enter the IP address of your peer: " $ip_peer
      ip_peer=$ans
      ip_netmask="255.255.255.255"
      ip_broadcast=$ans
      ip_network=$ans
      ip_gateway=$ans
    fi

    readip4 "Please enter the IP address of the DNS server: " $ip_dns
    ip_dns=$ans

    echo ""
    echo "Configuration will be:"
    echo "Host name        : $ip_host"
    echo "IP address       : $ip_addr"
    if [ $pointopoint -ne 0 ]; then
      echo "Net mask         : $ip_netmask"
      echo "Gateway address  : $ip_gateway"
    else
      echo "Peer IP address  : $ip_peer"
    fi
    echo "DNS IP address   : $ip_dns"
    echo "DNS search domain: $ip_search"
    if [ "$auto_network_setup" != "1" ]; then
      yes_no "Is this correct (Yes/No) ? "
      echo ""
      if [ "$ans" != "yes" ]; then
        continue
      fi
    fi

    hostname $ip_host
    echo "$ip_host" > /etc/hostname
    if [ $pointopoint -ne 0 ]; then
      ifconfig $network_interface $ip_addr netmask $ip_netmask broadcast $ip_broadcast
    else
      ifconfig $network_interface $ip_addr netmask $ip_netmask pointopoint $ip_peer
    fi
    if [ ! -z "$ip_gateway" ]; then
      route add default gw $ip_gateway
    fi
    if [ ! -z "$ip_dns" ]; then
      echo nameserver $ip_dns > /etc/resolv.conf
      if [ ! -z "$ip_search" ]; then
        echo search $ip_search >> /etc/resolv.conf
	echo "$ip_addr		$ip_host.$ip_search	$ip_host" >> /etc/hosts
      else
	echo "$ip_addr		$ip_host" >> /etc/hosts
      fi
      # save resolv.conf for possible restoration during dbootstrap restart
      cp -a /etc/resolv.conf /tmp/resolv.conf 2>/dev/null
    fi
    echo "localnet $ip_network" > /etc/networks
    echo ""

    echo "Waiting 3 seconds for the interface to come up..."
    sleep 3

    if [ ! -z "$ip_gateway" ]; then
      echo "Trying to ping your gateway..."
      ping -c 3 $ip_gateway
      if [ $? -ne 0 ]; then
        echo ""
	echo "Could not contact your gateway!"
        if [ "$auto_network_setup" != "1" ]; then
          yes_no "Do you want to use different network settings (Yes/No) ? "
          if [ "$ans" = "yes" ]; then
            continue
          fi
        fi
      fi
    fi

    if [ ! -z "$ip_dns" ]; then
      echo "Trying to ping your DNS server..."
      ping -c 3 $ip_dns
      if [ $? -ne 0 ]; then
        echo ""
	echo "Could not ping your DNS server!"
        if [ "$auto_network_setup" != "1" ]; then
          yes_no "Do you want to use different network settings (Yes/No) ? "
          if [ "$ans" = "yes" ]; then
            continue
          fi
        fi
      fi
    fi

    return 0
  done
}


export_netconfig() {
  echo "export ip_host=$ip_host"
  echo "export ip_addr=$ip_addr"
  echo "export ip_netmask=$ip_netmask"
  echo "export ip_network=$ip_network"
  echo "export ip_broadcast=$ip_broadcast"
  echo "export ip_peer=$ip_peer"
  echo "export ip_gateway=$ip_gateway"
  echo "export ip_search=$ip_search"
  echo "export ip_dns=$ip_dns"
  echo "export network_interface=$network_interface"
} 



#
# Main installation
#

echo ""
echo "Welcome to Debian GNU/Linux 3.0 for S/390"
echo ""

initial_setup
echo 3 4 1 7 >/proc/sys/kernel/printk
symlink_extra_modules /floppy
echo 6 4 1 7 >/proc/sys/kernel/printk
build_network_lists
load_kernel_parms

while :; do
  load_network_module
  if [ $? -eq 0 ]; then
    setup_network_interface
    export_netconfig > /tmp/netconfig
    chmod 755 /tmp/netconfig
    echo ""
    break;
  fi
done

# umount the second ram disk - dbootstrap will remount it to copy the foreign
# modules
umount -f /floppy >/dev/null 2>&1

grep -q loop /proc/devices || modprobe loop
[ -e /proc/net/rpc/nfs ] || modprobe nfs

