/***************************************************************************
 *            blank-dialog.c
 *
 *  mar jui 26 12:23:01 2005
 *  Copyright  2005  Philippe Rouquier
 *  brasero-app@wanadoo.fr
 ***************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib.h>
#include <glib/gi18n-lib.h>

#include <gtk/gtklabel.h>
#include <gtk/gtkstock.h>
#include <gtk/gtkbutton.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtkbox.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtkvbox.h>
#include <gtk/gtkwindow.h>
#include <gtk/gtkdialog.h>
#include <gtk/gtkprogressbar.h>
#include <gtk/gtkmessagedialog.h>
#include <gtk/gtkcheckbutton.h>

#include <nautilus-burn-drive-selection.h>
#include <nautilus-burn-recorder.h>

#include "burn.h"
#include "utils.h"
#include "burn-basics.h"
#include "blank-dialog.h"
#include "recorder-selection.h"
#include "progress.h"

extern gint debug;

static void brasero_blank_dialog_class_init (BraseroBlankDialogClass *klass);
static void brasero_blank_dialog_init (BraseroBlankDialog *sp);
static void brasero_blank_dialog_finalize (GObject *object);
static void brasero_blank_dialog_destroy (GtkObject *object);

static gboolean brasero_blank_dialog_delete (GtkWidget *widget,
					     GdkEventAny *event);
static void brasero_blank_dialog_cancel_clicked_cb (GtkWidget *button,
						     BraseroBlankDialog *dialog);
static void brasero_blank_dialog_blank_clicked_cb (GtkButton *button,
						    BraseroBlankDialog *dialog);
static void brasero_blank_dialog_device_changed_cb (NautilusBurnDriveSelection *selection,
						    NautilusBurnMediaType can_record,
						    BraseroBlankDialog *dialog);

static void brasero_blank_dialog_device_opts_setup (BraseroBlankDialog *dialog,
						    NautilusBurnMediaType type,
						    gboolean messages);

struct BraseroBlankDialogPrivate {
	BraseroBurn *burn;
	BraseroBurnCaps *caps;

	GtkWidget *upper_box;
	GtkWidget *lower_box;
	GtkWidget *fast_enabled;
	GtkWidget *dummy_toggle;
	GtkWidget *selector;
	GtkWidget *progress;
	GtkWidget *blank_button;
};

static GObjectClass *parent_class = NULL;

GType
brasero_blank_dialog_get_type ()
{
	static GType type = 0;

	if (type == 0) {
		static const GTypeInfo our_info = {
			sizeof (BraseroBlankDialogClass),
			NULL,
			NULL,
			(GClassInitFunc) brasero_blank_dialog_class_init,
			NULL,
			NULL,
			sizeof (BraseroBlankDialog),
			0,
			(GInstanceInitFunc) brasero_blank_dialog_init,
		};

		type = g_type_register_static (GTK_TYPE_DIALOG,
					       "BraseroBlankDialog",
					       &our_info,
					       0);
	}

	return type;
}

static void
brasero_blank_dialog_class_init (BraseroBlankDialogClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	GtkObjectClass *gtk_object_class = GTK_OBJECT_CLASS (klass);
	GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);
	object_class->finalize = brasero_blank_dialog_finalize;
	gtk_object_class->destroy = brasero_blank_dialog_destroy;
	widget_class->delete_event = brasero_blank_dialog_delete;
}

static void
brasero_blank_dialog_init (BraseroBlankDialog *obj)
{
	GtkWidget *cancel;
	GtkWidget *title;
	GtkWidget *blank;

	obj->priv = g_new0 (BraseroBlankDialogPrivate, 1);
	gtk_window_set_default_size (GTK_WINDOW (obj), 500, 300);

	/* upper part */
	obj->priv->upper_box = gtk_vbox_new (FALSE, 0);
	obj->priv->selector = brasero_recorder_selection_new ();
	g_object_set (G_OBJECT (obj->priv->selector),
		      "show-recorders-only", TRUE,
		      "show-properties", FALSE,
		      "file-image", FALSE,
		      NULL);

	gtk_box_pack_start (GTK_BOX (obj->priv->upper_box),
			    brasero_utils_pack_properties (_("<b>Select a recorder:</b>"),
							   obj->priv->selector,
							   NULL),
			    FALSE, FALSE, 0);

	brasero_recorder_selection_select_default_drive (BRASERO_RECORDER_SELECTION (obj->priv->selector),
							 BRASERO_MEDIA_REWRITABLE);

	obj->priv->fast_enabled = gtk_check_button_new_with_label (_("fast blanking"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (obj->priv->fast_enabled), TRUE);
	obj->priv->dummy_toggle = gtk_check_button_new_with_label (_("simulation"));
	gtk_box_pack_start (GTK_BOX (obj->priv->upper_box),
			    brasero_utils_pack_properties (_("<b>Options:</b>"),
							   obj->priv->fast_enabled,
							   obj->priv->dummy_toggle,
							   NULL), FALSE,
			    FALSE, 0);

	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (obj)->vbox),
			    obj->priv->upper_box,
			    FALSE,
			    FALSE,
			    0);
	gtk_widget_show_all (GTK_WIDGET (obj->priv->upper_box));

	/* lower part */
	obj->priv->lower_box = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (obj->priv->lower_box), 6);
	gtk_widget_set_sensitive (obj->priv->lower_box, FALSE);
	gtk_widget_show (obj->priv->lower_box);

	title = gtk_label_new (_("<b>Progress:</b>"));
	gtk_label_set_use_markup (GTK_LABEL (title), TRUE);
	gtk_misc_set_alignment (GTK_MISC (title), 0.0, 0.5);
	gtk_misc_set_padding(GTK_MISC (title), 0, 6);
	gtk_widget_show (title);
	gtk_box_pack_start (GTK_BOX (obj->priv->lower_box),
			    title,
			    FALSE,
			    FALSE,
			    0);

	obj->priv->progress = brasero_burn_progress_new ();
	gtk_widget_show (obj->priv->progress);
	g_object_set (G_OBJECT (obj->priv->progress),
		      "show-info", FALSE,
		      NULL);

	gtk_box_pack_start (GTK_BOX (obj->priv->lower_box),
			    obj->priv->progress,
			    FALSE,
			    FALSE,
			    0);

	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (obj)->vbox),
			    obj->priv->lower_box,
			    FALSE,
			    FALSE,
			    6);

	/* buttons */
	cancel = gtk_button_new_from_stock (GTK_STOCK_CANCEL);
	gtk_widget_show (cancel);
	g_signal_connect (G_OBJECT (cancel), "clicked",
			  G_CALLBACK (brasero_blank_dialog_cancel_clicked_cb),
			  obj);
	gtk_dialog_add_action_widget (GTK_DIALOG (obj), cancel, GTK_RESPONSE_CANCEL);

	blank = brasero_utils_make_button (_("Blank"), GTK_STOCK_CDROM);
	gtk_widget_show_all (blank);
	g_signal_connect (G_OBJECT (blank), "clicked",
			  G_CALLBACK (brasero_blank_dialog_blank_clicked_cb),
			  obj);
	gtk_dialog_add_action_widget (GTK_DIALOG (obj), blank, GTK_RESPONSE_OK);
	obj->priv->blank_button = blank;

	obj->priv->caps = brasero_burn_caps_get_default ();

	g_signal_connect (G_OBJECT (obj->priv->selector),
			  "media-changed",
			  G_CALLBACK (brasero_blank_dialog_device_changed_cb),
			  obj);
}

static void
brasero_blank_dialog_finalize (GObject *object)
{
	BraseroBlankDialog *cobj;

	cobj = BRASERO_BLANK_DIALOG (object);

	g_object_unref (cobj->priv->caps);

	g_free (cobj->priv);
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
brasero_blank_dialog_destroy (GtkObject *object)
{
	BraseroBlankDialog *cobj;

	cobj = BRASERO_BLANK_DIALOG (object);

	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		GTK_OBJECT_CLASS (parent_class)->destroy (object);
}

GtkWidget *
brasero_blank_dialog_new ()
{
	BraseroBlankDialog *obj;
	NautilusBurnMediaType type;
	NautilusBurnDrive *drive = NULL;

	obj = BRASERO_BLANK_DIALOG (g_object_new (BRASERO_TYPE_BLANK_DIALOG,
				    "title", _("Disc blanking"), NULL));

	brasero_recorder_selection_get_drive (BRASERO_RECORDER_SELECTION (obj->priv->selector),
					      &drive,
					      NULL);
	type = nautilus_burn_drive_get_media_type (drive);
	nautilus_burn_drive_unref (drive);
	brasero_blank_dialog_device_opts_setup (obj, type, FALSE);

	return GTK_WIDGET (obj);
}

static void
brasero_blank_dialog_device_opts_setup (BraseroBlankDialog *dialog,
					NautilusBurnMediaType type,
					gboolean messages)
{
	BraseroBurnResult result;
	gboolean fast_enabled = FALSE;
	gboolean fast_supported = FALSE;
	BraseroBurnFlag flags = BRASERO_BURN_FLAG_NONE;

	/* set the options */
	brasero_burn_caps_blanking_get_default_flags (dialog->priv->caps,
						      type,
						      &flags,
						      &fast_enabled);

	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog->priv->dummy_toggle),
				      (flags & BRASERO_BURN_FLAG_DUMMY));

	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog->priv->fast_enabled),
				      fast_enabled);

	result = brasero_burn_caps_blanking_get_supported_flags (dialog->priv->caps,
								 type,
								 &flags,
								 &fast_supported);

	gtk_widget_set_sensitive (dialog->priv->blank_button,
				  (result == BRASERO_BURN_OK));

	gtk_widget_set_sensitive (dialog->priv->dummy_toggle,
				  (flags & BRASERO_BURN_FLAG_DUMMY));

	gtk_widget_set_sensitive (dialog->priv->fast_enabled, fast_supported);

	if (result == BRASERO_BURN_NOT_SUPPORTED) {
		GtkWidget *message;
		gchar *string;

		if (!messages)
			return;

		/* FIXME: we don't want to break strings for 0.4.0 */
		string = g_strdup_printf ("<big><b>%s</b></big>", _("This type of disc can't or doesn't require to be blanked."));
			
		/* we don't need / can't blank(ing) so tell the user */
		message = gtk_message_dialog_new_with_markup (GTK_WINDOW (dialog),
							      GTK_DIALOG_MODAL |
							      GTK_DIALOG_DESTROY_WITH_PARENT,
							      GTK_MESSAGE_INFO,
							      GTK_BUTTONS_CLOSE,
							      string);
		g_free (string);

		gtk_window_set_title (GTK_WINDOW (message), _("Unneeded operation"));

		gtk_dialog_run (GTK_DIALOG (message));
		gtk_widget_destroy (message);
	}
	else if (result == BRASERO_BURN_ERR)
		return;

	if (!messages)
		return;

	/* FIXME: do we really need this following messages ? */
	if (type == NAUTILUS_BURN_MEDIA_TYPE_DVDRW) {
		if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->priv->dummy_toggle))) {
			GtkWidget *message;

			/* simulation doesn't work with DVDs */
			message = gtk_message_dialog_new (GTK_WINDOW (dialog),
							  GTK_DIALOG_MODAL |
							  GTK_DIALOG_DESTROY_WITH_PARENT,
							  GTK_MESSAGE_INFO,
							  GTK_BUTTONS_CLOSE,
							  _("DVDRW discs don't support simulated blanking."));

			gtk_window_set_title (GTK_WINDOW (message), _("Unneeded operation"));

			gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (message),
								  _("You can nevertheless blank it."));
			gtk_dialog_run (GTK_DIALOG (message));
			gtk_widget_destroy (message);
		}
	}
	else if (type == NAUTILUS_BURN_MEDIA_TYPE_DVD_PLUS_RW) {
		if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->priv->fast_enabled))) {
			GtkWidget *message;

			/* simulation doesn't work with DVDs */
			/* Tell the user fast blanking is not required with this kind of drive */
			message = gtk_message_dialog_new (GTK_WINDOW (dialog),
							  GTK_DIALOG_MODAL |
							  GTK_DIALOG_DESTROY_WITH_PARENT,
							  GTK_MESSAGE_INFO,
							  GTK_BUTTONS_CLOSE,
							  _("You can use this type of DVD without prior blanking.\n"
							    "NOTE: it doesn't support simulation."));

			gtk_window_set_title (GTK_WINDOW (message), _("Unneeded operation"));

			gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (message),
								  _("You can nevertheless blank it with the slow option if you want to."));
			gtk_dialog_run (GTK_DIALOG (message));
			gtk_widget_destroy (message);
		}
	}
}

static void
brasero_blank_dialog_device_changed_cb (NautilusBurnDriveSelection *selection,
					NautilusBurnMediaType media_type,
					BraseroBlankDialog *dialog)
{
	brasero_blank_dialog_device_opts_setup (dialog, media_type, TRUE);
}

static gboolean
brasero_blank_dialog_cancel_dialog (GtkWidget *toplevel)
{
	gint result;
	GtkWidget *button;
	GtkWidget *message;

	message = gtk_message_dialog_new (GTK_WINDOW (toplevel),
					  GTK_DIALOG_DESTROY_WITH_PARENT|
					  GTK_DIALOG_MODAL,
					  GTK_MESSAGE_QUESTION,
					  GTK_BUTTONS_NONE,
					  _("Do you really want to quit?"));

	gtk_window_set_title (GTK_WINDOW (message), _("Confirm"));

	gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (message),
						  _("Interrupting the process may make disc unusable."));
	gtk_dialog_add_buttons (GTK_DIALOG (message),
				GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				NULL);

	button = brasero_utils_make_button (_("Continue"), GTK_STOCK_OK);
	gtk_widget_show_all (button);
	gtk_dialog_add_action_widget (GTK_DIALOG (message),
				      button, GTK_RESPONSE_OK);

	result = gtk_dialog_run (GTK_DIALOG (message));
	gtk_widget_destroy (message);

	if (result != GTK_RESPONSE_OK)
		return TRUE;

	return FALSE;
}

static gboolean
brasero_blank_dialog_delete (GtkWidget *widget, GdkEventAny *event)
{
	BraseroBlankDialog *dialog;
	BraseroBurnResult result = BRASERO_BURN_OK;

	dialog = BRASERO_BLANK_DIALOG (widget);

	if (dialog->priv->burn)
		result = brasero_burn_cancel (dialog->priv->burn, TRUE);

	if (result == BRASERO_BURN_DANGEROUS) {
		if (brasero_blank_dialog_cancel_dialog (GTK_WIDGET (dialog))) {
			if (dialog->priv->burn)
				brasero_burn_cancel (dialog->priv->burn, FALSE);
		}
		else
			return FALSE;
	}
	else
		gtk_widget_destroy (widget);

	if (GTK_WIDGET_CLASS (parent_class)->delete_event)
		return GTK_WIDGET_CLASS (parent_class)->delete_event (widget, event);
	else
		return FALSE;
}

static void
brasero_blank_dialog_cancel_clicked_cb (GtkWidget *button,
					BraseroBlankDialog *dialog)
{
	if (dialog->priv->burn) {
		BraseroBurnResult result;

		if (dialog->priv->burn)
			result = brasero_burn_cancel (dialog->priv->burn, TRUE);

		if (result == BRASERO_BURN_DANGEROUS) {
			if (brasero_blank_dialog_cancel_dialog (GTK_WIDGET (dialog))) {
				if (dialog->priv->burn)
					brasero_burn_cancel (dialog->priv->burn, FALSE);
			}
		}
	}
	else
		gtk_widget_destroy (GTK_WIDGET (dialog));
}

static void
brasero_blank_dialog_progress_changed_cb (BraseroBurn *burn,
					  gdouble overall_progress,
					  gdouble task_progress,
					  glong remaining,
					  BraseroBlankDialog *dialog)
{
	/* FIXME: with DVDs we'd better write speed in DVD speed not CD */
	brasero_burn_progress_set_status (BRASERO_BURN_PROGRESS (dialog->priv->progress),
					  FALSE,
					  overall_progress,
					  task_progress,
					  remaining,
					  -1,
					  -1,
					  -1,
					  -1);
}

static void
brasero_blank_dialog_action_changed_cb (BraseroBurn *burn,
					BraseroBurnAction action,
					BraseroBlankDialog *dialog)
{
	gchar *string = NULL;

	brasero_burn_get_action_string (dialog->priv->burn, action, &string);
	brasero_burn_progress_set_action (BRASERO_BURN_PROGRESS (dialog->priv->progress),
					  action,
					  string);
	g_free (string);
}

static BraseroBurnResult
brasero_blank_dialog_blank_insert_media_cb (BraseroBurn *burn,
					    BraseroBurnError error,
					    BraseroMediaType type,
					    BraseroBlankDialog *dialog)
{
	return BRASERO_BURN_CANCEL;
}

static void
brasero_blank_dialog_blank_clicked_cb (GtkButton *button,
				       BraseroBlankDialog *dialog)
{
	NautilusBurnDrive *drive;
	BraseroBurnResult result;
	BraseroBurnFlag flags;
	GError *error = NULL;
	gboolean fast_blank;
	GdkCursor *cursor;

	/* set up */
	gtk_widget_set_sensitive (dialog->priv->upper_box, FALSE);
	gtk_widget_set_sensitive (dialog->priv->lower_box, TRUE);
	gtk_widget_set_sensitive (GTK_WIDGET (button), FALSE);

	cursor = gdk_cursor_new (GDK_WATCH);
	gdk_window_set_cursor (GTK_WIDGET (dialog)->window, cursor);
	gdk_cursor_unref (cursor);

	/* get the drive */
	brasero_recorder_selection_get_drive (BRASERO_RECORDER_SELECTION (dialog->priv->selector),
					      &drive,
					      NULL);

	/* get the flags */
	flags = BRASERO_BURN_FLAG_EJECT | BRASERO_BURN_FLAG_NOGRACE;
	if (debug)
		flags |= BRASERO_BURN_FLAG_DEBUG;

	if (GTK_WIDGET_SENSITIVE (dialog->priv->dummy_toggle)
	&&  gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->priv->dummy_toggle)))
		flags |= BRASERO_BURN_FLAG_DUMMY;

	if (GTK_WIDGET_SENSITIVE (dialog->priv->fast_enabled)
	&&  gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->priv->fast_enabled)))
		fast_blank = TRUE;
	else
		fast_blank = FALSE;

	dialog->priv->burn = brasero_burn_new ();
	g_signal_connect (dialog->priv->burn,
			  "progress-changed",
			  G_CALLBACK (brasero_blank_dialog_progress_changed_cb),
			  dialog);
	g_signal_connect (dialog->priv->burn,
			  "action-changed",
			  G_CALLBACK (brasero_blank_dialog_action_changed_cb),
			  dialog);
	g_signal_connect (G_OBJECT (dialog->priv->burn),
			  "insert_media",
			  G_CALLBACK (brasero_blank_dialog_blank_insert_media_cb),
			  dialog);

	result = brasero_burn_blank (dialog->priv->burn,
				     flags,
				     drive,
				     fast_blank,
				     &error);

	/* Tell the user the result of the operation */
	if (result == BRASERO_BURN_ERR || error) {
		GtkWidget *message;

		message =  gtk_message_dialog_new (GTK_WINDOW (dialog),
						   GTK_DIALOG_DESTROY_WITH_PARENT|
						   GTK_DIALOG_MODAL,
						   GTK_MESSAGE_ERROR,
						   GTK_BUTTONS_CLOSE,
						   _("Error Blanking:"));

		gtk_window_set_title (GTK_WINDOW (dialog), _("Blanking finished"));

		if (error) {
			gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (message),
								  "%s.",
								  error->message);
			g_error_free (error);
		}
		else
			gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (message),
								  _("Unexpected error"));

		gtk_dialog_run (GTK_DIALOG (message));
		gtk_widget_destroy (message);
	}
	else if (result == BRASERO_BURN_OK) {
		GtkWidget *message;

		message = gtk_message_dialog_new (GTK_WINDOW (dialog),
						  GTK_DIALOG_DESTROY_WITH_PARENT|
						  GTK_DIALOG_MODAL,
						  GTK_MESSAGE_INFO,
						  GTK_BUTTONS_CLOSE,
						  _("The disc was successfully blanked:"));

		gtk_window_set_title (GTK_WINDOW (dialog), _("Blanking finished"));

		gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (message),
							  _("the disc is ready for use."));
		gtk_dialog_run (GTK_DIALOG (message));
		gtk_widget_destroy (message);
	}
	else if (result == BRASERO_BURN_NOT_SUPPORTED) {
		g_warning ("operation not supported");
	}
	else if (result == BRASERO_BURN_NOT_READY) {
		g_warning ("operation not ready");
	}
	else if (result == BRASERO_BURN_NOT_RUNNING) {
		g_warning ("job not running");
	}
	else if (result == BRASERO_BURN_RUNNING) {
		g_warning ("job running");
	}

	nautilus_burn_drive_unref (drive);
	g_object_unref (dialog->priv->burn);
	dialog->priv->burn = NULL;
	gtk_widget_destroy (GTK_WIDGET (dialog));
}
