
/*
 *  Diverse Bristol audio routines.
 *  Copyright (c) by Nick Copeland <nick.copeland@ntlworld.com> 1996,2002
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


#include <fcntl.h>

#include "brighton.h"
#include "brightonMini.h"

static int hammondB3Init();
static int hammondB3Configure();
static int hammondB3Callback(void *, int, int, float);
static int hammondB3destroy();
//static int keyCallback(void *, int, int, float);
//static int modCallback(void *, int, int, float);

extern guimain global;
static guimain manual;

#include "brightonKeys.h"

#define FIRST_DEV 0
#define C_COUNT (26 + FIRST_DEV)
#define MEM_COUNT 16
#define VOL_COUNT 1
#define LESLIE_COUNT 2
#define OPTS_COUNT 25

#define ACTIVE_DEVS (C_COUNT + OPTS_COUNT + VOL_COUNT + LESLIE_COUNT - 1)
#define DEVICE_COUNT (C_COUNT + OPTS_COUNT + VOL_COUNT + LESLIE_COUNT + MEM_COUNT)

#define OPTS_PANEL 1

#define SLIDER_START 9

#define DEVICE_START FIRST_DEV
#define OPTS_START (DEVICE_START + C_COUNT)
#define VOL_START (OPTS_START + OPTS_COUNT)
#define LESLIE_START (VOL_START + VOL_COUNT)
#define MEM_START (LESLIE_START + LESLIE_COUNT)

#define DISPLAY_PAN 4
#define DISPLAY_DEV (MEM_COUNT - 1)

#define KEY_PANEL1 5
#define KEY_PANEL2 6

#define RADIOSET_1 MEM_START

/*
 * This structure is for device definition. The structure is defined in 
 * include/brighton.h, further definitions in brighton/brightonDevtable.h and
 * include/brightoninternals.h
 *
 *	typedef int (*brightonCallback)(int, float);
 *	typedef struct BrightonLocations {
 *		int device; 0=rotary, 1=scale, etc.
 *		float relx, rely; relative position with regards to 1000 by 1000 window
 *		float relw, relh; relative height.
 *		int from, to;
 *		brightonCallback callback; specific to this dev
 *		char *image; bitmap. If zero take a device default.
 *		int flags;
 *	} brightonLocations;
 *
 * This example is for a hammondB3Bristol type synth interface.
 */
#define SD1 30
#define SD2 50

#define W1 22
#define L1 800

#define R1 0

#define C1 230
#define C2 (C1 + SD1)
#define C3 (C2 + SD1)
#define C4 (C3 + SD1)
#define C5 (C4 + SD1)
#define C6 (C5 + SD1)
#define C7 (C6 + SD1)
#define C8 (C7 + SD1)
#define C9 (C8 + SD1)

#define C11 520
#define C12 (C11 + SD1)
#define C13 (C12 + SD1)
#define C14 (C13 + SD1)
#define C15 (C14 + SD1)
#define C16 (C15 + SD1)
#define C17 (C16 + SD1)
#define C18 (C17 + SD1)
#define C19 (C18 + SD1)

#define MC1 9
#define MC2 55
#define MC3 183

#define MC4 810
#define MC5 (MC4 + SD2)
#define MC6 (MC5 + SD2)
#define MC7 (MC6 + SD2)

#define MR1 360
#define MR2 650
#define MR3 700

#define MW1 36
#define MW2 50

#define MH1 525
#define MH2 200

static brightonLocations locations[C_COUNT] = {
	{"", 1, C1, R1, W1, L1, 0, 7, 0,
		"bitmaps/knobs/hammondbrown.xpm", 0, BRIGHTON_REVERSE|BRIGHTON_HAMMOND},
	{"", 1, C2, R1, W1, L1, 0, 7, 0,
		"bitmaps/knobs/hammondbrown.xpm", 0, BRIGHTON_REVERSE|BRIGHTON_HAMMOND},
	{"", 1, C3, R1, W1, L1, 0, 7, 0,
		"bitmaps/knobs/hammondwhite.xpm", 0, BRIGHTON_REVERSE|BRIGHTON_HAMMOND},
	{"", 1, C4, R1, W1, L1, 0, 7, 0,
		"bitmaps/knobs/hammondwhite.xpm", 0, BRIGHTON_REVERSE|BRIGHTON_HAMMOND},
	{"", 1, C5, R1, W1, L1, 0, 7, 0,
		"bitmaps/knobs/hammondblack.xpm", 0, BRIGHTON_REVERSE|BRIGHTON_HAMMOND},
	{"", 1, C6, R1, W1, L1, 0, 7, 0,
		"bitmaps/knobs/hammondwhite.xpm", 0, BRIGHTON_REVERSE|BRIGHTON_HAMMOND},
	{"", 1, C7, R1, W1, L1, 0, 7, 0,
		"bitmaps/knobs/hammondblack.xpm", 0, BRIGHTON_REVERSE|BRIGHTON_HAMMOND},
	{"", 1, C8, R1, W1, L1, 0, 7, 0,
		"bitmaps/knobs/hammondblack.xpm", 0, BRIGHTON_REVERSE|BRIGHTON_HAMMOND},
	{"", 1, C9, R1, W1, L1, 0, 7, 0,
		"bitmaps/knobs/hammondwhite.xpm", 0, BRIGHTON_REVERSE|BRIGHTON_HAMMOND},
	{"", 1, C11, R1, W1, L1, 0, 7, 0,
		"bitmaps/knobs/hammondbrown.xpm", 0, BRIGHTON_REVERSE|BRIGHTON_HAMMOND},
	{"", 1, C12, R1, W1, L1, 0, 7, 0,
		"bitmaps/knobs/hammondbrown.xpm", 0, BRIGHTON_REVERSE|BRIGHTON_HAMMOND},
	{"", 1, C13, R1, W1, L1, 0, 7, 0,
		"bitmaps/knobs/hammondwhite.xpm", 0, BRIGHTON_REVERSE|BRIGHTON_HAMMOND},
	{"", 1, C14, R1, W1, L1, 0, 7, 0,
		"bitmaps/knobs/hammondwhite.xpm", 0, BRIGHTON_REVERSE|BRIGHTON_HAMMOND},
	{"", 1, C15, R1, W1, L1, 0, 7, 0,
		"bitmaps/knobs/hammondblack.xpm", 0, BRIGHTON_REVERSE|BRIGHTON_HAMMOND},
	{"", 1, C16, R1, W1, L1, 0, 7, 0,
		"bitmaps/knobs/hammondwhite.xpm", 0, BRIGHTON_REVERSE|BRIGHTON_HAMMOND},
	{"", 1, C17, R1, W1, L1, 0, 7, 0,
		"bitmaps/knobs/hammondblack.xpm", 0, BRIGHTON_REVERSE|BRIGHTON_HAMMOND},
	{"", 1, C18, R1, W1, L1, 0, 7, 0,
		"bitmaps/knobs/hammondblack.xpm", 0, BRIGHTON_REVERSE|BRIGHTON_HAMMOND},
	{"", 1, C19, R1, W1, L1, 0, 7, 0,
		"bitmaps/knobs/hammondwhite.xpm", 0, BRIGHTON_REVERSE|BRIGHTON_HAMMOND},

	{"", 2, MC1, MR1, MW1, MH1, 0, 1, 0,
		"bitmaps/buttons/rockersmooth.xpm",
		"bitmaps/buttons/rockersmoothd.xpm", 0},
	{"", 2, MC2, MR1, MW1, MH1, 0, 1, 0,
		"bitmaps/buttons/rockersmooth.xpm",
		"bitmaps/buttons/rockersmoothd.xpm", 0},
	{"", 0, MC2 + 57, MR1 + 70, MW1 + 10, MH1, 0, 6, 0,
		0, 0, 0},
	{"", 2, MC3, MR1, MW1, MH1, 0, 1, 0,
		"bitmaps/buttons/rockersmooth.xpm",
		"bitmaps/buttons/rockersmoothd.xpm", 0},

	{"", 2, MC4, MR1, MW1, MH1, 0, 1, 0,
		"bitmaps/buttons/rockersmooth.xpm",
		"bitmaps/buttons/rockersmoothd.xpm", 0},
	{"", 2, MC5, MR1, MW1, MH1, 0, 1, 0,
		"bitmaps/buttons/rockersmooth.xpm",
		"bitmaps/buttons/rockersmoothd.xpm", 0},
	{"", 2, MC6, MR1, MW1, MH1, 0, 1, 0,
		"bitmaps/buttons/rockersmooth.xpm",
		"bitmaps/buttons/rockersmoothd.xpm", 0},
	{"", 2, MC7, MR1, MW1, MH1, 0, 1, 0,
		"bitmaps/buttons/rockersmooth.xpm",
		"bitmaps/buttons/rockersmoothd.xpm", 0},
};

static brightonLocations volume[1] = {
	{"", 0, 125, 125, 750, 750, 0, 1, 0, 0, 0, 0},
};

static brightonLocations leslie[2] = {
	{"", 2, 100, 600, 800, 200, 0, 1, 0,
		"bitmaps/buttons/sw3.xpm",
		"bitmaps/buttons/sw5.xpm", BRIGHTON_VERTICAL},
	{"", 2, 100, 100, 800, 200, 0, 1, 0,
		"bitmaps/buttons/rockerwhite.xpm", 0, 0}
};

#define mR1 100
#define mR2 300
#define mR3 550
#define mR4 800

#define mC1 100
#define mC2 293
#define mC3 486
#define mC4 679
#define mC5 875

#define S4 100
#define S5 150

static brightonLocations mem[MEM_COUNT] = {
	// memories
	{"", 2, mC5, mR4, S4, S5, 0, 1, 0,
		"bitmaps/buttons/pressoffg.xpm", 
		"bitmaps/buttons/pressong.xpm", 0},
	{"", 2, mC1, mR3, S4, S5, 0, 1, 0,
		"bitmaps/buttons/pressoffg.xpm", 
		"bitmaps/buttons/pressong.xpm", 0},
	{"", 2, mC2, mR3, S4, S5, 0, 1, 0,
		"bitmaps/buttons/pressoffg.xpm", 
		"bitmaps/buttons/pressong.xpm", 0},
	{"", 2, mC3, mR3, S4, S5, 0, 1, 0,
		"bitmaps/buttons/pressoffg.xpm", 
		"bitmaps/buttons/pressong.xpm", 0},
	{"", 2, mC4, mR3, S4, S5, 0, 1, 0,
		"bitmaps/buttons/pressoffg.xpm", 
		"bitmaps/buttons/pressong.xpm", 0},
	{"", 2, mC5, mR3, S4, S5, 0, 1, 0,
		"bitmaps/buttons/pressoffg.xpm", 
		"bitmaps/buttons/pressong.xpm", 0},
	{"", 2, mC1, mR4, S4, S5, 0, 1, 0,
		"bitmaps/buttons/pressoffg.xpm", 
		"bitmaps/buttons/pressong.xpm", 0},
	{"", 2, mC2, mR4, S4, S5, 0, 1, 0,
		"bitmaps/buttons/pressoffg.xpm", 
		"bitmaps/buttons/pressong.xpm", 0},
	{"", 2, mC3, mR4, S4, S5, 0, 1, 0,
		"bitmaps/buttons/pressoffg.xpm", 
		"bitmaps/buttons/pressong.xpm", 0},
	{"", 2, mC4, mR4, S4, S5, 0, 1, 0,
		"bitmaps/buttons/pressoffg.xpm", 
		"bitmaps/buttons/pressong.xpm", 0},
	// midi U, D, Load, Save
	{"", 2, mC1, mR2, S4, S5, 0, 1, 0,
		"bitmaps/buttons/pressoff.xpm", 
		"bitmaps/buttons/presson.xpm", BRIGHTON_CHECKBUTTON},
	{"", 2, mC2, mR2, S4, S5, 0, 1, 0,
		"bitmaps/buttons/pressoff.xpm", 
		"bitmaps/buttons/presson.xpm", BRIGHTON_CHECKBUTTON},
	{"", 2, mC4, mR2, S4, S5, 0, 1, 0,
		"bitmaps/buttons/pressoffg.xpm", 
		"bitmaps/buttons/pressong.xpm", BRIGHTON_CHECKBUTTON},
	{"", 2, mC3, mR2, S4, S5, 0, 1, 0,
		"bitmaps/buttons/pressoffo.xpm", 
		"bitmaps/buttons/pressono.xpm", BRIGHTON_CHECKBUTTON},
	{"", 2, mC5, mR2, S4, S5, 0, 1, 0,
		"bitmaps/buttons/pressoff.xpm", 
		"bitmaps/buttons/presson.xpm", BRIGHTON_CHECKBUTTON},
	// display
	{"", 3, mC1, mR1, 875, 125, 0, 1, 0, 0, 0, 0},
};

#define OD1 90
#define OD2 92
#define OD3 70

#define OC1 65
#define OC2 (OC1 + OD1 + 50)
#define OC3 (OC2 + OD1)
#define OC4 (OC3 + OD1)
#define OC5 (OC4 + OD1)
#define OC6 (OC5 + OD1)
#define OC7 (OC6 + OD1)
#define OC8 (OC7 + OD1)
#define OC9 (OC8 + OD1)
#define OC10 (OC9 + OD1)

#define OR1 23
#define OR2 (OR1 + OD2)
#define OR3 (OR2 + OD2)
#define OR4 (OR3 + OD2)

#define OR5 (OR1 + 250)
#define OR6 (OR5 + 250)
#define OR7 (OR6 + 250)

#define OS1 20
#define OS2 88
#define OS3 150
static brightonLocations opts[OPTS_COUNT] = {
	// Leslie type - 26
	{"", 2, OC1, OR1, OS1, OS2, 0, 1, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 2, OC1, OR2, OS1, OS2, 0, 1, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 2, OC1, OR3, OS1, OS2, 0, 1, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 2, OC1, OR4, OS1, OS2, 0, 1, 0,
		"bitmaps/buttons/pressoffg.xpm",
		"bitmaps/buttons/pressong.xpm", 0},
	// Leslie global - 30
	{"", 0, OC2, OR1, OS3, OS3, 0, 1, 0, 0, 0, 0},
	{"", 0, OC3, OR1, OS3, OS3, 0, 1, 0, 0, 0, 0},
	// Leslie HS - 32
	{"", 0, OC5, OR1, OS3, OS3, 0, 1, 0, 0, 0, 0},
	{"", 0, OC6, OR1, OS3, OS3, 0, 1, 0, 0, 0, 0},
	{"", 0, OC7, OR1, OS3, OS3, 0, 1, 0, 0, 0, 0},
	// Leslie global - 35
	{"", 0, OC4, OR1, OS3, OS3, 0, 1, 0, 0, 0, 0},
	// Leslie LS - 32
	{"", 0, OC8, OR1, OS3, OS3, 0, 1, 0, 0, 0, 0},
	{"", 0, OC9, OR1, OS3, OS3, 0, 1, 0, 0, 0, 0},
	{"", 0, OC10, OR1, OS3, OS3, 0, 1, 0, 0, 0, 0},
	// VC speed
	{"", 0, OC2, OR5, OS3, OS3, 0, 1, 0, 0, 0, 0},
	{"", 0, OC3, OR5, OS3, OS3, 0, 1, 0, 0, 0, 0},
	{"", 0, OC4, OR5, OS3, OS3, 0, 1, 0, 0, 0, 0},
	// Envelopes
	{"", 0, OC2, OR6, OS3, OS3, 0, 1, 0, 0, 0, 0},
	{"", 0, OC3, OR6, OS3, OS3, 0, 1, 0, 0, 0, 0},
	{"", 0, OC5, OR6, OS3, OS3, 0, 1, 0, 0, 0, 0},
	{"", 0, OC6, OR6, OS3, OS3, 0, 1, 0, 0, 0, 0},
	// Diverse
	{"", 2, OC1, OR7 - 10, OS1, OS2, 0, 1, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 2, OC1, OR7 + OD2 - 10, OS1, OS2, 0, 1, 0,
		"bitmaps/buttons/pressoffg.xpm",
		"bitmaps/buttons/pressong.xpm", 0},
	{"", 0, OC2, OR7, OS3, OS3, 0, 7, 0, 0, 0, 0},
	{"", 0, OC3, OR7, OS3, OS3, 0, 1, 0, 0, 0, 0},
	{"", 0, OC4, OR7, OS3, OS3, 0, 1, 0, 0, 0, 0},
// Bin two reverb levels
//	{"", 0, OC5, OR7, OS3, OS3, 0, 1, 0, 0, 0, 0},
//	{"", 0, OC6, OR7, OS3, OS3, 0, 1, 0, 0, 0, 0},
};

static int
lmCallback(void *cid, int panel, int index, float value)
{
	guiSynth *synth = findSynth(global.synths, (int) cid);

//	printf("keycallback(%x, %i, %i, %f): %i %i\n", synth, panel, index, value,
//		synth->transpose, global.controlfd);

	/*
	 * Want to send a note event, on or off, for this index + transpose.
	 */
	if (value)
		bristolMidiSendMsg(manual.controlfd, synth->midichannel + 1,
			BRISTOL_EVENT_KEYON, 0, index + synth->transpose);
	else
		bristolMidiSendMsg(manual.controlfd, synth->midichannel + 1,
			BRISTOL_EVENT_KEYOFF, 0, index + synth->transpose);
}

static int
hammondB3destroy(int cid)
{
	guiSynth *synth = findSynth(global.synths, cid);
	bristolMidiMsg msg;

printf("hammondB3destroy(%i): %i\n", cid, synth->midichannel);
	/*
	 * Since we registered two synths, we now need to remove the upper
	 * manual.
	 */
	bristolMidiSendMsg(manual.controlfd, synth->sid2, 127, 0,
		BRISTOL_EXIT_ALGO);
	if (bristolMidiRead(manual.controlfd, &msg) < 0)
		printf("socket closed\n");
	bristolMidiSendMsg(global.controlfd, synth->sid, 127, 0,
		BRISTOL_EXIT_ALGO);
	if (bristolMidiRead(global.controlfd, &msg) < 0)
		printf("socket closed\n");
}

/*
 * This is a set of globals for the main window rendering. Again taken from
 * include/brighton.h
 */
brightonApp hammondB3App = {
	"hammond-B3",
	0, // no blueprint on wood background.
	"bitmaps/textures/wood3.xpm",
	0, // BRIGHTON_STRETCH, //flags
	hammondB3Init,
	hammondB3Configure, // 3 callbacks, unused?
	0,
	hammondB3destroy,
	745, 325,
	15, // one panel only
	{
		{
			"HammondB3",
			"bitmaps/blueprints/hammondB3.xpm",
			"bitmaps/textures/metal5.xpm",
			0, //BRIGHTON_STRETCH, // flags
			0,
			0,
			hammondB3Callback,
			15, 150, 970, 293,
			C_COUNT,
			locations
		},
		{
			"Opts",
			"bitmaps/blueprints/hammondb3opts.xpm",
			"bitmaps/textures/wood3.xpm",
			0x020, //BRIGHTON_STRETCH, // flags
			0,
			0,
			hammondB3Callback,
			70, 440, 650, 533,
			//70, 440, 470, 500,
			OPTS_COUNT,
			opts
		},
		{
			"volume",
			0,
			"bitmaps/textures/metal5.xpm",
			0, //BRIGHTON_STRETCH, // flags
			0,
			0,
			hammondB3Callback,
			20, 525, 47, 110,
			1,
			volume
		},
		{
			"leslie",
			0,
			"bitmaps/textures/metal5.xpm",
			0, //BRIGHTON_STRETCH, // flags
			0,
			0,
			hammondB3Callback,
			20, 880, 47, 95,
			2,
			leslie
		},
		{
			"Memory",
			"bitmaps/blueprints/hammondb3mem.xpm",
			"bitmaps/textures/wood3.xpm",
			0x020, //BRIGHTON_STRETCH, // flags
			0,
			0,
			hammondB3Callback,
			720, 440, 250, 533,
			//70, 440, 470, 500,
			MEM_COUNT,
			mem
		},
		{
			"Keybaord",
			0,
			"bitmaps/keys/kbghammond.xpm", // flags
			BRIGHTON_STRETCH,
			0,
			0,
			keyCallback,
			70, 440, 900, 250,
			KEY_COUNT_6OCTAVE,
			keys6hammond
		},
		{
			"Keybaord",
			0,
			"bitmaps/keys/kbghammond.xpm", // flags
			BRIGHTON_STRETCH,
			0,
			0,
			lmCallback,
			70, 725, 900, 250,
			KEY_COUNT_6OCTAVE,
			keys6hammond
		},
		{
			"HammondB3",
			0,
			"bitmaps/textures/wood3.xpm",
			BRIGHTON_STRETCH|BRIGHTON_VERTICAL, // flags
			0,
			0,
			0,
			0, 0, 15, 1000,
			0,
			0
		},
		{
			"HammondB3",
			0,
			"bitmaps/textures/wood3.xpm",
			BRIGHTON_STRETCH|BRIGHTON_VERTICAL, //flags
			0,
			0,
			0,
			985, 0, 15, 1000,
			0,
			0
		},
		{
			"HammondB3",
			0,
			"bitmaps/textures/wood4.xpm",
			BRIGHTON_STRETCH|BRIGHTON_VERTICAL, //flags
			0,
			0,
			0,
			0, 632, 15, 125,
			0,
			0
		},
		{
			"HammondB3",
			0,
			"bitmaps/textures/wood4.xpm",
			BRIGHTON_STRETCH|BRIGHTON_VERTICAL, //flags
			0,
			0,
			0,
			985, 632, 15, 125,
			0,
			0
		},
/*
		{
			"HammondB3",
			0,
			"bitmaps/textures/wood4.xpm",
			0, //flags
			0,
			0,
			0,
			15, 408, 970, 35,
			0,
			0
		},
*/
		{
			"HammondB3",
			0,
			"bitmaps/textures/metal5.xpm",
			0, //flags
			0,
			0,
			0,
			15, 691, 970, 34,
			0,
			0
		},
		{
			"HammondB3",
			0,
			"bitmaps/textures/wood4.xpm",
			0, //flags
			0,
			0,
			0,
			15, 30, 970, 70,
			0,
			0
		},
		{
			"HammondB3",
			0,
			"bitmaps/textures/wood4.xpm",
			BRIGHTON_STRETCH, //flags
			0,
			0,
			0,
			15, 60, 970, 90,
			0,
			0
		},
		{
			"HammondB3",
			0,
			"bitmaps/textures/wood5.xpm",
			BRIGHTON_STRETCH, //flags
			0,
			0,
			0,
			15, 60, 970, 150,
			0,
			0
		}
	}
};

static int
b3SendMsg(int fd, int chan, int c, int o, int v)
{
//printf("b3SendMsg(%i, %i, %i, %i, %i) [%i, %i]\n", fd, chan, c, o, v,
//global.controlfd, manual.controlfd);
	bristolMidiSendMsg(global.controlfd, chan, c, o, v);
	bristolMidiSendMsg(manual.controlfd, chan + 1, c, o, v);
}

static int
hammondB3MidiSendMsg(void *synth, int fd, int chan, int c, int o, int v)
{
	bristolMidiSendMsg(fd, chan, c, o, v);
}

static int
multiTune(guiSynth *synth, int fd, int chan, int c, int o, int v)
{
	brightonEvent event;

	/*
	 * Configures all the tune settings to zero (ie, 0.5). Do the Osc-A first,
	 * since it is not visible, and then request the GUI to configure Osc-B.
	 */
	bristolMidiSendMsg(fd, chan, 0, 2, 8191);
	event.type = BRIGHTON_FLOAT;
	event.value = 0.5;
	brightonParamChange((void *) synth->connid, synth->panel, FIRST_DEV + 45,
		&event);
	brightonParamChange((void *) synth->connid, synth->panel, FIRST_DEV + 21,
		&event);
}

static int
midiRelease(guiSynth *synth, int fd, int chan, int c, int o, int v)
{
	if (!global.libtest)
	{
		bristolMidiMsg msg;

		bristolMidiSendMsg(fd, synth->sid, 127, 0,
			BRISTOL_ALL_NOTES_OFF);

		if (bristolMidiRead(fd, &msg) != BRISTOL_OK)
			cleanupBristol();
	}
}

static int
hammondB3DualDecay(void *synth, int fd, int chan, int c, int o, int v)
{
	bristolMidiSendMsg(fd, chan, c, 1, v);
	bristolMidiSendMsg(fd, chan, c, 3, v);
}

static int
hammondB3Memory(guiSynth *synth, int fd, int chan, int c, int o, int v)
{
	brightonEvent event;

//printf("hammondB3Memory(%i, %i)\n", c, o);

	if (synth->dispatch[RADIOSET_1].other2)
	{
		synth->dispatch[RADIOSET_1].other2 = 0;
		return;
	}

	switch (c) {
		default:
		case 0:
			/*
			 * We want to make these into memory buttons. To do so we need to
			 * know what the last active button was, and deactivate its 
			 * display, then send any message which represents the most
			 * recently configured value. Since this is a memory button we do
			 * not have much issue with the message, but we are concerned with
			 * the display.
			 */
			if (synth->dispatch[RADIOSET_1].other1 != -1)
			{
				synth->dispatch[RADIOSET_1].other2 = 1;

				if (synth->dispatch[RADIOSET_1].other1 != o)
					event.value = 0;
				else
					event.value = 1;

				brightonParamChange((void *) synth->connid, DISPLAY_PAN,
					synth->dispatch[RADIOSET_1].other1, &event);
			}
			synth->dispatch[RADIOSET_1].other1 = o;

			if (synth->flags & BANK_SELECT) {
				if ((synth->bank * 10 + o * 10) >= 1000)
				{
					synth->location = o;
					synth->flags &= ~BANK_SELECT;
					if (loadMemory(synth, "hammondB3", 0,
						synth->bank + synth->location,
						synth->mem.active, FIRST_DEV, BRISTOL_STAT) < 0)
						displayPanelText(synth, "FRE",
							synth->bank + synth->location,
							DISPLAY_PAN, DISPLAY_DEV);
					else
						displayPanelText(synth, "PRG",
							synth->bank + synth->location,
							DISPLAY_PAN, DISPLAY_DEV);
				} else {
					synth->bank = synth->bank * 10 + o * 10;
					if (loadMemory(synth, "hammondB3", 0,
						synth->bank + synth->location,
						synth->mem.active, FIRST_DEV, BRISTOL_STAT) < 0)
						displayPanelText(synth, "FRE",
							synth->bank + synth->location,
							DISPLAY_PAN, DISPLAY_DEV);
					else
						displayPanelText(synth, "BANK",
							synth->bank + synth->location,
							DISPLAY_PAN, DISPLAY_DEV);
				}
			} else {
				synth->location = o;
				if (loadMemory(synth, "hammondB3", 0,
					synth->bank + synth->location,
					synth->mem.active, FIRST_DEV, BRISTOL_STAT) < 0)
					displayPanelText(synth, "FRE",
						synth->bank + synth->location,
						DISPLAY_PAN, DISPLAY_DEV);
				else
					displayPanelText(synth, "PRG",
						synth->bank + synth->location,
						DISPLAY_PAN, DISPLAY_DEV);
			}
			break;
		case 1:
			synth->flags |= MEM_LOADING;
			if (loadMemory(synth, "hammondB3", 0, synth->bank + synth->location,
				synth->mem.active, FIRST_DEV, 0) < 0)
				displayPanelText(synth, "FRE", synth->bank + synth->location,
					DISPLAY_PAN, DISPLAY_DEV);
			else
				displayPanelText(synth, "PRG", synth->bank + synth->location,
					DISPLAY_PAN, DISPLAY_DEV);
			synth->flags &= ~MEM_LOADING;
			synth->flags &= ~BANK_SELECT;
			break;
		case 2:
			saveMemory(synth, "hammondB3", 0, synth->bank + synth->location,
				FIRST_DEV);
			displayPanelText(synth, "PRG", synth->bank + synth->location,
				DISPLAY_PAN, DISPLAY_DEV);
			synth->flags &= ~BANK_SELECT;
			break;
		case 3:
			if (synth->flags & BANK_SELECT) {
				synth->flags &= ~BANK_SELECT;
				if (loadMemory(synth, "hammondB3", 0,
					synth->bank + synth->location,
					synth->mem.active, FIRST_DEV, BRISTOL_STAT) < 0)
					displayPanelText(synth, "FRE",
						synth->bank + synth->location,
						DISPLAY_PAN, DISPLAY_DEV);
				else
					displayPanelText(synth, "PRG",
						synth->bank + synth->location,
						DISPLAY_PAN, DISPLAY_DEV);
			} else {
				synth->bank = 0;
				displayPanelText(synth, "BANK", synth->bank, DISPLAY_PAN,
					DISPLAY_DEV);
				synth->flags |= BANK_SELECT;
			}
			break;
	}
//	printf("	hammondB3Memory(B: %i L %i: %i)\n",
//		synth->bank, synth->location, o);
}

static int
hammondB3Midi(guiSynth *synth, int fd, int chan, int c, int o, int v)
{
	bristolMidiMsg msg;
	int newchan;
//	printf("hammondB3Midi(%i %i %i %i %i)\n", fd, chan, c, o, v);

	if (c == 1) {
		if ((newchan = synth->midichannel - 1) < 0)
		{
			synth->midichannel = 0;
			return;
		}
	} else {
		/*
		 * On the upper side we need to reserve two midi channels
		 */
		if ((newchan = synth->midichannel + 1) >= 15)
		{
			synth->midichannel = 14;
			return;
		}
	}

	/*
	 * Lower manual midi channel selection first
	 */
	bristolMidiSendMsg(manual.controlfd, synth->sid2,
		127, 0, BRISTOL_MIDICHANNEL|newchan + 1);
	if (bristolMidiRead(manual.controlfd, &msg) != BRISTOL_OK)
		cleanupBristol();

	bristolMidiSendMsg(global.controlfd, synth->sid,
		127, 0, BRISTOL_MIDICHANNEL|newchan);
	if (bristolMidiRead(global.controlfd, &msg) != BRISTOL_OK)
		cleanupBristol();

	synth->midichannel = newchan;

	displayPanelText(synth, "MIDI", synth->midichannel + 1,
		DISPLAY_PAN, DISPLAY_DEV);

}

/*
 * For the sake of ease of use, links have been placed here to be called
 * by any of the devices created. They would be better in some other file,
 * perhaps with this as a dispatch.
 *
 * Param refers to the device index in the locations table given below.
 */
static int
hammondB3Callback(void *cid, int panel, int index, float value)
{
	guiSynth *synth = findSynth(global.synths, (int) cid);
	int sendvalue;

//	printf("hammondB3Callback(%i, %i, %f): %x\n", panel, index, value, synth);

	if (synth == 0)
		return;

	if ((index >= DEVICE_COUNT) || (synth->flags & OPERATIONAL == 0))
		return;

	if (hammondB3App.resources[panel].devlocn[index].to == 1.0)
		sendvalue = value * (CONTROLLER_RANGE - 1);
	else
		sendvalue = value;

	switch (panel) {
		default:
		case 0:
			break;
		case 1:
			index += OPTS_START;
			break;
		case 2:
			index += VOL_START;
			break;
		case 3:
			index += LESLIE_START;
			break;
		case 4:
			index += MEM_START;
			break;
	}
//printf("index is now %i, %i %i\n", index, DEVICE_COUNT, ACTIVE_DEVS);

	synth->mem.param[index] = value;

	if ((!global.libtest) || (index >= ACTIVE_DEVS) || (index == 18))
		synth->dispatch[index].routine(synth,
			global.controlfd, synth->sid,
			synth->dispatch[index].controller,
			synth->dispatch[index].operator,
			sendvalue);

#define DEBUG
#ifdef DEBUG
	else
		printf("dispatch[%x,%i](%i, %i, %i, %i, %i)\n", synth, index,
			global.controlfd, synth->sid,
			synth->dispatch[index].controller,
			synth->dispatch[index].operator,
			sendvalue);
#endif

	return(0);
}

static int
hammondB3Passthrough(float value)
{
	printf("hammondB3Passthrough\n");
}

/*
static int
doSlider(guiSynth *synth, int fd, int chan, int cont, int op, int value)
{
	int slidervalue;

	slidervalue = cont * 8 + value;

	bristolMidiSendMsg(global.controlfd, synth->sid, 0, 2,
		slidervalue);
}
*/

static int
doVolume(guiSynth *synth)
{
	b3SendMsg(global.controlfd, synth->sid, 0, 4,
		(int) (synth->mem.param[VOL_START] * C_RANGE_MIN_1));
}

static int
doLMSlider(guiSynth *synth, int fd, int chan, int cont, int op, int value)
{
	int slidervalue;

//printf("doLMSlider(%x, %i, %i, %i, %i, %i)\n",
//synth, fd, chan, cont, op, value);

	slidervalue = cont * 8 + value;

	bristolMidiSendMsg(manual.controlfd, synth->sid2, 0, 2,
		slidervalue);
}

static int
doSlider(guiSynth *synth, int fd, int chan, int cont, int op, int value)
{
	int slidervalue;

	//printf("doSlider(%x, %i, %i, %i, %i, %i)\n",
	//	synth, fd, chan, cont, op, value);

	slidervalue = cont * 8 + value;

	bristolMidiSendMsg(global.controlfd, synth->sid, 0, 2,
		slidervalue);
}

static int
hammondB3Switch(guiSynth *id, int fd, int chan, int cont, int op, int value)
{
	brightonEvent event;

//	printf("hammondB3Switch(%x, %i, %i, %i, %i, %i)\n",
//		id, fd, chan, cont, op, value);

	/* 
	 * If the sendvalue is zero, then withdraw the opts window, draw the
	 * slider window, and vice versa.
	 */
	if (value == 0)
	{
		event.type = BRIGHTON_EXPOSE;
		event.intvalue = 0;
		brightonParamChange((void *) id->connid, 1, -1, &event);
		event.intvalue = 0;
		brightonParamChange((void *) id->connid, 4, -1, &event);
		event.intvalue = 1;
		brightonParamChange((void *) id->connid, 5, -1, &event);
		event.intvalue = 1;
		brightonParamChange((void *) id->connid, 6, -1, &event);
		event.intvalue = 1;
		brightonParamChange((void *) id->connid, 11, -1, &event);
	} else {
		event.type = BRIGHTON_EXPOSE;
		event.intvalue = 0;
		brightonParamChange((void *) id->connid, 5, -1, &event);
		event.intvalue = 0;
		brightonParamChange((void *) id->connid, 6, -1, &event);
		event.intvalue = 0;
		brightonParamChange((void *) id->connid, 11, -1, &event);
		event.intvalue = 1;
		brightonParamChange((void *) id->connid, 1, -1, &event);
		event.intvalue = 1;
		brightonParamChange((void *) id->connid, 4, -1, &event);
	}
}

static int
doVibra(guiSynth *synth)
{
	int speed, chorus = 0;

//printf("doVibra(%f)\n", synth->mem.param[20]);
	/*
	 * We get a value from 0 to 6.
	 * 0 1 2 are vibra only, with speeds from the opts panel.
	 * 3 is off
	 * 4 5 6 are vibrachorus, with same respective speeds
	 */
	switch ((int) synth->mem.param[20])
	{
		case 0:
			/*
			 * Fast vibra
			 */
			speed = synth->mem.param[OPTS_START + 15] * C_RANGE_MIN_1;
			break;
		case 1:
			/*
			 * mid vibra
			 */
			speed = synth->mem.param[OPTS_START + 14] * C_RANGE_MIN_1;
			break;
		case 2:
			/*
			 * slow vibra
			 */
			speed = synth->mem.param[OPTS_START + 13] * C_RANGE_MIN_1;
			break;
		case 3:
			/*
			 * Off
			 */
			bristolMidiSendMsg(global.controlfd, synth->sid, 126, 0, 0);
			return;
			break;
		case 4:
			/*
			 * slow vibrachorus
			 */
			speed = synth->mem.param[OPTS_START + 13] * C_RANGE_MIN_1;
			chorus = 1;
			break;
		case 5:
			/*
			 * mid vibrachorus
			 */
			speed = synth->mem.param[OPTS_START + 14] * C_RANGE_MIN_1;
			chorus = 1;
			break;
		case 6:
			/*
			 * fast vibrachorus
			 */
			speed = synth->mem.param[OPTS_START + 15] * C_RANGE_MIN_1;
			chorus = 1;
			break;
	}
	/*
	 * Turn it on
	 */
	bristolMidiSendMsg(global.controlfd, synth->sid, 126, 0, 1);
	/*
	 * Vibra/Chorus
	 */
	bristolMidiSendMsg(global.controlfd, synth->sid, 6, 2, chorus);
	/*
	 * Speed and depth
	 */
	bristolMidiSendMsg(global.controlfd, synth->sid, 6, 0, speed);
	bristolMidiSendMsg(global.controlfd, synth->sid, 6, 1, speed);
}

#define LESLIE_ONOFF 18

static int
doLeslieSpeed(guiSynth *synth)
{
	int speed, depth, phase;

//printf("doLeslieSpeed()\n");
	if (synth->mem.param[LESLIE_ONOFF] == 0)
	{
		bristolMidiSendMsg(global.controlfd, synth->sid, 100, 7, 0);
		return;
	}
	bristolMidiSendMsg(global.controlfd, synth->sid, 100, 7,
		(int) (synth->dispatch[OPTS_START + 1].other1));

	if (synth->mem.param[LESLIE_START] == 0)
	{
		speed = synth->mem.param[OPTS_START + 6] * C_RANGE_MIN_1;
		depth = synth->mem.param[OPTS_START + 7] * C_RANGE_MIN_1;
		phase = synth->mem.param[OPTS_START + 8] * C_RANGE_MIN_1;
	} else {
		speed = synth->mem.param[OPTS_START + 10] * C_RANGE_MIN_1;
		depth = synth->mem.param[OPTS_START + 11] * C_RANGE_MIN_1;
		phase = synth->mem.param[OPTS_START + 12] * C_RANGE_MIN_1;
	}
	bristolMidiSendMsg(global.controlfd, synth->sid, 100, 0, speed);
	bristolMidiSendMsg(global.controlfd, synth->sid, 100, 3, depth);
	bristolMidiSendMsg(global.controlfd, synth->sid, 100, 2, phase);
	bristolMidiSendMsg(global.controlfd, synth->sid, 100, 6,
		(int) (synth->mem.param[OPTS_START + 4] * C_RANGE_MIN_1));
	bristolMidiSendMsg(global.controlfd, synth->sid, 100, 1,
		(int) (synth->mem.param[OPTS_START + 5] * C_RANGE_MIN_1));
}

static int
doReverb(guiSynth *synth)
{
//printf("doReverb()\n");
	if (synth->mem.param[DEVICE_START + 19] == 0)
		bristolMidiSendMsg(global.controlfd, synth->sid, 100, 4, 0);
	else
		bristolMidiSendMsg(global.controlfd, synth->sid, 100, 4,
			(int) (synth->mem.param[OPTS_START + 24] * C_RANGE_MIN_1));
}

static int
doClick(guiSynth *synth)
{
//printf("doClick()\n");
	b3SendMsg(global.controlfd, synth->sid, 0, 6,
		(int) (synth->mem.param[OPTS_START + 23] * C_RANGE_MIN_1));
}

static int
doPreacher(guiSynth *synth)
{
//printf("doPreacher()\n");
	if (synth->mem.param[OPTS_START + 20] == 0)
	{
		b3SendMsg(global.controlfd, synth->sid, 126, 3, 0);
		b3SendMsg(global.controlfd, synth->sid, 0, 7, 0);
	} else {
		b3SendMsg(global.controlfd, synth->sid, 126, 3, 1);
		b3SendMsg(global.controlfd, synth->sid, 0, 7, 1);
	}
}

static int
doBright(guiSynth *synth)
{
//printf("doBright()\n");
	if (synth->mem.param[DEVICE_START + 21] == 0)
	{
		/*
		 * Once to the hammond manager
		 */
		b3SendMsg(global.controlfd, synth->sid, 126, 1, 0);
		/*
		 * And to the sine oscillator
		 */
		b3SendMsg(global.controlfd, synth->sid, 0, 0, 0);
		b3SendMsg(global.controlfd, synth->sid, 0, 0, 8);
		b3SendMsg(global.controlfd, synth->sid, 0, 0, 16);
		b3SendMsg(global.controlfd, synth->sid, 0, 0, 24);
		b3SendMsg(global.controlfd, synth->sid, 0, 0, 32);
		b3SendMsg(global.controlfd, synth->sid, 0, 0, 40);
		b3SendMsg(global.controlfd, synth->sid, 0, 0, 48);
		b3SendMsg(global.controlfd, synth->sid, 0, 0, 56);
		b3SendMsg(global.controlfd, synth->sid, 0, 0, 64);
	} else {
		b3SendMsg(global.controlfd, synth->sid, 126, 1,
			(int) (synth->mem.param[OPTS_START + 22]));
		b3SendMsg(global.controlfd, synth->sid, 0, 0, 0 +
			(int) (synth->mem.param[OPTS_START + 22]));
		b3SendMsg(global.controlfd, synth->sid, 0, 0, 8 +
			(int) (synth->mem.param[OPTS_START + 22]));
		b3SendMsg(global.controlfd, synth->sid, 0, 0, 16 +
			(int) (synth->mem.param[OPTS_START + 22]));
		b3SendMsg(global.controlfd, synth->sid, 0, 0, 24 +
			(int) (synth->mem.param[OPTS_START + 22]));
		b3SendMsg(global.controlfd, synth->sid, 0, 0, 32 +
			(int) (synth->mem.param[OPTS_START + 22]));
		b3SendMsg(global.controlfd, synth->sid, 0, 0, 40 +
			(int) (synth->mem.param[OPTS_START + 22]));
		b3SendMsg(global.controlfd, synth->sid, 0, 0, 48 +
			(int) (synth->mem.param[OPTS_START + 22]));
		b3SendMsg(global.controlfd, synth->sid, 0, 0, 56 +
			(int) (synth->mem.param[OPTS_START + 22]));
		b3SendMsg(global.controlfd, synth->sid, 0, 0, 64 +
			(int) (synth->mem.param[OPTS_START + 22]));
	}
}

static int
doCompress(guiSynth *synth)
{
//printf("doCompress()\n");
	if (synth->mem.param[OPTS_START + 21] == 0)
	{
		b3SendMsg(global.controlfd, synth->sid, 126, 2, 0);
	} else {
		b3SendMsg(global.controlfd, synth->sid, 126, 2, 1);
	}
}

static int
doGrooming(guiSynth *synth)
{
//printf("doGrooming()\n");
	if (synth->mem.param[DEVICE_START + 25] != 0)
	{
		b3SendMsg(global.controlfd, synth->sid, 1, 0,
			(int) (synth->mem.param[OPTS_START + 18] * C_RANGE_MIN_1));
	} else {
		b3SendMsg(global.controlfd, synth->sid, 1, 0,
			(int) (synth->mem.param[OPTS_START + 19] * C_RANGE_MIN_1));
	}
}

static int
doPerc(guiSynth *synth)
{
//printf("doPerc()\n");
	/*
	 * 4 foot percussive
	 */
	if (synth->mem.param[DEVICE_START + 22] == 0)
		bristolMidiSendMsg(global.controlfd, synth->sid, 0, 3, 24);
	else
		bristolMidiSendMsg(global.controlfd, synth->sid, 0, 3, 25);

	/*
	 * 2 1/3 foot percussive
	 */
	if (synth->mem.param[DEVICE_START + 23] == 0)
		bristolMidiSendMsg(global.controlfd, synth->sid, 0, 3, 32);
	else
		bristolMidiSendMsg(global.controlfd, synth->sid, 0, 3, 33);

	/*
	 * Slow fast decay
	 */
	if (synth->mem.param[DEVICE_START + 24] == 0)
		bristolMidiSendMsg(global.controlfd, synth->sid, 4, 1,
			(int) (synth->mem.param[OPTS_START + 16] * C_RANGE_MIN_1));
	else
		bristolMidiSendMsg(global.controlfd, synth->sid, 4, 1,
			(int) (synth->mem.param[OPTS_START + 17] * C_RANGE_MIN_1));
}

static int
hammondOption(guiSynth *synth, int fd, int chan, int cont, int op, int value)
{
	brightonEvent event;

//	printf("hammondOption(%x, %i, %i, %i, %i, %i)\n",
//		synth, fd, chan, cont, op, value);

	switch (cont) {
		case OPTS_START:
			/*
			 * Rotation type. Send 100.7 becomes op;
			 */
			if ((synth->flags & MEM_LOADING) == 0)
			{
				if (synth->dispatch[OPTS_START].other2)
				{
					synth->dispatch[OPTS_START].other2 = 0;
					return;
				}
				synth->dispatch[OPTS_START].other2 = 1;
				if (synth->dispatch[OPTS_START].other1 >= 0)
				{
					event.command = BRIGHTON_PARAMCHANGE;
					if (synth->dispatch[OPTS_START].other1 != (op - 1))
						event.value = 0;
					else
						event.value = 1;
					brightonParamChange((void *) synth->connid,
						OPTS_PANEL, synth->dispatch[OPTS_START].other1, &event);
				}
			}

			if (synth->mem.param[OPTS_START + op - 1] != 0)
			{
				synth->dispatch[OPTS_START].other1 = op - 1;
				synth->dispatch[OPTS_START + 1].other1 = op;

				if (synth->mem.param[LESLIE_ONOFF] == 0)
					bristolMidiSendMsg(global.controlfd, chan, 100, 7, 0);
				else {
					bristolMidiSendMsg(global.controlfd, chan, 100, 7, op);
					bristolMidiSendMsg(global.controlfd, chan, 100, 1,
						(int) (synth->mem.param[OPTS_START + 5]
							* C_RANGE_MIN_1));
				}
			}
			break;
		case OPTS_START + 3:
			/*
			 * Rotor break. Send 100.7 = 4 off, 100.7 = 5 on.
			 */
			if (value == 0)
				bristolMidiSendMsg(global.controlfd, chan, 100, 7, 4);
			else
				bristolMidiSendMsg(global.controlfd, chan, 100, 7, 5);
			break;
		case OPTS_START + 4:
		case OPTS_START + 5:
		case OPTS_START + 6:
		case OPTS_START + 7:
		case OPTS_START + 8:
		case OPTS_START + 10:
		case OPTS_START + 11:
		case OPTS_START + 12:
			doLeslieSpeed(synth);
			break;
		case OPTS_START + 9:
			// overdrive
			bristolMidiSendMsg(global.controlfd, synth->sid, 100, 5,
				(int) (synth->mem.param[OPTS_START + 9] * C_RANGE_MIN_1));
			break;
		case OPTS_START + 13:
		case OPTS_START + 14:
		case OPTS_START + 15:
			doVibra(synth);
			break;
		case OPTS_START + 16:
		case OPTS_START + 17:
			doPerc(synth);
			break;
		case OPTS_START + 18:
		case OPTS_START + 19:
			doGrooming(synth);
			break;
		case OPTS_START + 20:
			doPreacher(synth);
			break;
		case OPTS_START + 21:
			doCompress(synth);
			break;
		case OPTS_START + 22:
			doBright(synth);
			break;
		case OPTS_START + 23:
			doClick(synth);
			break;
		case OPTS_START + 24:
		case OPTS_START + 25:
		case OPTS_START + 26:
			doReverb(synth);
			break;
		default:
			break;
	}
}

/*
 * Any location initialisation required to run the callbacks. For bristol, this
 * will connect to the engine, and give it some base parameters.
 * May need to generate some application specific menus.
 * Will also then make specific requests to some of the devices to alter their
 * rendering.
 */
static int
hammondB3Init(void *reference)
{
	guiSynth *synth = findSynth(global.synths, (int) reference);
	dispatcher *dispatch;
	int i;

	if (synth == 0)
	{
		synth = findSynth(global.synths, (int) 0);
		if (synth == 0)
		{
			printf("cannot init\n");
			return(0);
		}
	}

	synth->connid = (int) reference;

	printf("Initialise the hammondB3 link to bristol: %x\n", synth->connid);

	synth->mem.param = (float *) brightonmalloc(DEVICE_COUNT * sizeof(float));
	synth->mem.count = DEVICE_COUNT;
	synth->mem.active = ACTIVE_DEVS;
	synth->dispatch = (dispatcher *)
		brightonmalloc(DEVICE_COUNT * sizeof(dispatcher));
	dispatch = synth->dispatch;

	if (!global.libtest)
	{
		int hold;
		bristolMidiMsg msg;

		if (synth->synthtype == BRISTOL_HAMMONDB3)
		{
			int sid;

			bcopy(&global, &manual, sizeof(guimain));

printf("starting upper manual connection\n");
			synth->synthtype = BRISTOL_HAMMOND;
			if ((synth->sid = initConnection(&global, synth)) < 0)
				exit(0);
printf("UM CID: %i\n", synth->sid);

			/*
			 * send a hello, with a voice count, then request starting of the
			 * synthtype. All of these will require an ACK, and we should wait
			 * here and read that ack before proceeding with each next step.
			 */
			manual.flags |= BRISTOL_CONN_FORCE|BRIGHTON_NOENGINE;

			synth->midichannel++;
			synth->synthtype = BRISTOL_HAMMONDB3;
			if ((synth->sid2 = initConnection(&manual, synth)) < 0)
				exit(0);
printf("LM CID: %i\n", synth->sid2);
			synth->midichannel--;
		} else {
			if ((synth->sid2 = initConnection(&manual, synth)) < 0)
				exit(0);
printf("LM CID: %i\n", synth->sid2);
		}
	}

	for (i = 0; i < DEVICE_COUNT; i++)
	{
		synth->dispatch[i].routine = (synthRoutine) hammondB3Passthrough;
	}

	/*
	 * Put in the drawbar configurations, upper manual
	 */
	dispatch[0].routine =
	dispatch[1].routine =
	dispatch[2].routine =
	dispatch[3].routine =
	dispatch[4].routine =
	dispatch[5].routine =
	dispatch[6].routine =
	dispatch[7].routine =
	dispatch[8].routine = (synthRoutine) doLMSlider;

	dispatch[0].controller = 0;
	dispatch[1].controller = 1;
	dispatch[2].controller = 2;
	dispatch[3].controller = 3;
	dispatch[4].controller = 4;
	dispatch[5].controller = 5;
	dispatch[6].controller = 6;
	dispatch[7].controller = 7;
	dispatch[8].controller = 8;

	/*
	 * Put in the drawbar configurations, upper manual
	 */
	dispatch[SLIDER_START].routine =
	dispatch[SLIDER_START + 1].routine =
	dispatch[SLIDER_START + 2].routine =
	dispatch[SLIDER_START + 3].routine =
	dispatch[SLIDER_START + 4].routine =
	dispatch[SLIDER_START + 5].routine =
	dispatch[SLIDER_START + 6].routine =
	dispatch[SLIDER_START + 7].routine =
	dispatch[SLIDER_START + 8].routine = (synthRoutine) doSlider;

	dispatch[SLIDER_START].controller = 0;
	dispatch[SLIDER_START + 1].controller = 1;
	dispatch[SLIDER_START + 2].controller = 2;
	dispatch[SLIDER_START + 3].controller = 3;
	dispatch[SLIDER_START + 4].controller = 4;
	dispatch[SLIDER_START + 5].controller = 5;
	dispatch[SLIDER_START + 6].controller = 6;
	dispatch[SLIDER_START + 7].controller = 7;
	dispatch[SLIDER_START + 8].controller = 8;

	// Options controllers
	dispatch[OPTS_START].routine = (synthRoutine) hammondOption;
	dispatch[OPTS_START].controller = OPTS_START;
	dispatch[OPTS_START].operator = 1;
	dispatch[OPTS_START + 1].routine = (synthRoutine) hammondOption;
	dispatch[OPTS_START + 1].controller = OPTS_START;
	dispatch[OPTS_START + 1].operator = 2;
	dispatch[OPTS_START + 2].routine = (synthRoutine) hammondOption;
	dispatch[OPTS_START + 2].controller = OPTS_START;
	dispatch[OPTS_START + 2].operator = 3;
	dispatch[OPTS_START].other1 = -1;
	dispatch[OPTS_START].other2 = 0;
	dispatch[OPTS_START + 3].routine = (synthRoutine) hammondOption;
	dispatch[OPTS_START + 3].controller = OPTS_START + 3;

	dispatch[OPTS_START + 4].routine = (synthRoutine) hammondOption;
	dispatch[OPTS_START + 4].controller = OPTS_START + 4;
	dispatch[OPTS_START + 5].routine = (synthRoutine) hammondOption;
	dispatch[OPTS_START + 5].controller = OPTS_START + 5;
	dispatch[OPTS_START + 6].routine = (synthRoutine) hammondOption;
	dispatch[OPTS_START + 6].controller = OPTS_START + 6;
	dispatch[OPTS_START + 7].routine = (synthRoutine) hammondOption;
	dispatch[OPTS_START + 7].controller = OPTS_START + 7;
	dispatch[OPTS_START + 8].routine = (synthRoutine) hammondOption;
	dispatch[OPTS_START + 8].controller = OPTS_START + 8;
	dispatch[OPTS_START + 9].routine = (synthRoutine) hammondOption;
	dispatch[OPTS_START + 9].controller = OPTS_START + 9;
	dispatch[OPTS_START + 10].routine = (synthRoutine) hammondOption;
	dispatch[OPTS_START + 10].controller = OPTS_START + 10;
	dispatch[OPTS_START + 11].routine = (synthRoutine) hammondOption;
	dispatch[OPTS_START + 11].controller = OPTS_START + 11;
	dispatch[OPTS_START + 12].routine = (synthRoutine) hammondOption;
	dispatch[OPTS_START + 12].controller = OPTS_START + 12;

	// vibra
	dispatch[OPTS_START + 13].routine = (synthRoutine) hammondOption;
	dispatch[OPTS_START + 13].controller = OPTS_START + 13;
	dispatch[OPTS_START + 14].routine = (synthRoutine) hammondOption;
	dispatch[OPTS_START + 14].controller = OPTS_START + 14;
	dispatch[OPTS_START + 15].routine = (synthRoutine) hammondOption;
	dispatch[OPTS_START + 15].controller = OPTS_START + 15;
	dispatch[DEVICE_START + 20].routine = (synthRoutine) doVibra;

	// Percussives
	dispatch[OPTS_START + 16].routine = (synthRoutine) hammondOption;
	dispatch[OPTS_START + 16].controller = OPTS_START + 16;
	dispatch[OPTS_START + 17].routine = (synthRoutine) hammondOption;
	dispatch[OPTS_START + 17].controller = OPTS_START + 17;
	dispatch[22].routine = (synthRoutine) doPerc;
	dispatch[23].routine = (synthRoutine) doPerc;
	dispatch[24].routine = (synthRoutine) doPerc;

	// Soft attack - grooming
	dispatch[OPTS_START + 18].routine = (synthRoutine) hammondOption;
	dispatch[OPTS_START + 18].controller = OPTS_START + 18;
	dispatch[OPTS_START + 19].routine = (synthRoutine) hammondOption;
	dispatch[OPTS_START + 19].controller = OPTS_START + 19;
	dispatch[DEVICE_START + 25].routine = (synthRoutine) doGrooming;

	// preacher
	dispatch[OPTS_START + 20].routine = (synthRoutine) doPreacher;
	dispatch[OPTS_START + 20].controller = OPTS_START + 20;
	dispatch[OPTS_START + 21].routine = (synthRoutine) hammondOption;
	dispatch[OPTS_START + 21].controller = OPTS_START + 21;
	dispatch[OPTS_START + 22].routine = (synthRoutine) hammondOption;
	dispatch[OPTS_START + 22].controller = OPTS_START + 22;
	dispatch[DEVICE_START + 21].routine = (synthRoutine) doBright;
	dispatch[OPTS_START + 23].routine = (synthRoutine) hammondOption;
	dispatch[OPTS_START + 23].controller = OPTS_START + 23;

	// reverb
	dispatch[OPTS_START + 24].routine = (synthRoutine) hammondOption;
	dispatch[OPTS_START + 24].controller = OPTS_START + 24;
	dispatch[OPTS_START + 25].routine = (synthRoutine) hammondOption;
	dispatch[OPTS_START + 25].controller = OPTS_START + 25;
	dispatch[OPTS_START + 26].routine = (synthRoutine) hammondOption;
	dispatch[OPTS_START + 26].controller = OPTS_START + 26;
	dispatch[DEVICE_START + 19].routine = (synthRoutine) doReverb;

	dispatch[DEVICE_START + 18].routine = (synthRoutine) doLeslieSpeed;
	dispatch[LESLIE_START].routine = (synthRoutine) doLeslieSpeed;

	dispatch[VOL_START].routine = (synthRoutine) doVolume;

	// Memory/Midi buttons
	dispatch[MEM_START + 10].controller = 12;
	dispatch[MEM_START + 10].operator = 0;

	/*
	 * These are for the memory radio buttons
	 */
	dispatch[MEM_START].other1 = 0;
	dispatch[MEM_START].other2 = 0;

	dispatch[MEM_START].operator = 0;
	dispatch[MEM_START + 1].operator = 1;
	dispatch[MEM_START + 2].operator = 2;
	dispatch[MEM_START + 3].operator = 3;
	dispatch[MEM_START + 4].operator = 4;
	dispatch[MEM_START + 5].operator = 5;
	dispatch[MEM_START + 6].operator = 6;
	dispatch[MEM_START + 7].operator = 7;
	dispatch[MEM_START + 8].operator = 8;
	dispatch[MEM_START + 9].operator = 9;

	dispatch[MEM_START].routine = dispatch[MEM_START + 1].routine =
	dispatch[MEM_START + 2].routine = dispatch[MEM_START + 3].routine =
	dispatch[MEM_START + 4].routine = dispatch[MEM_START + 5].routine =
	dispatch[MEM_START + 6].routine = dispatch[MEM_START + 7].routine =
	dispatch[MEM_START + 8].routine = dispatch[MEM_START + 9].routine =
		(synthRoutine) hammondB3Memory;

	/*
	 * Mem load and save
	 */
	dispatch[MEM_START + 12].controller = 1;
	dispatch[MEM_START + 13].controller = 2;
	dispatch[MEM_START + 14].controller = 3;
	dispatch[MEM_START + 12].routine = dispatch[MEM_START + 13].routine =
		dispatch[MEM_START + 14].routine = (synthRoutine) hammondB3Memory;

	/*
	 * Midi up/down
	 */
	dispatch[MEM_START + 10].controller = 2;
	dispatch[MEM_START + 11].controller = 1;
	dispatch[MEM_START + 10].routine = dispatch[MEM_START + 11].routine =
		(synthRoutine) hammondB3Midi;

	bristolMidiSendMsg(global.controlfd, synth->sid, 1, 0, 2);
	bristolMidiSendMsg(global.controlfd, synth->sid, 1, 1, 3);
	bristolMidiSendMsg(global.controlfd, synth->sid, 1, 2, 16383);
	bristolMidiSendMsg(global.controlfd, synth->sid, 1, 3, 10);
	bristolMidiSendMsg(global.controlfd, synth->sid, 1, 4, 13000);
	bristolMidiSendMsg(global.controlfd, synth->sid, 1, 5, 0);

	bristolMidiSendMsg(global.controlfd, synth->sid, 12, 7, 1);
	bristolMidiSendMsg(global.controlfd, synth->sid, 10, 0, 4);

	bristolMidiSendMsg(global.controlfd, synth->sid, 3, 0, 0);
	bristolMidiSendMsg(global.controlfd, synth->sid, 3, 1, 10);
	bristolMidiSendMsg(global.controlfd, synth->sid, 3, 2, 13000);
	bristolMidiSendMsg(global.controlfd, synth->sid, 3, 3, 20);
	bristolMidiSendMsg(global.controlfd, synth->sid, 3, 4, 13000);
	bristolMidiSendMsg(global.controlfd, synth->sid, 3, 5, 0);

	bristolMidiSendMsg(global.controlfd, synth->sid, 4, 0, 2);
	bristolMidiSendMsg(global.controlfd, synth->sid, 4, 1, 1200);
	bristolMidiSendMsg(global.controlfd, synth->sid, 4, 2, 0);
	bristolMidiSendMsg(global.controlfd, synth->sid, 4, 3, 20);
	bristolMidiSendMsg(global.controlfd, synth->sid, 4, 4, 13000);
	bristolMidiSendMsg(global.controlfd, synth->sid, 4, 5, 0);

	bristolMidiSendMsg(manual.controlfd, synth->sid2, 1, 0, 2);
	bristolMidiSendMsg(manual.controlfd, synth->sid2, 1, 1, 3);
	bristolMidiSendMsg(manual.controlfd, synth->sid2, 1, 2, 16383);
	bristolMidiSendMsg(manual.controlfd, synth->sid2, 1, 3, 10);
	bristolMidiSendMsg(manual.controlfd, synth->sid2, 1, 4, 13000);
	bristolMidiSendMsg(manual.controlfd, synth->sid2, 1, 5, 0);

	dispatch[LESLIE_START + 1].routine = (synthRoutine) hammondB3Switch;
}

/*
 * This will be called to make any routine specific parameters available.
 */
static int
hammondB3Configure(int cid)
{
	guiSynth *synth = findSynth(global.synths, cid);
	brightonEvent event;

	if (synth == 0)
	{
		printf("problems going operational\n");
		return(-1);
	}

	if (synth->flags & OPERATIONAL)
		return;

printf("	going operational: %x, %x\n", synth, cid);

	synth->flags |= OPERATIONAL;
	synth->transpose = 24;

	synth->bank = 0;
	synth->location = 0;
	loadMemory(synth, "hammondB3", 0, 0, synth->mem.active,
		FIRST_DEV, 0);

	bristolMidiSendMsg(global.controlfd, synth->midichannel,
		BRISTOL_EVENT_KEYON, 0, 10 + synth->transpose);
	bristolMidiSendMsg(global.controlfd, synth->midichannel,
		BRISTOL_EVENT_KEYOFF, 0, 10 + synth->transpose);
}

