/*  $Id: BurgerSpaceEngine.h,v 1.3 2002/04/03 06:11:27 sarrazip Exp $
    BurgerSpaceEngine.h - Main engine

    burgerspace - A hamburger-smashing video game.
    Copyright (C) 2001 Pierre Sarrazin <http://sarrazip.com/>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
    02111-1307, USA.
*/

#ifndef _H_BurgerSpaceEngine
#define _H_BurgerSpaceEngine

#include "IngredientGroup.h"
#include "IngredientSprite.h"

#include <gengameng/GameEngine.h>
#include <gengameng/Sprite.h>

#include <string>

using namespace std;

typedef char **XPM;


class BurgerSpaceEngine : public GameEngine
/*  A hamburger-smashing video game.
*/
{
public:

    typedef PixmapArray::Pixmap Pixmap;


    enum { RIGHT, UP, LEFT, DOWN };
    /*	Possible directions of movement, in trigonometric order.
	This enumeration is garanteed to start at zero.
	The order of the constants is guaranteed, so they are 0, 1, 2, 3.
    */

    BurgerSpaceEngine(const string &windowManagerCaption, int initLevelNumber)
							    throw(int, string);
    /*  See base class.

	'windowManagerCaption' must contain the title to give to the window.

	The game will start at level 'initLevelNumber'.
	This must be a positive integer.

	Throws a string or an integer code if an error occurs.
    */

    virtual ~BurgerSpaceEngine();
    /*  Nothing interesting.
    */

    virtual void processKey(SDLKey keysym, bool pressed);
    /*  Inherited.
    */

    virtual bool tick();
    /*  Inherited.
    */

private:

    ///////////////////////////////////////////////////////////////////////////
    //
    //  LOCAL TYPES, CLASSES AND CONSTANTS
    //
    //


    enum Key
    {
	NOKEY = 0,

	// The next four constants are guaranteed to have the values 1 to 4:
	UP_ARROW,
	DOWN_ARROW,
	LEFT_ARROW,
	RIGHT_ARROW,

	LETTER_P,
	ENTER,
	SPACE,
	CTRL,
	ALT,
	ESC,

	NUM_SUPPORTED_KEYS
    };


    struct IntPair
    {
	int first, second;
    };

    
    struct IntQuad
    {
	int first, second, third, fourth;
    };


    class Level
    {
    public:
	Couple sizeInTiles;
	Couple sizeInPixels;
	Couple positionInPixels;

	Level();
	~Level();

	void init(int no, int nCols, int nRows, Couple posInPixels);
	void setLevelNo(int no);
	int  getLevelNo() const;
	void setTileMatrixEntry(int colNo, int rowNo,
				XPM xpm, Pixmap pixmap);
	Pixmap *getTileMatrixRow(int rowNo);
	XPM getXPMAtPixel(Couple pos) const;

    private:
	int levelNo;
	XPM *xpmMatrix;   // vector of XPM pointers
	Pixmap *tileMatrix;  // vector of X11 pixmaps
    };

    class IngInit
    {
    public:
	enum IngType
	{
	    BOTTOM_BUN, MEAT, LETTUCE, RED_STUFF, YELLOW_STUFF, TOP_BUN

	    /*	The red stuff is a slice of tomato, and the yellow stuff
		is cheese.  This was not known at the time when this
		enumeration was defined...
	    */
	};

	int xInitTile, yInitTile, yTargetTile, rank;
	IngType type;
    };


    ///////////////////////////////////////////////////////////////////////////
    //
    //  DATA MEMBERS
    //
    //

    static const char
	*levelDescriptor1[],
	*levelDescriptor2[],
	*levelDescriptor3[],
	*levelDescriptor4[],
	*levelDescriptor5[],
	*levelDescriptor6[];

    static const char **levelDescriptorTable[];

    static const IngInit
	tableIngredientsLevel1[],
	tableIngredientsLevel2[],
	tableIngredientsLevel3[],
	tableIngredientsLevel4[],
	tableIngredientsLevel5[],
	tableIngredientsLevel6[];
    
    static const IngInit *tableOfTablesOfIngredientsLevel[];

    static const IntQuad enemyStartingHeights[];
    static const IntPair playerStartingPos[];


    int initLevelNo;
    int cumulLevelNo;

    bool paused;
    unsigned long tickCount;

    Pixmap tilePixmaps[5];


    /*  PLAYER:
    */
    bool userControlledPlayer;
    Couple initPlayerPos;  // initial player position in a level
    PixmapArray playerPA;
    Sprite *playerSprite;
    int lastPlayerDirection;

    PixmapArray pepperPA;
    SpriteList  pepperSprites;


    /*  ENEMIES:
    */
    unsigned long timeForNewEnemy;
    		// tick count at which a new enemy must be created;
		// 0 means none

    PixmapArray eggPA;
    PixmapArray hotdogPA;
    PixmapArray picklePA;
    SpriteList  enemySprites;


    /*  INGREDIENTS:
    */
    PixmapArray topBunPA;
    PixmapArray lettucePA;
    PixmapArray meatPA;
    PixmapArray redStuffPA;
    PixmapArray yellowStuffPA;
    PixmapArray bottomBunPA;
    IngredientSpriteList ingredientSprites;  // owns the contained objects
    IngredientGroupList  ingredientGroups;


    /*  TREATS (icecream, etc, that the player eats to get a pepper):
    */
    PixmapArray treatPA;
    SpriteList treatSprites;
    int timeForTreat;


    /*  DIGITS:
    */
    PixmapArray digitPA;
    SpriteList scoreSprites;


    /*  KEYBOARD COMMANDS:
    */
    bool escapePressed;
    bool leftArrowPressed;
    bool rightArrowPressed;
    bool upArrowPressed;
    bool downArrowPressed;
    bool spacePressed, spacePressedBefore;
    bool ctrlPressed, ctrlPressedBefore;
    bool letterPPressed, letterPPressedBefore;

    int    numHamburgersToDo;
    long   thePeakScore;  // player's best score yet in this game

    long   theScore;      // player's score in points
    bool   updateScore;   // indicates if player's score needs to be rewritten
    Couple scoreAreaPos;
    Couple scoreAreaSize;

    bool   celebrationMode;  // used when player has just won the level

    int    numLives;  // number of player lives left
    bool   updateNumLives;
    Couple numLivesAreaPos;
    Couple numLivesAreaSize;

    int    numAvailablePeppers;  // number of pepper shots available to player
    bool   updateNumAvailablePeppers;
    Couple numAvailablePeppersAreaPos;
    Couple numAvailablePeppersAreaSize;

    bool   updateLevelNo;
    Couple levelNoAreaPos;
    Couple levelNoAreaSize;

    Level theCurrentLevel;


    ///////////////////////////////////////////////////////////////////////////
    //
    //  IMPLEMENTATION FUNCTIONS
    //
    //
    bool isPepperRequested() const;
    void chooseDirectionAmongMany(bool directions[4]) const;
    int  chooseDirectionTowardTarget(
			    Couple startPos, Couple targetPos, int speedFactor,
			    const bool allowedDirections[4]) const;
    void avoidEnemies(bool desiredDirs[4]) const;
    void getPlayerDesiredDirections(bool desiredDirs[4]) const;

    bool animatePlayer();
    void animateAutomaticCharacters();
    void detectCollisions();
    void restoreBackground();
    void drawSprites();

    void putSprite(const Sprite &s);
    void showInstructions();
    void initGameParameters();
    void initNextLevel(int levelNo = 0) throw(int);
    void startNewLife();
    void animateTemporarySprites(SpriteList &slist) const;
    void givePlayerPepper();
    void makePlayerWin();
    void makePlayerDie();
    void releaseAllCarriedEnemies();
    void detectEnemyCollisions(SpriteList &slist);
    void detectCollisionBetweenIngredientGroupAndEnemyList(
	    const IngredientGroup &aGroup, SpriteList &enemies);
    bool ingredientGroupCollidesWithSprite(
			const Couple groupPos, const Couple groupSize,
			const Sprite &s) const;
    void loadPixmaps() throw(PixmapLoadError);
    void loadLevel(int levelNo) throw(string);
    void displayErrorMessage(const string &msg) throw();
    void initializeSprites() throw(PixmapLoadError);
    void initializeMisc() throw(string);
    void killSpritesInList(SpriteList &sl);
    IngredientGroup *findIngredientGroupRightBelow(
					const IngredientGroup &upperGroup);
    bool isIngredientSpriteOnFloor(const IngredientSprite &s) const;
    bool spriteTouchesIngredientGroup(
			    const Sprite &s, const IngredientGroup &g) const;
    size_t carryEnemies(IngredientGroup &g);
    size_t carryEnemiesInList(IngredientGroup &g, SpriteList &slist);
    size_t releaseCarriedEnemies(IngredientGroup &g);
    void createScoreSprites(long n, Couple center);
    void moveEnemyList(SpriteList &slist, int speedFactor);
    Couple getDistanceToPerfectPos(const Sprite &s) const;
    Couple determineAllowedDirections(const Sprite &s,
				    int speedFactor, int tolerance,
				    bool allowedDirections[4]) const;
    Couple attemptMove(const Sprite &s, bool attemptLeft, bool attemptRight,
			bool attemptUp, bool attemptDown,
			int speedFactor) const;
    bool positionAllowsLeftMovement(Couple pos, Couple size) const;
    bool positionAllowsRightMovement(Couple pos, Couple size) const;
    bool spriteBottomCenterIsOnLadder(const Sprite &s) const;
    bool spriteBottomCenterIsOverLadder(const Sprite &s) const;
    void drawComplexEnemySprites(const SpriteList &slist, int oscBit);
    void addToScore(long n);
    void addToNumLives(int n);
    void initTimeForTreat();
    void displayPauseMessage(bool display);
    void displayStartMessage(bool display);
    void displayMessage(const char *msg);


    /*	Forbidden operations:
    */
    BurgerSpaceEngine(const BurgerSpaceEngine &);
    BurgerSpaceEngine &operator = (const BurgerSpaceEngine &);
};


#endif  /* _H_BurgerSpaceEngine */
