(**************************************************************************)
(*                   Cameleon                                             *)
(*                                                                        *)
(*      Copyright (C) 2002 Institut National de Recherche en Informatique et   *)
(*      en Automatique. All rights reserved.                              *)
(*                                                                        *)
(*      This program is free software; you can redistribute it and/or modify  *)
(*      it under the terms of the GNU General Public License as published by  *)
(*      the Free Software Foundation; either version 2 of the License, or  *)
(*      any later version.                                                *)
(*                                                                        *)
(*      This program is distributed in the hope that it will be useful,   *)
(*      but WITHOUT ANY WARRANTY; without even the implied warranty of    *)
(*      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     *)
(*      GNU General Public License for more details.                      *)
(*                                                                        *)
(*      You should have received a copy of the GNU General Public License  *)
(*      along with this program; if not, write to the Free Software       *)
(*      Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA          *)
(*      02111-1307  USA                                                   *)
(*                                                                        *)
(*      Contact: Maxence.Guesdon@inria.fr                                *)
(**************************************************************************)

(** Gui to describe a report. *)

open Rep_desc
open GdkKeysyms
let print_DEBUG s = print_string s ; print_newline ()

module C = Configwin

type attribute = {
    mutable att_name : string ;
    mutable att_code : string ;
  } 

let params_for_att a = 
  let param_name = C.string
      ~f: (fun s -> a.att_name <- s)
      Rep_messages.name a.att_name 
  in
  let param_code = C.string
      ~f: (fun s -> a.att_code <- s)
      Rep_messages.code a.att_code
  in
  [param_name ; param_code] 

let tag_attributes_param tag =
  let f_string a = [ a.att_name ; a.att_code ] in
  let f_edit a =
    ignore (C.simple_get Rep_messages.edit (params_for_att a));
    a
  in
  let f_add () =
    let a = { att_name = "" ; att_code = "" } in
    match C.simple_get Rep_messages.edit (params_for_att a) with
      C.Return_cancel -> []
    | C.Return_apply
    | C.Return_ok -> [a]
  in
  let f_eq a1 a2 = a1.att_name = a2.att_name in
  let p = C.list
      ~f: (fun atts -> 
	    tag.atts <- List.map (fun a -> (a.att_name, a.att_code)) atts
	  )
      ~eq: f_eq
      ~edit: f_edit
      ~add: f_add
      ~titles: [ Rep_messages.name ; Rep_messages.code ] 
      Rep_messages.attributes
      f_string
      (List.map (fun (n,c) -> { att_name = n ; att_code = c}) tag.atts)
  in
  p

(** The selected report element and its item, with the parent element and its tree. *)
type t_selected_node = 
    report_ele * GTree.tree_item * GMisc.label * 
      report_ele option * GTree.tree

class gui data =
  (* The main vbox *)
  let box = GPack.vbox () in
  (* The menu bar *)
  let menubar = GMenu.menu_bar ~packing:(box#pack ~expand:false) () in
  (* we define the menus *)
  let factory = new GMenu.factory menubar in
  let accel_group = factory#accel_group in

  let menu_file = factory#add_submenu Rep_messages.mnFile in
  let menu_edit = factory#add_submenu Rep_messages.mnEdit in
  let menu_node = factory#add_submenu Rep_messages.mnNode in
  let menu_help = factory#add_submenu Rep_messages.mnHelp in

  let factory_file = new GMenu.factory ~accel_group: accel_group menu_file in
  let itemSave = factory_file#add_item ~key:_S Rep_messages.mnSave in
  let _ = factory_file#add_separator in
  let itemQuit = factory_file#add_item ~key:_Q Rep_messages.mnQuit in

  let factory_edit = new GMenu.factory ~accel_group: accel_group menu_edit in
  let itemCopy = factory_edit#add_item ~key:_C Rep_messages.mnCopy in
  let itemCut = factory_edit#add_item ~key:_X Rep_messages.mnCut in
  let itemPaste = factory_edit#add_item ~key:_V Rep_messages.mnPaste in

  let factory_node = new GMenu.factory ~accel_group: accel_group menu_node in
  let itemEdit_header = factory_node#add_item ~key:_U Rep_messages.mnEdit_header in
  let _ = factory_node#add_separator in
  let itemEdit_parameters = factory_node#add_item ~key:_U Rep_messages.mnEdit_params in
  let _ = factory_node#add_separator in
  let itemUp = factory_node#add_item ~key:_U Rep_messages.mnUp in
  let itemEdit_selected = factory_node#add_item ~key:_E Rep_messages.mnEdit_selected in
  let _ = factory_node#add_separator in
  let itemInsert_leaf = factory_node#add_item ~key:_L Rep_messages.mnInsert_leaf in
  let itemInsert_tag = factory_node#add_item ~key:_T Rep_messages.mnInsert_tag in
  let itemInsert_list = factory_node#add_item ~key:_F Rep_messages.mnInsert_list in
  let itemInsert_cond = factory_node#add_item ~key:_I Rep_messages.mnInsert_cond in
  let itemInsert_sub = factory_node#add_item ~key:_R Rep_messages.mnInsert_sub in
  let itemInsert_mark = factory_node#add_item ~key:_M Rep_messages.mnInsert_mark in

  let pane = GPack.paned `HORIZONTAL
      ~packing: (box#pack ~expand: true) ()
  in
  let wscroll_params = 
    GBin.scrolled_window 
      ~hpolicy: `AUTOMATIC
      ~vpolicy: `AUTOMATIC
      ~packing: (pane#add1) () 
  in
  let wlist_params = GList.clist 
      ~titles: [ Rep_messages.parameters ]
      ~titles_show: true
      ~selection_mode: `SINGLE
      ~packing: wscroll_params#add () 
  in
  (* The scroll window used for the tree *)
  let wscroll_tree = 
    GBin.scrolled_window
      ~hpolicy: `AUTOMATIC
      ~vpolicy: `AUTOMATIC
      ~packing: (pane#add2) () 
  in
  (* The tree view *)
  let wtree = GTree.tree ~packing: (wscroll_tree#add_with_viewport) () in

  object (self)
    val mutable selected_node = (None : t_selected_node option)

    method box = box

    method item_label r_ele =
      match r_ele with
	Leaf leaf -> leaf.leaf
      |	Tag t -> 
	  "<"^t.tag^
	  (List.fold_left
	     (fun acc -> fun (att,v) -> acc^" "^att^"="^v^"")
	     ""
	     t.atts
	  )^
	  ">"
      |	Mark m -> "<a name=\""^m.mark_name^"\"> ["^m.mark_id^"]"
      |	List l -> "for "^l.var^" in "^l.f
      |	Cond cond -> "if "^cond.cond
      |	Sub s -> s.sub_code
      |	Then _ -> "then"
      |	Else _ -> "else"
	  
    method set_widget_font w r_ele =
      match r_ele with
	Tag _ | Mark _ ->
	  let bold = Gdk.Font.load_fontset "7x13bold" in
	  let style = w#style#copy in
	  w#set_style style;
	  style#set_font bold

      |	List _ | Else _ | Then _ | Cond _ ->
	  let courier = Gdk.Font.load_fontset "fixed" in
	  let style = w#style#copy in
	  w#set_style style;
	  style#set_font courier
      |	Leaf _ | Sub _ ->
	  let courier = Gdk.Font.load_fontset "fixed" in
	  let style = w#style#copy in
	  w#set_style style;
	  style#set_font courier

    method set_widget_color (w : GObj.misc_ops) r_ele =
      match r_ele with
	Tag _ | Mark _ ->
	  let style = w#style#copy in
	  w#set_style style;
	  style#set_bg [ (`NORMAL, `NAME "SkyBlue") ;
		         (`INSENSITIVE, `NAME "SkyBlue") ;
			 (`PRELIGHT, `NAME "SkyBlue") ;
			 (`ACTIVE, `NAME "SkyBlue") ;
			 (`SELECTED, `NAME "SkyBlue")
		       ] ;
      |	List _ | Else _ | Then _ | Cond _ ->
	  let style = w#style#copy in
	  w#set_style style;
	  (* set_bg ne fait rien *)
	  style#set_bg [ (`NORMAL, `NAME "White") ;
		         (`INSENSITIVE, `NAME "White") ;
			 (`PRELIGHT, `NAME "White") ;
			 (`ACTIVE, `NAME "White") ;
			 (`SELECTED, `NAME "Black")
		       ] 
      |	Leaf _ ->
	  let style = w#style#copy in
	  w#set_style style;
	  (* set_bg ne fait rien *)
	  style#set_bg [ (`NORMAL, `NAME "MediumAquamarine") ;
		         (`INSENSITIVE, `NAME "MediumAquamarine") ;
			 (`PRELIGHT, `NAME "MediumAquamarine") ;
			 (`ACTIVE, `NAME "MediumAquamarine") ;
			 (`SELECTED, `NAME "MediumAquamarine")
		       ]
      |	Sub _ ->
	  let style = w#style#copy in
	  w#set_style style;
	  (* set_bg ne fait rien *)
	  style#set_bg [ (`NORMAL, `NAME "DarkKhaki") ;
		         (`INSENSITIVE, `NAME "DarkKhaki") ;
			 (`PRELIGHT, `NAME "DarkKhaki") ;
			 (`ACTIVE, `NAME "DarkKhaki") ;
			 (`SELECTED, `NAME "DarkKhaki")
		       ]

    method insert_report_ele ?pos wt parent_opt r_ele =
      let item = GTree.tree_item () in
      let ali = GBin.alignment ~x:0. ~xscale:0. ~packing:item#add () in
(*      let ev = GBin.event_box ~packing:ali#add () in*)
      let label = GMisc.label ~packing:ali#add ~text: (self#item_label r_ele) () in
(*      self#set_widget_color ev#misc r_ele ;*)
      self#set_widget_font label#misc r_ele ;

      let _ = 
	match pos with 
	  None -> wt#append item 
	| Some n -> wt#insert item ~pos: n
      in
      let children =
	match r_ele with
	  Leaf _
	| Sub _ 
	| Mark _ -> []
	| Cond c -> [ Then c ; Else c]
	| Then c -> c.subs_then
	| Else c -> c.subs_else
	| Tag t -> t.tag_subs
	| List l -> l.list_subs
      in
      (match children with
	[] -> ()
      | _ -> 
	  let wt_sub = GTree.tree () in
	  let _ = item#set_subtree wt_sub in
	  let _ = item#expand () in
	  List.iter (self#insert_report_ele wt_sub (Some r_ele)) children
      );
      let _ = item#connect#select
	  (fun () -> 
	    print_DEBUG ("selection "^(self#item_label r_ele));
	    selected_node <- Some (r_ele, item, label, parent_opt, wt)
	  )
      in
      let _ = item#connect#deselect
	  (fun () -> 
	    print_DEBUG "deselection";
	    selected_node <- None
	  ) 
      in
      ()

    method update_params =
      wlist_params#clear () ;
      List.iter
	(fun p -> 
	  let _ = wlist_params#append [p] in
	  ()
	)
	(data#report).rep_params

    method update =
      let report = data#report in
      let _ = wtree#remove_items (wtree#children) in
      self#update_params ;
      List.iter (self#insert_report_ele wtree None) report.rep_eles

    method copy () =
      match selected_node with
	None -> ()
      |	Some (ele, _, _, _, _) -> 
	  Rep_buffer.copy ele

    method cut () =
      match selected_node with
	None -> ()
      |	Some (ele, item, _, parent_opt, tree) ->
	  selected_node <- None ;
	  Rep_buffer.copy ele ;
	  data#remove_from_parent ele parent_opt ;
	  tree#remove_items [item]

    method paste_append () =
      match Rep_buffer.paste () with
	None -> ()
      | Some ele_paste ->
	  let (ele_opt, tree_opt) = 
	    match selected_node with
	      None -> (None, Some wtree)
	    | Some (Cond _, _, _, _, _) ->
		(* don't append anything to Cond *)
		(None, None)
	    | Some (ele, item, _, parent_opt, tree) -> 
	        (* when we remove the last item of a tree,
		  this tree gets deassociated from its parent item.
	           We recreate a subtree here, if needed *)
		let sub_opt = 
		  match item#subtree with
		    None ->
		      (match ele with 
		      | Tag _ | List _ | Else _ | Then _ ->
			  print_DEBUG "recreate subtree";
			  let wt_sub = GTree.tree () in
			  let _ = item#set_subtree wt_sub in
			  let _ = item#expand in
			  Some (wt_sub)
		      | Leaf _ | Mark _ | Sub _ ->
			  print_DEBUG "dont recreate subtree (1)" ;
			  None
		      |	Cond _ -> 
			  None (* sould not occur *)
		      )
		  | Some wt -> 
		      print_DEBUG "dont recreate subtree (2)" ;
		      Some wt
		in
		(Some ele, sub_opt)
	  in
	  match tree_opt with
	    None -> 		
	      print_DEBUG "paste_append : no subtree" ;
	      ()
	  | Some tree ->
	      print_DEBUG "paste_append : some subtree" ;
	      data#append_in_parent ele_paste ele_opt ;
	      self#insert_report_ele tree ele_opt ele_paste

    method up () =
      match selected_node with
	None -> ()
      | Some (ele, item, _, parent_opt, tree) -> 
	    let pos = tree#child_position item in
	    if pos < 1 then
	      ()
	    else
	      (
	       data#up_element ele parent_opt ;
	       tree#remove_items [item] ;
	       self#insert_report_ele ~pos: (pos - 1) tree parent_opt ele
	      )

    method insert_ele ele =
      let old_buffer = Rep_buffer.paste () in
      Rep_buffer.copy ele ;
      self#paste_append () ;
      (match old_buffer with None -> () | Some o -> Rep_buffer.copy o)

    method private params_for_leaf leaf =
      let param = C.string
	  ~f: (fun s -> leaf.leaf <- s)
	  Rep_messages.fun_unit leaf.leaf
      in
      [param]

    method private params_for_sub sub =
      let param = C.string
	  ~f: (fun s -> sub.sub_code <- s)
	  Rep_messages.code sub.sub_code
      in
      [param]

    method private params_for_tag tag =
      let param_tag = C.string
	  ~f: (fun s -> tag.tag <- s)
	  Rep_messages.tag tag.tag
      in
      let param_atts = tag_attributes_param tag in
      [param_tag ; param_atts]

    method private params_for_mark mark =
      let param_id = C.string
	  ~f: (fun s -> mark.mark_id <- s)
	  Rep_messages.ocaml_id mark.mark_id
      in
      let param_name = C.string
	  ~f: (fun s -> mark.mark_name <- s)
	  Rep_messages.name mark.mark_name
      in
      [param_id ; param_name] 

    method private params_for_list list =
      let param_f = C.string
	  ~f: (fun s -> list.f <- s)
	  Rep_messages.fun_unit list.f 
      in
      let param_var = C.string
	  ~f: (fun s -> list.var <- s)
	  Rep_messages.ocaml_id list.var
      in
      [param_var ; param_f] 

    method private params_for_cond cond =
      let param = C.string
	  ~f: (fun s -> cond.cond <- s)
	  Rep_messages.fun_unit cond.cond
      in
      [param]

    method insert_leaf () =
      let leaf = { leaf = "" } in
      match C.simple_get Rep_messages.mnInsert_leaf 
	  (self#params_for_leaf leaf)
      with
	C.Return_ok -> self#insert_ele (Leaf leaf)
      |	_ -> ()

    method insert_sub () =
      let sub = { sub_code = "" } in
      match C.simple_get Rep_messages.mnInsert_sub
	  (self#params_for_sub sub)
      with
	C.Return_ok -> self#insert_ele (Sub sub)
      |	_ -> ()

    method insert_tag () =
      let tag = { tag = "" ; atts = [] ; tag_subs = []} in
      match C.simple_get Rep_messages.mnInsert_tag 
	  (self#params_for_tag tag) 
      with
	C.Return_ok -> self#insert_ele (Tag tag)
      |	_ -> ()

    method insert_mark () =
      let mark = { mark_id = "" ; mark_name = ""} in
      match C.simple_get Rep_messages.mnInsert_mark
	  (self#params_for_mark mark)
      with
	C.Return_ok -> self#insert_ele (Mark mark)
      |	_ -> ()

    method insert_list () =
      let list = { f = "" ; var = "" ; list_subs = []} in
      match C.simple_get Rep_messages.mnInsert_list
	  (self#params_for_list list)
      with
	C.Return_ok -> self#insert_ele (List list)
      |	_ -> ()

    method insert_cond () =
      let cond = { cond = "" ; subs_then = [] ; subs_else = []} in
      match C.simple_get Rep_messages.mnInsert_cond
	  (self#params_for_cond cond)
      with
	C.Return_ok -> self#insert_ele (Cond cond)
      |	_ -> ()

    method edit_selected () =
      match selected_node with
	None -> ()
      |	Some (ele, item, label, parent_opt, tree) ->
	  let params = 
	    match ele with
	      Leaf l -> self#params_for_leaf l
	    | Mark m -> self#params_for_mark m
	    | Tag t -> self#params_for_tag t
	    | List l -> self#params_for_list l
	    | Else _ | Then _ -> []
	    | Cond c -> self#params_for_cond c
	    | Sub s -> self#params_for_sub s
	  in
	  match params with
	    [] -> ()
	  | _ ->
	      match C.simple_get Rep_messages.mnEdit_selected params
	      with
		C.Return_ok -> 
		  data#set_changed true ;
		  label#set_text (self#item_label ele)
	      | _ -> 
	      ()
	  

    method edit_params () =
      let param = C.strings
	  ~f: (fun l -> data#set_parameters l)
	  ~add: (fun () -> 
	           match GToolbox.input_string 
		       Rep_messages.add_parameter
		       Rep_messages.name
		   with
		     None -> []
		   | Some s -> [s]
		)
	  Rep_messages.parameters
	  (data#report).rep_params
      in
      match C.simple_get Rep_messages.mnEdit_params [param] with
	C.Return_ok -> self#update_params
      |	_ -> ()

    method edit_header () = 
      let param = C.text 
	  ~f: (fun s -> data#set_header s)
	  Rep_messages.header
	  (data#report).rep_header
      in
      match C.simple_get Rep_messages.mnEdit_params [param] with
	C.Return_ok -> ()
      |	_ -> ()

    method init_window (w : GWindow.window) =
      let _ = w#add_accel_group accel_group in
      ()

    method save = 
      try data#save
      with Failure s -> GToolbox.message_box Rep_messages.error s

    initializer
      let _ = itemQuit#connect#activate box#destroy in
      let _ = itemSave#connect#activate (fun () -> self#save) in
      let _ = itemCopy#connect#activate self#copy in
      let _ = itemCut#connect#activate self#cut in
      let _ = itemPaste#connect#activate self#paste_append in
      let _ = itemEdit_header#connect#activate self#edit_header in
      let _ = itemEdit_parameters#connect#activate self#edit_params in
      let _ = itemUp#connect#activate self#up in
      let _ = itemEdit_selected#connect#activate self#edit_selected in
      let _ = itemInsert_leaf#connect#activate self#insert_leaf in
      let _ = itemInsert_tag#connect#activate self#insert_tag in
      let _ = itemInsert_list#connect#activate self#insert_list in
      let _ = itemInsert_cond#connect#activate self#insert_cond in
      let _ = itemInsert_sub#connect#activate self#insert_sub in
      let _ = itemInsert_mark#connect#activate self#insert_mark in

      let _ = self#update in
      ()
  end



