/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2014 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

// -- Core image component stuff
#include "ImageComponent.h"
#include "ImageComponentExtension.h"
#include "SingleImageComponent.h"

// -- Core stuff
#include "Application.h"
#include "InteractiveViewer.h"
#include "Log.h"
#include "MeshComponent.h"

// -- vtk stuff
#include <vtkUniformGrid.h>
#include <vtkTransformFilter.h>
#include <vtkImageReslice.h>
#include <vtkImageFlip.h>
#include <vtkUnstructuredGrid.h>
#include <vtkVertex.h>
#include <vtkActor.h>

#include <vtkPiecewiseFunction.h>
#include <vtkColorTransferFunction.h>
#include <vtkVolumeProperty.h>
#include <vtkVolumeRayCastMapper.h>
#include <vtkImageCast.h>
#include <vtkVolumeRayCastCompositeFunction.h>
#include <vtkVolume.h>

#include <vtkCellArray.h>
#include <vtkFloatArray.h>
#include <vtkPointData.h>
#include <vtkPoints.h>
#include <vtkPolyData.h>


// -- QT stuff
#include <QDockWidget>
#include <QString>
#include <QMessageBox>
#include <QTextStream>

// -- stl stuff
#include <iostream>


namespace camitk {
// -------------------- constructors --------------------
ImageComponent::ImageComponent(const QString & file) throw(AbortException)
    : Component(file, "Image Volume") {
    init();
}

ImageComponent::ImageComponent(vtkSmartPointer<vtkImageData> anImageData, const QString & name, bool copy)
throw(AbortException) : Component("", name) {
    init();
    setImageData(anImageData, copy);
    setName(name);
    setModified();
}

// -------------------- destructor  --------------------
ImageComponent::~ImageComponent() {
}

// -------------------- init --------------------
void ImageComponent::init() {

    originalImageData = NULL;
    axialSlices     = NULL;
    coronalSlices   = NULL;
    sagittalSlices  = NULL;
    arbitrarySlices = NULL;
    volumeRenderingChild = NULL;
    viewIn3D        = false;

    currentPixelPicked[0] = -1;
    currentPixelPicked[1] = -1;
    currentPixelPicked[2] = -1;

    lut = vtkSmartPointer<vtkWindowLevelLookupTable>::New();


}

// -------------------- setImageData --------------------
void ImageComponent::setImageData(vtkSmartPointer<vtkImageData> anImageData, bool copy) {

    originalImageData = NULL;

    if (copy) {
        // We need to use a shall because of some strange behaviour of mingw
        // (In mingw variable seems to become out of scope when going from one DLL to another one,
        // which generates a loss of smartness in the vtkSmartPointer)
        // Disconnect the previous Vtk Pipeline without memcopy
        // (equivalent of ITK DisconnectPipeline()...)
        // Note : this (i.e disconnect/deepcopy/shallowcopy) should not have to be done
        // in the components...
        originalImageData = vtkImageData::New();
        originalImageData->ShallowCopy(anImageData);
    }
    else
        originalImageData = anImageData;


    // Build default lookup table
    initLookupTable();
    
    // Build the Axial, Sagittal and Coronal SingleImageComponents...
    buildImageComponents();
}

// -------------------- initLookupTable --------------------
void ImageComponent::initLookupTable() {

    // TODO set in a betterway information about the lut thanks to image data
    if (this->originalImageData->GetNumberOfScalarComponents() == 1) {
        // Default values for grey level lut
        lut->SetTableRange (0, 255);
        lut->SetSaturationRange (1.0, 1.0);
        lut->SetHueRange (0.667, 0.667);
        lut->SetValueRange (1.0, 1.0);
        lut->SetAlphaRange(1.0, 1.0);
        lut->SetLevel((int) ((getMaxColor() + this->getMinColor()) / 2));
        lut->SetWindow(getNumberOfColors());
        lut->Build (); // effective built
    }
    else {
        // For colored image, we need to extract the information from the image
        // but a classic lut is not really easy to use.
        // delete default LUT
        lut = NULL;
    }
}

// -------------------- replaceImageData --------------------
void ImageComponent::replaceImageData(vtkSmartPointer<vtkImageData> anImageData, bool copy) {
    // Delete ImageComponents
    if (axialSlices) {
        removeChild(axialSlices);
        delete axialSlices;
        axialSlices = NULL;
    }

    if (coronalSlices) {
        removeChild(coronalSlices);
        delete coronalSlices;
        coronalSlices = NULL;
    }

    if (sagittalSlices) {
        removeChild(sagittalSlices);
        delete sagittalSlices;
        sagittalSlices = NULL;
    }

    if (arbitrarySlices) {
        removeChild(arbitrarySlices);
        delete arbitrarySlices;
        arbitrarySlices = NULL;
    }

    if (volumeRenderingChild) {
        removeChild(volumeRenderingChild);
        delete volumeRenderingChild;
        volumeRenderingChild = NULL;
    }

    deleteChildren();

    this->refreshInterfaceNode();

    setImageData(anImageData, copy);

    this->refreshInterfaceNode();
}

// -------------------- getAxialSlices --------------------
SingleImageComponent * ImageComponent::getAxialSlices() {
    return this->axialSlices;
}

// -------------------- getCoronalSlices --------------------
SingleImageComponent * ImageComponent::getCoronalSlices() {
    return this->coronalSlices;
}

// -------------------- getSagittalSlices --------------------
SingleImageComponent * ImageComponent::getSagittalSlices() {
    return this->sagittalSlices;
}

// -------------------- getArbitrarySlices --------------------
SingleImageComponent * ImageComponent::getArbitrarySlices() {
    return this->arbitrarySlices;
}

MeshComponent * ImageComponent::getVolumeRenderingChild() {
    return this->volumeRenderingChild;
}

// -------------------- setImageName --------------------
void ImageComponent::setImageName(const QString& imageName) {
    // Change name in the ImageFilterAddon
    Component::setName(imageName);
    refresh();
}

// -------------------- setImageName --------------------
QString ImageComponent::getImageName() const {
    return getName();
}

// -------------------- getImageSize --------------------
QVariantMap ImageComponent::getImageSize() const {
    QVariantMap imageSize;

    if (originalImageData) {
        imageSize.insert("Voxels X", originalImageData->GetDimensions()[0]);
        imageSize.insert("Voxels Y", originalImageData->GetDimensions()[1]);
        imageSize.insert("Voxels Z", originalImageData->GetDimensions()[2]);
    }
    else {
        imageSize.insert("Voxels X", -1);
        imageSize.insert("Voxels Y", -1);
        imageSize.insert("Voxels Z", -1);
    }

    return imageSize;
}

QString ImageComponent::getDataType() const {
    QString dataType;
	
    if (originalImageData) {
	    dataType = originalImageData->GetScalarTypeAsString();
	}
	else {
	dataType = "unkown";
	}
	
	return dataType;
}

// -------------------- getVoxelSize --------------------
QVector3D ImageComponent::getVoxelSize() const {
    if (originalImageData) {
        double * spacing = originalImageData->GetSpacing();
        return QVector3D(spacing[0], spacing[1], spacing[2]);
    }
    else
        return QVector3D(-1.0, -1.0, -1.0);
}

// -------------------- getImageOrigin --------------------
QVector3D ImageComponent::getImageOrigin() const {
    if (originalImageData) {
        double * origin = originalImageData->GetOrigin();
        return QVector3D(origin[0], origin[1], origin[2]);
    }
    else {
        return QVector3D(0,0,0);
    }
}


bool ImageComponent::getViewIn3D() const {
    return viewIn3D;
}

void ImageComponent::setViewIn3D(bool toggle) {
    viewIn3D = toggle;

    if (viewIn3D) {
        if (axialSlices) {
            axialSlices->setViewSliceIn3D(true);
        }

        if (coronalSlices) {
            coronalSlices->setViewSliceIn3D(true);
        }

        if (sagittalSlices) {
            sagittalSlices->setViewSliceIn3D(true);
        }

        if (arbitrarySlices) {
            // by default not visible in 3D
            arbitrarySlices->setViewSliceIn3D(false);
        }

        if (volumeRenderingChild) {
            volumeRenderingChild->setVisibility(InteractiveViewer::get3DViewer(), true);
        }
    }
    else {
        if (axialSlices) {
            axialSlices->setViewSliceIn3D(false);
        }

        if (coronalSlices) {
            coronalSlices->setViewSliceIn3D(false);
        }

        if (sagittalSlices) {
            sagittalSlices->setViewSliceIn3D(false);
        }

        if (arbitrarySlices) {
            arbitrarySlices->setViewSliceIn3D(false);
        }

        if (volumeRenderingChild) {
            volumeRenderingChild->setVisibility(InteractiveViewer::get3DViewer(), false);
        }
    }

    InteractiveViewer::get3DViewer()->refresh();

}



// -------------------- buildImageComponents --------------------
void ImageComponent::buildImageComponents() {
    if (!axialSlices)
        axialSlices  = new SingleImageComponent(this, InterfaceBitMap::AXIAL_ORIENTATION, "Axial view", lut);

    if (this->originalImageData->GetDataDimension() == 3) {
        if (!coronalSlices)
            coronalSlices = new SingleImageComponent(this, InterfaceBitMap::CORONAL_ORIENTATION, "Coronal view", lut);

        if (!sagittalSlices)
            sagittalSlices = new SingleImageComponent(this, InterfaceBitMap::SAGITTAL_ORIENTATION, "Sagittal view", lut);

        if (!arbitrarySlices)
            arbitrarySlices = new SingleImageComponent(this, InterfaceBitMap::ARBITRARY_ORIENTATION, "Arbitrary view", lut);

        if (volumeRenderingChild)
            delete volumeRenderingChild;

        // compute bounding box
        vtkSmartPointer<vtkPolyData> bbox = getBoundingBox();
        volumeRenderingChild = new MeshComponent(this, bbox, "Volume Rendering");
        volumeRenderingChild->setRenderingModes(InterfaceGeometry::Wireframe);
    }
    else {
        if (coronalSlices)
            delete coronalSlices;

        coronalSlices  = NULL;

        if (sagittalSlices)
            delete sagittalSlices;

        sagittalSlices = NULL;

        if (arbitrarySlices)
            delete arbitrarySlices;

        arbitrarySlices = NULL;

        if (volumeRenderingChild)
            delete volumeRenderingChild;

        volumeRenderingChild = NULL;
    }

    // Let there be slices...
    setViewIn3D(true);

}

// -------------------- updateImageComponents --------------------
void ImageComponent::updateImageComponents() {
    if (axialSlices) {
        axialSlices->setOriginalVolume(originalImageData);
    }

    if (coronalSlices) {
        coronalSlices->setOriginalVolume(originalImageData);
    }

    if (sagittalSlices) {
        sagittalSlices->setOriginalVolume(originalImageData);
    }

    if (arbitrarySlices) {
        arbitrarySlices->setOriginalVolume(originalImageData);
    }

}

// -------------------- pixelPicked --------------------
void ImageComponent::pixelPicked(double x, double y, double z, SingleImageComponent *whoIsAsking) {

    currentPixelPicked[0] = rint(x);
    currentPixelPicked[1] = rint(y);
    currentPixelPicked[2] = rint(z);
    
    // Update each child even the one who is asking i order to display correctly the pixel pixed.
    foreach( Component *dc, getChildren() ) {
        SingleImageComponent *child = dynamic_cast<SingleImageComponent*>(dc);

        if (child) {
            child->setSlice(x, y, z);
        }
    }

    Application::showStatusBarMessage("Pixel (" + QString::number(currentPixelPicked[0]).rightJustified(3,' ') + ","
                                      + QString::number(currentPixelPicked[1]).rightJustified(3,' ') + ","
                                      + QString::number(currentPixelPicked[2]).rightJustified(3,' ') + "), "
                                      + "value: " + QString::number(originalImageData->GetScalarComponentAsDouble(currentPixelPicked[0], currentPixelPicked[1], currentPixelPicked[2], 0)));
                                      
}

// -------------------- getNumberOfColors --------------------
int ImageComponent::getNumberOfColors() const {
    double minColor = getMinColor();
    double maxColor = getMaxColor();
    int nbColors = (int)(maxColor - minColor + 1);
    return nbColors;
}

// -------------------- getMinColor --------------------
double ImageComponent::getMinColor() const {
    return (double) originalImageData->GetScalarTypeMin();
}

// -------------------- getMaxColor --------------------
double ImageComponent::getMaxColor() const {
    return (double) originalImageData->GetScalarTypeMax();
}

// -------------------- getNumberOfSlices --------------------
int ImageComponent::getNumberOfSlices() const {
    return this->axialSlices->getNumberOfSlices();
}

// -------------------- setLut --------------------
void ImageComponent::setLut(vtkSmartPointer<vtkWindowLevelLookupTable> lookupTable) {
    lut = lookupTable;
}

// -------------------- getLut --------------------
vtkSmartPointer<vtkWindowLevelLookupTable>  ImageComponent::getLut() {
    return this->lut;
}

// -------------------- setSelected --------------------
void ImageComponent::setSelected(const bool b, const bool) {
    foreach(Component *dc, getChildren()) {
        SingleImageComponent *child = dynamic_cast<SingleImageComponent*>(dc);

        if ( child ) {
            child->singleImageSelected(b);
        }
    }

    // do that only in the end, so that last selected will be this manager Component
    Component::setSelected(b, false);
}

// -------------------- getLastPixelPicked --------------------
void ImageComponent::getLastPixelPicked( int * x, int * y, int * z ) {
    *x = currentPixelPicked[0];
    *y = currentPixelPicked[1];
    *z = currentPixelPicked[2];
}

// -------------------- refresh --------------------
void ImageComponent::refresh() const {
    foreach(const Component *dc, childrenComponent) {
        dc->refresh();
    }
}

// -------------------- getBoundingBox --------------------
vtkSmartPointer<vtkPolyData> ImageComponent::getBoundingBox() {
    if (! originalImageData)
        return NULL;

    double * bounds = originalImageData->GetBounds();
    double x[8][3]= {{bounds[0],bounds[2],bounds[4]}, {bounds[1],bounds[2],bounds[4]},
        {bounds[1],bounds[3],bounds[4]}, {bounds[0],bounds[3],bounds[4]},
        {bounds[0],bounds[2],bounds[5]}, {bounds[1],bounds[2],bounds[5]},
        {bounds[1],bounds[3],bounds[5]}, {bounds[0],bounds[3],bounds[5]}
    };
    vtkIdType pts[6][4]= {{0,1,2,3}, {4,5,6,7}, {0,1,5,4},
        {1,2,6,5}, {2,3,7,6}, {3,0,4,7}
    };
    vtkSmartPointer<vtkPolyData>    bbox    = vtkSmartPointer<vtkPolyData>::New();
    vtkSmartPointer<vtkPoints>      points  = vtkSmartPointer<vtkPoints>::New();
    vtkSmartPointer<vtkCellArray>   polys   = vtkSmartPointer<vtkCellArray>::New();
    vtkSmartPointer<vtkFloatArray>  scalars  = vtkSmartPointer<vtkFloatArray>::New();

    for (int i=0; i<8; i++)
        points->InsertPoint(i,x[i]);

    for (int i=0; i<6; i++)
        polys->InsertNextCell(4,pts[i]);

    bbox->SetPoints(points);
    bbox->SetPolys(polys);

    return bbox;
}


}
