#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sstream>
#include <typeinfo>

#include "datetime.h"
#include "canlxx.h"
#include "opensslutil.h"

namespace AuthN {

using namespace AuthN::Utils;
using namespace AuthN::OpenSSL;

  static std::string tostring(unsigned int val) {
    std::stringstream ss;
    ss << val;
    return ss.str();
  }

  Credentials::Credentials(const Credentials& arg) :
     valid_(false), certctx_(NULL), context_(NULL), validator_(NULL) {
    if(arg.context_) context_ = &arg.context_->Copy();
    if(arg.validator_) validator_ = &arg.validator_->Copy();
    Assign(arg);
 }

  Credentials& Credentials::operator=(const Credentials& arg) {
    Assign(arg);
    return *this;
  }

  Credentials::Credentials(void) :
     valid_(false), certctx_(NULL), context_(NULL), validator_(NULL) {
  }

  Credentials::Credentials(const AuthN::Context& ctx):valid_(false), certctx_(NULL), validator_(NULL) {
    try {
    context_ = &ctx.Copy();
    certctx_ = new CertContext(*context_);
    valid_ = true;
    } catch (Status& err) {
      last_error_ = err;
    };
  };


  Credentials::~Credentials(void) {
    if(certctx_) delete certctx_;
    if(validator_) delete validator_;
    if(context_) delete context_;
  }

  Credentials& Credentials::Copy(void) const {
    std::string cert, key, chain;
    GetCertificate(cert);
    GetPrivateKey(key);
    GetChain(chain);
    Credentials* ncred = new Credentials(Context());
    ncred->Assign(cert, chain, key);
    if(context_) ncred->SetContext(*context_);
    if(validator_) ncred->SetValidator(*validator_); 
    return *ncred;
  }

  AuthN::Status Credentials::Assign(const Credentials& arg) {
    std::string cert, key, chain;
    arg.GetCertificate(cert); 
    arg.GetPrivateKey(key);
    arg.GetChain(chain);
    return Assign(cert, chain, key);
  }

  static void read_stream(std::istream& in, std::string& str) {
    int length;
    char* buffer;
    if(!in || !in.good()) return;
    in.seekg (0, std::ios::end);
    length = in.tellg();
    if(length <0) return;
    in.seekg (0, std::ios::beg);
    buffer = new char [length];
    in.read (buffer,length);
    str.assign(buffer,length);
    in.seekg (0, std::ios::beg);
    delete[] buffer;
  }

  AuthN::Status Credentials::Assign(std::istream& cert, std::istream& chain, std::istream& key) {
    std::string cert_str;
    std::string chain_str;
    std::string key_str;

    if(cert && cert.good()) read_stream(cert, cert_str);
    if(chain && chain.good()) read_stream(chain, chain_str);
    if(key && key.good()) read_stream(key, key_str);

    return(Assign(cert_str, chain_str, key_str));
  }

  AuthN::Status Credentials::Assign(const std::string& cert, const std::string& chain, const std::string& key) {
    try {
      std::string cert_str;
      if(!cert.empty()) cert_str.append(cert);
      if(!chain.empty()) cert_str.append("\n").append(chain);
      if(!cert_str.empty()) certctx_->certFromStr(cert_str);
      if(!key.empty()) certctx_->keyFromStr(key);
      
      if((bool)certctx_) valid_ = true;

      return Status(0);

    } catch (Status& err) {
      last_error_ = err;
    }

    return last_error_;
  }

  void Credentials::GetCertificate(std::string& str) const {
    if(!certctx_) return;
    std::string cert_and_chain;
    certctx_->certToPEM(cert_and_chain);
    if(cert_and_chain.empty()) return;

    std::string::size_type pos1, pos2;
    pos1 = cert_and_chain.find("END CERTIFICATE");
    if(pos1 != std::string::npos) {
      pos2 = cert_and_chain.find_first_of("\r\n", pos1);
      if(pos2 != std::string::npos) str = cert_and_chain.substr(0, pos2+1);
      else str = cert_and_chain;
    }
    return;
  }

  void Credentials::GetCertificate(std::ostream& o) const {
    std::string s;
    GetCertificate(s);
    o << s;
  }

  void Credentials::GetChain(std::string& str) const {
    if(!certctx_) return;
    std::string cert_and_chain;
    certctx_->certToPEM(cert_and_chain);
    if(cert_and_chain.empty()) return;

    std::string::size_type pos1, pos2, pos3;
    pos1 = cert_and_chain.find("END CERTIFICATE");
    if(pos1 != std::string::npos) { 
      pos2 = cert_and_chain.find_first_of("\r\n", pos1);
      if(pos2 != std::string::npos) {
        pos3 = cert_and_chain.find("BEGIN CERTIFICATE", pos2);
        if(pos3 != std::string::npos) str = cert_and_chain.substr(pos2+1);
      }
    }
    return;
  }

  void Credentials::GetChain(std::ostream& o) const {
    std::string s;
    GetCertificate(s);
    o << s;
  }

  void Credentials::GetPrivateKey(std::string& str, bool encrypt) const {
    certctx_->keyToPEM(str, encrypt);
    return;
  }
      
  void Credentials::GetPrivateKey(std::ostream& o, bool encrypt) const {
    std::string s;
    GetPrivateKey(s, encrypt);
    o << s;
  }

  Status Credentials::GetStatus(void) const {
    return last_error_;
  }

  AuthN::Status Credentials::Validate(void) {
    if(validator_ == NULL) {
      context_->Log(Context::LogError, "No validator is assigned");
      last_error_ = AuthN::Status(-1,"No validator is assigned");
    } else {
      last_error_ = validator_->Validate(*this);
    };
    return last_error_;
  }

  AuthN::Status Credentials::Sign(CredentialsRequest& request, Credentials& out, const std::string& conf_file) {
    CSRContext* csr = request.csrctx_;
    KeyContext* key = request.keyctx_;
    AuthN::OpenSSL::CertContext* signed_cert = NULL;
    if(csr == NULL) {
      context_->Log(Context::LogError, "The request is empty");
      last_error_ = Status(-1,"The request is empty");
      return last_error_;
    }

    if(!certctx_ || !(*certctx_)) {
      context_->Log(Context::LogDebug, "This object does not contain any certificate, self-signed certificate will be created");

      //If the CertContext is empty, we look the signing as self-signing,
      //for an EEC certificate signing, self-signing only happens if the EEC is a CA certificate

      AuthN::Context ctx(AuthN::Context::EmptyContext);

      AuthN::OpenSSL::CertificateOptions ca_opts;
      std::string sub_str = convert_rfc2253_to_ossdn(request.GetSubjectName());
      ca_opts.setSubject(sub_str);
      ca_opts.setAsCA();
      ca_opts.setSerialNumber(0);
      AuthN::Utils::Time start = request.GetValidFrom();
      AuthN::Utils::Time end = request.GetValidTill();
      ca_opts.setValidityPeriod(start, end);

      signed_cert = new AuthN::OpenSSL::CertContext(ctx);
      if(!conf_file.empty()) signed_cert->setOSSLConf(conf_file);
      if((signed_cert->createSelfSigned(ca_opts, *key)) != Status(0)) {
        context_->Log(Context::LogError, "Failed to sign a self-signed certificate");
        last_error_ = Status(-1,"Failed to sign a self-signed certificate");
        return last_error_;
      }
      std::string signed_cert_str;
      std::string signed_key_str;
      signed_cert->certToPEM(signed_cert_str);
      signed_cert->keyToPEM(signed_key_str);  //TODO: key encryption
      out.Assign(signed_cert_str, "", signed_key_str); 
      Assign(signed_cert_str, "", signed_key_str);

      context_->Log(Context::LogDebug, "Succeed to sign a CA certificate");

/*
      try {
        CACredentialsRequest& carequest = dynamic_cast<CACredentialsRequest&>(request);
        carequest.Assign(cert);
      } catch(const std::bad_cast& e) {
        last_error_ = Status(-1,"Credentials request is not CA request");
        if(cert != NULL) X509_free(cert);
        return last_error_;
      }
*/
    }
    else {
      AuthN::OpenSSL::CertificateOptions sign_opts;

      std::string proxypolicy;
      proxypolicy = csr->props_.proxyPolicy.value;
      if(!proxypolicy.empty()) sign_opts.setProxyPolicy(proxypolicy);

      std::string sub_str = convert_rfc2253_to_ossdn(request.GetSubjectName());
      if(!(sub_str.empty())) sign_opts.setSubject(sub_str);

      sign_opts.setAsUser();

      AuthN::Utils::Time start = request.GetValidFrom();
      AuthN::Utils::Time end = request.GetValidTill();
      sign_opts.setValidityPeriod(start, end);

      if(!conf_file.empty()) certctx_->setOSSLConf(conf_file);
      signed_cert = certctx_->signRequest(sign_opts, *csr);
      std::string signed_cert_str;
      const CertContextProps* props = certctx_->getProps();
      if(signed_cert != NULL) { 
        signed_cert->certToPEM(signed_cert_str);
        if(props!=NULL) {
          // If the signer is not a CA cert, then the signed cert should be a proxy
          // in this case, the signer's cert and chain needs to be assigned to the output
          if(!(props->isCA)) {
            std::string signer_cert_str; 
            certctx_->certToPEM(signer_cert_str);
            signed_cert_str.append(signer_cert_str);
          }
        }
        out.Assign(signed_cert_str); 
      }
      else {
        if(props != NULL) {
          if(props->isCA) { 
            context_->Log(Context::LogError, "Failed to sign an EEC certificate");
            last_error_ = Status(-1, "Failed to sign an EEC certificate");
          }
          else {
            context_->Log(Context::LogError, "Failed to sign proxy certificate");
            last_error_ = Status(-1, "Failed to sign a proxy certificate");
          }
        }
        return last_error_;
      }
      if(props != NULL) {
        if(props->isCA) context_->Log(Context::LogDebug, "Succeed to sign an EEC certificate");
        else context_->Log(Context::LogDebug, "Succeed to sign proxy certificate");
      }
    }

    if(signed_cert) delete signed_cert;
    return Status(0);
  }

  std::string Credentials::GetSubjectName(void) const {
    const CertContextProps* props = certctx_->getProps();
    std::string str = props->subject;
    str = convert_ossdn_to_rfc2253(str);
    return str;
  }

  std::string Credentials::GetIdentityName(void) const {
    const CertContextProps* props = certctx_->getProps();
    std::string str = props->identity;
    str = convert_ossdn_to_rfc2253(str);
    return str;
  }

  std::string Credentials::GetIssuerName(int position) const {
    std::string str;
    if(position == 0) {
      const CertContextProps* props = certctx_->getProps();
      str = props->issuer;
    }
    else {
      CertContext* cert_in_chain = certctx_->getChainCert(position);
      const CertContextProps* props;
      if(cert_in_chain != NULL) {
        props = cert_in_chain->getProps();
        str = props->issuer;
        delete cert_in_chain;
      }    
    }
    str = convert_ossdn_to_rfc2253(str);
    return str;
  }

  AuthN::Status Credentials::GetChainCert(Credentials& cert, int position) const {
    Status status;
    CertContext* cert_in_chain = certctx_->getChainCert(position);
    if(!cert_in_chain) return Status(-1,"Failed to get "+tostring(position)+" certificate in chain");
    cert.certctx_ = cert_in_chain;
    return Status(0);
  }

  time_t Credentials::GetValidFrom(void) const {
    const CertContextProps* props = certctx_->getProps();
    AuthN::Utils::Time time = props->start;
    return time.GetTime();
  }

  time_t Credentials::GetValidTill(void) const {
    const CertContextProps* props = certctx_->getProps();
    AuthN::Utils::Time time = props->end;
    return time.GetTime();
  }

  bool Credentials::GetSelfSigned(void) const {
    std::string sub, iss;
    sub = GetSubjectName();
    iss = GetIssuerName();
    if(!iss.empty() && (iss == sub)) return true;
    else return false;
  }

  bool Credentials::GetCA(void) const {
    const CertContextProps* props = certctx_->getProps();
    return props->isCA;
  }

  bool Credentials::GetProxy(void) const {
    const CertContextProps* props = certctx_->getProps();
    return props->isProxy;
  }

  bool Credentials::GetProxyLimited(void) const {
    std::string sub_name = (*this).GetSubjectName();
    if(sub_name.find("=limited proxy") != std::string::npos) return true;
    else return false;
  }

  bool Credentials::GetProxyPolicy(Extension& policy) const {
    const CertContextProps* props = certctx_->getProps();
    policy.oid = props->proxyPolicy.oid;
    policy.critical = props->proxyPolicy.critical;
    policy.value = props->proxyPolicy.value;
    return policy.oid.empty()?false:true;
  }

  bool Credentials::GetExtension(int pos, Extension& ext) const {
    if(!certctx_) return false;
    return certctx_->getCertExtension(pos, ext);
  }

  bool Credentials::GetExtension(const std::string& name, Extension& ext) const {
    if(!certctx_) return false;
    return certctx_->getCertExtension(name, ext);
  }

  bool Credentials::GetAttributes(const std::string& name, std::list<std::string>& attrs) const {
    if(!certctx_) return false;
    return certctx_->getCertAttributes(name, attrs);
  }

  void Credentials::SetContext(const AuthN::Context& env) {
    context_ = &env.Copy();
  }

  const AuthN::Context& Credentials::GetContext(void) const {
    return *context_;
  }

  void Credentials::SetValidator(const AuthN::Validator& validator) {
    if(validator_) delete validator_;
    validator_ = &(validator.Copy());
  }

  const AuthN::Validator& Credentials::GetValidator(void) const {
    return *validator_;
  }

}
