/**
 * @file pc64.c
 * Transfer routines for the 4-bit PET/C64/C128/Vic-20 to PC cable
 * designed by Wolfgang Lorenz
 * @author Marko Mkel (msmakela@nic.funet.fi)
 */

/*
 * Copyright  1994-1996 Marko Mkel and Olaf Seibert
 * Copyright  2001 Marko Mkel
 * Original Linux and Commodore 64/128/Vic-20 version by Marko Mkel
 * Ported to the PET and the Amiga series by Olaf Seibert
 * Restructured by Marko Mkel
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef COMM_PC
# include "info.h"
# include "pc64.h"
# include "pcpar.h"

/** Open the communication channel
 * @param dev		name of the communication device
 * @param hostinfo	(output) the computer model information
 * @return		zero on success, nonzero on failure
 */
int
pc64_init (const char* dev, struct hostinfo* hostinfo)
{
  unsigned char detectbuf[5];
  if (!open_port (dev))
    return 1;
  out_data (128); /* set the -FLAG signal to high */
  if (pc64_write ("", 1))
    return 1;
  if (pc64_read (detectbuf, sizeof detectbuf))
    return 2;
  hostinfo->host = detectbuf[0];
  hostinfo->driver = (unsigned) detectbuf[1] | (unsigned) detectbuf[2] << 8;
  hostinfo->basic = (unsigned) detectbuf[3] | (unsigned) detectbuf[4] << 8;
  return 0;
}
/** Close the communication channel */
void
pc64_close (void)
{
  close_port ();
}

#if !defined __BCC__ && (!defined __GNUC__ || defined USE_PPDEV)
/** Send data (unoptimised)
 * @param buf		the data to be sent
 * @param len		length of the data in bytes
 */
static void
pc64_write_slow (const void* buf, unsigned len)
{
  register const unsigned char* buffer = buf;

  while (len--) {
    register unsigned char byte = *buffer++;

    out_data (byte & 0x0f);
    while (!(128 & in_stat ()));
    out_data (128);
    out_data (byte >> 4);
    while (128 & in_stat ());
    out_data (128);
  }
}

/** Receive data (unoptimised)
 * @param buf		the data to be received
 * @param len		length of the data in bytes
 */
static void
pc64_read_slow (void* buf, unsigned len)
{
  register unsigned char data;
  register unsigned char* buffer = buf;

  while (len--) {
    out_data (0);
    while (!(128 & in_stat ()));
    out_data (128); 
    data = (in_stat () & 0x78) >> 3;
    out_data (0);
    while (128 & in_stat ());
    *buffer++ = data | ((in_stat () & 0x78) << 1);  
    out_data (128);
  }
}
#endif /* !__BCC__ && (!__GNUC__ || USE_PPDEV) */

/** Send data
 * @param buf		the data to be sent
 * @param len		length of the data in bytes
 * @return		zero on success, nonzero on failure
 */
int
pc64_write (const void* buf, unsigned len)
{
# ifdef __GNUC__
#  ifdef USE_PPDEV
  if (baseaddr == (unsigned) -1) {
    pc64_write_slow (buf, len);
    return 0;
  }
#  endif /* USE_PPDEV */
  __asm__ volatile ("0:\t"
		    "movb (%%ebx),%%al\n\t"
		    "andb $0x0f,%%al\n\t"
		    "outb %%al,%%dx\n\t"
		    "incw %%dx\n"
		    "1:\t"
		    "inb %%dx,%%al\n\t"
		    "testb $0x80,%%al\n\t"
		    "jz 1b\n\t"
		    "decw %%dx\n\t"
		    "movb $0x80,%%al\n\t"
		    "outb %%al,%%dx\n\t"
		    "movb (%%ebx),%%al\n\t"
		    "shrb $4,%%al\n\t"
		    "outb %%al,%%dx\n\t"
		    "incw %%dx\n"
		    "2:\t"
		    "inb %%dx,%%al\n\t"
		    "testb $0x80,%%al\n\t"
		    "jnz 2b\n\t"
		    "decw %%dx\n\t"
		    "movb $0x80,%%al\n\t"
		    "outb %%al,%%dx\n\t"
		    "inc %%ebx\n\t"
		    "loop 0b"
		    : /* no outputs */
		    : "b" (buf), "c" (len), "d" (baseaddr)
		    : "ax");
  return 0;
# else /* BCC does not recognize #elif */
# ifdef __BCC__
#  asm
    push bp
    mov bp,sp
    push es
    mov dx,_baseaddr
    mov cx,len_
    mov bx,buf_
    mov ax,I 0x40
    mov es,ax

.sloop:
    mov al,[bx]
    and al,I 0x0f
    out dx,al
    inc dx

.ewait:
    in al,dx
    test al,I 0x80
    jnz .newait
    eseg
    mov al,byte [0x71]	; is the keyboard interrupt flag set?
    test al,I 0x80
    jz .ewait
    jmp near _abreak

.newait:
    dec dx
    mov al,I 0x80
    out dx,al
    mov al,[bx]
    shr al,I 1
    shr al,I 1
    shr al,I 1
    shr al,I 1
    out dx,al
    inc dx

.zwait:
    in al,dx
    test al,I 0x80
    jz .nzwait
    eseg
    mov al,byte [0x71]	; is the keyboard interrupt flag set?
    test al,I 0x80
    jz .zwait
    jmp near _abreak

.nzwait:
    dec dx
    mov al,I 0x80
    out dx,al

    inc bx
    loop .sloop
    pop es
    pop bp
    xor ax,ax
#  endasm
# else
  pc64_write_slow (buf, len);
  return 0;
# endif
# endif /* BCC does not recognize #elif */
}

/** Receive data
 * @param buf		the data to be received
 * @param len		length of the data in bytes
 * @return		zero on success, nonzero on failure
 */
int
pc64_read (void* buf, unsigned len)
{
# ifdef __GNUC__
#  ifdef USE_PPDEV
  if (baseaddr == (unsigned) -1) {
    pc64_read_slow (buf, len);
    return 0;
  }
#  endif /* USE_PPDEV */
  __asm__ volatile ("0:\t"
		    "xorb %%al,%%al\n\t"
		    "outb %%al,%%dx\n\t"
		    "incw %%dx\n"
		    "1:\t"
		    "inb %%dx,%%al\n\t"
		    "testb $0x80,%%al\n\t"
		    "jz 1b\n\t"
		    "andb $0x78,%%al\n\t"
		    "shrb $3,%%al\n\t"
		    "movb %%al,(%%ebx)\n\t"
		    "decw %%dx\n\t"
		    "movb $0x80,%%al\n\t"
		    "outb %%al,%%dx\n\t"
		    "xorb %%al,%%al\n\t"
		    "outb %%al,%%dx\n\t"
		    "incw %%dx\n"
		    "2:\t"
		    "inb %%dx,%%al\n\t"
		    "testb $0x80,%%al\n\t"
		    "jnz 2b\n\t"
		    "andb $0x78,%%al\n\t"
		    "shlb $1,%%al\n\t"
		    "orb %%al,(%%ebx)\n\t"
		    "decw %%dx\n\t"
		    "movb $0x80,%%al\n\t"
		    "outb %%al,%%dx\n\t"
		    "inc %%ebx\n\t"
		    "loop 0b"
		    : /* no outputs */
		    : "b" (buf), "c" (len), "d" (baseaddr)
		    : "ax");
  return 0;
# else /* BCC does not recognize #elif */
# ifdef __BCC__
#  asm
    push bp
    mov bp,sp
    push es
    mov dx,_baseaddr
    mov cx,len_
    mov bx,buf_
    mov ax,I 0x40
    mov es,ax

.rloop:
    xor al,al
    out dx,al
    inc dx

.dwait:
    in al,dx
    test al,I 0x80
    jnz .ndwait
    eseg
    mov al,byte [0x71]	; is the keyboard interrupt flag set?
    test al,I 0x80
    jz .dwait
    jmp near _abreak

.ndwait:
    and al,I 0x78
    shr al,I 1
    shr al,I 1
    shr al,I 1
    mov [bx],al
    dec dx
    mov al,I 0x80
    out dx,al
    xor al,al
    out dx,al
    inc dx

.vwait:
    in al,dx
    test al,I 0x80
    jz .nvwait
    eseg
    mov al,byte [0x71]	; is the keyboard interrupt flag set?
    test al,I 0x80
    jz .vwait
    jmp near _abreak

.nvwait:
    and al,I 0x78
    shl al,I 1
    or [bx],al
    dec dx
    mov al,I 0x80
    out dx,al

    inc bx
    loop .rloop

    pop es
    pop bp
    xor ax,ax
#  endasm
# else
  pc64_read_slow (buf, len);
  return 0;
# endif
# endif /* BCC does not recognize #elif */
}

#endif /* COMM_PC */
