;;;; debugger.scm - Support library for executables with extended debugging information
;
; Copyright (c) 2000-2002, Felix L. Winkelmann
; All rights reserved.
;
; Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following
; conditions are met:
;
;   Redistributions of source code must retain the above copyright notice, this list of conditions and the following
;     disclaimer. 
;   Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following
;     disclaimer in the documentation and/or other materials provided with the distribution. 
;   Neither the name of the author nor the names of its contributors may be used to endorse or promote
;     products derived from this software without specific prior written permission. 
;
; THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS
; OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
; AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR
; CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
; SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
; THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
; OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
; POSSIBILITY OF SUCH DAMAGE.
;
; Send bugs, suggestions and ideas to: 
;
; felix@call-with-current-continuation.org
;
; Felix L. Winkelmann
; Steinweg 1A
; 37130 Gleichen, OT Weissenborn
; Germany


(declare
  (unit debugger) 
  (usual-integrations)
  (disable-interrupts)
  (fixnum) 
  (bound-to-procedure ##sys#error)
  (no-bound-checks) )


(include "parameters.scm")


;;; Globals:

(define ##sys#debug-frames '())
(define ##sys#selected-frame #f)
(define ##sys#break-level 0)
(define ##sys#continue-continuation #f)
(define ##sys#display-list '())
(define ##sys#debug-info-list '())
(define ##sys#breakpoints '())
(define ##sys#watchpoints '())
(define ##sys#debugger-history '())
(define ##sys#tracepoints '())
(define ##sys#call-stack '())
(define ##sys#single-stepping #f)
(define ##sys#step-frame #f)
(define ##sys#last-step-frames #f)
(define ##sys#last-step-command #f)


;;; Registration of available debug symbols:

(define (##sys#register-debug-info v)
  (set! ##sys#debug-info-list (cons v ##sys#debug-info-list)) )

(define ##sys#find-info
  (let ([display display])
    (lambda (name var)
      (##sys#call-with-current-continuation
       (lambda (return)
	 (let loop ([dl ##sys#debug-info-list])
	   (cond [(null? dl)
		  (display "| named item not found\n")
		  #f]
		 [else
		  (let* ([v (##sys#slot dl 0)]
			 [n (##sys#size v)] )
		    (do ([i 0 (add1 i)])
			((>= i n) (loop (##sys#slot dl 1)))
		      (let ([info (##sys#slot v i)])
			(when (and (eq? name (##sys#slot info 0))
				   (if var
				       (= debug-variable-info-size (##sys#size info))
				       (= debug-lambda-info-size (##sys#size info)) ) )
			  (return info) ) ) ) ) ] ) ) ) ) ) ) )


;;; Frame management:

(define (##sys#push-debug-frame info exit)
  ;; (vector info exit entry arguments)
  ;;   info: #(name lambda-list source argc rest-flag break? exec?)
  (set! ##sys#debug-frames (cons (##sys#vector info exit #f #f ##sys#call-stack) ##sys#debug-frames))
  (set! ##sys#call-stack '()) )

(define (##sys#pop-debug-frame)
  (set! ##sys#call-stack (##sys#slot (##sys#slot ##sys#debug-frames 0) 4))
  (set! ##sys#debug-frames (##sys#slot ##sys#debug-frames 1)) )

(define ##sys#get-frame
  (let ([display display]
	[list-ref list-ref] )
    (lambda (index)
      (let ([n (length ##sys#debug-frames)])
	(cond [(or (not (fixnum? index)) (< index 0) (>= index n))
	       (display "| invalid frame index\n")
	       #f]
	      [else (list-ref ##sys#debug-frames index)] ) ) ) ) )

(define ##sys#show-frame
  (let ([display display]
	[write write]
	[newline newline] )
    (lambda (frame)
      (if frame
	  (let* ([info (##sys#slot frame 0)]
		 [args (##sys#slot frame 3)]
		 [name (##sys#slot info 0)]
		 [llist (##sys#slot info 1)] )
	    (##sys#decompose-lambda-list
	     llist
	     (lambda (vars argc rest)
	       (write (cons name llist))
	       (newline)
	       (let loop ([vars vars] [args args] [end (null? args)])
		 (unless (null? vars)
		   (let ([var (car vars)])
		     (display "| ")
		     (write var)
		     (display "\t -> ")
		     (##sys#with-print-length-limit
		      debug-output-print-limit
		      (lambda ()
			(cond [end (display "-not supplied-")]
			      [(eq? rest var) (write args)]
			      [else (write (##sys#slot args 0))] ) ) )
		     (newline) 
		     (if (pair? args)
			 (loop (##sys#slot vars 1)
			       (##sys#slot args 1)
			       #f)
			 (loop (##sys#slot vars 1) '() #t) ) ) ) ) ) ) )
	  (display "| no frame selected\n") ) ) ) )


;;; Enter debugger programmatically:

(define ##sys#break
  (let ([display display]
	[newline newline] 
	[call-with-current-continuation call-with-current-continuation] )
    (lambda msg
      (let ([msgtxt (:optional msg "*** Break ***")]
	    [bl ##sys#break-level]
	    [pq ##sys#print-qualifiers]
	    [sf ##sys#selected-frame]
	    [cc ##sys#continue-continuation] )
	(when msgtxt
	  (display "| ")
	  (display msgtxt)
	  (newline) )
	(call-with-current-continuation
	 (lambda (return)
	   (##sys#dynamic-wind
	    (lambda ()
	      (set! ##sys#break-level (add1 bl))
	      (set! ##sys#print-qualifiers #t) 
	      (set! ##sys#continue-continuation return) )
	    (lambda ()
	      (unless (eq? ##sys#debug-frames ##sys#last-step-frames)
		(cond [(pair? ##sys#debug-frames)
		       (set! ##sys#selected-frame (car ##sys#debug-frames))
		       (display "| in frame: ")
		       (##sys#show-frame ##sys#selected-frame) ]
		      [else (display "| in toplevel frame\n")] ) )
	      (##sys#debugger-repl) )
	    (lambda () 
	      (set! ##sys#break-level bl)
	      (set! ##sys#print-qualifiers pq)
	      (set! ##sys#selected-frame sf)
	      (set! ##sys#continue-continuation cc) ) ) ) ) ) ) ) )

(define break ##sys#break)


;;; Check procedure argument count and update frame with argument values:

(define ##sys#check-debug-entry 
  (let ([string-append string-append] 
	[call-with-current-continuation call-with-current-continuation] )
    (lambda (args)
      (let* ([tframe (##sys#slot ##sys#debug-frames 0)]
	     [info (##sys#slot tframe 0)]
	     [argc (##sys#slot info 3)]
	     [rest (##sys#slot info 4)] )
	(call-with-current-continuation 
	 (lambda (k)
	   (##sys#setslot 
	    tframe 2			; entry continuation
	    (lambda (args2)
	      (set! args args2)
	      (k #f) ) ) ) )
	(##sys#setslot tframe 3 args)	; entry arguments
	(let ([n (length args)])
	  (when (< n argc)
	    (##sys#error
	     (string-append
	      "bad number of arguments - given " (number->string n) ", but expected "
	      (if rest "at least " "")
	      (number->string argc) ) ) )
	  (##sys#setslot tframe 3 args)
	  (let ([bp (##sys#slot info 5)])
	    (when (and bp (bp)) (##sys#break "*** Breakpoint hit ***")) )
	  (and-let* ([xp (##sys#slot info 6)]) (xp))
	  args) ) ) ) )


;;; Check variable assignments:

(define ##sys#check-debug-assignment
  (let ([string-append string-append]
	[make-string make-string] )
    (lambda (info val)
      (let ([wp (##sys#slot info 2)])
	(when (and wp (wp val))
	  (let* ([source (##sys#slot info 1)]
		 [len (##sys#size source)]
		 [msg (make-string len)] )
	    (##sys#copy-bytes source msg 0 0 len)
	    (##sys#break (string-append "*** Watchpoint hit ***\n" msg)) )  )
	val) ) ) )


;;; Single stepping and call-info:

(define ##sys#debug-call
  (let ([write write]
	[display display]
	[newline newline] )
    (lambda (name proc . args)
      (when (and ##sys#single-stepping
		 (or (not ##sys#step-frame)
		     (eq? ##sys#step-frame ##sys#debug-frames) ) )
	(display "| stepping [")
	(display ##sys#last-step-command)
	(display "]: ")
	(let ([pq ##sys#print-qualifiers])
	  (set! ##sys#print-qualifiers #t)
	  (##sys#with-print-length-limit debug-repl-result-print-limit (lambda () (write (cons name args))))
	  (set! ##sys#print-qualifiers pq) )
	(newline)
	(##sys#break #f) )
      (##sys#dynamic-wind
       (lambda () (set! ##sys#call-stack (cons (cons name args) ##sys#call-stack)))
       (lambda () (apply proc args))
       (lambda () (set! ##sys#call-stack (##sys#slot ##sys#call-stack 1))) ) ) ) )


;;; Set up things correctly:

(##sys#error-handler
 (let ([display display]
       [write write]
       [newline newline] )
   (lambda (msg . args)
     (display "| Error: ") 
     (display msg)
     (newline)
     (let ([pq ##sys#print-qualifiers])
       (set! ##sys#print-qualifiers #t)
       (for-each 
	(lambda (x)
	  (display "| ")
	  (##sys#with-print-length-limit ##sys#repl-print-length-limit (lambda () (write x)))
	  (newline) )
	args)
       (set! ##sys#print-qualifiers pq)
       (let loop ([msg #f])
	 (##sys#break msg)
	 (loop "can not resume computation") ) ) ) ) )


;;; Add commands to debugger REPL:

(define ##sys#execute-debugger-command
  (let ([eval eval]
	[exit exit]
	[display display]
	[write write]
	[newline newline]
	[print print]
	[system system]
	[frame-commands '(info i list ls up down u d r return r* return*)] )
    (lambda (cmd)
      
      (define (bad-command)
	(display "| undefined toplevel command ")
	(write cmd)
	(display "\n| (Enter '?' for help)\n") )

      (define (eval-in-frame exp)
	(eval
	 (if ##sys#selected-frame
	     (let ([llist (##sys#slot (##sys#slot ##sys#selected-frame 0) 1)] 
		   [tmp (gensym)] )
	       (##sys#decompose-lambda-list
		llist
		(lambda (vars argc rest)
		  `(let ([,tmp (##sys#slot ',##sys#selected-frame 3)])
		     ,(let fold ([vars vars])
			(if (null? vars)
			    exp
			    (let ([var (##sys#slot vars 0)])
			      (if (eq? var rest) 
				  `(let ([,var ,tmp])
				     ,exp)
				  `(let* ([,var (if (null? ,tmp) 
						    (##core#undefined) 
						    (##sys#slot ,tmp 0) ) ]
					  [,tmp (if (null? ,tmp) ,tmp (##sys#slot ,tmp 1))] )
				     ,(fold (##sys#slot vars 1)) ) ) ) ) ) ) ) ) )
	     exp) ) )

      (define (find-thread name)
	(let loop ([ts (##sys#all-threads)])
	  (cond [(null? ts) 
		 (display "| nonexistent thread\n") 
		 #f]
		[else
		 (let ([t (##sys#slot ts 0)])
		   (if (eq? name (##sys#slot t 6))
		       t
		       (loop (##sys#slot ts 1)) ) ) ] ) ) )

      (define (print-results vals)
	(for-each
	 (lambda (x)
	   (display "| ")
	   (##sys#with-print-length-limit debug-repl-result-print-limit (lambda () (write x)))
	   (newline) )
	 vals) )

      (define (show-call-stack cs)
	(for-each
	 (lambda (f)
	   (display "|     ")
	   (##sys#with-print-length-limit
	    debug-output-print-limit
	    (lambda () (write f)) )
	   (newline) )
	 cs) )

      (define (list-names size)
	(for-each 
	 (lambda (v)
	   (let ([n (##sys#size v)])
	     (do ([i 0 (add1 i)])
		 ((>= i n))
	       (let* ([info (##sys#slot v i)]
		      [name (##sys#slot info 0)] )
		 (when (and (= size (##sys#size info)) (not (eq? name anonymous-object-identifier)))
		   (display "|   ")
		   (write (##sys#slot info 0))
		   (newline) ) ) ) ) )
	 ##sys#debug-info-list) )

      (case cmd
	
	[(#f) #f]

	[(? help) (##sys#list-debugger-commands)]
		 		     
	[(q quit)
	 (##core#inline "C_exit_runtime" 0) ]

	[(lp listproc)
	 (let ([name (##sys#debugger-read)])
	   (let ([info (##sys#find-info name #f)])
	     (if info
		 (##core#inline 
		  "C_display_string"
		  (##sys#port-file-resolve ##sys#standard-output)
		  (##sys#slot info 2) )
		 (display "| undefined procedure\n") ) ) ) ]

	[(vs variables)
	 (print "| variables:")
	 (list-names debug-variable-info-size) ]

	[(ps procedures)
	 (print "| procedures:")
	 (list-names debug-lambda-info-size) ]

	[(hi history)
	 (let ([len (length ##sys#debugger-history)])
	   (do ([i len (sub1 i)]
		[h ##sys#debugger-history (##sys#slot h 1)] )
	       ((null? h))
	     (display "| ")
	     (display i)
	     (display ": ")
	     (write (##sys#slot h 0))
	     (newline) ) ) ]

	[(sh shell)
	 (print "| " (system (##sys#debugger-read))) ]

	[(f frame)
	 (let ([f (##sys#get-frame (##sys#debugger-read))])
	   (when f (set! ##sys#selected-frame f))
	   (##sys#show-frame ##sys#selected-frame) ) ]

	[(tf topframe)
	 (set! ##sys#selected-frame (and (pair? ##sys#debug-frames) (##sys#slot ##sys#debug-frames 0)))
	 (##sys#show-frame ##sys#selected-frame) ]

	[(bt backtrace)
	 (show-call-stack ##sys#call-stack)
	 (let loop ([frames ##sys#debug-frames] [index 0])
	   (unless (null? frames)
	     (let* ([frame (##sys#slot frames 0)]
		    [info (##sys#slot frame 0)] )
	       (display "| ")
	       (write-char (if (eq? frame ##sys#selected-frame) #\> #\space))
	       (display index)
	       (display ": ")
	       (##sys#with-print-length-limit 
		debug-output-print-limit
		(lambda () (write (cons (##sys#slot info 0) (##sys#slot frame 3)))) )
	       (newline)
	       (let ([cs (##sys#slot frame 4)])
		 (unless (null? cs) (show-call-stack (##sys#slot cs 1))) )
	       (loop (##sys#slot frames 1) (add1 index)) ) ) )
	 (display "|     -toplevel frame-\n") ]

	[(c continue)
	 (set! ##sys#single-stepping #f)
	 (set! ##sys#last-step-command 'continue)
	 (##sys#continue-continuation #f) ]

	[(re restart)
	 (let* ([index (##sys#debugger-read)]
		[args (##sys#debugger-read)]
		[f (##sys#get-frame index)] )
	   (set! ##sys#single-stepping #f)
	   (when f
	     (set! ##sys#selected-frame f)
	     (let* ([info (##sys#slot f 0)]
		    [entry (##sys#slot f 2)] 
		    [argc (##sys#slot info 3)]
		    [rest (##sys#slot info 4)] )
	       (if (not (list? args))
		   (display "| argument list expected\n")
		   (let ([len (length args)])
		     (cond [(< len argc)
			    (display "| too few arguments - ")
			    (when rest (display "at least "))
			    (print argc " arguments were expected") ]
			   [(and (> len argc) (not rest))
			    (display "| too many arguments\n") ]
			   [else (entry (map eval-in-frame args))] ) ) ) ) ) ) ]

	[(le leave le* leave*)
	 (let* ([index (##sys#debugger-read)]
		[vals (##sys#debugger-read)] 
		[f (##sys#get-frame index)] )
	   (set! ##sys#single-stepping #f)
	   (let ([multi (memq cmd '(le* leave*))])
	     (when f
	       (set! ##sys#selected-frame f)
	       (let* ([info (##sys#slot f 0)]
		      [exit (##sys#slot f 1)] )
		 (if multi
		     (if (not (list? vals))
			 (display "| value list expected\n")
			 (apply exit (map eval-in-frame vals)) )
		     (exit (eval-in-frame vals)) ) ) ) ) ) ]

	[(th threads)
	 (for-each
	  (lambda (t)
	    (print "| " (##sys#slot t 6) ": " (##sys#slot t 3)) )
	  (##sys#all-threads) )
	 (newline) ]

	[(k kill)
	 (and-let* ([t (find-thread (##sys#debugger-read))])
	   (##sys#setslot t 3 'terminated)
	   (##sys#setslot t 2 (##core#undefined))
	   (##sys#setslot t 7 (##sys#make-structure 'terminated-thread-exception))
	   (##sys#abandon-mutexes thread) ) ]

	[(st suspend)
	 (let ([t (find-thread (##sys#debugger-read))])
	   (when t (##sys#setslot t 3 'suspended)) ) ]

	[(rt resume)
	 (let ([t (find-thread (##sys#debugger-read))])
	   (cond [(not t)]
		 [(eq? (##sys#slot t 3) 'suspended)
		  (##sys#setslot thread 3 'ready)
		  (##sys#add-to-ready-queue thread) ]
		 [else (display "| can not resume unsuspended thread\n")] ) ) ]

	[(ds display)
	 (set! ##sys#display-list 
	   (##sys#append 
	    ##sys#display-list
	    (##sys#list (eval-in-frame `(lambda () ,(##sys#debugger-read)))) ) ) ]

	[(ud undisplay)
	 (let ([index (##sys#debugger-read)])
	   (let loop ([i 1] [ds ##sys#display-list] [prev #f])
	     (cond [(null? ds)
		    (display "| displayed expression not found\n") ]
		   [(eq? index i)
		    (let ([next (##sys#slot ds 1)])
		      (if prev
			  (##sys#setslot prev 1 next)
			  (set! ##sys#display-list next) ) ) ]
		   [else (loop (add1 i) (##sys#slot ds 1) ds)] ) ) ) ]

	[(b break)
	 (let ([names (##sys#debugger-read)])
	   (for-each
	    (lambda (name)
	      (let ([info (##sys#find-info name #f)])
		(when info
		  (print "| breakpoint: " name)
		  (##sys#setslot info 5 (lambda () #t))
		  (set! ##sys#breakpoints (##sys#append ##sys#breakpoints (list (cons info #f)))) ) ) )
	    (if (pair? names) names (list names)) ) ) ]

	[(cb cbreak)
	 (let* ([name (##sys#debugger-read)]
		[test (##sys#debugger-read)] )
	   (let ([info (##sys#find-info name #f)])
	     (when info
	       (let ([proc
		      (eval
		       `(lambda () 
			  (apply 
			   (lambda ,(##sys#slot info 1) ,test)
			   (##sys#slot (##sys#slot ##sys#debug-frames 0) 3) ) ) ) ] )				   
		 (display "| conditional breakpoint: ")
		 (display name)
		 (display " -> ")
		 (write test)
		 (newline)
		 (##sys#setslot info 5 proc)
		 (set! ##sys#breakpoints (##sys#append ##sys#breakpoints (list (cons info test))) ) ) ) ) ) ]

	[(nb nbreak)
	 (let* ([name (##sys#debugger-read)]
		[n (##sys#debugger-read)] )
	   (let ([info (##sys#find-info name #f)])
	     (when info
	       (if (integer? n)
		   (let ([proc 
			  (let ([count 0])
			    (lambda ()
			      (set! count (add1 count))
			      (>= count n) ) ) ] 
			 [test `(>= <count> ,n)] )
		     (display "| conditional breakpoint: ")
		     (display name)
		     (display " -> ")
		     (write test)
		     (newline)
		     (##sys#setslot info 5 proc)
		     (set! ##sys#breakpoints (##sys#append ##sys#breakpoints (list (cons info test))) ) )
		   (display "| invalid count\n") ) ) ) ) ]

	[(ub unbreak)
	 (let ([names (##sys#debugger-read)])
	   (for-each
	    (lambda (name)
	      (let loop ([bs ##sys#breakpoints] [prev #f])
		(if (null? bs)
		    (print "| no breakpoint defined: " name)
		    (let* ([p (##sys#slot bs 0)]
			   [info (##sys#slot p 0)] )
		      (cond [(and (eq? name (##sys#slot info 0)) (##sys#slot info 5))
			     (##sys#setslot info 5 #f)
			     (print "| breakpoint " name " removed")
			     (if prev
				 (##sys#setslot prev 1 (##sys#slot bs 1))
				 (set! ##sys#breakpoints (##sys#slot bs 1)) ) ]
			    [else (loop (##sys#slot bs 1) bs)] ) ) ) ) )
	    (if (pair? names) names (list names)) ) ) ]

	[(nob nobreak)
	 (for-each (lambda (bp) (##sys#setslot (##sys#slot bp 0) 5 #f)) ##sys#breakpoints)
	 (print "| all breakpoints removed")
	 (set! ##sys#breakpoints '()) ]

	[(bs breakpoints)
	 (if (null? ##sys#breakpoints)
	     (display "| no breakpoints defined\n")
	     (for-each
	      (lambda (bp)
		(display "| ")
		(display (##sys#slot (##sys#slot bp 0) 0))
		(let ([tst (##sys#slot bp 1)])
		  (when tst
		    (display ": ")
		    (write tst) )
		  (newline) ) ) 
	      ##sys#breakpoints) ) ]

	[(w watch)
	 (let ([names (##sys#debugger-read)])
	   (for-each
	    (lambda (name)
	      (let ([info (##sys#find-info name #t)])
		(when info
		  (print "| watchpoint: " name)
		  (##sys#setslot info 2 (lambda (x) #t))
		  (set! ##sys#watchpoints (##sys#append ##sys#watchpoints (list (cons info #f)))) ) ) )
	    (if (pair? names) names (list names)) ) ) ]

	[(cw cwatch)
	 (let* ([name (##sys#debugger-read)]
		[test (##sys#debugger-read)] )
	   (let ([info (##sys#find-info name #t)])
	     (when info
	       (display "| conditional watchpoint: ")
	       (display name)
	       (display " -> ")
	       (write test)
	       (newline)
	       (##sys#setslot info 2 (eval `(lambda (,name) ,test)))
	       (set! ##sys#watchpoints (##sys#append ##sys#watchpoints (list (cons info test)))) ) ) ) ]

	[(uw unwatch)
	 (let ([names (##sys#debugger-read)])
	   (for-each
	    (lambda (name)
	      (let loop ([bs ##sys#watchpoints] [prev #f])
		(if (null? bs)
		    (print "| no watchpoint defined: " name)
		    (let* ([p (##sys#slot bs 0)]
			   [info (##sys#slot p 0)] )
		      (cond [(and (eq? name (##sys#slot info 0)) (##sys#slot info 2))
			     (##sys#setslot info 2 #f)
			     (print "| watchpoint " name " removed")
			     (if prev
				 (##sys#setslot prev 1 (##sys#slot bs 1))
				 (set! ##sys#watchpoints (##sys#slot bs 1)) ) ]
			    [else (loop (##sys#slot bs 1) bs)] ) ) ) ) )
	    (if (pair? names) names (list names)) ) ) ]

	[(now nowatch)
	 (for-each (lambda (bp) (##sys#setslot (##sys#slot bp 0) 2 #f)) ##sys#watchpoints)
	 (display "| all watchpoints removed\n")
	 (set! ##sys#watchpoints '()) ]

	[(ws watchpoints)
	 (if (null? ##sys#watchpoints)
	     (display "| no watchpoints defined\n")
	     (for-each
	      (lambda (bp)
		(display "| ")
		(display (##sys#slot (##sys#slot bp 0) 0))
		(let ([tst (##sys#slot bp 1)])
		  (when tst
		    (display ": ")
		    (write tst) )
		  (newline) ) )
	      ##sys#watchpoints) ) ]

	[(t trace)
	 (let* ([name (##sys#debugger-read)]
		[exp (##sys#debugger-read)] )
	   (let ([info (##sys#find-info name #f)])
	     (when info
	       (let ([proc
		      (eval
		       `(lambda () 
			  (apply 
			   (lambda ,(##sys#slot info 1) ,exp)
			   (##sys#slot (##sys#slot ##sys#debug-frames 0) 3) ) ) ) ] )
		 (display "| tracepoint: ")
		 (display name)
		 (display " -> ")
		 (write exp)
		 (newline)
		 (##sys#setslot info 6 proc)
		 (set! ##sys#tracepoints (##sys#append ##sys#tracepoints (list (cons info exp))) ) ) ) ) ) ]

	[(ut untrace)
	 (let ([names (##sys#debugger-read)])
	   (for-each
	    (lambda (name)
	      (let loop ([bs ##sys#tracepoints] [prev #f])
		(if (null? bs)
		    (print "| no tracepoint defined: " name)
		    (let* ([p (##sys#slot bs 0)]
			   [info (##sys#slot p 0)] )
		      (cond [(and (eq? name (##sys#slot info 0)) (##sys#slot info 6))
			     (##sys#setslot info 6 #f)
			     (print "| tracepoint " name " removed")
			     (if prev
				 (##sys#setslot prev 1 (##sys#slot bs 1))
				 (set! ##sys#tracepoints (##sys#slot bs 1)) ) ]
			    [else (loop (##sys#slot bs 1) bs)] ) ) ) ) )
	    (if (pair? names) names (list names)) ) ) ]

	[(not notrace)
	 (for-each (lambda (bp) (##sys#setslot (##sys#slot bp 0) 6 #f)) ##sys#tracepoints)
	 (print "| all tracepoints removed")
	 (set! ##sys#tracepoints '()) ]

	[(ts tracepoints)
	 (if (null? ##sys#tracepoints)
	     (display "| no tracepoints defined\n")
	     (for-each
	      (lambda (bp)
		(display "| ")
		(display (##sys#slot (##sys#slot bp 0) 0))
		(let ([tst (##sys#slot bp 1)])
		  (when tst
		    (display ": ")
		    (write tst) )
		  (newline) ) ) 
	      ##sys#tracepoints) ) ]

	[(h hop)
	 (set! ##sys#last-step-frames ##sys#debug-frames)
	 (set! ##sys#step-frame #f)
	 (set! ##sys#single-stepping #t)
	 (set! ##sys#last-step-command 'hop)
	 (##sys#continue-continuation #f) ]

	[(s skip)
	 (set! ##sys#last-step-frames ##sys#debug-frames)
	 (set! ##sys#step-frame ##sys#debug-frames)
	 (set! ##sys#single-stepping #t)
	 (set! ##sys#last-step-command 'skip)
	 (##sys#continue-continuation #f) ]

	[(j jump)
	 (set! ##sys#last-step-frames ##sys#debug-frames)
	 (set! ##sys#single-stepping #t)
	 (when (pair? ##sys#debug-frames)
	   (set! ##sys#step-frame (##sys#slot ##sys#debug-frames 1)) )
	 (set! ##sys#last-step-command 'jump)
	 (##sys#continue-continuation #f) ]

	[(|#|)
	 (let ([len (length ##sys#debugger-history)]
	       [cmd (##sys#debugger-read)] )
	   (do ([i len (sub1 i)]
		[h ##sys#debugger-history (##sys#slot h 1)] )
	       ((or (null? h) (= i cmd))
		(if (null? h)
		    (display "| invalid history index\n")
		    (receive rs (eval-in-frame (##sys#slot h 0))
		      (print-results rs) ) ) ) ) ) ]

	[(: eval)
	 (let ([form (##sys#debugger-read)])
	   (set! ##sys#debugger-history (cons form ##sys#debugger-history))
	   (receive rs (eval-in-frame form)
	     (print-results rs) ) ) ]

	[else
	 (cond [##sys#selected-frame
		(case cmd
			   
		  [(i info) (##sys#show-frame ##sys#selected-frame)]

		  [(d down)
		   (let loop ([frames ##sys#debug-frames] [prev #f])
		     (cond [(null? frames)
			    (display "| can not move beyond current frame\n") ]
			   [(and (eq? (##sys#slot frames 0) ##sys#selected-frame)
				 prev)
			    (set! ##sys#selected-frame prev)
			    (##sys#show-frame ##sys#selected-frame) ]
			   [else (loop (##sys#slot frames 1) (##sys#slot frames 0))] ) ) ]

		  [(u up)
		   (let loop ([frames ##sys#debug-frames])
		     (cond [(null? frames)
			    (display "| can not move beyond current frame\n") ]
			   [(and (eq? (##sys#slot frames 0) ##sys#selected-frame)
				 (pair? (##sys#slot frames 1)) )
			    (set! ##sys#selected-frame (##sys#slot (##sys#slot frames 1) 0))
			    (##sys#show-frame ##sys#selected-frame) ]
			   [else (loop (##sys#slot frames 1))] ) ) ]

		  [(ls list) 
		   (##core#inline 
		    "C_display_string"
		    (##sys#port-file-resolve ##sys#standard-output)
		    (##sys#slot (##sys#slot ##sys#selected-frame 0) 2) ) ]

		  [(r return)
		   (set! ##sys#single-stepping #f)
		   ((##sys#slot ##sys#selected-frame 1) (eval-in-frame (##sys#debugger-read))) ]

		  [(r* return*)
		   (set! ##sys#single-stepping #f)
		   (apply (##sys#slot ##sys#selected-frame 1) (map eval-in-frame (##sys#debugger-read))) ]

		  [else (bad-command)] ) ]

	       [(memq cmd frame-commands) (display "| no frame selected\n")]

	       [else (bad-command)] ) ] ) ) ) )

(define ##sys#list-debugger-commands
  (let ([display display])
    (lambda ()
      (display #<<EOF
Status:
  bt   backtrace                  Show currently active frames
  ls   list                       Display source code of current frame
  lp   listproc NAME              Display source code of procedure
  i    info                       Display information about current frame
  ds   display EXPR               Display expression before prompting
  ud   undisplay INDEX            Removed display-expression
  ps   procedures                 List named procedures
  vs   variables                  List global variables

Frame:
  f    frame INDEX                Select frame number INDEX
  u    up                         Move to caller's frame
  d    down                       Move to callee's frame
  tf   topframe                   Move to topmost frame

Execution:
  q    quit                       Terminate program (no cleanup!)
  c    continue                   Resume execution
  r    return VALUE 
  r*   return* (VALUE ...)        Return from current frame
  re   restart INDEX (ARG ...)    Restart frame INDEX
  le   leave INDEX VAL
  le*  leave* INDEX (VAL ...)     Return from frame INDEX
  h    hop                        Break before next procedure call
  s    skip                       Break before next procedure call in this frame
  j    jump                       Break before next procedure call in caller frame

Breakpoints:
  b    break NAME | (NAME ...)    Set breakpoint
  cb   cbreak NAME EXP            Set conditional breakpoint
  nb   nbreak NAME COUNT          Set counted breakpoint
  ub   unbreak NAME | (NAME ...)  Remove breakpoint
  nob  nobreak                    Remove all breakpoints
  bs   breakpoints                List breakpoints
  w    watch NAME | (NAME ...)    Set watchpoint on global variable
  cw   cwatch NAME EXP            Set conditional watchpoint
  uw   unwatch NAME | (NAME ...)  Remove watchpoint
  now  nowatch                    Remove all watchpoints
  ws   watchpoints                List watchpoints

Tracing:
  t    trace NAME EXP             Execute expression after procedure entry
  ut   untrace NAME | (NAME ...)  Remove execution point
  not  notrace                    Remove all execution points
  ts   tracepoints                List tracepoints

Threads:
  th   threads                    Show threads that are ready for scheduling
  k    kill THREAD                Terminate thread
  st   suspend THREAD             Suspend thread
  rt   resume THREAD              Resume suspended thread

Miscellaneous:
  ?    help                       Show this text
  :    eval EXP                   Evaluate expression
  hi   history                    Show list of evaluated expressions
  # NUMBER                        Evaluate expression from history list
  sh   shell STRING               Execute shell command


When EXP starts with `(', then `eval' (`:') can be omitted.
Any expressions are evaluated in a lexical environment with
the procedure parameters bound to the arguments of the current
frame.

Entering an empty line repeats the last `hop', `skip', `jump'
or `continue' command.

EOF
)
      (##core#undefined) ) ) )

(define ##sys#read-debugger-command
  (let ([read-char read-char]
	[peek-char peek-char] 
	[reverse reverse]
	[list->string list->string] )
    (define (fini inp)
      (##sys#string->symbol (list->string (reverse inp))) )
    (lambda ()
      (let loop ([input '()])
	(let ([next (peek-char)])
	  (cond [(char=? #\( next)
		 (if (null? input) ': (fini input)) ]
		[(char=? #\: next)
		 (read-char)
		 (if (null? input)
		     ':
		     (begin
		       (display "| invalid debugger command syntax\n")
		       #f) ) ]
		[(char=? #\newline next)
		 (read-char)
		 (if (null? input)
		     (or ##sys#last-step-command
			 (begin
			   (display "| no previous stepping command\n")
			   #f) ) 
		     (fini input) ) ]
		[(char-whitespace? next)
		 (read-char)
		 (if (null? input)
		     (loop '())
		     (let skip ()
		       (let ([next (peek-char)])
			 (cond [(char-whitespace? next)
				(read-char)
				(if (char=? #\newline next)
				    (fini input)
				    (skip) ) ]
			       [else (fini input)] ) ) ) ) ]
		[else (loop (cons (read-char) input))] ) ) ) ) ) )

(define ##sys#debugger-read
  (let ([read read]
	[peek-char peek-char]
	[read-char read-char] )
    (lambda ()
      (fluid-let ([##sys#current-namespace #f])
	(let ([exp (read)])
	  (let skip ()
	    (let ([next (peek-char)])
	      (when (char-whitespace? next)
		(let ([next (read-char)])
		  (unless (char=? #\newline next) (skip)) ) ) ) )
	  exp) ) ) ) )

(define ##sys#debugger-repl
  (let ([eval eval]
	[read read]
	[write write]
	[call-with-current-continuation call-with-current-continuation]
	[display display]
	[reset reset]
	[flush-output flush-output]
	[newline newline] )
    (lambda ()

      (define (writeargs xs)
	(fluid-let ([##sys#print-qualifiers #t])
	  (if (or (null? xs) (pair? (cdr xs)) (not (eq? (##core#undefined) (car xs))))
	      (##sys#for-each 
	       (lambda (x)
		 (##sys#with-print-length-limit debug-repl-result-print-limit (lambda () (write x)))
		 (newline) )
	       xs) ) ) )

      (let ([stdin ##sys#standard-input]
	    [stdout ##sys#standard-output]
	    [stderr ##sys#standard-error] 
	    [ehandler (##sys#error-handler)] 
	    [rhandler (##sys#reset-handler)] )

	(define (saveports)
	  (set! stdin ##sys#standard-input)
	  (set! stdout ##sys#standard-output)
	  (set! stderr ##sys#standard-error) )

	(define (resetports)
	  (set! ##sys#standard-input stdin)
	  (set! ##sys#standard-output stdout)
	  (set! ##sys#standard-error stderr) )

	(saveports)
	(##sys#dynamic-wind
	 (lambda ()
	   (##sys#error-handler
	    (lambda (msg . args)
	      (resetports)
	      (display "Error: ")
	      (display msg)
	      (newline)
	      (writeargs args) ) ) )
	 (lambda ()
	   (let loop ()
	     (call-with-current-continuation
	      (lambda (c)
		(##sys#reset-handler
		 (lambda ()
		   (set! ##sys#read-error-with-line-number #f)
		   (set! ##sys#default-namespace-prefix #f)
		   (set! ##sys#enable-qualifiers #t)
		   (resetports)
		   (c #f) ) ) ) )
	     (when (or (##sys#fudge 12) (##sys#tty-port? ##sys#standard-input))
	       (let loop ([i 1] [ds ##sys#display-list])
		 (unless (null? ds)
		   (display "| ")
		   (display i)
		   (display ": ")
		   (write ((##sys#slot ds 0)))
		   (newline)
		   (loop (add1 i) (##sys#slot ds 1)) ) )
	       (display "| [")
	       (display ##sys#break-level)
	       (display "]: ")
	       (flush-output) )
	     (let ([cmd (##sys#read-debugger-command)])
	       (##sys#execute-debugger-command cmd) )
	     (loop) ) )
	 (lambda ()
	   (##sys#error-handler ehandler)
	   (##sys#reset-handler rhandler) ) ) ) ) ) )
