;;;; eval.scm - Interpreter for CHICKEN
;
; Copyright (c) 2000-2002, Felix L. Winkelmann
; All rights reserved.
;
; Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following
; conditions are met:
;
;   Redistributions of source code must retain the above copyright notice, this list of conditions and the following
;     disclaimer. 
;   Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following
;     disclaimer in the documentation and/or other materials provided with the distribution. 
;   Neither the name of the author nor the names of its contributors may be used to endorse or promote
;     products derived from this software without specific prior written permission. 
;
; THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS
; OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
; AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR
; CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
; SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
; THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
; OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
; POSSIBILITY OF SUCH DAMAGE.
;
; Send bugs, suggestions and ideas to: 
;
; felix@call-with-current-continuation.org
;
; Felix L. Winkelmann
; Steinweg 1A
; 37130 Gleichen, OT Weissenborn
; Germany


(declare
  (unit eval)
  (standard-bindings)
  (extended-bindings)
  (no-bound-checks)
  (bound-to-procedure 
   ##sys#check-char ##sys#check-exact ##sys#check-port ##sys#check-string
   ##sys#for-each ##sys#map ##sys#setslot ##sys#allocate-vector ##sys#check-pair ##sys#not-a-proper-list-error
   ##sys#check-symbol ##sys#check-vector ##sys#floor ##sys#ceiling ##sys#truncate ##sys#round 
   ##sys#check-number ##sys#cons-flonum ##sys#copy-environment
   ##sys#flonum-fraction ##sys#make-port ##sys#fetch-and-check-port-arg ##sys#print ##sys#check-structure 
   ##sys#make-structure ##sys#test-feature
   ##sys#error-handler ##sys#hash-symbol ##sys#register-macro ##sys#check-syntax
   ##sys#hash-table-ref ##sys#hash-table-set! ##sys#canonicalize-body ##sys#decompose-lambda-list
   ##sys#make-c-string ##sys#make-promise ##sys#resolve-include-filename ##sys#register-macro-2 
   ##sys#load ##sys#error ##sys#warn ##sys#hash-table-location
   ##sys#make-flonum ##sys#make-pointer ##sys#null-pointer ##sys#address->pointer 
   ##sys#pointer->address ##sys#compile-to-closure
   ##sys#ensure-heap-reserve ##sys#syntax-error-hook ##sys#read-prompt-hook
   ##sys#repl-eval-hook ##sys#append ##sys#secondary-macroexpand
   ##sys#macroexpand-hook ##sys#expand-module-definition ##sys#macroexpand-0) )


(cond-expand
 [unsafe
  (eval-when (compile)
    (define-macro (##sys#check-structure x y) '(##core#undefined))
    (define-macro (##sys#check-range x y z) '(##core#undefined))
    (define-macro (##sys#check-pair x) '(##core#undefined))
    (define-macro (##sys#check-list x) '(##core#undefined))
    (define-macro (##sys#check-symbol x) '(##core#undefined))
    (define-macro (##sys#check-string x) '(##core#undefined))
    (define-macro (##sys#check-char x) '(##core#undefined))
    (define-macro (##sys#check-exact x) '(##core#undefined))
    (define-macro (##sys#check-port x) '(##core#undefined))
    (define-macro (##sys#check-number x) '(##core#undefined))
    (define-macro (##sys#check-byte-vector x) '(##core#undefined)) ) ]
 [else] )


(include "parameters")


;;; Macro handling:

(define ##sys#macro-environment (make-vector macro-table-size '()))

(define (##sys#register-macro-2 name handler)
  (##sys#hash-table-set! 
   ##sys#macro-environment name
   (lambda (form) (handler (##sys#slot form 1))) ) )

(define ##sys#register-macro
  (lambda (name handler)
    (##sys#hash-table-set! 
     ##sys#macro-environment name
     (lambda (form) (apply handler (##sys#slot form 1))) ) ) )

(define (macro? sym)
  (##sys#check-symbol sym)
  (and (##sys#hash-table-ref ##sys#macro-environment sym) #t) )

(define (##sys#unregister-macro name)
  (##sys#hash-table-set! ##sys#macro-environment name #f) )

(define (undefine-macro! name)
  (##sys#check-symbol name)
  (##sys#unregister-macro name) )

(define (##sys#macroexpand-0 exp me)

  (define (expand exp head)
    (cond [(assq head me) => (lambda (mdef) (values ((##sys#slot mdef 1) exp) #t))]
	  [(##sys#hash-table-ref ##sys#macro-environment head) 
	   => (lambda (handler) (values (handler exp) #t)) ]
	  [else (values exp #f)] ) )

  (cond [(symbol? exp) (expand exp exp)]
	[(pair? exp)
	 (let ([head (##sys#slot exp 0)]
	       [body (##sys#slot exp 1)] )
	   (if (symbol? head)
	       (cond [(eq? head 'let)
		      (##sys#check-syntax 'let body '#(_ 2))
		      (let ([bindings (car body)])
			(cond [(symbol? bindings)
			       (##sys#check-syntax 'let body '(_ #((variable _) 0) . #(_ 1)))
			       (let ([bs (cadr body)])
				 `((letrec ([,bindings (##core#loop-lambda ,(map (lambda (b) (car b)) bs) ,@(cddr body))])
				     ,bindings)
				   ,@(##sys#map cadr bs) ) ) ]
			      [else (values exp #f)] ) ) ]
		     [else (expand exp head)] )
	       (values exp #f) ) ) ]
	[else (values exp #f)] ) )

(define (##sys#macroexpand-hook exp me)
  (let loop ([exp exp])
    (let-values ([(exp2 m) (##sys#macroexpand-0 exp me)])
      (if m
	  (loop exp2)
	  exp2) ) ) )

(define (##sys#macroexpand-1-hook exp me)
  (nth-value 0 (##sys#macroexpand-0 exp me)) )

(define (##sys#compiler-toplevel-macroexpand-hook exp) exp)

(define (macroexpand exp . me)
  (##sys#macroexpand-hook exp (if (pair? me) (car me) '())) )

(define (macroexpand-1 exp . me)
  (##sys#macroexpand-1-hook exp (if (pair? me) (car me) '())) )

(define ##sys#strict-mode #f)

(define (##sys#undefine-non-standard-macros leave)
  (let ([leave (##sys#append leave '(define and or cond case let* letrec do quasiquote delay))])
    (do ([i 0 (fx+ i 1)])
	((fx>= i macro-table-size))
      (##sys#setslot 
       ##sys#macro-environment i
       (let loop ([bs (##sys#slot ##sys#macro-environment i)])
	 (if (null? bs)
	     '()
	     (let ([b (##sys#slot bs 0)]
		   [r (##sys#slot bs 1)] )
	       (if (memq (##sys#slot b 0) leave)
		   (cons b (loop r))
		   (loop r) ) ) ) ) ) ) ) )


;;; "Compiler" macros:

(define ##sys#secondary-macro-table '())

(define (##sys#secondary-macroexpand x)
  (if (and (pair? x) (symbol? (##sys#slot x 0)))
      (let ([a (assq (##sys#slot x 0) ##sys#secondary-macro-table)])
	(if a 
	    (##sys#secondary-macroexpand ((##sys#slot a 1) x))
	    x) )
      x) )


;;; Expansion of bodies:

(define ##sys#canonicalize-body
  (let ([reverse reverse]
	[map map] )
    (define (fini vars vals mvars mvals body)
      (if (and (null? vars) (null? mvars))
	  `(begin ,@body)
	  (let ([vars (reverse vars)])
	    `(let ,(##sys#map (lambda (v) (##sys#list v (##sys#list '##core#undefined))) 
			  (apply ##sys#append vars mvars) )
	       ,@(map (lambda (v x) `(set! ,v ,x)) vars (reverse vals))
	       ,@(map (lambda (vs x)
			(let ([tmps (##sys#map gensym vs)])
			  `(##sys#call-with-values (lambda () ,x)
			     (lambda ,tmps 
			       ,@(map (lambda (v t) `(set! ,v ,t)) vs tmps) ) ) ) ) 
		      (reverse mvars)
		      (reverse mvals) )
	       ,@body) ) ) )
    (lambda (body)
      (let loop ([body body] [vars '()] [vals '()] [mvars '()] [mvals '()])
	(if (not (pair? body))
	    `(begin ,@body)
	    (let* ([x (##sys#slot body 0)]
		   [rest (##sys#slot body 1)] 
		   [head (and (pair? x) (##sys#slot x 0))] )
	      (cond [(not head) (fini vars vals mvars mvals body)]
		    [(eq? 'define head)
		     (##sys#check-syntax 'define x '(define _ . #(_ 1)) #f)
		     (let ([head (cadr x)])
		       (cond [(not (pair? head))
			      (##sys#check-syntax 'define x '(define variable _) #f)
			      (loop rest (cons head vars) (cons (caddr x) vals) mvars mvals) ]
			     [else
			      (##sys#check-syntax 'define x '(define (variable . lambda-list) . #(_ 1)) #f)
			      (loop rest
				    (cons (##sys#slot head 0) vars)
				    (cons `(lambda ,(##sys#slot head 1) ,@(cddr x)) vals)
				    mvars mvals) ] ) ) ]
		    [(eq? 'define-values head)
		     (##sys#check-syntax 'define-values x '(define-values #(_ 0) _) #f)
		     (loop rest vars vals (cons (cadr x) mvars) (cons (caddr x) mvals)) ]
		    [(eq? 'begin head)
		     (##sys#check-syntax 'begin x '(begin . #(_ 1)) #f)
		     (loop (##sys#append (##sys#slot x 1) rest) vars vals mvars mvals) ]
		    [else (fini vars vals mvars mvals body)] ) ) ) ) ) ) )

(define ##sys#match-expression
  (lambda (exp pat vars)
    (let ((env '()))
      (define (mwalk x p)
	(cond ((or (not (##core#inline "C_blockp" p)) (not (##core#inline "C_pairp" p)))
	       (cond ((assq p env) => (lambda (a) (equal? x (##sys#slot a 1))))
		     ((memq p vars)
		      (set! env (cons (cons p x) env))
		      #t)
		     (else (eq? x p)) ) )
	      ((or (not (##core#inline "C_blockp" x)) (not (##core#inline "C_pairp" x))) #f)
	      ((mwalk (##sys#slot x 0) (##sys#slot p 0))
	       (mwalk (##sys#slot x 1) (##sys#slot p 1)) )
	      (else #f) ) )
      (and (mwalk exp pat) env) ) ) )


;;; Lo-level hashtable support:

(define ##sys#hash-symbol
  (let ([cache-s #f]
	[cache-h #f] )
    (lambda (s n)
      (if (eq? s cache-s)
	  (##core#inline "C_fixnum_modulo" cache-h n)
	  (let ([h (##core#inline "C_hash_string" (##sys#slot s 1))])
	    (set! cache-s s)
	    (set! cache-h h)
	    (##core#inline "C_fixnum_modulo" h n) ) ) ) ) )

(define (##sys#hash-table-ref ht key)
  (let ((k (##sys#hash-symbol key (##core#inline "C_block_size" ht))))
    (let loop ((bucket (##sys#slot ht k)))
      (if (eq? bucket '())
	  #f
	  (let ((b (##sys#slot bucket 0)))
	    (if (eq? key (##sys#slot b 0))
		(##sys#slot b 1)
		(loop (##sys#slot bucket 1)) ) ) ) ) ) )

(define ##sys#hash-table-set! 
  (lambda (ht key val)
    (let* ((k (##sys#hash-symbol key (##core#inline "C_block_size" ht)))
	   (bucket0 (##sys#slot ht k)) )
      (let loop ((bucket bucket0))
	(if (eq? bucket '())
	    (##sys#setslot ht k (cons (cons key val) bucket0))
	    (let ((b (##sys#slot bucket 0)))
	      (if (eq? key (##sys#slot b 0))
		  (##sys#setslot b 1 val)
		  (loop (##sys#slot bucket 1)) ) ) ) ) ) ) )

(define (##sys#hash-table-for-each p ht)
  (let ((len (##core#inline "C_block_size" ht)))
    (do ((i 0 (fx+ i 1)))
	((fx>= i len))
      (##sys#for-each (lambda (bucket) 
		   (p (##sys#slot bucket 0)
		      (##sys#slot bucket 1) ) )
		 (##sys#slot ht i) ) ) ) )

(define (##sys#hash-table-location ht key)
  ;; This is for R5RS environments: undefined entries are initialized with #(<sym> <unbound> #t)
  (let* ([k (##sys#hash-symbol key (##core#inline "C_block_size" ht))]
	 [bucket0 (##sys#slot ht k)] 
	 [unbound (##sys#slot '##sys#arbitrary-unbound-symbol 0)] )
    (let loop ([bucket bucket0])
      (if (eq? bucket '())
	  (let ([p (vector key unbound #t)])
	    (##sys#setslot ht k (cons p bucket0))
	    p)
	  (let ([b (##sys#slot bucket 0)])
	    (if (eq? key (##sys#slot b 0))
		b
		(loop (##sys#slot bucket 1)) ) ) ) ) ) )


;;; Promises:

(define ##sys#make-promise
  (let ([values values])
    (lambda (proc)
      (let ([result-ready #f]
	    [results #f] )
	(##sys#make-structure
	 'promise
	 (lambda ()
	   (if result-ready
	       (apply values results)
	       (##sys#call-with-values 
		proc
		(lambda xs
		  (if result-ready
		      (apply values results)
		      (begin
			(set! result-ready #t)
			(set! results xs)
			(apply values results) ) ) ) ) ) ) ) ) ) ) )


;;; Compile lambda to closure:

(define ##sys#eval-environment #f)
(define ##sys#unqualified-quoted-symbols #f)

(define ##sys#compile-to-closure
  (let ([macroexpand-1 macroexpand-1]
	[macro? macro?]
	[write write]
	[cadadr cadadr]
	[reverse reverse]
	[with-input-from-file with-input-from-file]
	[unbound (##sys#slot '##sys#arbitrary-unbound-symbol 0)]
	[display display] )
    (lambda (exp env me)

      (define (lookup var e)
	(let loop ((envs e) (ei 0))
	  (cond ((null? envs) (values #f var))
		((posq var (##sys#slot envs 0)) => (lambda (p) (values ei p)))
		(else (loop (##sys#slot envs 1) (fx+ ei 1))) ) ) )

      (define (defined? var e)
	(receive (i j) (lookup var e) i) )

      (define (undefine vars e)
	(let loop ([envs e])
	  (if (null? envs)
	      '()
	      (let ([envi (##sys#slot envs 0)])
		(cons
		 (let delq ([ee envi])
		   (if (null? ee)
		       '()
		       (let ([h (##sys#slot ee 0)]
			     [r (##sys#slot ee 1)] )
			 (if (memq h vars)
			     r
			     (cons h (delq r)) ) ) ) )
		 (loop (##sys#slot envs 1)) ) ) ) ) )

      (define (posq x lst)
	(let loop ((lst lst) (i 0))
	  (cond ((null? lst) #f)
		((eq? x (##sys#slot lst 0)) i)
		(else (loop (##sys#slot lst 1) (fx+ i 1))) ) ) )

      (define (take-append-and-split lst n x)
	(let* ([rllist #f]
	       [llist2
		(let loop ([lst lst] [n n])
		  (cond [(fx= n 0)
			 (set! rllist lst)
			 x]
			[else (##sys#cons (##sys#slot lst 0) (loop (##sys#slot lst 1) (fx- n 1)))] ) ) ] )
	  (values llist2 rllist) ) )

      (define (macroexpand-1-checked x e me)
	(let ([x2 (##sys#secondary-macroexpand (macroexpand-1 x me))])
	  (if (pair? x2)
	      (let ([h (##sys#slot x2 0)])
		(if (and (eq? h 'let) (not (defined? 'let e)))
		    (let ([next (##sys#slot x2 1)])
		      (if (and (pair? next) (symbol? (##sys#slot next 0)))
			  (macroexpand-1-checked x2 e me)
			  x2) )
		    x2) )
	      x2) ) )

      (define (compile x e h me)
	(cond [(symbol? x)
	       (receive (i j) (lookup x e)
		 (cond [(not i)
			(let ([y (macroexpand-1-checked x e me)])
			  (if (eq? x y)
			      (if ##sys#eval-environment
				  (let ([loc (##sys#hash-table-location ##sys#eval-environment x)])
				    (cond-expand 
				     [unsafe (lambda v (##sys#slot loc 1))]
				     [else
				      (lambda v 
					(let ([val (##sys#slot loc 1)])
					  (if (eq? unbound val)
					      (##sys#error "unbound variable" x)
					      val) ) ) ] ) )
				  (cond-expand
				   [unsafe (lambda v (##core#inline "C_slot" x 0))]
				   [else (lambda v (##core#inline "C_retrieve" x))] ) )
			      (compile y e h me) ) ) ]
		       [(zero? i) (lambda (v) (##sys#slot (##sys#slot v 0) j))]
		       [else (lambda (v) (##sys#slot (##core#inline "C_u_i_list_ref" v i) j))] ) ) ]
	      [(number? x)
	       (case x
		 [(-1) (lambda v -1)]
		 [(0) (lambda v 0)]
		 [(1) (lambda v 1)]
		 [(2) (lambda v 2)]
		 [else (lambda v x)] ) ]
	      [(boolean? x)
	       (if x
		   (lambda v #t)
		   (lambda v #f) ) ]
	      [(or (char? x)
		   (string? x) )
	       (lambda v x) ]
	      [(not (pair? x)) (##sys#error "syntax error - illegal non-atomic object" x)]
	      [(symbol? (##sys#slot x 0))
	       (let ([head (##sys#slot x 0)])
		 (if (defined? head e)
		     (compile-call x e me)
		     (let ([x2 (macroexpand-1-checked x e me)])
		       (if (eq? x2 x)
			   (case head

			     [(quote)
			      (##sys#check-syntax 'quote x '(quote _) #f)
			      (let* ([c0 (cadr x)]
				     [c (if ##sys#unqualified-quoted-symbols
					    (##sys#unqualify-quoted-symbols c0)
					    c0) ] )
				(case c
				  [(-1) (lambda v -1)]
				  [(0) (lambda v 0)]
				  [(1) (lambda v 1)]
				  [(2) (lambda v 2)]
				  [(#t) (lambda v #t)]
				  [(#f) (lambda v #f)]
				  [(()) (lambda v '())]
				  [else (lambda v c)] ) ) ]

			     [(##core#qualified)
			      (fluid-let ([##sys#unqualified-quoted-symbols #f])
				(compile (cadr x) e h me) ) ]

			     [(##core#immutable)
			      (##sys#check-syntax '##core#immutable x '(##core#immutable (quote _)) #f)
			      (compile (cadr x) e #f me) ]
		   
			     [(##core#undefined) (lambda (v) (##core#undefined))]

			     [(if)
			      (##sys#check-syntax 'if x '(if _ _ . #(_)) #f)
			      (let* ([test (compile (cadr x) e #f me)]
				     [cns (compile (caddr x) e #f me)]
				     [alt (if (pair? (cdddr x))
					      (compile (cadddr x) e #f me)
					      (compile '(##core#undefined) e #f me) ) ] )
				(lambda (v) (if (test v) (cns v) (alt v))) ) ]

			     [(begin)
			      (##sys#check-syntax 'begin x '(begin . #(_ 0)) #f)
			      (let* ([body (##sys#slot x 1)]
				     [len (length body)] )
				(case len
				  [(0) (compile '(##core#undefined) e #f me)]
				  [(1) (compile (##sys#slot body 0) e #f me)]
				  [(2) (let* ([x1 (compile (##sys#slot body 0) e #f me)]
					      [x2 (compile (cadr body) e #f me)] )
					 (lambda (v) (x1 v) (x2 v)) ) ]
				  [else
				   (let* ([x1 (compile (##sys#slot body 0) e #f me)]
					  [x2 (compile (cadr body) e #f me)] 
					  [x3 (compile `(begin ,@(##sys#slot (##sys#slot body 1) 1)) e #f me)] )
				     (lambda (v) (x1 v) (x2 v) (x3 v)) ) ] ) ) ]

			     [(set!)
			      (##sys#check-syntax 'set! x '(set! variable _) #f)
			      (let ([var (cadr x)])
				(receive (i j) (lookup var e)
				  (let ([val (compile (caddr x) e var me)])
				    (cond [(not i)
					   (when (macro? var)
					     (##sys#warn "assigned global variable is a macro" var) )
					   (if ##sys#eval-environment
					       (let ([loc (##sys#hash-table-location ##sys#eval-environment var)])
						 (if (##sys#slot loc 2)
						     (lambda (v) (##sys#setslot loc 1 (val v)))
						     (lambda v (##sys#error "assignment to immutable variable" var)) ) )
					       (lambda (v) (##sys#setslot j 0 (val v))) ) ]
					  [(zero? i) (lambda (v) (##sys#setslot (##sys#slot v 0) j (val v)))]
					  [else
					   (lambda (v)
					     (##sys#setslot (##core#inline "C_u_i_list_ref" v i) j (val v)) ) ] ) ) ) ) ]

			     [(let)
			      (##sys#check-syntax 'let x '(let #((variable _) 0) . #(_ 1)) #f)
			      (let ([bindings (cadr x)])
				(compile
				 `((lambda ,(##sys#map (lambda (x) (car x)) bindings) 
				     ,@(##sys#slot (##sys#slot x 1) 1) )
				   ,@(##sys#map (lambda (x) (cadr x)) bindings) )
				 e #f me) ) ]

			     [(lambda)
			      (##sys#check-syntax 'lambda x '(lambda lambda-list . #(_ 1)) #f)
			      (##sys#decompose-lambda-list
			       (cadr x)
			       (lambda (vars argc rest)
				 (let ([body (##sys#compile-to-closure
					      (##sys#canonicalize-body (cddr x))
					      (cons vars e)
					      me) ] 
				       [vars (##sys#list->vector vars)] )
				   (case argc
				     [(0) (if rest
					      (lambda (v) (lambda r (body (cons (vector r) v))))
					      (lambda (v) (lambda () (body (cons #f v))))) ]
				     [(1) (if rest
						  (lambda (v) (lambda (a1 . r) (body (cons (vector a1 r) v))))
						  (lambda (v) (lambda (a1) (body (cons (vector a1) v))))) ]
				     [(2) (if rest
					      (lambda (v) (lambda (a1 a2 . r) (body (cons (vector a1 a2 r) v))))
					      (lambda (v) (lambda (a1 a2) (body (cons (vector a1 a2) v))))) ]
				     [(3) (if rest
					      (lambda (v) (lambda (a1 a2 a3 . r) (body (cons (vector a1 a2 a3 r) v))))
					      (lambda (v) (lambda (a1 a2 a3) (body (cons (vector a1 a2 a3) v))))) ]
				     [(4) (if rest
					      (lambda (v)
						(lambda (a1 a2 a3 a4 . r) (body (cons (vector a1 a2 a3 a4 r) v))) )
					      (lambda (v)
						(lambda (a1 a2 a3 a4) (body (##sys#cons (##sys#vector a1 a2 a3 a4) v))))) ]
				     [else (if rest
					       (lambda (v)
						 (lambda as
						   (body (##sys#cons (apply ##sys#vector (fudge-argument-list argc as)) v)) ) )
					       (lambda (v)
						 (lambda as 
						   (let ([len (length as)])
						     (if (not (fx= len argc))
							 (##sys#error "bad argument count" argc len)
							 (body (##sys#cons (apply ##sys#vector as) v))) ) ) ) ) ] ) ) ) ) ]

			     [(##core#loop-lambda)
			      (compile `(lambda ,@(cdr x)) e #f me) ]

			     [(##core#named-lambda)
			      (##sys#check-syntax '##core#named-lambda x '(##core#named-lambda symbol lambda-list . #(_ 1)))
			      (compile `(lambda ,@(cddr x)) e (cadr x) me) ]

			     [(##core#enable-unqualified-quoted-symbols)
			      (set! ##sys#unqualified-quoted-symbols #t)
			      (compile `(##core#undefined) e h me) ]

			     [(##core#disable-unqualified-quoted-symbols)
			      (set! ##sys#unqualified-quoted-symbols #f)
			      (compile `(##core#undefined) e h me) ]

			     [(##core#elaborationtimeonly ##core#elaborationtimetoo) ; <- Note this!
			      ((##sys#compile-to-closure (cadr x) '() '()) '())
			      (compile '(##core#undefined) e #f me) ]

			     [(##core#compiletimetoo)
			      (compile (cadr x) e #f me) ]

			     [(##core#compiletimeonly ##core#declare ##core#callunit) 
			      (compile '(##core#undefined) e #f me) ]

                             [(##core#define-inline ##core#define-constant)
                              (compile `(define ,(cadadr x) ,@(cddr x)) e #f me) ]
                   
			     [(##core#include)
			      (compile
			       (with-input-from-file (##sys#resolve-include-filename (cadadr x))
				 (lambda ()
				   (do ([x (read) (read)]
					[xs '() (cons x xs)] )
				       ((eof-object? x) 
					#;(pretty-print (reverse xs))
					`(begin ,@(reverse xs))) ) ) )
			       e #f me) ]

			     [(##core#define-module)
			      (compile
			       (##sys#expand-module-definition
				(cadadr x)
				(cadr (caddr x)) 
				##sys#error) 
			       e #f me) ]

			     [(##core#primitive ##core#inline ##core#inline_allocate ##core#foreign-lambda 
			       ##core#define-foreign-variable ##core#define-external-variable
			       ##core#define-foreign-parameter ##core#define-foreign-type ##core#foreign-lambda*)
			      (##sys#error "syntax error - can not evaluate compiler-special-form" x) ]

			     [else
			      (cond [##sys#strict-mode (compile-call x e me)]

				    [(eq? head 'let-macro)
				     (##sys#check-syntax 'let-macro x '(let-macro #(list 0) . #(_ 1)) #f)
				     (set! ##sys#syntax-error-culprit #f)
				     (let ([me2 (##sys#expand-local-macrodefs (cadr x))])
				       (compile
					(##sys#canonicalize-body (cddr x))
					(undefine (map (lambda (x) (car x)) me2) e)
					#f
					(##sys#append me2 me) ) ) ]

				    [(eq? head 'let-id-macro)
				     (##sys#check-syntax 'let-id-macro x '(let-id-macro #((symbol _) 0) . #(_ 1)) #f)
				     (let ([me2 (map (lambda (mdef) 
						       (cons (car mdef) (lambda (form) (cadr mdef))) )
						     (cadr x) ) ] )
				       (compile
					(##sys#canonicalize-body (cddr x))
					(undefine (map (lambda (m) (##sys#slot m 0)) me2) e)
					#f
					(##sys#append me2 me) ) ) ]

				    [(eq? head 'external-pointer)
				     (##sys#error "syntax error - can not evaluate compiler-special-form" x) ]

				    [else (compile-call x e me)] ) ] )

			   (compile x2 e h me) ) ) ) ) ]

	      [else (compile-call x e me)] ) )

      (define (fudge-argument-list n alst)
	(if (null? alst) 
	    (list alst)
	    (do ([n n (fx- n 1)]
		 [args alst (##sys#slot args 1)]
		 [last #f args] )
		((fx= n 0)
		 (##sys#setslot last 1 (list args))
		 alst) ) ) )

      (define (compile-call x e me)
	(let* ([fn (compile (##sys#slot x 0) e #f me)]
	       [args (##sys#slot x 1)]
	       [argc (length args)] )
	  (case argc
	    [(0) (lambda (v) ((fn v)))]
	    [(1) (let ([a1 (compile (##sys#slot args 0) e #f me)])
		   (lambda (v) ((fn v) (a1 v))) ) ]
	    [(2) (let* ([a1 (compile (##sys#slot args 0) e #f me)]
			[a2 (compile (##core#inline "C_u_i_list_ref" args 1) e #f me)] )
		   (lambda (v) ((fn v) (a1 v) (a2 v))) ) ]
	    [(3) (let* ([a1 (compile (##sys#slot args 0) e #f me)]
			[a2 (compile (##core#inline "C_u_i_list_ref" args 1) e #f me)]
			[a3 (compile (##core#inline "C_u_i_list_ref" args 2) e #f me)] )
		   (lambda (v) ((fn v) (a1 v) (a2 v) (a3 v))) ) ]
	    [(4) (let* ([a1 (compile (##sys#slot args 0) e #f me)]
			[a2 (compile (##core#inline "C_u_i_list_ref" args 1) e #f me)]
			[a3 (compile (##core#inline "C_u_i_list_ref" args 2) e #f me)] 
			[a4 (compile (##core#inline "C_u_i_list_ref" args 3) e #f me)] )
		   (lambda (v) ((fn v) (a1 v) (a2 v) (a3 v) (a4 v))) ) ]
	    [else (let ([as (##sys#map (lambda (a) (compile a e #f me)) args)])
		    (lambda (v) (apply (fn v) (##sys#map (lambda (a) (a v)) as))) ) ] ) ) )

      (compile exp env #f me) ) ) )

(define ##sys#expand-module-definition
  (lambda args
    (##sys#error "module system not available") ) )

(define ##sys#eval-handler 
  (make-parameter
   (lambda (x . env)
     (let ([sm #f]
	   [e #f] )
       (when (pair? env)
	 (let ([env (car env)])
	   (when env
	     (##sys#check-structure env 'environment)
	     (set! e (##sys#slot env 1)) )
	   (when e (set! sm #t)) ) )
       ((fluid-let ([##sys#strict-mode sm]
		    [##sys#eval-environment e] )
	  (##sys#compile-to-closure x '() '()) )
	'() ) ) ) ) )

(define eval-handler ##sys#eval-handler)
(define (eval . args) (apply (##sys#eval-handler) args))


;;; R5RS environments:

(define ##sys#r4rs-environment (make-vector 499 '()))
(define ##sys#r5rs-environment #f)

(define ##sys#copy-environment
  (let ([make-vector make-vector])
    (lambda (e)
      (let* ([s (##sys#size e)]
	     [e2 (make-vector s '())] )
	(do ([i 0 (fx+ i 1)])
	    ((fx>= i s) e2)
	  (##sys#setslot 
	   e2 i
	   (let copy ([b (##sys#slot e i)])
	     (if (null? b)
		 '()
		 (let ([bi (##sys#slot b 0)])
		   (cons (vector (##sys#slot bi 0) (##sys#slot bi 1) (##sys#slot bi 2))
			 (copy (##sys#slot b 1)) ) ) ) ) ) ) ) ) ) )

(let ()
  (define (initb ht) 
    (lambda (b)
      (let ([loc (##sys#hash-table-location ht b)])
	(##sys#setslot loc 1 (##sys#slot b 0))
	(##sys#setslot loc 2 #f) ) ) )
  (for-each 
   (initb ##sys#r4rs-environment)
   '(not boolean? eq? eqv? equal? pair? cons car cdr caar cadr cdar cddr caaar caadr cadar caddr cdaar
     cdadr cddar cdddr caaaar caaadr caadar caaddr cadaar cadadr cadddr cdaaar cdaadr cdadar cdaddr
     cddaar cddadr cdddar cddddr set-car! set-cdr! null? list? list length list-tail list-ref
     append reverse memq memv member assq assv assoc symbol? symbol->string string->symbol
     number? integer? exact? real? complex? inexact? rational? zero? odd? even? positive? negative?
     max min + - * / = > < >= <= quotient remainder modulo gcd lcm abs floor ceiling truncate round
     exact->inexact inexact->exact exp log expt sqrt sin cos tan asin acos atan number->string
     string->number char? char=? char>? char<? char>=? char<=? char-ci=? char-ci<? char-ci>?
     char-ci>=? char-ci<=? char-alphabetic? char-whitespace? char-numeric? char-upper-case?
     char-lower-case? char-upcase char-downcase char->integer integer->char string? string=?
     string>? string<? string>=? string<=? string-ci=? string-ci<? string-ci>? string-ci>=? string-ci<=?
     make-string string-length string-ref string-set! string-append string-copy string->list 
     list->string substring string-fill! vector? make-vector vector-ref vector-set! string vector
     vector-length vector->list list->vector vector-fill! procedure? map for-each apply force 
     call-with-current-continuation input-port? output-port? current-input-port current-output-port
     call-with-input-file call-with-output-file open-input-file open-output-file close-input-port
     close-output-port load transcript-on transcript-off read eof-object? read-char peek-char
     write display write-char newline with-input-from-file with-output-to-file
     ##sys#list->vector ##sys#list ##sys#append ##sys#cons ##sys#make-promise) )
  (set! ##sys#r5rs-environment (##sys#copy-environment ##sys#r4rs-environment))
  (for-each
   (initb ##sys#r5rs-environment)
   '(dynamic-wind values call-with-values eval scheme-report-environment null-environment interaction-environment) ) )

(let ([make-vector make-vector])
  (set! interaction-environment (lambda () (##sys#make-structure 'environment #f #t)))
  (set! scheme-report-environment
    (lambda (n)
      (##sys#check-exact n)
      (case n
	[(4) (##sys#make-structure 'environment (##sys#copy-environment ##sys#r4rs-environment) #f)]
	[(5) (##sys#make-structure 'environment (##sys#copy-environment ##sys#r5rs-environment) #f)]
	[else (##sys#error "no support for scheme-report-environment" n)] ) ) )
  (set! null-environment 
    (lambda (n)
      (##sys#check-exact n)
      (when (or (fx< n 4) (fx> n 5))
	(##sys#error "no support for null-environment" n) )
      (##sys#make-structure 'environment (make-vector 499 '()) #t) ) ) )


;;; Expand local macro definitions:

(define (##sys#expand-local-macrodefs defs)
  (let loop ([defs defs])
    (if (null? defs) 
	'()
	(let ([def (##sys#slot defs 0)])
	  (cond [(and (pair? def) (symbol? (car def)))
		 (##sys#check-syntax 'let-macro def '(variable (lambda lambda-list . #(_ 1))))
		 (let ([expander ((##sys#eval-handler) (cadr def))])
		   (cons (cons (car def) (lambda (form) (apply expander (cdr form))))
			 (loop (##sys#slot defs 1))) ) ]
		[else
		 (##sys#check-syntax 'let-macro def '((variable . lambda-list) . #(_ 1)))
		 (let ([expander ((##sys#eval-handler) `(lambda ,(cdar def) ,@(cdr def)))])
		   (cons (cons (caar def) (lambda (form) (apply expander (cdr form))))
			 (loop (##sys#slot defs 1)) ) ) ] ) ) ) ) )


;;; Split lambda-list into its parts:

(define ##sys#decompose-lambda-list
  (let ([reverse reverse])
    (lambda (llist0 k)

      (define (err)
	(set! ##sys#syntax-error-culprit #f)
	(##sys#syntax-error-hook "illegal lambda-list syntax" llist0) )

      (let loop ([llist llist0] [vars '()] [argc 0])
	(cond [(eq? llist '()) (k (reverse vars) argc #f)]
	      [(not (##core#inline "C_blockp" llist)) (err)]
	      [(##core#inline "C_symbolp" llist) (k (reverse (cons llist vars)) argc llist)]
	      [(not (##core#inline "C_pairp" llist)) (err)]
	      [else (loop (##sys#slot llist 1)
			  (cons (##sys#slot llist 0) vars)
			  (fx+ argc 1) ) ] ) ) ) ) )


;;; Loading source files:

(define ##sys#load-verbose #t)

(let ([read read]
      [write write]
      [display display]
      [newline newline]
      [call-with-input-file call-with-input-file]
      [with-input-from-file with-input-from-file]
      [file-exists? file-exists?]
      [string-append string-append] )
  (set! ##sys#load 
    (lambda (filename evaluator pf)
      (let ([fname (if (file-exists? filename)
		       filename
		       (let ((fname2 (string-append filename ".scm")))
			 (if (file-exists? fname2)
			     fname2
			     filename) ) ) ]
	    [evproc (or evaluator (##sys#eval-handler))] 
	    [orlc ##sys#read-line-counter] 
	    [rewln ##sys#read-error-with-line-number] )
	(when ##sys#load-verbose
	  (display "; loading ")
	  (display fname)
	  (display " ...\n") )
	(set! ##sys#read-line-counter 1)
	(set! ##sys#read-error-with-line-number #t)
	(call-with-input-file fname
	  (lambda (in)
	    (do ((x (read in) (read in)))
		((eof-object? x) 
		 (set! ##sys#read-line-counter orlc)
		 (set! ##sys#read-error-with-line-number rewln) )
	      (##sys#call-with-values
	       (lambda () (evproc x)) 
	       (lambda results
		 (when pf
		   (for-each
		    (lambda (r) 
		      (write r)
		      (newline) )
		    results) ) ) ) ) ) ) ) ) )
  (set! load
    (lambda (filename . evaluator)
      (fluid-let ([##sys#current-namespace ##sys#current-namespace])
	(##sys#load filename (:optional evaluator #f) #f) ) ) )
  (set! load-noisily
    (lambda (filename . evaluator)
      (fluid-let ([##sys#current-namespace ##sys#current-namespace])
	(##sys#load filename (:optional evaluator #f) #t) ) ) )
  (set! load-srfi-7-program
    (lambda (filename . evaluator)
      (fluid-let ([##sys#current-namespace ##sys#current-namespace])
	((:optional evaluator (##sys#eval-handler))
	 (##sys#expand-srfi-7-program
	  ""
	  (with-input-from-file filename read)
	  ##sys#syntax-error-hook) ) ) ) ) )


;;; Find included file:

(define ##sys#include-pathnames '())

(define ##sys#resolve-include-filename
  (let ((file-exists? file-exists?)
	(string string)
	(string-append string-append) )
    (lambda (fname)

      (define (test fname)
	(if (file-exists? fname) 
	    fname
	    (let ((fname2 (string-append fname ".scm")))
	      (and (file-exists? fname2) fname2) ) ) )
		
      (or (test fname)
	  (let loop ((paths ##sys#include-pathnames))
	    (cond ((eq? paths '()) fname)
		  ((test (string-append (##sys#slot paths 0)
					(string pathname-directory-separator)
					fname) ) )
		  (else (loop (##sys#slot paths 1))) ) ) ) ) ) )


;;; Print timing information (support for "time" macro):

(define ##sys#display-times
  (let* ((display display)
	 (spaces 
	  (lambda (n)
	    (do ((i n (fx- i 1)))
		((fx<= i 0))
	      (display #\space) ) ) )
	 (display-rj 
	  (lambda (x w)
	    (let* ((xs (if (zero? x) "0" (number->string x)))
		   (xslen (##core#inline "C_block_size" xs)) )
	      (spaces (fx- w xslen))
	      (display xs) ) ) ) )
    (lambda (info)
      (display-rj (##sys#slot info 0) 8)
      (display " seconds elapsed\n") 
      (display-rj (##sys#slot info 1) 8)
      (display " seconds in GC\n")
      (display-rj (##sys#slot info 2) 8)
      (display " mutations\n")
      (display-rj (##sys#slot info 3) 8)
      (display " minor GCs\n")
      (display-rj (##sys#slot info 4) 8)
      (display " major GCs\n")
      (display-rj (##sys#slot info 5) 8)
      (display " bytes of heap-space currently in use\n") ) ) )


;;; General syntax checking routine:

(define ##sys#line-number-database #f)
(define ##sys#syntax-error-hook ##sys#error)
(define ##sys#syntax-error-culprit #f)

(define (get-line-number sexp)
  (and ##sys#line-number-database
       (##core#inline "C_blockp" sexp)
       (##core#inline "C_pairp" sexp)
       (let ([head (##sys#slot sexp 0)])
	 (and (symbol? head)
	      (cond [(##sys#hash-table-ref ##sys#line-number-database head)
		     => (lambda (pl)
			  (let ([a (assq sexp pl)])
			    (and a (##sys#slot a 1)) ) ) ]
		    [else #f] ) ) ) ) )

(define ##sys#check-syntax
  (let ([string-append string-append]
	[keyword? keyword?]
	[get-line-number get-line-number]
	[symbol->string symbol->string] )
    (lambda (id exp pat . culprit)

      (define (test x pred msg)
	(unless (pred x) (err msg)) )

      (define (err msg)
	(let* ([sexp ##sys#syntax-error-culprit]
	       [ln (get-line-number sexp)] )
	  (##sys#syntax-error-hook
	   (if ln 
	       (string-append "(" (symbol->string id) ") in line " (number->string ln) " - " msg)
	       (string-append "(" (symbol->string id) ") " msg) )
	   exp) ) )

      (define (lambda-list? x)
	(let loop ((x x))
	  (cond ((eq? x '()))
		((not (##core#inline "C_blockp" x)) #f)
		((##core#inline "C_symbolp" x) (not (keyword? x)))
		((##core#inline "C_pairp" x)
		 (let ((s (##sys#slot x 0)))
		   (if (or (not (##core#inline "C_blockp" s)) (not (##core#inline "C_symbolp" s)))
		       #f
		       (loop (##sys#slot x 1)) ) ) ) 
		(else #f) ) ) )

      (define (proper-list? x)
	(let loop ((x x))
	  (cond ((eq? x '()))
		((and (##core#inline "C_blockp" x) (##core#inline "C_pairp" x)) (loop (##sys#slot x 1)))
		(else #f) ) ) )

      (when (pair? culprit) (set! ##sys#syntax-error-culprit (car culprit)))
      (let walk ((x exp) (p pat))
	(cond ((and (##core#inline "C_blockp" p) (##core#inline "C_vectorp" p))
	       (let* ((p2 (##sys#slot p 0))
		      (vlen (##core#inline "C_block_size" p))
		      (min (if (fx> vlen 1) 
			       (##sys#slot p 1)
			       0) )
		      (max (cond ((eq? vlen 1) 1)
				 ((fx> vlen 2) (##sys#slot p 2))
				 (else 99999) ) ) )
		 (do ((x x (##sys#slot x 1))
		      (n 0 (fx+ n 1)) )
		     ((eq? x '())
		      (if (fx< n min)
			  (err "not enough arguments") ) )
		   (cond ((fx>= n max) 
			  (err "too many arguments") )
			 ((or (not (##core#inline "C_blockp" x)) (not (##core#inline "C_pairp" x)))
			  (err "not a proper list") )
			 (else (walk (##sys#slot x 0) p2) ) ) ) ) )
	      ((not (##core#inline "C_blockp" p))
	       (if (not (eq? p x)) (err "unexpected object")) )
	      ((##core#inline "C_symbolp" p)
	       (case p
		 ((_) #t)
		 ((pair) (test x pair? "pair expected"))
		 ((variable) (test x (lambda (x) (and (symbol? x) (not (keyword? x)))) "identifer expected"))
		 ((symbol) (test x symbol? "symbol expected"))
		 ((list) (test x proper-list? "proper list expected"))
		 ((number) (test x number? "number expected"))
		 ((string) (test x string? "string expected"))
		 ((lambda-list) (test x lambda-list? "lambda-list expected"))
		 (else (test x (lambda (y) (eq? y p)) "missing keyword")) ) )
	      ((or (not (##core#inline "C_blockp" x)) (not (##core#inline "C_pairp" x)))
	       (err "incomplete form") )
	      (else
	       (walk (##sys#slot x 0) (##sys#slot p 0))
	       (walk (##sys#slot x 1) (##sys#slot p 1)) ) ) ) ) ) )


;;; Macro definitions:

(##sys#register-macro-2
 'define
 (lambda (form)
   (let ((head (car form))
	 (body (cdr form)) )
     (cond ((not (and (##core#inline "C_blockp" head) (##core#inline "C_pairp" head)))
	    (##sys#check-syntax 'define head 'symbol)
	    (##sys#check-syntax 'define body '#(_ 1))
	    `(set! ,head ,(car body)) )
	   (else
	    (##sys#check-syntax 'define head '(symbol . lambda-list))
	    (##sys#check-syntax 'define body '#(_ 1))
	    `(set! ,(car head) (lambda ,(cdr head) ,@body)) ) ) ) ) )

(##sys#register-macro-2
 'and
 (lambda (body)
   (if (eq? body '())
       #t
       (let ((rbody (##sys#slot body 1))
	     (hbody (##sys#slot body 0)) )
	 (if (eq? rbody '())
	     hbody
	     `(if ,hbody (and ,@rbody) #f) ) ) ) ) )

(##sys#register-macro-2
 'or 
 (let ((gensym gensym))
   (lambda (body)
     (if (eq? body '())
	 #f
	 (let ((rbody (##sys#slot body 1))
	       (hbody (##sys#slot body 0)) )
	   (if (eq? rbody '())
	       hbody
	       (let ((tmp (gensym)))
		 `(let ((,tmp ,hbody))
		    (if ,tmp ,tmp (or ,@rbody)) ) ) ) ) ) ) ) )

(##sys#register-macro-2
 'cond
 (let ((gensym gensym))
   (lambda (body)
     (let expand ((clauses body))
       (if (not (pair? clauses))
	   '(##core#undefined)
	   (let ((clause (##sys#slot clauses 0))
		 (rclauses (##sys#slot clauses 1)) )
	     (##sys#check-syntax 'cond clause '#(_ 1))
	     (cond ((eq? 'else (car clause)) `(begin ,@(cdr clause)))
		   ((eq? (cdr clause) '()) `(or ,(car clause) ,(expand rclauses)))
		   ((eq? '=> (car (cdr clause)))
		    (let ((tmp (gensym)))
		      `(let ((,tmp ,(car clause)))
			 (if ,tmp
			     (,(car (cdr (cdr clause))) ,tmp)
			     ,(expand rclauses) ) ) ) )
		   (else `(if ,(car clause) 
			      (begin ,@(cdr clause))
			      ,(expand rclauses) ) ) ) ) ) ) ) ) )

(##sys#register-macro-2
 'case
 (let ((gensym gensym))
   (lambda (form)
     (let ((exp (car form))
	   (body (cdr form)) )
       (let ((tmp (gensym)))
	 `(let ((,tmp ,exp))
	    ,(let expand ((clauses body))
	       (if (not (pair? clauses))
		   '(##core#undefined)
		   (let ((clause (##sys#slot clauses 0))
			 (rclauses (##sys#slot clauses 1)) )
		     (##sys#check-syntax 'case clause '#(_ 1))
		     (if (eq? 'else (car clause))
			 `(begin ,@(cdr clause))
			 `(if (or ,@(##sys#map (lambda (x) `(eqv? ,tmp ',x)) (car clause)))
			      (begin ,@(cdr clause)) 
			      ,(expand rclauses) ) ) ) ) ) ) ) ) ) ) )

(##sys#register-macro-2
 'let*
 (lambda (form)
   (let ((bindings (car form))
	 (body (cdr form)) )
     (##sys#check-syntax 'let* bindings '#((symbol _) 0))
     (##sys#check-syntax 'let* body '#(_ 1))
     (let expand ((bs bindings))
       (if (eq? bs '())
	   (##sys#canonicalize-body body)
	   `(let (,(car bs)) ,(expand (cdr bs))) ) ) ) ) )

(##sys#register-macro-2
 'letrec
 (lambda (form)
   (let ((bindings (car form))
	 (body (cdr form)) )
     (##sys#check-syntax 'letrec bindings '#((symbol _) 0))
     (##sys#check-syntax 'letrec body '#(_ 1))
     `(let ,(##sys#map (lambda (b) (list (car b) '(##core#undefined))) bindings)
	(begin ,@(##sys#append (##sys#map (lambda (b) `(set! ,(car b) ,(cadr b))) bindings)
			   (list (##sys#canonicalize-body body)) ) ) ) ) ) )

(##sys#register-macro
 'do
 (let ((gensym gensym))
   (lambda (bindings test . body)
     (##sys#check-syntax 'do bindings '#((symbol _ . #(_)) 0))
     (##sys#check-syntax 'do test '#(_ 1))
     (let ((dovar (gensym "do")))
       `(let ,dovar ,(##sys#map (lambda (b) (list (car b) (car (cdr b)))) bindings)
	     (if ,(car test)
		 ,(let ((tbody (cdr test)))
		    (if (eq? tbody '())
			'(##core#undefined)
			`(begin ,@tbody) ) )
		 (begin
		   ,(if (eq? body '())
			'(##core#undefined)
			(##sys#canonicalize-body body) )
		   (,dovar ,@(##sys#map (lambda (b) 
				     (if (eq? (cdr (cdr b)) '())
					 (car b)
					 (car (cdr (cdr b))) ) )
				   bindings) ) ) ) ) ) ) ) )

(##sys#register-macro
 'quasiquote
 (let ((vector->list vector->list))
   (lambda (form)
     
     (define (walk x n) (simplify (walk1 x n)))

     (define (walk1 x n)
       (if (##core#inline "C_blockp" x)
	   (cond ((##core#inline "C_vectorp" x)
		  `(##sys#list->vector ,(walk (vector->list x) n)) )
		 ((not (##core#inline "C_pairp" x)) `(quote ,x))
		 (else
		  (let ((head (##sys#slot x 0))
			(tail (##sys#slot x 1)) )
		    (case head
		      ((unquote)
		       (if (and (##core#inline "C_blockp" tail) (##core#inline "C_pairp" tail))
			   (let ((hx (##sys#slot tail 0)))
			     (if (eq? n 0)
				 hx
				 (list '##sys#list '(quote unquote)
				       (walk hx (fx- n 1)) ) ) )
			   '(quote unquote) ) )
		      ((quasiquote)
		       (if (and (##core#inline "C_blockp" tail) (##core#inline "C_pairp" tail))
			   `(##sys#list (quote quasiquote) 
				   ,(walk (##sys#slot tail 0) (fx+ n 1)) ) 
			   (list '##sys#cons (list 'quote 'quasiquote) (walk tail n)) ) )
		      (else
		       (if (and (##core#inline "C_blockp" head) (##core#inline "C_pairp" head))
			   (let ((hx (##sys#slot head 0))
				 (tx (##sys#slot head 1)) )
			     (if (and (eq? hx 'unquote-splicing)
				      (##core#inline "C_blockp" tx)
				      (##core#inline "C_pairp" tx) )
				 (let ((htx (##sys#slot tx 0)))
				   (if (eq? n 0)
				       `(##sys#append ,htx
						 ,(walk tail n) )
				       `(##sys#cons (##sys#list 'unquote-splicing
							,(walk htx (fx- n 1)) )
					       ,(walk tail n) ) ) )
				 `(##sys#cons ,(walk head n) ,(walk tail n)) ) )
			   `(##sys#cons ,(walk head n) ,(walk tail n)) ) ) ) ) ) )
	   `(quote ,x) ) )

     (define (simplify x)
       (cond ((##sys#match-expression x '(##sys#cons a '()) '(a))
	      => (lambda (env) (simplify `(##sys#list ,(##sys#slot (assq 'a env) 1)))) )
	     ((##sys#match-expression x '(##sys#cons a (##sys#list . b)) '(a b))
	      => (lambda (env)
		   (let ([bxs (assq 'b env)])
		     (if (fx< (length bxs) 32)
			 (simplify `(##sys#list ,(##sys#slot (assq 'a env) 1)
					    ,@(##sys#slot bxs 1) ) ) 
			 x) ) ) )
	     ((##sys#match-expression x '(##sys#append a '()) '(a))
	      => (lambda (env) (##sys#slot (assq 'a env) 1)) )
	     (else x) ) )
     
     (walk form 0) ) ) )

(##sys#register-macro
 'delay
 (lambda (x) `(##sys#make-promise (lambda () ,x))) )


;;; Other non-standard macros:

(##sys#register-macro 
 'define-record
 (let ((symbol->string symbol->string)
       (string->symbol string->symbol)
       (string-append string-append) )
   (lambda (name . slots)
     (##sys#check-syntax 'define-record name 'symbol)
     (##sys#check-syntax 'define-record slots '#(symbol 0))
     (let ([prefix (symbol->string name)]
	   [nsprefix (##sys#qualified-symbol-prefix name)] )
       `(begin
	  (set! ,(##sys#string->qualified-symbol nsprefix (string-append "make-" prefix))
		(lambda ,slots (##sys#make-structure ',name ,@slots)) )
	  (set! ,(##sys#string->qualified-symbol nsprefix (string-append prefix "?"))
		(lambda (x) (##sys#structure? x ',name)) )
	  ,@(let mapslots ((slots slots) (i 1))
	      (if (eq? slots '())
		  slots
		  (let ((slotname (symbol->string (##sys#slot slots 0))))
		    (cons
		     `(begin
			(set! ,(##sys#string->qualified-symbol nsprefix (string-append prefix "-" slotname))
			      (lambda (x) 
				(##sys#check-structure x ',name)
				(##sys#slot x ,i) ) )
			(set! ,(##sys#string->qualified-symbol nsprefix (string-append prefix "-" slotname "-set!"))
			      (lambda (x val)
				(##sys#check-structure x ',name)
				(##sys#setslot x ,i val) ) ) ) 
		     (mapslots (##sys#slot slots 1) (fx+ i 1)) ) ) ) ) ) ) ) ) )

(##sys#register-macro
 'define-macro
 (lambda (head . body)
   (cond [(symbol? head) 
	  (let ([b (car body)])
	    (##sys#check-syntax 'define-macro b '(lambda lambda-list . #(_ 1)))
	    (set! head (cons head (cadr b)))
	    (set! body (cddr b)) ) ]
	 [else
	  (##sys#check-syntax 'define-macro head '(symbol . lambda-list))
	  (##sys#check-syntax 'define-macro body '#(_ 1)) ] )
   `(##core#elaborationtimetoo
     ,(let ((name (##sys#slot head 0))
	    (llist (##sys#slot head 1)) )
	(if (and (##core#inline "C_blockp" llist) (##core#inline "C_symbolp" llist))
	    `(##sys#register-macro-2 (##core#qualified ',name) (lambda (,llist) ,@body))
	    `(##sys#register-macro (##core#qualified ',name) (lambda ,llist ,@body) ) ) ) ) ) )

(##sys#register-macro
 'define-id-macro
 (lambda (head form)
   (##sys#check-syntax 'define-id-macro head 'symbol)
   `(##core#elaborationtimetoo
     (##sys#hash-table-set!
      ##sys#macro-environment
      (##core#qualified ',head) 
      (lambda (form) (##core#qualified ',form))) ) ) )

(##sys#register-macro
 'receive
 (lambda (vars exp . exps)
   (##sys#check-syntax 'receive vars 'lambda-list)
   `(##sys#call-with-values (lambda () ,exp)
      (lambda ,vars ,@exps) ) ) )

(##sys#register-macro
 'time 
 (let ((gensym gensym))
   (lambda (exp)
     (let ((rvar (gensym 't)))
       `(begin
	  (##sys#start-timer)
	  (##sys#call-with-values 
	   (lambda () ,exp)
	   (lambda ,rvar
	     (##sys#display-times (##sys#stop-timer))
	     (##sys#apply ##sys#values ,rvar) ) ) ) ) ) ) )

(##sys#register-macro
 'declare
 (lambda specs 
   `(##core#declare ,@(##sys#map (lambda (x) `(quote ,x)) specs)) ) ); hides specifiers from macroexpand

(##sys#register-macro
 'include
 (lambda (filename) 
   `(##core#include ',filename) ) )

(##sys#register-macro
 'assert
 (lambda (exp . msg-and-args)
   (let ((msg (if (eq? '() msg-and-args)
		  "assertion failed"
		  (##sys#slot msg-and-args 0) ) ) )
     `(if (not ,exp)
	  (##sys#error ,msg ',exp ,@(if (fx> (length msg-and-args) 1)
				  (##sys#slot msg-and-args 1)
				  '() ) ) ) ) ) )

(##sys#register-macro
 'fluid-let
 (let ((gensym gensym))
   (lambda (clauses . body)
     (##sys#check-syntax 'fluid-let clauses '#((symbol _) 0))
     (let ((ids (##sys#map car clauses))
	   (new-tmps (##sys#map (lambda (x) (gensym)) clauses))
	   (old-tmps (##sys#map (lambda (x) (gensym)) clauses)))
       `(let (,@(map ##sys#list new-tmps (##sys#map cadr clauses))
	      ,@(map ##sys#list old-tmps
		     (let loop ((n (length clauses)))
		       (if (eq? n 0)
			   '()
			   (cons #f (loop (fx- n 1))) ) ) ) )
	  (##sys#dynamic-wind
	      (lambda ()
		,@(map (lambda (ot id) `(set! ,ot ,id))
		       old-tmps ids)
		,@(map (lambda (id nt) `(set! ,id ,nt))
		       ids new-tmps))
	      (lambda () ,@body)
	      (lambda ()
		,@(map (lambda (nt id) `(set! ,nt ,id))
		       new-tmps ids)
		,@(map (lambda (id ot) `(set! ,id ,ot))
		       ids old-tmps) ) ) ) ) ) ) )

(##sys#register-macro
 'eval-when
 (lambda (situations . body)
   (let ([e #f]
	 [c #f]
	 [l #f] 
	 [body `(begin ,@body)] )
     (let loop ([ss situations])
       (if (pair? ss)
	   (begin
	     (case (##sys#slot ss 0)
	       [(eval) (set! e #t)]
	       [(load) (set! l #t)]
	       [(compile) (set! c #t)]
	       [else (##sys#error "invalid situation specifier" (##sys#slot ss 0))] )
	     (loop (##sys#slot ss 1)) ) ) )
     (if (memq '#:compiling ##sys#features)
	 (cond [(and c l) `(##core#compiletimetoo ,body)]
	       [c `(##core#compiletimeonly ,body)]
	       [l body]
	       [else '(##core#undefined)] )
	 (if e 
	     body
	     '(##core#undefined) ) ) ) ) )

(##sys#register-macro
 'parameterize 
 (let ([car car]
       [cadr cadr] 
       [map map] )
   (lambda (bindings . body)
     (##sys#check-syntax 'parameterize bindings '#((symbol _) 0))
     (let* ([swap (gensym)]
	    [params (##sys#map car bindings)]
	    [vals (##sys#map cadr bindings)]
	    [aliases (##sys#map gensym params)]
	    [aliases2 (##sys#map (lambda (z) (gensym)) params)] )
       `(let ,(##sys#append (map ##sys#list aliases params) (map ##sys#list aliases2 vals))
	  (let ((,swap (lambda ()
			 ,@(map (lambda (a a2) `(let ((t (,a))) (,a ,a2) (set! ,a2 t)))
				aliases aliases2) ) ) )
	    (##sys#dynamic-wind 
		,swap
		(lambda () ,@body)
		,swap) ) ) ) ) ) )

(##sys#register-macro
 'when
 (lambda (test . body)
   `(if ,test (begin ,@body)) ) )

(##sys#register-macro
 'unless
 (lambda (test . body)
   `(if ,test (##core#undefined) (begin ,@body)) ) )

(let* ([map map]
       [assign
	(lambda (vars exp)
	  (##sys#check-syntax 'set!-values/define-values vars '#(symbol 0))
	  (let ([aliases (map gensym vars)])
	    `(##sys#call-with-values (lambda () ,exp)
	       (lambda ,aliases
		 ,@(map (lambda (v a) `(set! ,v ,a)) vars aliases) ) ) ) ) ] )
  (##sys#register-macro 'set!-values assign)
  (##sys#register-macro 'define-values assign) )

(##sys#register-macro-2
 'let-values
 (lambda (form)
   (##sys#check-syntax 'let-values form '(#(_ 0) . #(_ 1)))
   (let* ([vbindings (car form)]
	  [body (cdr form)]
	  [llists (map (lambda (x) (car x)) vbindings)]
	  [vars (apply ##sys#append llists)]
	  [aliases (map (lambda (v) (cons v (gensym v))) vars)] 
	  [lookup (lambda (v) (cdr (assq v aliases)))]
	  [llists2 (map (lambda (llist) (map lookup llist)) llists)] )
     (let fold ([llists llists]
		[exps (map (lambda (x) (cadr x)) vbindings)]
		[llists2 llists2] )
       (if (null? llists)
	   `(let ,(map (lambda (v) (##sys#list v (lookup v))) vars) ,@body)
	   `(##sys#call-with-values (lambda () ,(car exps))
	      (lambda ,(car llists2) ,(fold (cdr llists) (cdr exps) (cdr llists2))) ) ) ) ) ) )

(##sys#register-macro-2
 'let*-values
 (lambda (form)
   (##sys#check-syntax 'let*-values form '(#(_ 0) . #(_ 1)))
   (let ([vbindings (car form)]
	 [body (cdr form)] )
     (let fold ([vbindings vbindings])
       (if (null? vbindings)
	   `(let () ,@body)
	   `(##sys#call-with-values (lambda () ,(cadar vbindings))
	      (lambda ,(caar vbindings) ,(fold (cdr vbindings))) ) ) ) ) ) )

(##sys#register-macro-2 
 'letrec-values
 (lambda (form)
   (##sys#check-syntax 'letrec-values form '(#(_ 0) . #(_ 1)))
   (let* ([vbindings (car form)]
	  [body (cdr form)] 
	  [vars (apply ##sys#append (map (lambda (x) (car x)) vbindings))] 
	  [aliases (map (lambda (v) (cons v (gensym v))) vars)] 
	  [lookup (lambda (v) (cdr (assq v aliases)))] )
     `(let ,(map (lambda (v) (##sys#list v '(##core#undefined))) vars)
	,@(map (lambda (vb)
		 `(##sys#call-with-values (lambda () ,(cadr vb))
		    (lambda ,(map lookup (car vb))
		      ,@(map (lambda (v) `(set! ,v ,(lookup v))) (car vb)) ) ) )
	       vbindings)
	,@body) ) ) )

(##sys#register-macro
 'nth-value
 (lambda (i exp)
   (let ([v (gensym)])
     `(##sys#call-with-values
       (lambda () ,exp)
       (lambda ,v (list-ref ,v ,i)) ) ) ) )

(##sys#register-macro-2
 'record-case
 (lambda (form)
   (##sys#check-syntax 'record-case form '(_ . #(list 0)))
   (let* ([val (car form)]
	  [clauses (cdr form)] 
	  [valtmp (gensym)]
	  [headtmp (gensym)]
	  [sizetmp (gensym)] )
     `(let* ((,valtmp ,val)
	     (,headtmp (and (##sys#generic-structure? ,valtmp) (##sys#slot ,valtmp 0)))
	     (,sizetmp (and ,headtmp (##sys#size ,valtmp))) )
	(cond 
	 ,@(map (lambda (clause)
		  (let ([head (car clause)])
		    (if (eq? 'else head)
			clause
			`((and (eq? ,headtmp ',(car head))
			       (fx>= ,sizetmp ,(length head)) )
			  (let ,(let loop ([i 1] [slots (cdr head)])
				  (if (null? slots)
				      '()
				      (cons (list (car slots) `(##sys#slot ,valtmp ,i)) (loop (fx+ i 1) (cdr slots))) ) )
			    ,@(cdr clause) ) ) ) ) )
		clauses) ) ) ) ) )

(letrec ([quotify 
	  (lambda (xs id)
	    (##sys#check-syntax id xs '#(_ 1))
	    (map (lambda (x) (list 'quote x)) xs) ) ]
	 [quotify2
	  (lambda (xs id)
	    (##sys#check-syntax id xs '#(_ 2 4))
	    `((quote ,(car xs))
	      (quote ,(cadr xs))
	      ,@(cond [(pair? (cddr xs))
		       (cddr xs) ]
		      [else '()] ) ) ) ]
         [quotify-proc 
           (lambda (xs id)
	     (##sys#check-syntax id xs '#(_ 1))
             (let* ([head (car xs)]
                    [name (if (pair? head) (car head) head)]
                    [val (if (pair? head)
                           `(lambda ,(cdr head) ,@(cdr xs))
                           (cadr xs) ) ] )
               (list (list 'quote name) val) ) ) ] )
  (##sys#register-macro-2 'define-inline (lambda (form) `(##core#define-inline ,@(quotify-proc form 'define-inline ))))
  (##sys#register-macro-2 'define-integrable (lambda (form) `(##core#define-inline ,@(quotify-proc form 'define-integrable))))
  (##sys#register-macro-2 'define-foreign-type (lambda (form) `(##core#define-foreign-type ,@(quotify2 form 'define-foreign-type))))
  (##sys#register-macro-2 
   'define-foreign-variable
   (lambda (form) `(##core#define-foreign-variable ,@(quotify form 'define-foreign-variable))) )
  (##sys#register-macro-2 
   'define-foreign-parameter 
   (lambda (form) `(##core#define-foreign-parameter ,@(quotify form 'define-foreign-parameter))) )
  (##sys#register-macro-2 'foreign-lambda (lambda (form) `(##core#foreign-lambda ,@(quotify form 'foreign-lambda))))
  (##sys#register-macro-2 'foreign-callback-lambda
			  (lambda (form) `(##core#foreign-callback-lambda ,@(quotify form 'foreign-callback-lambda))))
  (##sys#register-macro-2 'foreign-callback-lambda*
			  (lambda (form) `(##core#foreign-callback-lambda* ,@(quotify form 'foreign-callback-lambda*))))
  (##sys#register-macro-2 'foreign-lambda* (lambda (form) `(##core#foreign-lambda* ,@(quotify form 'foreign-lambda*))))
  (##sys#register-macro-2 'define-constant (lambda (form) `(##core#define-constant ,@(quotify-proc form 'define-constant)))) )

(##sys#register-macro-2
 'foreign-callback-wrapper
 (lambda (form)
   (##sys#check-syntax 'foreign-callback-wrapper form '(_ string string _ (lambda lambda-list . #(_ 1))))
   `(##core#foreign-callback-wrapper
     ',(cadr form)
     ',(caddr form)
     ',(car form)
     ',(cadddr form)
     ,(cadddr (cdr form)) ) ) )

(##sys#register-macro-2
 'define-external
 (lambda (form)
   (let* ([quals (and (pair? form) (string? (car form)))]
	  [var (and (not quals) (pair? form) (symbol? (car form)))] )
     (cond [var
	    (##sys#check-syntax 'define-external form '(symbol _ . #(_ 0 1)))
	    (let ([var (car form)])
	      `(begin
		 (##core#define-foreign-variable ',var ',(cadr form))
		 (##core#define-external-variable ',var ',(cadr form))
		 ,@(if (pair? (cddr form))
		       `((set! ,var ,(caddr form)))
		       '() ) ) ) ]
	   [else
	    (if quals
		(##sys#check-syntax 'define-external form '(string (symbol . #((_ symbol) 0)) _ . #(_ 1)))
		(##sys#check-syntax 'define-external form '((symbol . #((_ symbol) 0)) _ . #(_ 1))) )
	    (let* ([head (if quals (cadr form) (car form))]
		   [args (cdr head)] )
	      `(define ,(car head)
		 (##core#foreign-callback-wrapper
		  ',(car head)
		  ',(if quals (car form) "")
		  ',(if quals (caddr form) (cadr form))
		  ',(map (lambda (a) (car a)) args)
		  (lambda ,(map (lambda (a) (cadr a)) args) ,@(if quals (cdddr form) (cddr form)) ) ) ) ) ] ) ) ) )

(##sys#register-macro-2
 'critical-section
 (lambda (form)
   `(##sys#dynamic-wind
	##sys#disable-interrupts
	(lambda () ,@form)
	##sys#enable-interrupts) ) )


;;; SRFI-2:

(##sys#register-macro-2
 'and-let*
   (lambda (forms)
     (##sys#check-syntax 'and-let* forms '(#(_ 0) . #(_ 1)))
     (if (or (not (list? forms)) (fx< (length forms) 2))
	 (##sys#syntax-error-hook "syntax error in 'and-let*' form" forms) 
	 (let ([bindings (##sys#slot forms 0)]
	       [body (##sys#slot forms 1)] )
	   (let fold ([bs bindings])
	     (if (null? bs)
		 `(begin ,@body)
		 (let ([b (##sys#slot bs 0)]
		       [bs2 (##sys#slot bs 1)] )
		   (cond [(not-pair? b) `(if ,b ,(fold bs2) #f)]
			 [(null? (##sys#slot b 1)) `(if ,(##sys#slot b 0) ,(fold bs2) #f)]
			 [else
			  (let ([var (##sys#slot b 0)])
			    `(let ((,var ,(cadr b)))
			       (if ,var ,(fold bs2) #f) ) ) ] ) ) ) ) ) ) ) )


;;; cond-expand (srfi-0):

(set! ##sys#features (append '(#:srfi-8 #:srfi-6 #:srfi-2 #:srfi-0 #:srfi-10 #:srfi-7) ##sys#features))

(define (##sys#test-feature f)
  (let ([f (##sys#->feature-id f)])
    (memq f ##sys#features) ) )

(##sys#register-macro-2
 'cond-expand
   (lambda (clauses)

     (define (err x) 
       (##sys#error "syntax error in `cond-expand' form" x (cons 'cond-expand clauses)) )

     (define (test fx)
       (cond ((symbol? fx) (##sys#test-feature fx))
	     ((not (pair? fx)) (err fx))
	     (else
	      (let ((rest (##sys#slot fx 1)))
		(case (##sys#slot fx 0)
		  ((and)
		   (or (eq? rest '())
		       (if (pair? rest)
			   (and (test (##sys#slot rest 0))
				(test `(and ,@(##sys#slot rest 1))) )
			   (err fx) ) ) )
		  ((or) 
		   (and (not (eq? rest '()))
			(if (pair? rest)
			    (or (test (##sys#slot rest 0))
				(test `(or ,@(##sys#slot rest 1))) )
			    (err fx) ) ) )
		  ((not) (not (test (cadr fx))))
		  (else (err fx)) ) ) ) ) )

     (let expand ((cls clauses))
       (cond ((eq? cls '())
	      (##sys#apply
	       ##sys#error "no matching clause in `cond-expand' form" 
	       (map (lambda (x) (car x)) clauses) ) )
	     ((not (pair? cls)) (err cls))
	     (else
	      (let ((clause (##sys#slot cls 0))
		    (rclauses (##sys#slot cls 1)) )
		(if (not (pair? clause)) 
		    (err clause)
		    (let ((id (##sys#slot clause 0)))
		      (cond ((eq? id 'else)
			     (let ((rest (##sys#slot clause 1)))
			       (if (eq? rest '())
				   '(##core#undefined)
				   `(begin ,@rest) ) ) )
			    ((test id) `(begin ,@(##sys#slot clause 1)))
			    (else (expand rclauses)) ) ) ) ) ) ) ) ) )


;;; Entry points:

(##sys#register-macro-2 
 'define-entry-point
   (lambda (form)
     (##sys#check-syntax 'define-entry-point form '(_ list list . #(_ 1)))
     (let ([id (car form)]
	   [args (cadr form)]
	   [results (caddr form)]
	   [body (cdddr form)]
	   [wordsperdouble (lambda (n) (fx* n (##sys#fudge 8)))]
	   [buffer (gensym)] )

       (define (convert-argument type index)
	 (let ([err (lambda () (##sys#error "can not generate entry-point argument conversion for foreign type" type))])
	   (case type
	     [(int unsigned-int) `(##sys#peek-fixnum ,buffer ,(wordsperdouble index))]
	     [(integer short long) `(##sys#peek-signed-integer ,buffer ,(wordsperdouble index))]
	     [(unsigned-integer unsigned-short unsigned-long)
	      `(##sys#peek-unsigned-integer ,buffer ,(wordsperdouble index)) ]
	     [(char) `(integer->char (##sys#peek-signed-integer ,buffer ,(wordsperdouble index)))]
	     [(unsigned-char) `(integer->char (##sys#peek-unsigned-integer ,buffer ,(wordsperdouble index)))]
	     [(bool) `(not (eq? 0 (##sys#peek-fixnum ,buffer ,(wordsperdouble index))))]
	     [(c-pointer pointer) `(##sys#address->pointer (##sys#peek-unsigned-integer ,buffer ,(wordsperdouble index)))]
	     [(float double) `(##sys#peek-double ,buffer ,index)]
	     [(c-string) `(##sys#peek-c-string ,buffer ,(wordsperdouble index))]
	     [else
	      (if (pair? type)
		  (case (car type)
		    [(pointer function) `(##sys#address->pointer (##sys#peek-unsigned-integer ,buffer ,(wordsperdouble index)))]
		    [else (err)] )
		  (err) ) ] ) ) )

       (define (convert-result type index val)
	 (let ([err (lambda () (##sys#error "can not generate entry-point result conversion for foreign type" type))])
	   (case type
	     [(int integer short long) `(##sys#poke-integer ,buffer ,(wordsperdouble index) ,val)]
	     [(unsigned-int unsigned-integer unsigned-short unsigned-long)
	      `(##sys#poke-integer ,buffer ,(wordsperdouble index) ,val) ]
	     [(char unsigned-char) `(##sys#poke-integer ,buffer ,(wordsperdouble index) (char->integer ,val))]
	     [(bool) `(##sys#poke-integer ,buffer ,(wordsperdouble index) (if ,val 1 0))]
	     [(c-pointer pointer) `(##sys#poke-integer ,buffer ,(wordsperdouble index) (##sys#pointer->address ,val))]
	     [(float double) `(##sys#poke-double ,buffer ,index ,val)]
	     [(c-string) `(##sys#poke-c-string ,buffer ,(wordsperdouble index) ,val)]
	     [else
	      (if (pair? type)
		  (case (car type)
		    [(pointer function) `(##sys#poke-integer ,buffer ,(wordsperdouble index) (##sys#pointer->address ,val))]
		    [else (err)] )
		  (err) ) ] ) ) )

       `(##sys#register-entry-point
	 ,id
	 (lambda (,(gensym) ,buffer)
	   (##sys#call-with-values
	    (lambda ()
	      (let ,(let loop ([args args] [i 0])
		      (if (null? args)
			  '()
			  (##sys#cons
			   (##sys#list (caar args) (convert-argument (cadar args) i))
			   (loop (cdr args) (fx+ i 1)) ) ) )
		,@body) )
	    (lambda resultvalues
	      ,(if (null? results)
		   '(##core#undefined)
		   (let loop ([results results] [args args] [i 0])
		     (let ([type (car results)])
		       (when (and (eq? 'c-string type)
				  (or (null? args)
				      (not (eq? 'c-string (cadar args))) ) )
			 (##sys#error "can not return result of type `c-string' without corresponding string argument") )
		       (let ([r (convert-result type i '(car resultvalues))])
			 (if (null? (cdr results))
			     r
			     `(begin
				,r
				(let ([resultvalues (cdr resultvalues)])
				  ,(loop (cdr results) (cdr args) (fx+ i 1)) ) ) ) ) ) ) ) ) ) ) ) ) ) )


;;;; Read-Eval-Print loop:

(define ##sys#repl-eval-hook #f)
(define ##sys#repl-print-length-limit #f)
(define ##sys#repl-read-hook #f)

(define ##sys#read-prompt-hook
  (let ((display display))
    (lambda () (display ">>> ")) ) )

(define read-eval-print-loop
  (let ([eval eval]
	[read read]
	[write write]
	[call-with-current-continuation call-with-current-continuation]
	[display display]
	[reset reset]
	[newline newline] )
    (lambda ()

      (define (writeargs xs)
	(fluid-let ([##sys#print-qualifiers #t])
	  (if (or (null? xs) (pair? (cdr xs)) (not (eq? (##core#undefined) (car xs))))
	      (##sys#for-each 
	       (lambda (x)
		 (##sys#with-print-length-limit ##sys#repl-print-length-limit (lambda () (write x)))
		 (newline) )
	       xs) ) ) )

      (let ([stdin ##sys#standard-input]
	    [stdout ##sys#standard-output]
	    [stderr ##sys#standard-error] 
	    [ehandler (##sys#error-handler)] 
	    [rhandler (##sys#reset-handler)] )

	(define (saveports)
	  (set! stdin ##sys#standard-input)
	  (set! stdout ##sys#standard-output)
	  (set! stderr ##sys#standard-error) )

	(define (resetports)
	  (set! ##sys#standard-input stdin)
	  (set! ##sys#standard-output stdout)
	  (set! ##sys#standard-error stderr) )

	(saveports)
	(##sys#dynamic-wind
	 (lambda ()
	   (##sys#error-handler
	    (lambda (msg . args)
	      (resetports)
	      (display "Error: ")
	      (display msg)
	      (newline)
	      (writeargs args) ) ) )
	 (lambda ()
	   (let loop ()
	     (call-with-current-continuation
	      (lambda (c)
		(##sys#reset-handler
		 (lambda ()
		   (set! ##sys#read-error-with-line-number #f)
		   (set! ##sys#default-namespace-prefix #f)
		   (set! ##sys#enable-qualifiers #t)
		   (resetports)
		   (c #f) ) ) ) )
	     (##sys#read-prompt-hook)
	     (let ([exp ((or ##sys#repl-read-hook read))])
	       (unless (eof-object? exp)
		 (receive result ((or ##sys#repl-eval-hook eval) exp)
		   (writeargs result) 
		   (loop) ) ) ) ) )
	 (lambda ()
	   (##sys#error-handler ehandler)
	   (##sys#reset-handler rhandler) ) ) ) ) ) )


;;; SRFI-10:

(define ##sys#sharp-comma-reader-ctors (make-vector 301 '()))

(define (define-reader-ctor spec proc)
  (##sys#check-symbol spec)
  (##sys#hash-table-set! ##sys#sharp-comma-reader-ctors spec proc) )

(set! ##sys#user-read-hook
  (let ([old ##sys#user-read-hook]
	[read read] )
    (lambda (char port)
      (cond [(char=? char #\,)
	     (read-char port)
	     (let* ([exp (read port)]
		    [err (lambda () (##sys#read-error "invalid sharp-comma external form" exp))] )
	       (if (or (null? exp) (not (list? exp)))
		   (err)
		   (let ([spec (##sys#slot exp 0)])
		     (if (not (symbol? spec))
			 (err) 
			 (let ([ctor (##sys#hash-table-ref ##sys#sharp-comma-reader-ctors spec)])
			   (if ctor
			       (apply ctor (##sys#slot exp 1))
			       (##sys#read-error "undefined sharp-comma constructor" spec) ) ) ) ) ) ) ]
	    [else (old char port)] ) ) ) )


;;; SRFI-7:

(define ##sys#expand-srfi-7-program
  (let ([string-append string-append]
	[string string] )
    (lambda (path prg failed)

      (define (check fr)
	(if (pair? fr)
	    (case (##sys#slot fr 0)
	      [(not)
	       (##sys#check-syntax 'program-feature-cond fr '(not _))
	       (not (check (##sys#slot (##sys#slot 1) 0))) ]
	      [(and)
	       (##sys#check-syntax 'program-feature-cond fr '(and . #(_ 0)))
	       (let fold ([frs (##sys#slot fr 1)])
		 (or (null? frs)
		     (and (check (##sys#slot frs 0))
			  (fold (##sys#slot frs 1)) ) ) ) ]
	      [(or)
	       (##sys#check-syntax 'program-feature-cond fr '(or . #(_ 0)))
	       (let fold ([frs (##sys#slot fr 1)])
		 (and (pair? frs)
		      (or (check (##sys#slot frs 0))
			  (fold (##sys#slot frs 1)) ) ) ) ]
	      [else (##sys#syntax-error-hook "invalid feature requirement" fr)] )
	    (##sys#test-feature fr) ) )

      (define (check-file f)
	(if (file-exists? f)
	    f
	    (string-append path (string pathname-directory-separator) f) ) )

      (##sys#check-syntax 'program prg '(program . #(_ 0)))
      (let loop ([clauses (##sys#slot prg 1)])
	(if (null? clauses) 
	    '(begin)
	    (let ([clause (##sys#slot clauses 0)]
		  [rest (##sys#slot clauses 1)] )
	      (##sys#check-syntax 'program clause '(symbol . #(_ 0)))
	      (let ([r (##sys#slot clause 1)])
		(case (##sys#slot clause 0)
		  [(requires)
		   (##sys#check-syntax 'program-requires r '#(symbol 1))
		   (for-each
		    (lambda (f)
		      (unless (##sys#test-feature f)
			(failed "missing required feature" f) ) )
		    r)
		   (loop rest) ]
		  [(files)
		   (##sys#check-syntax 'program-files r '#(string 0))
		   `(begin 
		      ,@(map (lambda (f) `(include ,(check-file f))) r)
		      ,(loop rest) ) ]
		  [(code)
		   (##sys#check-syntax 'program-code r '#(_ 0))
		   `(begin ,@r ,(loop rest)) ]
		  [(feature-cond)
		   (##sys#check-syntax 'program-feature-cond r '#(_ 0))
		   (let fold ([fcs r])
		     (if (null? fcs)
			 (failed "no matching `feature-cond' clause")
			 (let ([fc (##sys#slot fcs 0)]
			       [r (##sys#slot fcs 1)] )
			   (##sys#check-syntax 'program-feature-cond fc '#(_ 2))
			   (let ([fr (##sys#slot fc 0)]
				 [frr (##sys#slot fc 1)] )
			     (if (or (eq? 'else fr) (check fr))
				 (loop frr)
				 (fold r) ) ) ) ) ) ]
		  [else (##sys#syntax-error-hook "invalid program clause" clause)] ) ) ) ) ) ) ) )


;;; To catch missing `-hygienic' option:

(##sys#register-macro-2
 'define-syntax
 (lambda (form)
   (##sys#error "highlevel macros not available - try `-hygienic' option") ) )
