;;;; prolog.scm - Chicken user pass for compiling a simple Prolog subset into Scheme - felix


(declare
  (unit prolog)
  (uses srfi-1 match extras)
  (usual-integrations)
  (interrupts-disabled)
  (fixnum)
  (block) )


;----------------------------------------------------------------------------------------------------


; *** This file starts with a copy of the file multilex.scm ***
; Copyright (C) 1997 Danny Dube', Universite' de Montre'al.
; All rights reserved.
; SILex 1.0.

;
; Gestion des Input Systems
; Fonctions a utiliser par l'usager:
;   lexer-make-IS, lexer-get-func-getc, lexer-get-func-ungetc,
;   lexer-get-func-line, lexer-get-func-column et lexer-get-func-offset
;

; Taille initiale par defaut du buffer d'entree
(define lexer-init-buffer-len 1024)

; Numero du caractere newline
(define lexer-integer-newline (char->integer #\newline))

; Constructeur d'IS brut
(define lexer-raw-IS-maker
  (lambda (buffer read-ptr input-f counters)
    (let ((input-f          input-f)                ; Entree reelle
	  (buffer           buffer)                 ; Buffer
	  (buflen           (string-length buffer))
	  (read-ptr         read-ptr)
	  (start-ptr        1)                      ; Marque de debut de lexeme
	  (start-line       1)
	  (start-column     1)
	  (start-offset     0)
	  (end-ptr          1)                      ; Marque de fin de lexeme
	  (point-ptr        1)                      ; Le point
	  (user-ptr         1)                      ; Marque de l'usager
	  (user-line        1)
	  (user-column      1)
	  (user-offset      0)
	  (user-up-to-date? #t))                    ; Concerne la colonne seul.
      (letrec
	  ((start-go-to-end-none         ; Fonctions de depl. des marques
	    (lambda ()
	      (set! start-ptr end-ptr)))
	   (start-go-to-end-line
	    (lambda ()
	      (let loop ((ptr start-ptr) (line start-line))
		(if (= ptr end-ptr)
		    (begin
		      (set! start-ptr ptr)
		      (set! start-line line))
		    (if (char=? (string-ref buffer ptr) #\newline)
			(loop (+ ptr 1) (+ line 1))
			(loop (+ ptr 1) line))))))
	   (start-go-to-end-all
	    (lambda ()
	      (set! start-offset (+ start-offset (- end-ptr start-ptr)))
	      (let loop ((ptr start-ptr)
			 (line start-line)
			 (column start-column))
		(if (= ptr end-ptr)
		    (begin
		      (set! start-ptr ptr)
		      (set! start-line line)
		      (set! start-column column))
		    (if (char=? (string-ref buffer ptr) #\newline)
			(loop (+ ptr 1) (+ line 1) 1)
			(loop (+ ptr 1) line (+ column 1)))))))
	   (start-go-to-user-none
	    (lambda ()
	      (set! start-ptr user-ptr)))
	   (start-go-to-user-line
	    (lambda ()
	      (set! start-ptr user-ptr)
	      (set! start-line user-line)))
	   (start-go-to-user-all
	    (lambda ()
	      (set! start-line user-line)
	      (set! start-offset user-offset)
	      (if user-up-to-date?
		  (begin
		    (set! start-ptr user-ptr)
		    (set! start-column user-column))
		  (let loop ((ptr start-ptr) (column start-column))
		    (if (= ptr user-ptr)
			(begin
			  (set! start-ptr ptr)
			  (set! start-column column))
			(if (char=? (string-ref buffer ptr) #\newline)
			    (loop (+ ptr 1) 1)
			    (loop (+ ptr 1) (+ column 1))))))))
	   (end-go-to-point
	    (lambda ()
	      (set! end-ptr point-ptr)))
	   (point-go-to-start
	    (lambda ()
	      (set! point-ptr start-ptr)))
	   (user-go-to-start-none
	    (lambda ()
	      (set! user-ptr start-ptr)))
	   (user-go-to-start-line
	    (lambda ()
	      (set! user-ptr start-ptr)
	      (set! user-line start-line)))
	   (user-go-to-start-all
	    (lambda ()
	      (set! user-ptr start-ptr)
	      (set! user-line start-line)
	      (set! user-column start-column)
	      (set! user-offset start-offset)
	      (set! user-up-to-date? #t)))
	   (init-lexeme-none             ; Debute un nouveau lexeme
	    (lambda ()
	      (if (< start-ptr user-ptr)
		  (start-go-to-user-none))
	      (point-go-to-start)))
	   (init-lexeme-line
	    (lambda ()
	      (if (< start-ptr user-ptr)
		  (start-go-to-user-line))
	      (point-go-to-start)))
	   (init-lexeme-all
	    (lambda ()
	      (if (< start-ptr user-ptr)
		  (start-go-to-user-all))
	      (point-go-to-start)))
	   (get-start-line               ; Obtention des stats du debut du lxm
	    (lambda ()
	      start-line))
	   (get-start-column
	    (lambda ()
	      start-column))
	   (get-start-offset
	    (lambda ()
	      start-offset))
	   (peek-left-context            ; Obtention de caracteres (#f si EOF)
	    (lambda ()
	      (char->integer (string-ref buffer (- start-ptr 1)))))
	   (peek-char
	    (lambda ()
	      (if (< point-ptr read-ptr)
		  (char->integer (string-ref buffer point-ptr))
		  (let ((c (input-f)))
		    (if (char? c)
			(begin
			  (if (= read-ptr buflen)
			      (reorganize-buffer))
			  (string-set! buffer point-ptr c)
			  (set! read-ptr (+ point-ptr 1))
			  (char->integer c))
			(begin
			  (set! input-f (lambda () 'eof))
			  #f))))))
	   (read-char
	    (lambda ()
	      (if (< point-ptr read-ptr)
		  (let ((c (string-ref buffer point-ptr)))
		    (set! point-ptr (+ point-ptr 1))
		    (char->integer c))
		  (let ((c (input-f)))
		    (if (char? c)
			(begin
			  (if (= read-ptr buflen)
			      (reorganize-buffer))
			  (string-set! buffer point-ptr c)
			  (set! read-ptr (+ point-ptr 1))
			  (set! point-ptr read-ptr)
			  (char->integer c))
			(begin
			  (set! input-f (lambda () 'eof))
			  #f))))))
	   (get-start-end-text           ; Obtention du lexeme
	    (lambda ()
	      (substring buffer start-ptr end-ptr)))
	   (get-user-line-line           ; Fonctions pour l'usager
	    (lambda ()
	      (if (< user-ptr start-ptr)
		  (user-go-to-start-line))
	      user-line))
	   (get-user-line-all
	    (lambda ()
	      (if (< user-ptr start-ptr)
		  (user-go-to-start-all))
	      user-line))
	   (get-user-column-all
	    (lambda ()
	      (cond ((< user-ptr start-ptr)
		     (user-go-to-start-all)
		     user-column)
		    (user-up-to-date?
		     user-column)
		    (else
		     (let loop ((ptr start-ptr) (column start-column))
		       (if (= ptr user-ptr)
			   (begin
			     (set! user-column column)
			     (set! user-up-to-date? #t)
			     column)
			   (if (char=? (string-ref buffer ptr) #\newline)
			       (loop (+ ptr 1) 1)
			       (loop (+ ptr 1) (+ column 1)))))))))
	   (get-user-offset-all
	    (lambda ()
	      (if (< user-ptr start-ptr)
		  (user-go-to-start-all))
	      user-offset))
	   (user-getc-none
	    (lambda ()
	      (if (< user-ptr start-ptr)
		  (user-go-to-start-none))
	      (if (< user-ptr read-ptr)
		  (let ((c (string-ref buffer user-ptr)))
		    (set! user-ptr (+ user-ptr 1))
		    c)
		  (let ((c (input-f)))
		    (if (char? c)
			(begin
			  (if (= read-ptr buflen)
			      (reorganize-buffer))
			  (string-set! buffer user-ptr c)
			  (set! read-ptr (+ read-ptr 1))
			  (set! user-ptr read-ptr)
			  c)
			(begin
			  (set! input-f (lambda () 'eof))
			  'eof))))))
	   (user-getc-line
	    (lambda ()
	      (if (< user-ptr start-ptr)
		  (user-go-to-start-line))
	      (if (< user-ptr read-ptr)
		  (let ((c (string-ref buffer user-ptr)))
		    (set! user-ptr (+ user-ptr 1))
		    (if (char=? c #\newline)
			(set! user-line (+ user-line 1)))
		    c)
		  (let ((c (input-f)))
		    (if (char? c)
			(begin
			  (if (= read-ptr buflen)
			      (reorganize-buffer))
			  (string-set! buffer user-ptr c)
			  (set! read-ptr (+ read-ptr 1))
			  (set! user-ptr read-ptr)
			  (if (char=? c #\newline)
			      (set! user-line (+ user-line 1)))
			  c)
			(begin
			  (set! input-f (lambda () 'eof))
			  'eof))))))
	   (user-getc-all
	    (lambda ()
	      (if (< user-ptr start-ptr)
		  (user-go-to-start-all))
	      (if (< user-ptr read-ptr)
		  (let ((c (string-ref buffer user-ptr)))
		    (set! user-ptr (+ user-ptr 1))
		    (if (char=? c #\newline)
			(begin
			  (set! user-line (+ user-line 1))
			  (set! user-column 1))
			(set! user-column (+ user-column 1)))
		    (set! user-offset (+ user-offset 1))
		    c)
		  (let ((c (input-f)))
		    (if (char? c)
			(begin
			  (if (= read-ptr buflen)
			      (reorganize-buffer))
			  (string-set! buffer user-ptr c)
			  (set! read-ptr (+ read-ptr 1))
			  (set! user-ptr read-ptr)
			  (if (char=? c #\newline)
			      (begin
				(set! user-line (+ user-line 1))
				(set! user-column 1))
			      (set! user-column (+ user-column 1)))
			  (set! user-offset (+ user-offset 1))
			  c)
			(begin
			  (set! input-f (lambda () 'eof))
			  'eof))))))
	   (user-ungetc-none
	    (lambda ()
	      (if (> user-ptr start-ptr)
		  (set! user-ptr (- user-ptr 1)))))
	   (user-ungetc-line
	    (lambda ()
	      (if (> user-ptr start-ptr)
		  (begin
		    (set! user-ptr (- user-ptr 1))
		    (let ((c (string-ref buffer user-ptr)))
		      (if (char=? c #\newline)
			  (set! user-line (- user-line 1))))))))
	   (user-ungetc-all
	    (lambda ()
	      (if (> user-ptr start-ptr)
		  (begin
		    (set! user-ptr (- user-ptr 1))
		    (let ((c (string-ref buffer user-ptr)))
		      (if (char=? c #\newline)
			  (begin
			    (set! user-line (- user-line 1))
			    (set! user-up-to-date? #f))
			  (set! user-column (- user-column 1)))
		      (set! user-offset (- user-offset 1)))))))
	   (reorganize-buffer            ; Decaler ou agrandir le buffer
	    (lambda ()
	      (if (< (* 2 start-ptr) buflen)
		  (let* ((newlen (* 2 buflen))
			 (newbuf (make-string newlen))
			 (delta (- start-ptr 1)))
		    (let loop ((from (- start-ptr 1)))
		      (if (< from buflen)
			  (begin
			    (string-set! newbuf
					 (- from delta)
					 (string-ref buffer from))
			    (loop (+ from 1)))))
		    (set! buffer    newbuf)
		    (set! buflen    newlen)
		    (set! read-ptr  (- read-ptr delta))
		    (set! start-ptr (- start-ptr delta))
		    (set! end-ptr   (- end-ptr delta))
		    (set! point-ptr (- point-ptr delta))
		    (set! user-ptr  (- user-ptr delta)))
		  (let ((delta (- start-ptr 1)))
		    (let loop ((from (- start-ptr 1)))
		      (if (< from buflen)
			  (begin
			    (string-set! buffer
					 (- from delta)
					 (string-ref buffer from))
			    (loop (+ from 1)))))
		    (set! read-ptr  (- read-ptr delta))
		    (set! start-ptr (- start-ptr delta))
		    (set! end-ptr   (- end-ptr delta))
		    (set! point-ptr (- point-ptr delta))
		    (set! user-ptr  (- user-ptr delta)))))))
	(list (cons 'start-go-to-end
		    (cond ((eq? counters 'none) start-go-to-end-none)
			  ((eq? counters 'line) start-go-to-end-line)
			  ((eq? counters 'all ) start-go-to-end-all)))
	      (cons 'end-go-to-point
		    end-go-to-point)
	      (cons 'init-lexeme
		    (cond ((eq? counters 'none) init-lexeme-none)
			  ((eq? counters 'line) init-lexeme-line)
			  ((eq? counters 'all ) init-lexeme-all)))
	      (cons 'get-start-line
		    get-start-line)
	      (cons 'get-start-column
		    get-start-column)
	      (cons 'get-start-offset
		    get-start-offset)
	      (cons 'peek-left-context
		    peek-left-context)
	      (cons 'peek-char
		    peek-char)
	      (cons 'read-char
		    read-char)
	      (cons 'get-start-end-text
		    get-start-end-text)
	      (cons 'get-user-line
		    (cond ((eq? counters 'none) #f)
			  ((eq? counters 'line) get-user-line-line)
			  ((eq? counters 'all ) get-user-line-all)))
	      (cons 'get-user-column
		    (cond ((eq? counters 'none) #f)
			  ((eq? counters 'line) #f)
			  ((eq? counters 'all ) get-user-column-all)))
	      (cons 'get-user-offset
		    (cond ((eq? counters 'none) #f)
			  ((eq? counters 'line) #f)
			  ((eq? counters 'all ) get-user-offset-all)))
	      (cons 'user-getc
		    (cond ((eq? counters 'none) user-getc-none)
			  ((eq? counters 'line) user-getc-line)
			  ((eq? counters 'all ) user-getc-all)))
	      (cons 'user-ungetc
		    (cond ((eq? counters 'none) user-ungetc-none)
			  ((eq? counters 'line) user-ungetc-line)
			  ((eq? counters 'all ) user-ungetc-all))))))))

; Construit un Input System
; Le premier parametre doit etre parmi "port", "procedure" ou "string"
; Prend un parametre facultatif qui doit etre parmi
; "none", "line" ou "all"
(define lexer-make-IS
  (lambda (input-type input . largs)
    (let ((counters-type (cond ((null? largs)
				'line)
			       ((memq (car largs) '(none line all))
				(car largs))
			       (else
				'line))))
      (cond ((and (eq? input-type 'port) (input-port? input))
	     (let* ((buffer   (make-string lexer-init-buffer-len #\newline))
		    (read-ptr 1)
		    (input-f  (lambda () (read-char input))))
	       (lexer-raw-IS-maker buffer read-ptr input-f counters-type)))
	    ((and (eq? input-type 'procedure) (procedure? input))
	     (let* ((buffer   (make-string lexer-init-buffer-len #\newline))
		    (read-ptr 1)
		    (input-f  input))
	       (lexer-raw-IS-maker buffer read-ptr input-f counters-type)))
	    ((and (eq? input-type 'string) (string? input))
	     (let* ((buffer   (string-append (string #\newline) input))
		    (read-ptr (string-length buffer))
		    (input-f  (lambda () 'eof)))
	       (lexer-raw-IS-maker buffer read-ptr input-f counters-type)))
	    (else
	     (let* ((buffer   (string #\newline))
		    (read-ptr 1)
		    (input-f  (lambda () 'eof)))
	       (lexer-raw-IS-maker buffer read-ptr input-f counters-type)))))))

; Les fonctions:
;   lexer-get-func-getc, lexer-get-func-ungetc,
;   lexer-get-func-line, lexer-get-func-column et lexer-get-func-offset
(define lexer-get-func-getc
  (lambda (IS) (cdr (assq 'user-getc IS))))
(define lexer-get-func-ungetc
  (lambda (IS) (cdr (assq 'user-ungetc IS))))
(define lexer-get-func-line
  (lambda (IS) (cdr (assq 'get-user-line IS))))
(define lexer-get-func-column
  (lambda (IS) (cdr (assq 'get-user-column IS))))
(define lexer-get-func-offset
  (lambda (IS) (cdr (assq 'get-user-offset IS))))

;
; Gestion des lexers
;

; Fabrication de lexer a partir d'arbres de decision
(define lexer-make-tree-lexer
  (lambda (tables IS)
    (letrec
	(; Contenu de la table
	 (counters-type        (vector-ref tables 0))
	 (<<EOF>>-pre-action   (vector-ref tables 1))
	 (<<ERROR>>-pre-action (vector-ref tables 2))
	 (rules-pre-actions    (vector-ref tables 3))
	 (table-nl-start       (vector-ref tables 5))
	 (table-no-nl-start    (vector-ref tables 6))
	 (trees-v              (vector-ref tables 7))
	 (acc-v                (vector-ref tables 8))

	 ; Contenu du IS
	 (IS-start-go-to-end    (cdr (assq 'start-go-to-end IS)))
	 (IS-end-go-to-point    (cdr (assq 'end-go-to-point IS)))
	 (IS-init-lexeme        (cdr (assq 'init-lexeme IS)))
	 (IS-get-start-line     (cdr (assq 'get-start-line IS)))
	 (IS-get-start-column   (cdr (assq 'get-start-column IS)))
	 (IS-get-start-offset   (cdr (assq 'get-start-offset IS)))
	 (IS-peek-left-context  (cdr (assq 'peek-left-context IS)))
	 (IS-peek-char          (cdr (assq 'peek-char IS)))
	 (IS-read-char          (cdr (assq 'read-char IS)))
	 (IS-get-start-end-text (cdr (assq 'get-start-end-text IS)))
	 (IS-get-user-line      (cdr (assq 'get-user-line IS)))
	 (IS-get-user-column    (cdr (assq 'get-user-column IS)))
	 (IS-get-user-offset    (cdr (assq 'get-user-offset IS)))
	 (IS-user-getc          (cdr (assq 'user-getc IS)))
	 (IS-user-ungetc        (cdr (assq 'user-ungetc IS)))

	 ; Resultats
	 (<<EOF>>-action   #f)
	 (<<ERROR>>-action #f)
	 (rules-actions    #f)
	 (states           #f)
	 (final-lexer      #f)

	 ; Gestion des hooks
	 (hook-list '())
	 (add-hook
	  (lambda (thunk)
	    (set! hook-list (cons thunk hook-list))))
	 (apply-hooks
	  (lambda ()
	    (let loop ((l hook-list))
	      (if (pair? l)
		  (begin
		    ((car l))
		    (loop (cdr l)))))))

	 ; Preparation des actions
	 (set-action-statics
	  (lambda (pre-action)
	    (pre-action final-lexer IS-user-getc IS-user-ungetc)))
	 (prepare-special-action-none
	  (lambda (pre-action)
	    (let ((action #f))
	      (let ((result
		     (lambda ()
		       (action "")))
		    (hook
		     (lambda ()
		       (set! action (set-action-statics pre-action)))))
		(add-hook hook)
		result))))
	 (prepare-special-action-line
	  (lambda (pre-action)
	    (let ((action #f))
	      (let ((result
		     (lambda (yyline)
		       (action "" yyline)))
		    (hook
		     (lambda ()
		       (set! action (set-action-statics pre-action)))))
		(add-hook hook)
		result))))
	 (prepare-special-action-all
	  (lambda (pre-action)
	    (let ((action #f))
	      (let ((result
		     (lambda (yyline yycolumn yyoffset)
		       (action "" yyline yycolumn yyoffset)))
		    (hook
		     (lambda ()
		       (set! action (set-action-statics pre-action)))))
		(add-hook hook)
		result))))
	 (prepare-special-action
	  (lambda (pre-action)
	    (cond ((eq? counters-type 'none)
		   (prepare-special-action-none pre-action))
		  ((eq? counters-type 'line)
		   (prepare-special-action-line pre-action))
		  ((eq? counters-type 'all)
		   (prepare-special-action-all  pre-action)))))
	 (prepare-action-yytext-none
	  (lambda (pre-action)
	    (let ((get-start-end-text IS-get-start-end-text)
		  (start-go-to-end    IS-start-go-to-end)
		  (action             #f))
	      (let ((result
		     (lambda ()
		       (let ((yytext (get-start-end-text)))
			 (start-go-to-end)
			 (action yytext))))
		    (hook
		     (lambda ()
		       (set! action (set-action-statics pre-action)))))
		(add-hook hook)
		result))))
	 (prepare-action-yytext-line
	  (lambda (pre-action)
	    (let ((get-start-end-text IS-get-start-end-text)
		  (start-go-to-end    IS-start-go-to-end)
		  (action             #f))
	      (let ((result
		     (lambda (yyline)
		       (let ((yytext (get-start-end-text)))
			 (start-go-to-end)
			 (action yytext yyline))))
		    (hook
		     (lambda ()
		       (set! action (set-action-statics pre-action)))))
		(add-hook hook)
		result))))
	 (prepare-action-yytext-all
	  (lambda (pre-action)
	    (let ((get-start-end-text IS-get-start-end-text)
		  (start-go-to-end    IS-start-go-to-end)
		  (action             #f))
	      (let ((result
		     (lambda (yyline yycolumn yyoffset)
		       (let ((yytext (get-start-end-text)))
			 (start-go-to-end)
			 (action yytext yyline yycolumn yyoffset))))
		    (hook
		     (lambda ()
		       (set! action (set-action-statics pre-action)))))
		(add-hook hook)
		result))))
	 (prepare-action-yytext
	  (lambda (pre-action)
	    (cond ((eq? counters-type 'none)
		   (prepare-action-yytext-none pre-action))
		  ((eq? counters-type 'line)
		   (prepare-action-yytext-line pre-action))
		  ((eq? counters-type 'all)
		   (prepare-action-yytext-all  pre-action)))))
	 (prepare-action-no-yytext-none
	  (lambda (pre-action)
	    (let ((start-go-to-end    IS-start-go-to-end)
		  (action             #f))
	      (let ((result
		     (lambda ()
		       (start-go-to-end)
		       (action)))
		    (hook
		     (lambda ()
		       (set! action (set-action-statics pre-action)))))
		(add-hook hook)
		result))))
	 (prepare-action-no-yytext-line
	  (lambda (pre-action)
	    (let ((start-go-to-end    IS-start-go-to-end)
		  (action             #f))
	      (let ((result
		     (lambda (yyline)
		       (start-go-to-end)
		       (action yyline)))
		    (hook
		     (lambda ()
		       (set! action (set-action-statics pre-action)))))
		(add-hook hook)
		result))))
	 (prepare-action-no-yytext-all
	  (lambda (pre-action)
	    (let ((start-go-to-end    IS-start-go-to-end)
		  (action             #f))
	      (let ((result
		     (lambda (yyline yycolumn yyoffset)
		       (start-go-to-end)
		       (action yyline yycolumn yyoffset)))
		    (hook
		     (lambda ()
		       (set! action (set-action-statics pre-action)))))
		(add-hook hook)
		result))))
	 (prepare-action-no-yytext
	  (lambda (pre-action)
	    (cond ((eq? counters-type 'none)
		   (prepare-action-no-yytext-none pre-action))
		  ((eq? counters-type 'line)
		   (prepare-action-no-yytext-line pre-action))
		  ((eq? counters-type 'all)
		   (prepare-action-no-yytext-all  pre-action)))))

	 ; Fabrique les fonctions de dispatch
	 (prepare-dispatch-err
	  (lambda (leaf)
	    (lambda (c)
	      #f)))
	 (prepare-dispatch-number
	  (lambda (leaf)
	    (let ((state-function #f))
	      (let ((result
		     (lambda (c)
		       state-function))
		    (hook
		     (lambda ()
		       (set! state-function (vector-ref states leaf)))))
		(add-hook hook)
		result))))
	 (prepare-dispatch-leaf
	  (lambda (leaf)
	    (if (eq? leaf 'err)
		(prepare-dispatch-err leaf)
		(prepare-dispatch-number leaf))))
	 (prepare-dispatch-<
	  (lambda (tree)
	    (let ((left-tree  (list-ref tree 1))
		  (right-tree (list-ref tree 2)))
	      (let ((bound      (list-ref tree 0))
		    (left-func  (prepare-dispatch-tree left-tree))
		    (right-func (prepare-dispatch-tree right-tree)))
		(lambda (c)
		  (if (< c bound)
		      (left-func c)
		      (right-func c)))))))
	 (prepare-dispatch-=
	  (lambda (tree)
	    (let ((left-tree  (list-ref tree 2))
		  (right-tree (list-ref tree 3)))
	      (let ((bound      (list-ref tree 1))
		    (left-func  (prepare-dispatch-tree left-tree))
		    (right-func (prepare-dispatch-tree right-tree)))
		(lambda (c)
		  (if (= c bound)
		      (left-func c)
		      (right-func c)))))))
	 (prepare-dispatch-tree
	  (lambda (tree)
	    (cond ((not (pair? tree))
		   (prepare-dispatch-leaf tree))
		  ((eq? (car tree) '=)
		   (prepare-dispatch-= tree))
		  (else
		   (prepare-dispatch-< tree)))))
	 (prepare-dispatch
	  (lambda (tree)
	    (let ((dicho-func (prepare-dispatch-tree tree)))
	      (lambda (c)
		(and c (dicho-func c))))))

	 ; Fabrique les fonctions de transition (read & go) et (abort)
	 (prepare-read-n-go
	  (lambda (tree)
	    (let ((dispatch-func (prepare-dispatch tree))
		  (read-char     IS-read-char))
	      (lambda ()
		(dispatch-func (read-char))))))
	 (prepare-abort
	  (lambda (tree)
	    (lambda ()
	      #f)))
	 (prepare-transition
	  (lambda (tree)
	    (if (eq? tree 'err)
		(prepare-abort     tree)
		(prepare-read-n-go tree))))

	 ; Fabrique les fonctions d'etats ([set-end] & trans)
	 (prepare-state-no-acc
	   (lambda (s r1 r2)
	     (let ((trans-func (prepare-transition (vector-ref trees-v s))))
	       (lambda (action)
		 (let ((next-state (trans-func)))
		   (if next-state
		       (next-state action)
		       action))))))
	 (prepare-state-yes-no
	  (lambda (s r1 r2)
	    (let ((peek-char       IS-peek-char)
		  (end-go-to-point IS-end-go-to-point)
		  (new-action1     #f)
		  (trans-func (prepare-transition (vector-ref trees-v s))))
	      (let ((result
		     (lambda (action)
		       (let* ((c (peek-char))
			      (new-action
			       (if (or (not c) (= c lexer-integer-newline))
				   (begin
				     (end-go-to-point)
				     new-action1)
				   action))
			      (next-state (trans-func)))
			 (if next-state
			     (next-state new-action)
			     new-action))))
		    (hook
		     (lambda ()
		       (set! new-action1 (vector-ref rules-actions r1)))))
		(add-hook hook)
		result))))
	 (prepare-state-diff-acc
	  (lambda (s r1 r2)
	    (let ((end-go-to-point IS-end-go-to-point)
		  (peek-char       IS-peek-char)
		  (new-action1     #f)
		  (new-action2     #f)
		  (trans-func (prepare-transition (vector-ref trees-v s))))
	      (let ((result
		     (lambda (action)
		       (end-go-to-point)
		       (let* ((c (peek-char))
			      (new-action
			       (if (or (not c) (= c lexer-integer-newline))
				   new-action1
				   new-action2))
			      (next-state (trans-func)))
			 (if next-state
			     (next-state new-action)
			     new-action))))
		    (hook
		     (lambda ()
		       (set! new-action1 (vector-ref rules-actions r1))
		       (set! new-action2 (vector-ref rules-actions r2)))))
		(add-hook hook)
		result))))
	 (prepare-state-same-acc
	  (lambda (s r1 r2)
	    (let ((end-go-to-point IS-end-go-to-point)
		  (trans-func (prepare-transition (vector-ref trees-v s)))
		  (new-action #f))
	      (let ((result
		     (lambda (action)
		       (end-go-to-point)
		       (let ((next-state (trans-func)))
			 (if next-state
			     (next-state new-action)
			     new-action))))
		    (hook
		     (lambda ()
		       (set! new-action (vector-ref rules-actions r1)))))
		(add-hook hook)
		result))))
	 (prepare-state
	  (lambda (s)
	    (let* ((acc (vector-ref acc-v s))
		   (r1 (car acc))
		   (r2 (cdr acc)))
	      (cond ((not r1)  (prepare-state-no-acc   s r1 r2))
		    ((not r2)  (prepare-state-yes-no   s r1 r2))
		    ((< r1 r2) (prepare-state-diff-acc s r1 r2))
		    (else      (prepare-state-same-acc s r1 r2))))))

	 ; Fabrique la fonction de lancement du lexage a l'etat de depart
	 (prepare-start-same
	  (lambda (s1 s2)
	    (let ((peek-char    IS-peek-char)
		  (eof-action   #f)
		  (start-state  #f)
		  (error-action #f))
	      (let ((result
		     (lambda ()
		       (if (not (peek-char))
			   eof-action
			   (start-state error-action))))
		    (hook
		     (lambda ()
		       (set! eof-action   <<EOF>>-action)
		       (set! start-state  (vector-ref states s1))
		       (set! error-action <<ERROR>>-action))))
		(add-hook hook)
		result))))
	 (prepare-start-diff
	  (lambda (s1 s2)
	    (let ((peek-char         IS-peek-char)
		  (eof-action        #f)
		  (peek-left-context IS-peek-left-context)
		  (start-state1      #f)
		  (start-state2      #f)
		  (error-action      #f))
	      (let ((result
		     (lambda ()
		       (cond ((not (peek-char))
			      eof-action)
			     ((= (peek-left-context) lexer-integer-newline)
			      (start-state1 error-action))
			     (else
			      (start-state2 error-action)))))
		    (hook
		     (lambda ()
		       (set! eof-action <<EOF>>-action)
		       (set! start-state1 (vector-ref states s1))
		       (set! start-state2 (vector-ref states s2))
		       (set! error-action <<ERROR>>-action))))
		(add-hook hook)
		result))))
	 (prepare-start
	  (lambda ()
	    (let ((s1 table-nl-start)
		  (s2 table-no-nl-start))
	      (if (= s1 s2)
		  (prepare-start-same s1 s2)
		  (prepare-start-diff s1 s2)))))

	 ; Fabrique la fonction principale
	 (prepare-lexer-none
	  (lambda ()
	    (let ((init-lexeme IS-init-lexeme)
		  (start-func  (prepare-start)))
	      (lambda ()
		(init-lexeme)
		((start-func))))))
	 (prepare-lexer-line
	  (lambda ()
	    (let ((init-lexeme    IS-init-lexeme)
		  (get-start-line IS-get-start-line)
		  (start-func     (prepare-start)))
	      (lambda ()
		(init-lexeme)
		(let ((yyline (get-start-line)))
		  ((start-func) yyline))))))
	 (prepare-lexer-all
	  (lambda ()
	    (let ((init-lexeme      IS-init-lexeme)
		  (get-start-line   IS-get-start-line)
		  (get-start-column IS-get-start-column)
		  (get-start-offset IS-get-start-offset)
		  (start-func       (prepare-start)))
	      (lambda ()
		(init-lexeme)
		(let ((yyline   (get-start-line))
		      (yycolumn (get-start-column))
		      (yyoffset (get-start-offset)))
		  ((start-func) yyline yycolumn yyoffset))))))
	 (prepare-lexer
	  (lambda ()
	    (cond ((eq? counters-type 'none) (prepare-lexer-none))
		  ((eq? counters-type 'line) (prepare-lexer-line))
		  ((eq? counters-type 'all)  (prepare-lexer-all))))))

      ; Calculer la valeur de <<EOF>>-action et de <<ERROR>>-action
      (set! <<EOF>>-action   (prepare-special-action <<EOF>>-pre-action))
      (set! <<ERROR>>-action (prepare-special-action <<ERROR>>-pre-action))

      ; Calculer la valeur de rules-actions
      (let* ((len (quotient (vector-length rules-pre-actions) 2))
	     (v (make-vector len)))
	(let loop ((r (- len 1)))
	  (if (< r 0)
	      (set! rules-actions v)
	      (let* ((yytext? (vector-ref rules-pre-actions (* 2 r)))
		     (pre-action (vector-ref rules-pre-actions (+ (* 2 r) 1)))
		     (action (if yytext?
				 (prepare-action-yytext    pre-action)
				 (prepare-action-no-yytext pre-action))))
		(vector-set! v r action)
		(loop (- r 1))))))

      ; Calculer la valeur de states
      (let* ((len (vector-length trees-v))
	     (v (make-vector len)))
	(let loop ((s (- len 1)))
	  (if (< s 0)
	      (set! states v)
	      (begin
		(vector-set! v s (prepare-state s))
		(loop (- s 1))))))

      ; Calculer la valeur de final-lexer
      (set! final-lexer (prepare-lexer))

      ; Executer les hooks
      (apply-hooks)

      ; Resultat
      final-lexer)))

; Fabrication de lexer a partir de listes de caracteres taggees
(define lexer-make-char-lexer
  (let* ((char->class
	  (lambda (c)
	    (let ((n (char->integer c)))
	      (list (cons n n)))))
	 (merge-sort
	  (lambda (l combine zero-elt)
	    (if (null? l)
		zero-elt
		(let loop1 ((l l))
		  (if (null? (cdr l))
		      (car l)
		      (loop1
		       (let loop2 ((l l))
			 (cond ((null? l)
				l)
			       ((null? (cdr l))
				l)
			       (else
				(cons (combine (car l) (cadr l))
				      (loop2 (cddr l))))))))))))
	 (finite-class-union
	  (lambda (c1 c2)
	    (let loop ((c1 c1) (c2 c2) (u '()))
	      (if (null? c1)
		  (if (null? c2)
		      (reverse u)
		      (loop c1 (cdr c2) (cons (car c2) u)))
		  (if (null? c2)
		      (loop (cdr c1) c2 (cons (car c1) u))
		      (let* ((r1 (car c1))
			     (r2 (car c2))
			     (r1start (car r1))
			     (r1end (cdr r1))
			     (r2start (car r2))
			     (r2end (cdr r2)))
			(if (<= r1start r2start)
			    (cond ((< (+ r1end 1) r2start)
				   (loop (cdr c1) c2 (cons r1 u)))
				  ((<= r1end r2end)
				   (loop (cdr c1)
					 (cons (cons r1start r2end) (cdr c2))
					 u))
				  (else
				   (loop c1 (cdr c2) u)))
			    (cond ((> r1start (+ r2end 1))
				   (loop c1 (cdr c2) (cons r2 u)))
				  ((>= r1end r2end)
				   (loop (cons (cons r2start r1end) (cdr c1))
					 (cdr c2)
					 u))
				  (else
				   (loop (cdr c1) c2 u))))))))))
	 (char-list->class
	  (lambda (cl)
	    (let ((classes (map char->class cl)))
	      (merge-sort classes finite-class-union '()))))
	 (class-<
	  (lambda (b1 b2)
	    (cond ((eq? b1 'inf+) #f)
		  ((eq? b2 'inf-) #f)
		  ((eq? b1 'inf-) #t)
		  ((eq? b2 'inf+) #t)
		  (else (< b1 b2)))))
	 (finite-class-compl
	  (lambda (c)
	    (let loop ((c c) (start 'inf-))
	      (if (null? c)
		  (list (cons start 'inf+))
		  (let* ((r (car c))
			 (rstart (car r))
			 (rend (cdr r)))
		    (if (class-< start rstart)
			(cons (cons start (- rstart 1))
			      (loop c rstart))
			(loop (cdr c) (+ rend 1))))))))
	 (tagged-chars->class
	  (lambda (tcl)
	    (let* ((inverse? (car tcl))
		   (cl (cdr tcl))
		   (class-tmp (char-list->class cl)))
	      (if inverse? (finite-class-compl class-tmp) class-tmp))))
	 (charc->arc
	  (lambda (charc)
	    (let* ((tcl (car charc))
		   (dest (cdr charc))
		   (class (tagged-chars->class tcl)))
	      (cons class dest))))
	 (arc->sharcs
	  (lambda (arc)
	    (let* ((range-l (car arc))
		   (dest (cdr arc))
		   (op (lambda (range) (cons range dest))))
	      (map op range-l))))
	 (class-<=
	  (lambda (b1 b2)
	    (cond ((eq? b1 'inf-) #t)
		  ((eq? b2 'inf+) #t)
		  ((eq? b1 'inf+) #f)
		  ((eq? b2 'inf-) #f)
		  (else (<= b1 b2)))))
	 (sharc-<=
	  (lambda (sharc1 sharc2)
	    (class-<= (caar sharc1) (caar sharc2))))
	 (merge-sharcs
	  (lambda (l1 l2)
	    (let loop ((l1 l1) (l2 l2))
	      (cond ((null? l1)
		     l2)
		    ((null? l2)
		     l1)
		    (else
		     (let ((sharc1 (car l1))
			   (sharc2 (car l2)))
		       (if (sharc-<= sharc1 sharc2)
			   (cons sharc1 (loop (cdr l1) l2))
			   (cons sharc2 (loop l1 (cdr l2))))))))))
	 (class-= eqv?)
	 (fill-error
	  (lambda (sharcs)
	    (let loop ((sharcs sharcs) (start 'inf-))
	      (cond ((class-= start 'inf+)
		     '())
		    ((null? sharcs)
		     (cons (cons (cons start 'inf+) 'err)
			   (loop sharcs 'inf+)))
		    (else
		     (let* ((sharc (car sharcs))
			    (h (caar sharc))
			    (t (cdar sharc)))
		       (if (class-< start h)
			   (cons (cons (cons start (- h 1)) 'err)
				 (loop sharcs h))
			   (cons sharc (loop (cdr sharcs)
					     (if (class-= t 'inf+)
						 'inf+
						 (+ t 1)))))))))))
	 (charcs->tree
	  (lambda (charcs)
	    (let* ((op (lambda (charc) (arc->sharcs (charc->arc charc))))
		   (sharcs-l (map op charcs))
		   (sorted-sharcs (merge-sort sharcs-l merge-sharcs '()))
		   (full-sharcs (fill-error sorted-sharcs))
		   (op (lambda (sharc) (cons (caar sharc) (cdr sharc))))
		   (table (list->vector (map op full-sharcs))))
	      (let loop ((left 0) (right (- (vector-length table) 1)))
		(if (= left right)
		    (cdr (vector-ref table left))
		    (let ((mid (quotient (+ left right 1) 2)))
		      (if (and (= (+ left 2) right)
			       (= (+ (car (vector-ref table mid)) 1)
				  (car (vector-ref table right)))
			       (eqv? (cdr (vector-ref table left))
				     (cdr (vector-ref table right))))
			  (list '=
				(car (vector-ref table mid))
				(cdr (vector-ref table mid))
				(cdr (vector-ref table left)))
			  (list (car (vector-ref table mid))
				(loop left (- mid 1))
				(loop mid right))))))))))
    (lambda (tables IS)
      (let ((counters         (vector-ref tables 0))
	    (<<EOF>>-action   (vector-ref tables 1))
	    (<<ERROR>>-action (vector-ref tables 2))
	    (rules-actions    (vector-ref tables 3))
	    (nl-start         (vector-ref tables 5))
	    (no-nl-start      (vector-ref tables 6))
	    (charcs-v         (vector-ref tables 7))
	    (acc-v            (vector-ref tables 8)))
	(let* ((len (vector-length charcs-v))
	       (v (make-vector len)))
	  (let loop ((i (- len 1)))
	    (if (>= i 0)
		(begin
		  (vector-set! v i (charcs->tree (vector-ref charcs-v i)))
		  (loop (- i 1)))
		(lexer-make-tree-lexer
		 (vector counters
			 <<EOF>>-action
			 <<ERROR>>-action
			 rules-actions
			 'decision-trees
			 nl-start
			 no-nl-start
			 v
			 acc-v)
		 IS))))))))

; Fabrication d'un lexer a partir de code pre-genere
(define lexer-make-code-lexer
  (lambda (tables IS)
    (let ((<<EOF>>-pre-action   (vector-ref tables 1))
	  (<<ERROR>>-pre-action (vector-ref tables 2))
	  (rules-pre-action     (vector-ref tables 3))
	  (code                 (vector-ref tables 5)))
      (code <<EOF>>-pre-action <<ERROR>>-pre-action rules-pre-action IS))))

(define lexer-make-lexer
  (lambda (tables IS)
    (let ((automaton-type (vector-ref tables 4)))
      (cond ((eq? automaton-type 'decision-trees)
	     (lexer-make-tree-lexer tables IS))
	    ((eq? automaton-type 'tagged-chars-lists)
	     (lexer-make-char-lexer tables IS))
	    ((eq? automaton-type 'code)
	     (lexer-make-code-lexer tables IS))))))

;
; Table generated from the file prolog.l by SILex 1.0
;

(define lexer-default-table
  (vector
   'all
   (lambda (yycontinue yygetc yyungetc)
     (lambda (yytext yyline yycolumn yyoffset)
                    (list *EOI*-tok)
       ))
   (lambda (yycontinue yygetc yyungetc)
     (lambda (yytext yyline yycolumn yyoffset)
                    (lexing-error yyline yycolumn)
       ))
   (vector
    #f
    (lambda (yycontinue yygetc yyungetc)
      (lambda (yyline yycolumn yyoffset)
                    (yycontinue)
        ))
    #f
    (lambda (yycontinue yygetc yyungetc)
      (lambda (yyline yycolumn yyoffset)
                    (yycontinue)
        ))
    #t
    (lambda (yycontinue yygetc yyungetc)
      (lambda (yytext yyline yycolumn yyoffset)
                    (cons VAR-tok yytext)
        ))
    #t
    (lambda (yycontinue yygetc yyungetc)
      (lambda (yytext yyline yycolumn yyoffset)
                    (cons NUM-tok yytext)
        ))
    #f
    (lambda (yycontinue yygetc yyungetc)
      (lambda (yyline yycolumn yyoffset)
                    (list PERIOD-tok)
        ))
    #f
    (lambda (yycontinue yygetc yyungetc)
      (lambda (yyline yycolumn yyoffset)
                    (list REL-tok)
        ))
    #f
    (lambda (yycontinue yygetc yyungetc)
      (lambda (yyline yycolumn yyoffset)
                    (list GOAL-tok)
        ))
    #t
    (lambda (yycontinue yygetc yyungetc)
      (lambda (yytext yyline yycolumn yyoffset)
                    (cons OP1-tok yytext)
        ))
    #t
    (lambda (yycontinue yygetc yyungetc)
      (lambda (yytext yyline yycolumn yyoffset)
                    (cons OP2-tok yytext)
        ))
    #t
    (lambda (yycontinue yygetc yyungetc)
      (lambda (yytext yyline yycolumn yyoffset)
                    (cons OP3-tok yytext)
        ))
    #t
    (lambda (yycontinue yygetc yyungetc)
      (lambda (yytext yyline yycolumn yyoffset)
                    (cons OP4-tok yytext)
        ))
    #t
    (lambda (yycontinue yygetc yyungetc)
      (lambda (yytext yyline yycolumn yyoffset)
                    (cons OP5-tok yytext)
        ))
    #t
    (lambda (yycontinue yygetc yyungetc)
      (lambda (yytext yyline yycolumn yyoffset)
                    (cons ATOM-tok yytext)
        ))
    #f
    (lambda (yycontinue yygetc yyungetc)
      (lambda (yyline yycolumn yyoffset)
                    (list OPENPAREN-tok)
        ))
    #f
    (lambda (yycontinue yygetc yyungetc)
      (lambda (yyline yycolumn yyoffset)
                    (list CLOSEPAREN-tok)
        ))
    #f
    (lambda (yycontinue yygetc yyungetc)
      (lambda (yyline yycolumn yyoffset)
                    (list OPENBRACKET-tok)
        ))
    #f
    (lambda (yycontinue yygetc yyungetc)
      (lambda (yyline yycolumn yyoffset)
                    (list CLOSEBRACKET-tok)
        ))
    #f
    (lambda (yycontinue yygetc yyungetc)
      (lambda (yyline yycolumn yyoffset)
                    (list VBAR-tok)
        ))
    #f
    (lambda (yycontinue yygetc yyungetc)
      (lambda (yyline yycolumn yyoffset)
                    (list COMMA-tok)
        ))
    #f
    (lambda (yycontinue yygetc yyungetc)
      (lambda (yyline yycolumn yyoffset)
                    (list SEMICOLON-tok)
        )))
   'decision-trees
   0
   1
   '#((62 (41 (34 (11 (9 err 29) (32 err (33 29 9))) (38 (36 err (37 9 28))
    (39 9 (40 10 8)))) (47 (44 (42 7 (43 14 16)) (45 3 (46 26 24))) (59 (48
    14 (58 25 23)) (60 2 (61 18 20))))) (100 (92 (64 (63 17 22) (65 err (91
    27 6))) (95 (93 21 (94 5 9)) (96 27 (97 err 11)))) (111 (106 (101 13
    (105 11 19)) (109 11 (110 12 15))) (125 (123 11 (124 err 4)) (= 126 9
    err))))) (62 (41 (34 (11 (9 err 29) (32 err (33 29 9))) (38 (36 err (37
    9 28)) (39 9 (40 10 8)))) (47 (44 (42 7 (43 32 34)) (45 3 (46 42 41)))
    (59 (48 32 (58 25 40)) (60 2 (61 36 38))))) (100 (92 (64 (63 35 39) (65
    err (91 27 6))) (95 (93 21 (94 5 9)) (96 27 (97 err 11)))) (111 (106
    (101 31 (105 11 37)) (109 11 (110 30 33))) (125 (123 11 (124 err 4)) (=
    126 9 err))))) err err err err err err err (45 (37 (34 (33 err 9) (36
    err 9)) (39 (38 err 9) (42 err (44 9 err)))) (64 (58 (48 9 err) (= 59
    err 9)) (95 (94 err 9) (= 126 9 err)))) (= 39 44 43) (91 (58 (48 err
    45) (65 err 45)) (96 (95 err 45) (97 err (123 45 err)))) (95 (58 (48
    err 45) (65 err (91 45 err))) (111 (= 96 err 45) (112 46 (123 45
    err)))) (95 (58 (48 err 45) (65 err (91 45 err))) (105 (= 96 err 45)
    (106 47 (123 45 err)))) (45 (37 (34 (33 err 9) (36 err 9)) (39 (38 err
    9) (42 err (44 9 err)))) (64 (58 (48 9 err) (= 59 err 9)) (95 (94 err
    9) (= 126 9 err)))) (95 (58 (48 err 45) (65 err (91 45 err))) (111 (=
    96 err 45) (112 48 (123 45 err)))) (45 (37 (34 (33 err 9) (36 err 9))
    (39 (38 err 9) (42 err (44 9 err)))) (64 (58 (48 9 err) (= 59 err 9))
    (95 (94 err 9) (= 126 9 err)))) (48 (38 (34 (33 err 9) (= 36 9 err))
    (42 (39 9 err) (= 44 err 9))) (62 (59 (58 err 9) (60 err (61 9 49)))
    (95 (64 9 (94 err 9)) (= 126 9 err)))) (45 (37 (34 (33 err 9) (36 err
    9)) (39 (38 err 9) (42 err (44 9 err)))) (64 (58 (48 9 err) (= 59 err
    9)) (95 (94 err 9) (= 126 9 err)))) (95 (58 (48 err 45) (65 err (91 45
    err))) (115 (= 96 err 45) (116 50 (123 45 err)))) (58 (38 (34 (33 err
    9) (= 36 9 err)) (44 (39 9 (42 err 9)) (45 err (48 9 err)))) (92 (61
    (59 51 (60 err 54)) (62 53 (64 9 err))) (95 (93 52 (94 err 9)) (= 126 9
    err)))) (= 61 55 err) (46 (38 (34 (33 err 9) (= 36 9 err)) (42 (39 9
    err) (44 9 (45 err 56)))) (64 (58 (48 9 err) (= 59 err 9)) (95 (94 err
    9) (= 126 9 err)))) (46 (38 (34 (33 err 9) (= 36 9 err)) (42 (39 9 err)
    (44 9 (45 err 57)))) (64 (58 (48 9 err) (= 59 err 9)) (95 (94 err 9) (=
    126 9 err)))) (45 (37 (34 (33 err 9) (36 err 9)) (39 (38 err 9) (42 err
    (44 9 err)))) (64 (58 (48 9 err) (= 59 err 9)) (95 (94 err 9) (= 126 9
    err)))) (47 (46 err 58) (48 err (58 25 err))) (45 (37 (34 (33 err 9)
    (36 err 9)) (39 (38 err 9) (42 err (44 9 err)))) (64 (58 (48 9 25) (=
    59 err 9)) (95 (94 err 9) (= 126 9 err)))) (91 (58 (48 err 59) (65 err
    59)) (96 (95 err 59) (97 err (123 59 err)))) (= 10 err 60) (11 (9 err
    29) (= 32 29 err)) (95 (58 (48 err 45) (65 err (91 45 err))) (111 (= 96
    err 45) (112 61 (123 45 err)))) (95 (58 (48 err 45) (65 err (91 45
    err))) (105 (= 96 err 45) (106 62 (123 45 err)))) (45 (37 (34 (33 err
    9) (36 err 9)) (39 (38 err 9) (42 err (44 9 err)))) (64 (58 (48 9 err)
    (= 59 err 9)) (95 (94 err 9) (= 126 9 err)))) (95 (58 (48 err 45) (65
    err (91 45 err))) (111 (= 96 err 45) (112 63 (123 45 err)))) (45 (37
    (34 (33 err 9) (36 err 9)) (39 (38 err 9) (42 err (44 9 err)))) (64 (58
    (48 9 err) (= 59 err 9)) (95 (94 err 9) (= 126 9 err)))) (48 (38 (34
    (33 err 9) (= 36 9 err)) (42 (39 9 err) (= 44 err 9))) (62 (59 (58 err
    9) (60 err (61 9 64))) (95 (64 9 (94 err 9)) (= 126 9 err)))) (45 (37
    (34 (33 err 9) (36 err 9)) (39 (38 err 9) (42 err (44 9 err)))) (64 (58
    (48 9 err) (= 59 err 9)) (95 (94 err 9) (= 126 9 err)))) (95 (58 (48
    err 45) (65 err (91 45 err))) (115 (= 96 err 45) (116 65 (123 45
    err)))) (58 (38 (34 (33 err 9) (= 36 9 err)) (44 (39 9 (42 err 9)) (45
    err (48 9 err)))) (92 (61 (59 66 (60 err 68)) (62 67 (64 9 err))) (95
    (93 52 (94 err 9)) (= 126 9 err)))) (46 (38 (34 (33 err 9) (= 36 9
    err)) (42 (39 9 err) (44 9 (45 err 69)))) (64 (58 (48 9 err) (= 59 err
    9)) (95 (94 err 9) (= 126 9 err)))) (46 (38 (34 (33 err 9) (= 36 9
    err)) (42 (39 9 err) (44 9 (45 err 70)))) (64 (58 (48 9 err) (= 59 err
    9)) (95 (94 err 9) (= 126 9 err)))) (45 (37 (34 (33 err 9) (36 err 9))
    (39 (38 err 9) (42 err (44 9 err)))) (64 (58 (48 9 err) (= 59 err 9))
    (95 (94 err 9) (= 126 9 err)))) (45 (37 (34 (33 err 9) (36 err 9)) (39
    (38 err 9) (42 err (44 9 err)))) (64 (58 (48 9 25) (= 59 err 9)) (95
    (94 err 9) (= 126 9 err)))) (= 39 44 43) err (91 (58 (48 err 45) (65
    err 45)) (96 (95 err 45) (97 err (123 45 err)))) (95 (58 (48 err 45)
    (65 err (91 45 err))) (100 (= 96 err 45) (101 71 (123 45 err)))) (95
    (58 (48 err 45) (65 err (91 45 err))) (118 (= 96 err 45) (119 72 (123
    45 err)))) (95 (58 (48 err 45) (65 err (91 45 err))) (116 (= 96 err 45)
    (117 73 (123 45 err)))) (45 (37 (34 (33 err 9) (36 err 9)) (39 (38 err
    9) (42 err (44 9 err)))) (64 (58 (48 9 err) (= 59 err 9)) (95 (94 err
    9) (= 126 9 err)))) (91 (58 (48 err 45) (65 err 45)) (96 (95 err 45)
    (97 err (123 45 err)))) (48 (38 (34 (33 err 9) (= 36 9 err)) (42 (39 9
    err) (= 44 err 9))) (62 (59 (58 err 9) (60 err (61 9 74))) (95 (64 9
    (94 err 9)) (= 126 9 err)))) (= 61 75 err) (45 (37 (34 (33 err 9) (36
    err 9)) (39 (38 err 9) (42 err (44 9 err)))) (64 (58 (48 9 err) (= 59
    err 9)) (95 (94 err 9) (= 126 9 err)))) (45 (37 (34 (33 err 9) (36 err
    9)) (39 (38 err 9) (42 err (44 9 err)))) (64 (58 (48 9 err) (= 59 err
    9)) (95 (94 err 9) (= 126 9 err)))) (= 61 76 err) (45 (37 (34 (33 err
    9) (36 err 9)) (39 (38 err 9) (42 err (44 9 err)))) (64 (58 (48 9 err)
    (= 59 err 9)) (95 (94 err 9) (= 126 9 err)))) (45 (37 (34 (33 err 9)
    (36 err 9)) (39 (38 err 9) (42 err (44 9 err)))) (64 (58 (48 9 err) (=
    59 err 9)) (95 (94 err 9) (= 126 9 err)))) (48 err (58 77 err)) (91 (58
    (48 err 59) (65 err 59)) (96 (95 err 59) (97 err (123 59 err)))) (= 10
    err 60) (95 (58 (48 err 45) (65 err (91 45 err))) (100 (= 96 err 45)
    (101 78 (123 45 err)))) (95 (58 (48 err 45) (65 err (91 45 err))) (118
    (= 96 err 45) (119 79 (123 45 err)))) (95 (58 (48 err 45) (65 err (91
    45 err))) (116 (= 96 err 45) (117 80 (123 45 err)))) (45 (37 (34 (33
    err 9) (36 err 9)) (39 (38 err 9) (42 err (44 9 err)))) (64 (58 (48 9
    err) (= 59 err 9)) (95 (94 err 9) (= 126 9 err)))) (91 (58 (48 err 45)
    (65 err 45)) (96 (95 err 45) (97 err (123 45 err)))) (48 (38 (34 (33
    err 9) (= 36 9 err)) (42 (39 9 err) (= 44 err 9))) (62 (59 (58 err 9)
    (60 err (61 9 81))) (95 (64 9 (94 err 9)) (= 126 9 err)))) (45 (37 (34
    (33 err 9) (36 err 9)) (39 (38 err 9) (42 err (44 9 err)))) (64 (58 (48
    9 err) (= 59 err 9)) (95 (94 err 9) (= 126 9 err)))) (45 (37 (34 (33
    err 9) (36 err 9)) (39 (38 err 9) (42 err (44 9 err)))) (64 (58 (48 9
    err) (= 59 err 9)) (95 (94 err 9) (= 126 9 err)))) (45 (37 (34 (33 err
    9) (36 err 9)) (39 (38 err 9) (42 err (44 9 err)))) (64 (58 (48 9 err)
    (= 59 err 9)) (95 (94 err 9) (= 126 9 err)))) (45 (37 (34 (33 err 9)
    (36 err 9)) (39 (38 err 9) (42 err (44 9 err)))) (64 (58 (48 9 err) (=
    59 err 9)) (95 (94 err 9) (= 126 9 err)))) (91 (58 (48 err 45) (65 err
    45)) (96 (95 err 45) (97 err (123 45 err)))) (91 (58 (48 err 45) (65
    err 45)) (96 (95 err 45) (97 err (123 45 err)))) (91 (58 (48 err 45)
    (65 err 45)) (96 (95 err 45) (97 err (123 45 err)))) (45 (37 (34 (33
    err 9) (36 err 9)) (39 (38 err 9) (42 err (44 9 err)))) (64 (58 (48 9
    err) (= 59 err 9)) (95 (94 err 9) (= 126 9 err)))) err err (69 (48 err
    (58 77 err)) (101 (70 82 err) (102 82 err))) (91 (58 (48 err 45) (65
    err 45)) (96 (95 err 45) (97 err (123 45 err)))) (91 (58 (48 err 45)
    (65 err 45)) (96 (95 err 45) (97 err (123 45 err)))) (91 (58 (48 err
    45) (65 err 45)) (96 (95 err 45) (97 err (123 45 err)))) (45 (37 (34
    (33 err 9) (36 err 9)) (39 (38 err 9) (42 err (44 9 err)))) (64 (58 (48
    9 err) (= 59 err 9)) (95 (94 err 9) (= 126 9 err)))) (48 err (58 83
    err)) (48 err (58 83 err)))
   '#((#f . #f) (#f . #f) (19 . 19) (18 . 18) (17 . 17) (16 . 16) (15 . 15)
    (14 . 14) (13 . 13) (12 . 12) (#f . #f) (12 . 12) (12 . 12) (12 . 12)
    (10 . 10) (12 . 12) (8 . 8) (7 . 7) (7 . 7) (12 . 12) (7 . 7) (#f . #f)
    (12 . 12) (12 . 12) (4 . 4) (3 . 3) (8 . 8) (2 . 2) (1 . #f) (0 . 0)
    (12 . 12) (12 . 12) (10 . 10) (12 . 12) (8 . 8) (7 . 7) (7 . 7) (12 .
    12) (7 . 7) (12 . 12) (12 . 12) (4 . 4) (8 . 8) (#f . #f) (12 . 12) (12
    . 12) (12 . 12) (12 . 12) (12 . 12) (7 . 7) (7 . 7) (12 . 12) (#f . #f)
    (7 . 7) (7 . 7) (#f . #f) (6 . 6) (5 . 5) (#f . #f) (2 . 2) (1 . #f)
    (12 . 12) (12 . 12) (12 . 12) (7 . 7) (7 . 7) (12 . 12) (7 . 7) (7 . 7)
    (6 . 6) (5 . 5) (11 . 11) (10 . 10) (9 . 9) (7 . 7) (7 . 7) (7 . 7) (3
    . 3) (11 . 11) (10 . 10) (9 . 9) (7 . 7) (#f . #f) (3 . 3))))

;
; User functions
;

(define lexer #f)

(define lexer-get-line   #f)
(define lexer-get-column #f)
(define lexer-get-offset #f)
(define lexer-getc       #f)
(define lexer-ungetc     #f)

(define lexer-init
  (lambda (input-type input)
    (let ((IS (lexer-make-IS input-type input 'all)))
      (set! lexer (lexer-make-lexer lexer-default-table IS))
      (set! lexer-get-line   (lexer-get-func-line IS))
      (set! lexer-get-column (lexer-get-func-column IS))
      (set! lexer-get-offset (lexer-get-func-offset IS))
      (set! lexer-getc       (lexer-get-func-getc IS))
      (set! lexer-ungetc     (lexer-get-func-ungetc IS)))))


;----------------------------------------------------------------------------------------------------



;; ---------------------------------------------------------------------- ;;
;; FICHIER               : lr-dvr.scm                                     ;;
;; DATE DE CREATION      : Fri May 31 15:47:05 1996                       ;;
;; DERNIERE MODIFICATION : Fri May 31 15:51:13 1996                       ;;
;; ---------------------------------------------------------------------- ;;
;; Copyright (c) 1996 Dominique Boucher                                   ;;
;; ---------------------------------------------------------------------- ;;
;; The LR parser driver                                                   ;;
;;                                                                        ;;
;; lr-dvr.scm is part of the lalr.scm distribution which is free          ;;
;; software; you can redistribute it and/or modify                        ;;
;; it under the terms of the GNU General Public License as published by   ;;
;; the Free Software Foundation; either version 2, or (at your option)    ;;
;; any later version.                                                     ;;
;;                                                                        ;;
;; lalr.scm is distributed in the hope that it will be useful,            ;;
;; but WITHOUT ANY WARRANTY; without even the implied warranty of         ;;
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          ;;
;; GNU General Public License for more details.                           ;;
;;                                                                        ;;
;; You should have received a copy of the GNU General Public License      ;;
;; along with lalr.scm; see the file COPYING.  If not, write to           ;;
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  ;;
;; ---------------------------------------------------------------------- ;;

(define max-stack-size 500)
(define *debug* #f)

(define (push stack sp new-cat goto-table lval)
  (let* ((state     (vector-ref stack sp))
	 (new-state (cdr (assq new-cat (vector-ref goto-table state))))
	 (new-sp    (+ sp 2)))
    (if (>= new-sp max-stack-size)
	(error "PARSE ERROR : stack overflow")
	(begin
	  (vector-set! stack new-sp new-state)
	  (vector-set! stack (- new-sp 1) lval)
	  new-sp))))

(define (make-parser action-table goto-table reduction-table token-defs)
  (lambda (lexerp errorp)

    (define (action x l)
      (let ((y (assq x l)))
	(if y 
	    (cdr y) 
	    (cdar l))))
  
    (let ((stack (make-vector max-stack-size 0)))
      (let loop ((sp 0) (input (lexerp)))
	(let* ((state (vector-ref stack sp))
	       (i     (car input))
	       (act   (action i (vector-ref action-table state))))

	  (if *debug*
	      (begin
		(display "** PARSER TRACE: i=") 
		(display (cdr (assq i token-defs)))
		(display "  state=") 
		(display state) 
		(display "  sp=")
		(display sp) 
		(newline)))

	  (cond

	   ;; Input succesfully parsed
	   ((eq? act 'accept)
	    (vector-ref stack 1))

	   ;; Syntax error in input
	   ((eq? act '*error*)
	    (errorp "PARSE ERROR : unexpected token : " 
		    (cdr (assq i token-defs))))

	   ;; Shift current token on top of the stack
	   ((>= act 0)
	    (vector-set! stack (+ sp 1) (cdr input))
	    (vector-set! stack (+ sp 2) act)
	    (loop (+ sp 2) (lexerp)))

	   ;; Reduce by rule (- act)
	   (else 
	    (loop ((vector-ref reduction-table (- act)) stack sp goto-table) 
		  input))))))))


; *** Token Definitions ***
(define *eoi*-tok	0)
(define var-tok	1)
(define atom-tok	2)
(define num-tok	3)
(define op1-tok	4)
(define op2-tok	5)
(define op3-tok	6)
(define op4-tok	7)
(define op5-tok	8)
(define openparen-tok	9)
(define closeparen-tok	10)
(define openbracket-tok	11)
(define closebracket-tok	12)
(define vbar-tok	13)
(define comma-tok	14)
(define semicolon-tok	15)
(define rel-tok	16)
(define goal-tok	17)
(define period-tok	18)

(define token-defs
  (list 
   (cons 0 "*eoi*")
   (cons 1 "var")
   (cons 2 "atom")
   (cons 3 "num")
   (cons 4 "op1")
   (cons 5 "op2")
   (cons 6 "op3")
   (cons 7 "op4")
   (cons 8 "op5")
   (cons 9 "openparen")
   (cons 10 "closeparen")
   (cons 11 "openbracket")
   (cons 12 "closebracket")
   (cons 13 "vbar")
   (cons 14 "comma")
   (cons 15 "semicolon")
   (cons 16 "rel")
   (cons 17 "goal")
   (cons 18 "period")
  ))

; *** Action Table ***
(define action-table
  '#(
     ((default . *error*) (1 . 8) (2 . 7) (3 . 6) (6 . 5) (9 . 4) (11 . 3) (16 . 2) (17 . 1))
     ((default . *error*) (1 . 8) (2 . 7) (3 . 6) (6 . 5) (9 . 4) (11 . 3))
     ((default . *error*) (1 . 8) (2 . 7) (3 . 6) (6 . 5) (9 . 4) (11 . 3))
     ((default . *error*) (1 . 8) (2 . 7) (3 . 6) (6 . 5) (9 . 4) (11 . 3) (12 . 23))
     ((default . *error*) (1 . 8) (2 . 7) (3 . 6) (6 . 5) (9 . 4) (11 . 3) (16 . 2) (17 . 1))
     ((default . *error*) (1 . 8) (2 . 7) (3 . 6) (6 . 5) (9 . 4) (11 . 3))
     ((default . -22))
     ((default . -25) (9 . 27))
     ((default . -23))
     ((default . -27))
     ((default . -24))
     ((default . -21) (8 . 28))
     ((default . -19))
     ((default . -17) (7 . 29))
     ((default . -15))
     ((default . -13) (4 . 31) (5 . 30))
     ((default . -11))
     ((default . -9) (14 . 32))
     ((default . -7) (15 . 34) (16 . 33))
     ((default . *error*) (18 . 35))
     ((default . *error*) (0 . 36))
     ((default . -4) (15 . 34))
     ((default . -5) (15 . 34))
     ((default . -31))
     ((default . *error*) (12 . 39) (13 . 38) (14 . 37))
     ((default . *error*) (10 . 41))
     ((default . -16))
     ((default . *error*) (1 . 8) (2 . 7) (3 . 6) (6 . 5) (9 . 4) (11 . 3))
     ((default . *error*) (1 . 8) (2 . 7) (3 . 6) (9 . 4) (11 . 3))
     ((default . *error*) (1 . 8) (2 . 7) (3 . 6) (9 . 4) (11 . 3))
     ((default . *error*) (1 . 8) (2 . 7) (3 . 6) (6 . 5) (9 . 4) (11 . 3))
     ((default . *error*) (1 . 8) (2 . 7) (3 . 6) (6 . 5) (9 . 4) (11 . 3))
     ((default . *error*) (1 . 8) (2 . 7) (3 . 6) (6 . 5) (9 . 4) (11 . 3))
     ((default . *error*) (1 . 8) (2 . 7) (3 . 6) (6 . 5) (9 . 4) (11 . 3))
     ((default . *error*) (1 . 8) (2 . 7) (3 . 6) (6 . 5) (9 . 4) (11 . 3))
     ((default . -3) (1 . 8) (2 . 7) (3 . 6) (6 . 5) (9 . 4) (11 . 3) (16 . 2) (17 . 1))
     ((default . -1) (0 . accept))
     ((default . *error*) (1 . 8) (2 . 7) (3 . 6) (6 . 5) (9 . 4) (11 . 3))
     ((default . *error*) (1 . 8) (2 . 7) (3 . 6) (6 . 5) (9 . 4) (11 . 3))
     ((default . -33))
     ((default . -32))
     ((default . -26))
     ((default . *error*) (10 . 54) (14 . 53))
     ((default . -20))
     ((default . -18))
     ((default . -14))
     ((default . -12) (5 . 30))
     ((default . -10))
     ((default . -6) (15 . 34))
     ((default . -8) (14 . 32))
     ((default . -2))
     ((default . *error*) (12 . 39) (13 . 38) (14 . 37))
     ((default . *error*) (12 . 57))
     ((default . *error*) (1 . 8) (2 . 7) (3 . 6) (6 . 5) (9 . 4) (11 . 3))
     ((default . -29))
     ((default . -28))
     ((default . -35))
     ((default . -34))
     ((default . *error*) (10 . 54) (14 . 53))
     ((default . -30))
    ))

; *** Goto Table ***
(define goto-table
  '#(
     ((13 . 9)(11 . 10)(10 . 11)(9 . 12)(8 . 13)(7 . 14)(6 . 15)(5 . 16)(4 . 17)(3 . 18)(2 . 19)(1 . 20))
     ((13 . 9)(11 . 10)(10 . 11)(9 . 12)(8 . 13)(7 . 14)(6 . 15)(5 . 16)(4 . 17)(3 . 21))
     ((13 . 9)(11 . 10)(10 . 11)(9 . 12)(8 . 13)(7 . 14)(6 . 15)(5 . 16)(4 . 17)(3 . 22))
     ((13 . 9)(11 . 10)(10 . 11)(9 . 12)(8 . 13)(7 . 14)(6 . 15)(5 . 24))
     ((13 . 9)(11 . 10)(10 . 11)(9 . 12)(8 . 13)(7 . 14)(6 . 15)(5 . 16)(4 . 17)(3 . 18)(2 . 25))
     ((13 . 9)(11 . 10)(10 . 11)(9 . 12)(8 . 13)(7 . 26))
     ()
     ()
     ()
     ()
     ()
     ()
     ()
     ()
     ()
     ()
     ()
     ()
     ()
     ()
     ()
     ()
     ()
     ()
     ((14 . 40))
     ()
     ()
     ((13 . 9)(11 . 10)(10 . 11)(9 . 12)(8 . 13)(7 . 14)(6 . 15)(5 . 42))
     ((13 . 9)(11 . 10)(10 . 43))
     ((13 . 9)(11 . 10)(10 . 11)(9 . 44))
     ((13 . 9)(11 . 10)(10 . 11)(9 . 12)(8 . 13)(7 . 45))
     ((13 . 9)(11 . 10)(10 . 11)(9 . 12)(8 . 13)(7 . 14)(6 . 46))
     ((13 . 9)(11 . 10)(10 . 11)(9 . 12)(8 . 13)(7 . 14)(6 . 15)(5 . 47))
     ((13 . 9)(11 . 10)(10 . 11)(9 . 12)(8 . 13)(7 . 14)(6 . 15)(5 . 16)(4 . 17)(3 . 48))
     ((13 . 9)(11 . 10)(10 . 11)(9 . 12)(8 . 13)(7 . 14)(6 . 15)(5 . 16)(4 . 49))
     ((13 . 9)(11 . 10)(10 . 11)(9 . 12)(8 . 13)(7 . 14)(6 . 15)(5 . 16)(4 . 17)(3 . 18)(2 . 19)(1 . 50))
     ()
     ((13 . 9)(11 . 10)(10 . 11)(9 . 12)(8 . 13)(7 . 14)(6 . 15)(5 . 51))
     ((13 . 9)(11 . 10)(10 . 11)(9 . 12)(8 . 13)(7 . 14)(6 . 15)(5 . 52))
     ()
     ()
     ()
     ((12 . 55))
     ()
     ()
     ()
     ()
     ()
     ()
     ()
     ()
     ((14 . 56))
     ()
     ((13 . 9)(11 . 10)(10 . 11)(9 . 12)(8 . 13)(7 . 14)(6 . 15)(5 . 58))
     ()
     ()
     ()
     ()
     ((12 . 59))
     ()
    ))

; *** Reduction Table ***
(define reduction-table
  (vector
    '()
    (lambda (stack sp goto-table)
      (let* (($2 (vector-ref stack (- sp 1)))
             ($1 (vector-ref stack (- sp 3))))
          (accept $1)))
    (lambda (stack sp goto-table)
      (let* (($3 (vector-ref stack (- sp 1)))
             ($2 (vector-ref stack (- sp 3)))
             ($1 (vector-ref stack (- sp 5))))
          (push stack (- sp 6) 1 goto-table (cons $1 $3))))
    (lambda (stack sp goto-table)
      (let* (($2 (vector-ref stack (- sp 1)))
             ($1 (vector-ref stack (- sp 3))))
          (push stack (- sp 4) 1 goto-table (list $1))))
    (lambda (stack sp goto-table)
      (let* (($2 (vector-ref stack (- sp 1)))
             ($1 (vector-ref stack (- sp 3))))
          (push stack (- sp 4) 2 goto-table (quasiquote (goal (unquote $2))))))
    (lambda (stack sp goto-table)
      (let* (($2 (vector-ref stack (- sp 1)))
             ($1 (vector-ref stack (- sp 3))))
          (push stack (- sp 4) 2 goto-table (quasiquote (rel (unquote $2))))))
    (lambda (stack sp goto-table)
      (let* (($3 (vector-ref stack (- sp 1)))
             ($2 (vector-ref stack (- sp 3)))
             ($1 (vector-ref stack (- sp 5))))
          (push stack (- sp 6) 2 goto-table (quasiquote (rel (unquote $1) (unquote $3))))))
    (lambda (stack sp goto-table)
      (let* (($1 (vector-ref stack (- sp 1))))
          (push stack (- sp 2) 2 goto-table $1)))
    (lambda (stack sp goto-table)
      (let* (($3 (vector-ref stack (- sp 1)))
             ($2 (vector-ref stack (- sp 3)))
             ($1 (vector-ref stack (- sp 5))))
          (push stack (- sp 6) 3 goto-table (quasiquote (pred "or" (unquote $1) (unquote $3))))))
    (lambda (stack sp goto-table)
      (let* (($1 (vector-ref stack (- sp 1))))
          (push stack (- sp 2) 3 goto-table $1)))
    (lambda (stack sp goto-table)
      (let* (($3 (vector-ref stack (- sp 1)))
             ($2 (vector-ref stack (- sp 3)))
             ($1 (vector-ref stack (- sp 5))))
          (push stack (- sp 6) 4 goto-table (quasiquote (pred "and" (unquote $1) (unquote $3))))))
    (lambda (stack sp goto-table)
      (let* (($1 (vector-ref stack (- sp 1))))
          (push stack (- sp 2) 4 goto-table $1)))
    (lambda (stack sp goto-table)
      (let* (($3 (vector-ref stack (- sp 1)))
             ($2 (vector-ref stack (- sp 3)))
             ($1 (vector-ref stack (- sp 5))))
          (push stack (- sp 6) 5 goto-table (quasiquote (pred (unquote $2) (unquote $1) (unquote $3))))))
    (lambda (stack sp goto-table)
      (let* (($1 (vector-ref stack (- sp 1))))
          (push stack (- sp 2) 5 goto-table $1)))
    (lambda (stack sp goto-table)
      (let* (($3 (vector-ref stack (- sp 1)))
             ($2 (vector-ref stack (- sp 3)))
             ($1 (vector-ref stack (- sp 5))))
          (push stack (- sp 6) 6 goto-table (quasiquote (pred (unquote $2) (unquote $1) (unquote $3))))))
    (lambda (stack sp goto-table)
      (let* (($1 (vector-ref stack (- sp 1))))
          (push stack (- sp 2) 6 goto-table $1)))
    (lambda (stack sp goto-table)
      (let* (($2 (vector-ref stack (- sp 1)))
             ($1 (vector-ref stack (- sp 3))))
          (push stack (- sp 4) 7 goto-table (quasiquote (pred (unquote $1) (unquote $2))))))
    (lambda (stack sp goto-table)
      (let* (($1 (vector-ref stack (- sp 1))))
          (push stack (- sp 2) 7 goto-table $1)))
    (lambda (stack sp goto-table)
      (let* (($3 (vector-ref stack (- sp 1)))
             ($2 (vector-ref stack (- sp 3)))
             ($1 (vector-ref stack (- sp 5))))
          (push stack (- sp 6) 8 goto-table (quasiquote (pred (unquote $2) (unquote $1) (unquote $3))))))
    (lambda (stack sp goto-table)
      (let* (($1 (vector-ref stack (- sp 1))))
          (push stack (- sp 2) 8 goto-table $1)))
    (lambda (stack sp goto-table)
      (let* (($3 (vector-ref stack (- sp 1)))
             ($2 (vector-ref stack (- sp 3)))
             ($1 (vector-ref stack (- sp 5))))
          (push stack (- sp 6) 9 goto-table (quasiquote (pred (unquote $2) (unquote $1) (unquote $3))))))
    (lambda (stack sp goto-table)
      (let* (($1 (vector-ref stack (- sp 1))))
          (push stack (- sp 2) 9 goto-table $1)))
    (lambda (stack sp goto-table)
      (let* (($1 (vector-ref stack (- sp 1))))
          (push stack (- sp 2) 10 goto-table (quasiquote (num (unquote $1))))))
    (lambda (stack sp goto-table)
      (let* (($1 (vector-ref stack (- sp 1))))
          (push stack (- sp 2) 10 goto-table (quasiquote (var (unquote $1))))))
    (lambda (stack sp goto-table)
      (let* (($1 (vector-ref stack (- sp 1))))
          (push stack (- sp 2) 10 goto-table $1)))
    (lambda (stack sp goto-table)
      (let* (($1 (vector-ref stack (- sp 1))))
          (push stack (- sp 2) 10 goto-table (quasiquote (pred (unquote $1))))))
    (lambda (stack sp goto-table)
      (let* (($3 (vector-ref stack (- sp 1)))
             ($2 (vector-ref stack (- sp 3)))
             ($1 (vector-ref stack (- sp 5))))
          (push stack (- sp 6) 10 goto-table $2)))
    (lambda (stack sp goto-table)
      (let* (($1 (vector-ref stack (- sp 1))))
          (push stack (- sp 2) 10 goto-table $1)))
    (lambda (stack sp goto-table)
      (let* (($4 (vector-ref stack (- sp 1)))
             ($3 (vector-ref stack (- sp 3)))
             ($2 (vector-ref stack (- sp 5)))
             ($1 (vector-ref stack (- sp 7))))
          (push stack (- sp 8) 11 goto-table (quasiquote (pred (unquote $1) (unquote $3) (unquote-splicing $4))))))
    (lambda (stack sp goto-table)
      (let* (($1 (vector-ref stack (- sp 1))))
          (push stack (- sp 2) 12 goto-table (quote ()))))
    (lambda (stack sp goto-table)
      (let* (($3 (vector-ref stack (- sp 1)))
             ($2 (vector-ref stack (- sp 3)))
             ($1 (vector-ref stack (- sp 5))))
          (push stack (- sp 6) 12 goto-table (cons $2 $3))))
    (lambda (stack sp goto-table)
      (let* (($2 (vector-ref stack (- sp 1)))
             ($1 (vector-ref stack (- sp 3))))
          (push stack (- sp 4) 13 goto-table (quote (null)))))
    (lambda (stack sp goto-table)
      (let* (($3 (vector-ref stack (- sp 1)))
             ($2 (vector-ref stack (- sp 3)))
             ($1 (vector-ref stack (- sp 5))))
          (push stack (- sp 6) 13 goto-table (quasiquote (pred "." (unquote $2) (unquote $3))))))
    (lambda (stack sp goto-table)
      (let* (($1 (vector-ref stack (- sp 1))))
          (push stack (- sp 2) 14 goto-table (quote (null)))))
    (lambda (stack sp goto-table)
      (let* (($3 (vector-ref stack (- sp 1)))
             ($2 (vector-ref stack (- sp 3)))
             ($1 (vector-ref stack (- sp 5))))
          (push stack (- sp 6) 14 goto-table $2)))
    (lambda (stack sp goto-table)
      (let* (($3 (vector-ref stack (- sp 1)))
             ($2 (vector-ref stack (- sp 3)))
             ($1 (vector-ref stack (- sp 5))))
          (push stack (- sp 6) 14 goto-table (quasiquote (pred "." (unquote $2) (unquote $3))))))
  ))

; *** Parser Definition ***
(define parser
   (make-parser
    action-table
    goto-table
    reduction-table
    token-defs))


;----------------------------------------------------------------------------------------------------


;;; Error handling:

(define (lexing-error row col)
  (fprintf (current-error-port) "Syntax error at line ~S, column ~S~%" row col)
  (exit 1) )

(define (parse-error msg . args)
  (with-output-to-port (current-error-port)
    (lambda ()
      (printf "~A~%" msg)
      (for-each (lambda (x) (printf "~S~%" x)) args)
      (exit 2) ) ) )


;;; Convert Prolog names to Scheme symbols:

(define (prolog-id->scheme-id name arity)
  (string->symbol (sprintf "%~A/~S" name arity)) )


;;; Collect clauses with same name and arity:

(define (collect-similar-clauses prg sets)

  (define (update name arity entry)
    (let* ([head (prolog-id->scheme-id name arity)]
	   [prev (or (hash-table-ref sets head) '())] )
      (hash-table-set! sets head (cons entry prev)) ) )

  (define sequentialize-body
    (match-lambda
      [`(pred "," ,x ,y) (cons x (sequentialize-body y))]
      [x (list x)] ) )

  (for-each 
   (match-lambda
     [`(goal ,body) (update "goal" 0 body)]
     [`(rel ,(and head `(pred ,name ,@args)) ,body)
      (update name (length args) (cons args (sequentialize-body body))) ]
     [(and head `(pred ,name ,@args)) (update name (length args) (list args))] )
   prg) )


;;; Parse complete source code and translate to Schelog:

(define (parse-tree->schelog name clauses)

  (define extract-vars
    (match-lambda
      [`(var "_") '()]
      [`(var ,var) (list (string->symbol (string-append "_" var)))]
      [(x . y) (append (extract-vars x) (extract-vars y))]
      [_ '()] ) )

  (define translate
    (match-lambda
      ['(null) ''()]
      [`(num ,n) (string->number n)]
      [`(var "_") '(_)]
      [`(var ,x) (string->symbol (string-append "_" x))]
      ['(pred "!") '!]
      [`(pred "." ,x ,y) `(cons ,(translate x) ,(translate y))]
      [`(pred "and" (pred "and" ,xs1 ...) ,xs2 ...) (translate `(pred "and" ,@xs1 ,@xs2))]
      [`(pred "and" ,xs ...) `(%and ,@(map translate xs))]
      [`(pred "or" ,xs ...) `(%or ,@(map translate xs))]
      [`(pred ,hd ,args ...) (cons (prolog-id->scheme-id hd (length args)) (map translate args))]
      [(xs ...) (map translate xs)] ) )

  (if (eq? name '%goal/0)
      `(%repeat-query
	(lambda ()
	  (%which ,(delete-duplicates (extract-vars clauses) eq?) ,(translate clauses)) ) )
      `(define ,name
	 (%rel ,(delete-duplicates (extract-vars clauses) eq?)
	   ,@(map (lambda (clause)
		    (cons (translate (car clause)) 
			  (map translate (cdr clause)) ) )
		  clauses) ) ) ) )


;;; User read pass:

(user-read-pass
 (lambda (prelude files postlude)
   (let ([goals '()]
	 [sets (make-hash-table)] 
	 [exprs '((declare (uses schelog-support))
		  (include "schelog-macros") ) ] )

     (define (parse-input)
       (let ([prg (parser lexer parse-error)])
	 (collect-similar-clauses prg sets) ) )

     (define (parse-strings ss)
       (for-each 
	(lambda (s)
	  (lexer-init 'string s)
	  (parse-input) ) 
	ss) )

     (parse-strings prelude)
     (for-each
      (lambda (f)
	(call-with-input-file f
	  (lambda (in) 
	    (lexer-init 'port in)
	    (parse-input) ) ) )
      files)
     (parse-strings postlude)
     (hash-table-for-each
      (lambda (head entry)
	(if (eq? '%goal/0 head)
	    (set! goals entry)
	    (set! exprs (cons (parse-tree->schelog head (reverse entry)) exprs)) ) )
      sets)
     (for-each
      (lambda (goal)
	(set! exprs (cons (parse-tree->schelog '%goal/0 goal) exprs)) )
      goals)
     (reverse exprs) ) ) )

