;;;; highlevel-macros.scm


;;; macro-defs.ss
;;; Robert Hieb & Kent Dybvig
;;; 92/06/18

(define-syntax with-syntax
   (lambda (x)
      (syntax-case x ()
         ((_ () e1 e2 ...)
          (syntax (begin e1 e2 ...)))
         ((_ ((out in)) e1 e2 ...)
          (syntax (syntax-case in () (out (begin e1 e2 ...)))))
         ((_ ((out in) ...) e1 e2 ...)
          (syntax (syntax-case (list in ...) ()
                     ((out ...) (begin e1 e2 ...))))))))

(define-syntax syntax-rules
   (lambda (x)
      (syntax-case x ()
         ((_ (k ...) ((keyword . pattern) template) ...)
          (with-syntax (((dummy ...)
                         (generate-temporaries (syntax (keyword ...)))))
             (syntax (lambda (x)
                        (syntax-case x (k ...)
                           ((dummy . pattern) (syntax template))
                           ...))))))))

(define-syntax or
   (lambda (x)
      (syntax-case x ()
         ((_) (syntax #f))
         ((_ e) (syntax e))
         ((_ e1 e2 e3 ...)
          (syntax (let ((t e1)) (if t t (or e2 e3 ...))))))))

(define-syntax and
   (lambda (x)
      (syntax-case x ()
         ((_ e1 e2 e3 ...) (syntax (if e1 (and e2 e3 ...) #f)))
         ((_ e) (syntax e))
         ((_) (syntax #t)))))

(define-syntax cond
   (lambda (x)
      (syntax-case x (else =>)
         ((_ (else e1 e2 ...))
          (syntax (begin e1 e2 ...)))
         ((_ (e0))
          (syntax (let ((t e0)) (if t t))))
         ((_ (e0) c1 c2 ...)
          (syntax (let ((t e0)) (if t t (cond c1 c2 ...)))))
         ((_ (e0 => e1)) (syntax (let ((t e0)) (if t (e1 t)))))
         ((_ (e0 => e1) c1 c2 ...)
          (syntax (let ((t e0)) (if t (e1 t) (cond c1 c2 ...)))))
         ((_ (e0 e1 e2 ...)) (syntax (if e0 (begin e1 e2 ...))))
         ((_ (e0 e1 e2 ...) c1 c2 ...)
          (syntax (if e0 (begin e1 e2 ...) (cond c1 c2 ...)))))))

(define-syntax let*
   (lambda (x)
      (syntax-case x ()
         ((let* () e1 e2 ...)
          (syntax (let () e1 e2 ...)))
         ((let* ((x1 v1) (x2 v2) ...) e1 e2 ...)
          (##syncase#andmap identifier? (syntax (x1 x2 ...)))
          (syntax (let ((x1 v1)) (let* ((x2 v2) ...) e1 e2 ...)))))))

(define-syntax case
   (lambda (x)
      (syntax-case x (else)
	 ((_ v (else e1 e2 ...))
          (syntax (begin e1 e2 ...)))
         ((_ v ((k1 ...) e1 e2 ...))
	  (syntax (let ((x v))
		    (if (or (eqv? x 'k1) ...) (begin e1 e2 ...)) ) ) )
         ((_ v ((k1 ...) e1 e2 ...) c1 c2 ...)
	  (syntax (let ((x v))
		    (if (or (eqv? x 'k1) ...)
			(begin e1 e2 ...)
			(case x c1 c2 ...))))))) )

(define-syntax do
   (lambda (orig-x)
      (syntax-case orig-x ()
         ((_ ((var init . step) ...) (e0 e1 ...) c ...)
          (with-syntax (((step ...)
                         (map (lambda (v s)
                                 (syntax-case s ()
                                    (() v)
                                    ((e) (syntax e))
                                    (_ (syntax-error orig-x))))
                              (syntax (var ...))
                              (syntax (step ...)))))
             (syntax-case (syntax (e1 ...)) ()
                (() (syntax (let doloop ((var init) ...)
                               (if (not e0)
                                   (begin c ... (doloop step ...))))))
                ((e1 e2 ...)
                 (syntax (let doloop ((var init) ...)
                            (if e0
                                (begin e1 e2 ...)
                                (begin c ... (doloop step ...))))))))))))

(define-syntax quasiquote
   (letrec
      ((gen-cons
        (lambda (x y)
           (syntax-case x (quote)
              ((quote x)
               (syntax-case y (quote ##sys#list)
                  ((quote y) (syntax (quote (x . y))))
                  ((##sys#list y ...) (syntax (##sys#list (quote x) y ...)))
                  (y (syntax (##sys#cons (quote x) y)))))
              (x (syntax-case y (quote ##sys#list)
                   ((quote ()) (syntax (##sys#list x)))
                   ((##sys#list y ...) (syntax (##sys#list x y ...)))
                   (y (syntax (##sys#cons x y))))))))

       (gen-append
        (lambda (x y)
           (syntax-case x (quote ##sys#list ##sys#cons)
              ((quote (x1 x2 ...))
               (syntax-case y (quote)
                  ((quote y) (syntax (quote (x1 x2 ... . y))))
                  (y (syntax (##sys#append (quote (x1 x2 ...) y))))))
              ((quote ()) y)
              ((##sys#list x1 x2 ...)
               (gen-cons (syntax x1) (gen-append (syntax (##sys#list x2 ...)) y)))
              (x (syntax-case y (quote ##sys#list)
                   ((quote ()) (syntax x))
                   (y (syntax (##sys#append x y))))))))

       (gen-vector
        (lambda (x)
           (syntax-case x (quote ##sys#list)
              ((quote (x ...)) (syntax (quote #(x ...))))
              ((##sys#list x ...) (syntax (##sys#vector x ...)))
              (x (syntax (##sys#list->vector x))))))

       (gen
        (lambda (p lev)
           (syntax-case p (unquote unquote-splicing quasiquote)
              ((unquote p)
               (if (= lev 0)
                   (syntax p)
                   (gen-cons (syntax (quote unquote))
                             (gen (syntax (p)) (- lev 1)))))
              (((unquote-splicing p) . q)
               (if (= lev 0)
                   (gen-append (syntax p) (gen (syntax q) lev))
                   (gen-cons (gen-cons (syntax (quote unquote-splicing))
                                       (gen (syntax p) (- lev 1)))
                             (gen (syntax q) lev))))
              ((quasiquote p)
               (gen-cons (syntax (quote quasiquote))
                         (gen (syntax (p)) (+ lev 1))))
              ((p . q)
               (gen-cons (gen (syntax p) lev) (gen (syntax q) lev)))
              (#(x ...) (gen-vector (gen (syntax (x ...)) lev)))
              (p (syntax (quote p)))))))

    (lambda (x)
       (syntax-case x ()
          ((- e) (gen (syntax e) 0))))))


;;; CHICKEN specific macros:

(define-syntax delay
   (lambda (x)
      (syntax-case x ()
         ((delay exp)
          (syntax (##sys#make-promise (lambda () exp)))))))

(define-syntax receive
  (lambda (x)
    (syntax-case x ()
      ((_ vars x0 x1 x2 ...)
       (syntax (##sys#call-with-values (lambda () x0)
		 (lambda vars x1 x2 ...) ) ) ) ) ) )

(define-syntax time
  (lambda (x)
    (syntax-case x ()
      ((_ exp)
       (syntax (begin
		 (##sys#start-timer)
		 (##sys#call-with-values
		  (lambda () exp)
		  (lambda tmp
		    (##sys#display-times (##sys#stop-timer))
		    (##sys#apply ##sys#values tmp) ) ) ) ) ) ) ) )

(define-syntax declare
  (lambda (x)
    (syntax-case x ()
      ((_ spec1 ...) (syntax (##core#declare 'spec1 ...))) ) ) )

(define-syntax include
  (lambda (x)
    (syntax-case x ()
      ((_ filename) (syntax (##core#include 'filename))) ) ) )

(define-syntax assert
  (lambda (x)
    (syntax-case x ()
      ((_ exp)
       (syntax (assert exp "assertion failed")) )
      ((_ exp msg arg1 ...)
       (syntax (if (not exp) (##sys#error (##core#immutable 'msg) 'exp arg1 ...))) ) ) ) )

(define-syntax define-record
  (lambda (x)
    (define construct-name
      (lambda (template-identifier prefix . args)
	(implicit-identifier
	 template-identifier
	 (##sys#string->qualified-symbol 
	  prefix
	  (##sys#apply string-append
		 (map (lambda (x)
			(if (string? x)
			    x
			    (symbol->string (syntax-object->datum x))))
		      args))))))
    (syntax-case x ()
      ((_ name id1 ...)
       (let* ([nm (syntax-object->datum (syntax name))]
	      [prefix (##sys#qualified-symbol-prefix nm)] )
       (with-syntax
	((constructor (construct-name (syntax name) prefix "make-" (syntax name)))
	 (predicate (construct-name (syntax name) prefix (syntax name) "?"))
	 ((access ...)
	  (map (lambda (x) (construct-name x prefix (syntax name) "-" x))
	       (syntax (id1 ...))))
	 ((assign ...)
	  (map (lambda (x)
		 (construct-name x prefix (syntax name) "-" x "-set!"))
	       (syntax (id1 ...))))
	 ((index ...)
	  (let f ((i 1) (ids (syntax (id1 ...))))
	    (if (null? ids)
		'()
		(cons i (f (+ i 1) (cdr ids)))))))
	(syntax (begin
		  (define constructor
		    (lambda (id1 ...)
		      (##sys#make-structure 'name id1 ...)))
		  (define predicate
		    (lambda (x) (##sys#structure? x 'name)) )
		  (define access
		    (lambda (x)
		      (##sys#check-structure x 'name)
		      (##sys#slot x index)))
		  ...
		  (define assign
		    (lambda (x update)
		      (##sys#check-structure x 'name)
		      (##sys#setslot x index update)))
		  ...))))))))

(define-syntax cond-expand
  (lambda (x)
    (syntax-case x (else not or and)
      [(_)
       (##sys#error
	(##core#immutable '"no matching clause in `cond-expand' form") ) ]
      [(_ (else body ...)) 
       (syntax (begin body ...)) ]
      [(_ ((and) body ...) more ...)
       (syntax (begin body ...)) ]
      [(_ ((and req1 req2 ...) body ...) more ...)
       (syntax (cond-expand
		(req1
		 (cond-expand
		  ((and req2 ...) body ...)
		  more ...))
		more ...) ) ]
      [(_ ((or) body ...) more ...)
       (syntax (cond-expand more ...)) ]
      [(_ ((or req1 req2 ...) body ...) more ...)
       (syntax (cond-expand
		(req1 (begin body ...))
		(else (cond-expand
		       ((or req2 ...) body ...)
		       more ...) ) ) ) ]
      [(_ ((not req) body ...) more ...)
       (syntax (cond-expand
		(req (cond-expand more ...))
		(else body ...) ) ) ]
      [(_ (req body ...) more ...)
       (if (##sys#test-feature (syntax-object->datum (syntax req)))
	   (syntax (begin body ...))
	   (syntax (cond-expand more ...)) ) ] ) ) )

(define-syntax fluid-let
  (lambda (x)
    (syntax-case x ()
      [(_ ((var val) ...) x1 x2 ...)
       (with-syntax ([(new ...) (generate-temporaries (syntax (var ...)))]
		     [(old ...) (generate-temporaries (syntax (var ...)))] )
	 (syntax
	  (let ([new val] ...
		[old #f] ...)
	    (##sys#dynamic-wind
		(lambda ()
		  (set! old var) ...
		  (set! var new) ... )
		(lambda () x1 x2 ...)
		(lambda ()
		  (set! new var) ...
		  (set! var old) ...) ) ) ) ) ] ) ) )

(define-syntax case-lambda		; (reference implementation)
  (syntax-rules ()
      ((case-lambda 
	(?a1 ?e1 ...) 
	?clause1 ...)
       (lambda args
	 (let ((l (length args)))
	   (case-lambda "CLAUSE" args l 
			(?a1 ?e1 ...)
			?clause1 ...))))
      ((case-lambda "CLAUSE" ?args ?l 
		    ((?a1 ...) ?e1 ...) 
		    ?clause1 ...)
       (if (eq? ?l (length '(?a1 ...)))
	   (##sys#apply (lambda (?a1 ...) ?e1 ...) ?args)
	   (case-lambda "CLAUSE" ?args ?l 
			?clause1 ...)))
      ((case-lambda "CLAUSE" ?args ?l
		    ((?a1 . ?ar) ?e1 ...) 
		    ?clause1 ...)
       (case-lambda "IMPROPER" ?args ?l 1 (?a1 . ?ar) (?ar ?e1 ...) 
		    ?clause1 ...))
      ((case-lambda "CLAUSE" ?args ?l 
		    (?a1 ?e1 ...)
		    ?clause1 ...)
       (let ((?a1 ?args))
	 ?e1 ...))
      ((case-lambda "CLAUSE" ?args ?l)
       (##sys#error (##core#immutable '"wrong number of arguments to CASE-LAMBDA.")))
      ((case-lambda "IMPROPER" ?args ?l ?k ?al ((?a1 . ?ar) ?e1 ...)
		    ?clause1 ...)
       (case-lambda "IMPROPER" ?args ?l (+ ?k 1) ?al (?ar ?e1 ...) 
		    ?clause1 ...))
      ((case-lambda "IMPROPER" ?args ?l ?k ?al (?ar ?e1 ...) 
		    ?clause1 ...)
       (if (fx>= ?l ?k)
	   (##sys#apply (lambda ?al ?e1 ...) ?args)
	   (case-lambda "CLAUSE" ?args ?l 
			?clause1 ...)))))

(register-feature! 'srfi-16)

(define-syntax and-let*
   (syntax-rules ()
      ((and-let* () body ...)
       (begin body ...))

      ((and-let* ((var expr) clauses ...) body ...)
       (let ((var expr))
	  (if var (and-let* (clauses ...) body ...) #f)))

      ((and-let* ((expr) clauses ...) body ...)
       (if expr (and-let* (clauses ...) body ...) #f))
      
      ((and-let* (var clauses ...) body ...)
       (if var (and-let* (clauses ...) body ...) #f))))

(define-syntax eval-when
  (lambda (x)
    (syntax-case x ()
      [(_ (situations ...) body ...)
       (let ([e #f]
	     [c #f]
	     [l #f] )
	 (let loop ([ss (##sys#map syntax-object->datum (syntax (situations ...)))])
	   (if (pair? ss)
	       (begin
		 (case (car ss)
		   [(eval) (set! e #t)]
		   [(load) (set! l #t)]
		   [(compile) (set! c #t)]
		   [else (##sys#error (##core#immutable '"invalid situation specifier") (car ss))] )
		 (loop (cdr ss)) ) ) )
	 (if (memq '#:compiling ##sys#features)
	     (cond [(and c l) (syntax (##core#compiletimetoo (begin body ...)))]
		   [c (syntax (##core#compiletimeonly (begin body ...)))]
		   [l (syntax (begin body ...))]
		   [else (syntax (##core#undefined))] )
	     (if e 
		 (syntax (begin body ...))
		 (syntax (##core#undefined)) ) ) ) ] ) ) )

(define-syntax parameterize
  (lambda (x)
    (syntax-case x ()
      [(_ () e1 e2 ...) (syntax (begin e1 e2 ...))]
      [(_ ([x v] ...) e1 e2 ...)
       (##syncase#andmap identifier? (syntax (x ...)))
       (with-syntax ([(p ...) (generate-temporaries (syntax (x ...)))]
		     [(y ...) (generate-temporaries (syntax (x ...)))])
	 (syntax
	  (let ([p x] ... [y v] ...)
	    (let ([swap (lambda () (let ([t (p)]) (p y) (set! y t)) ...)])
	      (##sys#dynamic-wind swap (lambda () e1 e2 ...) swap)))))])))

(define-syntax when
  (syntax-rules ()
    [(_ x y z ...) (if x (begin y z ...))] ) )

(define-syntax unless
  (syntax-rules ()
    [(_ x y z ...) (if x (##core#undefined) (begin y z ...))] ) )

(define-syntax set!-values
  (lambda (x)
    (syntax-case x ()
      [(_ (var1 ...) exp)
       (with-syntax ([(tmp1 ...) (generate-temporaries (syntax (var1 ...)))])
	 (syntax
	  (##sys#call-with-values (lambda () exp)
	    (lambda (tmp1 ...)
	      (set! var1 tmp1) ...) ) ) ) ] ) ) )

(define-syntax define-values
  (syntax-rules ()
    [(_ (var1 ...) exp) (set!-values (var1 ...) exp)] ) )

(define-syntax let*-values
  (syntax-rules ()
    [(_ () exp1 ...) (let () exp1 ...)]
    [(_ (((var1 ...) exp) . rest) exp1 ...)
     (##sys#call-with-values (lambda () exp)
       (lambda (var1 ...)
	 (let*-values rest exp1 ...) ) ) ] ) )

(define-syntax let-values
  (lambda (x)
    (syntax-case x ()
      [(_ "INTERNAL" () tbindings body) (syntax (let tbindings body))]
      [(_ "INTERNAL" (((var1 ...) exp1) . rest) (tbindings ...) body)
       (with-syntax ([(tmp1 ...) (generate-temporaries (syntax (var1 ...)))])
	 (syntax
	  (##sys#call-with-values (lambda () exp1)
	    (lambda (tmp1 ...)
	      (let-values "INTERNAL" rest (tbindings ... (var1 tmp1) ...) body) ) ) ) ) ]
      [(_ bindings body1 ...) (syntax (let-values "INTERNAL" bindings () (let () body1 ...)))] ) ) )

(define-syntax letrec-values
  (lambda (x)
    (syntax-case x ()
      [(_ "INTERNAL1" () bindings body)
       (syntax (letrec-values "INTERNAL2" bindings body)) ]
      [(_ "INTERNAL1" (((var1 ...) exp) . rest) (binding1 ...) body)
       (syntax
	(let ((var1 (##core#undefined)) ...)
	  (letrec-values "INTERNAL1" rest (binding1 ... ((var1 ...) exp)) body) ) ) ]
      [(_ "INTERNAL2" () body) (syntax body)]
      [(_ "INTERNAL2" (((var1 ...) exp) . rest) body)
       (with-syntax ([(tmp1 ...) (generate-temporaries (syntax (var1 ...)))])
	 (syntax
	  (##sys#call-with-values 
	   (lambda () exp)
	   (lambda (tmp1 ...)
	     (set! var1 tmp1) ...
	     (letrec-values "INTERNAL2" rest body) ) ) ) ) ]
      [(_ bindings body1 ...)
       (syntax
	(letrec-values "INTERNAL1" bindings () (let () body1 ...)) ) ] ) ) )

(define-syntax record-case
  (lambda (x)
    (syntax-case x (else)
      [(_ "INTERNAL" rec head size ())
       (syntax (##core#undefined)) ]
      [(_ "INTERNAL" rec head size ((else body1 ...)))
       (syntax (begin body1 ...)) ]
      [(_ "INTERNAL" rec head size (((name slot1 ...) body1 ...) . rest))
       (let ([slotlist (syntax-object->datum (syntax (slot1 ...)))])
	 (with-syntax ([len (fx+ 1 (length slotlist))]
		       [(index1 ...)
			(let loop ([i 1] [slist slotlist])
			  (if (null? slist)
			      '()
			      (cons i (loop (fx+ i 1) (cdr slist))) ) ) ] )
	   (syntax
	    (if (and (eq? head 'name) (fx>= size len))
		(let ((slot1 (##sys#slot rec index1)) ...)
		  body1 ...)
		(record-case "INTERNAL" rec head size rest) ) ) ) ) ]
      [(_ val clause1 ...)
       (syntax
	(let* ((rec val)
	       (head (and (##sys#generic-structure? rec) (##sys#slot rec 0)))
	       (size (and head (##sys#size rec))) )
	  (record-case "INTERNAL" rec head size (clause1 ...)) ) ) ] ) ) )

(define-syntax :optional
  (syntax-rules ()
    [(_ rest default)
     (let ((tmp rest))
       (cond ((null? tmp) default)
	     ((null? (cdr tmp)) (car tmp))
	     (else (##sys#error (##core#immutable '"too many optional arguments") tmp)) ) ) ] ) )

(define-syntax let-optionals*
  (syntax-rules ()
    [(_ rest () body ...) (let () body ...)]
    [(_ rest ((var default) . more) body ...)
     (let* ((tmp rest)
	    (var (if (null? tmp) default (car tmp)))
	    (rest2 (if (null? tmp) '() (cdr tmp))) )
       (let-optionals* rest2 more body ...) ) ]
    [(_ rest (var) body ...) (let ((var rest)) body ...)] ) )

(define-syntax define-inline
  (syntax-rules ()
    [(_ (name . vars) . body) (##core#define-inline 'name (lambda vars . body))]
    [(_ name val) (##core#define-inline 'name val)] ) )

(define-syntax define-constant
  (syntax-rules ()
    [(_ (name . vars) . body) (##core#define-constant 'name (lambda vars . body))]
    [(_ name val) (##core#define-constant 'name val)] ) )

(define-syntax define-integrable
  (syntax-rules ()
    [(_ rest ...) (define-inline rest ...)] ) )

(define-syntax foreign-lambda
  (syntax-rules () [(_ rest ...) (##core#foreign-lambda (quote rest) ...)]) )

(define-syntax foreign-callback-lambda
  (syntax-rules () [(_ rest ...) (##core#foreign-callback-lambda (quote rest) ...)]) )

(define-syntax foreign-lambda*
  (syntax-rules () [(_ rest ...) (##core#foreign-lambda* (quote rest) ...)]) )

(define-syntax foreign-callback-lambda*
  (syntax-rules () [(_ rest ...) (##core#foreign-callback-lambda* (quote rest) ...)]) )

(define-syntax define-foreign-type
  (syntax-rules ()
    [(_ x y) (##core#define-foreign-type (quote x) (quote y))]
    [(_ x y p q) (##core#define-foreign-type (quote x) (quote y) p q)] ) )

(define-syntax define-foreign-variable
  (syntax-rules () [(_ rest ...) (##core#define-foreign-variable (quote rest) ...)]) )

(define-syntax define-foreign-parameter
  (syntax-rules () [(_ rest ...) (##core#define-foreign-parameter (quote rest) ...)]) )

(define-syntax foreign-callback-wrapper
  (syntax-rules () 
    [(_ rtype quals str atypes proc)
     (##core#foreign-callback-wrapper (quote str) (quote quals) (quote rtype) (quote atypes) proc) ] ) )

(define-syntax define-external
  (syntax-rules ()
    [(_ qualifiers (name (argtypes argvars) ...) rtype body ...)
     (define name
       (##core#foreign-callback-wrapper
	(quote name)
	(quote qualifiers)
	(quote rtype)
	(quote (argtypes ...))
	(lambda (argvars ...) body ...) ) ) ]
    [(_ (name (argtypes argvars) ...) rtype body ...)
     (define name
       (##core#foreign-callback-wrapper
	(quote name)
	(quote "")
	(quote rtype)
	(quote (argtypes ...))
	(lambda (argvars ...) body ...) ) ) ]
    [(_ name type)
     (begin
       (##core#define-foreign-variable (quote name) (quote type))
       (##core#define-external-variable (quote name) (quote type)) ) ]
    [(_ name type init)
     (begin
       (##core#define-foreign-variable (quote name) (quote type))
       (##core#define-external-variable (quote name) (quote type))
       (set! name init) ) ] ) )

(define-syntax critical-section
  (syntax-rules () 
    [(_ body ...)
     (##sys#dynamic-wind
	 ##sys#disable-interrupts
	 (lambda () body ...)
	 ##sys#enable-interrupts) ] ) )

(define-syntax nth-value
  (syntax-rules ()
    [(_ i exp)
     (##sys#call-with-values
      (lambda () exp)
      (lambda lst (list-ref lst i)) ) ] ) )

(define-syntax define-module
  (syntax-rules ()
    [(_ name clauses ...) (##core#define-module 'name '(clauses ...))] ) )

(define-syntax define-library-interface
  (syntax-rules ()
    [(_ name exports ...) (##core#define-module 'name '((unit name) (export exports ...)))] ) )

(define-syntax define-library-implementation
  (syntax-rules ()
    [(_ name clauses ...)
     (##core#define-module 'name '((unit name) clauses ...)) ] ) )

(define-syntax define-record-printer
  (syntax-rules ()
    [(_ (name var1 var2) body ...)
     (##sys#register-record-printer 'name (lambda (var1 var2) body ...)) ]
    [(_ name proc) (##sys#register-record-printer 'name proc)] ) )

(define-syntax handle-exceptions
  (syntax-rules ()
    ((_ var handle-body e1 e2 ...)      
     ((call-with-current-continuation
       (lambda (k)
	 (with-exception-handler 
	  (lambda (var) (k (lambda () handle-body)))
	  (lambda ()
	    (##sys#call-with-values 
	     (lambda () e1 e2 ...)
	     (lambda args (k (lambda () (##sys#apply ##sys#values args)))))))))))))

(define-syntax define-class
  (syntax-rules ()
    [(_ name () slots)
     (define-class name (<object>) slots) ]
    [(_ name supers slots)
     (define-class name supers slots <class>) ]
    [(_ name () slots meta)
     (define-class name (<object>) slots meta) ]
    [(_ cname (supers ...) (slots ...) meta)
     (set! cname (make meta 'name 'cname 'direct-supers (list supers ...) 'direct-slots (list 'slots ...))) ] ) )

(define-syntax define-generic
  (syntax-rules () 
    [(_ name) (set! name (make-generic 'name))] ) )

(define-syntax define-method
  (lambda (x)
    (syntax-case x ()
      [(_ "gather" name ((var spec) . rest) body (specs ...) (vars ...))
       (syntax (define-method "gather" name rest body (specs ... spec) (vars ... var))) ]
      [(_ "gather" name rest body (specs ...) (vars ...))
       (with-syntax ([call-next-method (implicit-identifier (syntax name) 'call-next-method)])
	 (syntax
	  (##tinyclos#add-global-method
	   'name
	   (list specs ...) (##core#named-lambda name (call-next-method vars ... . rest) . body)) ) ) ]
      [(_ (name . llist) body ...) 
       (syntax (define-method "gather" name llist (body ...) () ())) ] ) ) )


;;; For SRFI-13:

;;; This macro parses optional start/end arguments from arg lists, defaulting
;;; them to 0/(string-length s), and checks them for correctness.

(define-syntax let-string-start+end
  (syntax-rules ()
    ((let-string-start+end (start end) proc s-exp args-exp body ...)
     (receive (start end) (string-parse-final-start+end proc s-exp args-exp)
       body ...))
    ((let-string-start+end (start end rest) proc s-exp args-exp body ...)
     (receive (rest start end) (string-parse-start+end proc s-exp args-exp)
       body ...))))


;;; Interpreter stuff:

(cond-expand [(not compiling)

(define-syntax trace
  (lambda (x)
    (syntax-case x ()
      [(_) (syntax (##core#undefined))]
      [(_ name ...)
       (with-syntax ([(tmp ...) (generate-temporaries (syntax (name ...)))])
	 (syntax
	  (let ([tmp name] ...)
	    (##sys#hash-table-set! ##sys#traced-procedures 'name tmp) ...
	    (set! name (lambda args
			 (##sys#traced-procedure-entry 'name args)
			 (##sys#call-with-values (lambda () (##sys#apply tmp args))
			   (lambda results
			     (##sys#traced-procedure-exit 'name results)
			     (##sys#apply ##sys#values results) ) ) ) )
	    ...) ) ) ] ) ) )

(define-syntax untrace
  (lambda (x)
    (syntax-case x ()
      [(_) (syntax (##core#undefined))]
      [(_ name ...)
       (with-syntax ([(tmp ...) (generate-temporaries (syntax (name ...)))])
	 (syntax
	  (let ([tmp (##sys#hash-table-ref ##sys#traced-procedures 'name)] ...)
	    (if tmp (set! name tmp)) ...
	    (##sys#hash-table-set! ##sys#traced-procedures 'name #f) ...) ) ) ] ) ) )

(define-syntax autoload
  (syntax-rules ()
    [(_ filename (symbol ...))
     (begin
       (define (symbol . args)
	 (load filename)
	 (##sys#apply symbol args) )
       ...) ] ) )

] [else] )
