#include "cp_types.h"
#include "cp_proto.h"
#include "cp_post.h"

/* Drawing flag scheme: flags indicate drawing instructions
   for objects -- circles/lines/faces. Bits set as follows:
      1  ->  draw object?
      2  ->  fill? (4 and 16 imply 2, also)
      4  ->  off = foreground, on = background 
             (applies to interior, not border, overriden by bit 16)
      8  ->  border color? (default=foreground, else recorded color)
      16 ->  interior color? (default set by bit 4, on --> recorded color)
      32 ->  display label?

Eg.  flag=3: filled object, in foreground
     flag=9: open object, border in (recorded) color
             (for 'edge', this gives colored edge)
     flag=19: filled in color, border in foreground
     flag=27: filled, border and interior in color
     flag=15: filled with background, border in color
     flag=32: label only (face or circle)

Normally, flag for each type of object; often passed on to
subroutine, so may need color code with it:
Eg. (cflag, ccol, ecol) for circle flag, int color, border color.
*/

int dum_int;

int post_any_circle(FILE *fp,struct p_data *p,int vert,
		    int cflag,int ccol,int col)
/* posts circle in approp geometry with edge/interior colors. */
{
  int front;
  complex ctr;
  double r;
  
  if (!p->packK_ptr[vert].plot_flag) return 0;
  ctr=p->packR_ptr[vert].center;
  r=p->packR_ptr[vert].rad;
  if (p->hes>0)
    {
      if (cflag & 1) post_s_cir(fp,r,ss_view(p->screen,ctr,1,&front),
	  cflag,ccol,col);
      if (cflag & 32 && front) post_cir_label(fp,p,vert);
      return 1;
    }
  if (p->hes <0) h_to_e_data(ctr,r,&ctr,&r);
  if (!cir_ck(ctr,r,p->screen->box)) return 0;
  if (cflag & 1) post_cir(fp,r,ctr,cflag,ccol,col);
  if (cflag & 32) post_cir_label(fp,p,vert);
  return 1;
} /* post_any_circle */

int post_any_circle_data(FILE *fp,struct s_data *q,int hes,double r,
			 complex ctr,int cflag,int ccol,int col,int label)
/* like post_any_circle, except data r and ctr given directly, allow for
label. */
{
  double stretch;

  if (hes <0) 
    {
      h_to_e_data(ctr,r,&ctr,&r);
      post_cir(fp,r,ctr,cflag,ccol,col);
      if (cflag & 32)
	{
	  if (r<0) /* horocycle */
	    {
	      stretch=1.0+
		(q->box.rx-q->box.lx)*15/
		(q->pix_box.rx-q->pix_box.lx); 
	      /* shift label to miss unit circle */
	      ctr.re=ctr.re*stretch;
	      ctr.im=ctr.im*stretch;
	    }
	  post_num(fp,ctr,label,q->box);
	  return 1;
	}
    }
  if (hes > 0) 
      post_s_cir(fp,r,ss_view(q,ctr,1,&dum_int),cflag,ccol,col);
  else post_cir(fp,r,ctr,cflag,ccol,col);
  if (cflag & 32) post_num(fp,ctr,label,q->box);
  return 1;
} /* post_any_circle_data */

int post_cir(FILE *fp,double rad,complex ctr,
	     int cflag,int ccol,int col)
/* store postscript eucl circle. */
{
  if (ccol==FG_COLOR && (!(cflag & 2) || col==FG_COLOR))
    {
      if (cflag & 2)
	fprintf(fp,"%f     %f %f %f d\n",
	  1.0-graylevel,ctr.re,ctr.im,rad); /* filled disc */
      else 
	fprintf(fp,"%f %f %f c\n",
	  ctr.re,ctr.im,rad);	/* circle */
      return 1;
    }
  fprintf(fp,"gs ");
  if (ccol!=FG_COLOR) /* edge colors? */
    fprintf(fp,"%f %f %f srgb  ",
      (double)red[ccol]/255,(double)green[ccol]/255,(double)blue[ccol]/255);
  if (cflag & 2) 
    {
      if (col==BG_COLOR)
	fprintf(fp,"gs 1 1 1 %f %f %f cd gr\n",ctr.re,ctr.im,rad);
      else if (col!=FG_COLOR)
	fprintf(fp,"gs %f %f %f %f %f %f cd gr\n",
          (double)red[col]/255,(double)green[col]/255,(double)blue[col]/255,
          ctr.re,ctr.im,rad);
      else 
	fprintf(fp,"gs %f     %f %f %f d gr\n",
	  1.0-graylevel,ctr.re,ctr.im,rad);
    }
  fprintf(fp,"%f %f %f c gr\n",ctr.re,ctr.im,rad);
  return 1;
} /* post_cir */

int post_any_dot(FILE *fp,struct p_data *p,int v,double diam,
		 int cflag,int shape_flag)
/* post shape at center of circle. diam gives size, cf is color 
flag (shape set in post_shape command), shape is just dot 
(more shapes later). */
{
  double rad;
  complex ctr;

  if (!p->packK_ptr[v].plot_flag) return 0;
  if (p->hes>0) /* sphere */
    {
      ctr=ss_view(p->screen,p->packR_ptr[v].center,1,&dum_int);
      if (cos(ctr.re)<=0) return 0; /* not on front */
      ctr.re=sin(ctr.im)*sin(ctr.re);
      ctr.im=cos(ctr.im);
    }
  else if (p->hes < 0) 
    h_to_e_data(p->packR_ptr[v].center,
            p->packR_ptr[v].rad,&ctr,&rad);
  else ctr=p->packR_ptr[v].center;
  if (!cir_ck(ctr,diam,p->screen->box)) return 0; /* not on screen */
  if (cflag)
    fprintf(fp,"%f %f %f   %f %f %f mark\n",
	(double)(red[p->packK_ptr[v].color])/255,
	(double)(green[p->packK_ptr[v].color])/255,
	(double)(blue[p->packK_ptr[v].color])/255,
	ctr.re,ctr.im,diam/2.0);
  else
    fprintf(fp,"0 0 0   %f %f %f mark\n",
	ctr.re,ctr.im,diam/2.0);
  return 1;
} /* post_any_dot */

