;;;; -*- Mode: Lisp; Syntax: ANSI-Common-Lisp; Base: 10 -*-
;;;; *************************************************************************
;;;; FILE IDENTIFICATION
;;;;
;;;; Name:          mop.lisp
;;;; Purpose:       Metaobject Protocol Interface
;;;; Programmer:    Kevin M. Rosenberg
;;;; Date Started:  Apr 2000
;;;;
;;;; This metaclass as functions to classes to allow display
;;;; in Text, HTML, and XML formats. This includes hyperlinking
;;;; capability and sub-objects.
;;;;
;;;; $Id: mop.lisp 8216 2003-11-16 02:57:17Z kevin $
;;;;
;;;; This file is Copyright (c) 2000-2003 by Kevin M. Rosenberg
;;;; *************************************************************************
 
(in-package #:hyperobject)

;; Main class

(defclass hyperobject-class (standard-class)
  ( ;; slots initialized in defclass
   (user-name :initarg :user-name :type string :initform nil
	      :accessor user-name
	      :documentation "User name for class")
   (user-name-plural :initarg :user-name-plural :type string :initform nil
		     :accessor user-name-plural
		     :documentation "Plural user name for class")
   (default-print-slots :initarg :default-print-slots :type list :initform nil
			:accessor default-print-slots
			:documentation "Defaults slots for a view")
   (description :initarg :description :initform nil
		:accessor description
		:documentation "Class description")
   (version :initarg :version :initform nil
	    :accessor version
	    :documentation "Version number for class")
   (closures :initarg :closures :initform nil
	     :accessor closures
	     :documentation "Closures to call on slot chnages")
   (sql-name :initarg :sql-name :accessor sql-name :initform nil
	     :documentation "SQL Name for this class")
   (guid :initarg :guid :accessor guid :initform nil
	 :documentation "ID string for this class")

   ;;; The remainder of these fields are calculated one time
   ;;; in finalize-inheritence.
   
   (subobjects :initform nil :accessor subobjects
	       :documentation
	       "List of fields that contain a list of subobjects objects.")
   (hyperlinks :type list :initform nil :accessor hyperlinks
	       :documentation "List of fields that have hyperlinks")
   (direct-rules :type list :initform nil :initarg :direct-rules
		 :accessor direct-rules
		 :documentation "List of rules to fire on slot changes.")
   (direct-views :type list :initform nil :initarg :direct-views
		 :accessor direct-views
		 :documentation "List of views")
   (class-id :type integer :initform nil
	     :accessor class-id
	     :documentation "Unique ID for the class")
   (default-view :initform nil :initarg :default-view :accessor default-view
		 :documentation "The default view for a class")
   (documementation :initform nil :initarg :documentation
		    :documentation "Documentation string for hyperclass.")
   
   ;; SQL commands
   (create-table-cmd :initform nil :reader create-table-cmd)
   (create-indices-cmds :initform nil :reader create-index-cmds)
   (drop-table-cmd :initform nil :reader drop-table-cmd)

   (views :type list :initform nil :initarg :views :accessor views
	  :documentation "List of views")
   (rules :type list :initform nil :initarg :rules :accessor rules
	  :documentation "List of rules")
   )
  (:documentation "Metaclass for Markup Language classes."))

(defclass subobject ()
  ((name-class :type symbol :initarg :name-class :reader name-class)
   (name-slot :type symbol :initarg :name-slot :reader name-slot)
   (subobj-class :type symbol :initarg :subobj-class :reader subobj-class)
   (lookup :type (or function symbol) :initarg :lookup :reader lookup)
   (lookup-keys :type list :initarg :lookup-keys :reader lookup-keys))
  (:documentation "subobject information")
  (:default-initargs :name-class nil :name-slot nil :subobj-class nil
		     :lookup nil :lookup-keys nil))


(defmethod print-object ((obj subobject) (s stream))
  (print-unreadable-object (obj s :type t)
    (format s "~S" (name-slot obj))))

(defclass hyperlink ()
  ((name :type symbol :initform nil :initarg :name :reader name)
   (lookup
    ;; The type specifier seems to break sbcl
    :type (or function symbol)
    ;;    :type t
    :initform nil :initarg :lookup :reader lookup)
   (link-parameters :type list :initform nil :initarg :link-parameters
		    :reader link-parameters)))

(defmethod print-object ((obj hyperlink) (s stream))
  (print-unreadable-object (obj s :type t :identity t)
    (format s "~S" (name obj))))

(defmethod validate-superclass ((class hyperobject-class) (superclass standard-class))
  t)

(defmethod finalize-inheritance :after ((cl hyperobject-class))
  ;; Work-around needed for OpenMCL
  #+ignore
  (unless (find-class (class-name cl))
    (setf (find-class (class-name cl)) cl))
  
  (init-hyperobject-class cl)
  )

(eval-when (:compile-toplevel :load-toplevel :execute)
  (when (>= (length (generic-function-lambda-list
		     (ensure-generic-function
		      'compute-effective-slot-definition)))
	    3)
    (pushnew :ho-normal-cesd cl:*features*))
  
    (when (>= (length (generic-function-lambda-list
		       (ensure-generic-function
			'direct-slot-definition-class)))
	    3)
      (pushnew :ho-normal-dsdc cl:*features*))
    
    (when (>= (length (generic-function-lambda-list
		       (ensure-generic-function
			'effective-slot-definition-class)))
	      3)
      (pushnew :ho-normal-esdc cl:*features*)))

;; Slot definitions
(defmethod direct-slot-definition-class ((cl hyperobject-class)
					 #+ho-normal-dsdc &rest iargs)
  (find-class 'hyperobject-dsd))

(defmethod effective-slot-definition-class ((cl hyperobject-class) 
					    #+ho-normal-esdc &rest iargs)
  (find-class 'hyperobject-esd))


;;; Slot definitions

(eval-when (:compile-toplevel :load-toplevel :execute)
  (defmacro process-class-option (slot-name &optional required)
    #+lispworks
    `(defmethod clos:process-a-class-option ((class hyperobject-class)
					     (name (eql ,slot-name))
					     value)
      (when (and ,required (null value))
	(error "hyperobject class slot ~A must have a value" name))
      (list name `',value))
    #+(or allegro sbcl cmu scl openmcl)
    (declare (ignore slot-name required))
    )

  (defmacro process-slot-option (slot-name)
    #+lispworks
    `(defmethod clos:process-a-slot-option ((class hyperobject-class)
					    (option (eql ,slot-name))
					    value
					    already-processed-options
					    slot)
      (list* option `',value already-processed-options))
    #-lispworks
    (declare (ignore slot-name))
    )
  
  (dolist (option *class-options*)
    (eval `(process-class-option ,option)))
  (dolist (option *slot-options*)
    (eval `(process-slot-option ,option)))

  (eval
   `(defclass hyperobject-dsd (standard-direct-slot-definition)
     (,@(mapcar #'(lambda (x)
		    `(,(intern (symbol-name x))
		      :initform nil))
		*slot-options-no-initarg*)
      ,@(mapcar #'(lambda (x)
		    `(,(intern (symbol-name x))
		      :initarg
		      ,(intern (symbol-name x) (symbol-name :keyword))
		      :initform nil
		      :accessor
		      ,(intern (concatenate 'string
					    (symbol-name :dsd-)
					    (symbol-name x)))))
		*slot-options*))))
  (eval
   `(defclass hyperobject-esd (standard-effective-slot-definition)
     (,@(mapcar #'(lambda (x)
		    `(,(intern (symbol-name x))
		      :initarg
		      ,(intern (symbol-name x) (symbol-name :keyword))
		      :initform nil
		      :accessor
		      ,(intern (concatenate 'string
					    (symbol-name :esd-)
					    (symbol-name x)))))
		(append *slot-options* *slot-options-no-initarg*)))))
  ) ;; eval-when

(defun intern-in-keyword (obj)
  (cond
    ((null obj)
     nil)
    ((eq t obj)
     t)
    ((atom obj)
     (intern (symbol-name obj) (find-package 'keyword)))
    ((consp obj)
     (cons (intern-in-keyword (car obj) ) (intern-in-keyword (cdr obj))))
    (t
     obj)))

(defun canonicalize-value-type (vt)
  (typecase vt
    (atom
     (ensure-keyword vt))
    (cons
     (list (ensure-keyword (car vt)) (cadr vt)))
    (t
     t)))

(defmethod compute-effective-slot-definition :around ((cl hyperobject-class)
						      #+ho-normal-cesd name
						      dsds)
  (declare (ignore #+ho-normal-cesd name))
  (let ((esd (call-next-method)))
    (if (typep esd 'hyperobject-esd)
	(compute-hyperobject-esd esd dsds)
	esd)))

(defun compute-hyperobject-esd (esd dsds)
  (let* ((dsd (car dsds))
	 (value-type (canonicalize-value-type (slot-value dsd 'value-type))))
    (multiple-value-bind (sql-type sql-length) 
	(value-type-to-sql-type value-type)
      (setf (esd-sql-type esd) sql-type)
      (setf (esd-sql-length esd) sql-length))
    (setf (slot-value esd 'type) (value-type-to-lisp-type value-type))
    (setf (esd-value-type esd) value-type)
    (setf (esd-user-name esd)
	  (aif (dsd-user-name dsd)
	       it
	       (string-downcase (symbol-name (slot-definition-name dsd)))))
    (setf (esd-sql-name esd)
	  (aif (dsd-sql-name dsd)
	       it
	       (lisp-name-to-sql-name (slot-definition-name dsd))))
    (setf (esd-sql-name esd)
	  (aif (dsd-sql-name dsd)
	       it
	       (lisp-name-to-sql-name (slot-definition-name dsd))))
    (dolist (name '(print-formatter subobject hyperlink hyperlink-parameters
		    description value-constraint indexed null-allowed
		    unique short-description void-text read-only-groups
		    hidden-groups unit disable-predicate view-type
		    list-of-values stored))
      (setf (slot-value esd name) (slot-value dsd name)))
    esd))

(defun lisp-name-to-sql-name (lisp)
  "Convert a lisp name (atom or list, string or symbol) into a canonical
SQL name"
  (unless (stringp lisp)
    (setq lisp
	  (typecase lisp
	    (symbol (symbol-name lisp))
	    (t (write-to-string lisp)))))
  (do* ((len (length lisp))
	(sql (make-string len))
	(i 0 (1+ i)))
      ((= i len) (string-upcase sql))
    (declare (fixnum i)
	     (simple-string sql))
    (setf (schar sql i)
	  (let ((c (char lisp i)))
	    (case c
	      ((#\- #\$ #\+ #\#) #\_)
	      (otherwise c))))))

#+ho-normal-cesd
(setq cl:*features* (delete :ho-normal-cesd cl:*features*))
#+ho-normal-dsdc
(setq cl:*features* (delete :ho-normal-dsdc cl:*features*))
#+ho-normal-esdc
(setq cl:*features* (delete :ho-normal-esdc cl:*features*))

(defun lisp-type-is-a-string (type)
  (or (eq type 'string)
      (and (listp type) (some #'(lambda (x) (eq x 'string)) type))))

(defun base-value-type (value-type)
  (if (atom value-type)
      value-type
    (car value-type)))

(defun value-type-to-lisp-type (value-type)
  (case (base-value-type value-type)
    ((:string :cdata :varchar :char)
     '(or null string))
    (:datetime
     '(or null integer))
    (:character
     '(or null character))
    (:fixnum
     '(or null fixnum))
    (:boolean
     '(or null boolean))
    ((:integer :long-integer)
     '(or null integer))
    ((:float :single-float)
     '(or null single-float))
    (:double-float
     '(or null double-float))
    (otherwise
     t)))

(defun value-type-to-sql-type (value-type)
  "Return two values, the sql type and field length."
  (let ((type (base-value-type value-type))
	(length (when (consp value-type)
		  (cadr value-type))))
    (values
     (case type
       ((:char :character)
	:char)
       (:varchar
	:varchar)
       ((:fixnum :integer)
	:integer)
       (:long-integer
	:long-integer)
       (:boolean
	:boolean)
       ((:float :single-float)
	:single-float)
       (:double-float
	:double-float)
       (:datetime
        :long-integer)
       (otherwise
	:text))
     length)))

;;;; Class initialization function

;; One entry for each class with lazy readers defined.  The value is a plist mapping
;; slot-name to a lazy reader, each of which is a list of a function and slot-names.
(defvar *lazy-readers* (make-hash-table))

(defmethod slot-unbound ((class hyperobject-class) instance slot-name)
  (let ((lazy-reader
	 (loop for super in (class-precedence-list class)
	       as lazy-reader = (getf (gethash super *lazy-readers*) slot-name)
	       when lazy-reader return it)))
    (if lazy-reader
	(setf (slot-value instance slot-name)
	      (if (atom lazy-reader)
		  (make-instance lazy-reader)
		  (apply (car lazy-reader)
			 (loop for arg-slot-name in (cdr lazy-reader)
			       collect (slot-value instance arg-slot-name)))))
	;; No lazy reader -- defer to regular slot-unbound handling.
	(call-next-method))))

;; The reader is a function and the reader-keys are slot names.  The slot is lazily set to
;; the result of applying the function to the slot-values of those slots, and that value
;; is also returned.
(defun ensure-lazy-reader (cl class-name slot-name subobj-class reader 
			   &rest reader-keys)
  (setf (getf (gethash cl *lazy-readers*) slot-name)
    (aif subobj-class
	 it
	 (list* reader (copy-list reader-keys)))))

(defun remove-lazy-reader (class-name slot-name)
  (setf (getf (gethash (find-class class-name) *lazy-readers*) slot-name)
    nil))


(defun finalize-subobjects (cl)
  "Process class subobjects slot"
  (setf (subobjects cl)
	(let ((subobjects '()))
	  (dolist (slot (class-slots cl))
	    (let-when
	     (subobj-def (esd-subobject slot))
	     (let ((subobject
		    (make-instance 'subobject
				   :name-class (class-name cl)
				   :name-slot (slot-definition-name slot)
				   :subobj-class (when (atom subobj-def)
						   subobj-def)
				   :lookup (when (listp subobj-def)
					     (car subobj-def))
				   :lookup-keys (when (listp subobj-def)
						  (cdr subobj-def)))))
	       (unless (eq (lookup subobject) t)
		 (apply #'ensure-lazy-reader
			cl
			(name-class subobject) (name-slot subobject)
			(subobj-class subobject)
			(lookup subobject) (lookup-keys subobject))
		 (push subobject subobjects)))))
	  ;; sbcl/cmu reverse class-slots compared to the defclass form
	  ;; so re-reverse on cmu/sbcl
	  #+(or cmu sbcl) subobjects
	  #-(or cmu sbcl) (nreverse subobjects)
	  )))

(defun finalize-class-slots (cl)
  "Make sure all class slots have an expected value"
  (unless (user-name cl)
    (setf (user-name cl) (format nil "~:(~A~)" (class-name cl))))
  
  (setf (user-name-plural cl)
	(if (and (consp (user-name cl)) (cadr (user-name cl)))
	    (cadr (user-name cl))
	    (format nil "~A~P" (if (consp (user-name cl))
				   (car (user-name cl))
				   (user-name cl))
		    2)))

  (dolist (name '(user-name description version guid sql-name))
    (awhen (slot-value cl name)
	   (setf (slot-value cl name)
	     (if (listp it)
		 (car it)
	       it))))
  
  (unless (sql-name cl)
    (setf (sql-name cl) (lisp-name-to-sql-name (class-name cl))))
  )

(defun finalize-documentation (cl)
  "Calculate class documentation slot"
  (let ((*print-circle* nil))
    (setf (documentation cl 'type)
	  (format nil "Hyperobject~A~A~A~A"
		  (aif (user-name cl)
		       (format nil ": ~A" it ""))
		  (aif (description cl)
		       (format nil "~%Class description: ~A" it) "")
		  (aif (subobjects cl)
		       (format nil "~%Subobjects:~{ ~A~}" (mapcar #'name-slot it)) "")
		  (aif (default-print-slots cl)
		       (format nil "~%Default print slots:~{ ~A~}" it) "")
		  ))))

(defun finalize-hyperlinks (cl)
  (let ((hyperlinks '()))
    (dolist (esd (class-slots cl))
      (awhen (slot-value esd 'hyperlink)
	     (push
	      (make-instance 'hyperlink
			     :name (slot-definition-name esd)
			     :lookup it
			     :link-parameters (slot-value esd 'hyperlink-parameters))
	      hyperlinks)))
    ;; cmu/sbcl reverse class-slots compared to the defclass form
    ;; hyperlinks is already reversed from the dolist/push loop, so re-reverse on sbcl/cmu
    #-(or cmu sbcl) (setq hyperlinks (nreverse hyperlinks))
    (setf (slot-value cl 'hyperlinks) hyperlinks)))

(defun init-hyperobject-class (cl)
  "Initialize a hyperobject class. Calculates all class slots"
  (finalize-subobjects cl)
  (finalize-views cl)
  (finalize-hyperlinks cl)
  (finalize-sql cl)
  (finalize-rules cl)
  (finalize-class-slots cl)
  (finalize-documentation cl))


;;;; *************************************************************************
;;;;  Metaclass Slot Accessors
;;;; *************************************************************************

(defun find-slot-by-name (cl name)
  (find name (class-slots cl) :key #'slot-definition-name))

(defun hyperobject-class-user-name (obj)
  (user-name (class-of obj)))

(defun hyperobject-class-user-name-plural (obj)
  (user-name-plural (class-of obj)))

(defun hyperobject-class-subobjects (obj)
  (subobjects (class-of obj)))

(defun hyperobject-class-hyperlinks (obj)
  (hyperlinks (class-of obj)))

(defun hyperobject-class-slots (obj)
  ;; cmucl/sbcl reverse class-slots
  #+(or cmu sbcl) (reverse (class-slots (class-of obj)))
  #-(or cmu sbcl) (class-slots (class-of obj)))

(defun all-subobjects (obj)
  "Returns a list of all subobjects in an object"
  (let ((so-list '()))
    (dolist (subobj-obj (subobjects (class-of obj)) (nreverse so-list))
      (dolist (so (funcall (name-slot subobj-obj) obj))
        (push so so-list)))))
