;;; -*- Syntax: Ansi-Common-Lisp; Base: 10; Mode: lisp; Package: imho -*-
;;; $Id: application.lisp,v 1.162 2002/03/29 21:20:07 craig Exp $
;;;
;;; Copyright (c) 1999, 2000, 2001 onShore Development, Inc.
;;;
;;; See the file 'COPYING' in this directory for terms.
;;;
;;; Entry points, application and session machinery for the request
;;; handler.

(in-package :imho)

(defun handle-error (application condition element stream)
  "Return an HMTL document containing a formatted lisp error."
  (log-application-error application *active-session*
                         (format nil "request: ~a element: ~a" *active-request* element)
                         condition)
  (html-stream
   stream
   (:html
    (:head
     (:title
      (:princ "Application Error")))
    (:body
     (:pre
      (let ((msg (format nil "~s: An error occurred during processing:~%~A"
                         application condition)))
        (write-string (html-escape msg) stream)))))))

(defmethod log-application-error ((self t) (session http-session) context (error t))
  (declare (ignore context)))


(defun parse-uri (request uristring)
  (let* ((path (odcl::split uristring #\/))
         (length (length path)))
    (flet ((parse-method (methodstring)
             (let ((parts (odcl::split methodstring #\?)))
               (setf (request-args request) (apply #'concatenate 'string (cdr parts)))
               (setf (request-method request) (first parts))))
           (session (id)
             (application-session (request-application request) id)))
      (destructuring-bind (mount-point app-id &optional three four five six)
          path
        (setf (request-mount-point request) mount-point)
        (setf (request-application request) (find-application app-id))
        (cond ((and (session three) (= length 6))
               ;; /mount-point/app/session-id/caller/callee/method[?args]*
               (setf (request-session request) (session three)
                     (request-caller request) four
                     (request-callee request) five)
               (parse-method six))
              ((= length 6)
               (parse-method six))
              
              ((and (nth 2 path) (< 0 (length three)))
               ;; at least find a method
               (parse-method three))
              ;; NOT (YET) REACHED
              ((= length 5)
               ;; /mount-point/app/caller/callee/method[?args]*
               ;; session from cookie
               (setf (request-session request) (session (get-cookie :session-id))
                     (request-caller request) three
                     (request-callee request) four)
               (parse-method five))
              ((= length 3)
               ;; /mount-point/app/method[?args]*
               ;; session from cookie
               (setf (request-session request) (session (get-cookie :session-id)))
               (parse-method three))
              )))))

(defun print-session-contents (session)
  (maphash (lambda (k v)
	     (cmsg "~A ==> ~S" k v))
	   (session-html-elements session)))

;; ------------------------------------------------------------
;; method: handle-request
;;
;; The main request/response loop processor.
;;
;; Once a request object has been created, and a running application
;; has been identified, we can process the request.
;;
;; 1. Check for a session
;;
;; 2. Identify the originating html-element
;;
;; 3. Extract values from it
;;
;; 4. Run a method, if one was requested, which will return the
;; response element, or nil, to indicate that the request element
;; should be rerendered.
;;
;; 5. Render the response.

(defun handler-base-url (request session)
  (concatenate 'string
	       "/" (request-mount-point request)
	       "/" (slot-value (request-application request) 'base-url)
	       "/" (session-id session) "/" ))

(defun session-root ()
  (concatenate 'string
	       "/" (request-mount-point *active-request*)
	       "/" (slot-value (request-application *active-request*) 'base-url)
	       "/" (session-id (request-session *active-request*)) "/-/-"))

(defun lookup-initial-method (application)
  (with-slots (initial-method)
    application
    (or (lookup-wm initial-method)
        (error 'method-not-found :method-name initial-method))))

(defun accept-languages (string)
  (let ((segments (split string #\,)))
    (remove-if #'null
               (mapcar (lambda (x)
                         (let ((pair (split (string-trim " " x) #\-)))
                           (case (length pair)
                             (1 (cons (intern (string-upcase (car pair)) :keyword)
                                      nil))
                             (2 (cons (intern (string-upcase (car pair)) :keyword)
                                      (intern (string-upcase (cadr pair)) :keyword))))))
                       segments))))

(defun pick-language (request)
  (let ((lang (cdr (assoc :accept-language (request-headers-in request)))))
    (if lang
        (accept-languages lang)
        '((:en . :US)))))

(defun handle-request (http-request)
  "The center of everything: take an HTTP-REQUEST, apply it to a
running APPLICATION, and render a resultant element."
  (let* ((*active-application* (request-application http-request))
         (*active-request*     http-request)
         (*lang*               (pick-language http-request))
         (*active-components*  (make-hash-table))
         (*active-session*     (with-lock-held (*session-timeout-lock*)
                                 (lookup-session http-request)))
         (element              (session-element *active-session*
                                                (request-callee http-request)))
         (*active-url*         (handler-base-url http-request *active-session*)))
    (when *log-requests*
      (log-event (format nil "Starting request on ~s for ~s"
                         (request-application http-request)
                         (request-method http-request))))
    (handler-case-if *production*
      (handler-case
          (progn
            (generate-response *active-application* element))
        (imho-session-error ()
          ;; session timed out, reinvoke handler
          (setf (request-session http-request) nil)
          (setf (request-method http-request) nil)
          (handle-request http-request)))
      (error (cond)
        (let ((errstream (make-string-output-stream)))
          (handle-error *active-application* cond element errstream)
          (setf (request-response-body http-request)
                (get-output-stream-string errstream))

          (funcall (request-response-callback http-request) http-request))))))

(defun write-body-tag (alist stream)
  (flet ((make-attr (attr)
           (concatenate 'string (symbol-name (car attr)) "=\"" (cdr attr) "\"")))
    (format stream "<BODY ~{~a~^ ~}>" (mapcar #'make-attr alist))))

(defmethod default-body-attrs ((app application))
  nil)

(defun html-prologue (request application &key (frameset nil))
  (let ((stream (make-string-output-stream))
	(style (application-style-sheet application)))
    (write-string *default-dtd* stream)
    (terpri stream)
    (write-string "<HTML>" stream)
    (terpri stream)
    (html-stream
     stream
     (:head
      (:title
       (:princ (request-doc-title request)))
      ;; write styles/layers into header
      (:comment "BEGIN: CSS Components")
      ((:style :type "text/css")
       (:comment
        (terpri stream)
        
        ;; instances
        (when (request-css-entries request)
          (write-string
           (get-output-stream-string (request-css-entries request))
           stream))
        
        (write-string "// " stream)))

      ;; link to wco stylee sheet -- this should be abstracted somehow ASAP
      (when (stringp style)
	(html-stream
	 stream
	 ((:link :rel "stylesheet" :type "text/css" :href style))))

      ;; write scripts into header
      (:comment
       (format stream "BEGIN: Script Components"))
      ((:script :type "text/javascript")
       (:comment
        (terpri stream)
        
        ;; instances
        (format stream "// Start of Instances~%~%")
        (format stream "function imho_init_instances() {}~%~%")
        (when (request-scripted-instances request)
          (format stream "imho_instances = new Array;~%")
          (format stream "IMHOActions = new Object;~%")
          (dolist (init (request-scripted-instances request))
            (format stream "~A~%" init))
          (format stream "~%// End of Instances~%~%"))
        
        (dformat :scripts "Scripted elements: ~s" (hashkeys *active-components*))

        ;; functions
        (maphash (lambda (k v)
                   (declare (ignore v))
                   (write-script application k stream))
                 *active-components*)
        
        (format stream "// "))))
     (if (not frameset)
         (write-body-tag (append (default-body-attrs application)
                                 (request-body-attrs request))
                         stream))
     (when (request-div-elements request)
       (write-string (get-output-stream-string (request-div-elements request)) stream))
     )
    (get-output-stream-string stream)))

(defmethod authorize ((element html-element))
  element)

(defmethod generate-response/process-url ((application application) request)
  (when-bind (method (request-method request))
    (let ((method-object (lookup-wm method)))
      (unless method-object
        (error 'method-not-found :method-name method))
      (values method-object
              (process-wm-args (wmethod-args method-object)
                               (list (request-args request)))))))

(defmethod generate-response/take-values ((application application) element)
  (take-values-from-request element *active-request*))

(defmethod generate-response/invoke ((application application) method &optional (arglist '(nil)))
  (when method
    (if (debugging :profile)
        #-cmu
        (apply (wmethod-body method) arglist)
        #+cmu
        (let (before after start end element)
          (without-scheduling
           (setq before (ext:get-bytes-consed))
           (setq start (get-utime))
           (setq element (apply (wmethod-body method) arglist))
           (setq end (get-utime))
           (setq after (ext:get-bytes-consed)))
          (cmsg "~12d ~6f INVOKE ~a => ~a" (- after before)
                (/ (- end start) 1000000.0)
                (wmethod-name method) (cadr arglist))
          element)
        (apply (wmethod-body method) arglist))))

(defmethod generate-response/authorize (application element)
  (declare (ignore application))
  (authorize element))

(defmethod generate-response/awake (application element)
  (declare (ignore application))
  (preawake element)
  (awake element)
  (setf (request-response-element *active-request*) element)
  (setf (slot-value *active-session* 'active-response) element))

(defmethod generate-response/render (application element stream)
  (declare (ignore application))
  (if (debugging :profile)
      #-cmu
      (render-html element stream)
      #+cmu 
      (let (before after start end)
        (without-scheduling
         (setq before (ext:get-bytes-consed))
         (setq start (get-utime))
         (render-html element stream)
         (setq end (get-utime))
         (setq after (ext:get-bytes-consed))
         (cmsg "~12d ~6f RENDER ~s" (- after before)
                 (/ (- end start) 1000000.0)
                 element)))
      (render-html element stream)))

(defmethod generate-response ((application application) element)
  (when element
    (generate-response/take-values application element))
  (multiple-value-bind (method args)
      (generate-response/process-url application *active-request*)

    (let ((response (or (generate-response/invoke application method
                                                  (cons element args))
                        element
                        (slot-value *active-session* 'active-response)

			(when-bind (element (slot-value application 'initial-element))
				   (apply #'session-instance (list element)))
                        (generate-response/invoke application (lookup-initial-method application)))))

      (setq response (or (when (typep response 'html-element)
                           (generate-response/authorize application (element-root response)))
                         element))

      (generate-response/awake application response)
      (let ((body nil)
            (mime-type (request-response-type *active-request*)))
        (if (string= mime-type "text/html")
            (generate-response/render application response
                                      (request-html-stream *active-request*))
            (without-element-comments
             (generate-response/render application response
                                       (request-html-stream *active-request*))))
        (cond ((or (and (> (length mime-type) 11)
                        (string-equal mime-type "application" :end1 11)) ; Fixed
                   (and (> (length mime-type) 5)
                        (string-equal mime-type "image" :end1 5))) ; Added
               (setq body (cons :data (get-output-stream-data
                                       (request-binary-stream *active-request*)))))
              ((string= mime-type "text/plain")
               (setq body (get-output-stream-string (request-html-stream
                                                     *active-request*))))
              ((typep response 'html-frameset)
               (setq body (concatenate 'string
                                       (html-prologue *active-request*
                                                      application :frameset t)
                                       (get-output-stream-string
                                        (request-html-stream *active-request*))
                                       "</HTML>"))
               (setf (request-response-length *active-request*) (length body)))
              (t
               (setq body (concatenate 'string
                                       (html-prologue *active-request* application)
                                       (get-output-stream-string
                                        (request-html-stream *active-request*))
                                       "</BODY></HTML>"))
               (when *postprocess-html*
                 (setq body (frob-html body)))
               (setf (request-response-length *active-request*) (length body))))
        (setf (request-response-body *active-request*) body)
        (funcall (request-response-callback *active-request*) *active-request*)))))

#+cmu
(defun frob-html (body)
  (let ((sgml-input (make-string-input-stream body))
        (sgml-output (make-string-output-stream))
        (sgml-errors (make-string-output-stream)))
    (ext:run-program "/usr/bin/sgmlnorm" nil
                     :wait t
                     :error sgml-errors
                     :input sgml-input
                     :output sgml-output)
    (cmsg "SGML errors: ~a" (get-output-stream-string sgml-errors))
    (get-output-stream-string sgml-output)))

;; ------------------------------------------------------------
;; Remove a renderer

(defmacro kill-renderer (html-element-class)
  `(remove-method (symbol-function 'render-html)
    (find-method (symbol-function 'render-html)
     ()
     (list (find-class ,html-element-class)
      (find-class 't)
      (find-class 't)))))


(defmacro defapplication (name &key (initial-method nil)
			  (session-class 'http-session)
			  (base-url nil)
			  (initial-element nil)
			  (doc-root "/")
			  (template-root nil)
			  (script-root nil)
			  (style-sheet nil))
  `(defclass ,name (application)
    ((base-url
      :initform ,(string-downcase (or base-url (symbol-name name))))
     (initial-method
      :initform ',(or initial-method (symbol-name name)))
     (initial-element
      :initform ',initial-element)
     (session-class
      :initform ',session-class)
     (doc-root
      :initform ,doc-root)
     (template-root
      :initform ,template-root)
     (script-root
      :initform ,script-root)
     (style-sheet
      :initform ,style-sheet))))

(defun init/browser-debugger (&optional (state :start))
  (declare (ignore state))
  (compile-file
   (logical-pathname "systems:imho;experimental;production-debug")
   :load t)
  (compile-file
   (logical-pathname "systems:imho;experimental;application")
   :load t)
  (compile-file
   (logical-pathname "systems:imho;experimental;cmucl-error-patch")
   :load t))

