;;; -*- Syntax: Ansi-Common-Lisp; Base: 10; Mode: lisp; Package: imho -*-
;;; $Id: html-elements.lisp,v 1.69 2001/11/12 20:07:08 jesse Exp $
;;;
;;; Copyright (c) 1999, 2000, 2001 onShore Development, Inc.
;;;
;;; See the file 'COPYING' in this directory for terms.
;;;
;;; Basic non-form elements

(in-package :imho)

(eval-when (:compile-toplevel :load-toplevel)

(defmacro render-child (element child stream)
  `(render-html (child-element ,element ,child) ,stream))
  
(defclass frame-targetting-mixin ()
  ((window
    :accessor window
    :initarg :window
    :initform "_self")))

(defun local-href (url)
  (concatenate 'string (application-doc-root *active-application*) url))

(declaim (inline local-href))

(defun write-spacer (stream &key (height 1) (width 1))
  (html-stream
   stream
   ((:img :src (local-href "images/spacer.gif") :width width :height height))))

;; ------------------------------------------------------------
;; html-element: html-page

(defclass html-page (html-element)
  ())

(defclass html-frameset (html-element)
  ((cols
    :initarg :cols
    :initform nil)
   (rows
    :initarg :rows
    :initform nil)
   (border
    :initarg :border
    :initform 0)
   (frames
    :initarg :frames)))

(defmethod preawake ((self html-frameset))
  (with-slots (cols rows frames)
    self
    (dolist (frame frames)
      (let ((child (child-element self frame))) 
        (when (element-value self)
          (setf (element-value child)
                (element-value self)))
        (preawake child)))))

(defmethod render-html ((element html-frameset) stream)
  (with-slots (cols rows frames border)
    element
    (if rows
        (html-stream
         stream
         ((:frameset :rows rows :border border)
          (dolist (frame frames)
            (render-child element frame stream))))
        (html-stream
         stream
         ((:frameset :cols cols :border border)
          (dolist (frame frames)
            (render-child element frame stream)))))))

(defclass html-frame (html-element)
  ((src
    :initarg :src)
   (name
    :initarg :name
    :initform "")
   (method
    :initarg :method)
   (scrolling
    :initarg :scrolling
    :initform :auto)))

(defmethod render-html ((element html-frame) stream)
  (with-slots (method name scrolling)
    element
    (let ((frame-url (build-element-url element :reference (dyn-refer-wm method)))
          (frame-element (apply-wm method)))
      (html-stream
       stream
       ((:frame :src frame-url :name name :scrolling (symbol-name scrolling))))
      (when (element-value element)
        (setf (element-value frame-element)
              (element-value element))))))
      

;; ------------------------------------------------------------
;; html-element: static-string

(defclass static-string (html-element)
  ((escaped-p
    :accessor escaped-p
    :initarg :escaped
    :initform nil))
  )

(defmethod render-html ((element static-string) stream)
  (let* ((value (element-value element))
	 (string (typecase value
		   (string	(if (equal value "") "&nbsp;" value))
		   (function	(funcall value))
		   (null "&nbsp;")
		   (t (princ-to-string value)))))
    
    (write-string (if (escaped-p element)
		      (html-escape string)
		    string) stream)))

(defclass sized-mixin ()
  ((width
    :initarg :width
    :initform 0)
   (height
    :initarg :height
    :initform 0)))

(defclass enabling-mixin ()
  ((enabled
    :initarg :enabled
    :accessor element-enabled
    :initform t)))


;; ------------------------------------------------------------
;; html-element: hyperlink
;;
;; A text hyperlink.

(defclass hyperlink (targeted enabling-mixin html-element)
  ())

;; ------------------------------------------------------------
;; framework-method: element-value
;;
;; Allow the 'value' slot of a hyperlink to be a zero-arity function
;; that returns the content for the link.

(defmethod element-value ((element hyperlink))
  (with-slots (value)
    element
    (typecase value
      (function (funcall value))
      (string value)
      (null "")
      (t value))))

;; By default, spaces inside of hyperlinks are converted to &nbsp;'s

(defmethod render-html ((self hyperlink) stream)
  (let ((link-text (string-replace #\Space "&nbsp;" (element-value self))))
    (if (element-enabled self)
        (html-stream
         stream
         ((:a :href (element-url self)
              :target (element-target-frame self))
          (:princ link-text)))
        (write-string link-text stream))))

(defclass tooltip-mixin ()
  ((tooltip
    :initarg :tooltip
    :initform nil)))

(defclass status-msg-mixin ()
  ((status-msg
    :initarg :status-msg
    :initform nil)))

(defclass image (targeted sized-mixin tooltip-mixin status-msg-mixin enabling-mixin html-element)
  ((source
    :initarg :source
    :initform nil)
   (alttext
    :initarg :alt
    :initform nil)
   (border
    :initarg :border
    :initform 0)))

(defmethod render-html ((element image) stream)
  (with-slots (alttext source width height enabled border tooltip element-external-name)
    element
    (let ((image-src (get-image-url source (if enabled :default :disabled))))
      (if (and width height)
          (html-stream
           stream
           ((:img :src image-src
                  :alt (if (and alttext (not tooltip)) alttext "")
                  :width width
                  :height height
                  :name element-external-name
                  :border 0)))
          (html-stream
           stream
           ((:img :src image-src
                  :alt (if (and alttext (not tooltip)) alttext "")
                  ;; :width (format nil "~d" width)
                  ;; :height (format nil "~d" height)
                  :name element-external-name
                  :border 0)))))))

;; ------------------------------------------------------------
;; image-button

(defclass image-button (image)
  ())

(defmethod render-html ((element image-button) stream)
  (with-slots (source width height enabled tooltip status-msg element-external-name)
    element
    (if (not enabled)
        (call-next-method)
        (let ((standard (get-image-url source :default))
              (highlit  (get-image-url source :highlit))
              (clicked  (get-image-url source :clicked))
              (tooltip-out "")
              (tooltip-over "")
	      (status-over "")
	      (status-out ""))
          (scripted-element-init "IMHOLoadImage" element-external-name standard highlit clicked "")
          (when tooltip
            (let ((divname (concatenate 'string element-external-name "DIV")))
              (setq tooltip-over (format nil "doToolTip(event, '~a'); "
                                         divname))
              (setq tooltip-out (format nil "clearToolTip('~a'); "
                                        divname))
              (write-layer divname
                           '(("position" . "absolute")
                             ("visibility" . "hidden")
                             ("margin-left" . "250 px")
                             ("margin-right" . "250 px")
                             ("margin-top" . "25")
                             ("z-index" . "1")
                             ("text-align" . "center")
                             ("font-family" . "arial, helvetica")
                             ("font-size" . "12pt")
                             ("border-width" . "0")
                             ("background-color" . "e4e4e4")
                             ("text-color" . "000033")))
              (with-div (divstream divname)
                (html-stream
                 divstream
                 ((:table :width "300" :height "28" :border 0)
                  (:tr
                   ((:td :align :center :class "flabel_2")
                    (write-string tooltip divstream))))))))
	  (when status-msg
	    (setq status-over (format nil "doStatusMsg('~a'); "
				      status-msg))
	    (setq status-out (format nil "clearStatusMsg(); ")))
          (let ((frame-args (element-frame-args element)))
            (if frame-args
                (html-stream
                 stream
                 ((:a :href "#"
                      :onclick
                      (format nil "window.open('~a',~a)"
                              (element-url element)
                              frame-args)
                      :target (element-target-frame element)
                      :onmouseover (format nil "~a; ~areturn CSIShow('~a',1);"
                                           status-over tooltip-over element-external-name)
                      :onmouseout (format nil "~a; ~areturn CSIShow('~a',0);"
                                          status-out tooltip-out element-external-name)
                      :onmousedown (format nil "CSIShow('~a',2); return CSButtonReturn()"
                                           element-external-name))
                  (call-next-method)))
                (html-stream
                 stream
                 ((:a :href (element-url element)
                      :target (element-target-frame element)
                      :onmouseover (format nil "~a; ~areturn CSIShow('~a',1);"
                                           status-over tooltip-over element-external-name)
                      :onmouseout (format nil "~a; ~areturn CSIShow('~a',0);"
                                          status-over tooltip-out element-external-name)
                      :onmousedown (format nil "CSIShow('~a',2); return CSButtonReturn()"
                                           element-external-name))
                  (call-next-method)))))))))

(defclass splash-button (image)
  ())

(defmethod render-html ((element splash-button) stream)
  (with-slots (source status-msg width height enabled element-external-name)
     element
     (let ((status-over "")
	   (status-out ""))
       (when status-msg
	 (setq status-over (format nil "doStatusMsg('~a'); "
				   status-msg))
	 (setq status-out (format nil "clearStatusMsg(); ")))
       (scripted-element-init "CSNullInit" element-external-name "splash.html" "/splash.html" 400 400)
       (html-stream
	stream
	((:a :href (element-url element)
	     :onmouseover status-over
	     :onmouseout status-out
	     :onclick (format nil "DOSplash(); return false;"
			      element-external-name))
	 (call-next-method))))))
  
(defclass link-text-field (html-element)
  ((cols
    :initarg :cols)))

(defmethod render-html ((self link-text-field) stream)
  (with-slots (element-external-name)
      self
      ;;     (scripted-element-init "CSNullInit" element-external-name "splash.html" "/splash.html" 400 400)
      (html-stream
       stream
       ((:a :href (element-url self)
	    :onclick (format nil "DOSplash(); return false;"
			     element-external-name))
	(:princ-safe (element-value self))))))

)
