;; -*- mode: lisp; package: net.html.generator -*-
;;
;; htmlgen.cl
;;
;; copyright (c) 1986-2000 Franz Inc, Berkeley, CA 
;;
;; This code is free software; you can redistribute it and/or
;; modify it under the terms of the version 2.1 of
;; the GNU Lesser General Public License as published by 
;; the Free Software Foundation, as clarified by the AllegroServe
;; prequel found in license-allegroserve.txt.
;;
;; This code is distributed in the hope that it will be useful,
;; but without any warranty; without even the implied warranty of
;; merchantability or fitness for a particular purpose.  See the GNU
;; Lesser General Public License for more details.
;;
;; Version 2.1 of the GNU Lesser General Public License is in the file 
;; license-lgpl.txt that was distributed with this file.
;; If it is not present, you can access it from
;; http://www.gnu.org/copyleft/lesser.txt (until superseded by a newer
;; version) or write to the Free Software Foundation, Inc., 59 Temple Place, 
;; Suite 330, Boston, MA  02111-1307  USA
;;

;;
;; $Id: htmlgen.lisp,v 1.14 2001/01/18 18:02:05 jesse Exp $

;; Description:
;;   html generator

;;- This code in this file obeys the Lisp Coding Standard found in
;;- http://www.franz.com/~jkf/coding_standards.html
;;-

(defpackage :net.html.generator
  (:use :common-lisp :excl)
  (:export #:html 
	   #:html-print
	   #:html-print-list
	   #:html-stream 
           #:print-template
	   #:*html-stream*
           #:phtml-internal
           #:parse-html
	   ;; should export with with-html-xxx things too I suppose
	   ))

(in-package :net.html.generator)

;; html generation

(defvar *html-stream*)                  ; all output sent here
(defvar *template-bits*)                ; all output sent here

(defstruct (html-process (:type list) (:constructor
				       make-html-process (key has-inverse
							      macro special
							      print)))
  key	; keyword naming this
  has-inverse	; t if the / form is used
  macro  ; the macro to define this
  special  ; if true then call this to process the keyword
  print    ; function used to handle this in html-print
  )


(defparameter *html-process-table* 
    (make-hash-table :test #'equal) ; #'eq is accurate but want to avoid rehashes
  )

(defmacro html (&rest forms)
  ;; just emit html to the current stream
  (process-html-forms forms))

(defmacro html-out-stream-check (stream)
  ;; ensure that a real stream is passed to this function
  `(let ((.str. ,stream))
     (if* (not (streamp .str.))
	then (error "html-stream must be passed a stream object, not ~s"
		    .str.))
     .str.))


(defmacro html-stream (stream &rest forms)
  ;; set output stream and emit html
  `(let ((*html-stream* (html-out-stream-check ,stream))) (html ,@forms)))

(defun print-template (forms)
  (let ((*html-stream* (make-string-output-stream))
        (*template-bits* nil))
    (html-print-list forms *html-stream*)
    (nreverse (push (list :string (get-output-stream-string *html-stream*))
                    *template-bits*))))

(defun process-html-forms (forms)
  (let (res)
    (flet ((do-ent (ent args argsp body)
	     ;; ent is an html-process object associated with the 
	     ;;	    html tag we're processing
	     ;; args is the list of values after the tag in the form
	     ;;     ((:tag &rest args) ....)
	     ;; argsp is true if this isn't a singleton tag  (i.e. it has
	     ;;     a body) .. (:tag ...) or ((:tag ...) ...)
	     ;; body is the body if any of the form
	     ;; 
	     (let (spec)
	       (if* (setq spec (html-process-special ent))
		  then ; do something different
		       (push (funcall spec ent args argsp body) res)
		elseif (null argsp)
		  then ; singleton tag, just do the set
		       (push `(,(html-process-macro ent) :set) res)
		       nil
		  else (if* (equal args '(:unset))
			  then ; ((:tag :unset)) is a special case.
			       ; that allows us to close off singleton tags
			       ; printed earlier.
			       (push `(,(html-process-macro ent) :unset) res)
			       nil
			  else ; some args
			       (push `(,(html-process-macro ent) ,args
								 ,(process-html-forms body))
				     res)
			       nil)))))

      (do* ((xforms forms (cdr xforms))
	    (form (car xforms) (car xforms)))
	  ((null xforms))
	
	(if* (atom form)
	   then (if* (keywordp form)
		   then (let ((ent (gethash form *html-process-table*)))
			  (if* (null ent)
			     then (error "unknown html keyword ~s"
					 form)
			     else (do-ent ent nil nil nil)))
		 elseif (stringp form)
		   then ; turn into a print of it
			(push `(write-string ,form *html-stream*) res)
		   else (push form res))
	   else (let ((first (car form)))
		  (if* (keywordp first)
		     then ; (:xxx . body) form
			  (let ((ent (gethash first
					    *html-process-table*)))
			    (if* (null ent)
			       then (error "unknown html keyword ~s"
					   form)
			       else (do-ent ent nil t (cdr form))))
		   elseif (and (consp first) (keywordp (car first)))
		     then ; ((:xxx args ) . body)
			  (let ((ent (gethash (car first)
					    *html-process-table*)))
			    (if* (null ent)
			       then (error "unknown html keyword ~s"
					   form)
			       else (do-ent ent (cdr first) t (cdr form))))
		     else (push form res))))))
    `(progn ,@(nreverse res))))


(defun html-atom-check (args open close body)
  (if* (and args (atom args))
     then (let ((ans (case args
		       (:set `(write-string  ,open *html-stream*))
		       (:unset `(write-string  ,close *html-stream*))
		       (t (error "illegal arg ~s to ~s" args open)))))
	    (if* (and ans body)
	       then (error "can't have a body form with this arg: ~s"
			   args)
	       else ans))))

(defun html-body-form (open close body)
  ;; used when args don't matter
  `(progn (write-string  ,open *html-stream*)
	  ,@body
	  (write-string  ,close *html-stream*)))


(defun html-body-key-form (string-code has-inv args body)
  ;; do what's needed to handle given keywords in the args
  ;; then do the body
  (if* (and args (atom args))
     then ; single arg 
	  (return-from html-body-key-form
	    (case args
	      (:set `(write-string  ,(format nil "<~a>" string-code)
				    *html-stream*))
	      (:unset (if* has-inv
			 then `(write-string  ,(format nil "</~a>" string-code)
					      *html-stream*)))
	      (t (error "illegal arg ~s to ~s" args string-code)))))
  
  (if* (not (evenp (length args)))
     then (warn "arg list ~s isn't even" args))
  
  
  (if* args
     then `(progn (write-string ,(format nil "<~a" string-code)
				*html-stream*)
		  ,@(do ((xx args (cddr xx))
			 (res))
			((null xx)
			 (nreverse res))
		      (if* (eq :if* (car xx))
			 then ; insert following conditionally
			      (push `(if* ,(cadr xx)
					then (write-string 
					      ,(format nil " ~a=" (caddr xx))
					      *html-stream*)
					     (prin1-safe-http-string ,(cadddr xx)))
				    res)
			      (pop xx) (pop xx)
			 else 
					     
			      (push `(write-string 
				      ,(format nil " ~a=" (car xx))
				      *html-stream*)
				    res)
			      (push `(prin1-safe-http-string ,(cadr xx)) res)))
						    
		      
		  (write-string ">" *html-stream*)
		  ,@body
		  ,(if* (and body has-inv)
		      then `(write-string ,(format nil "</~a>" string-code)
					  *html-stream*)))
     else `(progn (write-string ,(format nil "<~a>" string-code)
				*html-stream*)
		  ,@body
		  ,(if* (and body has-inv)
		      then `(write-string ,(format nil "</~a>" string-code)
					  *html-stream*)))))
			     
		 

(defun princ-http (val)
  ;; print the given value to the http stream using ~a
  (format *html-stream* "~a" val))

(defun prin1-http (val)
  ;; print the given value to the http stream using ~s
  (format *html-stream* "~s" val))


(defun princ-safe-http (val)
  (emit-safe *html-stream* (format nil "~a" val)))

(defun prin1-safe-http (val)
  (emit-safe *html-stream* (format nil "~s" val)))


(defun prin1-safe-http-string (val)
  ;; print the contents inside a string double quotes (which should
  ;; not be turned into &quot;'s
  ;; symbols are turned into their name
  (if* (or (stringp val)
	   (and (symbolp val) 
		(setq val (symbol-name val))))
     then (write-char #\" *html-stream*)
	  (emit-safe *html-stream* val)
	  (write-char #\" *html-stream*)
     else (prin1-safe-http val)))



(defun emit-safe (stream string)
  ;; send the string to the http response stream watching out for
  ;; special html characters and encoding them appropriately
  (do* ((i 0 (1+ i))
	(start i)
	(end (length string)))
      ((>= i end)
       (if* (< start i)
	  then  (write-sequence string
				stream
				:start start
				:end i)))
	 
      
    (let ((ch (schar string i))
	  (cvt ))
      (if* (eql ch #\<)
	 then (setq cvt "&lt;")
       elseif (eq ch #\>)
	 then (setq cvt "&gt;")
       elseif (eq ch #\&)
	 then (setq cvt "&amp;")
       elseif (eq ch #\")
	 then (setq cvt "&quot;"))
      (if* cvt
	 then ; must do a conversion, emit previous chars first
		
	      (if* (< start i)
		 then  (write-sequence string
				       stream
				       :start start
				       :end i))
	      (write-string cvt stream)
		
	      (setq start (1+ i))))))
	
		

(defun html-print-list (list-of-forms stream)
  ;; html print a list of forms
  (dolist (x list-of-forms)
    (html-print x stream)))

(defun html-print (form stream)
  ;; Print the given lhtml form to the given stream
  (assert (streamp stream))
  (let ((possible-kwd (if* (atom form)
			 then form
		       elseif (consp (car form))
			 then (caar form)
			 else (car form)))
	print-handler
	ent)
    (if* (keywordp possible-kwd)
       then (if* (null (setq ent (gethash possible-kwd *html-process-table*)))
	       then (error "unknown html tag: ~s" possible-kwd)
	       else (setq print-handler
		      (html-process-print ent))))
    (if* (atom form)
       then (if* (keywordp form)
	       then (funcall print-handler ent :set nil nil stream)
	     elseif (stringp form)
	       then (write-string form stream)
	       else (error "bad form: ~s" form))
     elseif ent
       then (funcall print-handler 
		     ent
		     :full
		     (if* (consp (car form))
			then (cdr (car form)))
		     form 
		     stream)
       else (error "Illegal form: ~s" form))))
	  
(defun html-standard-print (ent cmd args form stream)
  ;; the print handler for the normal html operators
  (ecase cmd
    (:set ; just turn it on
     (format stream "<~a>" (html-process-key ent)))
    (:full ; set, do body and then unset
     (if* args
	then (format stream "<~a" (html-process-key ent))
	     (do ((xx args (cddr xx)))
		 ((null xx))
	       (format stream " ~a=\"" (car xx))
	       (emit-safe stream (format nil "~a" (cadr xx)))
	       (format stream "\""))
	     (format stream ">")
	else (format stream "<~a>" (html-process-key ent)))
     (dolist (ff (cdr form))
       (html-print ff stream))
     (if* (html-process-has-inverse ent)
	then ; end the form
	     (format stream "</~a>~%" (html-process-key ent))))))
     
  
  
		  
		    
  
					 
		      
      
  
  

(defmacro def-special-html (kwd fcn print-fcn)
  `(setf (gethash ,kwd *html-process-table*) 
     (make-html-process ,kwd nil nil ,fcn ,print-fcn)))


(def-special-html :newline 
    #'(lambda (ent args argsp body)
	(declare (ignore ent args argsp))
	(if* body
	   then (error "can't have a body with :newline -- body is ~s" body))
			       
	`(terpri *html-stream*))
  
  #'(lambda (ent cmd args form stream)
      (declare (ignore args ent))
      (if* (eq cmd :set)
	 then (terpri stream)
	 else (error ":newline in an illegal place: ~s" form)))
  )
			       

(def-special-html :princ 
    #'(lambda (ent args argsp body)
	(declare (ignore ent args argsp))
	`(progn ,@(mapcar #'(lambda (bod)
			      `(princ-http ,bod))
			  body)))
  
  #'(lambda (ent cmd args form stream)
      (declare (ignore args ent))
      (assert (eql 2 (length form)))
      (if* (eq cmd :full)
	 then (format stream "~a" (cadr form))
	 else (error ":princ must be given an argument")))
  )

#+nil
(def-special-html :action
    #'(lambda (ent args argsp body)
        `(imho:with-action (*html-stream* nil ,args nil)
          ,@body))
  #'(lambda (ent cmd args form stream)
      (declare (ignore args ent))
      (break))
  )

(def-special-html :princ-safe 
    #'(lambda (ent args argsp body)
	(declare (ignore ent args argsp))
	`(progn ,@(mapcar #'(lambda (bod)
			      `(princ-safe-http ,bod))
			  body)))
  #'(lambda (ent cmd args form stream)
      (declare (ignore args ent))
      (assert (eql 2 (length form)))
      (if* (eq cmd :full)
	 then (emit-safe stream (format nil "~a" (cadr form)))
	 else (error ":princ-safe must be given an argument"))))

(def-special-html :prin1 
    #'(lambda (ent args argsp body)
	(declare (ignore ent args argsp))
	`(progn ,@(mapcar #'(lambda (bod)
			      `(prin1-http ,bod))
			  body)))
  #'(lambda (ent cmd args form stream)
      (declare (ignore ent args))
      (assert (eql 2 (length form)))
      (if* (eq cmd :full)
	 then (format stream "~s" (cadr form))
	 else (error ":prin1 must be given an argument")))
  
  )


(def-special-html :prin1-safe 
    #'(lambda (ent args argsp body)
	(declare (ignore ent args argsp))
	`(progn ,@(mapcar #'(lambda (bod)
			      `(prin1-safe-http ,bod))
			  body)))
  #'(lambda (ent cmd args form stream)
      (declare (ignore args ent))
      (assert (eql 2 (length form)))
      (if* (eq cmd :full)
	 then (emit-safe stream (format nil "~s" (cadr form)))
	 else (error ":prin1-safe must be given an argument"))
      )
  )


(def-special-html :comment
  #'(lambda (ent args argsp body)
      ;; must use <!--   --> syntax
      (declare (ignore ent args argsp))
      `(progn (write-string "<!--" *html-stream*)
	      ,@body
	      (write-string "-->" *html-stream*)))
  
  #'(lambda (ent cmd args form stream)
      (declare (ignore ent cmd args))
      (format stream "<!--~a-->" (cadr form))))

(defun ensure-keyword (name)
  (etypecase name
    (string     (intern (string-upcase name) :keyword))
    (keyword    name)
    (symbol     (intern (string-upcase (symbol-name name)) :keyword))))

(defun ensure-symbol (name)
  (let ((lc (position #\: name :from-end t))
        (fc (position #\: name)))
    (setq name (string-upcase name))
    (if fc
        (let ((package (find-package (intern (subseq name 0 fc) :keyword))))
          (intern (subseq name (1+ lc)) package))
        (intern name))))

(defun process-imho-tag (args)
  (let ((name (position :name args))
        (id   (position :id args))
        (method (position :method args)))
    (cond (name
           (push (list :child (ensure-keyword (nth (1+ name) args)))
                 *template-bits*))
          (method
           (push (list :method (ensure-symbol (nth (1+ method) args))
                       (and id (nth (1+ id) args)))
                 *template-bits*)))))
      
(def-special-html :imho
    #'(lambda (ent args argsp body)
        (declare (ignore ent argsp body))
        `(progn (push (list :string (get-output-stream-string *html-stream*)) *template-bits*)
          (process-imho-tag ',args)
          ))
  #'(lambda (ent cmd args form stream)
      (declare (ignore ent cmd form stream))
      (push (list :string (get-output-stream-string *html-stream*)) *template-bits*)
      (process-imho-tag args)))

(defmacro def-std-html (kwd has-inverse)
  (let ((mac-name (intern (format nil "~a-~a" :with-html kwd)))
	(string-code (string-downcase (string kwd))))
    `(progn (setf (gethash ,kwd *html-process-table*)
	      (make-html-process ,kwd ,has-inverse
				     ',mac-name
				     nil
				     #'html-standard-print))
	    (defmacro ,mac-name (args &rest body)
	      (html-body-key-form ,string-code ,has-inverse args body)))))

(def-std-html :a        t)
(def-std-html :abbrl     t)
(def-std-html :acronym  t)
(def-std-html :address  t)
(def-std-html :applet   t)
(def-std-html :area    nil)

(def-std-html :b        t)
(def-std-html :base     nil)
(def-std-html :basefont nil)
(def-std-html :bdo      t)
(def-std-html :bgsound  nil)
(def-std-html :big      t)
(def-std-html :blink    t)
(def-std-html :blockquote  t)
(def-std-html :body      t)
(def-std-html :br       nil)
(def-std-html :button   nil)

(def-std-html :center   t)
(def-std-html :cite     t)
(def-std-html :code     t)
(def-std-html :col      nil)
(def-std-html :colgroup nil)

(def-std-html :dd        t)
(def-std-html :del       t)
(def-std-html :dfn       t)
(def-std-html :dir       t)
(def-std-html :div       t)
(def-std-html :dl        t)
(def-std-html :dt        t)

(def-std-html :em        t)
(def-std-html :embed     nil)

(def-std-html :fieldset        t)
(def-std-html :font        t)
(def-std-html :form        t)
(def-std-html :frame        t)
(def-std-html :frameset        t)

(def-std-html :h1        t)
(def-std-html :h2        t)
(def-std-html :h3        t)
(def-std-html :h4        t)
(def-std-html :h5        t)
(def-std-html :h6        t)
(def-std-html :head        t)
(def-std-html :hr        nil)
(def-std-html :html        t)

(def-std-html :i     t)
(def-std-html :iframe     t)
(def-std-html :ilayer     t)
(def-std-html :img     nil)
(def-std-html :input     nil)
(def-std-html :ins     t)
(def-std-html :isindex    nil)

(def-std-html :kbd  	t)
(def-std-html :keygen  	nil)

(def-std-html :label  	t)
(def-std-html :layer  	t)
(def-std-html :legend  	t)
(def-std-html :li  	t)
(def-std-html :link  	nil)
(def-std-html :listing  t)

(def-std-html :map  	t)
(def-std-html :marquee  t)
(def-std-html :menu  	t)
(def-std-html :meta  	nil)
(def-std-html :multicol t)

(def-std-html :nobr  	t)
(def-std-html :noembed  t)
(def-std-html :noframes t)
(def-std-html :noscript t)

(def-std-html :object  	nil)
(def-std-html :ol  	t)
(def-std-html :optgroup t)
(def-std-html :option  	t)

(def-std-html :p  	t)
(def-std-html :param  	nil)
(def-std-html :plaintext  nil)
(def-std-html :pre  	t)

(def-std-html :q  	t)

(def-std-html :s  	t)
(def-std-html :samp  	t)
(def-std-html :script  	t)
(def-std-html :select  	t)
(def-std-html :server  	t)
(def-std-html :small  	t)
(def-std-html :spacer  	nil)
(def-std-html :span  	t)
(def-std-html :strike  	t)
(def-std-html :strong  	t)
(def-std-html :style    t)  
(def-std-html :sub  	t)
(def-std-html :sup  	t)

(def-std-html :table  	t)
(def-std-html :tbody  	t)
(def-std-html :td  	t)
(def-std-html :textarea  t)
(def-std-html :tfoot  	t)
(def-std-html :th  	t)
(def-std-html :thead  	t)
(def-std-html :title  	t)
(def-std-html :tr  	t)
(def-std-html :tt  	t)

(def-std-html :u 	t)
(def-std-html :ul 	t)

(def-std-html :var 	t)

(def-std-html :wbr  	nil)

(def-std-html :xmp 	t)




(defmacro tag-auto-close (tag) `(get ,tag 'tag-auto-close))
(defmacro tag-auto-close-stop (tag) `(get ,tag 'tag-auto-close-stop))
(defmacro tag-no-end (tag) `(get ,tag 'tag-no-end))

; only subelements allowed in this element, no strings
(defmacro tag-no-pcdata (tag) `(get ,tag 'tag-no-pcdata))

;; given :foo or (:foo ...) return :foo
(defmacro tag-name (expr)
  `(let ((.xx. ,expr))
     (if* (consp .xx.)
	then (car .xx.)
	else .xx.)))





(eval-when (compile load eval)
  (defconstant state-pcdata 0) ; scanning for chars or a tag
  (defconstant state-readtagfirst 1)
  (defconstant state-readtag      2)
  (defconstant state-findattribname 3)
  (defconstant state-attribname    4)
  (defconstant state-attribstartvalue 5)
  (defconstant state-attribvaluedelim 6)
  (defconstant state-attribvaluenodelim 7)
  (defconstant state-readcomment 8)
  (defconstant state-readcomment-one 9)
  (defconstant state-readcomment-two 10)
  (defconstant state-findvalue 11)
  (defconstant state-rawdata 12)
)


(defstruct collector 
  next  ; next index to set
  max   ; 1+max index to set
  data  ; string vector
  )

;; keep a cache of collectors on this list

(defparameter *collectors* (list nil nil nil nil))

(defun get-collector ()
  (declare (optimize (speed 3) (safety 1)))
  (let (col)
    (mp::without-scheduling
      (do* ((cols *collectors* (cdr cols))
	    (this (car cols) (car cols)))
	  ((null cols))
	(if* this
	   then (setf (car cols) nil)
		(setq col this)
		(return))))
    (if*  col
       then (setf (collector-next col) 0)
	    col
       else (make-collector
	     :next 0
	     :max  100
	     :data (make-string 100)))))

(defun put-back-collector (col)
  (declare (optimize (speed 3) (safety 1)))
  (mp::without-scheduling 
    (do ((cols *collectors* (cdr cols)))
	((null cols)
	 ; toss it away
	 nil)
      (if* (null (car cols))
	 then (setf (car cols) col)
	      (return)))))
	 


(defun grow-and-add (coll ch)
  (declare (optimize (speed 3) (safety 1) #+cmu (extensions:inhibit-warnings 3)))
  ;; increase the size of the data portion of the collector and then
  ;; add the given char at the end
  (let* ((odata (collector-data coll))
	 (ndata (make-string (* 2 (length odata)))))
    (dotimes (i (length odata))
      (setf (schar ndata i) (schar odata i)))
    (setf (collector-data coll) ndata)
    (setf (collector-max coll) (length ndata))
    (let ((next (collector-next coll)))
      (setf (schar ndata next) ch)
      (setf (collector-next coll) (1+ next)))))

	 


    
  
  
;; character characteristics
(defconstant char-tagcharacter   1) ; valid char for a tag
(defconstant char-attribnamechar 2) ; valid char for an attribute name
(defconstant char-attribundelimattribvalue 4) ; valid for undelimited value
(defconstant char-spacechar 8)

(defparameter *characteristics* 
    ;; array of bits describing character characteristics
    (let ((arr (make-array 128 :initial-element 0)))
      (declare (optimize (speed 3) (safety 1) #+cmu (extensions:inhibit-warnings 3)))
      (macrolet ((with-range ((var from to) &rest body)
		   `(do ((,var (char-code ,from) (1+ ,var))
			 (mmax  (char-code ,to)))
			((> ,var mmax))
		      ,@body))
		 
		 (addit (index charistic)
		   `(setf (svref arr ,index)
		      (logior (svref arr ,index)
			      ,charistic)))
		 )
	
	(with-range (i #\A #\Z)
	    (addit i (+ char-tagcharacter
	       char-attribnamechar
	       char-attribundelimattribvalue)))
	
	(with-range (i #\a #\z)
	    (addit i (+ char-tagcharacter
	       char-attribnamechar
	       char-attribundelimattribvalue)))
		      
	(with-range (i #\0 #\9)
	    (addit i (+ char-tagcharacter
	       char-attribnamechar
	       char-attribundelimattribvalue)))
	
	;; let colon be legal tag character
	(addit (char-code #\:) char-tagcharacter)
	
	;; NY times special tags have _
	(addit (char-code #\_) char-tagcharacter)
	
	; now the unusual cases
	(addit (char-code #\-) char-attribundelimattribvalue)
	(addit (char-code #\.) char-attribundelimattribvalue)
	
	;; adding all typeable chars except for whitespace and >
	(addit (char-code #\:) char-attribundelimattribvalue)
	(addit (char-code #\@) char-attribundelimattribvalue)
	(addit (char-code #\/) char-attribundelimattribvalue)
	(addit (char-code #\!) char-attribundelimattribvalue)
	(addit (char-code #\#) char-attribundelimattribvalue)
	(addit (char-code #\$) char-attribundelimattribvalue)
	(addit (char-code #\%) char-attribundelimattribvalue)
	(addit (char-code #\^) char-attribundelimattribvalue)
	(addit (char-code #\&) char-attribundelimattribvalue)
	(addit (char-code #\() char-attribundelimattribvalue)
	(addit (char-code #\)) char-attribundelimattribvalue)
	(addit (char-code #\_) char-attribundelimattribvalue)
	(addit (char-code #\=) char-attribundelimattribvalue)
	(addit (char-code #\+) char-attribundelimattribvalue)
	(addit (char-code #\\) char-attribundelimattribvalue)
	(addit (char-code #\|) char-attribundelimattribvalue)
	(addit (char-code #\{) char-attribundelimattribvalue)
	(addit (char-code #\}) char-attribundelimattribvalue)
	(addit (char-code #\[) char-attribundelimattribvalue)
	(addit (char-code #\]) char-attribundelimattribvalue)
	(addit (char-code #\;) char-attribundelimattribvalue)
	(addit (char-code #\') char-attribundelimattribvalue)
	(addit (char-code #\") char-attribundelimattribvalue)
	(addit (char-code #\,) char-attribundelimattribvalue)
	(addit (char-code #\<) char-attribundelimattribvalue)
	(addit (char-code #\?) char-attribundelimattribvalue)
	
	; i'm not sure what can be in a tag name but we know that
	; ! and - must be there since it's used in comments
	
	(addit (char-code #\-) char-tagcharacter)
	(addit (char-code #\!) char-tagcharacter)
	
	; spaces
	(addit (char-code #\space) char-spacechar)
	(addit (char-code #\tab) char-spacechar)
	(addit (char-code #\return) char-spacechar)
	(addit (char-code #\linefeed) char-spacechar)
	
	)
      
      
      
      arr))
	

(defun char-characteristic (char bit)
  (declare (optimize (speed 3) (safety 1) #+cmu (extensions:inhibit-warnings 3)))
  ;; return true if the given char has the given bit set in 
  ;; the characteristic array
  (let ((code (char-code char)))
    (if* (<= 0 code 127)
       then ; in range
	    (not (zerop (logand (svref *characteristics* code) bit))))))


(defstruct tokenbuf
  cur ;; next index to use to grab from tokenbuf
  max ;; index one beyond last character
  data ;; character array
  )

;; cache of tokenbuf structs
(defparameter *tokenbufs* (list nil nil nil nil))

(defun get-tokenbuf ()
  (declare (optimize (speed 3) (safety 1)))
  (let (buf)
    (mp::without-scheduling
      (do* ((bufs *tokenbufs* (cdr bufs))
	    (this (car bufs) (car bufs)))
	  ((null bufs))
	(if* this
	   then (setf (car bufs) nil)
		(setq buf this)
		(return))))
    (if* buf
       then (setf (tokenbuf-cur buf) 0)
	    (setf (tokenbuf-max buf) 0)
	    buf
       else (make-tokenbuf
	     :cur 0
	     :max  0
	     :data (make-array 1024 :element-type 'character)))))

(defun put-back-tokenbuf (buf)
  (declare (optimize (speed 3) (safety 1)))
  (mp::without-scheduling 
    (do ((bufs *tokenbufs* (cdr bufs)))
	((null bufs)
	 ; toss it away
	 nil)
      (if* (null (car bufs))
	 then (setf (car bufs) buf)
	      (return)))))


    
    
(defun next-token (stream ignore-strings raw-mode-delimiter
		   read-sequence-func tokenbuf)
  (declare (optimize (speed 3) (safety 1) #+cmu (extensions:inhibit-warnings 3)))
  ;; return two values: 
  ;;    the next token from the stream.
  ;; 	the kind of token (:pcdata, :start-tag, :end-tag, :eof)
  ;;
  ;; if read-sequence-func is non-nil,
  ;; read-sequence-func is called to fetch the next character
  (macrolet ((next-char (stream)
	       `(let ((cur (tokenbuf-cur tokenbuf))
		      (tb (tokenbuf-data tokenbuf)))
		  (if* (>= cur (tokenbuf-max tokenbuf))
		     then ; fill buffer
			  (if* (zerop (setf (tokenbuf-max tokenbuf)
					(if* read-sequence-func
					   then (funcall read-sequence-func tb ,stream)
					   else (read-sequence tb stream))))
			     then (setq cur nil) ; eof
			     else (setq cur 0)))
		  (if* cur
		     then (prog1 (schar tb cur)
			    (setf (tokenbuf-cur tokenbuf) (1+ cur))))))
			  
	     
	     (un-next-char ()
	       `(decf (tokenbuf-cur tokenbuf)))
	     
	     (clear-coll (coll)
	       `(setf (collector-next ,coll) 0))
		     
	     (add-to-coll (coll ch)
	       `(let ((.next. (collector-next ,coll)))
		  (if* (>= .next. (collector-max ,coll))
		     then (grow-and-add ,coll ,ch)
		     else (setf (schar (collector-data ,coll) .next.)
			    ,ch)
			  (setf (collector-next ,coll) (1+ .next.)))))
	     
	     (to-preferred-case (ch)
	       ;; should check the case mode
	       (if* (eq excl:*current-case-mode* :CASE-INSENSITIVE-UPPER)
		       then `(char-upcase ,ch)
		       else `(char-downcase ,ch)))
	       
	     )
    
    (let ((state (if* raw-mode-delimiter then state-rawdata else state-pcdata))
	  (coll  (get-collector))
	  (ch)

	  (value-delim)
	  
	  (tag-to-return)
	  (attribs-to-return)
	  
	  (end-tag)
	  
	  (attrib-name)
	  (attrib-value)
	  
	  (name-length 0) ;; count only when it could be a comment
	  
	  (raw-length 0)
          (xml-bailout)
	  )
    
      (loop
      
	(setq ch (next-char stream))
	;;(format t "ch: ~s state: ~s~%" ch state)
      
	(if* (null ch)
	   then (return) ; eof -- exit loop
		)
      
      
	(case state
	  (#.state-pcdata
	   ; collect everything until we see a <
	   (if* (eq ch #\<)
	      then ; if we've collected nothing then get a tag 
		   (if* (> (collector-next coll) 0)
		      then ; have collected something, return this string
			   (un-next-char) ; push back the <
			   (return)
		      else ; collect a tag
			   (setq state state-readtagfirst))
	      else ; we will check for & here eventually
		   (if* (not (eq ch #\return))
		      then (add-to-coll coll ch))))
	
	  (#.state-readtagfirst
	   ; starting to read a tag name
	   (if* (eq #\/ ch)
	      then ; end tag
		   (setq end-tag t)
	      else (if* (eq #\! ch) ; possible comment
		      then (setf xml-bailout t)
			   (setq name-length 0))
		   (un-next-char))
	   (setq state state-readtag))
	
	  (#.state-readtag
	   ;; reading the whole tag name
	   (if* (char-characteristic ch char-tagcharacter)
	      then (add-to-coll coll (to-preferred-case ch))
		   (incf name-length)
		   (if* (and (eq name-length 3)
			     (coll-has-comment coll))
		      then (clear-coll coll)
			   (setq state state-readcomment))
			   
	      else (setq tag-to-return (compute-tag coll))
		   (clear-coll coll)
		   (if* (eq ch #\>)
		      then (return)	; we're done
		    elseif xml-bailout then 
			   (un-next-char)
			   (return)
		      else (if* (eq tag-to-return :!--)
			      then ; a comment
				   (setq state state-readcomment)
			      else (un-next-char)
				   (setq state state-findattribname)))))
	
	  (#.state-findattribname
	   ;; search until we find the start of an attribute name
	   ;; or the end of the tag
	   (if* (eq ch #\>)
	      then ; end of the line
		   (return)
	    elseif (eq ch #\=)
	      then ; value for previous attribute name
		   ; (syntax  "foo = bar" is bogus I think but it's
		   ; used some places, here is where we handle this
		   (pop attribs-to-return)
		   (setq attrib-name (pop attribs-to-return))
		   (setq state state-findvalue)
	    elseif (char-characteristic ch char-attribnamechar)
	      then (un-next-char)
		   (setq state state-attribname)
	      else nil ; ignore other things
		   ))
	  
	  (#.state-findvalue
	   ;; find the start of the value
	   (if* (char-characteristic ch char-spacechar)
	      thenret ; keep looking
	    elseif (eq ch #\>)
	      then ; no value, set the value to be the
		   ; name as a string
		   (setq attrib-value 
		     (string-downcase (string attrib-name)))
		   
		   (push attrib-name attribs-to-return)
		   (push attrib-value attribs-to-return)
		   (un-next-char)
		   (setq state state-findattribname)
	      else (un-next-char)
		   (setq state state-attribstartvalue)))
	   
	
	  (#.state-attribname
	   ;; collect attribute name

	   (if* (char-characteristic ch char-attribnamechar)
	      then (add-to-coll coll (to-preferred-case ch))
	    elseif (eq #\= ch)
	      then ; end of attribute name, value is next
		   (setq attrib-name (compute-tag coll))
		   (clear-coll coll)
		   (setq state state-attribstartvalue)
	      else ; end of attribute name with no value, 
		   (setq attrib-name (compute-tag coll))
		   (clear-coll coll)
		   (setq attrib-value 
		     (string-downcase (string attrib-name)))
		   (push attrib-name attribs-to-return)
		   (push attrib-value attribs-to-return)
		   (un-next-char)
		   (setq state state-findattribname)))
	
	  (#.state-attribstartvalue
	   ;; begin to collect value
	   (if* (or (eq ch #\")
		    (eq ch #\'))
	      then (setq value-delim ch)
		   (setq state state-attribvaluedelim)
		   ;; gobble spaces; assume since we've seen a '=' there really is a value
	    elseif (eq #\space ch) then nil
	      else (un-next-char)
		   (setq state state-attribvaluenodelim)))
	
	  (#.state-attribvaluedelim
	   (if* (eq ch value-delim)
	      then (setq attrib-value (compute-coll-string coll))
		   (clear-coll coll)
		   (push attrib-name attribs-to-return)
		   (push attrib-value attribs-to-return)
		   (setq state state-findattribname)
	      else (add-to-coll coll ch)))
	
	  (#.state-attribvaluenodelim
	   ;; an attribute value not delimited by ' or " and thus restricted
	   ;; in the possible characters
	   (if* (char-characteristic ch char-attribundelimattribvalue)
	      then (add-to-coll coll ch)
	      else (un-next-char)
		   (setq attrib-value (compute-coll-string coll))
		   (clear-coll coll)
		   (push attrib-name attribs-to-return)
		   (push attrib-value attribs-to-return)
		   (setq state state-findattribname)))
	  
	  (#.state-readcomment
	   ;; a comment ends on the first --, but we'll look for -->
	   ;; since that's what most people expect
	   (if* (eq ch #\-)
	      then (setq state state-readcomment-one)
	      else (add-to-coll coll ch)))
	  
	  (#.state-readcomment-one
	   ;; seen one -, looking for ->
	   
	   (if* (eq ch #\-)
	      then (setq state state-readcomment-two)
	      else ; not a comment end, put back the -'s
		   (add-to-coll coll #\-)
		   (add-to-coll coll ch)
		   (setq state state-readcomment)))
	  
	  (#.state-readcomment-two
	   ;; seen two -'s, looking for >
	   
	   (if* (eq ch #\>)
	      then ; end of the line
		   (return)
	    elseif (eq ch #\-)
	      then ; still at two -'s, have to put out first
		   (add-to-coll coll #\-)
	      else ; put out two hypens and back to looking for a hypen
		   (add-to-coll coll #\-)
		   (add-to-coll coll #\-)
		   (setq state state-readcomment)))
	  
	  (#.state-rawdata
	   ;; collect everything until we see the delimiter
	   (if* (eq (to-preferred-case ch) (elt raw-mode-delimiter raw-length))
	      then
		   (incf raw-length)
		   (when (= raw-length (length raw-mode-delimiter))
		     ;; push the end tag back so it can then be lexed
		     ;; but don't do it for xml stuff
		     (when (/= (length  raw-mode-delimiter) 1)
		       (dotimes (i (length raw-mode-delimiter))
			 (un-next-char)))
		     ;; set state to state-pcdata for next section
		     (return))
	      else
		   ;; push partial matches into data string
		   (dotimes (i raw-length)
		     (add-to-coll coll (elt raw-mode-delimiter i)))
		   (setf raw-length 0)
		   (add-to-coll coll ch)))
		     
	  ))
      
      
      ;; out of the loop. 
      ;; if we're in certain states then it means we should return a value
      ;;
      (case state
	((#.state-pcdata #.state-rawdata)
	 ;; return the buffer as a string
	 (if* (zerop (collector-next coll))
	    then (values nil (if (eq state state-pcdata) :eof :pcdata))
	    else (values (prog1 
			     (if* (null ignore-strings)
				then (compute-coll-string coll))
			   (put-back-collector coll))
			 :pcdata)))
	
	(#.state-readtag
	 (when (null tag-to-return)
	       (error "unexpected end of input encountered"))
	 ;; we've read a tag with no attributes
	 (put-back-collector coll)
	 (values tag-to-return
		 (if* end-tag
		    then :end-tag
		    else (if* xml-bailout then :xml else :start-tag))
		 ))
	
	(#.state-findattribname
	 ;; returning a tag with possible attributes
	 (put-back-collector coll)
	 (if* end-tag
	    then ; ignore any attributes
		 (values tag-to-return :end-tag)
	  elseif attribs-to-return
	    then (values (cons tag-to-return 
			       (nreverse attribs-to-return))
			 :start-tag)
	    else (values tag-to-return :start-tag)))
	
	(#.state-readcomment-two
	 ;; returning a comment
	 (values (prog1 (if* (null ignore-strings)
			   then (compute-coll-string coll))
		   (put-back-collector coll))
		 :comment))
	
	(t 
	 (if* (null ch) then (error "unexpected end of input encountered")
	    else (error "internal error, can't be here in state ~d" state)))))))


(defvar *kwd-package* (find-package :keyword))

(defun compute-tag (coll)
  (declare (optimize (speed 3) (safety 1) #+cmu (extensions:inhibit-warnings 3)))
  ;; compute the symbol named by what's in the collector
  (intern (string-upcase (subseq (collector-data coll) 0 (collector-next coll))) :keyword))


(defun compute-coll-string (coll)
  (declare (optimize (speed 3) (safety 1) #+cmu (extensions:inhibit-warnings 3)))
  ;; return the string that's in the collection
  (let ((str (make-string (collector-next coll)))
	(from (collector-data coll)))
    (dotimes (i (collector-next coll))
      (setf (schar str i) (schar from i)))
    
    str))

(defun coll-has-comment (coll)
  (declare (optimize (speed 3) (safety 1)))
  ;; true if the collector has exactly "!--" in it
  (and (eq 3 (collector-next coll))
       (let ((data (collector-data coll)))
	 (and (eq #\! (schar data 0))
	      (eq #\- (schar data 1))
	      (eq #\- (schar data 2))))))
		 

;;;;;;;;;;; quick and dirty parse

; the elements with no body and thus no end tag
(dolist (opt '(:area :base :basefont :bgsound :br :button :col 
	       ;;:colgroup - no, this is an element with contents
	       :embed :hr :img :imho :frame
	       :input :isindex :keygen :link :meta 
	       :plaintext :spacer :wbr))
  (setf (tag-no-end opt) t))

(defvar *in-line* '(:tt :i :b :big :small :em :strong :dfn :code :samp :kbd
		    :var :cite :abbr :acronym :a :img :object :br :script :map
		    :q :sub :sup :span :bdo :input :select :textarea :label :button :font))

(defvar *ch-format* '(:i :b :tt :big :small :strike :s :u
		      :em :strong :font))

(defvar *known-tags* '(:!doctype :a :acronym :address :applet :area :b :base
                       :basefont :bdo :bgsound :big :blink :blockquote :body :br
                       :button :caption :center :cite :code :col :colgroup
                       :comment :dd :del :dfn :dir :div :dl :dt :em :embed
                       :fieldset :font :form :frame :frameset :h1 :h2 :h3 :h4
                       :h5 :h6 :head :hr :html :i :iframe :imho :img :input :ins
                       :isindex :kbd :label :layer :legend :li :link :listing
                       :map :marquee :menu :meta :multicol :nobr :noframes
                       :noscript :object :ol :option :p :param :plaintext :pre
                       :q :samp :script :select :small :spacer :span :s :strike
                       :strong :style :sub :sup :table :tbody :td :textarea
                       :tfoot :th :thead :title :tr :tt :u :ul :var :wbr :xmp))

; the elements whose start tag can end a previous tag

(setf (tag-auto-close :tr) '(:tr :td :th :colgroup))
(setf (tag-auto-close-stop :tr) '(:table))

(setf (tag-auto-close :td) '(:td :th))
(setf (tag-auto-close-stop :td) '(:table))

(setf (tag-auto-close :th) '(:td :th))
(setf (tag-auto-close-stop :td) '(:table))

(setf (tag-auto-close :dt) '(:dt :dd))
(setf (tag-auto-close-stop :dt) '(:dl))

(setf (tag-auto-close :li) '(:li))
(setf (tag-auto-close-stop :li) '(:ul :ol))

;; new stuff to close off tags with optional close tags
(setf (tag-auto-close :address) '(:head :p))
(setf (tag-auto-close :blockquote) '(:head :p))
(setf (tag-auto-close :body) '(:body :frameset :head))

(setf (tag-auto-close :dd) '(:dd :dt))
(setf (tag-auto-close-stop :dd) '(:dl))

(setf (tag-auto-close :dl) '(:head :p))
(setf (tag-auto-close :div) '(:head :p))
(setf (tag-auto-close :fieldset) '(:head :p))
(setf (tag-auto-close :form) '(:head :p))
(setf (tag-auto-close :frameset) '(:body :frameset :head))
(setf (tag-auto-close :hr) '(:head :p))
(setf (tag-auto-close :h1) '(:head :p))
(setf (tag-auto-close :h2) '(:head :p))
(setf (tag-auto-close :h3) '(:head :p))
(setf (tag-auto-close :h4) '(:head :p))
(setf (tag-auto-close :h5) '(:head :p))
(setf (tag-auto-close :h6) '(:head :p))
(setf (tag-auto-close :noscript) '(:head :p))
(setf (tag-auto-close :ol) '(:head :p))

(setf (tag-auto-close :option) '(:option))
(setf (tag-auto-close-stop :option) '(:select))

(setf (tag-auto-close :p) '(:head :p))

(setf (tag-auto-close :pre) '(:head :p))
(setf (tag-auto-close :table) '(:head :p))

(setf (tag-auto-close :tbody) '(:colgroup :tfoot :tbody :thead))
(setf (tag-auto-close-stop :tbody) '(:table))

(setf (tag-auto-close :tfoot) '(:colgroup :tfoot :tbody :thead))
(setf (tag-auto-close-stop :tfoot) '(:table))

(setf (tag-auto-close :thead) '(:colgroup :tfoot :tbody :thead))
(setf (tag-auto-close-stop :thead) '(:table))

(setf (tag-auto-close :ul) '(:head :p))

(setf (tag-no-pcdata :table) t)
(setf (tag-no-pcdata :tr) t)


(defmethod parse-html ((p stream) &key callback-only callbacks collect-rogue-tags
				       no-body-tags)
  (declare (optimize (speed 3) (safety 1)))
  (phtml-internal p nil callback-only callbacks collect-rogue-tags
		  no-body-tags))

(defmacro tag-callback (tag)
  `(rest (assoc ,tag callbacks)))

(defun phtml-internal (p read-sequence-func callback-only callbacks collect-rogue-tags
		       no-body-tags)
  (declare (optimize (speed 3) (safety 1) #+cmu (extensions:inhibit-warnings 3)))
  (let ((first-pass nil)
	(raw-mode-delimiter nil)
	(pending nil)
	(current-tag :start-parse)
	(last-tag :start-parse)
	(current-callback-tags nil)
	(pending-ch-format nil)
	(closed-pending-ch-format nil)
	(new-opens nil)
	(tokenbuf (get-tokenbuf))
	(guts)
	(rogue-tags)
	)
    (labels ((close-off-tags (name stop-at collect-rogues)
	       ;; close off an open 'name' tag, but search no further
	       ;; than a 'stop-at' tag.
	       (if* (member (tag-name current-tag) name :test #'eq)
		  then ;; close current tag(s)
		       (loop
			 (when (and collect-rogues
				    (not (member (tag-name current-tag)
						 *known-tags*)))
			   (push (tag-name current-tag) rogue-tags))
			 (close-current-tag)
			 (when (or (member (tag-name current-tag)
					   *ch-format*)
				(not (member 
				      (tag-name current-tag) name :test #'eq)))
			     (return)))
		elseif (member (tag-name current-tag) stop-at :test #'eq)
		  then nil
		  else ; search if there is a tag to close
		       (dolist (ent pending)
			 (if* (member (tag-name (car ent)) name :test #'eq)
			    then ; found one to close
				 (loop
				   (when (and collect-rogues
					      (not (member (tag-name current-tag)
							   *known-tags*)))
				     (push (tag-name current-tag) rogue-tags))
				   (close-current-tag)
				   (if* (member (tag-name current-tag) name
						:test #'eq)
				      then (close-current-tag)
					   (return)))
				 (return)
			  elseif (member (tag-name (car ent)) stop-at
					 :test #'eq)
			    then (return) ;; do nothing
				 ))))
	   
	     (close-current-tag ()
	       ;; close off the current tag and open the pending tag
	       (when (member (tag-name current-tag) *ch-format* :test #'eq)
		 (push current-tag closed-pending-ch-format)
		 )
	       (let (element)
		 (if* (tag-no-pcdata (tag-name current-tag))
		    then (setq element `(,current-tag
					 ,@(strip-rev-pcdata guts)))
		    else (setq element `(,current-tag ,@(nreverse guts))))
		 (let ((callback (tag-callback (tag-name current-tag))))
		   (when callback
		     (setf current-callback-tags (rest current-callback-tags))
		     (funcall callback element)))
		 (let* ((prev (pop pending)))
		   (setq current-tag (car prev)
			 guts (cdr prev))
		   (push element guts))))
	     
	     (save-state ()
	       ;; push the current tag state since we're starting
	       ;; a new open tag
	       (push (cons current-tag guts) pending))
	     
	     
	     (strip-rev-pcdata (stuff)
	       ;; reverse the list stuff, omitting all the strings
	       (let (res)
		 (dolist (st stuff)
		   (if* (not (stringp st)) then (push st res)))
		 res))
	     (check-in-line (check-tag)
	       (setf new-opens nil)
	       (let (val kind (i 0)
		     (length (length first-pass)))
		 (loop
		   (if* (< i length) then
			   (setf val (nth i first-pass))
			   (setf kind (nth (+ i 1) first-pass))
			   (setf i (+ i 2))
			   (if* (= i length) then (setf first-pass (nreverse first-pass)))
		      else
			   (multiple-value-setq (val kind)
			     (get-next-token t))
			   (push val first-pass)
			   (push kind first-pass)
			   )
		   (when (eq kind :eof)
		     (if* (= i length) then (setf first-pass (nreverse first-pass)))
		     (return))
		   (when (and (eq val check-tag) (eq kind :end-tag))
		     (if* (= i length) then (setf first-pass (nreverse first-pass)))
		     (return))
		   (when (member val *ch-format* :test #'eq)
		     (if* (eq kind :start-tag) then (push val new-opens)
		      elseif (member val new-opens :test #'eq) then
			     (setf new-opens (remove val new-opens :count 1))
			else (close-off-tags (list val) nil nil)
			     )))))
		 
	     (get-next-token (force)
	       (if* (or force (null first-pass)) then
		       (multiple-value-bind (val kind)
			   (next-token p nil raw-mode-delimiter read-sequence-func
				       tokenbuf)
			(values val kind))
		  else
		       (let ((val (first first-pass))
			     (kind (second first-pass)))
			 (setf first-pass (rest (rest first-pass)))
			 (values val kind))))
	     )
      (loop
	(multiple-value-bind (val kind)
	    (get-next-token nil)
	  ;;(format t "val: ~s kind: ~s~%" val kind)
	  (case kind
	    (:pcdata
	     (when (or (and callback-only current-callback-tags)
		       (not callback-only))
	       (if* (member last-tag *in-line*)
		  then
		       (push val guts)
		  else
		       (when (dotimes (i (length val) nil)
			       (when (not (char-characteristic (elt val i) 
							       char-spacechar))
				 (return t)))
			 (push val guts))))
	     (when (and (= (length raw-mode-delimiter) 1) ;; xml tag...
			(or (and callback-only current-callback-tags)
			    (not callback-only)))
	       (close-off-tags (list last-tag) nil nil))
	     (setf raw-mode-delimiter nil)
	     )
	    
	    (:xml
	     (setf last-tag val)
	     (setf raw-mode-delimiter ">")
	     (let* ((name (tag-name val)))
	       (when (and callback-only (tag-callback name))
		 (push name current-callback-tags))
	       (save-state)
	       (setq current-tag val)
	       (setq guts nil)
	       ))
	    
	    (:start-tag
	     (setf last-tag val)
	     (if* (or (eq last-tag :style)
		      (and (listp last-tag) (eq (first last-tag) :style)))
		then
		     (setf raw-mode-delimiter
		       (if* (eq excl:*current-case-mode* :CASE-INSENSITIVE-UPPER)
			  then "</STYLE>"
			  else "</style>"))
	      elseif (or (eq last-tag :script)
		      (and (listp last-tag) (eq (first last-tag) :script)))
		then
		     (setf raw-mode-delimiter
		       (if* (eq excl:*current-case-mode* :CASE-INSENSITIVE-UPPER)
			  then "</SCRIPT>"
			  else "</script>")))
	     ; maybe this is an end tag too
	     (let* ((name (tag-name val))
		    (auto-close (tag-auto-close name))
		    (auto-close-stop nil)
		    (no-end (or (tag-no-end name) (member name no-body-tags))))
	       (when (and callback-only (tag-callback name))
		 (push name current-callback-tags))
	       (when (or (and callback-only current-callback-tags)
			 (not callback-only))
		 (if* auto-close
		    then (setq auto-close-stop (tag-auto-close-stop name))
			 (close-off-tags auto-close auto-close-stop nil))
		 (when (and pending-ch-format (not no-end))
		   (if* (member name *ch-format* :test #'eq) then nil
		    elseif (member name *in-line* :test #'eq) then
			   ;; close off only tags that are within *in-line* block
			   (check-in-line name)
		      else ;; close ALL pending char tags and then reopen 
			   (dolist (this-tag (reverse pending-ch-format))
			     (close-off-tags (list (if (listp this-tag) (first this-tag) this-tag)) nil nil))
			   ))
		 (if* no-end
		    then		; this is a singleton tag
			 (push (if* (atom val)
				  then val
				  else (list val))
			       guts)
		    else (save-state)
			 (setq current-tag val)
			 (setq guts nil))
		 (if* (member name *ch-format* :test #'eq)
		    then (push val pending-ch-format)
		    else (dolist (tmp (reverse closed-pending-ch-format))
			   (save-state)
			   (setf current-tag tmp)
			   (setf guts nil))
			 )
		 (setf closed-pending-ch-format nil)
		 )))
	  
	    (:end-tag
	     (setf raw-mode-delimiter nil)
	     (when (or (and callback-only current-callback-tags)
		       (not callback-only))
	       (close-off-tags (list val) nil nil)
	       (when (member val *ch-format* :test #'eq)
		 (setf pending-ch-format 
		   (remove val pending-ch-format :count 1
			   :test #'(lambda (x y) (eq x (if (listp y) (first y) y)))))
		 (setf closed-pending-ch-format 
		   (remove val closed-pending-ch-format :count 1
			   :test #'(lambda (x y) (eq x (if (listp y) (first y) y)))))
		 )
	       (dolist (tmp (reverse closed-pending-ch-format))
		 (save-state)
		 (setf current-tag tmp)
		 (setf guts nil))
	       (setf closed-pending-ch-format nil)
	       ))

	    (:comment
	     (setf raw-mode-delimiter nil)
	     (when (or (and callback-only current-callback-tags)
		       (not callback-only))
	       (push `(:comment ,val) guts)))
	    
	    (:eof
	     (setf raw-mode-delimiter nil)
	     ;; close off all tags
	     (when (or (and callback-only current-callback-tags)
		       (not callback-only))
	       (close-off-tags '(:start-parse) nil collect-rogue-tags))
	     (put-back-tokenbuf tokenbuf)
	     (if collect-rogue-tags
		 (return (values (cdar guts) rogue-tags))
	       (return (cdar guts))))))))))

	      

(defmethod parse-html (file &key callback-only callbacks collect-rogue-tags
				 no-body-tags)
  (declare (optimize (speed 3) (safety 1)))
  (with-open-file (p file :direction :input)
    (parse-html p :callback-only callback-only :callbacks callbacks
		:collect-rogue-tags collect-rogue-tags
		:no-body-tags no-body-tags)))	     
	     

(defmethod parse-html ((str string) &key callback-only callbacks collect-rogue-tags
					 no-body-tags)
  (declare (optimize (speed 3) (safety 1)))
  (parse-html (make-string-input-stream str) 
	      :callback-only callback-only :callbacks callbacks
	      :collect-rogue-tags collect-rogue-tags
	      :no-body-tags no-body-tags))
