;;; -*- Syntax: Ansi-Common-Lisp; Base: 10; Mode: lisp; Package: imho -*-
;;; $Id: template.lisp,v 1.26 2001/11/12 20:07:08 jesse Exp $
;;;
;;; Copyright (c) 1999, 2000, 2001 onShore Development, Inc.
;;;
;;; See the file 'COPYING' in this directory for terms.

(in-package :imho)

(defstruct html-template
  file-name
  file-date
  content)

;; ------------------------------------------------------------
;; Scan a template file for IMHO tags, assembling outputable strings
;; and children.

(defun read-template (filename)
  (let ((pathname (or (ignore-errors
			(translate-logical-pathname filename))
		      filename)))
    (if (probe-file pathname)
        (with-open-file (stream pathname)
          (let* ((parsed-html (parse-html stream))
                 (parsed-template (print-template parsed-html)))
            (make-html-template :file-date (file-write-date pathname)
                                :file-name filename
                                :content parsed-template))))))

(defun imho-superclasses (element)
  (let ((classes (class-precedence-list (find-class element))))
    (subseq classes 0 (- (length classes) 5)))) ; wah!

;; ------------------------------------------------------------
;;

(defun html-template (application element-class)
  "Try to find an HTML template corresponding to the provided
application and html-element class; if none is found, return a stub
template indicating where the application thought it was going to
find a template file."
  (flet ((missing (classname filename)
	     `((:string
		,(format nil
			 "[Missing template for class '~A' at default path ~A]"
			 classname filename)))))
    (with-slots (templates template-root)
      application
      (let* ((t-key (string-downcase (symbol-name element-class)))
	     (template (gethash t-key templates)))
	(if (and template
		 (ignore-errors
		   (probe-file (html-template-file-name template)))
		 (html-template-file-date template)
		 (= (file-write-date (html-template-file-name template))
		    (html-template-file-date template)))
	    template
	  (let* ((semicolon-p (when template-root
				(char/= #\; (char template-root
						  (1- (length template-root)))))))
            (if template-root
                (setq template-root (format nil "~a~:[~;~]" template-root semicolon-p))
                (setq template-root ""))
            (let ((files (mapcar (lambda (class)
                                   (string-downcase (symbol-name (class-name class))))
                                 (imho-superclasses element-class))))
              (block find
                (dolist (file files)
                  (let ((filename (concatenate 'string template-root file ".html")))
                    (when-bind (template (read-template filename))
                      (setf (gethash t-key templates) template)
                      (return-from find template))))
                (make-html-template :content (missing t-key template-root))))))))))
