(in-package :cl-user)

;;; PNM image routines.
;;;
;;; Images are either PBM (bitmap), PGM (greyscale) or PPM (color).
;;;
;;; PGM images are treated internally as 8-bits per pixel; PPM
;;; are 8 bits for each red, green, blue.  PGM and PPM allow larger pixels
;;; (set maxval to something different than 255), but we only allow 255 for
;;; now.
;;;
;;; The intent is to use these routines for I/O of images.  The functions
;;; {PBM,PGM,PPM}-PTR return a pointer to the beginning of the image data
;;; for the Mesa/OpenGL interface (package :GL).
;;;
;;;
;;; TODO:
;;;
;;; Error handling needs work (ie., when file is empty or of ends early.)
;;;
;;; Make generic accessors (eg., pnm-width instead of pbm, pgm, ppm-width).
;;;
;;; Later (maybe) translate some of these to C (or just use faster code
;;; such as optimized read from C or LISP).
;;;
;;; Note! I notice that small PPM images (eg., 63x63x3) do not seem to be
;;; stored as "packed" byte arrays.  Perhaps I need to allocate on C side
;;; after all.  This should not be a problem since all the accesses to PNM
;;; are via the functions exported in this file.  I am not going to worry
;;; about this now though.
;;;
;;; Richard Mann
;;; 10 November 1996

(defpackage :pnm (:use :cl)
 (:export create-pbm
	  create-pgm
	  create-ppm
	  pbm?
	  pgm?
	  ppm?
	  pbm-width
	  pbm-height
	  pgm-width
	  pgm-height
	  pgm-maxval
	  ppm-width
	  ppm-height
	  ppm-maxval
	  pbm-ref
	  pbm-set!
	  pgm-ref
	  pgm-set!
	  ppm-ref
	  ppm-set!
	  pbm-xels
	  pgm-xels
	  ppm-xels
	  pbm-flip!
	  pgm-flip!
	  ppm-flip!
	  pbm-copy
	  pgm-copy
	  ppm-copy
	  read-pnm
	  write-pnm
	  pgm->ppm
	  ppm->pgm))

(in-package :pnm)

;;;
;;; These structures should be not accessed by the user.
;;; Instead, we use specific accessors below (plus some provided by
;;; defstruct).
;;;

(defstruct pbm width height xels)
(defstruct pgm maxval width height xels)
(defstruct ppm maxval width height xels)

(defun pbm? (x) (pbm-p x))
(defun pgm? (x) (pgm-p x))
(defun ppm? (x) (ppm-p x))

(defun create-pbm (width height) nil)

;;;
;;; LISP arrays should be in row-major order (that is, rightmost indices
;;; move the fastest).  This is important when we pass a pointer of the
;;; pixel array to GL/Xlib routines.
;;;

(defun create-pgm (maxval width height)
 (make-pgm :maxval maxval :width width :height height
	   :xels (make-array `(,height ,width)
			     :element-type '(unsigned-byte 8))))

(defun create-ppm (maxval width height)
 (make-ppm :maxval maxval :width width :height height
	   :xels (make-array `(,height ,width 3)
			     :element-type '(unsigned-byte 8))))

;;; Scheme-style accessors for image data (3rd arg of ppm is pixel color)
(defun pbm-ref (image y x)
 (declare (fixnum y x))
 (aref (pbm-xels image) y x))
(defun pbm-set! (image y x v)
 (declare (fixnum y x))
 (setf (aref (pbm-xels image) y x) v))
(defun pgm-ref (image y x)
 (declare (fixnum y x))
 (aref (pgm-xels image) y x))
(defun pgm-set! (image y x v)
 (declare (fixnum y x))
 (setf (aref (pgm-xels image) y x) v))
(defun ppm-ref (image y x i)
 (declare (fixnum y x i))
 (aref (ppm-xels image) y x i))
(defun ppm-set! (image y x i v)
 (declare (fixnum y x i))
 (setf (aref (ppm-xels image) y x i) v))

;;;
;;;
;;;

(defun pbm-copy (image)
 (let* ((width (pbm-width image))
	(height (pbm-height image))
	(image2 (create-pbm width height)))
  (dotimes (row height)
   (dotimes (col width)
    (pbm-set! image2 row col (pbm-ref image row col))))
  image2))

(defun pgm-copy (image)
 (let* ((width (pgm-width image))
	(height (pgm-height image))
	(maxval (pgm-maxval image))
	(image2 (create-pgm maxval width height)))
  (dotimes (row height)
   (dotimes (col width)
    (pgm-set! image2 row col (pgm-ref image row col))))
  image2))

(defun ppm-copy (image)
 (let* ((width (ppm-width image))
	(height (ppm-height image))
	(maxval (ppm-maxval image))
	(image2 (create-ppm maxval width height)))
  (dotimes (row height)
   (dotimes (col width)
    (dotimes (i 3)
     (ppm-set! image2 row col i (ppm-ref image row col i)))))
  image2))

(defun pbm-flip! (image)
 (let* ((width (pbm-width image))
	(height (pbm-height image))
	(offset (floor height 2)))
  (declare (fixnum width height offset) (inline pbm-ref pbm-set!))
  (dotimes (row offset)
   (declare (fixnum row))
   (dotimes (col width)
    (declare (fixnum col))
    (let ((temp (pbm-ref image row col))
	  (row-prime (- height row 1)))
     (pbm-set! image row col (pbm-ref image row-prime col))
     (pbm-set! image row-prime col temp))))))

(defun pgm-flip! (image)
 (let* ((width (pgm-width image))
	(height (pgm-height image))
	(offset (floor height 2)))
  (dotimes (row offset)
   (dotimes (col width)
    (let ((temp (pgm-ref image row col))
	  (row-prime (- height row 1)))
     (pgm-set! image row col (pgm-ref image row-prime col))
     (pgm-set! image row-prime col temp))))))

(defun ppm-flip! (image)
 (let* ((width (ppm-width image))
	(height (ppm-height image))
	(offset (floor height 2)))
  (dotimes (row offset)
   (dotimes (col width)
    (let ((row-prime (- height row 1)))
     (dotimes (i 3)
      (let ((temp (ppm-ref image row col i)))
       (ppm-set! image row col i (ppm-ref image row-prime col i))
       (ppm-set! image row-prime col i temp))))))))

;;; Generic accessor
(defun read-pnm (pathname)
 (flet
  ((read-pbm (s raw?)
	     (error "READ-PNM: Cannot (yet) read pbm."))
   ;;
   (read-pgm (s raw?)
	     (let* ((width (read s))
		    (height (read s))
		    (maxval (read s))
		    (image (create-pgm maxval width height)))
	      (unless (= maxval 255)
	       (error "READ-PNM: pgm ~a not byte image." pathname))
	      (cond (raw?
		     (dotimes (row height)
		      (dotimes (col width)
		       (pgm-set! image row col (char-int (read-char s))))))
		    (t
		     (dotimes (row height)
		      (dotimes (col width)
		       (pgm-set! image row col (read s))))))
	      image))
   ;;
   (read-ppm (s raw?)
	     (let* ((width (read s))
		    (height (read s))
		    (maxval (read s))
		    (image (create-ppm maxval width height)))
	      (unless (= maxval 255)
	       (error "READ-PNM: ppm ~a not byte image." pathname))
	      (cond (raw?
		     (dotimes (row height)
		      (dotimes (col width)
		       (dotimes (i 3)
			(ppm-set! image row col i (char-int (read-char s)))))))
		    (t
		     (dotimes (row height)
		      (dotimes (col width)
		       (dotimes (i 3)
			(ppm-set! image row col i (read s)))))))
	      image)))
  ;;
  (flet
   ((read-pnm (s)
	      ;;
	      ;; Need work: skip comments in header
	      (let* ((code (symbol-name (read s))))
	       (cond ((string= code "P1") (read-pbm s nil))
		     ((string= code "P2") (read-pgm s nil))
		     ((string= code "P3") (read-ppm s nil))
		     ((string= code "P4") (read-pbm s t))
		     ((string= code "P5") (read-pgm s t))
		     ((string= code "P6") (read-ppm s t))
		     (t (error "READ-PNM: Empty or bad pnm file."))))))
   ;;
   (if (string= pathname "-")
       (read-pnm *standard-input*)
       (with-open-file (s pathname :if-does-not-exist nil)
	(unless s
	 (error "READ-PNM: File ~a does not exist." pathname))
	(read-pnm s))))))

;;; Not tested yet.
(defun write-pnm (image pathname &optional (raw? t))
 (flet
  ((write-pbm (s)
	      (error "WRITE-PNM: Cannot (yet) write pbm."))
   ;;
   (write-pgm (s)
	      (let ((width (pgm-width image))
		    (height (pgm-height image))
		    (maxval (pgm-maxval image)))
	       (unless (= maxval 255)
		(error "WRITE-PNM: pgm not byte image."))
	       (format s "~a~%" (if raw? 'p5 'p2))
	       (format s "~a ~a~%~a~%" width height maxval)
	       (cond
		(raw?
		 (dotimes (row height)
		  (dotimes (col width)
		   (write-char (int-char (pgm-ref image row col)) s))))
		(t
		 (dotimes (row height)
		  (dotimes (col width)
		   (format s "~a " (pgm-ref image row col)))
		  (format s "~%"))))))
   ;;
   (write-ppm (s)
	      (let ((width (pgm-width image))
		    (height (pgm-height image))
		    (maxval (pgm-maxval image)))
	       (unless (= maxval 255)
		(error "WRITE-PNM: ppm not byte image."))
	       (format s "~a~%" (if raw? 'p6 'p3))
	       (format s "~a ~a~%~a~%" width height maxval)
	       (cond
		(raw?
		 (dotimes (row height)
		  (dotimes (col width)
		   (dotimes (i 3)
		    (write-char (int-char (ppm-ref image row col i)) s)))))
		(t
		 (dotimes (row height)
		  (dotimes (col width)
		   (dotimes (i 3)
		    (format s "~a " (ppm-ref image row col i))))
		  (format s "~%")))))))
  ;;
  (flet
   ((write-pnm (s)
	       (cond ((pbm? image) (write-pbm s))
		     ((pgm? image) (write-pgm s))
		     ((ppm? image) (write-ppm s))
		     (t (error "WRITE-PNM: Type error.")))))
   ;;
   (if (string= pathname "-")
       (write-pnm *standard-output*)
       (with-open-file
	 (s (concatenate
	     'string pathname
	     (cond ((pbm? image) ".pbm")
		   ((pgm? image) ".pgm")
		   ((ppm? image) ".ppm")
		   (t (error "WRITE-PNM: Type error."))))
	    :direction :output)
	(write-pnm s))))))

;;; Convert grey to color (just replicate each color)
(defun pgm->ppm (pgm) nil)

;;; Convert color to greylevel image (use standard weighting)
(defun ppm->pgm (ppm) nil)
