;;;; MaiSQL --- Common Lisp Interface Layer to SQL Databases
;;;; This is copyrighted software.  See documentation for terms.
;;;; 
;;;; mysql.cl --- FFI Interface to MySQL on Unix
;;;; 
;;;; Checkout Tag: $Name:  $
;;;; $Id: mysql.lisp,v 1.1.1.1 1999/09/20 23:04:47 jesse Exp $

(in-package :MAISQL-MYSQL)

;;;; %File Description:
;;;; 
;;;; 
;;;; 

;;;; Alien Type definitions

;;; Note: These are based on Version 3.21.33a of the MySQL Client API
;;; includes.  If/When these change, we have to redo most of this, although
;;; it seems to me that the API should treat especially MYSQL as opaque
;;; blocks of memory.  But since the current API implementation doesn't
;;; really do this (it is the user's obligation to allocate and free MYSQL
;;; and other structures), so we might as well do it right...

;;; Basic Types

(def-alien-type mysql-socket int)
(def-alien-type mysql-bool char)
(def-alien-type mysql-byte char)

;;; NET Structure, see mysql_com.h

(def-alien-type mysql-net (struct st-net
				  (fd mysql-socket)
				  (fcntl int)
				  (buff (* unsigned-char))
				  (buff-end (* unsigned-char))
				  (write-pos (* unsigned-char))
				  (last-error (array char 200))
				  (last-errno unsigned-int)
				  (max-packet unsigned-int)
				  (timeout unsigned-int)
				  (pkt-nr unsigned-int)
				  (error mysql-bool)
				  (return-errno mysql-bool)))

;;; Mem-Root
(def-alien-type mysql-used-mem (struct st-used-mem
				       (next (* (struct st-used-mem)))
				       (left unsigned-int)
				       (size unsigned-int)))

(def-alien-type mysql-mem-root (struct st-mem-root
				       (free (* mysql-used-mem))
				       (used (* mysql-used-mem))
				       (min-alloc unsigned-int)
				       (block-size unsigned-int)
				       (error-handler (* t))))

;;; MYSQL-FIELD
(def-alien-type mysql-field-types (enum enum-field-types
					:decimal
					:tiny
					:short
					:long
					:float
					:double
					:null
					:timestamp
					:longlong
					:int24
					:date
					:time
					:datetime
					:year
					:newdate
					(:enum 247)
					(:set 248)
					(:tiny-blob 249)
					(:medium-blob 250)
					(:long-blob 251)
					(:blob 252)
					(:var-string 253)
					(:string 254)))

(def-alien-type mysql-field (struct st-mysql-field
				    (name (* char))
				    (table (* char))
				    (def (* char))
				    (type mysql-field-types)
				    (length unsigned-int)
				    (max-length unsigned-int)
				    (flags unsigned-int)
				    (decimals unsigned-int)))

;;; MYSQL-ROWS
(def-alien-type mysql-row (* c-string))
(def-alien-type mysql-field-offset unsigned-int)

(def-alien-type mysql-rows (struct st-mysql-rows
				   (next (* (struct st-mysql-rows)))
				   (data mysql-row)))

(def-alien-type mysql-row-offset (* mysql-rows))

(def-alien-type mysql-data (struct st-mysql-data
				   (rows unsigned-int)
				   (fields unsigned-int)
				   (data (* mysql-rows))
				   (alloc mysql-mem-root)))

;;; MYSQL
(def-alien-type mysql-status (enum enum-status
				   :ready :get-result :use-result))

(def-alien-type mysql-mysql (struct st-mysql
				    (net mysql-net)
				    (host (* char))
				    (user (* char))
				    (passwd (* char))
				    (unix-socket (* char))
				    (server-version (* char))
				    (host-info (* char))
				    (info (* char))
				    (db (* char))
				    (port unsigned-int)
				    (client-flag unsigned-int)
				    (server-capabilities unsigned-int)
				    (protocol-version unsigned-int)
				    (field-count unsigned-int)
				    (thread-id unsigned-long)
				    (affected-rows unsigned-long)
				    (insert-id unsigned-long)
				    (extra-info unsigned-long)
				    (status mysql-status)
				    (fields (* mysql-field))
				    (field-alloc mysql-mem-root)
				    (free-me mysql-bool)
				    (reconnect mysql-bool)))
				  
;;; MYSQL-RES
(def-alien-type mysql-mysql-res (struct st-mysql-res
					(row-count unsigned-long)
					(field-count unsigned-int)
					(current-field unsigned-int)
					(fields (* mysql-field))
					(data (* mysql-data))
					(data-cursor (* mysql-rows))
					(field-alloc mysql-mem-root)
					(row mysql-row)
					(current-row mysql-row)
					(lengths (* unsigned-int))
					(handle (* mysql-mysql))
					(eof mysql-bool)))

;;;; The Alien C routines
(declaim (inline mysql-connect))
(def-alien-routine "mysql_connect" (* mysql-mysql)
  (mysql (* mysql-mysql) :in)
  (host c-string :in)
  (user c-string :in)
  (passwd c-string :in))

;;; The following function's signature depends on the version of MySQL.
;;; The default form is only valid for MYSQL_VERSION_ID >= 32200. If you
;;; experience problems, try to use the commented form.
(declaim (inline mysql-real-connect))
(def-alien-routine "mysql_real_connect" (* mysql-mysql)
  (mysql (* mysql-mysql) :in)
  (host c-string :in)
  (user c-string :in)
  (passwd c-string :in)
  (db c-string :in)
  (port unsigned-int :in)
  (unix-socket c-string :in)
  (clientflag unsigned-int :in))

#+NIL
(def-alien-routine "mysql_real_connect" (* mysql-mysql)
  (mysql (* mysql-mysql) :in)
  (host c-string :in)
  (user c-string :in)
  (passwd c-string :in)
  (port unsigned-int :in)
  (unix-socket c-string :in)
  (clientflag unsigned-int :in))

(declaim (inline mysql-close))
(def-alien-routine "mysql_close" void
  (sock (* mysql-mysql) :in))

(declaim (inline mysql-select-db))
(def-alien-routine "mysql_select_db" int
  (mysql (* mysql-mysql) :in)
  (db c-string :in))

(declaim (inline mysql-query))
(def-alien-routine "mysql_query" int
  (mysql (* mysql-mysql) :in)
  (query c-string :in))

;;; I doubt that this function is really useful for direct Lisp usage,
;;; but it is here for completeness...

(declaim (inline mysql-real-query))
(def-alien-routine "mysql_real_query" int
  (mysql (* mysql-mysql) :in)
  (query c-string :in)
  (length unsigned-int :in))

(declaim (inline mysql-create-db))
(def-alien-routine "mysql_create_db" int
  (mysql (* mysql-mysql) :in)
  (db c-string :in))

(declaim (inline mysql-drop-db))
(def-alien-routine "mysql_drop_db" int
  (mysql (* mysql-mysql) :in)
  (db c-string :in))

(declaim (inline mysql-shutdown))
(def-alien-routine "mysql_shutdown" int
  (mysql (* mysql-mysql) :in))

(declaim (inline mysql-dump-debug-info))
(def-alien-routine "mysql_dump_debug_info" int
  (mysql (* mysql-mysql) :in))

(declaim (inline mysql-refresh))
(def-alien-routine "mysql_refresh" int
  (mysql (* mysql-mysql) :in)
  (refresh-options unsigned-int :in))

(declaim (inline mysql-kill))
(def-alien-routine "mysql_kill" int
  (mysql (* mysql-mysql) :in)
  (pid unsigned-long :in))

(declaim (inline mysql-stat))
(def-alien-routine "mysql_stat" c-string
  (mysql (* mysql-mysql) :in))

(declaim (inline mysql-get-server-info))
(def-alien-routine "mysql_get_server_info" c-string
  (mysql (* mysql-mysql) :in))

(declaim (inline mysql-get-client-info))
(def-alien-routine "mysql_get_client_info" c-string)

(declaim (inline mysql-get-host-info))
(def-alien-routine "mysql_get_host_info" c-string
  (mysql (* mysql-mysql) :in))

(declaim (inline mysql-get-proto-info))
(def-alien-routine "mysql_get_server_info" unsigned-int
  (mysql (* mysql-mysql) :in))

(declaim (inline mysql-list-dbs))
(def-alien-routine "mysql_list_dbs" (* mysql-mysql-res)
  (mysql (* mysql-mysql) :in)
  (wild c-string :in))

(declaim (inline mysql-list-tables))
(def-alien-routine "mysql_list_tables" (* mysql-mysql-res)
  (mysql (* mysql-mysql) :in)
  (wild c-string :in))

(declaim (inline mysql-list-fields))
(def-alien-routine "mysql_list_fields" (* mysql-mysql-res)
  (mysql (* mysql-mysql) :in)
  (table c-string :in)
  (wild c-string :in))

(declaim (inline mysql-list-processes))
(def-alien-routine "mysql_list_processes" (* mysql-mysql-res)
  (mysql (* mysql-mysql) :in))

(declaim (inline mysql-store-result))
(def-alien-routine "mysql_store_result" (* mysql-mysql-res)
  (mysql (* mysql-mysql) :in))

(declaim (inline mysql-use-result))
(def-alien-routine "mysql_use_result" (* mysql-mysql-res)
  (mysql (* mysql-mysql) :in))

(declaim (inline mysql-free-result))
(def-alien-routine "mysql_free_result" void
  (res (* mysql-mysql-res) :in))

(declaim (inline mysql-data-seek))
(def-alien-routine "mysql_data_seek" void
  (res (* mysql-mysql-res) :in)
  (offset unsigned-int :in))

(declaim (inline mysql-row-seek))
(def-alien-routine "mysql_row_seek" mysql-row-offset
  (res (* mysql-mysql-res) :in)
  (offset mysql-row-offset :in))

(declaim (inline mysql-field-seek))
(def-alien-routine "mysql_field_seek" mysql-field-offset
  (res (* mysql-mysql-res) :in)
  (offset mysql-field-offset :in))

(declaim (inline mysql-fetch-row))
(def-alien-routine "mysql_fetch_row" mysql-row
  (res (* mysql-mysql-res) :in))

(declaim (inline mysql-fetch-lengths))
(def-alien-routine "mysql_fetch_lengths" (* unsigned-int)
  (res (* mysql-mysql-res) :in))

(declaim (inline mysql-fetch-field))
(def-alien-routine "mysql_fetch_field" (* mysql-field)
  (res (* mysql-mysql-res) :in))

(declaim (inline mysql-escape-string))
(def-alien-routine "mysql_escape_string" unsigned-int
  (to c-string :in)
  (from c-string :in)
  (length unsigned-int :in))

(declaim (inline mysql-debug))
(def-alien-routine "mysql_debug" void
  (debug c-string :in))

;;;; Equivalents of C Macro definitions for accessing various fields
;;;; in the internal MySQL Datastructures

(declaim (inline mysql-num-rows))
(defun mysql-num-rows (result)
  "Returns the number of rows in the given result set."
  (declare (type (alien (* mysql-mysql-res)) result))
  (slot result 'row-count))

(declaim (inline mysql-num-fields))
(defun mysql-num-fields (result)
  "Returns the number of fields in the given result set."
  (declare (type (alien (* mysql-mysql-res)) result))
  (slot result 'field-count))

(declaim (inline mysql-eof))
(defun mysql-eof (result)
  "Returns true if the end of the result set has been reached."
  (declare (type (alien (* mysql-mysql-res)) result))
  (slot result 'eof))

(declaim (inline mysql-error))
(defun mysql-error (mysql)
  "Returns the last error message."
  (declare (type (alien (* mysql-mysql)) mysql))
  (slot (slot mysql 'net) 'last-error))

(declaim (inline mysql-error-string))
(defun mysql-error-string (mysql)
  "Returns the last error message as a lisp string."
  (declare (type (alien (* mysql-mysql)) mysql))
  (c-call::%naturalize-c-string (alien-sap (mysql-error mysql))))

(declaim (inline mysql-errno))
(defun mysql-errno (mysql)
  "Returns the last error number."
  (declare (type (alien (* mysql-mysql)) mysql))
  (slot (slot mysql 'net) 'last-errno))

(declaim (inline mysql-info))
(defun mysql-info (mysql)
  "Returns the last information string."
  (declare (type (alien (* mysql-mysql)) mysql))
  (slot mysql 'info))
