;;;; MaiSQL --- Common Lisp Interface Layer to SQL Databases
;;;; This is copyrighted software.  See documentation for terms.
;;;; 
;;;; postgresql.cl --- FFI interface to PostgreSQL on Unix
;;;; 
;;;; Checkout Tag: $Name:  $
;;;; $Id: glue-lispworks.lisp,v 1.2 2001/02/08 22:21:46 jesse Exp $

(in-package :maisql-postgresql)

(defun %pg-database-connection (connection-spec)
  (check-connection-spec connection-spec :postgresql
			 (host db user password &optional port options tty))
  (destructuring-bind (host db user password &optional port options tty)
      connection-spec
    (fli:with-dynamic-foreign-objects ()
      (let* ((fh (fli:convert-to-dynamic-foreign-string host))
             (fp (fli:convert-to-dynamic-foreign-string port))
             (fopts (fli:convert-to-dynamic-foreign-string options))
             (fu (fli:convert-to-dynamic-foreign-string user))
             (fdb (fli:convert-to-dynamic-foreign-string db))
             (fpass (fli:convert-to-dynamic-foreign-string password))
             (ftty (fli:convert-to-dynamic-foreign-string tty))
             (connection (pqsetdblogin fh fp fopts ftty fdb fu fpass)))
        (declare (type fli::pointer connection))
        (when (not (eq (pqstatus connection) :connection-ok))
	  ;; Connect failed
	  (error 'maisql-connect-error
	         :database-type database-type
	         :connection-spec connection-spec
	         :errno (pqstatus connection)
	         :error (pqerrormessage connection)))
        connection))))

(defun %pg-query (database expr)
  (let ((conn-ptr (database-conn-ptr database)))
    (declare (type fli::pointer conn-ptr))
    (fli:with-dynamic-foreign-objects ()
      (let ((fquery (fli:convert-to-dynamic-foreign-string expr)))
        (pqexec conn-ptr fquery)))))

(defun %pg-exec (database sql-expression)
  (let ((conn-ptr (database-conn-ptr database)))
    (declare (type fli::pointer conn-ptr))
    (fli:with-dynamic-foreign-objects ()
      (let* ((fexp (fli:convert-to-dynamic-foreign-string sql-expression))
             (result (pqexec conn-ptr fexp)))
        (when (null result)
	  (format t ";; ~s~%~%" (pqerrormessage conn-ptr))
	  #+nil
	  (error 'maisql-sql-error
	         :database database
	         :expression sql-expression
	         :errno nil
	         :error (%trim-crlf (pqerrormessage conn-ptr))))
        result))))


(defun %pg-query-result-set (database query-expression)
  (let ((conn-ptr (database-conn-ptr database)))
    (declare (type fli::pointer conn-ptr))
    (fli:with-dynamic-foreign-objects ()
      (let* ((fexp (fli:convert-to-dynamic-foreign-string query-expression))
             (result (pqexec conn-ptr fexp)))
        (when (null result)
	  (error 'maisql-sql-error
	         :database database
	         :expression query-expression
	         :errno nil
	         :error "PQexec failed to return result structure!"))
        result))))
