
//! component="Resources"

#ifndef header_resourcetype_string
#define header_resourcetype_string

#include "resource_manager.h"
#include "resourcetype.h"
#include "resource.h"
#include "../System/clanstring.h"
#include "../System/error.h"
#include "../IOData/inputsource.h"
#include "../IOData/inputsource_provider.h"

class CL_Res_String : public CL_ResourceType
//: The String resource is used to read string values from a resource file.
// CL_Res_String, CL_Res_Integer and CL_Res_String helps you with seperating
// the game code from the constants used in the game.
// <br>
// Using these resource types, you can easily change the text used in the game.
// Just by changing the resource file, you can make you game speak Danish (or Dutch), or
// any other language. You can change the constants like weapon damage, movement
// speed, and so on.
// <br>
// These resources can also be used to apply theme support into your application.
// By letting each theme become a datafile, you can give the player full control
// of the visual parts of the game - and without using much work on it yourself.
//!res_type: string
//!res_desc: The string resource is used to read a string from a resourcefile
{
public:
	static std::string load(std::string res_id, CL_ResourceManager *manager, std::string _default);
	static std::string load(std::string res_id, CL_ResourceManager *manager );

	CL_Res_String();

private:
friend CL_ResourceManager;
	virtual CL_Resource *create_from_location(
		std::string name,
		std::string location,
		CL_ResourceOptions *options,
		CL_ResourceManager *parent);

	virtual CL_Resource *create_from_serialization(
		std::string name,
		CL_ResourceManager *parent);
};

class CL_StringResource : public CL_Resource
{
public:
	CL_StringResource(
		std::string name,
		std::string value)
	: CL_Resource("string", name)
	{
		this->value = value;
		load_count = 0;
	}

	CL_StringResource(
		std::string name,
		std::string location,
		CL_ResourceOptions *options,
		CL_ResourceManager *parent)
	: CL_Resource("string", name)
	{
		value = location;
		load_count = 0;
	}

	CL_StringResource(
		std::string name,
		CL_ResourceManager *parent)
	: CL_Resource("string", name)
	{
		CL_InputSource *input =
			parent->get_resource_provider()->open_source(name.c_str());

		value = input->read_string();
		
		delete input;
		load_count = 0;
	}
	
	std::string get_value() const { return value; }

	virtual void load() { load_count++; }
	virtual void unload() { load_count--; }
	virtual int get_load_count() { return load_count; }

	virtual void serialize_save(CL_OutputSource *output)
	{
		output->write_string(value.c_str());
	}

private:
	std::string value;
	int load_count;
};

#endif
