/*
	$Id: inputsource_memory_generic.h,v 1.1.1.1 2000/04/09 12:18:01 mbn Exp $

	------------------------------------------------------------------------
	ClanLib, the platform independent game SDK.

	This library is distributed under the GNU LIBRARY GENERAL PUBLIC LICENSE
	version 2. See COPYING for details.

	For a total list of contributers see CREDITS.

	------------------------------------------------------------------------
*/

#ifndef header_inputsource_memory_generic
#define header_inputsource_memory_generic

#include "API/Core/IOData/inputsource.h"

class CL_InputSource_MemoryGeneric : public CL_InputSource
//: Interface to read data from a memory_generic source.
// CL_InputSource_MemoryGeneric is used to read data from a memory_generic source.
//!also: CL_InputSourceProvider - Interface to open input sources with.
{
public:
	CL_InputSource_MemoryGeneric(void *data, int size, bool delete_data = false);
	virtual ~CL_InputSource_MemoryGeneric();

	virtual void set_system_mode();
	//: Changes input data endianess to the local systems mode.

	virtual void set_big_endian_mode();
	//: Changes input data endianess to big endian mode.

	virtual void set_little_endian_mode();
	//: Changes input data endianess to little endian mode. This is the 
	//: default setting.

	virtual int read_int32();
	//: Reads a signed 32 bit integer from input source.
	//!retval: The integer read.

	virtual unsigned int read_uint32();
	//: Reads a unsigned 32 bit integer from input source.
	//!retval: The unsigned integer read.

	virtual short read_short16();
	//: Reads a signed 16 bit integer (short) from input source.
	//!retval: The short read.

	virtual unsigned short read_ushort16();
	//: Reads a unsigned 16 bit integer (short) from input source.
	//!retval: The unsigned short read.

	virtual char read_char8();
	//: Reads a signed 8 bit integer (char) from input source.
	//!retval: The char read.

	virtual unsigned char read_uchar8();
	//: Reads a unsigned 8 bit integer (char) from input source.
	//!retval: The unsigned char read.

	virtual float read_float32();
	//: Reads an ieee format floating point from input source.
	//!retval: The float read.

	virtual bool read_bool();
	//: Reads a bool from input source.
	//!retval: The bool read.

	virtual int read(void *data, int size);
	//: Reads larger amounts of data (no endian and 64 bit conversion).
	//!param: data - Points to an array where the read data is stored.
	//!param: size - Number of bytes to read.
	//!retval: Num bytes actually read.
	
	virtual void open();
	//: Opens the input source. By default, it is open.
	
	virtual void close();
	//: Closes the input source.

	virtual CL_InputSource *clone() const;
	//: Make a copy of the current InputSource, standing at the same position.
	//!retval: The clone of the input source.

	virtual int tell() const;
	//: Returns current position in input source.
	//!retval: Current position in input source.

	virtual void seek(int pos, SeekEnum seek_type);
	//: Seeks to the specified position in the input source.
	//!param: pos - Position relative to 'seek_type'.
	//!param: seek_type - Defines what the 'pos' is relative to. Can be either seek_set, seek_cur og seek_end.

	virtual int size() const;
	//: Returns the size of the input source
	//!retval: Size of the input source.

	virtual std::string read_string();
	//: Reads a string from the input source.
	//!retval: The string read.

	virtual void push_position();
	//: Pushes the current input source position. The position can be restored again with pop_position.
	
	virtual void pop_position();
	//: Pops a previous pushed input source position (returns to the position).

private:
	unsigned char *m_data;
	unsigned int m_pos, m_size;
	bool m_delete_data;
};

#endif
