/*
  Copyright (C) 1999-2001 Ricardo Ueda Karpischek

  This is free software; you can redistribute it and/or modify
  it under the terms of the version 2 of the GNU General Public
  License as published by the Free Software Foundation.

  This software is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this software; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
  USA.
*/

/*

skel.c: Skeleton computation

*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include "common.h"
#include "gui.h"

#ifndef PI
#define PI M_PI
#endif

/* (devel)

Skeleton pixels
---------------

The first method implemented by Clara OCR for symbol classification
was skeleton fitting. Two symbols are considered similar when each
one contains the skeleton of the other.

Clara OCR implements five heuristics to compute skeletons. The
heuristic to be used is informed through the command-line option
-k as the SA parameter. The value of SA may be 0, 1, 2, 3 or 4.

Heuristics 0, 1 and 2 considerer a pixel as being a skeleton pixel
if it is the center of a circle inscribed within the closure, and
tangent to the pattern boundary in more than one point.

The discrete implementation of this idea is as follows: for each
pixel p of the closure, compute the minimum distance d from p to
some boundary pixel. Now try to find two pixels on the closure
boundary such that the distance from each of them to p does not
differ too much from d (must be less than or equal to RR). These
pixels are called "BPs".

To make the algorithm faster, the maximum distance from p to the
boundary pixels considered is RX. In fact, if there exists a
square of size 2*BT+1 centered at p, then p is considered a
skeleton pixel.

As this criteria alone produces fat skeletons and isolated
skeleton pixels along the closure boundary, two other conditions
are imposed: the angular distance between the radiuses from p to
each of those two pixels must be "sufficiently large" (larger
than MA), and a small path joining these two boundary pixels
(built only with boundary pixels) must not exist (the "joined"
function computes heuristically the smallest boundary path
between the two pixels, and that distance is then compared to
MP).

The heuristics 1 and 2 are variants of heuristic 0:

1. (SA = 1) The minimum linear distance between the two BPs
is specified as a factor (ML) of the square of the radius. This
will avoid the conversion from rectangular to polar coordinates
and may save some CPU time, but the results will be slightly
different.

2. (SA = 2) No minimum distance checks are performed, but a
minimum of MB BPs is required to exist in order to consider the
pixel p a skeleton pixel.

The heuristic 3 is very simple. It computes the skeleton removing
BT times the boundary.

The heuristic 4 uses "growing lines". For angles varying in steps
of approximately 22 degrees, a line of lenght RX pixels is drawn
from each pixel. The heuristic check if the line can or cannot be
entirely drawn using black pixels. Depending on the results, it
decides if the pixel is an skeleton pixel or not. For instance:
if all lines could be drawn, then the pixel is center of an
inscribed circle, so it's considered an skeleton pixels. All
considered cases can be found on the source code.

The heuristic 5 computes the distance from each pixel to the
border, for some definition of distance. When the distance is
at least RX, it is considered a skeleton pixel. Otherwise,
it will be considered a skeleton pixel if its distance to the
border is close to the maximum distance around it (see the code
for details).

All parameters for skeleton computation are informed to Clara
through the -k command-line option, as a list in the following
order: SA,RR,MA,MP,ML,MB,RX,BT. For instance:

    clara -k 2,1.4,1.57,10,3.8,10,4,4

The default values and the valid ranges for each parameter must
be checked on the source code (see the declaration of the
variables SA, RR, MA, MP, ML, MB, RX, and BT, and the function
skel_parms). Note that BT must be at most RX.

*/

/*

Parameters for skeleton computation of closures.

*/
int   DEF_SA = 2;
float DEF_RR = 1.4;
float DEF_MA = (PI/2);
int   DEF_MP = 10;
float DEF_ML = 3.8;
int   DEF_MB = 10;
int   DEF_RX = 1; /* was 4 */
int   DEF_BT = 1; /* was 4 */

int   SA;
float RR;
float MA;
int   MP;
float ML;
int   MB;
int   RX;
int   BT;

/* arrays required for skeleton computation */
int *xbp=NULL,*ybp=NULL;
float *bpd=NULL;
float *rmdist=NULL,*rmdistp=NULL;

/* derived from RX */
int RX2,DRX;

/* geometric limits of the skeleton */
int fc_bp,lc_bp,fl_bp,ll_bp;

/* cb and cb2 are buffers for symbols, used to compute skeletons */
char cb[LFS*FS],cb2[LFS*FS];

/*

Copy the skeleton parameters to/from the specified
locations. Both locations may be the macro SP_GLOBAL or the macro
SP_DEF or a valid pattern index.

global skeleton parameters to the default buffers.

*/
void spcpy(int to,int from)
{
    pdesc *d;

    if ((to == SP_DEF) && (from == SP_GLOBAL)) {
        DEF_SA = SA;
        DEF_RR = RR;
        DEF_MA = MA;
        DEF_MP = MP;
        DEF_ML = ML;
        DEF_MB = MB;
        DEF_RX = RX;
        DEF_BT = BT;
    }

    else if ((to == SP_GLOBAL) && (from == SP_DEF)) {
        SA = DEF_SA;
        RR = DEF_RR;
        MA = DEF_MA;
        MP = DEF_MP;
        ML = DEF_ML;
        MB = DEF_MB;
        RX = DEF_RX;
        BT = DEF_BT;
    }

    else if ((to == SP_GLOBAL) && (0 <= from) && (from <= topp)) {

        d = pattern + from;
        if (d->p[0] >= 0) {
            SA = d->p[0];
            RR = ((float)d->p[1]) / 10;
            MA = ((float)d->p[2]) / 100;
            MP = d->p[3];
            ML = ((float)d->p[4]) / 10;
            MB = d->p[5];
            RX = d->p[6];
            BT = d->p[7];
        }
    }

    else if ((0 <= to) && (to <= topp) && (from == SP_GLOBAL)) {

        pattern[to].p[0] = SA;
        pattern[to].p[1] = RR * 10;
        pattern[to].p[2] = MA * 100;
        pattern[to].p[3] = MP;
        pattern[to].p[4] = ML * 10;
        pattern[to].p[5] = MB;
        pattern[to].p[6] = RX;
        pattern[to].p[7] = BT;
    }
}

/*

Converts rectangular to polar coordinates.

*/
float ro(int x,int y)
{
    float r=0.0;

    if (x == 0) {
        if (y > 0)
            r = (PI/2);
        else if (y < 0)
            r = (3*PI/2);
        else {
            fatal(IE,"unexpected conversion");
        }
    }
    else if (x > 0) {
        if (y > 0)
            r = (atan(((double)y)/x));
        else
            r = (2*PI+atan(((double)y)/x));
    }
    else
        r = (PI+atan(((double)y)/x));
    return(r);
}

/*

Consist skeleton parameters.

*/
void consist_skel(void)
{
    int u,v;

    if ((SA < 0) || (SA > 6))
        SA = DEF_SA;
    if ((RR < 0.0) || (RR > 4.0))
        RR = DEF_RR;
    if ((MA < 0.0) || (MA >= PI))
        MA = DEF_MA;
    if ((MP < 0) || (MP > 20))
        MP = DEF_MP;
    if ((ML < 0.0) || (ML >= 4))
        ML = DEF_ML;
    if ((MB < 0) || (MB >= 15))
        MB = DEF_MB;
    if ((RX < 0) || (RX >= 10))
        RX = DEF_RX;
    if ((BT < 0) || (BT > RX))
        BT = (DEF_BT > RX) ? RX : DEF_BT;

    RX2 = 2 * RX + 1;
    DRX = ((2*RX+1)*(2*RX+1));

    /* arrays required for skeleton and boxes computation */
    if (xbp != NULL)
        c_free(xbp);
    if (ybp != NULL)
        c_free(ybp);
    if (bpd != NULL)
        c_free(bpd);
    if (rmdistp != NULL)
        c_free(rmdistp);
    xbp = c_realloc(NULL,sizeof(int)*DRX,NULL);
    ybp = c_realloc(NULL,sizeof(int)*DRX,NULL);
    rmdistp = c_realloc(NULL,sizeof(float)*(2*RX+1)*(2*RX+1),NULL);
    bpd = c_realloc(NULL,sizeof(float)*DRX,NULL);

    /* build rmdist */
    rmdist = rmdistp + RX*RX2 + RX;
    for (u=-RX; u<=RX; ++u)
        for (v=-RX; v<=RX; ++v)
            rmdist[u+v*RX2] = sqrt(u*u+v*v);

}

/*

Process the list of parameters for skeleton computation.

This service is called while the program is processing the
command-line parameters. The argument to -k is passed to
skel_parms to be parsed.

*/
void skel_parms(char *s)
{
    int i,j,n;
    char b[21];

    for (n=i=j=0; j<=strlen(s); ++j) {
        if ((s[j] == ',') || (s[j] == 0)) {

            /* copy next argument to b */
            if (j-i > 20) {
                fatal(DI,"argument to -k too long");
            }
            if (j > i)
                strncpy(b,s+i,j-i);
            b[j-i] = 0;

            /* algorithm to use */
            if (n == 0) {
                if (strlen(b) > 0)
                    SA = atoi(b);
            }

            /* acceptable error for radius */
            else if (n == 1) {
                if (strlen(b) > 0)
                    RR = atof(b);
            }

            /* minimum acceptable angular distance */
            else if (n == 2) {
                if (strlen(b) > 0)
                    MA = atof(b);
            }

            /* maximum path threshold */
            else if (n == 3) {
                if (strlen(b) > 0)
                    MP = atoi(b);
            }

            /* minimum acceptable linear distance */
            else if (n == 4) {
                if (strlen(b) > 0)
                    ML = atof(b);
            }

            /* minimum acceptable number of BPs */
            else if (n == 5) {
                if (strlen(b) > 0)
                    MB = atoi(b);
            }

            /* maximum radius for squares */
            else if (n == 6) {
                if (strlen(b) > 0)
                    RX = atoi(b);
            }

            /* black square threshold */
            else if (n == 7) {
                if (strlen(b) > 0)
                    BT = atoi(b);
            }

            else {
                fatal(DI,"too many arguments to -k");
            }

            /* prepare next iteration */
            i = j + 1;
            ++n;
        }
    }

    if (n < 8) {
        fatal(DI,"too few arguments to -k (%d)",n);
    }

    consist_skel();
}

/*

Computes recursively the border from the seed (i,j).

*/
int border(int i,int j)
{
    int g=1;

    if ((i<0) || (j<0) || (i>=LFS) || (j>=FS))
        return(0);
    if (cb2[i+j*LFS] == BLACK) {
        cb2[i+j*LFS] = WHITE;
        g &= border(i-1,j-1);
        g &= border(i-1,j);
        g &= border(i-1,j+1);
        g &= border(i,j-1);
        g &= border(i,j+1);
        g &= border(i+1,j-1);
        g &= border(i+1,j);
        g &= border(i+1,j+1);
        if (g == 0)
            cb[i+j*LFS] = GRAY;
    }
    return((cb[i+j*LFS] == WHITE) ? 0 : 1);
}

/*

Computes the border of the bitmap stored on cb

W and H may be used to make this service faster. By default use
W=0 and H=0. To achieve time savings, pass on W and H the
width and heigth of the bitmap.

*/
void cb_border(int W,int H)
{
    int i,j;

    /* skel computation requires a buffer */
    memcpy(cb2,cb,LFS*FS);

    /* by default analyse the entire bitmap */
    if ((!shape_opt) || (W == 0))
        W = LFS;
    if ((!shape_opt) || (H == 0))
        H = FS;

    /*
        Computes the boundary for each component. This
        block will call border(i,j) only when (i,j) is a black pixel
        with a white neighbour. So if the border of the component
        that contains (i,j) was already computed, then (i,j) will not
        have a white neighbour because the border function makes the
        boundary gray.
    */
    for (i=0; i<W; ++i)
        for (j=0; j<H; ++j)
            if ((cb[i+j*LFS] == BLACK) &&
               (((i>0) && (cb[(i-1)+j*LFS] == WHITE)) ||
                ((j>0) && (cb[i+(j-1)*LFS] == WHITE)) ||
                ((i+1<LFS) && (cb[(i+1)+j*LFS] == WHITE)) ||
                ((j+1<FS) && (cb[i+(j+1)*LFS] == WHITE))))
                border(i,j);
}


/*

Compute the smallest size of a gray path joining (x,y) and (u,v)
on cb. Returns 1 if this size is not larger than MP.

This function assumes some intuitive topological properties of
the closure boundary that we're not sure if are always
true. Basically, as the boundary is a closed line, there will be
exactly two paths joining each two boundary points (well, if one
point is on the internal boundary and other on the external, then
there will be no paths at all). We assume also that it is
possible to walk from (x,y) to (u,v) simply moving to the left,
right, top or botton neighbour, and avoiding to go back to the
last pixel visited (as this strategy could potentially produce an
infinite loop, the walk stopp after MP iterations). These
hypothesis allow us to implement this functions as a very simple
loop, and not as a typical recursive minimum path computation.

BUGS: this function is assymetric, that is, joined(x,y,u,v) may
differ from joined(u,v,x,y). It's not clear how to fix this.

*/
int joined(int x,int y,int u,int v)
{
    int a,b,c,d,r;
    int k[8],t,l;

    /* compute candidates to the second pixel */
    t = 0;
    if ((x+1<LFS) && (cb[(x+1)+y*LFS]==GRAY)) {
        k[t++] = x+1;
        k[t++] = y;
    }
    if ((y-1>=0) && (cb[x+(y-1)*LFS]==GRAY)) {
        k[t++] = x;
        k[t++] = y-1;
    }
    if ((x-1>=0) && (cb[(x-1)+y*LFS]==GRAY)) {
        k[t++] = x-1;
        k[t++] = y;
    }
    if ((y+1<FS) && (cb[x+(y+1)*LFS]==GRAY)) {
        k[t++] = x;
        k[t++] = y+1;
    }

    /* there will be at most two paths (?) */
    for (r=0; r<t/2; ++r) {

        /* select candidate */
        a = x;
        b = y;
        if (t >= 2*r+2) {
            c = k[2*r];
            d = k[2*r+1];
        }
        else
            return(0);

        /* build path to (u,v) */
        for (l=1; (l<MP) && (((c!=u) || (d!=v)) && ((c!=x) || (d!=y))); ++l) {

            /* compute the next pixel */
            if ((c+1<LFS) && ((c+1!=a) || (d!=b)) && (cb[(c+1)+d*LFS]==GRAY)) {
                a = c;
                b = d;
                c = c+1;
            }
            else if ((d-1>=0) && ((c!=a) || (d-1!=b)) && (cb[c+(d-1)*LFS]==GRAY)) {
                a = c;
                b = d;
                d = d-1;
            }
            else if ((c-1>=0) && ((c-1!=a) || (d!=b)) && (cb[(c-1)+d*LFS]==GRAY)) {
                a = c;
                b = d;
                c = c-1;
            }
            else if ((d+1<FS) && ((c!=a) || (d+1!=b)) && (cb[c+(d+1)*LFS]==GRAY)) {
                a = c;
                b = d;
                d = d+1;
            }
        }

        /* (x,y) reached: there is no path */
        if ((c==x) && (d==y))
            return(0);

        /* path is short: return true */
        if ((c==u) && (d==v) && (l < MP))
            return(1);
    }

    /* there is no short path: return false */
    return(0);
}

/*

Computes the skeleton of the bitmap stored on the
buffer cb. The skeleton pixels are painted BLACK. The others
(non-white) are painted GRAY.

Also computes the leftmost, rightmost, topmost and bottomost
coordinates of skeleton pixels (the horizontal coordinate of the
leftmost|righmost|topmost|bottomost skeleton pixel is stored on
fc_bp|lc_bp|fl_bp|ll_bp).

For algorithms 0, 1 and 2, if (i0,j0) is a valid pixel coordinate
of the buffer cb, then use it as a seed and computes only the
skeleton of the component that contains (i0,j0).

For algorithms 0, 1 and 2, if (i0==-2) then compute only the
skeleton of the components that belong to the symbol j0. In this
case, assumes that cb is aligned at the top and left limits of
symbol j0.

In all other cases compute the closure of all components found on
the buffer cb.

W and H may be used to make this service faster. By default use
W=0 and H=0. To achieve time savings, pass on W and H the
width and heigth of the bitmap.

*/
void skel(int i0,int j0,int W,int H)
{
    char cb3[FS*LFS];
    int i,j,x,y;

    /*
    printf("SA=%d\n",SA);
    printf("RR=%1.2f\n",RR);
    printf("MA=%1.2f\n",MA);
    printf("MP=%d\n",MP);
    printf("ML=%1.2f\n",ML);
    printf("MB=%d\n",MB);
    printf("RX=%d\n",RX);
    printf("BT=%d\n\n",BT);
    fflush(stdout);
    */

    /* by default analyse the entire bitmap */
    if ((!shape_opt) || (W == 0))
        W = LFS;
    if ((!shape_opt) || (H == 0))
        H = FS;

    /* limits for black pixels */
    fc_bp = FS;
    lc_bp = -1;
    fl_bp = FS;
    ll_bp = -1;

    /*
        Algorithm 6
        -----------
    */
    if (SA == 6) {
        int d;

        for (i=0; i<LFS; ++i) {
            for (j=0; j<FS; ++j) {
                if (cb[i+j*LFS] == BLACK)
                    cb3[i+j*LFS] = GRAY;
                else
                    cb3[i+j*LFS] = WHITE;
            }
        }

        cb_border(W,H);

        for (d=1; d; ) {


            for (i=0; i<FS; ++i) {
                for (j=0; j<FS; ++j) {
                    if (cb[i+j*LFS] == GRAY)
                        cb[i+j*LFS] = WHITE;
                }
            }

            cb_border(W,H);

            for (i=d=0; i<FS; ++i) {
                for (j=0; j<FS; ++j) {
                    if (cb[i+j*LFS] == GRAY) {
                        int ia,ib,ja,jb;

                        ia = i-1;
                        ib = i+1;
                        ja = j-1;
                        jb = j+1;

                        if (((i==0) || (cb[ia+j*LFS] != BLACK)) &&
                            ((ib==FS) || (cb[ib+j*LFS] != BLACK)) &&
                            ((j==0) || (cb[i+ja*LFS] != BLACK)) &&
                            ((jb==FS) || (cb[i+jb*LFS] != BLACK)) &&
                            ((i==0) || (j==0) || (cb[ia+ja*LFS] != BLACK)) &&
                            ((i==0) || (jb==FS) || (cb[ia+jb*LFS] != BLACK)) &&
                            ((ib==FS) || (ja==0) || (cb[ib+ja*LFS] != BLACK)) &&
                            ((ib==FS) || (jb==FS) || (cb[ib+jb*LFS] != BLACK))) {

                            cb3[i+j*LFS] = BLACK;

                        }
                        cb[i+j*LFS] = WHITE;
                    }
                    else if (cb[i+j*LFS] == BLACK) {
                        d = 1;
                    }
                }
            }


        }

        memcpy(cb,cb3,FS*LFS);

        /* compute limits */
        for (y=0; y<FS; ++y) {
            char *p;

            p = cb + y*LFS;
            for (x=0; x<FS; ++x,++p) {
                if (*p == BLACK) {
                    if (fc_bp >= FS)
                        fc_bp = x;
                    lc_bp = x;
                    if (fl_bp >= FS)
                        fl_bp = y;
                    ll_bp = y;
                }
            }
        }

        return;
    }

    /*
        Algorithm 5
        -----------

        Based on the distance from each pixel to the border.
    */
    if (SA == 5) {
        int f,t;

        /* compute the border */
        cb_border(W,H);

        /*
            Skeleton pixels are scored 0, other bitpmap pixels are
            scored -1 and white pixels are scored -2.
        */
        for (i=0; i<FS; ++i) {
            for (j=0; j<FS; ++j) {
                if (cb[i+j*LFS] == GRAY)
                    cb[i+j*LFS] = 0;
                else if (cb[i+j*LFS] == BLACK)
                    cb[i+j*LFS] = -1;
                else
                    cb[i+j*LFS] = -2;
            }
        }

        /*
            Computes the "distance" from each bitmap pixel to the
            border, for some definition of "distance".
        */
        for (t=0, f=1; f; ++t) {
            f = 0;
            for (i=0; i<FS; ++i) {
                for (j=0; j<FS; ++j) {
                    if (cb[i+j*LFS] == -1) {
                        if (((i>0) && (cb[(i-1)+j*LFS]==t)) ||
                            ((i+1<LFS) && (cb[(i+1)+j*LFS]==t)) ||
                            ((j>0) && (cb[i+(j-1)*LFS]==t)) ||
                            ((j+1<LFS) && (cb[i+(j+1)*LFS]==t))) {

                            cb[i+j*LFS] = t+1;
                            f = 1;
                        }
                    }
                }
            }
        }

        /* store the distances */
        memcpy(cb2,cb,LFS*FS);

        for (i=0; i<FS; ++i) {
            for (j=0; j<FS; ++j) {
                int a,pa,ua,b,pb,ub,m,n,t=3,v;

                if ((m = v = cb2[i+j*LFS]) >= 0) {

                    /*
                        Computee the limits of a rectangle around (i,j).
                        BUG: the magic number t==3 is hardcoded.
                    */
                    if ((pa = i-t) < 0)
                        pa = 0;
                    if ((ua = i+t) >= FS)
                        ua = FS-1;
                    if ((pb = j-t) < 0)
                        pb = 0;
                    if ((ub = j+t) >= FS)
                        ub = FS-1;

                    /* compute the maximum distance within the rectangle */
                    for (a = pa; a <= ua; ++a) {
                        for (b = pb; b <= ub; ++b) {

                            if ((n=cb2[a+b*LFS]) > m)
                                m = n;
                        }
                    }

                    /*
                        Mark (i,j) as a skeleton pixel if its distance
                        is large enough or if it is close to the local
                        maximum.
                    */
                    if (((v <= BT) && (m == v)) ||
                        ((v > BT) && (v+1 >= m)) ||
                        (v >= RX))

                        cb[i+j*LFS] = -1;
                }
            }
        }

        /* make skeleton pixels black */
        for (j=0; j<FS; ++j) {
            for (i=0; i<FS; ++i) {
                if (cb[i+j*LFS] == -2)
                    cb[i+j*LFS] = WHITE;
                else if (cb[i+j*LFS] == -1)
                    cb[i+j*LFS] = BLACK;
                else
                    cb[i+j*LFS] = GRAY;
            }
        }

        /* compute limits */
        for (y=0; y<FS; ++y) {
            char *p;

            p = cb + y*LFS;
            for (x=0; x<FS; ++x,++p) {
                if (*p == BLACK) {
                    if (fc_bp >= FS)
                        fc_bp = x;
                    lc_bp = x;
                    if (fl_bp >= FS)
                        fl_bp = y;
                    ll_bp = y;
                }
            }
        }

        return;
    }

    /*
        Algorithm 4
        -----------

        Growing lines heuristic.

        The considered angles are numbered from 0 to 15. All paths can
        be performed alternating the steps 1 and 2.

            number angle step 1 step 2
            ------+-----+------+------
               0      0   1, 0   1, 0
               1     27   1, 0   1,-1
               2     45   1,-1   1,-1
               3     63   0,-1   1,-1
               4     90   0,-1   0,-1
               5    117   0,-1  -1,-1 
               6    135  -1,-1  -1,-1 
               7    153  -1, 0  -1,-1
               8    180  -1, 0  -1, 0
               9    207  -1, 0  -1, 1
              10    225  -1, 1  -1, 1
              11    243   0, 1  -1, 1
              12    270   0, 1   0, 1
              13    287   0, 1   1, 1
              14    315   1, 1   1, 1
              15    333   1, 0   1, 1

        For instance: starting from (23,14) the first 5 pixels on the
        path number 9 are:

              (23,14)
              (22,14) = (23,14) + (-1,0)
              (21,13) = (22,14) + (-1,-1)
              (20,13) = (21,13) + (-1,0)
              (19,12) = (20,13) + (-1,-1)

    */
    if (SA == 4) {

        /* results, centers and distances */
        int R[16],C[16],D[16];

        /* the steps */
        int dx1[16] = { 1, 1, 1, 0, 0, 0,-1,-1,-1,-1,-1, 0, 0, 0, 1, 1};
        int dy1[16] = { 0, 0,-1,-1,-1,-1,-1, 0, 0, 0, 1, 1, 1, 1, 1, 0};
        int dx2[16] = { 1, 1, 1, 1, 0,-1,-1,-1,-1,-1,-1,-1, 0, 1, 1, 1};
        int dy2[16] = { 0,-1,-1,-1,-1,-1,-1,-1, 0, 1, 1, 1, 1, 1, 1, 1};

        /* other */
        int c,c1,c2,f,g,n,r,s,u,v;

        /* must remember original bitmap */
        memcpy(cb3,cb,LFS*FS);

        /* test each pixel */
        for (i=0; i<LFS; ++i) {
            for (j=0; j<FS; ++j) {

                /* ignore non-BLACK pixels */
                if (cb3[i+j*LFS] != BLACK)
                    continue;

                /* draw each line */
                for (f=r=0; r<16; ++r) {

                    /* try RX steps */
                    for (g=1, s=0, u=i, v=j; g && (s<RX); ++s) {

                        /* next step */
                        if (s & 1) {
                            u += dx2[r];
                            v += dy2[r];
                        }
                        else {
                            u += dx1[r];
                            v += dy1[r];
                        }

                        /* failure to extend */
                        if ((u < 0) || (LFS <= u) ||
                            (v < 0) || (FS <= v) ||
                            cb3[u+v*LFS] != BLACK) {

                            g = 0;
                            ++f;
                        }
                    }

                    /* result for angle r */
                    R[r] = g;
                    D[r] = s-1;
                }

                /* centers of inscribed circles are skeleton pixels */
                if (f == 0)
                    continue;

                /* discard pixels with no successfull path */
                if (f == 16) {
                    cb[i+j*LFS] = GRAY;
                    continue;
                }

                /* initialize the centers */
                for (r=0; r<16; ++r)
                    C[r] = 0;

                /*
                    Search the angles that are centers of maximal
                    arrays of consecutive failures.
                */
                for (c1=c2=-1, n=0, r=15, g=-1; r>g; --r) {

                    if (R[r])
                        continue;

                    /*
                        Now that we know that for angle r we've failed to
                        draw the line, let's try to  extend the failure
                        clockwise and counterclockwise, finding the limits
                        u and v:

                                u r  v    0
                           1111100000011111

                               ------>
                              clockwise
                    */

                    /* extend the failure conter-clockwise */
                    for (u=r; R[s=(u==15)?0:u+1] == 0; u=s);
		    
                    /*
                        Extend the failure clockwise.
		    
                        As we're starting with r=15, the v limit will never
                        cross the boundary from angle 0 to angle 15, because
                        if R[0] == R[15] == 0 happens, then on the first
                        failure (that is, r==15), the u limit already crossed
                        the boundary from 15 to 0.
                    */
                    for (v=r; (v>0) && (R[v-1] == 0); --v);
		    
                    /* choose center */
                    if (r <= u) {
                        c = (u+v) / 2;
                        C[c] = u-v+1;
                    }
                    else {
		    
                        /* (no need to visit again the entries from 0 to u) */
                        g = u;
		    
                        /* translate, compute center and translate center */
                        u += 16;
                        c = (u+v) / 2;
                        if (c >= 16)
                            c -= 16;
                        C[c] = u-v+1;
                    }

                    /* account centers */
                    if (++n == 1)
                        c1 = c;
                    else if (n == 2)
                        c2 = c;

                    /* prepare next search */
                    r = v-2;
                }

                /*
                    1. If we have only one failure, than the
                    pixel is on the extreme portion of
                    one not-thin component. Such pixels are not
                    considered skeleton pixels, unless the failure
                    covers at most three angles. This exception
                    tries to cover those cases where the component
                    has width 2*RX. In this extreme case, all
                    pixels will be rejected unless we admit that
                    exception.
                */
                if (n == 1) {
                    if (C[c1] > 3)
                        cb[i+j*LFS] = GRAY;
                }

                /*
                    2. If we have just two large opposed failures,
                    then the pixel is on a thin portion
                    of the component. Such pixels are considered
                    skeleton pixels. However, we try to discard some
                    pixels when the component width is exactly 2.
                */
                else if (n == 2) {

                    /* width 2 */
                    if ((D[c1] == 1) && (D[c2] == 0))
                        cb[i+j*LFS] = GRAY;
                }

                /*
                    All other cases are discarded by now.
                */
                else {
                    cb[i+j*LFS] = GRAY;
                }
            }
        }

        /* compute limits */
        for (y=0; y<FS; ++y) {
            char *p;

            p = cb + y*LFS;
            for (x=0; x<FS; ++x,++p) {
                if (*p == BLACK) {
                    if (fc_bp >= FS)
                        fc_bp = x;
                    lc_bp = x;
                    if (fl_bp >= FS)
                        fl_bp = y;
                    ll_bp = y;
                }
            }
        }

        return;
    }

    /*
        Algorithm 3
        -----------

        Compute skeleton removing the boundary BT times
    */
    if (SA == 3) {
        int n;
        char *p;

        /* must remember original bitmap */
        memcpy(cb3,cb,LFS*FS);

        for (n=0; n<BT; ++n) {

            /* make the boundary gray */
            cb_border(W,H);

            /* make the boundary white */
            for (i=0; i<LFS; ++i)
                for (j=0; j<FS; ++j)
                    if (cb[i+j*LFS] == GRAY)
                        cb[i+j*LFS] = WHITE;
        }

        /* make the non-skeleton pixels gray */
        for (i=0; i<LFS; ++i)
            for (j=0; j<FS; ++j)
                if ((cb3[i+j*LFS] == BLACK) && (cb[i+j*LFS] == WHITE))
                    cb[i+j*LFS] = GRAY;

        /* compute limits */
        for (y=0; y<FS; ++y) {
            p = cb + y*LFS;
            for (x=0; x<FS; ++x,++p) {
                if (*p == BLACK) {
                    if (fc_bp >= FS)
                        fc_bp = x;
                    lc_bp = x;
                    if (fl_bp >= FS)
                        fl_bp = y;
                    ll_bp = y;
                }
            }
        }

        return;
    }

    /*

        Here begins algorithms 0, 1 and 2. Both depend on the
        computation of the border. Border pixels are used by
        the heuristics that build the skeleton. So we need to
        compute the border of all components on the buffer cb.

        Alternatively, we can compute the border only of those
        components to be considered. So skel() offers three
        options:

    */

    /*
        1. compute the closure boundary only of the component
           specified by the seed (i0,j0).
    */
    if ((0<=i0) && (i0<LFS) && (0<=j0) && (j0<FS)) {
        memcpy(cb2,cb,LFS*FS);
        border(i0,j0);
    }

    /*
        2. compute the closure boundary of all components of
           the symbol j0.
    */
    else if (i0 == -2) {
        sdesc *m;
        cldesc *c;
        int n;

        memcpy(cb2,cb,LFS*FS);
        m = mc + j0;
        for (n=0; n < m->ncl; ++n) {
            c = cl + m->cl[n];
            border(c->seed[0]-c->l,c->seed[1]-c->t);
        }
    }

    /*
        3. compute the border of all components.
    */
    else {
        cb_border(W,H);
    }

    /* clear the buffers */
    memset(cb2,WHITE,LFS*FS);
    memset(cb3,0,LFS*FS);

    /* for each black pixel on the board */
    for (y=0, x=0; y<H; ((void)((++x<W) || (x=0,++y)))) {
        int r,rmin,rc;
        float d,dmin,ma;
        int topbp;
        int umin,vmin;
        int a,u,v;

        /* ignore non-black pixels */
        if (cb[x+y*LFS] != BLACK)
            continue;

        /* maximum size for a r-square */
        if (y < (r = x))
            r = y;
        if ((x + r) >= LFS)
            r = LFS-x-1;
        if ((y + r) >= FS)
            r = FS-y-1;
        if (r > RX)
            r = RX;
	
        /*
            Search the gray pixels around (x,y) using squares
            centered at (x,y).
        */
        dmin = r + 1;
        umin = vmin = r+1;
        topbp = 0;
        rmin = 1;

        /*
            For each pixel (x,y),
            remembers the size of the largest black square centered
            at (x,y) as cb3[x+y*LFS]. This information may be used
            to save time when the pixels (x+1,y) or (x,y+1) are
            processed, if the square optimization is on.
        */
        if (square_opt != 0) {
            if ((x > 0) && ((a=cb3[(x-1)+y*LFS]-2) > rmin))
                rmin = a;
            if ((y > 0) && ((a=cb3[x+(y-1)*LFS]-2) > rmin))
                rmin = a;
        }

        for (rc=rmin; (rc<=r) && (rc-dmin<=RR); ++rc) {
            char *pp,*pl;
            int mu,mv;

            /* top and bottom */
            mu = rc;
            pp = cb + x - mu + (y-rc)*LFS;
            pl = cb + x - mu + (y+rc)*LFS;
            for (u=-mu; (u<=mu); ++u,++pp,++pl) {
                if ((d = rmdist[u+rc*RX2]) < dmin+RR) {
                    if (*pp == GRAY) {
                        if (d < dmin)
                            dmin = d, umin = abs(u), vmin = rc;
                        xbp[topbp] = u, ybp[topbp] = -rc, bpd[topbp++] = d;
                    }
                    if (*pl == GRAY) {
                        if (d < dmin)
                            dmin = d, umin = abs(u), vmin = rc;
                        xbp[topbp] = u, ybp[topbp] = rc, bpd[topbp++] = d;
                    }
                }
            }

            /* left and right */
            mv = rc - 1;
            pp = cb + x - rc + (y-mv)*LFS;
            pl = cb + x + rc + (y-mv)*LFS;
            for (v=-mv; v<=mv; ++v,pp+=LFS,pl+=LFS) {
                if ((d = rmdist[rc+v*RX2]) < dmin+RR) {
                    if (*pp == GRAY) {
                        if (d < dmin)
                            dmin = d, umin = rc, vmin = abs(v);
                        xbp[topbp] = -rc, ybp[topbp] = v, bpd[topbp++] = d;
                    }
                    if (*pl == GRAY) {
                        if (d < dmin)
                            dmin = d, umin = rc, vmin = abs(v);
                        xbp[topbp] = rc, ybp[topbp] = v, bpd[topbp++] = d;
                    }
                }
            }
        }

        /* remember */
        cb3[x+y*LFS] = (umin > vmin) ? umin-1 : vmin-1;

        /*
            If there exists a black square of size 2*BT+1 centered at
            (x,y), then the pixel is automatically considered a
            skeleton pixel.
        */
        if (dmin > BT)
            ma = MA;

        /* Otherwise the BPs (boundary pixels) will be analysed */
        else {

            /*
                remove pixels at distance from (x,y) outside
                the interval [dmin-RR,dmin+RR].
            */
            ma = 0;
            for (i=j=0; i<topbp; ++i)
                if (fabs(dmin-bpd[i]) <= RR) {
                    if (i != j) {
                        xbp[j] = xbp[i];
                        ybp[j] = ybp[i];
                        bpd[j] = bpd[i];
                    }
                    ++j;
                }
            topbp = j;

            /*
                Algorithm 2
                -----------

                algorithm 2 means "accept those pixels with at
                least MB BPs". Faster but less accurate.
            */
            if (SA == 2) {
                if (topbp >= MB)
                    ma = MA;
            }

            /*
                locate the pairs of BPs (i,j) where the distance
                from i to j is "large" check if their angular
                distance is greater then MA and if there is not
                a small path joining them.
            */
            else if (topbp > 2) {
                float d2=dmin*dmin;

                /* for each pair of BPs */
                for (i=0, j=1;
                    (ma<MA) && (i+1<topbp);
                    ((void)((++j<topbp) || (j=++i+1)))) {

                    /*
                        Algorithm 1
                        -----------

                        Ignore if the linear distance is too small.
                    */
                    if (SA == 1) {
                        u = xbp[i] - xbp[j];
                        v = ybp[i] - ybp[j];
                        if ((u*u + v*v) < ML*d2)
                            continue;
                        ma = MA;
                    }

                    /*
                        Algorithm 0
                        -----------

                        Ignore if the angular distance is too small.
                    */
                    else {
                        if ((ma = fabs(ro(xbp[i],ybp[i])-ro(xbp[j],ybp[j]))) > PI)
                            ma = 2*PI - ma;
                        if (ma < MA)
                            continue;
                    }

                    if ((debug) &&
                        (joined(x+xbp[i],y+ybp[i],x+xbp[j],y+ybp[j]) !=
                         joined(x+xbp[j],y+ybp[j],x+xbp[i],y+ybp[i]))) {

                        db("assymetric behaviour of joined");
                    }

                    /* ignore if there is a small path joining them */
                    if ((joined(x+xbp[i],y+ybp[i],x+xbp[j],y+ybp[j]) != 0) ||
                        (joined(x+xbp[j],y+ybp[j],x+xbp[i],y+ybp[i]) != 0))
                        ma = 0;
                }
            }
        }

        /* accept pixel */
        if (ma >= MA)
            cb2[x+y*LFS] = BLACK;
    }

    /*
        Finally, make all skeleton pixels BLACK. Skeleton pixels
        are currently painted BLACK on the cb2 auxiliar buffer.

        Pixels from cb that originally were BLACK but do not
        belong to the skeleton are painted GRAY.
    */
    for (y=0; y<H; ++y) {
        for (x=0; x<W; ++x) {
            if (cb2[x+y*LFS] == BLACK) {

                /* skeleton pixel */
                cb[x+y*LFS] = BLACK;

                /* adjust limits for black pixels */
                if (x < FS) {
                    if (x < fc_bp)
                        fc_bp = x;
                    if (x > lc_bp)
                        lc_bp = x;
                    if (y < fl_bp)
                        fl_bp = y;
                    if (y > ll_bp)
                        ll_bp = y;
                }
            }

            /* non-skeleton non-white pixel */
            else if (cb[x+y*LFS] == BLACK)
                cb[x+y*LFS] = GRAY;
        }
    }
}

/*

Rate the skeleton quality for pattern p.

The skeleton quality is defined by the code.

*/
int skel_quality(int p)
{
    pdesc *d;
    int cx,cy,i,j,t,x,y;
    int S,C;
    int f,bp,sp,bad;

    /* prepare */
    d = pattern + p;
    x = d->sx;
    y = d->sy;

    /*
        Clearance
        ---------
    */
    cx = d->bw - d->sw;
    cy = d->bh - d->sh;
    if ((cx <= 0) || (cy <= 0) || (10 <= cx) || (10 <= cy))
        return(0);
    if ((cx <= 2) || (cy <= 2) || (8 <= cx) || (8 <= cy))
        C = 8;
    else
        C = 10;


    /*
        Skeleton coverage
        -----------------
    */

    /* copy the bitmap and computes the border */
    bp = bm2cb(d->b,0,0);
    cb_border(0,0);

    /*
        Border pixels are scored 0, other bitpmap pixels are
        scored -1 and white pixels are scored -2.
    */
    for (i=0; i<d->bw; ++i) {
        for (j=0; j<d->bh; ++j) {
            if (cb[i+j*LFS] == GRAY)
                cb[i+j*LFS] = 0;
            else if (cb[i+j*LFS] == BLACK)
                cb[i+j*LFS] = -1;
            else
                cb[i+j*LFS] = -2;
        }
    }

    /*
        Computes the "distance" of each bitmap pixel to the
        border, for some definition of "distance".
    */
    for (f=1, t=0; f && (t<127); ++t) {
        f = 0;
        for (i=0; i<d->bw; ++i) {
            for (j=0; j<d->bh; ++j) {
                if (cb[i+j*LFS] == -1) {
                    if (((i>0) && (cb[(i-1)+j*LFS]==t)) ||
                        ((i+1<LFS) && (cb[(i+1)+j*LFS]==t)) ||
                        ((j>0) && (cb[i+(j-1)*LFS]==t)) ||
                        ((j+1<LFS) && (cb[i+(j+1)*LFS]==t))) {

                        cb[i+j*LFS] = t+1;
                        f = 1;
                    }
                }
            }
        }
    }

    /* dump map of distances (for debug) */
    /*
    {
        for (j=0; j<d->bh; ++j) {
            for (i=0; i<d->bw; ++i) {
                if (cb[i+j*LFS] < 0)
                    putchar(' ');
                else if ((0 <= cb[i+j*LFS]) && (cb[i+j*LFS] <= 9))
                    putchar('0'+cb[i+j*LFS]);
                else
                    putchar('*');
            }
            putchar('\n');
        }
    */

    /* Analyse skeleton pixels */
    memcpy(cb2,cb,LFS*FS);
    sp = bm2cb(d->s,x,y);
    bad = 0;
    for (i=0; i<d->bw; ++i) {
        for (j=0; j<d->bh; ++j) {
            int a,pa,ua,b,pb,ub,m,n,t=3,v;

            m = v = cb[i+j*LFS];
            if ((cb[i+j*LFS] == BLACK) && (cb2[i+j*LFS] < 3)) {

                if ((pa = i-t) < 0)
                    pa = 0;
                if ((ua = i+t) >= FS)
                    ua = FS-1;
                if ((pb = j-t) < 0)
                    pb = 0;
                if ((ub = j+t) >= FS)
                    ub = FS-1;

                for (a = pa; a <= ua; ++a) {
                    for (b = pb; b <= ub; ++b) {

                        if ((n=cb2[a+b*LFS]) > m)
                            m = n;
                    }
                }

                if (((v<=1) && (m>v)) ||
                    ((v>1) && ((m>v+1)))) {

                    ++bad;
                }
            }

            else if ((cb[i+j*LFS] != BLACK) && (cb2[i+j*LFS] >= 4)) {

                if (((i>0) && (cb[(i-1)+j*LFS] != BLACK)) &&
                    ((i+1<LFS) && (cb[(i+1)+j*LFS] != BLACK)) &&
                    ((j>0) && (cb[i+(j-1)*LFS] != BLACK)) &&
                    ((j+1<LFS) && (cb[i+(j+1)*LFS] != BLACK))) {

                    ++bad;
                }
            }

        }
    }

    /* estimate the coverage */
    if (10*sp < bp) {
        S = 0;
    }
    else if (2*sp > bp) {
        S = 0;
    }
    else if (bad*10 >= bp) {
        S = 0;
    }
    else if (bad*40 < bp) {
        S = 10;
    }
    else {
        S = ((bp/bad)-6)/4;
    }

    /* final result */
    return(S+C);
}

/*


*/
int tune_skel(int p)
{
    static int st=0,c=0,bc,bv;
    int v;

    static short C[] = {

       /* SA   RR   MA   MP   ML   MB   RX   BT */

           0,  14, 157,  10,  38,  10,   2,   2,
           0,  14, 157,  10,  38,  10,   3,   3,

           1,  14, 157,  10,  38,  10,   2,   2,
           1,  14, 157,  10,  38,  10,   3,   3,

           2,  14, 157,  10,  38,  10,   2,   2,
           2,  14, 157,  10,  38,  10,   3,   3,

           3,   0,   0,   0,   0,   0,   0,   2,
           3,   0,   0,   0,   0,   0,   0,   3,

           4,   0,   0,   0,   0,   0,   3,   0,
           4,   0,   0,   0,   0,   0,   4,   0,

           5,   0,   0,   0,   0,   0,   3,   2,
           5,   0,   0,   0,   0,   0,   3,   3,
           5,   0,   0,   0,   0,   0,   4,   2,
           5,   0,   0,   0,   0,   0,   4,   3,
           5,   0,   0,   0,   0,   0,   4,   4,

          -1
    };

    /* reset */
    if (st == 0) {
        st = 1;
        bc = c = 0;
        bv = 0;
    }

    /* try next candidate */
    else if (st == 1) {

        if (C[c] >= 0) {

            SA = C[c++];
            RR = ((float)C[c++]) / 10;
            MA = ((float)C[c++]) / 100;
            MP = C[c++];
            ML = ((float)C[c++]) / 10;
            MB = C[c++];
            RX = C[c++];
            BT = C[c++];

            printf("trying %d %f %f %d %f %d %d %d\n",SA,RR,MA,MP,ML,MB,RX,BT);

            v = skel_quality(p);

            spcpy(SP_GLOBAL,SP_DEF);

            if (v > bv) {
                bc = c-8;
                bv = v;
            }

            /* bug: rendering of TUNE_PATTERN will call skel_quality again */
            if (*cm_v_st != ' ') {
                dw[TUNE_PATTERN].rg = 1;
                dw[TUNE_SKEL].rg = 1;
                redraw_dw = 1;
            }
        }

        /* finish */
        else {

            /* store best parameters */
            pattern[p].p[0] = C[bc++];
            pattern[p].p[1] = C[bc++] * 10;
            pattern[p].p[2] = C[bc++] * 100;
            pattern[p].p[3] = C[bc++];
            pattern[p].p[4] = C[bc++] * 10;
            pattern[p].p[5] = C[bc++];
            pattern[p].p[6] = C[bc++];
            pattern[p].p[7] = C[bc++];

            st = 0;
            return(0);
        }
    }

    return(1);
}

