/*
 *   This file is part of Clinica.
 *
 *   Clinica is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   Clinica is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with Clinica.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Authors: Leonardo Robol <leo@robol.it>
 *            Gianmarco Brocchi <brocchi@poisson.phc.unipi.it>
 */
 
 using Gtk;
 
 namespace Clinica {
    
    public class EventListStore : ListStore {
    
        public signal void event_added   (int event_id);
        public signal void event_removed (int event_id);
    
        /**
         * @brief Fields present in the store, i.e. header of
         * the columns.
         */
        enum Field {
            EVENT,
            TITLE,
            DESCRIPTION,
            VENUE,
            PATIENT,
            VISIT,
            DATE,
        }
    
        /**
         * @brief This signal is emitted if an error occur in the
         * list_store
         */
        public signal void error (string message);
        
        /**
         * @brief ResourceManager associated to this clinica application
         * that will be used to retrieve various informationa about the
         * other components running
         */
        private ResourceManager resource_manager { get; set; }
    
        public EventListStore (ResourceManager resources) {
            resource_manager = resources;
            error.connect ((widget, message) => resource_manager.error_callback (widget, message));
            
            Type [] column_headers = { typeof (Event), typeof (string), typeof (string), 
                typeof (string), typeof (Patient?), typeof (Visit?), typeof (DateTime) };
                
            set_column_types (column_headers);
            GLib.Idle.add (load_events_from_db);
        }
        
        /**
         * @brief Add an event to the liststore.
         */
        public void add_event (Event event) {
            TreeIter iter;
            append (out iter);
            
            /* Set values in the treeview */
            set_value (iter, Field.EVENT, event);
            set_value (iter, Field.DATE, event.date);
            set_value (iter, Field.DESCRIPTION, event.description);
            set_value (iter, Field.PATIENT, event.patient);
            set_value (iter, Field.TITLE, event.title);
            set_value (iter, Field.VENUE, event.venue);
            set_value (iter, Field.VISIT, event.visit);
            
            /* Emit event_added to make calendars and other pieces of interface
             * using events know that they must reload their dates */
            event_added (event.get_id ());
        }
        
        public void remove_event (Event event) {
            TreeIter iter;
            Value value;
            if (!get_iter_first (out iter)) {
                error (_("Events database seems corrupted. This is likely to be a bug in the application"));
            }
            
            do {
                get_value (iter, Field.EVENT, out value);
                if ((value as Event).get_id () == event.get_id ()) {
                    /* Delete this event from the liststore */
                    int event_id = event.get_id ();
                    remove (iter);
                    event_removed (event_id);
                    return;
                }
            } while (iter_next (ref iter));
            
            assert_not_reached ();
        }
        
        /**
         * @brief Function to be schedule with GLib.Idle_add () when loading of
         * the events from the db is needed at the start of the application.
         *
         * This function is now called by default from the constructor.
         */
        private bool load_events_from_db () {
            foreach (var event in Event.all (resource_manager)) {
                add_event (event);
            }
            
            return false;
        }
    }
 }
