/*
 *   This file is part of Clinica.
 *
 *   Clinica is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   Clinica is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with Clinica.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Authors: Leonardo Robol <leo@robol.it>
 *            Gianmarco Brocchi <brocchi@poisson.phc.unipi.it>
 */

using Gtk;

namespace Clinica {

    public class PatientListPage : Alignment, Page {
    
        public  PatientListView view;
        private Alignment sidebar;
        private VBox left_vbox;
        private ScrolledWindow scrolled_window;
        
        public ResourceManager resource_manager { get; set; }
        
        /**
         * @brief PatientListPage is the Page loaded when the user
         * asks for a list of the Patients in the database 
         */
        public PatientListPage (ResourceManager resources) {
            var hbox = new HBox (false, 6);
            resource_manager = resources;
            
            connect_signal ();
            
            /* Load files from the XML */
            var builder = new Builder ();
            
            try {
                builder.add_from_file (resource_manager.get_ui_file ("patient_list_toolbar.glade"));
                builder.add_from_file (resource_manager.get_ui_file ("patient_list_sidebar.glade"));
            } catch (Error e) {
                error ("Error opening patient_list_toolbar.glade. Check you installation.\n");
            }
            
            left_vbox = builder.get_object ("left_vbox") as VBox;
            sidebar = builder.get_object ("sidebar") as Alignment;
            scrolled_window = builder.get_object ("treeview_scrolled_window") as ScrolledWindow;
            
            view = new PatientListView (resource_manager, builder.get_object ("find_entry") as Entry);
            scrolled_window.add (view);
            
            hbox.pack_start (left_vbox);
            hbox.pack_start (sidebar, false);
            add (hbox);
            show_all ();
            
            /* Connect XML specified signals */
            builder.connect_signals (this);
        }
        
        /**
         * @brief Routine called for initialization
         */
        public void setup () {
            new_title (_("Patients"));
        }
        
        /**
         * @brief Callback for the back_button 
         */
        [CCode (instance_pos = -1)]
        public void on_back_button_clicked (Button button) {
            start_page_requested ();
        }
        
        /**
         * @brief Callback for the remove_button
         */
        [CCode (instance_pos = -1)]
        public void on_remove_button_clicked (Button button) {
        	view.remove_selected_patient ();
        }
        
        /**
         * @brief Callback for the add_button
         */
        [CCode (instance_pos = -1)]
        public void on_add_button_clicked (Button button) {
        	var new_patient_dialog = new PatientEditor (resource_manager);
        	new_patient_dialog.dialog.set_transient_for (resource_manager.user_interface.window);
        	new_patient_dialog.run ();
      		new_patient_dialog.destroy ();
        }
        
        [CCode (instance_pos = -1)]
        public void on_modify_button_clicked (Button button) {
        	var patient = view.get_selected_patient ();
        	if (patient == null)
        		return;
        	
        	/*  Create the patient browser */
        	var edit_patient_dialog = new PatientEditor.with_patient (resource_manager, patient);
        	edit_patient_dialog.dialog.set_transient_for (resource_manager.user_interface.window);
        	edit_patient_dialog.run ();
        	edit_patient_dialog.destroy ();
        }
        
        /**
         * @brief Callback for the Edit visits button that opens the default
         * VisitBrowser for the selected patient.
         */
        [CCode (instance_pos = -1)]
        public void on_edit_visit_button_clicked (Button button) {	
        	var patient = view.get_selected_patient ();
        	if (patient == null) {
        		return;
        	}
        	
            resource_manager.user_interface.show_visit_window (patient);
        }
    }
    
}
