#include <arpa/inet.h>
#include <assert.h>
#include <ctype.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <net/if.h>
#include <sys/socket.h>
#include <map>
#include <string>
#include <utility>
#include <vector>

#include "acceptor.h"
#include "config.h"
#include "log.h"
#include "parse.h"

using namespace std;

#define DEFAULT_BACKLOG_SIZE 10485760

struct ConfigLine {
	string keyword;
	vector<string> arguments;
	map<string, string> parameters;
};

namespace {

bool parse_hostport(const string &hostport, sockaddr_in6 *addr)
{
	memset(addr, 0, sizeof(*addr));
	addr->sin6_family = AF_INET6;

	string port_string;

	// See if the argument if on the type [ipv6addr]:port.
	if (!hostport.empty() && hostport[0] == '[') {
		size_t split = hostport.find("]:");
		if (split == string::npos) {
			log(ERROR, "address '%s' is malformed; must be either [ipv6addr]:port or ipv4addr:port");
			return false;
		}

		string host(hostport.begin() + 1, hostport.begin() + split);
		port_string = hostport.substr(split + 2);

		if (inet_pton(AF_INET6, host.c_str(), &addr->sin6_addr) != 1) {
			log(ERROR, "'%s' is not a valid IPv6 address");
			return false;
		}
	} else {
		// OK, then it must be ipv4addr:port.
		size_t split = hostport.find(":");
		if (split == string::npos) {
			log(ERROR, "address '%s' is malformed; must be either [ipv6addr]:port or ipv4addr:port");
			return false;
		}

		string host(hostport.begin(), hostport.begin() + split);
		port_string = hostport.substr(split + 1);

		// Parse to an IPv4 address, then construct a mapped-v4 address from that.
		in_addr addr4;

		if (inet_pton(AF_INET, host.c_str(), &addr4) != 1) {
			log(ERROR, "'%s' is not a valid IPv4 address");
			return false;
		}

		addr->sin6_addr.s6_addr32[2] = htonl(0xffff);
		addr->sin6_addr.s6_addr32[3] = addr4.s_addr;
	}

	int port = atoi(port_string.c_str());
	if (port < 1 || port >= 65536) {
		log(ERROR, "port %d is out of range (must be [1,65536>).", port);
		return false;
	}
	addr->sin6_port = ntohs(port);

	return true;
}

bool read_config(const string &filename, vector<ConfigLine> *lines)
{
	FILE *fp = fopen(filename.c_str(), "r");
	if (fp == NULL) {
		log_perror(filename.c_str());
		return false;
	}

	char buf[4096];
	while (!feof(fp)) {
		if (fgets(buf, sizeof(buf), fp) == NULL) {
			break;
		}

		// Chop off the string at the first #, \r or \n.
		buf[strcspn(buf, "#\r\n")] = 0;

		// Remove all whitespace from the end of the string.
		size_t len = strlen(buf);
		while (len > 0 && isspace(buf[len - 1])) {
			buf[--len] = 0;
		}

		// If the line is now all blank, ignore it.
		if (len == 0) {
			continue;
		}

		vector<string> tokens = split_tokens(buf);
		assert(!tokens.empty());
		
		ConfigLine line;
		line.keyword = tokens[0];

		for (size_t i = 1; i < tokens.size(); ++i) {
			// foo=bar is a parameter; anything else is an argument.
			size_t equals_pos = tokens[i].find_first_of('=');
			if (equals_pos == string::npos) {
				line.arguments.push_back(tokens[i]);
			} else {
				string key = tokens[i].substr(0, equals_pos);
				string value = tokens[i].substr(equals_pos + 1, string::npos);
				line.parameters.insert(make_pair(key, value));
			}
		}

		lines->push_back(line);
	}

	if (fclose(fp) == EOF) {
		log_perror(filename.c_str());
		return false;
	}
	return true;
}

bool fetch_config_string(const vector<ConfigLine> &config, const string &keyword, string *value)
{
	for (unsigned i = 0; i < config.size(); ++i) {
		if (config[i].keyword != keyword) {
			continue;
		}
		if (config[i].parameters.size() > 0 ||
		    config[i].arguments.size() != 1) {
			log(ERROR, "'%s' takes one argument and no parameters", keyword.c_str());
			return false;
		}
		*value = config[i].arguments[0];
		return true;
	}
	return false;
}

bool fetch_config_int(const vector<ConfigLine> &config, const string &keyword, int *value)
{
	for (unsigned i = 0; i < config.size(); ++i) {
		if (config[i].keyword != keyword) {
			continue;
		}
		if (config[i].parameters.size() > 0 ||
		    config[i].arguments.size() != 1) {
			log(ERROR, "'%s' takes one argument and no parameters", keyword.c_str());
			return false;
		}
		*value = atoi(config[i].arguments[0].c_str());  // TODO: verify int validity.
		return true;
	}
	return false;
}

bool parse_port(const ConfigLine &line, Config *config)
{
	if (line.arguments.size() != 1) {
		log(ERROR, "'port' takes exactly one argument");
		return false;
	}

	int port = atoi(line.arguments[0].c_str());
	if (port < 1 || port >= 65536) {
		log(ERROR, "port %d is out of range (must be [1,65536>).", port);
		return false;
	}

	AcceptorConfig acceptor;
	acceptor.addr = create_any_address(port);

	config->acceptors.push_back(acceptor);
	return true;
}

bool parse_listen(const ConfigLine &line, Config *config)
{
	if (line.arguments.size() != 1) {
		log(ERROR, "'listen' takes exactly one argument");
		return false;
	}

	AcceptorConfig acceptor;
	if (!parse_hostport(line.arguments[0], &acceptor.addr)) {
		return false;
	}
	config->acceptors.push_back(acceptor);
	return true;
}

bool parse_stream(const ConfigLine &line, Config *config)
{
	if (line.arguments.size() != 1) {
		log(ERROR, "'stream' takes exactly one argument");
		return false;
	}

	StreamConfig stream;
	stream.url = line.arguments[0];

	map<string, string>::const_iterator src_it = line.parameters.find("src");
	if (src_it == line.parameters.end()) {
		log(WARNING, "stream '%s' has no src= attribute, clients will not get any data.",
		        stream.url.c_str());
	} else {
		stream.src = src_it->second;
		// TODO: Verify that the URL is parseable?
	}

	map<string, string>::const_iterator backlog_it = line.parameters.find("backlog_size");
	if (backlog_it == line.parameters.end()) {
		stream.backlog_size = DEFAULT_BACKLOG_SIZE;
	} else {
		stream.backlog_size = atoi(backlog_it->second.c_str());
	}

	// Parse encoding.
	map<string, string>::const_iterator encoding_parm_it = line.parameters.find("encoding");
	if (encoding_parm_it == line.parameters.end() ||
	    encoding_parm_it->second == "raw") {
		stream.encoding = StreamConfig::STREAM_ENCODING_RAW;
	} else if (encoding_parm_it->second == "metacube") {
		stream.encoding = StreamConfig::STREAM_ENCODING_METACUBE;
	} else {
		log(ERROR, "Parameter 'encoding' must be either 'raw' (default) or 'metacube'");
		return false;
	}

	// Parse the pacing rate, converting from kilobits to bytes as needed.
	map<string, string>::const_iterator pacing_rate_it = line.parameters.find("pacing_rate_kbit");
	if (pacing_rate_it == line.parameters.end()) {
		stream.pacing_rate = ~0U;
	} else {
		stream.pacing_rate = atoi(pacing_rate_it->second.c_str()) * 1024 / 8;
	}

	config->streams.push_back(stream);
	return true;
}

bool parse_udpstream(const ConfigLine &line, Config *config)
{
	if (line.arguments.size() != 1) {
		log(ERROR, "'udpstream' takes exactly one argument");
		return false;
	}

	UDPStreamConfig udpstream;

	string hostport = line.arguments[0];
	if (!parse_hostport(hostport, &udpstream.dst)) {
		return false;
	}

	map<string, string>::const_iterator src_it = line.parameters.find("src");
	if (src_it == line.parameters.end()) {
		// This is pretty meaningless, but OK, consistency is good.
		log(WARNING, "udpstream to %s has no src= attribute, clients will not get any data.",
		        hostport.c_str());
	} else {
		udpstream.src = src_it->second;
		// TODO: Verify that the URL is parseable?
	}

	// Parse the pacing rate, converting from kilobits to bytes as needed.
	map<string, string>::const_iterator pacing_rate_it = line.parameters.find("pacing_rate_kbit");
	if (pacing_rate_it == line.parameters.end()) {
		udpstream.pacing_rate = ~0U;
	} else {
		udpstream.pacing_rate = atoi(pacing_rate_it->second.c_str()) * 1024 / 8;
	}

	// Parse the TTL. The same value is used for unicast and multicast.
	map<string, string>::const_iterator ttl_it = line.parameters.find("ttl");
	if (ttl_it == line.parameters.end()) {
		udpstream.ttl = -1;
	} else {
		udpstream.ttl = atoi(ttl_it->second.c_str());
	}

	// Parse the multicast interface index.
	map<string, string>::const_iterator multicast_iface_it = line.parameters.find("multicast_output_interface");
	if (multicast_iface_it == line.parameters.end()) {
		udpstream.multicast_iface_index = -1;
	} else {
		udpstream.multicast_iface_index = if_nametoindex(multicast_iface_it->second.c_str());
		if (udpstream.multicast_iface_index == 0) {
			log(ERROR, "Interface '%s' does not exist", multicast_iface_it->second.c_str());
			return false;
		}
	}

	config->udpstreams.push_back(udpstream);
	return true;
}

bool parse_error_log(const ConfigLine &line, Config *config)
{
	if (line.arguments.size() != 0) {
		log(ERROR, "'error_log' takes no arguments (only parameters type= and filename=)");
		return false;
	}

	LogConfig log_config;
	map<string, string>::const_iterator type_it = line.parameters.find("type");
	if (type_it == line.parameters.end()) {
		log(ERROR, "'error_log' has no type= parameter");
		return false; 
	}

	string type = type_it->second;
	if (type == "file") {
		log_config.type = LogConfig::LOG_TYPE_FILE;
	} else if (type == "syslog") {
		log_config.type = LogConfig::LOG_TYPE_SYSLOG;
	} else if (type == "console") {
		log_config.type = LogConfig::LOG_TYPE_CONSOLE;
	} else {
		log(ERROR, "Unknown log type '%s'", type.c_str());
		return false; 
	}

	if (log_config.type == LogConfig::LOG_TYPE_FILE) {
		map<string, string>::const_iterator filename_it = line.parameters.find("filename");
		if (filename_it == line.parameters.end()) {
			log(ERROR, "error_log type 'file' with no filename= parameter");
			return false; 
		}
		log_config.filename = filename_it->second;
	}

	config->log_destinations.push_back(log_config);
	return true;
}

}  // namespace

bool parse_config(const string &filename, Config *config)
{
	vector<ConfigLine> lines;
	if (!read_config(filename, &lines)) {
		return false;
	}

	config->daemonize = false;

	if (!fetch_config_int(lines, "num_servers", &config->num_servers)) {
		log(ERROR, "Missing 'num_servers' statement in config file.");
		return false;
	}
	if (config->num_servers < 1 || config->num_servers >= 20000) {  // Insanely high max limit.
		log(ERROR, "'num_servers' is %d, needs to be in [1, 20000>.", config->num_servers);
		return false;
	}

        // See if the user wants stats.
	config->stats_interval = 60;
	bool has_stats_file = fetch_config_string(lines, "stats_file", &config->stats_file);
	bool has_stats_interval = fetch_config_int(lines, "stats_interval", &config->stats_interval);
	if (has_stats_interval && !has_stats_file) {
		log(WARNING, "'stats_interval' given, but no 'stats_file'. No client statistics will be written.");
	}

	config->input_stats_interval = 60;
	bool has_input_stats_file = fetch_config_string(lines, "input_stats_file", &config->input_stats_file);
	bool has_input_stats_interval = fetch_config_int(lines, "input_stats_interval", &config->input_stats_interval);
	if (has_input_stats_interval && !has_input_stats_file) {
		log(WARNING, "'input_stats_interval' given, but no 'input_stats_file'. No input statistics will be written.");
	}
	
	fetch_config_string(lines, "access_log", &config->access_log_file);

	for (size_t i = 0; i < lines.size(); ++i) {
		const ConfigLine &line = lines[i];
		if (line.keyword == "num_servers" ||
		    line.keyword == "stats_file" ||
		    line.keyword == "stats_interval" ||
		    line.keyword == "input_stats_file" ||
		    line.keyword == "input_stats_interval" ||
		    line.keyword == "access_log") {
			// Already taken care of, above.
		} else if (line.keyword == "port") {
			if (!parse_port(line, config)) {
				return false;
			}
		} else if (line.keyword == "listen") {
			if (!parse_listen(line, config)) {
				return false;
			}
		} else if (line.keyword == "stream") {
			if (!parse_stream(line, config)) {
				return false;
			}
		} else if (line.keyword == "udpstream") {
			if (!parse_udpstream(line, config)) {
				return false;
			}
		} else if (line.keyword == "error_log") {
			if (!parse_error_log(line, config)) {
				return false;
			}
		} else if (line.keyword == "daemonize") {
			config->daemonize = true;
		} else {
			log(ERROR, "Unknown configuration keyword '%s'.",
			        line.keyword.c_str());
			return false;
		}
	}

	return true;
}
