package f1.events;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import com.db4o.Db4o;
import com.db4o.ObjectContainer;
import com.db4o.ObjectSet;
import com.db4o.config.ConfigScope;
import com.db4o.drs.ObjectState;
import com.db4o.drs.Replication;
import com.db4o.drs.ReplicationEvent;
import com.db4o.drs.ReplicationEventListener;
import com.db4o.drs.ReplicationSession;

public class EventsExample {
	public static void main(String[] args) {
		
		replicationInterruptionExample();
		conflictResolutionExample();
	}
	// end main

	private static void replicationInterruptionExample() {
		Db4o.configure().generateUUIDs(ConfigScope.GLOBALLY);
		Db4o.configure().generateVersionNumbers(ConfigScope.GLOBALLY);
		
		new File("handheld.db4o").delete();
		new File("desktop.db4o").delete();
		//		Open the db4o database
		ObjectContainer desktop = Db4o.openFile("desktop.db4o");
		ObjectContainer handheld = Db4o.openFile("handheld.db4o");

		// create a list of objects and save to handheld
		SensorPanel sensors = new SensorPanel().createList(10);
		handheld.set(sensors);
		
		//Start a Replication Session
		// The replication must stop after replicating the root object
		ReplicationEventListener listener;
		listener = new ReplicationEventListener() {
			public void onReplicate(ReplicationEvent event) {
				if (event.stateInProviderA().getObject() instanceof SensorPanel)
					event.stopTraversal();
			}
		};
		ReplicationSession replication = Replication.begin(handheld, desktop, listener);

		//Query for changed objects
		ObjectSet changedObjects = replication.providerA().objectsChangedSinceLastReplication();
		while (changedObjects.hasNext())
			replication.replicate(changedObjects.next());

		//Commit
		replication.commit();
		replication.close();

		ObjectSet result = desktop.get(null);
		while (result.hasNext()) {
			System.out.println(result.next());
		}
		handheld.close();
		desktop.close();
		
		new File("handheld.db4o").delete();
		new File("desktop.db4o").delete();

	}
	// end replicationInterruptionExample
	
	private static void conflictResolutionExample() {
		Db4o.configure().generateUUIDs(ConfigScope.GLOBALLY);
		Db4o.configure().generateVersionNumbers(ConfigScope.GLOBALLY);

		//	Open databases
		ObjectContainer desktop = Db4o.openFile("desktop.db4o");
		ObjectContainer handheld = Db4o.openFile("handheld.db4o");

		Pilot pilot = new Pilot("Scott Felton", 200);
		handheld.set(pilot);
		handheld.commit();
		/* Clean the reference cache to make sure that objects in memory
		* won't interfere
		*/ 
		handheld.ext().refresh(Pilot.class, Integer.MAX_VALUE);
		
		/* Replicate changes from handheld to desktop
		 * Note, that only objects replicated from one database to another will 
		 * be treated as the same. If you will create an object and save it to both
		 * databases, dRS will count them as 2 different objects with identical 
		 * fields.
		 */
		ReplicationSession replication = Replication.begin(handheld, desktop);
		ObjectSet changedObjects = replication.providerA().objectsChangedSinceLastReplication();
		while (changedObjects.hasNext())
			replication.replicate(changedObjects.next());
		replication.commit();
		
		// change object on the handheld
		pilot = (Pilot)handheld.query(Pilot.class).next();
		pilot.setName("S.Felton");
		handheld.set(pilot);
		handheld.commit();
		
		//	change object on the desktop
		pilot = (Pilot)desktop.query(Pilot.class).next();
		pilot.setName("Scott");
		desktop.set(pilot);
		desktop.commit();
		
		/* The replication will face a conflict: Pilot object was changed on the 
		* handheld and on the desktop.
		* To resolve this conflict we will add an event handler, which makes
		* desktop changes dominating.
		*/
		ReplicationEventListener listener;
		listener = new ReplicationEventListener() {
			public void onReplicate(ReplicationEvent event) {
				if (event.isConflict()) {
					ObjectState chosenObjectState = event.stateInProviderB();
					event.overrideWith(chosenObjectState);
				}
			}
		};

		replication = Replication.begin(handheld, desktop, listener);

		//The state of the desktop after the replication should not change, as it dominates
		changedObjects = replication.providerA().objectsChangedSinceLastReplication();
		while (changedObjects.hasNext())
			replication.replicate(changedObjects.next());

		//Commit
		replication.commit();
		replication.close();
		
		// Check what we've got on the desktop
		ObjectSet result = desktop.query(Pilot.class);
		System.out.println(result.size());
		while (result.hasNext()) {
			System.out.println(result.next());
		}
		handheld.close();
		desktop.close();
		
		new File("handheld.db4o").delete();
		new File("desktop.db4o").delete();

	}
	// end conflictResolutionExample
}

