/* dc_gui2 - a GTK+2 GUI for DCTC
 * Copyright (C) 2002 Eric Prevoteau
 *
 * dctc_process.c: Copyright (C) Eric Prevoteau <www@a2pb.gotdns.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: dctc_process.c,v 1.49 2004/01/09 15:14:03 ericprev Exp $
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/socket.h>
#include <string.h>

/* Added for printer(), using isprint() */
#include <ctype.h>

#include <errno.h>
#include <gtk/gtk.h>
#include <gnome.h>

#include "dctc_process.h"
#include "main.h"
#include "misc_gtk.h"
#include "misc.h"
#include "do_connect.h"
#include "init_fnc.h"
#include "str_array.h"
#include "tos_key.h"
#include "timed_out_string.h"
#include "gui_define.h"
#include "ls_cache.h"
#include "gdl_ctree.h"
#include "find_result_clist.h"
#include "user_file_list_clist.h"
#include "uaddr_clist.h"
#include "userinfo.h"
#include "user_clist.h"
#include "gtk_helper.h"
#include "xpm_draw.h"
#include "sound.h"
#include "global_user.h"
#include "manage_chat.h"

/* forward declaration */
static void foreach_selected_entry_send_cmd_ulong_numeric_col(char *cmd,char *clist_name, int column_num);

/********************************/
/* print a line to stdout       */
/* filter out unprintable chars */
/* adds a newline to the end    */
/********************************/
static void printer(const GString *s)
{
	char *p = s->str;

	while ( p[0] != '\0' ) {
		if ( isprint(p[0]) )
			printf("%c", p[0]);
		else
			printf(" ");
		
		p++;
	}
	printf("\n");
}

/************************/
/* get a line from DCTC */
/******************************************************************************/
/* input: with_recv == 1, we read the sock_fd. ==0, we use the data in buffer */
/******************************************************************************/
static GString *get_dctc_line(int with_recv)
{
	GString *s;
	char *tmp;

	if(current_dctc==NULL)
		return NULL;

	/* if we should read the socket, read it and append data to the yet received one */
	if(with_recv)
	{
		char buf[8192];
		int n;

		read_again:
		n=recv(current_dctc->dctc_fd,buf,sizeof(buf)-1,MSG_NOSIGNAL);
		if(n==-1)
		{
			if(errno==EINTR)
				goto read_again;

			close_dctc_com(&current_dctc);
			return NULL;
		}

		if(n==0)
		{
			close_dctc_com(&current_dctc);
			return NULL;
		}

		buf[n]='\0';
		current_dctc->incoming_data=g_string_append(current_dctc->incoming_data,buf);
	}

	/* full line received ? */
	tmp=strchr(current_dctc->incoming_data->str,'\n');
	if(tmp==NULL)
		return NULL;

	/* split the data in buffer at the first '\n'. The second part becomes the new data in buffer */
	/* the first part is returned to the calling function */
	s=current_dctc->incoming_data;
	current_dctc->incoming_data=g_string_sized_new(128);
	current_dctc->incoming_data=g_string_assign(current_dctc->incoming_data,tmp+1);

	s=g_string_truncate(s,tmp+1-s->str);
	return s;
}

/****************************/
/* get a line from GDL DCTC */
/******************************************************************************/
/* input: with_recv == 1, we read the sock_fd. ==0, we use the data in buffer */
/******************************************************************************/
static GString *get_dctc_line_from_gdl_dctc(int with_recv)
{
	GString *s;
	char *tmp;

	if(gdl_dctc==NULL)
		return NULL;

	/* if we should read the socket, read it and append data to the yet received one */
	if(with_recv)
	{
		char buf[8192];
		int n;

		read_again:
		n=recv(gdl_dctc->dctc_fd,buf,sizeof(buf)-1,MSG_NOSIGNAL);
		if(n==-1)
		{
			if(errno==EINTR)
				goto read_again;

			close_dctc_com(&gdl_dctc);
			return NULL;
		}

		if(n==0)
		{
			close_dctc_com(&gdl_dctc);
			return NULL;
		}

		buf[n]='\0';
		gdl_dctc->incoming_data=g_string_append(gdl_dctc->incoming_data,buf);
	}

	/* full line received ? */
	tmp=strchr(gdl_dctc->incoming_data->str,'\n');
	if(tmp==NULL)
		return NULL;

	/* split the data in buffer at the first '\n'. The second part becomes the new data in buffer */
	/* the first part is returned to the calling function */
	s=gdl_dctc->incoming_data;
	gdl_dctc->incoming_data=g_string_sized_new(128);
	gdl_dctc->incoming_data=g_string_assign(gdl_dctc->incoming_data,tmp+1);

	s=g_string_truncate(s,tmp+1-s->str);
	return s;
}

static void dummy_fnc(const GString *s)
{
	/* Using printer() instead
	    printf("--> %s\n",s->str);
	 */

	GString *s2 = g_string_new("");
	g_string_sprintf(s2, "--> %s", s->str);
	printer(s2);    
	g_string_free(s2,TRUE);
}

/* ------------------------------------------------------------------------------------- */
/***************************************************/
/* the following function processes "OP   ]" lines */
/***************************************************/
/* GTK2: to test */
/*****************/
static void op_fnc(const GString *s)
{
	char *t;
	char *nick;
	GLOB_USER *gu;

	/* s format: OP   ] "xxxxx"nickname| */

	t=strchr(s->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;
	nick=t;
	t=strchr(nick,'|');
	if(t==NULL)
		return;
	*t='\0';

	gu=gu_add_user(nick,TRUE,NULL);
	gu_set_op_flag(gu,TRUE,TRUE);		/* set OP flag and redraw */
}

/* ----------------------------------------------------------------- */
/* ----------------------------------------------------------------- */
/* ----------------------------------------------------------------- */
typedef struct
{
	const char *pattern;
	unsigned int pattern_length;
	const char *new_pattern;
} RECOD_PATTERN;

static GString *enhance_chat_output(const gchar *utf8_string)
{
	static RECOD_PATTERN rp[]={
										/* longest pattern must be first */
										{"euro",4,"\xE2\x82\xAC"},
										{"<=>",3,"\xE2\x87\x94"},
										{":(",2,"\xE2\x98\xB9"},
										{":)",2,"\xE2\x98\xBA"},
										{"=>",2,"\xE2\x87\x92"},
										{"<=",2,"\xE2\x87\x90"},
										{NULL,0,NULL}
									  };
	GString *rec;
	int i;
	int fnd;

	rec=g_string_new("");

	while(*utf8_string!='\0')
	{
		if((*utf8_string)&0x80)
		{
			g_string_append_c(rec,utf8_string[0]);
			utf8_string++;
		}
		else
		{
			i=0;
			fnd=0;
			while(rp[i].pattern!=NULL)
			{
				if(!strncmp(utf8_string,rp[i].pattern,rp[i].pattern_length))
				{
					g_string_append(rec,rp[i].new_pattern);
					utf8_string+=rp[i].pattern_length;
					break;
				}
				i++;
			}

			if(!fnd)
			{
				g_string_append_c(rec,utf8_string[0]);
				utf8_string++;
			}
		}
	}

	return rec;
}

static void untranslate_chars(GString *str, int start_pos, int end_pos)
{
	int i;
	int p;
#define SUB_PAT(encoded_p,decoded_p)		{encoded_p, sizeof(encoded_p)-1, decoded_p, sizeof(decoded_p)-1 }

	static struct
	{
		char *encoded_pattern;
		int encoded_pattern_len;
		char *decoded_pattern;
		int decoded_pattern_len;
	} subst_pattern[]={ SUB_PAT("&#124;","|"),
								{NULL,0,NULL,0}};

	end_pos++;	/* don't forget to test the last character */

	p=0;
	while(subst_pattern[p].encoded_pattern!=NULL)
	{
		i=start_pos;
		while(i<end_pos-subst_pattern[p].encoded_pattern_len)
		{
			if(!strncmp(str->str+i,subst_pattern[p].encoded_pattern,subst_pattern[p].encoded_pattern_len))
			{
				g_string_erase(str,i,subst_pattern[p].encoded_pattern_len);
				g_string_insert(str,i,subst_pattern[p].decoded_pattern);

				/* don't forget to adjust i and end_pos */
				i+=subst_pattern[p].decoded_pattern_len;
				end_pos+=(subst_pattern[p].decoded_pattern_len-subst_pattern[p].encoded_pattern_len);
			}
			else
				i++;
		}

		p++;
	}
}

/* ----------------------------------------------------------------- */
/* ----------------------------------------------------------------- */
/* ----------------------------------------------------------------- */
/***************************************************/
/* the following function processes "CHAT ]" lines */
/***************************************************/
static void chat_fnc(const GString *s)
{
	char *t;
	char *o_msg;

	/* s format: CHAT ] "xxxxx"message| */

	t=strchr(s->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;
	o_msg=t;
	t=strchr(o_msg,'|');
	if(t==NULL)
		return;
	*t='\0';

	/* replace \r by \n */
	t=o_msg;
	while(*t!='\0')
	{
		if(*t=='\r')
		{
			if(t[1]=='\r')
				*t++=' ';

			*t='\n';
		}
		t++;
	}

	{
		GtkWidget *w;
		GtkTextView *gtv;
		GtkTextBuffer *gtb;
		GtkTextIter iter;
		char *eonick;
 		/* Morra: Timestamp thingy */
		struct tm tm;					 
		time_t tt;	
		char timestamp[512];	
		GString *msg;
 
		tt=time(NULL);		
		localtime_r(&tt,&tm);		
		strftime(timestamp,sizeof(timestamp),"[ %H:%M:%S ]",&tm);

		w=get_widget_by_widget_name(main_window,"chat_output");
		if(w==NULL)
			return;

		msg=g_string_new(o_msg);
		untranslate_chars(msg,0,msg->len);

		eonick=strstr(msg->str,"> ");

		gtb=gtk_text_view_get_buffer(gtv=GTK_TEXT_VIEW(w));

		gtk_text_buffer_get_end_iter(gtb,&iter);
		
		if((msg->str[0]!='<')||(eonick==NULL))
		{
			if(utf8_mode==TRUE)
			{
				gtk_text_buffer_insert(gtb,&iter,msg->str,-1);
			}
			else
			{
				gchar *utf8;
				utf8=g_locale_to_utf8(msg->str,-1,NULL,NULL,NULL);
			
				if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(get_widget_by_widget_name(main_window,"enhanced_chat_display_checkbutton")))==FALSE)
					gtk_text_buffer_insert(gtb,&iter,utf8,-1);
				else
				{
					GString *utf8_recoded;

					utf8_recoded=enhance_chat_output(utf8);
					gtk_text_buffer_insert(gtb,&iter,utf8_recoded->str,-1);
					g_string_free(utf8_recoded,TRUE);
				}
				g_free(utf8);
			}
		}
		else
		{
			gtk_text_buffer_insert_with_tags_by_name(gtb,&iter,timestamp,-1,"timestamp",NULL);

			if(utf8_mode==TRUE)
			{
				gtk_text_buffer_insert_with_tags_by_name(gtb,&iter,msg->str,eonick+1-msg->str,"nickname",NULL);
				gtk_text_buffer_insert(gtb,&iter,eonick+1,-1);
			}
			else
			{
				gchar *utf8;

				utf8=g_locale_to_utf8(msg->str,eonick+1-msg->str,NULL,NULL,NULL);
				gtk_text_buffer_insert_with_tags_by_name(gtb,&iter,utf8,-1,"nickname",NULL);
				g_free(utf8);

				utf8=g_locale_to_utf8(eonick+1,-1,NULL,NULL,NULL);
			
				if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(get_widget_by_widget_name(main_window,"enhanced_chat_display_checkbutton")))==FALSE)
					gtk_text_buffer_insert(gtb,&iter,utf8,-1);
				else
				{
					GString *utf8_recoded;

					utf8_recoded=enhance_chat_output(utf8);
					gtk_text_buffer_insert(gtb,&iter,utf8_recoded->str,-1);
					g_string_free(utf8_recoded,TRUE);
				}
				g_free(utf8);
			}
		}

		{
			GtkTextIter iter2;
			gtk_text_buffer_get_end_iter(gtb,&iter2);

			gtk_text_buffer_insert(gtb,&iter2,"\n",-1);
		}

		/* keep the window at the bottom place, we must use a mark, an iter does not work here */
		auto_scroll_view(gtv,gtb,w);
		g_string_free(msg,TRUE);
	}
}

/***************************************************/
/* the following function processes "HUBNM]" lines */
/***************************************************/
static void hubnm_fnc(const GString *s)
{
	char *t;
	char *msg;
	
	/* we valid here the user list. It is possible because the initial /ULIST comes after the initial /HUBNAME */

	/* s format: HUBNM] "xxxxx"hubname| */

	t=strchr(s->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;
	msg=t;
	t=strchr(msg,'|');
	if(t==NULL)
		return;
	*t='\0';

	{
		GtkWidget *w;
		GString *str;

		w=get_widget_by_widget_name(main_window,"xfer_notebook");
		if(w==NULL)
		{
			printf("no widget app1\n");
			return;
		}
		w=gtk_widget_get_toplevel(w);
		/* for an unknown reason, get_widget_by_widget_name doesn't work for "app1" */

		str=g_string_new("CHAT ] \"\">>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>\r"
										    "entering hub ");

		if(utf8_mode==TRUE)
		{
			gtk_window_set_title(GTK_WINDOW(w),msg);
		}
		else
		{
			gchar *utf8;

			utf8=g_locale_to_utf8(msg,-1,NULL,NULL,NULL);
			gtk_window_set_title(GTK_WINDOW(w),utf8);
			g_free(utf8);
		}
		g_string_sprintfa(str,  "%s\r<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<|",msg);

		/* display a message in the chat to notify hub change */
		chat_fnc(str);
		g_string_free(str,TRUE);
	}
}

/***************************************************/
/* the following function processes "ERR  ]" lines */
/***************************************************/
/* GTK2: to test */
/*****************/
static void err_fnc(const GString *s)
{
	char *t;
	char *fnc;
	char *msg;

	/* s format: ERR  ] "xxxxx"errmsg| */

	t=strchr(s->str,'"');
	if(t==NULL)
		return;
	fnc=t;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;
	msg=t;
	t=strrchr(msg,'|');
	if(t==NULL)
		return;
	*t='\0';

	/* replace | by \n */
	t=msg;
	while(*t!='\0')
	{
		if(*t=='|')
			*t='\n';
		t++;
	}

	if(strstr(msg,"use /QUIT to quit")==NULL)
	{
		/* don't display the "use /QUIT to quit" message */
		GtkWidget *w;
		GtkTextView *gtv;
		GtkTextBuffer *gtb;
		GtkTextIter iter;

		struct tm tm;
		time_t tt;
		char timestamp[512];

		w=get_widget_by_widget_name(main_window,"error_messages_text");
		if(w==NULL)
			return;

		gtb=gtk_text_view_get_buffer(gtv=GTK_TEXT_VIEW(w));
		gtk_text_buffer_get_end_iter(gtb,&iter);

		tt=time(NULL);
		localtime_r(&tt,&tm);
		strftime(timestamp,sizeof(timestamp),"%c: ",&tm);

		gtk_text_buffer_insert_with_tags_by_name(gtb,&iter,timestamp,-1,"timestamp",NULL);
		{
			/* ERR ] messages come from DCTC and are always in ascii thus we always have to convert the message in UTF8 */
			gchar *utf8;
			utf8=g_locale_to_utf8(msg,-1,NULL,NULL,NULL);
			gtk_text_buffer_insert(gtb,&iter,utf8,-1);
			g_free(utf8);
		}
		gtk_text_buffer_insert(gtb,&iter,"\n",-1);

		/* keep the window at the bottom place, we must use a mark, an iter does not work here */
		auto_scroll_view(gtv,gtb,w);
	}
}


/*****************************************************************/
/* compare the given string to the pattern entered in find_entry */
/*****************************************************************/
/* output: 0=not invalid, !=0: is invalid */
/******************************************/
static int is_invalid_search_result(char *fname)
{
	int i;
	GString to_find;

	if(last_started_search==NULL)
		return 0;		/* no filter list -> result is ok */

	i=0;
	while(last_started_search[i]!=NULL)
	{
		/* build a fake GString */
		to_find.str=last_started_search[i];
		to_find.len=strlen(to_find.str);
	
		if(my_strcasestr(&to_find,fname)==NULL)
			return 1;	/* filtering string not found -> result is invalid */
		i++;
	}
	return 0;		/* all strings in the filter are available -> result is ok */
}

static void set_same_color_as_row(GtkTreeModel *gtm, GtkListStore *gls, GtkTreeIter *current, GtkTreeIter *reference)
{
	char *base_color;

	gtk_tree_model_get(gtm,reference,FRC_LINE_BACKGROUND,&base_color,-1);

	gtk_list_store_set(gls,current, FRC_LINE_BACKGROUND,base_color,-1);

	free(base_color);
}

static void set_different_color_as_row(GtkTreeModel *gtm, GtkListStore *gls, GtkTreeIter *current, GtkTreeIter *reference)
{
	char *base_color;

	gtk_tree_model_get(gtm,reference,FRC_LINE_BACKGROUND,&base_color,-1);

	if(base_color==NULL)
	{
		gtk_list_store_set(gls,current, FRC_LINE_BACKGROUND,"grey94",-1);
	}
	else if(!strcmp(base_color,"grey94"))
	{
		gtk_list_store_set(gls,current, FRC_LINE_BACKGROUND,"grey87",-1);
	}
	else
		gtk_list_store_set(gls,current, FRC_LINE_BACKGROUND,NULL,-1);

	free(base_color);
}

static void set_different_color_as_2_rows(GtkTreeModel *gtm, GtkListStore *gls, GtkTreeIter *current, GtkTreeIter *ref_prev, GtkTreeIter *ref_next)
{
	char *prev_color;
	char *next_color;

	gtk_tree_model_get(gtm,ref_prev,FRC_LINE_BACKGROUND,&prev_color,-1);
	gtk_tree_model_get(gtm,ref_next,FRC_LINE_BACKGROUND,&next_color,-1);

	if(prev_color==NULL)
	{
		if(!strcmp(next_color,"grey94"))
			gtk_list_store_set(gls,current, FRC_LINE_BACKGROUND,"grey87",-1);
		else
			gtk_list_store_set(gls,current, FRC_LINE_BACKGROUND,"grey94",-1);
	}
	else if(!strcmp(prev_color,"grey94"))
	{
		if(next_color==NULL)
			gtk_list_store_set(gls,current, FRC_LINE_BACKGROUND,"grey87",-1);
		else
			gtk_list_store_set(gls,current, FRC_LINE_BACKGROUND,NULL,-1);
	}
	else
	{
		if(next_color==NULL)
			gtk_list_store_set(gls,current, FRC_LINE_BACKGROUND,"grey94",-1);
		else
			gtk_list_store_set(gls,current, FRC_LINE_BACKGROUND,NULL,-1);
	}

	free(prev_color);
	free(next_color);
}

/***************************************************/
/* the following function processes "SREST]" lines */
/***************************************************/
static void srest_fnc(const GString *s)
{
	char *t;
	char *nick;
	char *fname;
	char *size;
	char *dl_ratio;
	char *hubname;
	char *hubip;
	char *localfname;
	GString *hub;
	GLOB_USER *gu;

	/* s format: SREST] "xxxxx"nickname|file|size|ratio|hubname|hubip| */
	/* (result of SEARCH and MSEARCH) */
	/* or */
	/* s format: SREST] "xxxxx"nickname|file|size|ratio|hubname|hubip|localfname| */
	/* (result of CSEARCH) */


	t=strchr(s->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;
	nick=t;
	t=strchr(nick,'|');
	if(t==NULL)
		return;
	*t++='\0';

	fname=t;
	t=strchr(fname,'|');
	if(t==NULL)
		return;
	*t++='\0';

	/* check post-filtering option */
	{
		GtkWidget *w;
		w=get_widget_by_widget_name(main_window,"postfiltering_checkbutton");
		if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w))==TRUE)
		{
			/* we must do post-filtering */
			if(is_invalid_search_result(fname))
				return;
		}
	}

	size=t;
	t=strchr(size,'|');
	if(t==NULL)
		return;
	*t++='\0';

	dl_ratio=t;
	t=strchr(dl_ratio,'|');
	if(t==NULL)
		return;
	*t++='\0';

	hubname=t;
	t=strchr(hubname,'|');
	if(t==NULL)
		return;
	*t++='\0';

	hubip=t;
	t=strchr(hubip,'|');
	if(t==NULL)
		return;
	*t++='\0';

	localfname=t;
	t=strchr(localfname,'|');
	if(t==NULL)
		localfname="";
	else
		*t++='\0';

	hub=g_string_new("");
	
	g_string_sprintf(hub,"%s (%s)",hubname,hubip);
	{
		GtkWidget *w;

		w=get_widget_by_widget_name(main_window,"find_result");
		if(w!=NULL)
		{
			GtkListStore *gls;
			GtkTreeModel *gtm;
			GtkTreeIter iter;

			unsigned long fsize;
			unsigned int free_slot,ttl_slot;
			char buf_size[64];
			LMP_UINFO nickuinfo;
			gint sort_column;
			GtkSortType sort_type;

			gchar *nw[6];

			gls=GTK_LIST_STORE(gtm=gtk_tree_view_get_model(GTK_TREE_VIEW(w)));

			if(get_user_info(current_dctc->user_info_lmp,nick,&nickuinfo))
			{
				nickuinfo.cnx_type[0]='\0';	/* no information available */
			}

			sscanf(size,"%lu",&fsize);
#ifndef NO_PRINTF_LOCALE
			sprintf(buf_size,"%'lu",fsize);	/* NO_PRINTF_LOCAL support added */
#else
			sprintf(buf_size,"%lu",fsize);
#endif

			sscanf(dl_ratio,"%u/%u",&free_slot,&ttl_slot);

			nw[0]=nick;
			nw[1]=fname;
			nw[2]=buf_size;
			nw[3]=dl_ratio;
			nw[4]=nickuinfo.cnx_type;
			nw[5]=hub->str;

			/* get/create this user */
			gu=gu_add_user(nick,TRUE,NULL);
			
			gtk_list_store_append(gls,&iter);
			gu_ref_from_iter(gu,gtm,&iter);

			if(utf8_mode==TRUE)
			{
				gtk_list_store_set(gls,&iter,
   													FRC_FULL_PATH_COL,nw[1],
   													FRC_SIZE_COL,nw[2],
   													FRC_SLOT_COL,nw[3],
   													FRC_HUBNAME_COL,nw[5],
   													FRC_LINE_BACKGROUND,NULL,
														FRC_SIZE_AS_VAL,(gulong)fsize,
														FRC_FREE_SLOT,(int)free_slot,
														FRC_TTL_SLOT,(int)ttl_slot,
														FRC_FULL_PATH,nw[1],
														FRC_USER_POINTER,gu,
														-1);
			}
			else
			{
				gchar *utf8_n1, *utf8_n5;

				utf8_n1=g_locale_to_utf8(nw[1],-1,NULL,NULL,NULL);
				utf8_n5=g_locale_to_utf8(nw[5],-1,NULL,NULL,NULL);
				gtk_list_store_set(gls,&iter,	
   													FRC_FULL_PATH_COL,utf8_n1,
   													FRC_SIZE_COL,nw[2],
   													FRC_SLOT_COL,nw[3],
   													FRC_HUBNAME_COL,utf8_n5,
   													FRC_LINE_BACKGROUND,NULL,
														FRC_SIZE_AS_VAL,(gulong)fsize,
														FRC_FREE_SLOT,(int)free_slot,
														FRC_TTL_SLOT,(int)ttl_slot,
														FRC_FULL_PATH,nw[1],
														FRC_USER_POINTER,gu,
														-1);
				g_free(utf8_n1);
				g_free(utf8_n5);
			}

			if(gtk_tree_sortable_get_sort_column_id(GTK_TREE_SORTABLE(gtm),&sort_column,&sort_type))
			{
				if(sort_column==FRC_SIZE_COL)
				{
					GtkTreeIter prev;
					GtkTreeIter *next;
					gboolean has_prev,has_next;
					GtkTreePath *tp;
				
					next=gtk_tree_iter_copy(&iter);

					has_next=gtk_tree_model_iter_next(gtm,next);

					/* unfortunatelly, there is a direct call to have the next row but none to have the previous row */
					tp=gtk_tree_model_get_path(gtm,&iter);
					has_prev=gtk_tree_path_prev(tp);
					if(has_prev==TRUE)
					{
						has_prev=gtk_tree_model_get_iter(gtm,&prev,tp);
					}
					gtk_tree_path_free(tp);
					

					/* now, has_prev,has_next,prev and next are initialized */
					if((has_prev==FALSE)&&(has_next==FALSE))
					{
						/* only one row in the list, let is as is */
					}
					else
					{
						if((has_prev==FALSE)&&(has_next==TRUE))
						{
							/* first row in the list */
							gulong other_size;
							gtk_tree_model_get(gtm,next,FRC_SIZE_AS_VAL,&other_size,-1);

							if(other_size==fsize)
								set_same_color_as_row(gtm,gls,&iter,next);
							else
								set_different_color_as_row(gtm,gls,&iter,next);
							
						}
						else if((has_prev==TRUE)&&(has_next==FALSE))
						{
							/* last row in the list */
							gulong other_size;
							gtk_tree_model_get(gtm,&prev,FRC_SIZE_AS_VAL,&other_size,-1);

							if(other_size==fsize)
								set_same_color_as_row(gtm,gls,&iter,&prev);
							else
								set_different_color_as_row(gtm,gls,&iter,&prev);
						}
						else
						{
							/* row in the middle of the list */
							gulong other_size;
							gtk_tree_model_get(gtm,&prev,FRC_SIZE_AS_VAL,&other_size,-1);

							if(other_size==fsize)
								set_same_color_as_row(gtm,gls,&iter,&prev);
							else
							{
								gtk_tree_model_get(gtm,next,FRC_SIZE_AS_VAL,&other_size,-1);

								if(other_size==fsize)
									set_same_color_as_row(gtm,gls,&iter,next);
								else
									set_different_color_as_2_rows(gtm,gls,&iter,&prev,next);
							}
						}
					}

					gtk_tree_iter_free(next);
				}
			}
		}
	}
	g_string_free(hub,TRUE);
}

#if 1

static void update_lists(const GString *s)
{
		send_data_to_gdl_dctc("/SLOWGDLLST\n");
		send_data_to_dctc("/XFER\n");
}

static void update_lists_force_gdl(const GString *s)
{
	send_data_to_gdl_dctc("/GDLLST\n");
}

/****************************************************************/
/* free the given GPtrArray of gchar ** (freed with g_strfreev) */
/****************************************************************/
static void free_temp_array(GPtrArray **temp_array)
{
	int i;

	if(*temp_array==NULL)
		return;

	for(i=0;i<(*temp_array)->len;i++)
	{
		gchar **k;

		k=g_ptr_array_index((*temp_array),i);
		if(k)
			g_strfreev(k);
	}
	g_ptr_array_free((*temp_array),TRUE);
	*temp_array=NULL;
}


static GPtrArray *xferr_temp_array=NULL;		/* it is an array of gchar ** (each has been obtain using g_strsplit on one line of XFERR) */
static GPtrArray *xferq_temp_array=NULL;		/* it is an array of gchar ** (each has been obtain using g_strsplit on one line of XFERQ) */
static GPtrArray *cmdkb_temp_array=NULL;		/* it is an array of gchar ** (each has been obtain using g_strsplit on one line of CMDKB) */

/************************/
/* reset the xfer lists */
/************************/
/* input: NULL */
/***************/
void begin_xfer_list_fnc(const GString *s)
{
	if(xferr_temp_array!=NULL)
		free_temp_array(&xferr_temp_array);
	xferr_temp_array=g_ptr_array_new();

	if(xferq_temp_array!=NULL)
		free_temp_array(&xferq_temp_array);
	xferq_temp_array=g_ptr_array_new();

	if(cmdkb_temp_array!=NULL)
		free_temp_array(&cmdkb_temp_array);
	cmdkb_temp_array=g_ptr_array_new();
}

/****************************************/
/* add an xferr entry to its temp array */
/****************************************/
static void add_xferr_fnc(const GString *in)
{
	gchar **fields;
	char *t;

	t=strchr(in->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;

	fields=g_strsplit(t,"|",0);
	if((fields[0]!=NULL)&&(fields[1]!=NULL)&&(fields[2]!=NULL))
	{
		g_ptr_array_add(xferr_temp_array,fields);
	}
	else
		g_strfreev(fields);
}

/****************************************/
/* add an xferq entry to its temp array */
/****************************************/
static void add_xferq_fnc(const GString *in)
{
	gchar **fields;
	char *t;

	t=strchr(in->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;

	fields=g_strsplit(t,"|",0);
	if((fields[0]!=NULL)&&(fields[1]!=NULL))
	{
		g_ptr_array_add(xferq_temp_array,fields);
	}
	else
		g_strfreev(fields);
}

/****************************************/
/* add an cmdkb entry to its temp array */
/****************************************/
static void add_cmdkb_fnc(const GString *in)
{
	gchar **fields;
	char *t;

	t=strchr(in->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;

	fields=g_strsplit(t,"|",0);
	if((fields[0]!=NULL)&&(fields[1]!=NULL)&&(fields[2]!=NULL))
	{
		g_ptr_array_add(cmdkb_temp_array,fields);
	}
	else
		g_strfreev(fields);
}

/*====================================================================================================*/
/*= XFERR clist update =*/
/*======================*/
/******************************************************************/
/* update the given row of the download clist using ptr parameter */
/******************************************************************/
static void update_xferr_dl_row(GtkTreeModel *dl_gtm,GtkListStore *dl_gls,GtkTreeIter *iter,gchar **entry)
{
	if(!strncmp(entry[2],"LS/",3))
	{
		gtk_list_store_set(dl_gls,iter,DLC_SIZE_COL,"",DLC_SPEED_COL,"",DLC_FNAME_COL,_("File list"),DLC_LFNAME_COL,"",-1);
	}
}

/***************************************************************/
/* update the given row of the given clist using ptr parameter */
/***************************************************************/
static void update_xferr_ul_row(GtkTreeModel *ul_gtm,GtkListStore *ul_gls,GtkTreeIter *iter,gchar **entry)
{
	int set=0;
	unsigned long val1;
	unsigned long *ptr_status;
	int size_status;

	if(utf8_mode==TRUE)
	{
		gtk_list_store_set(ul_gls,iter,ULC_FNAME_COL,entry[2]+3,
								-1);
	}
	else
	{
		gchar *utf8_2;

		utf8_2=g_locale_to_utf8(entry[2]+3,-1,NULL,NULL,NULL);
		gtk_list_store_set(ul_gls,iter,ULC_FNAME_COL,utf8_2,
								-1);

		g_free(utf8_2);
	}

	/* try to restore the current transfer status */
	val1=strtoul(entry[0],NULL,10);
	if(get_tos_entry(ULST_TOSKEY,(char*)&val1,sizeof(unsigned long),0, (char**)&ptr_status, &size_status))
	{
		if(size_status==(5*sizeof(unsigned long)))
		{
			char stt[512];
			char stt1[512];
				
#ifndef NO_PRINTF_LOCALE
			sprintf(stt,"%'lu",ptr_status[2]);	/* NO_PRINTF_LOCAL support added */
			sprintf(stt1,"%'lu (%.2f%%)",ptr_status[1]+ptr_status[3] /* => current global position */, 	/* NO_PRINTF_LOCAL support added */
												  	100*((float)(ptr_status[1]+ptr_status[3]))/(float)(ptr_status[2]));
#else
			sprintf(stt,"%lu",ptr_status[2]);
			sprintf(stt1,"%lu (%.2f%%)",ptr_status[1]+ptr_status[3] /* => current global position */, 
												  	100*((float)(ptr_status[1]+ptr_status[3]))/(float)(ptr_status[2]));
#endif
			gtk_list_store_set(ul_gls,iter,ULC_SIZE_COL,stt,ULC_SPEED_COL,stt1,-1);
			set=1;
		}
		if(ptr_status)
			free(ptr_status);
	}
	
	if(!set)
	{
		gtk_list_store_set(ul_gls,iter,ULC_SIZE_COL,"",ULC_SPEED_COL,"",-1);
	}
}

static void	create_xferr_row(GtkTreeModel *dl_gtm,GtkListStore *dl_gls,GtkTreeModel *ul_gtm,GtkListStore *ul_gls,gchar **entry)
{
	GtkTreeIter iter;

	/* entry[0]= id */
	/* entry[1]= nickname */
	/* entry[2]= the command */
	/* entry[3...]= end with null array of command arguments */
	
	if(!strncmp(entry[2],"LS/",3))
	{
		GLOB_USER *gu;
		gu=gu_add_user(entry[1],TRUE,NULL);

		gtk_list_store_append(dl_gls,&iter);
		gu_ref_from_iter(gu,dl_gtm,&iter);

		gtk_list_store_set(dl_gls,&iter,DLC_ID_COL,strtoul(entry[0],NULL,10),
												DLC_SIZE_COL,"",DLC_SIZE_AS_VAL,0UL,
												DLC_SPEED_COL,"",DLC_FNAME_COL,"",
												DLC_LFNAME_COL,"",
												DLC_SIZE_RIGHT_ALIGN,1.0,
												DLC_HAS_START_INFO,FALSE,
												DLC_START_TIME,0,
												DLC_START_POS,0,
												DLC_USER_POINTER,gu,
												-1);
	}
	else if(!strncmp(entry[2],"UL/",3))
	{
		unsigned long val1;
		unsigned long *ptr_status;
		int size_status;
		GLOB_USER *gu;

		val1=strtoul(entry[0],NULL,10);

		gu=gu_add_user(entry[1],TRUE,NULL);

		gtk_list_store_append(ul_gls,&iter);
		gu_ref_from_iter(gu,ul_gtm,&iter);

		if(utf8_mode==TRUE)
		{
			gtk_list_store_set(ul_gls,&iter,ULC_ID_COL,val1,
													ULC_SIZE_COL,"",
													ULC_SIZE_AS_VAL,0UL,
													ULC_SPEED_COL,"",
													ULC_FNAME_COL,entry[2]+3,
													ULC_SIZE_RIGHT_ALIGN,1.0,
													ULC_USER_POINTER,gu,
													-1);
		}
		else
		{
			gchar *utf8_2;

			utf8_2=g_locale_to_utf8(entry[2]+3,-1,NULL,NULL,NULL);

			gtk_list_store_set(ul_gls,&iter,ULC_ID_COL,val1,
													ULC_SIZE_COL,"",
													ULC_SIZE_AS_VAL,0UL,
													ULC_SPEED_COL,"",
													ULC_FNAME_COL,utf8_2,
													ULC_SIZE_RIGHT_ALIGN,1.0,
													ULC_USER_POINTER,gu,
													-1);
			g_free(utf8_2);
		}

		/* try to restore the current transfer status */
		if(get_tos_entry(ULST_TOSKEY,(char*)&val1,sizeof(unsigned long),0, (char**)&ptr_status, &size_status))
		{
			if(size_status==(5*sizeof(unsigned long)))
			{
				char stt[512];
				char stt1[512];
					
#ifndef NO_PRINTF_LOCALE
				sprintf(stt,"%'lu",ptr_status[2]);		/* NO_PRINTF_LOCAL support added */
				sprintf(stt1,"%'lu (%.2f%%)",ptr_status[1]+ptr_status[3] /* => current global position */, 	/* NO_PRINTF_LOCAL support added */
														  100*((float)(ptr_status[1]+ptr_status[3]))/(float)(ptr_status[2]));
#else
				sprintf(stt,"%lu",ptr_status[2]);
				sprintf(stt1,"%lu (%.2f%%)",ptr_status[1]+ptr_status[3] /* => current global position */, 
														  100*((float)(ptr_status[1]+ptr_status[3]))/(float)(ptr_status[2]));
#endif
				gtk_list_store_set(ul_gls,&iter,ULC_SIZE_COL,stt,ULC_SIZE_AS_VAL,ptr_status[2],
														  ULC_SPEED_COL,stt1,-1);
			}
			if(ptr_status)
				free(ptr_status);
		}
	}
}

/*******************************************************************************/
/* scan download and upload clist, remove all running xfer which no more exist */
/* and update the matching one                                                 */
/*******************************************************************************/
static void clean_running_ul_and_dl_clist(GtkTreeModel *dl_gtm, GtkListStore *dl_gls, GtkTreeModel *ul_gtm, GtkListStore *ul_gls)
{
	int j;
	GtkTreeIter iter;

#if 0
	/* the following code does not work with gtk2.0.6 */
	int valid;

	/* first, the download clist */
	valid=gtk_tree_model_get_iter_first(dl_gtm,&iter);
	while(valid)
	{
		unsigned long xfer_id;
		int found=0;

		gtk_tree_model_get(dl_gtm,&iter,DLC_ID_COL,&xfer_id,-1);

		if(xfer_id==0)        /* xferq entries has a NULL ID */
			goto skip_to_next;

		for(j=0;j<xferr_temp_array->len;j++)
		{
			gchar **ptr;
	
			ptr=g_ptr_array_index(xferr_temp_array,j);
			if((ptr!=NULL)&&(ptr[0][0]!='\001'))
			{
				if(xfer_id==strtoul(ptr[0],NULL,10))
				{
					update_xferr_dl_row(dl_gtm,dl_gls,&iter,ptr);
					ptr[0][0]='\001';
					found=1;
					break;
				}
			}
		}

		if(!found)
		{
			/* the value no more exists */
			/* FIXME don't ask me why, according to the manual, the following line is enough */
			/* but it produce a warning when I compile it (at least with gtk 2.0.6) */
			valid=gtk_list_store_remove(dl_gls,&iter);
		}
		else
		{
			skip_to_next:
			valid=gtk_tree_model_iter_next(dl_gtm,&iter);
		}
	}

	/* and the upload clist */
	valid=gtk_tree_model_get_iter_first(ul_gtm,&iter);
	while(valid)
	{
		unsigned long xfer_id;
		int found=0;

		gtk_tree_model_get(ul_gtm,&iter,ULC_ID_COL,&xfer_id,-1);

		for(j=0;j<xferr_temp_array->len;j++)
		{
			gchar **ptr;
	
			ptr=g_ptr_array_index(xferr_temp_array,j);
			if((ptr!=NULL)&&(ptr[0][0]!='\001'))
			{
				if(xfer_id==strtoul(ptr[0],NULL,10))
				{
					update_xferr_ul_row(ul_gtm,ul_gls,&iter,ptr);
					ptr[0][0]='\001';
					found=1;
					break;
				}
			}
		}

		if(!found)
		{
			/* the value no more exists */
			/* FIXME don't ask me why, according to the manual, the following line is enough */
			/* but it produce a warning when I compile it (at least with gtk 2.0.6) */
			valid=gtk_list_store_remove(ul_gls,&iter);
		}
		else
			valid=gtk_tree_model_iter_next(ul_gtm,&iter);
	}
#else
	int i;

	i=gtk_tree_model_iter_n_children(dl_gtm,NULL)-1;
	while(i>=0)
	{
		if(gtk_tree_model_iter_nth_child(dl_gtm,&iter,NULL,i)==TRUE)
		{
			unsigned long xfer_id;
			int found=0;

			gtk_tree_model_get(dl_gtm,&iter,DLC_ID_COL,&xfer_id,-1);

			if(xfer_id!=0)        /* xferq entries has a NULL ID */
			{
				for(j=0;j<xferr_temp_array->len;j++)
				{
					gchar **ptr;
	
					ptr=g_ptr_array_index(xferr_temp_array,j);
					if((ptr!=NULL)&&(ptr[0][0]!='\001'))
					{
						if(xfer_id==strtoul(ptr[0],NULL,10))
						{
							update_xferr_dl_row(dl_gtm,dl_gls,&iter,ptr);
							ptr[0][0]='\001';
							found=1;
							break;
						}
					}
				}

				if(!found)
				{
					/* the value no more exists */
					gtk_list_store_remove(dl_gls,&iter);
				}
			}
			i--;
		}
		else
		{
			fprintf(stderr,"clear_queue_clist: idx too big\n");
			break;
		}
	}
	
	/* and the upload clist */
	i=gtk_tree_model_iter_n_children(ul_gtm,NULL)-1;
	while(i>=0)
	{
		if(gtk_tree_model_iter_nth_child(ul_gtm,&iter,NULL,i)==TRUE)
		{
			unsigned long xfer_id;
			int found=0;

			gtk_tree_model_get(ul_gtm,&iter,ULC_ID_COL,&xfer_id,-1);
	
			for(j=0;j<xferr_temp_array->len;j++)
			{
				gchar **ptr;
		
				ptr=g_ptr_array_index(xferr_temp_array,j);
				if((ptr!=NULL)&&(ptr[0][0]!='\001'))
				{
					if(xfer_id==strtoul(ptr[0],NULL,10))
					{
						update_xferr_ul_row(ul_gtm,ul_gls,&iter,ptr);
						ptr[0][0]='\001';
					found=1;
					break;
					}
				}
			}

			if(!found)
			{
				/* the value no more exists */
				gtk_list_store_remove(ul_gls,&iter);
			}
			i--;
		}
		else
		{
			fprintf(stderr,"clear_queue_clist: idx too big 2\n");
			break;
		}
	}
#endif
}

/*****************/
/* GTK2: to test */
/*****************/
static void end_xfer_update_xferr(void)
{
	GtkWidget *dl_clst, *ul_clst;
	GtkTreeModel *dl_gtm, *ul_gtm;
	GtkListStore *dl_gls, *ul_gls;
	int i;
	SORT_VARS(_ul)
	SORT_VARS(_dl)

	dl_clst=get_widget_by_widget_name(main_window,"download_clist");
	if(dl_clst==NULL)
		return;

	ul_clst=get_widget_by_widget_name(main_window,"upload_clist");
	if(ul_clst==NULL)
		return;

	dl_gls=GTK_LIST_STORE(dl_gtm=gtk_tree_view_get_model(GTK_TREE_VIEW(dl_clst)));
	ul_gls=GTK_LIST_STORE(ul_gtm=gtk_tree_view_get_model(GTK_TREE_VIEW(ul_clst)));

	SORT_SAVE_GLS_TO_UNSORTED(_ul,ul_gtm,ul_gls)
	SORT_SAVE_GLS_TO_UNSORTED(_dl,dl_gtm,dl_gls)

	/* remove all entries of the queue_clist which are no more inside cmdkb_temp_array */
	clean_running_ul_and_dl_clist(dl_gtm,dl_gls,ul_gtm,ul_gls);

	/* and add all missing entries */
	for(i=0;i<xferr_temp_array->len;i++)
	{
		gchar **ptr;

		ptr=g_ptr_array_index(xferr_temp_array,i);
		if(ptr!=NULL)
		{
			if(ptr[0][0]!='\001')
			{	/* unseen entry, time to add it */
				create_xferr_row(dl_gtm,dl_gls,ul_gtm,ul_gls,ptr);
			}
		}
	}
	
	SORT_RESTORE(_ul,ul_gtm)
	SORT_RESTORE(_dl,dl_gtm)

	/* free temp array buffer */
	free_temp_array(&xferr_temp_array);
}

/*====================================================================================================*/
/*= XFERQ clist update =*/
/*======================*/

/************************/
/* update one xferq row */
/************************/
/* GTK2: to test */
/*****************/
static void update_xferq_row(GtkTreeModel *gtm, GtkListStore *gls,GtkTreeIter *iter,gchar **ptr)
{
	if(!strncmp(ptr[1],"LS/",3))
	{
		gtk_list_store_set(gls,iter,DLC_FNAME_COL,_("File list"),-1);
	}
	else
	{
		gtk_list_store_set(gls,iter,DLC_SPEED_COL,"?",DLC_FNAME_COL,"?",DLC_LFNAME_COL,"?",-1);
	}
}

/********************************************************************/
/* search in the download clist for the entry in the download clist */
/********************************************************************/
/* GTK2: to test */
/*****************/
static void clean_queued_dl_clist(GtkTreeModel *gtm, GtkListStore *gls)
{
	GtkTreeIter iter;
	int i;

	i=gtk_tree_model_iter_n_children(gtm,NULL)-1;
	while(i>=0)
	{
		if(gtk_tree_model_iter_nth_child(gtm,&iter,NULL,i)==TRUE)
		{
			unsigned long xfer_id;
			int found=0;
			int j;
			GLOB_USER *gu;

			gtk_tree_model_get(gtm,&iter,DLC_ID_COL,&xfer_id,DLC_USER_POINTER,&gu,-1);

			if(xfer_id==0)        /* xferq entries has a NULL ID */
			{
				for(j=0;j<xferq_temp_array->len;j++)
				{
					gchar **ptr;
			
					ptr=g_ptr_array_index(xferq_temp_array,j);
					if((ptr!=NULL)&&(ptr[0][0]!='\001'))
					{
						if(!strcmp(gu->unfmt_nick,ptr[0]))
						{
							update_xferq_row(gtm,gls,&iter,ptr);
							ptr[0][0]='\001';
							found=1;
							break;
						}
					}
				}
		
				if(!found)
				{
					gu_unref_from_iter(gu,gtm,&iter);
					gtk_list_store_remove(gls,&iter);
				}
			}
			i--;
		}
		else
		{
			fprintf(stderr,"clear_queued_dl_clist: idx too big\n");
			break;
		}
	}
}

/* ------------------------------------------------------------------------------------- */
/**************************/
/* create a new xferq row */
/**************************/
/* GTK2: to test */
/*****************/
static void create_xferq_row(GtkTreeModel *gtm, GtkListStore *gls,gchar **ptr)
{
	GtkTreeIter iter;
	GLOB_USER *gu;

	gu=gu_add_user(ptr[0],TRUE,NULL);

	gtk_list_store_append(gls,&iter);
	gu_ref_from_iter(gu,gtm,&iter);

	gtk_list_store_set(gls,&iter,
											DLC_SIZE_COL,_("trying"),DLC_SIZE_AS_VAL,0,
											DLC_ID_COL,0,
											DLC_SIZE_RIGHT_ALIGN,1.0,
											DLC_HAS_START_INFO,FALSE,
											DLC_USER_POINTER,gu,
											-1);
										
	if(!strncmp(ptr[1],"LS/",3))
	{
		gtk_list_store_set(gls,&iter,DLC_FNAME_COL,_("File list"),-1);
	}
	else
	{
		gtk_list_store_set(gls,&iter,DLC_FNAME_COL,"?",DLC_LFNAME_COL,"?",-1);
	}
}

/* ------------------------------------------------------------------------------------- */
/************************************************/
/* update the XFERQ clist                       */
/* this is the list of xfer in the trying stage */
/************************************************/
static void end_xfer_update_xferq(void)
{
	GtkWidget *dl_clst;	/* queued xfers only appear in download clist */
	GtkTreeModel *gtm;
	GtkListStore *gls;
	int i;

	dl_clst=get_widget_by_widget_name(main_window,"download_clist");
	if(dl_clst==NULL)
		return;

	gls=GTK_LIST_STORE(gtm=gtk_tree_view_get_model(GTK_TREE_VIEW(dl_clst)));

	/* remove all entries of the queue_clist which are no more inside cmdkb_temp_array */
	clean_queued_dl_clist(gtm,gls);

	/* and add all missing entries */
	for(i=0;i<xferq_temp_array->len;i++)
	{
		gchar **ptr;

		ptr=g_ptr_array_index(xferq_temp_array,i);
		if(ptr!=NULL)
		{
			if(ptr[0][0]!='\001')
			{	/* unseen entry, time to add it */
				create_xferq_row(gtm,gls,ptr);
			}
		}
	}
	
	/* free temp array buffer */
	free_temp_array(&xferq_temp_array);
}

/*====================================================================================================*/
/*= CMDKB clist update =*/
/*======================*/

/********************************************************************************************/
/* scan the queue_clist and remove all entries which are not inside cmdkb_temp_array        */
/* at the same time, mark ('\001') each cmdkb_temp_array entry which are found in the clist */
/********************************************************************************************/
/* GTK2: to test */
/*****************/
static void clean_queue_clist(GtkTreeModel *gtm,GtkListStore *gls)
{
	GtkTreeIter iter;
	int i;

	i=gtk_tree_model_iter_n_children(gtm,NULL)-1;
	while(i>=0)
	{
		if(gtk_tree_model_iter_nth_child(gtm,&iter,NULL,i)==TRUE)
		{
			unsigned long xfer_id;
			int found=0;
			int j;
			char cmdkb_id[25];
			GLOB_USER *gu;

			gtk_tree_model_get(gtm,&iter,QUC_ID_COL,&xfer_id,QUC_USER_POINTER,&gu,-1);
			sprintf(cmdkb_id,"%lu",xfer_id);

			for(j=0;j<cmdkb_temp_array->len;j++)
			{
				gchar **ptr;

				ptr=g_ptr_array_index(cmdkb_temp_array,j);
				if((ptr!=NULL)&&(ptr[0][0]!='\001'))
				{
					if(!strcmp(cmdkb_id,ptr[0]))
					{
						ptr[0][0]='\001';
						found=1;
						break;
					}
				}
			}

			if(!found)
			{
				gu_unref_from_iter(gu,gtm,&iter);
				gtk_list_store_remove(gls,&iter);
			}
			i--;
		}
		else
		{
			fprintf(stderr,"clear_queue_clist: idx too big\n");
			break;
		}
	}
}

/* ------------------------------------------------------------------------------------- */
/**************************/
/* update the queue_clist */
/**************************/
/* GTK2: to test */
/*****************/
static void end_xfer_update_cmdkb(void)
{
	GtkWidget *w;
	int i;
	GtkTreeModel *gtm;
	GtkListStore *gls;
	GtkTreeIter iter;

	w=get_widget_by_widget_name(main_window,"queue_clist");
	if(w==NULL)
		return;

	gls=GTK_LIST_STORE(gtm=gtk_tree_view_get_model(GTK_TREE_VIEW(w)));

	/* remove all entries of the queue_clist which are no more inside cmdkb_temp_array */
	clean_queue_clist(gtm,gls);

	/* and add all missing entries */
	for(i=0;i<cmdkb_temp_array->len;i++)
	{
		gchar **ptr;

		ptr=g_ptr_array_index(cmdkb_temp_array,i);
		if(ptr!=NULL)
		{
			if(ptr[0][0]!='\001')
			{	/* unseen entry, time to add it */
				if(!strncmp(ptr[2],"/LS ",4))		/* is the waiting command a /LS ? */
				{
					char *nw[2];
					GLOB_USER *gu;

					nw[0]=ptr[0];			/* the waiting ID */
					nw[1]=ptr[2]+4;		/* the nickname */

					/* get/create this user entry */
					gu=gu_add_user(nw[1],TRUE,NULL);

					/* add a row and reference it */
					gtk_list_store_append(gls,&iter);
					gu_ref_from_iter(gu,gtm,&iter);

					gtk_list_store_set(gls,&iter, QUC_FILENAME_COL,_("File list"),
															QUC_ID_COL,strtoul(nw[0],NULL,10),
															QUC_USER_POINTER,gu,
															-1);
				}
			}
		}
	}
	
	/* free temp array buffer */
	free_temp_array(&cmdkb_temp_array);
}

/*====================================================================================================*/
/****************************************************************/
/* all temp arrays are filled, now, it is time to do the update */
/****************************************************************/
/* GTK2: to test */
/*****************/
static void end_xfer_list_fnc(const GString *s)
{
	/* update xferr list */
	end_xfer_update_xferr();

	/* update xferq list, easiest than previous one */
	end_xfer_update_xferq();

	/* and the easiest of all, the cmdkb array */
	end_xfer_update_cmdkb();
}

/* ------------------------------------------------------------------------------------------------------------- */
/**************************************/
/* update one row of the upload clist */
/**************************************/
/* GTK2: to test */
/*****************/
static void update_ul_list(const GString *in)
{
	unsigned long values[5];
	GtkTreeModel *gtm;
	GtkListStore *gls;
	GtkTreeIter iter;
	GtkWidget *w;

	char *t;

	t=strchr(in->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;
	
	values[0]=strtoul(t,&t,10);	/* thread id */
	if((t==NULL)||(*t!=':'))
		return;
	t++;
	
	values[1]=strtoul(t,&t,10);	/* upload start position */
	if((t==NULL)||(*t!='/'))
		return;
	t++;
	
	values[2]=strtoul(t,&t,10);	/* file length */
	if((t==NULL)||(*t!='/'))
		return;
	t++;
	
	values[3]=strtoul(t,&t,10);	/* sent data */
	if((t==NULL)||(*t!='/'))
		return;
	t++;

	values[4]=strtoul(t,&t,10);	/* total to send */
	if((t==NULL)||(*t!='|'))
		return;
	
	/* keep a copy of the status so we can refresh the display at anytime */
	add_tos_entry_v1_uniq(ULST_TOSKEY,300,(char*)&values[0],sizeof(unsigned long), (char*)&values[0],5*sizeof(unsigned long));

	/* cmd is "UL/ficname" */
	w=get_widget_by_widget_name(main_window,"upload_clist");
	if(w==NULL)
		return;

	gls=GTK_LIST_STORE(gtm=gtk_tree_view_get_model(GTK_TREE_VIEW(w)));

	/* search and update the row on the clist */
	if(get_iter_from_ulong(gtm,ULC_ID_COL,values[0],&iter))
	{
		/* search the row having the same xfer ID */
		char stt[512];
		char stt1[512];
		
#ifndef NO_PRINTF_LOCALE
		sprintf(stt,"%'lu",values[2]);	/* NO_PRINTF_LOCAL support added */
		sprintf(stt1,"%'lu (%.2f%%)",values[1]+values[3] /* => current global position */, 	/* NO_PRINTF_LOCAL support added */
												  100*((float)(values[1]+values[3]))/(float)(values[2]));
#else
		sprintf(stt,"%lu",values[2]);
		sprintf(stt1,"%lu (%.2f%%)",values[1]+values[3] /* => current global position */, 
												  100*((float)(values[1]+values[3]))/(float)(values[2]));
#endif
		gtk_list_store_set(gls,&iter,ULC_SIZE_COL,stt,ULC_SIZE_AS_VAL,values[2],ULC_SPEED_COL,stt1,-1);
	}
}
/* ------------------------------------------------------------------------------------------------------------- */

/***************************************************************************************/
/* search into the download_clist to find a row matching the nickname and the filename */
/***************************************************************************************/
/* output: TRUE if iter is initialized with the matching row */
/*************************************************************/
static gboolean find_xfer_row(GtkTreeModel *gtm,char *nick, char *fname, GtkTreeIter *iter)
{
	gchar *true_nick, *true_fname;
	gboolean valid;

	if((valid=gtk_tree_model_get_iter_first(gtm,iter))==FALSE)
		return FALSE;

	if(utf8_mode==TRUE)
	{
		true_nick=nick;
		true_fname=fname;
	}
	else
	{
		true_nick=g_locale_to_utf8(nick,-1,NULL,NULL,NULL);
		true_fname=g_locale_to_utf8(fname,-1,NULL,NULL,NULL);
	}

	while(valid)
	{
		gchar *cur_str;

		gtk_tree_model_get(gtm,iter,DLC_NICK_COL,&cur_str,-1);
		if(!strcmp(cur_str,true_nick))
		{
			g_free(cur_str);
			gtk_tree_model_get(gtm,iter,DLC_LFNAME_COL,&cur_str,-1);
			if(!strcmp(cur_str,true_fname))
			{
				g_free(cur_str);
				break;		/* we have found the row */
			}
		}
		g_free(cur_str);
		valid=gtk_tree_model_iter_next(gtm,iter);
	}

	if(utf8_mode==FALSE)
	{
		g_free(true_nick);
		g_free(true_fname);
	}

	return valid;
}

#if 0
/********************************************************/
/* copy a row from the download_clist to the done clist */
/********************************************************/
static void copy_row(GtkTreeModel *source, GtkTreeIter *iter_source, GtkListStore *dest, GtkTreeIter *iter_dest)
{
	gchar *size_col;
	gchar *fname_col;
	gchar *lfname_col;
	gulong size_as_val;

	gtk_tree_model_get(source,iter_source,	
														DLC_SIZE_COL,&size_col,
														DLC_FNAME_COL,&fname_col,
														DLC_LFNAME_COL,&lfname_col,
														DLC_SIZE_AS_VAL,&size_as_val,
														-1);

	gtk_list_store_set(dest,iter_dest,	DOC_FNAME_COL,fname_col,
													DOC_SIZE_COL,size_col,
													DOC_LFNAME_COL,lfname_col,
													DOC_SIZE_AS_VAL,size_as_val,
													-1);

	g_free(size_col);
	g_free(fname_col);
	g_free(lfname_col);
}
#endif

/***************************************************/
/* the following function processes "$DL- ]" lines */
/***************************************************/
static void remove_dl_xfer_on_success_fnc(const GString *s)
{
	GtkWidget *w;
	GtkTreeIter iter;
	GtkTreeModel *gtm_dl;
	GtkListStore *gls_dl;

	char *t;
	char *nick;
	char *fname;

	/* s format: ] "xxxxx"nickname|localfile| */

	t=strchr(s->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;
	nick=t;
	t=strchr(nick,'|');
	if(t==NULL)
		return;
	*t++='\0';
	
	fname=t;
	t=strchr(fname,'|');
	if(t==NULL)
		return;
	*t='\0';
	
	w=get_widget_by_widget_name(main_window,"download_clist");
	gls_dl=GTK_LIST_STORE(gtm_dl=gtk_tree_view_get_model(GTK_TREE_VIEW(w)));

	if(find_xfer_row(gtm_dl,nick,fname,&iter)==TRUE)
	{
#if 0
		GtkListStore *gls_done;
		GtkTreeIter iter_done;

		w=get_widget_by_widget_name(main_window,"done_clist");
		gls_done=GTK_LIST_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(w)));

		gtk_list_store_append(gls_done,&iter_done);
		copy_row(gtm_dl,&iter,gls_done,&iter_done);
#endif
		gtk_list_store_remove(gls_dl,&iter);
	}
	update_lists(NULL);
}

#endif

/***************************************************/
/* the following function processes "VAR  ]" lines */
/***************************************************/
static void var_fnc(const GString *s)
{
	char *t;
	char *vname;
	char *vval;

	/* s format: ] "xxxxx"var_name|var_value| */
	/* var value may contain | */

	t=strchr(s->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;
	vname=t;
	t=strchr(vname,'|');
	if(t==NULL)
		return;
	*t++='\0';
	
	vval=t;
	t=strrchr(vval,'|');
	if(t==NULL)
		return;
	*t='\0';

	add_var(vname,vval);
	fix_pref_window();
}
	
/***************************************************/
/* the following function processes "LUSER]" lines */
/***************************************************/
/* GTK2: ok, not tested */
/************************/
static void luser_fnc(const GString *s)
{
	char *t;
	gchar **tmp_split;

	/* s format: ] "xxxxx"nickname|hubip:port|share size|userip:port */
	/* var value may contain | */

	t=strchr(s->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;

	tmp_split=g_strsplit(t,"|",4+1);
	if((tmp_split[0]!=NULL)&&(tmp_split[1]!=NULL)&&(tmp_split[2]!=NULL)&&(tmp_split[3]!=NULL))
	{
		char *pattern;

		gtk_label_get(GTK_LABEL(get_widget_by_widget_name(main_window,"currently_searched_user_label")),&pattern);

		if(!strcmp(pattern,tmp_split[0]))
		{
			GtkWidget *w;
			GtkListStore *gls;
			GtkTreeIter iter;

			w=get_widget_by_widget_name(main_window,"locate_user_clist");

			gls=GTK_LIST_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(w)));

			gtk_list_store_append(gls,&iter);
			gtk_list_store_set(gls,&iter, LUC_NICK_COL, tmp_split[0],
													LUC_SIZE_COL, tmp_split[1],
													LUC_IP_COL, tmp_split[2],
													LUC_RIGHT_ALIGN_COL, 1.0,
													-1);
		}
	}

	g_strfreev(tmp_split);
}
	

/***************************************************/
/* the following function processes "PRIV ]" lines */
/***************************************************/
static void pchat_fnc(const GString *s)
{
	char *t;
	char *nick;
	char *omsg;
	GtkWidget *chat_text;
	char *pchat_label;
	int pchat_num;

	/* the user has disabled private chat ? */
	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(get_widget_by_widget_name(main_window,"ignore_all_pmsg_checkbutton")))==TRUE)
		return;
	
	/* s format: ] "xxxxx"var_name|var_value| */
	/* var value may contain | */

	t=strchr(s->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;
	nick=t;
	t=strchr(nick,'|');
	if(t==NULL)
		return;
	*t++='\0';
	
	omsg=t;
	t=strrchr(omsg,'|');
	if(t==NULL)
		return;
	*t='\0';

	/* replace \r by \n */
	t=omsg;
	while(*t!='\0')
	{
		if(*t=='\r')
		{
			if(t[1]=='\r')
			{
				*t++=' ';
			}

			*t='\n';
		}
		t++;
	}

	{
		GLOB_USER *gu;
		gu=gu_add_user(nick,TRUE,NULL);
		chat_text=get_pchat_text_widget_from_nick(gu,TRUE,&pchat_label,&pchat_num);
	}

	if(chat_text!=NULL)
	{
		GtkTextBuffer *gtb;
		GtkTextView *gtv;
		GtkTextIter iter;
		char *eonick;
	   /* Morra: Timestamp thingy */
		struct tm tm;
		time_t tt;						
		char timestamp[512];	
		GString *msg;
		
		tt=time(NULL);						
		localtime_r(&tt,&tm);			
		strftime(timestamp,sizeof(timestamp),"[ %H:%M:%S ]",&tm); 

		msg=g_string_new(omsg);
		untranslate_chars(msg,0,msg->len);

		eonick=strstr(msg->str,"> ");

		gtb=gtk_text_view_get_buffer(gtv=GTK_TEXT_VIEW(chat_text));
		gtk_text_buffer_get_end_iter(gtb,&iter);

		if((msg->str[0]!='<')||(eonick==NULL))
		{
			if(utf8_mode==TRUE)
			{
				gtk_text_buffer_insert(gtb,&iter,msg->str,-1);
			}
			else
			{
				gchar *utf8;
				utf8=g_locale_to_utf8(msg->str,-1,NULL,NULL,NULL);
	
				if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(get_widget_by_widget_name(main_window,"enhanced_chat_display_checkbutton")))==FALSE)
					gtk_text_buffer_insert(gtb,&iter,utf8,-1);
				else
				{
					GString *utf8_recoded;

					utf8_recoded=enhance_chat_output(utf8);
					gtk_text_buffer_insert(gtb,&iter,utf8_recoded->str,-1);
					g_string_free(utf8_recoded,TRUE);
				}
				g_free(utf8);
			}
		}
		else
		{
			gtk_text_buffer_insert_with_tags_by_name(gtb,&iter,timestamp,-1,"timestamp",NULL);
			if(utf8_mode==TRUE)
			{
				gtk_text_buffer_insert_with_tags_by_name(gtb,&iter,msg->str,eonick+1-msg->str,"nickname",NULL);
				gtk_text_buffer_insert(gtb,&iter,eonick+1,-1);
			}
			else
			{
				gchar *utf8;
				utf8=g_locale_to_utf8(msg->str,eonick+1-msg->str,NULL,NULL,NULL);
				gtk_text_buffer_insert_with_tags_by_name(gtb,&iter,utf8,-1,"nickname",NULL);
				g_free(utf8);
				utf8=g_locale_to_utf8(eonick+1,-1,NULL,NULL,NULL);
				if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(get_widget_by_widget_name(main_window,"enhanced_chat_display_checkbutton")))==FALSE)
					gtk_text_buffer_insert(gtb,&iter,utf8,-1);
				else
				{
					GString *utf8_recoded;

					utf8_recoded=enhance_chat_output(utf8);
					gtk_text_buffer_insert(gtb,&iter,utf8_recoded->str,-1);
					g_string_free(utf8_recoded,TRUE);
				}
				g_free(utf8);
			}
		}
		gtk_text_buffer_insert(gtb,&iter,"\n",-1);

		/* keep the window at the bottom place, we must use a mark, an iter does not work here */
		auto_scroll_view(gtv,gtb,chat_text);

		{
			int has_changed=FALSE;

			if(gtk_notebook_get_current_page(GTK_NOTEBOOK(get_widget_by_widget_name(main_window,"main_notebook")))!=CHAT_TAB)
			{
				blink_on("chat_page");
				has_changed=TRUE;
			}

			if(gtk_notebook_get_current_page(GTK_NOTEBOOK(get_widget_by_widget_name(main_window,"chat_notebook")))!=pchat_num)
			{
				blink_on(pchat_label);
				has_changed=TRUE;
			}

			if(has_changed)
				play_the_sound_of_event(SOUND_PRIV_MSG);
		}

		g_string_free(msg,TRUE);
	}
	else
	{
		GString *out;

		out=g_string_new("");

		g_string_sprintfa(out,"WARNING: all private chats are busy, close unused one.\n"
									 "From %s\n%s\n",nick,omsg);

		gnome_app_error(GNOME_APP(main_window),out->str);
		g_string_free(out,TRUE);
	}
}

/* ===================================================================================== */
/* ===================================================================================== */
/* ===================================================================================== */
/* ===================================================================================== */
/******************************/
/* end the FLST clist refresh */
/******************************/
void user_share_full_list_fnc(const GString *in)
{
	char *t;
	char *nick;

	/* s format: ] "xxxxx"nick| */
	t=strchr(in->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;
	nick=t;
	t=strchr(nick,'|');
	if(t==NULL)
		return;
	*t='\0';

	put_nick_ls_into_user_file_list_clist(nick);
}

/* ------------------------------------------------------------------------------ */
/********************************************************************/
/* this function is called when the user has validated its password */
/* string is a g_malloc'd string which should be freed, or NULL if  */
/* the user cancelled.                                              */
/********************************************************************/
/* GTK2: ok */
/************/
static void password_entered(gchar *string, gpointer data)
{
	if(string==NULL)
	{
		/* user has cancelled */
		send_data_to_dctc("/FORCEQUIT\n");
	}
	else
	{
		GString *cmd;

		cmd=g_string_new("/PASSWD ");
		g_string_sprintfa(cmd,"%s\n",string);
		send_data_to_dctc(cmd->str);
		g_string_free(cmd,TRUE);
		g_free(string);
	}
}

/************************************************/
/* open password dialog to enter a new password */
/************************************************/
/* GTK2: ok */
/************/
void enter_passwd_fnc(const GString *in)
{
	/* see gnome_request_dialog for information */
	gnome_app_request_password(GNOME_APP(main_window),_("Your nickname is password protected on this hub\nPlease enter your password."),
																	  password_entered,NULL);
}

/* ------------------------------------------------------------------------------ */
/*************************************/
/* decode and display a progress bar */
/*************************************/
void prog_bar_fnc(const GString *in)
{
	char *t;
	gchar **splt;
	/* s format: ] "xxxxx"barname|progress|string|[string|...] */

	t=strchr(in->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;

	splt=g_strsplit(t+1,"|",0);
	if(splt!=NULL)
	{
		if((splt[0]!=NULL)&&(splt[1]!=NULL)&&(splt[2]!=NULL))
		{	/* at least 3 fields */
			float v;
			int pb_len;
			char *pb=NULL;
			GtkWidget *nw[3];		/* a progress bar is 2 widgets: the window containing the progressbar, the progressbar itself, the label containing the text */

			v=atof(splt[1]);
			if(v!=100.0)
			{	/* it is not a bar destruction */
				if(get_tos_entry(PGBAR_TOSKEY,splt[0],strlen(splt[0]),0,&pb,&pb_len))
				{
					/* the bar still exists */
					if(pb!=NULL)
					{
						/* and seems to be valid */
						nw[0]=((GtkWidget **)pb)[0];
						nw[1]=((GtkWidget **)pb)[1];
						nw[2]=((GtkWidget **)pb)[2];
						goto set_field;
					}
				}
				else
				{
					/* the bar does not exist */
					{
						nw[0]=gtk_window_new(GTK_WINDOW_TOPLEVEL);
						gtk_container_border_width (GTK_CONTAINER (nw[0]), 5);
						{
							GtkWidget *vbox;
							vbox=gtk_vbox_new(FALSE,0);
							gtk_container_add(GTK_CONTAINER(nw[0]),vbox);
							gtk_widget_show(vbox);
							{
								nw[2]=gtk_label_new(NULL);
								gtk_box_pack_start(GTK_BOX(vbox),nw[2],TRUE,TRUE,0);
								gtk_widget_show(nw[2]);
							}
							{
								nw[1]=gtk_progress_bar_new();
								gtk_progress_bar_set_bar_style(GTK_PROGRESS_BAR(nw[1]),GTK_PROGRESS_CONTINUOUS);
								gtk_progress_bar_set_orientation(GTK_PROGRESS_BAR(nw[1]),GTK_PROGRESS_LEFT_TO_RIGHT);
								gtk_box_pack_start(GTK_BOX(vbox),nw[1],TRUE,TRUE,0);
								gtk_widget_show(nw[1]);
							}
						}
						gtk_widget_show(nw[0]);
	
						add_tos_entry(PGBAR_TOSKEY,1000000000,splt[0],strlen(splt[0]),(void*)&nw,sizeof(nw));
					}

					set_field:
					{
						GString *grp;
						int i;

						/* concat all strings */
						grp=g_string_new(splt[2]);
						i=3;
						while(splt[i]!=NULL)
						{
							grp=g_string_append_c(grp,'\n');
							grp=g_string_append(grp,splt[i]);
							i++;
						}

						gtk_progress_set_percentage(GTK_PROGRESS(nw[1]), v/100.0);		/* the value should be between 0 and 1 and we have it between 0 and 100 */
						gtk_label_set_text(GTK_LABEL(nw[2]),grp->str);
						g_string_free(grp,TRUE);
					}
				}

				if(pb!=NULL)
					free(pb);
			}
			else
			{
				/* it is a bar destruction */
				if(get_tos_entry(PGBAR_TOSKEY,splt[0],strlen(splt[0]),0,&pb,&pb_len))
				{
					if(pb!=NULL)
					{
						gtk_widget_destroy(((GtkWidget **)pb)[0]);
					}
					if(pb!=NULL)
						free(pb);
					delete_this_tos_entry(PGBAR_TOSKEY,splt[0],strlen(splt[0]),0);
				}
			}
		}
		g_strfreev(splt);
	}
}

/* ===================================================================================== */
/* ===================================================================================== */
/* ===================================================================================== */
/* ===================================================================================== */

static GHashTable *ualst_temp_array=NULL;		/* each entry is composed by an address (key) and a value (nick) obtained from one line of UALSC) */
/**************************************************/
/* start the beginning of the UADDR clist refresh */
/**************************************************/
void ualst_begin_fnc(const GString *in)
{
	if(ualst_temp_array!=NULL)
		g_hash_table_destroy(ualst_temp_array);
	ualst_temp_array=g_hash_table_new_full(g_str_hash,g_str_equal,g_free,g_free);
}

/***********************************/
/* temporary store one UADDR entry */
/***********************************/
void ualst_content_fnc(const GString *in)
{
	char *t;
	gchar **fields;

	/* s format: ] "xxxxx"nick|hostip:port */
	t=strchr(in->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;

	fields=g_strsplit(t,"|",0);
	g_hash_table_insert(ualst_temp_array,fields[1],fields[0]);	/* the key is hostip:port */
}

/* ------------------------------------------------------------------------------ */
/*************************************************************/
/* search if the given list entry exists in the update list. */
/* If yes, it is updated else, it is deleted                 */
/*************************************************************/
/* GTK2: to test */
/*****************/
static gboolean update_one_ualst_entry(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	char *ipport;
	int fnd=0;
	GLOB_USER *gu;
	gchar *gch;


	gtk_tree_model_get(model,iter,UAC_USER_POINTER,&gu,UAC_ADDR_PORT_COL,&ipport,-1);

	gch=g_hash_table_lookup(ualst_temp_array,ipport);

	/* first, we try to find an UADDR entry matching the current UADDR list entry */
	if(gch!=NULL)
	{
		if(gu==NULL)
		{	/* ip without name */
			if(strlen(gch))
			{	/* add name only if available */
				gu=gu_add_user(gch,TRUE,NULL);
				gu_ref_from_iter(gu,model,iter);
				gtk_list_store_set(GTK_LIST_STORE(model),iter,UAC_USER_POINTER,gu,-1);
			}
		}
		else
		{	/* ip with name */
			if(strcmp(gu->unfmt_nick,gch))
			{
				/* ip but with different name */
				/* remove previous reference */
				gu_unref_from_iter(gu,model,iter);		/* gu is not null here */

				/* and add the new one */
				gu=gu_add_user(gch,TRUE,NULL);
				gu_ref_from_iter(gu,model,iter);
				gtk_list_store_set(GTK_LIST_STORE(model),iter,UAC_USER_POINTER,gu,-1);
			}
		}
		g_hash_table_remove(ualst_temp_array,ipport);
		fnd=1;
	}
	else
	{
		/* no such address exists in the new list */
		if(gu!=NULL)
		{
			gu_unref_from_iter(gu,model,iter);
		}
		gtk_list_store_remove(GTK_LIST_STORE(model),iter);
	}

	free(ipport);
	
	return FALSE;	/* don't stop */
}

typedef struct
{
	GtkTreeModel *gtm;
	GtkListStore *gls;
} NW_UADDR_P;

/************************************/
/* add a new row to the uaddr_clist */
/************************************/
/* key is "host:ip"                 */
/* value is "nickname" or ""        */
/* user_data = NW_UADDR_P *         */
/************************************/
static void add_newly_visible_uaddr(gpointer key, gpointer value, gpointer user_data)
{
	gchar *hip=key;
	gchar *nick=value;
	NW_UADDR_P *nup=user_data;
	GLOB_USER *gu;

	GtkTreeIter iter;

	gtk_list_store_append(nup->gls,&iter);
	if((nick!=NULL)&&(strlen(nick)))
	{
		gu=gu_add_user(nick,TRUE,NULL);
		gu_ref_from_iter(gu,nup->gtm,&iter);
	}
	else
		gu=NULL;

	gtk_list_store_set(nup->gls,&iter,UAC_USER_POINTER,gu,
												  UAC_ADDR_PORT_COL,hip,
												  -1);
}

/*******************************/
/* end the UADDR clist refresh */
/*******************************/
/* GTK2: to test */
/*****************/
void ualst_end_fnc(const GString *in)
{
	GtkWidget *w;
	SORT_VARS( )
	NW_UADDR_P nup;

	if(ualst_temp_array==NULL)
		return;

	w=get_widget_by_widget_name(main_window,"uaddr_clist");
	if(w==NULL)
		return;

	nup.gtm=gtk_tree_view_get_model(GTK_TREE_VIEW(w));
	nup.gls=GTK_LIST_STORE(nup.gtm);
	SORT_SAVE_GLS_TO_UNSORTED(,nup.gtm,nup.gls)

	generic_uaddr_entry_list_calls(update_one_ualst_entry,NULL);	/* the function uses ualst_temp_array */

	/* the only thing to do know is to add newly create UADDR with their sources */
	g_hash_table_foreach(ualst_temp_array,add_newly_visible_uaddr,&nup);

	SORT_RESTORE(,nup.gtm)

	g_hash_table_destroy(ualst_temp_array);
	ualst_temp_array=NULL;

	return;
}

/* ------------------------------------------------------------------------------ */
/**********************************/
/* Add an entry to the UADDR list */
/**********************************/
/* GTK2: to test */
/*****************/
void ualst_entry_add(const GString *in)
{
	char *t;
	gchar **fields;
	GtkWidget *w;
	GtkListStore *gls;
	GtkTreeModel *gtm;
	GtkTreeIter iter;
	GLOB_USER *gu;

	w=get_widget_by_widget_name(main_window,"uaddr_clist");
	if(w==NULL)
		return;

	gls=GTK_LIST_STORE(gtm=gtk_tree_view_get_model(GTK_TREE_VIEW(w)));

	/* s format: ] "xxxxx"nick|hostip:port */
	t=strchr(in->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;

	fields=g_strsplit(t,"|",0);

	/* too easy to add a new entry */
	gtk_list_store_append(gls,&iter);
	if((fields[0]!=NULL)&&(strlen(fields[0])))
	{
		gu=gu_add_user(fields[0],TRUE,NULL);
		gu_ref_from_iter(gu,gtm,&iter);
	}
	else
		gu=NULL;

	gtk_list_store_set(gls,&iter,UAC_USER_POINTER,gu,
										  UAC_ADDR_PORT_COL,fields[1],
										  -1);
	g_strfreev(fields);
}

/* ------------------------------------------------------------------------------ */
/*************************************************/
/* search the matching ualst entry and delete it */
/*************************************************/
static gboolean discard_matching_ualst_entry(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	gchar **fields=data;
	char *ipport;
	gboolean ret=FALSE;	/* don't stop */
	GLOB_USER *gu;

	gtk_tree_model_get(model,iter,UAC_USER_POINTER,&gu,UAC_ADDR_PORT_COL,&ipport,-1);
	if( (gu!=NULL)&&(!strcmp(gu->unfmt_nick,fields[0])) &&
		 (!strcmp(ipport,fields[1])) )
	{
		gu_unref_from_iter(gu,model,iter);
		gtk_list_store_remove(GTK_LIST_STORE(model),iter);
		ret=TRUE;
	}
	free(ipport);
	return ret;
}

/*************************************/
/* remove an entry to the UADDR list */
/*************************************/
/* GTK2: to test */
/*****************/
void ualst_entry_remove(const GString *in)
{
	char *t;
	gchar **fields;
	GtkWidget *w;
	GtkTreeModel *gtm;

	w=get_widget_by_widget_name(main_window,"uaddr_clist");
	if(w==NULL)
		return;

	gtm=gtk_tree_view_get_model(GTK_TREE_VIEW(w));

	/* s format: ] "xxxxx"nick|hostip:port */
	t=strchr(in->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;

	fields=g_strsplit(t,"|",0);

	gtk_tree_model_foreach(gtm,discard_matching_ualst_entry,fields);
	
	/* it is not necessary to resort the list, removing something don't break the sort */
	g_strfreev(fields);
}
/* ------------------------------------------------------------------------------ */

/* ===================================================================================== */
/* ===================================================================================== */
/* ===================================================================================== */
/* ===================================================================================== */
static GPtrArray *glst_temp_array=NULL;		/* it is an array of gchar ** (each has been obtain using g_strsplit on one line of GLSTC) */

/************************************************/
/* start the beginning of the GDL ctree refresh */
/************************************************/
void glst_begin_fnc(const GString *in)
{
	free_temp_array(&glst_temp_array);
	glst_temp_array=g_ptr_array_new();
}

#define FIXED_GLSTC_HEADER  (3+4+2+1)		/* number of fixed strings in the GLSTC header (the GDL ID, the local filename, the local filesize, */
													/* (FIXED ?) the byte offset, the received byte, the start time and the speed, the rename dir, the rename file, */
													/* the script to start at end */
/*******************************************************************************************************************/
/* for each source/range entry of this GDL, we check if it appear in the given glst_temp_array (data)              */
/* on success, we update it and mark the string of data as used (1st byte:'\001'). On error the entry is destroyed */
/*******************************************************************************************************************/
/* output: FALSE: node updated, TRUE: node to delete */
/***************************************************************/
/* if the row alters any download size, set *update_dl to TRUE */
/***************************************************************/
static gboolean second_level_glst_end_fnc(GtkTreeModel *gtm, GtkTreeStore *gts, GtkTreeIter *iter, gchar **data, gboolean *update_dl)
{
	gchar **strv=data+FIXED_GLSTC_HEADER;		/* skip the header */
	int i;
	int ct_type;

	gtk_tree_model_get(gtm,iter,GCC_TYPE,&ct_type,-1);

	switch(ct_type)
	{
		case GDL_SEARCH_PATTERN:
										{
											GString *temp_str;

											{
												gulong autoscan_id,autoscan_type;
												char *search_pattern;

												gtk_tree_model_get(gtm,iter,GCC_ID,&autoscan_id,GCC_STR1,&search_pattern,GCC_VAL1,&autoscan_type,-1);
												/* only the autoscan search pattern has an icon */
												/* this ctree entry is a GDL autoscan */
												temp_str=g_string_new("");
												g_string_sprintf(temp_str,"%lu$%lu?%s", autoscan_id, autoscan_type, search_pattern);
												free(search_pattern);
											}

											i=0;

											while(strv[i]!=NULL)
											{
												if(strlen(strv[i])==0)		/* we have found the empty string between sources and range in the GDLSTC message */
												{
													i++;							/* now, i is the index of the first range string */
													while(strv[i]!=NULL)
													{
														if(strlen(strv[i])==0)		/* we have found the empty string between range and autoscan */
														{
															i++;
															while(strv[i]!=NULL)
															{
																if(strlen(strv[i])==0)		/* we have found the empty string at the end of the autoscan */
																	break;
								
																if(strv[i][0]!='\001')		/* string unused ? */
																{
																	/* now, we have to check if we found the wanted string */
																	if(!strcmp(strv[i],temp_str->str))
																	{
																		strv[i][0]='\001';	/* mark the string as used */
																		g_string_free(temp_str,TRUE); /* we have found it, it is over */
																		return FALSE; /* entry updated */
																	}
																}
																i++;
															}
															break;
														}
														i++;
													}
													break;
												}
												i++;
											}
											g_string_free(temp_str,TRUE);
										}
										break;

		case GDL_STORED_SEGMENT:
										{
											/* build the temporary comparaison string */
											gchar *temp_str;

											/* create a string "filename$range" */
											{
												char *stored_filename,*stored_interval;
												gtk_tree_model_get(gtm,iter,GCC_STR1,&stored_filename,GCC_STR2,&stored_interval,-1);
												temp_str=g_strconcat(stored_filename,"$",stored_interval,NULL);	
												free(stored_filename);
												free(stored_interval);
											}
	
											i=0;
											while(strv[i]!=NULL)
											{
												if(strlen(strv[i])==0)		/* we have found the empty string between sources and range in the GDLSTC message */
												{
													i++;							/* now, i is the index of the first range string */
							
													while(strv[i]!=NULL)
													{
														if(strlen(strv[i])==0)		/* we have found the empty string between range and autoscan */
															break;
								
														if(strv[i][0]!='\001')		/* string unused ? */
														{
															/* now, we have to check if we found the wanted string */
															if(!strcmp(strv[i],temp_str))
															{
																strv[i][0]='\001';	/* mark the string as used */
																g_free(temp_str);					/* we have found it, it is over */
																return FALSE; /* entry updated */
															}
														}
														i++;
													}
													break;
												}
												i++;
											}
											g_free(temp_str);					/* we have found it, it is over */
										}
										break;
		case GDL_RENAME_AT_END:
										if(strlen(strv[-2]))		/* renaming information */
										{
											int changed=0;
											char *final_filename,*final_directory;
											const char *cur_f, *cur_d;
											gtk_tree_model_get(gtm,iter,GCC_STR1,&final_directory,GCC_STR2,&final_filename,-1);
											cur_f=final_filename;
											cur_d=final_directory;

											/* update filename when needed */
											if(strcmp(final_filename,strv[-2]))
											{
												gtk_tree_store_set(gts,iter,GCC_STR2,strv[-2],-1);
												cur_f=strv[-2];
												changed=1;
											}

											/* update directory when needed */
											if(strlen(strv[-3]))
											{
												if(strcmp(final_directory,strv[-3]))
												{
													gtk_tree_store_set(gts,iter,GCC_STR1,strv[-3],-1);
													cur_d=strv[-3];
													changed=1;
												}
											}
											else
											{
												if(final_directory)
												{
													gtk_tree_store_set(gts,iter,GCC_STR1,"",-1);
													cur_d="";
													changed=1;
												}
											}

											if(changed)
											{
												gchar *temp_str;
												if(strlen(cur_d))
												{
													temp_str=g_strconcat(cur_d,"/",cur_f,NULL);
												}
												else
												{
													temp_str=g_strconcat(cur_f,NULL);
												}
	
												/* renaming information differs */
												if(utf8_mode==TRUE)
												{
													gtk_tree_store_set(gts,iter,GCC_FULL_PATH_COL,temp_str,-1);
												}
												else
												{
													gchar *utf8;

													utf8=g_locale_to_utf8(temp_str,-1,NULL,NULL,NULL);
													gtk_tree_store_set(gts,iter,GCC_FULL_PATH_COL,utf8,-1);
													g_free(utf8);
												}
												g_free(temp_str);
											}
											strv[-3][0]='\0';
											strv[-2][0]='\0';
											free(final_filename);
											free(final_directory);
											return FALSE; /* entry updated */
										}
										break;

		case GDL_SCRIPT_AT_END:
										{
											int changed=0;
											char *script_to_start;
											char *cur_c;
											gtk_tree_model_get(gtm,iter,GCC_STR1,&script_to_start,-1);
											cur_c=script_to_start;

											if(strlen(strv[-1]))
											{
												if(strcmp(script_to_start,strv[-1]))
												{
													gtk_tree_store_set(gts,iter,GCC_STR1,strv[-1],-1);
													cur_c=strv[-1];
													changed=1;
												}
											}
											else
											{
												if(script_to_start)
												{
													gtk_tree_store_set(gts,iter,GCC_STR1,"",-1);
													cur_c="";
													changed=1;
												}
											}

											if(changed)
											{
												/* script information differs */
												if(utf8_mode==TRUE)
												{
													gtk_tree_store_set(gts,iter,GCC_FULL_PATH_COL,cur_c,-1);
												}
												else
												{
													gchar *utf8;

													utf8=g_locale_to_utf8(cur_c,-1,NULL,NULL,NULL);
													gtk_tree_store_set(gts,iter,GCC_FULL_PATH_COL,utf8,-1);
													g_free(utf8);
												}
											}
											strv[-1][0]='\0';
											free(script_to_start);
											return FALSE; /* entry updated */
										}
										break;

		case GDL_ACTIVE_SEGMENT:
										{
											int temp_str_len;
											char *status_temp;
											gchar *temp_str;
											gulong remote_file_size;

											{
												char *remote_filename;
												GLOB_USER *gu;
												gtk_tree_model_get(gtm,iter,GCC_MISC_POINTER,&gu,GCC_STR2,&remote_filename,GCC_VAL1,&remote_file_size,-1);
												temp_str=g_strconcat(gu->unfmt_nick,"$",remote_filename,"$",NULL);
												free(remote_filename);
											}

											/* create a string "nickname$filename$" */
											temp_str_len=strlen(temp_str);
											i=0;
											while(strv[i]!=NULL)
											{
												if(strlen(strv[i])==0)		/* we have found the empty string between sources and range in the GDLSTC message */
													break;

												if(strv[i][0]!='\001')		/* string unused ? */
												{
													/* now, we have to check if we found the wanted string */
													if(!strncmp(strv[i],temp_str,temp_str_len))
													{
														/* we have found the string, check if we must update the row */
														status_temp=strchr(strv[i]+temp_str_len,'$');
														if(status_temp!=NULL)
														{
															unsigned long cur_size;
															char *status;
															*status_temp++='\0';
															/* strv[i]+temp_str_len is the remote file size */
															/* status_temp is the xfer status */
														
															if((cur_size=strtoul(strv[i]+temp_str_len,NULL,10))!=remote_file_size)
															{
																char buf_size[64];
#ifndef NO_PRINTF_LOCALE
																sprintf(buf_size,"%'lu",cur_size);	/* NO_PRINTF_LOCAL support added */
#else
																sprintf(buf_size,"%lu",cur_size);
#endif
																gtk_tree_store_set(gts,iter,GCC_SIZE_COL,buf_size,GCC_VAL1,(gulong)cur_size,-1);
															}
								
															gtk_tree_model_get(gtm,iter,GCC_STR3,&status,-1);
															if(strcmp(status,status_temp))
															{
																gtk_tree_store_set(gts,iter,GCC_STATUS_COL,
																					((status_temp[0]=='W')?_("Waiting"): ((status_temp[0]=='T')?_("Trying"):status_temp)),
																						GCC_STR3,status_temp,
																						-1);
																*update_dl=TRUE;
															}
															free(status);
									
															strv[i][0]='\001';	/* mark the string as used */
															g_free(temp_str);					/* we have found it, it is over */
															return FALSE; /* entry updated */
														}
													}
												}
												i++;
											}
											g_free(temp_str);					/* we have found it, it is over */
										}
										break;
		default:
						printf("Unsupported entry type: %d\n",ct_type);
						break;
	}
	/* the entry does not exist anymore */
	return TRUE;	/* destroy the node */
}

static const char *ftype2str(int file_type)
{
#define NB_TYPE_ENT 8
	static const char *ftype_str[]={ "any", "audio", "compressed", "document", "exe", "picture", "video", "folder", NULL};

	file_type--;

	if((file_type<0)||(file_type>=NB_TYPE_ENT))
		return "";

	if(ftype_str[file_type]==NULL)
		return "";
	return ftype_str[file_type];
}

/*******************************************************************************************/
/* for each unused string of the given glst_temp_array (data), we create all ctree entries */
/*******************************************************************************************/
/* set *update_dl to TRUE if the added rows alter the amount of downloaded bytes */
/*********************************************************************************/
static void second_level_add_newly_created_entries(GtkTreeModel *gtm,GtkTreeStore *gts, GtkTreeIter *parent_iter, gchar **data, gboolean *update_dl)
{
	gchar **strv=data+FIXED_GLSTC_HEADER;		/* skip the header */
	int i;
	char *ent_array[4];		/* 4 columns in the list */
	GtkTreeIter iter;
	
	if(strlen(strv[-2]))
	{	/* there is renaming information */
		gchar *temp_str;
		if(strlen(strv[-3]))
			temp_str=g_strconcat(strv[-3],"/",strv[-2],NULL);
		else
			temp_str=g_strconcat(strv[-2],NULL);

		gtk_tree_store_insert_before(gts,&iter,parent_iter,NULL);
		if(utf8_mode==TRUE)
		{
			gtk_tree_store_set(gts,&iter,GCC_GID_NICK_COL,NULL,
												GCC_GID_NICK_VISI,(gboolean)FALSE,
												GCC_GID_NICK_NOT_VISI,(gboolean)TRUE,
												GCC_FULL_PATH_COL,temp_str,
												GCC_SIZE_COL,NULL,
												GCC_STATUS_COL,_("At end, rename"),
												GCC_SIZE_RIGHT_ALIGN,1.0,
												GCC_TYPE,(guint)GDL_RENAME_AT_END,
												GCC_STR1,strv[-3],
												GCC_STR2,strv[-2],
												GCC_MISC_POINTER,NULL,
												-1);
		}
		else
		{
			gchar *utf8;

			utf8=g_locale_to_utf8(temp_str,-1,NULL,NULL,NULL);

			gtk_tree_store_set(gts,&iter,GCC_GID_NICK_COL,NULL,
												GCC_GID_NICK_VISI,(gboolean)FALSE,
												GCC_GID_NICK_NOT_VISI,(gboolean)TRUE,
												GCC_FULL_PATH_COL,utf8,
												GCC_SIZE_COL,NULL,
												GCC_STATUS_COL,_("At end, rename"),
												GCC_SIZE_RIGHT_ALIGN,1.0,
												GCC_TYPE,(guint)GDL_RENAME_AT_END,
												GCC_STR1,strv[-3],
												GCC_STR2,strv[-2],
												GCC_MISC_POINTER,NULL,
												-1);
			g_free(utf8);
		}
		g_free(temp_str);
	}

	if(strlen(strv[-1]))
	{
		/* there is script at end information */
		gtk_tree_store_insert_before(gts,&iter,parent_iter,NULL);
		if(utf8_mode==TRUE)
		{
			gtk_tree_store_set(gts,&iter,GCC_GID_NICK_COL,NULL,
												GCC_GID_NICK_VISI,(gboolean)FALSE,
												GCC_GID_NICK_NOT_VISI,(gboolean)TRUE,
												GCC_FULL_PATH_COL,strv[-1],
												GCC_SIZE_COL,NULL,
												GCC_STATUS_COL,_("At end, start script"),
												GCC_SIZE_RIGHT_ALIGN,1.0,
												GCC_TYPE,(guint)GDL_SCRIPT_AT_END,
												GCC_STR1,strv[-1],
												GCC_MISC_POINTER,NULL,
												-1);
		}
		else
		{
			gchar *utf8;

			utf8=g_locale_to_utf8(strv[-1],-1,NULL,NULL,NULL);
			gtk_tree_store_set(gts,&iter,GCC_GID_NICK_COL,NULL,
												GCC_GID_NICK_VISI,(gboolean)FALSE,
												GCC_GID_NICK_NOT_VISI,(gboolean)TRUE,
												GCC_FULL_PATH_COL,utf8,
												GCC_SIZE_COL,NULL,
												GCC_STATUS_COL,_("At end, start script"),
												GCC_SIZE_RIGHT_ALIGN,1.0,
												GCC_TYPE,(guint)GDL_SCRIPT_AT_END,
												GCC_STR1,strv[-1],
												GCC_MISC_POINTER,NULL,
												-1);
			g_free(utf8);
		}
	}

	i=0;
	/* scan source parts of the string */
	while((strv[i]!=NULL)&&(strlen(strv[i])))
	{
		if(strv[i][0]!='\001')		/* string unused ? */
		{
			gchar **f4;

			f4=g_strsplit(strv[i],"$",3+1);
			if((f4[0]!=NULL)&&(f4[1]!=NULL)&&(f4[2]!=NULL)&&(f4[3]!=NULL))
			{
				char buf_size[64];
				unsigned long cur_size;
				GLOB_USER *ngu;

				cur_size=strtoul(f4[2],NULL,10);
#ifndef NO_PRINTF_LOCALE
				sprintf(buf_size,"%'lu",cur_size);		/* NO_PRINTF_LOCAL support added */
#else
				sprintf(buf_size,"%lu",cur_size);
#endif

				ent_array[0]=f4[0];
				ent_array[1]=f4[1];
				ent_array[2]=buf_size;
				if(f4[3][0]=='W')
					ent_array[3]=_("Waiting");
				else if(f4[3][0]=='T')
					ent_array[3]=_("Trying");
				else
				{
					*update_dl=TRUE;
					ent_array[3]=f4[3];
				}

				ngu=gu_add_user(ent_array[0],TRUE,NULL);
				gtk_tree_store_insert_before(gts,&iter,parent_iter,NULL);
				gu_ref_from_iter(ngu,gtm,&iter);

				if(utf8_mode==TRUE)
				{
					gtk_tree_store_set(gts,&iter,
												GCC_GID_NICK_VISI,(gboolean)TRUE,
												GCC_GID_NICK_NOT_VISI,(gboolean)FALSE,
												GCC_FULL_PATH_COL,ent_array[1],
												GCC_SIZE_COL,ent_array[2],
												GCC_STATUS_COL,ent_array[3],
												GCC_SIZE_RIGHT_ALIGN,1.0,
												GCC_TYPE,(guint)GDL_ACTIVE_SEGMENT,
												GCC_STR1,f4[0],
												GCC_STR2,f4[1],
												GCC_VAL1,(gulong)cur_size,
												GCC_STR3,f4[3],
												GCC_MISC_POINTER,ngu,
												-1);

				}
				else
				{
					gchar *utf8_1;

					utf8_1=g_locale_to_utf8(ent_array[1],-1,NULL,NULL,NULL);

					gtk_tree_store_set(gts,&iter,
												GCC_GID_NICK_VISI,(gboolean)TRUE,
												GCC_GID_NICK_NOT_VISI,(gboolean)FALSE,
												GCC_FULL_PATH_COL,utf8_1,
												GCC_SIZE_COL,ent_array[2],
												GCC_STATUS_COL,ent_array[3],
												GCC_SIZE_RIGHT_ALIGN,1.0,
												GCC_TYPE,(guint)GDL_ACTIVE_SEGMENT,
												GCC_STR1,f4[0],
												GCC_STR2,f4[1],
												GCC_VAL1,(gulong)cur_size,
												GCC_STR3,f4[3],
												GCC_MISC_POINTER,ngu,
												-1);
					g_free(utf8_1);
				}
			}
			g_strfreev(f4);
		}
		i++;
	}

	if(strv[i]!=NULL)
	{
		/* we have found the empty string between sources and range in the GDLSTC message */
		i++;							/* now, i is the index of the first range string */
		while((strv[i]!=NULL)&&(strlen(strv[i])))
		{
			if(strv[i][0]!='\001')		/* string unused ? */
			{
				gchar **f2;

				f2=g_strsplit(strv[i],"$",1+1);
				if((f2[0]!=NULL)&&(f2[1]!=NULL))
				{
					char out_range[512];
					unsigned long lower, upper;
					sscanf(f2[1],"[%lu;%lu]",&lower,&upper);
#ifndef NO_PRINTF_LOCALE
					sprintf(out_range,"[%'lu : %'lu]",lower,upper);	/* NO_PRINTF_LOCAL support added */
#else
					sprintf(out_range,"[%lu : %lu]",lower,upper);
#endif

#if 0
					ent_array[0]="";
#endif
					ent_array[1]=f2[0];
					ent_array[2]=out_range;
#if 0
					ent_array[3]="";
#endif
					*update_dl=TRUE;
					gtk_tree_store_insert_before(gts,&iter,parent_iter,NULL);
					if(utf8_mode==TRUE)
					{
						gtk_tree_store_set(gts,&iter,GCC_GID_NICK_COL,NULL,
																GCC_GID_NICK_VISI,(gboolean)FALSE,
																GCC_GID_NICK_NOT_VISI,(gboolean)TRUE,
																GCC_FULL_PATH_COL,ent_array[1],
																GCC_SIZE_COL,ent_array[2],
																GCC_STATUS_COL,NULL,
																GCC_SIZE_RIGHT_ALIGN,1.0,
																GCC_TYPE,(guint)GDL_STORED_SEGMENT,
																GCC_STR1,f2[0],
																GCC_STR2,f2[1],
																GCC_MISC_POINTER,NULL,
																-1);
					}
					else
					{
						gchar *utf8_1;

						utf8_1=g_locale_to_utf8(ent_array[1],-1,NULL,NULL,NULL);

						gtk_tree_store_set(gts,&iter,GCC_GID_NICK_COL,NULL,
																GCC_GID_NICK_VISI,(gboolean)FALSE,
																GCC_GID_NICK_NOT_VISI,(gboolean)TRUE,
																GCC_FULL_PATH_COL,utf8_1,
																GCC_SIZE_COL,ent_array[2],
																GCC_STATUS_COL,NULL,
																GCC_SIZE_RIGHT_ALIGN,1.0,
																GCC_TYPE,(guint)GDL_STORED_SEGMENT,
																GCC_STR1,f2[0],
																GCC_STR2,f2[1],
																GCC_MISC_POINTER,NULL,
																-1);
						g_free(utf8_1);
					}
				}
				g_strfreev(f2);
			}
			i++;
		}

		if(strv[i]!=NULL)
		{
			/* we have found the empty string between range and autoscan */
			i++;							/* now, i is the index of the first autoscan string */
			while((strv[i]!=NULL)&&(strlen(strv[i])))
			{
				if(strv[i][0]!='\001')		/* string unused ? */
				{
					gchar **f2;

					f2=g_strsplit(strv[i],"$",1+1);
					if((f2[0]!=NULL)&&(f2[1]!=NULL))
					{
						GString *long_desc;
						int ftype;
						char *t;
						char *tmp_desc;

						long_desc=g_string_new(f2[1]);
						ftype=atoi(long_desc->str);
						t=strchr(long_desc->str,'?');
						if(t!=NULL)
						{
							long_desc=g_string_erase(long_desc,0,1+t-long_desc->str);
						}
						/* all the values are available only here */
						tmp_desc=strdup(long_desc->str);
						long_desc=g_string_prepend(long_desc,"] ");
						long_desc=g_string_prepend(long_desc,ftype2str(ftype));
						long_desc=g_string_prepend_c(long_desc,'[');

#if 0
						ent_array[0]=f2[0];
#endif
						ent_array[1]=long_desc->str;
#if 0
						ent_array[2]="";
						ent_array[3]="";
#endif

						gtk_tree_store_insert_before(gts,&iter,parent_iter,NULL);
						if(utf8_mode==TRUE)
						{
							gtk_tree_store_set(gts,&iter,GCC_GID_NICK_COL,NULL,
																	GCC_GID_NICK_VISI,(gboolean)TRUE,
																	GCC_GID_NICK_NOT_VISI,(gboolean)FALSE,
																	GCC_FULL_PATH_COL,ent_array[1],
																	GCC_SIZE_COL,NULL,
																	GCC_STATUS_COL,NULL,
																	GCC_SIZE_RIGHT_ALIGN,1.0,
																	GCC_TYPE,(guint)GDL_SEARCH_PATTERN,
																	GCC_ID,(gulong)strtoul(f2[0],NULL,10),
																	GCC_STR1,tmp_desc,
																	GCC_VAL1,(gulong)ftype,
																	GCC_MISC_POINTER,NULL,
																	-1);
						}
						else
						{
							gchar *utf8_1;
	
							utf8_1=g_locale_to_utf8(ent_array[1],-1,NULL,NULL,NULL);
							gtk_tree_store_set(gts,&iter,GCC_GID_NICK_COL,NULL,
																	GCC_GID_NICK_VISI,(gboolean)TRUE,
																	GCC_GID_NICK_NOT_VISI,(gboolean)FALSE,
																	GCC_FULL_PATH_COL,utf8_1,
																	GCC_SIZE_COL,NULL,
																	GCC_STATUS_COL,NULL,
																	GCC_SIZE_RIGHT_ALIGN,1.0,
																	GCC_TYPE,(guint)GDL_SEARCH_PATTERN,
																	GCC_ID,(gulong)strtoul(f2[0],NULL,10),
																	GCC_STR1,tmp_desc,
																	GCC_VAL1,(gulong)ftype,
																	GCC_MISC_POINTER,NULL,
																	-1);
							g_free(utf8_1);
						}
						free(tmp_desc);
						g_string_free(long_desc,TRUE);
					}
					g_strfreev(f2);
				}
				i++;
			}
		}
	}
}

/****************************************************************************/
/* scan the children of the given node to purge and update existing entries */
/****************************************************************************/
static void scan_second_level_of_gdl_ctree(GtkTreeModel *gtm,GtkTreeStore *gts,GtkTreeIter *first_level_parent,gchar **ptr, gboolean *update_dl)
{
	GtkTreeIter iter;

	if(gtk_tree_model_iter_children(gtm,&iter,first_level_parent)==TRUE)
	{
#if 0
	/* this code should work according to the manual but gtk_tree_store_remove does not return a boolean on my version (gtk2.0.6) */
		gboolean valid;
		do
		{
			if(second_level_glst_end_fnc(gtm,gts,&iter,ptr)==TRUE)
			{
				/* this line is not valid with gtk2.0.6 but according to the manual */
				/* some latest versions support this */
				valid=gtk_tree_store_remove(gts,&iter);
			}
			else
				valid=gtk_tree_model_iter_next(gtm,&iter);
		} while(valid);
#else
		/* We must use a slower code */
		int i;

		i=gtk_tree_model_iter_n_children(gtm,first_level_parent)-1;
		while(i>=0)
		{
			if(gtk_tree_model_iter_nth_child(gtm,&iter,first_level_parent,i)==TRUE)
			{
				if(second_level_glst_end_fnc(gtm,gts,&iter,ptr,update_dl)==TRUE)
				{
					gtk_tree_store_remove(gts,&iter);
				}
				i--;
			}
			else
			{
				fprintf(stderr,"scan_second_level_of_gdl_ctree: idx too big\n");
				break;
			}
		}
	}
#endif
}

/**********************************************************************************/
/* for each GDL of the ctree, we do this:                                         */
/* 1) check if it still exists. If not, delete it                                 */
/* 2) if it exists, we scan its sources and ranges to remove no more existing one */
/* 3) if it exists, we add newly inserted ranges and sources                      */
/**********************************************************************************/
/* output: FALSE: the node has been updated, TRUE, the node must be deleted */
/****************************************************************************/
static gboolean first_level_glst_end_fnc(GtkTreeModel *gtm,GtkTreeStore *gts,GtkTreeIter *iter)
{
	gchar gdl_id_txt[20];
	int i;
	guint ct_type;
	gulong gdl_id;
	gulong gdl_size;
	gboolean update_dl=FALSE;

	gtk_tree_model_get(gtm,iter,GCC_TYPE,&ct_type,GCC_ID,&gdl_id,GCC_VAL1,&gdl_size,-1);
	if(ct_type!=GDL_ROOT)
		return FALSE;

	sprintf(gdl_id_txt,"%lu",gdl_id);

	/* now, search for this id in the glst_temp_array */
	for(i=0;i<glst_temp_array->len;i++)
	{
		gchar **ptr;

		ptr=g_ptr_array_index(glst_temp_array,i);
		if(ptr!=NULL)
		{
			if(!strcmp(gdl_id_txt,ptr[0]))		/* compare this temp entry to the gdl_id we search. On success, we start level2 scan */
			{
				GString *spd_str;
				unsigned long missing_bytes;
				time_t duration;
				double spd;
				unsigned long cur_size;
				
				gulong byte_offset;
				gulong received_bytes;
				guint64 start_time;
				gulong last_10sec_speed;

				gdl_size=strtoul(ptr[2],NULL,10);    /* ttl_size */
				byte_offset=strtoul(ptr[3],NULL,10);		/* yet_here */
				received_bytes=strtoul(ptr[4],NULL,10);
				start_time=strtoul(ptr[5],NULL,10);
				last_10sec_speed=strtoul(ptr[6],NULL,10);		/* last10sec_here */

				gtk_tree_store_set(gts,iter,GCC_VAL1,gdl_size,
													GCC_BYTE_OFFSET,byte_offset,
													GCC_RECEIVED_BYTES,received_bytes,
													GCC_START_TIME,start_time,
													GCC_LAST_10S_SPD,last_10sec_speed,
													-1);

				
				missing_bytes=gdl_size-received_bytes;

				spd_str=g_string_new("");
#ifndef NO_PRINTF_LOCALE
				g_string_sprintf(spd_str,"%'15lu (%.2f%%) ",received_bytes,100.0*(double)received_bytes/(double)gdl_size);	/* NO_PRINTF_LOCAL support added */
#else
				g_string_sprintf(spd_str,"%15lu (%.2f%%) ",received_bytes,100.0*(double)received_bytes/(double)gdl_size);
#endif

				cur_size=received_bytes-byte_offset;		/* number of bytes downloaded */
				duration=time(NULL)-(time_t)start_time;
					
				spd=(double)(last_10sec_speed)/10.0;		/* compute the speed of the last 10 seconds */
				if(spd<1024.0)
				{
					g_string_sprintfa(spd_str,"[%.2fB/s]",spd);
				}
				else if(spd<(1024.0*1024.0))
				{
					g_string_sprintfa(spd_str,"[%.2fKB/s]",spd/1024.0);
				}

				/* Estimated Time to Arrival computation: now (in second) + missing_bytes/spd */
				if(spd>=10.0)		/* speed above 10bytes/s */
				{
					GString *tmp_dur;

					tmp_dur=duration_to_readable(((double)missing_bytes/spd));
					g_string_append_c(spd_str,' ');
					g_string_append(spd_str,tmp_dur->str);
					g_string_append(spd_str,_(" left"));
					g_string_free(tmp_dur,TRUE);
				}

				gtk_tree_store_set(gts,iter,GCC_STATUS_COL,spd_str->str,-1);
				g_string_free(spd_str,TRUE);

				scan_second_level_of_gdl_ctree(gtm,gts,iter,ptr,&update_dl);
					
				second_level_add_newly_created_entries(gtm,gts,iter,ptr,&update_dl);
				g_strfreev(ptr);	/* and free memory */
				g_ptr_array_index(glst_temp_array,i)=NULL;

				if(update_dl)
				{
					gtk_tree_store_set(gts,iter,GCC_PIXBUF_DIRTY,update_dl,
													-1);
				}
				return FALSE;
			}
		}
	}
		
	/* the ctree entry no more exists, delete it */
	return TRUE;
}

/****************************************/
/* scan all GDL and updates its sources */
/****************************************/
static void scan_first_level_of_gdl_ctree(GtkTreeModel *gtm, GtkTreeStore *gts)
{
	GtkTreeIter iter;

#if 0
	/* this code should work according to the manual but gtk_tree_store_remove does not return a boolean on my version (gtk2.0.6) */
	gboolean valid;

	valid=gtk_tree_model_get_iter_first(gtm,&iter);
	while(valid)
	{
		if(first_level_glst_end_fnc(gtm,gts,&iter)==TRUE)
		{
			/* this line is not valid with gtk2.0.6 but according to the manual */
			/* some latest versions support this */
			valid=gtk_tree_store_remove(gts,&iter);
		}
		else
			valid=gtk_tree_model_iter_next(gtm,&iter);
	}
#else
	/* We must use a slower code */
	int nb_children;
	int i;
	int end_of_dl=FALSE;

	nb_children=gtk_tree_model_iter_n_children(gtm,NULL);
	i=0;
	while(i<nb_children)
	{
		if(gtk_tree_model_iter_nth_child(gtm,&iter,NULL,i)==TRUE)
		{
			if(first_level_glst_end_fnc(gtm,gts,&iter)==TRUE)
			{
				gtk_tree_store_remove(gts,&iter);		/* don't increase i because we destroy the current node */
																	/* thus we are automatically on the next one after */
				nb_children--;
				end_of_dl=TRUE;
			}
			else
				i++;
		}
		else
		{
			fprintf(stderr,"scan_first_level_of_gdl_ctree: idx too big\n");
			break;
		}
	}

	if(end_of_dl)
		play_the_sound_of_event(SOUND_END_OF_DOWNLOAD);
#endif
}

#define XPM_DATA_WIDTH 100
#define XPM_DATA_HEIGHT 18
#define XPM_OFFSET_TO_PIX 6		/* string index to the picture */

/* XPM */
static const char *org_xpm_data[] = {
"100 18 5 1",
" 	c None",		/* transparent background */
". c #00B800",	/* top progressbar: dark green */
"o c #FF5C00",	/* not yet received area: dark orange */
"b c #DBDBFF",	/* receiving area: light blue */
"B c #6E6E80",	/* received area: dark blue */
"                                                                                                    ",
"                                                                                                    ",
"                                                                                                    ",
"                                                                                                    ",
"                                                                                                    ",
"                                                                                                    ",
"                                                                                                    ",
"                                                                                                    ",
"                                                                                                    ",
"                                                                                                    ",
"                                                                                                    ",
"                                                                                                    ",
"                                                                                                    ",
"                                                                                                    ",
"                                                                                                    ",
"                                                                                                    ",
"                                                                                                    ",
"                                                                                                    "};

static char **xpm_data=NULL;

static void init_xpm_data(void)
{
	int nb_ptr=sizeof(org_xpm_data)/sizeof(char*);
	int i;

	xpm_data=malloc(nb_ptr*sizeof(char*));

	for(i=0;i<nb_ptr;i++)
	{
		xpm_data[i]=strdup(org_xpm_data[i]);
	}
}

static void populate_gdl_pixmap(GtkTreeModel *gtm, GtkTreeStore *gts,GtkTreeIter *parent_iter, gulong gdl_size)
{
	GtkTreeIter iter;

	if(gtk_tree_model_iter_children(gtm,&iter,parent_iter)==FALSE)
		return;

	do
	{
		guint row_type;
		char *str2=NULL, *str3=NULL;
		gulong xbegin,xend;
		char fill_char;
		gfloat min_range, max_range;

		gtk_tree_model_get(gtm,&iter,GCC_TYPE,&row_type,GCC_STR2,&str2,GCC_STR3,&str3,-1);

		switch(row_type)
		{
			case GDL_ACTIVE_SEGMENT:		/* str3= 'W'|'T'|'Ramount[begin:end] */
											if(str3[0]=='R')
											{
												if(sscanf(str3,"R%lu[%lu:",&xend,&xbegin)==2)
												{
													xend+=xbegin;
													fill_char='b';
													goto calc_box;
												}
											}
											break;

			case GDL_STORED_SEGMENT:		/* str2= [begin:end] */
											if(sscanf(str2,"[%lu;%lu]",&xbegin,&xend)==2)
											{
												fill_char='B';
												calc_box:

												min_range=((gfloat)xbegin)/((gfloat)gdl_size);
												max_range=((gfloat)xend)/((gfloat)gdl_size);
												cyclic_box_xpm(xpm_data,XPM_DATA_WIDTH,XPM_DATA_HEIGHT,XPM_OFFSET_TO_PIX,fill_char,min_range,max_range,0,XPM_DATA_WIDTH-1,3,XPM_DATA_HEIGHT-2);
											}
											break;
		}

		if(str2)
			free(str2);
		if(str3)
			free(str3);
	}while(gtk_tree_model_iter_next(gtm,&iter));
}

/******************************************/
/* scan all GDL to find pixbuf to rebuild */
/******************************************/
static void scan_first_level_of_gdl_ctree_for_dirty_pixbuf(GtkTreeModel *gtm, GtkTreeStore *gts)
{
	GtkTreeIter iter;
	int nb_children;
	int i;
	gboolean update_dl;

	if(xpm_data==NULL)
		init_xpm_data();

	nb_children=gtk_tree_model_iter_n_children(gtm,NULL);
	i=0;
	while(i<nb_children)
	{
		if(gtk_tree_model_iter_nth_child(gtm,&iter,NULL,i)==TRUE)
		{
			gtk_tree_model_get(gtm,&iter,GCC_PIXBUF_DIRTY,&update_dl,-1);

			if(update_dl==TRUE)
			{
				GdkPixbuf *gpb=NULL;
				gulong gdl_size;
				gulong received_bytes;

				gtk_tree_model_get(gtm,&iter,GCC_VAL1,&gdl_size,GCC_RECEIVED_BYTES,&received_bytes,-1);

				/* fill the top 2 lines with the transparent color */
				box_xpm(xpm_data,XPM_DATA_WIDTH,XPM_DATA_HEIGHT,XPM_OFFSET_TO_PIX,' ',0,XPM_DATA_WIDTH-1,0,3);
				/* and all the remaining bottom lines with the light orange */
				box_xpm(xpm_data,XPM_DATA_WIDTH,XPM_DATA_HEIGHT,XPM_OFFSET_TO_PIX,'o',0,XPM_DATA_WIDTH-1,3,XPM_DATA_HEIGHT-1);

				/* draw a green line at the top with a length linked to the amount of received bytes */
				box_xpm(xpm_data,XPM_DATA_WIDTH,XPM_DATA_HEIGHT,XPM_OFFSET_TO_PIX,'.',0,((100.0*(float)received_bytes)/(float)gdl_size),0,2);
			
				populate_gdl_pixmap(gtm,gts,&iter,gdl_size);

				gpb=gdk_pixbuf_new_from_xpm_data((const char **)xpm_data);		/* create a 24 bits RGB (no alpha) pixmap of 100x18 pixels */
				update_dl=FALSE;
				gtk_tree_store_set(gts,&iter,GCC_PIXBUF,gpb,GCC_PIXBUF_DIRTY,&update_dl,-1);
				g_object_unref(G_OBJECT(gpb));
			}
			i++;
		}
		else
		{
			fprintf(stderr,"scan_first_level_of_gdl_ctree: idx too big\n");
			break;
		}
	}
}

/*****************************/
/* end the GDL ctree refresh */
/*****************************/
void glst_end_fnc(const GString *in)
{
	GtkWidget *w;
	GtkTreeModel *gtm;
	GtkTreeStore *gts;
	SORT_VARS( )

	int i;

	if(glst_temp_array==NULL)
		return;

	w=get_widget_by_widget_name(main_window,"gdl_ctree");
	if(w==NULL)
		return;

	gts=GTK_TREE_STORE(gtm=gtk_tree_view_get_model(GTK_TREE_VIEW(w)));
	SORT_SAVE(,gtm,GCC_SIZE_RIGHT_ALIGN)

	/* now, we will scans the ctree to remove all sources entries which no more exists and add newly added sources */
	/* at the same time, we remove GDL entries which no more exist */
	scan_first_level_of_gdl_ctree(gtm,gts);

	/* the only thing to do know is to add newly create GDL with their sources */
	for(i=0;i<glst_temp_array->len;i++)
	{
		gchar **k;

		k=g_ptr_array_index(glst_temp_array,i);
		if(k)
		{
			char buf_size[64];
			unsigned long size_gdl=strtoul(k[2],NULL,10);
			GtkTreeIter iter;
			gboolean update_dl=FALSE;

			/* remove an entry with the same gdl_id */
#ifndef NO_PRINTF_LOCALE
			sprintf(buf_size,"%'lu",size_gdl);	/* NO_PRINTF_LOCAL support added */
#else
			sprintf(buf_size,"%lu",size_gdl);
#endif

			/* add a new entry in the root list */
			gtk_tree_store_insert_before(gts,&iter,NULL,NULL);
			if(utf8_mode==TRUE)
			{
				gtk_tree_store_set(gts,&iter,
													GCC_GID_NICK_COL,NULL,
													GCC_GID_NICK_VISI,(gboolean)FALSE,
													GCC_GID_NICK_NOT_VISI,(gboolean)TRUE,
													GCC_FULL_PATH_COL,k[1],
													GCC_SIZE_COL,buf_size,
													GCC_STATUS_COL,NULL,
													GCC_SIZE_RIGHT_ALIGN,1.0,
													GCC_LINE_BACKGROUND,"LightGoldenrod1",
													GCC_TYPE,(guint)GDL_ROOT,
													GCC_ID,(gulong)strtoul(k[0],NULL,10),
													GCC_STR1,k[1],
													GCC_VAL1,(gulong)size_gdl,
													GCC_BYTE_OFFSET,(gulong)0,
													GCC_RECEIVED_BYTES,(gulong)0,
													GCC_START_TIME,(guint64)0,
													GCC_LAST_10S_SPD,(gulong)0,
													GCC_PIXBUF,NULL,
													GCC_MISC_POINTER,NULL,
													-1);
			}
			else
			{
				gchar *utf8_1;
	
				utf8_1=g_locale_to_utf8(k[1],-1,NULL,NULL,NULL);
				gtk_tree_store_set(gts,&iter,
													GCC_GID_NICK_COL,NULL,
													GCC_GID_NICK_VISI,(gboolean)FALSE,
													GCC_GID_NICK_NOT_VISI,(gboolean)TRUE,
													GCC_FULL_PATH_COL,utf8_1,
													GCC_SIZE_COL,buf_size,
													GCC_STATUS_COL,NULL,
													GCC_SIZE_RIGHT_ALIGN,1.0,
													GCC_LINE_BACKGROUND,"LightGoldenrod1",
													GCC_TYPE,(guint)GDL_ROOT,
													GCC_ID,(gulong)strtoul(k[0],NULL,10),
													GCC_STR1,k[1],
													GCC_VAL1,(gulong)size_gdl,
													GCC_BYTE_OFFSET,(gulong)0,
													GCC_RECEIVED_BYTES,(gulong)0,
													GCC_START_TIME,(guint64)0,
													GCC_LAST_10S_SPD,(gulong)0,
													GCC_PIXBUF,NULL,
													GCC_PIXBUF_DIRTY,(gboolean)TRUE,
													GCC_MISC_POINTER,NULL,
													-1);
				g_free(utf8_1);
			}

			/* and put its info inside it */
			second_level_add_newly_created_entries(gtm,gts,&iter,k,&update_dl);

			g_strfreev(k);
			g_ptr_array_index(glst_temp_array,i)=NULL;
			
			if(update_dl)
				gtk_tree_store_set(gts,&iter,GCC_PIXBUF_DIRTY,update_dl,-1);
		}
	}

	/* and free unnecessary data of the glst_temp_array */
	/* this loop should be unuseful because the temp_array should already by empty */
	for(i=0;i<glst_temp_array->len;i++)
	{
		gchar **k;

		k=g_ptr_array_index(glst_temp_array,i);
		if(k)
			g_strfreev(k);
	}
	g_ptr_array_free(glst_temp_array,TRUE);
	glst_temp_array=NULL;

	/* now, foreach first level, we check if the GCC_PIXBUF_DIRTY is set */
	scan_first_level_of_gdl_ctree_for_dirty_pixbuf(gtm,gts);

	SORT_RESTORE(,gtm)
	return;
}

/*********************************/
/* temporary store one GDL entry */
/*********************************/
void glst_content_fnc(const GString *in)
{
	char *t;
	gchar **fields;

	/* s format: ] "xxxxx"gdlID|localfname|fsize|offset|downloaded|start_time|speed|runningxfer|donexfer|autoscan */
	t=strchr(in->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;

	fields=g_strsplit(t,"|",0);
	g_ptr_array_add(glst_temp_array,fields);
}

static struct 
{
	const char *msg_type;
	void (*fnc)(const GString *);
	gboolean master_ignore;			/* TRUE= message from master to ignore */
} msg_fnc[]={
                              {"ERR  ]",err_fnc, FALSE},			/* done, not checked */
                              {"DEBUG]",dummy_fnc, FALSE},
                              {"INFO ]",dummy_fnc, FALSE},
                              {"OP   ]",op_fnc, TRUE},
                              {"ADMIN]",op_fnc, TRUE},
                              {"CHAT ]",chat_fnc, TRUE},			/* done, checked */
                              {"HUBNM]",hubnm_fnc, FALSE},		/* done, checked */
                              {"SREST]",srest_fnc, FALSE},		/* done, checked */
                              {"$USER]",dummy_fnc, TRUE},

                              {"$UL+ ]",update_lists, FALSE},		/* done, not checked */
                              {"$UL- ]",update_lists, FALSE},		/* same function on error and success */
                              {"$UL# ]",update_lists, FALSE},	/* done, not checked */
                              {"$DL+ ]",update_lists, FALSE},		/* done, not checked */
                              {"$DL- ]",remove_dl_xfer_on_success_fnc, FALSE},	/* done, not checked */
                              {"$DL# ]",update_lists, FALSE},	/* done, not checked */
                              {"$LS+ ]",update_lists, FALSE},		/* in progress */
                              {"$LS- ]",update_lists, FALSE},
                              {"$LS# ]",update_lists, FALSE},
										{"$UL= ]",update_lists, FALSE},
										{"$DL= ]",update_lists, FALSE},
										{"$LS= ]",update_lists, FALSE},
										{"ASTRT]",update_lists, FALSE},
										{"ASTOP]",update_lists, FALSE},
                              {"$DL~ ]",update_lists, FALSE},
										{"$LS~ ]",update_lists, FALSE},
										{"RFRSH]",update_lists, FALSE},

                              {"XFERR]",add_xferr_fnc, TRUE},
                              {"XFERQ]",add_xferq_fnc, TRUE},
                              {"CMDKB]",add_cmdkb_fnc, TRUE},
										{"BXFER]",begin_xfer_list_fnc, TRUE},
										{"EXFER]",end_xfer_list_fnc, TRUE},

										{"$ULST]",update_ul_list, TRUE},
                              {"HUB- ]",dummy_fnc, TRUE},
                              {"PRIV ]",pchat_fnc, TRUE},
                              {"HUBGO]",dummy_fnc, TRUE},
                              {"HUB+ ]",dummy_fnc, TRUE},
                              {"VAR  ]",var_fnc, TRUE},
                              {"PASWD]",enter_passwd_fnc, TRUE},
                              {"PRGBR]",prog_bar_fnc, TRUE},
										{"GLSTB]",glst_begin_fnc, FALSE},
										{"GLSTC]",glst_content_fnc, FALSE},
										{"GLSTE]",glst_end_fnc, FALSE},
										{"UALSB]",ualst_begin_fnc, FALSE},
										{"UALSC]",ualst_content_fnc, FALSE},
										{"UALSE]",ualst_end_fnc, FALSE},
										{"UALS+]",ualst_entry_add, FALSE},
										{"UALS-]",ualst_entry_remove, FALSE},

										{"LUSER]",luser_fnc, FALSE},
										{"LSCCH]",user_share_full_list_fnc, FALSE},
										{NULL,NULL}
				};


/**********************************************************************************/
/* this function is called when something comes from DCTC client                  */
/* data is always NULL, source == current_dctc_fd and condition == GTK_INPUT_READ */
/**********************************************************************************/
void process_data_from_dctc(gpointer data, gint source, GdkInputCondition condition)
{
	GString *input;
	int i;

	input=get_dctc_line(1);
	while(input!=NULL)
	{
		i=0;
		while(msg_fnc[i].msg_type!=NULL)
		{
			if(!strncmp(input->str,msg_fnc[i].msg_type,6))
			{
				(msg_fnc[i].fnc)(input);
				break;
			}
			i++;
		}

		g_string_free(input,TRUE);
		input=get_dctc_line(0);
	}
	return;
}

/**********************************************************************************/
/* this function is called when something comes from DCTC client                  */
/* data is always NULL, source == current_dctc_fd and condition == GTK_INPUT_READ */
/**********************************************************************************/
/* technically, the function performs exactly the same tasks as the previous one */
/* except it uses gdl_dctc instead of current_dctc *DCTC_COM                     */
/*********************************************************************************/
void process_data_from_gdl_dctc(gpointer data, gint source, GdkInputCondition condition)
{
	GString *input;
	int i;

	input=get_dctc_line_from_gdl_dctc(1);
	while(input!=NULL)
	{
		i=0;
		while(msg_fnc[i].msg_type!=NULL)
		{
			if(!strncmp(input->str,msg_fnc[i].msg_type,6))
			{
				if(msg_fnc[i].master_ignore==FALSE)
					(msg_fnc[i].fnc)(input);
				break;
			}
			i++;
		}

		g_string_free(input,TRUE);
		input=get_dctc_line_from_gdl_dctc(0);
	}
	return;
}

/************************************************************************************/
/* same function as previous one except it tries to send queued data to DCTC client */
/************************************************************************************/
/* data is always NULL, source == current_dctc_fd and condition == GTK_INPUT_WRITE */
/***********************************************************************************/
void process_data_to_dctc(gpointer data, gint source, GdkInputCondition condition)
{
	if(current_dctc!=NULL)
	{	/* the previous test should never fail */

		/* as long as there is something queued and it is possible to send data */
		/* on the socket, we send */
		while(current_dctc->write_q->len!=0)
		{
			char *str;
			int l;
			int ret;
		
			str=g_ptr_array_index(current_dctc->write_q,0);
			l=strlen(str);

			ret=send(current_dctc->dctc_fd,str,l,MSG_DONTWAIT);
			if(ret==l)
			{
				/* successfully sent => go to next one */
				g_ptr_array_remove_index(current_dctc->write_q,0);
				free(str);
			}
			else
			{
				/* oh oh, an error occurs */
				if(errno!=EAGAIN)
				{
					/* and it is a fatal one */
					close_dctc_com(&current_dctc);
					/* we can no longer do anything because the structure has been destroyed */
					gnome_app_error(GNOME_APP(main_window),_("Error while sending data to DCTC.\nConnection with DCTC terminated."));
					return;
				}
				break;
			}
		}

		if(current_dctc->write_q->len==0)
		{
			gdk_input_remove(current_dctc->tag_write);
			current_dctc->tag_write=-1;
		}
	}
	else
	{
		fprintf(stderr,"process_data_to_dctc: invalid handler.\n");
	}
}

/****************************************/
/* send data to the current DCTC client */
/* the function handles all errors      */
/****************************************/
void send_data_to_dctc(char *str)
{
	char *tm;

	if(current_dctc==NULL)
		return;

	/* if the wait_q is empty, we try to send the string without queueing */
	/* else the string is queued and gdk_input handler is enabled */
	if(current_dctc->write_q->len==0)
	{
		int l;
		int ret;
		int attempt=0;

		l=strlen(str);

		retry:
		ret=send(current_dctc->dctc_fd,str,l,MSG_DONTWAIT);
		if(ret==l)
			return;		/* everything ok, we end here */

		/* oh oh, an error occurs */
		if(errno!=EAGAIN)
		{
			if((errno==EPIPE)&&(attempt<2))
			{
				reconnect_dctc_com(current_dctc);
				if(current_dctc->dctc_fd!=-1)
				{
					attempt++;
					goto retry;
				}
			}
			close_dctc_com(&current_dctc);
			gnome_app_error(GNOME_APP(main_window),_("Error while sending data to DCTC.\nConnection with DCTC terminated."));
		}
	}

	/* if a queue still exists or the atomic send fails, the string is queued */
	if(current_dctc!=NULL)
	{
		/* the connection may be dead and the current_dctc becomes NULL */
		tm=strdup(str);
		if(tm==NULL)
		{
			close_dctc_com(&current_dctc);
			gnome_app_error(GNOME_APP(main_window),_("Out of memory.\nConnection with DCTC terminated."));
			return;
		}

		g_ptr_array_add(current_dctc->write_q,tm);

		if(current_dctc->tag_write==-1)
		{
			current_dctc->tag_write=gdk_input_add(current_dctc->dctc_fd,GDK_INPUT_WRITE,process_data_to_dctc,NULL);
		}
	}
}

/************************************************************************************/
/* same function as previous one except it tries to send queued data to DCTC client */
/************************************************************************************/
/* data is always NULL, source == current_dctc_fd and condition == GTK_INPUT_WRITE */
/***********************************************************************************/
void process_data_to_gdl_dctc(gpointer data, gint source, GdkInputCondition condition)
{
	if(gdl_dctc!=NULL)
	{	/* the previous test should never fail */

		/* as long as there is something queued and it is possible to send data */
		/* on the socket, we send */
		while(gdl_dctc->write_q->len!=0)
		{
			char *str;
			int l;
			int ret;
		
			str=g_ptr_array_index(gdl_dctc->write_q,0);
			l=strlen(str);

			ret=send(gdl_dctc->dctc_fd,str,l,MSG_DONTWAIT);
			if(ret==l)
			{
				/* successfully sent => go to next one */
				g_ptr_array_remove_index(gdl_dctc->write_q,0);
				free(str);
			}
			else
			{
				/* oh oh, an error occurs */
				if(errno!=EAGAIN)
				{
					/* and it is a fatal one */
					close_dctc_com(&gdl_dctc);
					/* we can no longer do anything because the structure has been destroyed */
					gnome_app_error(GNOME_APP(main_window),_("Error while sending data to DCTC.\nConnection with master_DCTC terminated."));
					return;
				}
				break;
			}
		}

		if(gdl_dctc->write_q->len==0)
		{
			gdk_input_remove(gdl_dctc->tag_write);
			gdl_dctc->tag_write=-1;
		}
	}
	else
	{
		fprintf(stderr,"process_data_to_gdl_dctc: invalid handler.\n");
	}
}

/************************************/
/* send data to the GDL DCTC client */
/* the function handles all errors  */
/************************************/
void send_data_to_gdl_dctc(char *str)
{
	char *tm;

	if(gdl_dctc==NULL)
	{
		connect_to_the_master_dctc();
		if(gdl_dctc==NULL)
			return;
	}

	/* if the wait_q is empty, we try to send the string without queueing */
	/* else the string is queued and gdk_input handler is enabled */
	if(gdl_dctc->write_q->len==0)
	{
		int l;
		int ret;
		int attempt=0;

		l=strlen(str);

		retry:
		ret=send(gdl_dctc->dctc_fd,str,l,MSG_DONTWAIT);
		if(ret==l)
			return;		/* everything ok, we end here */

		/* oh oh, an error occurs */
		if(errno!=EAGAIN)
		{
			if((errno==EPIPE)&&(attempt<2))
			{
				close_dctc_com(&gdl_dctc);
				connect_to_the_master_dctc();
				if((gdl_dctc!=NULL)&&(gdl_dctc->dctc_fd!=-1))
				{
					attempt++;
					goto retry;
				}
			}
			close_dctc_com(&gdl_dctc);
			gnome_app_error(GNOME_APP(main_window),_("Error while sending data to DCTC.\nConnection with master_DCTC terminated."));
		}
	}

	/* if a queue still exists or the atomic send fails, the string is queued */
	if(gdl_dctc!=NULL)
	{
		/* the connection may be dead and the current_dctc becomes NULL */
		tm=strdup(str);
		if(tm==NULL)
		{
			close_dctc_com(&gdl_dctc);
			gnome_app_error(GNOME_APP(main_window),_("Out of memory.\nConnection with DCTC terminated."));
			return;
		}

		g_ptr_array_add(gdl_dctc->write_q,tm);

		if(gdl_dctc->tag_write==-1)
		{
			gdl_dctc->tag_write=gdk_input_add(gdl_dctc->dctc_fd,GDK_INPUT_WRITE,process_data_to_gdl_dctc,NULL);
		}
	}
}

/* ------------------------------------------------------------------------------------------------------------- */
static int as_created=0;

/*************************************************************************/
/* create autoscan configuration window for the given entry of the ctree */
/*************************************************************************/
/* entry is a GtkCTreeNode, ct is a GtkCTree */
/*********************************************/
/* GTK2: to test */
/*****************/
static void create_autoscan_window_for_this_gdl_entry(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	guint ct_type;

	if(as_created==1)
		return;

	gtk_tree_model_get(model,iter,GCC_TYPE,&ct_type,-1);
	if(ct_type==GDL_ROOT)
	{
		GtkWidget *w;
		gulong gdl_id;
		char *local_filename;

		gtk_tree_model_get(model,iter,GCC_ID,&gdl_id,GCC_STR1,&local_filename,-1);

		w=get_widget_by_widget_name(main_window,"as_gid_label");
		if(w!=NULL)
		{
			char bf[64];
			sprintf(bf,"%lu",gdl_id);
			gtk_label_set(GTK_LABEL(w),bf);
		}

		set_gtk_label_by_name(main_window,"as_fname_label",local_filename);

		/* switch to the find config tab */
  		w=get_widget_by_widget_name(main_window,"main_notebook");
  		gtk_notebook_set_page(GTK_NOTEBOOK(w),FIND_CONFIG_TAB);

		w=get_widget_by_widget_name(main_window,"gdl_as_pattern_entry");
		if(w!=NULL)
		{
			gtk_editable_set_position(GTK_EDITABLE(w),0);
			gtk_editable_delete_text(GTK_EDITABLE(w),0,-1);
			gtk_widget_grab_focus(w);
		}

		free(local_filename);
	}
}

/* ------------------------------------------------------------------------------------------------------------- */
/*****************************************************************/
/* create autoscan configuration window for selected GDL entries */
/*****************************************************************/
/* GTK2: to test */
/*****************/
void create_autoscan_window_for_selected_gdl_entry(void)
{
	GtkWidget *w;
	GtkTreeSelection *slc;

	/* get the nickname of the remote side of the private chat */
	w=get_widget_by_widget_name(main_window,"gdl_ctree");
	if(w==NULL)
		return;

	slc=gtk_tree_view_get_selection(GTK_TREE_VIEW(w));
	as_created=0;
	gtk_tree_selection_selected_foreach(slc,create_autoscan_window_for_this_gdl_entry,NULL);

	update_lists_force_gdl(NULL);
}

/* ------------------------------------------------------------------------------------------------------------- */
/**************************************************************/
/* create end program window for the given entry of the ctree */
/**************************************************************/
/* entry is a GtkCTreeNode, ct is a GtkCTree */
/*********************************************/
/* GTK2: to test */
/*****************/
static void create_script_window_for_this_gdl_entry(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	guint ct_type;

	if(as_created==1)
		return;

	gtk_tree_model_get(model,iter,GCC_TYPE,&ct_type,-1);
	if(ct_type==GDL_ROOT)
	{
		GtkWidget *w;
		gulong gdl_id;
		char *local_filename;

		gtk_tree_model_get(model,iter,GCC_ID,&gdl_id,GCC_STR1,&local_filename,-1);

		w=get_widget_by_widget_name(main_window,"gdl_script_gid_label");
		if(w!=NULL)
		{
			char bf[64];
			sprintf(bf,"%lu",gdl_id);
			gtk_label_set(GTK_LABEL(w),bf);
		}

		set_gtk_label_by_name(main_window,"gdl_script_fname_label",local_filename);

		/* switch to the find config tab */
  		w=get_widget_by_widget_name(main_window,"main_notebook");
  		gtk_notebook_set_page(GTK_NOTEBOOK(w),FIND_CONFIG_TAB);

		w=get_widget_by_widget_name(main_window,"gdl_endname_filename_entry");
		if(w!=NULL)
		{
			gtk_editable_set_position(GTK_EDITABLE(w),0);
			gtk_editable_delete_text(GTK_EDITABLE(w),0,-1);
			gtk_widget_grab_focus(w);
		}
		free(local_filename);
	}
}

/* ------------------------------------------------------------------------------------------------------------- */
/*************************************************/
/* create rename window for selected GDL entries */
/*************************************************/
/* GTK2: to test */
/*****************/
void create_script_window_for_selected_gdl_entry(void)
{
	GtkWidget *w;
	GtkTreeSelection *slc;

	/* get the nickname of the remote side of the private chat */
	w=get_widget_by_widget_name(main_window,"gdl_ctree");
	if(w==NULL)
		return;

	slc=gtk_tree_view_get_selection(GTK_TREE_VIEW(w));
	as_created=0;
	gtk_tree_selection_selected_foreach(slc,create_script_window_for_this_gdl_entry,NULL);

	update_lists_force_gdl(NULL);
}

/* ------------------------------------------------------------------------------------------------------------- */
/*********************************************************/
/* create rename window for the given entry of the ctree */
/*********************************************************/
/* entry is a GtkCTreeNode, ct is a GtkCTree */
/*********************************************/
static void create_rename_window_for_this_gdl_entry(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	guint ct_type;

	if(as_created==1)
		return;

	gtk_tree_model_get(model,iter,GCC_TYPE,&ct_type,-1);
	if(ct_type==GDL_ROOT)
	{
		GtkWidget *w;
		gulong gdl_id;
		char *local_filename;

		gtk_tree_model_get(model,iter,GCC_ID,&gdl_id,GCC_STR1,&local_filename,-1);

		w=get_widget_by_widget_name(main_window,"gdl_rename_gid_label");
		if(w!=NULL)
		{
			char bf[64];
			sprintf(bf,"%lu",gdl_id);
			gtk_label_set(GTK_LABEL(w),bf);
		}

		set_gtk_label_by_name(main_window,"gdl_rename_fname_label",local_filename);

		/* switch to the find config tab */
  		w=get_widget_by_widget_name(main_window,"main_notebook");
  		gtk_notebook_set_page(GTK_NOTEBOOK(w),FIND_CONFIG_TAB);

		w=get_widget_by_widget_name(main_window,"gdl_rename_file_entry");
		if(w!=NULL)
		{
			gtk_editable_set_position(GTK_EDITABLE(w),0);
			gtk_editable_delete_text(GTK_EDITABLE(w),0,-1);
			gtk_widget_grab_focus(w);
		}
		free(local_filename);
	}
}

/* ---------------------------------------------------------------------------------------------------- */
/*************************************************/
/* create rename window for selected GDL entries */
/*************************************************/
/* GTK2: to test */
/*****************/
void create_rename_window_for_selected_gdl_entry(void)
{
	GtkWidget *w;
	GtkTreeSelection *slc;

	/* get the nickname of the remote side of the private chat */
	w=get_widget_by_widget_name(main_window,"gdl_ctree");
	if(w==NULL)
		return;

	slc=gtk_tree_view_get_selection(GTK_TREE_VIEW(w));
	as_created=0;
	gtk_tree_selection_selected_foreach(slc,create_rename_window_for_this_gdl_entry,NULL);

	update_lists_force_gdl(NULL);
}

/* ---------------------------------------------------------------------------------------------------- */
/**********************************************************************/
/* kill upload or download task using their ID (and /KILL or /KILLKB) */
/**********************************************************************/
void kill_selected_entry(char *cmd,char *clist_name, int clist_row)
{
	foreach_selected_entry_send_cmd_ulong_numeric_col(cmd,clist_name,clist_row);
	update_lists(NULL);
}

/* ---------------------------------------------------------------------------------------------------- */
typedef struct
{
	const char *cmd;
	int column_num;
	void (*send_data_to_fnc)(char *);
} TMP_FORSELECTCMD;

/*****************************************************************************/
/* send the given command to one selected entry from a list containing users */
/*****************************************************************************/
static void for_one_selected_entry_send_cmd(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	TMP_FORSELECTCMD *tf=data;
	char buf[51200];
	char *nickname;

	gtk_tree_model_get(model,iter,tf->column_num,&nickname,-1);
	sprintf(buf,"%s %s\n",tf->cmd,nickname);
	(tf->send_data_to_fnc)(buf);
	free(nickname);
}

/*********************************************************************************************/
/* foreach selected line, send the given command followed by the content of the given column */
/*********************************************************************************************/
/* the clist_name is the name of a clist containing a GtkListStore */
/* the clist column is G_TYPE_STRING                               */
/*******************************************************************/
void foreach_selected_entry_send_cmd(char *cmd,char *clist_name, int column_num)
{
	GtkWidget *w;
	GtkTreeSelection *slc;
	TMP_FORSELECTCMD tf;

	w=get_widget_by_widget_name(main_window,clist_name);
	if(w==NULL)
		return;

	tf.cmd=cmd;
	tf.column_num=column_num;
	tf.send_data_to_fnc=send_data_to_dctc;

	slc=gtk_tree_view_get_selection(GTK_TREE_VIEW(w));
	gtk_tree_selection_selected_foreach(slc,for_one_selected_entry_send_cmd,&tf);
}

void foreach_selected_entry_send_cmd_to_gdl(char *cmd,char *clist_name, int column_num)
{
	GtkWidget *w;
	GtkTreeSelection *slc;
	TMP_FORSELECTCMD tf;

	w=get_widget_by_widget_name(main_window,clist_name);
	if(w==NULL)
		return;

	tf.cmd=cmd;
	tf.column_num=column_num;
	tf.send_data_to_fnc=send_data_to_gdl_dctc;

	slc=gtk_tree_view_get_selection(GTK_TREE_VIEW(w));
	gtk_tree_selection_selected_foreach(slc,for_one_selected_entry_send_cmd,&tf);
}

/* ---------------------------------------------------------------------------------------------------- */
/*****************************************************************************/
/* send the given command to one selected entry from a list containing users */
/*****************************************************************************/
static void for_one_selected_entry_send_cmd_with_glob_user(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	TMP_FORSELECTCMD *tf=data;
	char buf[51200];
	GLOB_USER *gu;

	gtk_tree_model_get(model,iter,tf->column_num,&gu,-1);
	if(gu!=NULL)
	{
		sprintf(buf,"%s %s\n",tf->cmd,gu->unfmt_nick);
		(tf->send_data_to_fnc)(buf);
	}
}

/*************************************************************************************/
/* foreach selected line, send the given command followed by the nickname of the row */
/*************************************************************************************/
/* the clist_name is the name of a clist containing a GtkListStore */
/* the clist column is G_TYPE_POINTER containing a GLOB_USER *     */
/*******************************************************************/
void foreach_selected_entry_send_cmd_with_glob_user(char *clist_name, int column_num,char *cmd)
{
	GtkWidget *w;
	GtkTreeSelection *slc;
	TMP_FORSELECTCMD tf;

	w=get_widget_by_widget_name(main_window,clist_name);
	if(w==NULL)
		return;

	tf.cmd=cmd;
	tf.column_num=column_num;
	tf.send_data_to_fnc=send_data_to_dctc;

	slc=gtk_tree_view_get_selection(GTK_TREE_VIEW(w));
	gtk_tree_selection_selected_foreach(slc,for_one_selected_entry_send_cmd_with_glob_user,&tf);
}

void foreach_selected_entry_send_cmd_to_gdl_with_glob_user(char *clist_name, int column_num, char *cmd)
{
	GtkWidget *w;
	GtkTreeSelection *slc;
	TMP_FORSELECTCMD tf;

	w=get_widget_by_widget_name(main_window,clist_name);
	if(w==NULL)
		return;

	tf.cmd=cmd;
	tf.column_num=column_num;
	tf.send_data_to_fnc=send_data_to_gdl_dctc;

	slc=gtk_tree_view_get_selection(GTK_TREE_VIEW(w));
	gtk_tree_selection_selected_foreach(slc,for_one_selected_entry_send_cmd_with_glob_user,&tf);
}

/* ---------------------------------------------------------------------------------------------------- */
/*****************************************************************************/
/* send the given command to one selected entry from a list containing users */
/*****************************************************************************/
/* the clist column is G_TYPE_ULONG                                */
/*******************************************************************/
/* GTK 2: ok */
/*************/
static void for_one_selected_entry_send_cmd_ulong_numeric(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	TMP_FORSELECTCMD *tf=data;
	char buf[51200];
	unsigned long val;

	gtk_tree_model_get(model,iter,tf->column_num,&val,-1);
	sprintf(buf,"%s %lu\n",tf->cmd,val);
	(tf->send_data_to_fnc)(buf);
}

/*********************************************************************************************/
/* foreach selected line, send the given command followed by the content of the given column */
/*********************************************************************************************/
/* the clist_name is the name of a clist containing a GtkListStore */
/* the clist column is G_TYPE_ULONG                                */
/*******************************************************************/
/* GTK2: to test */
/*****************/
static void foreach_selected_entry_send_cmd_ulong_numeric_col(char *cmd,char *clist_name, int column_num)
{
	GtkWidget *w;
	GtkTreeSelection *slc;
	TMP_FORSELECTCMD tf;

	w=get_widget_by_widget_name(main_window,clist_name);
	if(w==NULL)
		return;

	tf.cmd=cmd;
	tf.column_num=column_num;
	tf.send_data_to_fnc=send_data_to_dctc;

	slc=gtk_tree_view_get_selection(GTK_TREE_VIEW(w));
	gtk_tree_selection_selected_foreach(slc,for_one_selected_entry_send_cmd_ulong_numeric,&tf);
}

