# -*- coding: utf-8 -*-
#
# Author: Ingelrest François (Athropos@gmail.com)
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

import cgi, collections, gtk, gui, media, modules, os, shutil, tools, urllib

from gui             import fileChooser, help, questionMsgBox, extTreeview, extListview, progressDlg, selectPath
from tools           import consts, prefs, pickleLoad, pickleSave
from gettext         import gettext as _
from os.path         import isdir, isfile
from gobject         import idle_add, TYPE_STRING, TYPE_INT, TYPE_PYOBJECT
from gui.progressDlg import ProgressDlg

MOD_INFO = ('Library', _('Library'), _('Organize your music by tags instead of files'), [], False, True)
MOD_L10N = MOD_INFO[modules.MODINFO_L10N]


# Constants
ROOT_PATH                = os.path.join(consts.dirCfg, 'Library') # Path where libraries are stored
PREFS_DEFAULT_PREFIXES   = {'the ': None}                         # Prefixes are put at the end of artists' names
PREFS_DEFAULT_LIBRARIES  = {}                                     # No libraries at first
PREFS_DEFAULT_TREE_STATE = {}                                     # No state at first


# Information associated with libraries
(
    LIB_PATH,        # Physical location of media files
    LIB_NB_ARTISTS,  # Number of artists
    LIB_NB_ALBUMS,   # Number of albums
    LIB_NB_TRACKS    # Number of tracks
) = range(4)


# Information associated with artists
(
    ART_NAME,       # Its name
    ART_INDEX,      # Name of the directory: avoid the use of the artist name as a filename (may contain invalid characters)
    ART_NB_ALBUMS   # How many albums
) = range(3)


# Information associated with albums
(
    ALB_NAME,       # Its name
    ALB_INDEX,      # Name of the file: avoid the use of the artist name as a filename (may contain invalid characters)
    ALB_NB_TRACKS,  # Number of tracks
    ALB_LENGTH      # Complete duration (include all tracks)
) = range(4)


# Possible types for a node of the tree
(
    TYPE_ARTIST,    # Artist
    TYPE_ALBUM,     # Album
    TYPE_TRACK,     # Single track
    TYPE_HEADER,    # Alphabetical header
    TYPE_NONE       # Used for fake children
) = range(5)


# The format of a row in the treeview
(
    ROW_PIXBUF,    # Item icon
    ROW_NAME,      # Item name
    ROW_TYPE,      # The type of the item (e.g., directory, file)
    ROW_FULLPATH,  # The full path to the item
    ROW_TAGS       # The tags of the track, valid only for rows of type TYPE_TRACK
) = range(5)


class Library(modules.Module):


    def __init__(self):
        """ Constructor """
        modules.Module.__init__(self, (consts.MSG_EVT_APP_STARTED, consts.MSG_EVT_EXPLORER_CHANGED, consts.MSG_EVT_MOD_LOADED,
                                       consts.MSG_EVT_APP_QUIT,    consts.MSG_EVT_MOD_UNLOADED))


    def onAppStarted(self):
        """ This is the real initialization function, called when the module has been loaded """
        self.popup     = None
        self.currLib   = None
        self.cfgWindow = None
        self.libraries = prefs.get(__name__, 'libraries',  PREFS_DEFAULT_LIBRARIES)
        self.treeState = prefs.get(__name__, 'tree-state', PREFS_DEFAULT_TREE_STATE)
        # Create the tree
        txtRdr    = gtk.CellRendererText()
        pixbufRdr = gtk.CellRendererPixbuf()

        columns = (('',   [(pixbufRdr, gtk.gdk.Pixbuf), (txtRdr, TYPE_STRING)], False),
                   (None, [(None, TYPE_INT)],                                   False),
                   (None, [(None, TYPE_STRING)],                                False),
                   (None, [(None, TYPE_PYOBJECT)],                              False))

        self.tree = extTreeview.ExtTreeView(columns, True)

        self.tree.get_column(0).set_cell_data_func(txtRdr,    self.__drawCell)
        self.tree.get_column(0).set_cell_data_func(pixbufRdr, self.__drawCell)
        self.tree.setIsDraggableFunc(self.__isDraggable)
        self.tree.setDNDSources([consts.DND_TARGETS[consts.DND_DAP_TRACKS]])
        # Scroll window
        self.scrolled = gtk.ScrolledWindow()
        self.scrolled.add(self.tree)
        self.scrolled.set_shadow_type(gtk.SHADOW_IN)
        self.scrolled.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        self.scrolled.show()
        # GTK handlers
        self.tree.connect('drag-data-get',              self.onDragDataGet)
        self.tree.connect('key-press-event',            self.onKeyPressed)
        self.tree.connect('exttreeview-row-expanded',   self.onRowExpanded)
        self.tree.connect('exttreeview-button-pressed', self.onButtonPressed)


    def __drawCell(self, column, cell, model, iter):
        """ Use a different background color for alphabetical headers """
        if model.get_value(iter, ROW_TYPE) == TYPE_HEADER: cell.set_property('cell-background-gdk', self.tree.style.bg[gtk.STATE_PRELIGHT])
        else:                                              cell.set_property('cell-background',     None)


    def __isDraggable(self):
        """ Return True if the selection is draggable """
        for row in self.tree.iterSelectedRows():
            if row[ROW_TYPE] == TYPE_HEADER:
                return False
        return True


    def refreshLibrary(self, parent, libName, path, creation=False):
        """ Refresh the given library, must be called through idle_add() """
        # First show a progress dialog
        if creation: header = _('Creating library')
        else:        header = _('Refreshing library')

        progress = ProgressDlg(parent, header, _('The directory is scanned for media files. This can take some time.\nPlease wait.'))
        yield True

        db         = {}                                         # The dictionnary used to create the library
        queue      = collections.deque((path,))                 # Faster structure for appending/removing elements
        libPath    = os.path.join(ROOT_PATH, libName)           # Location of the library
        mediaFiles = []                                         # All media files found
        newLibrary = {}                                         # Reflect the current file structure of the library
        oldLibrary = pickleLoad(os.path.join(libPath, 'files')) # Previous file structure of the same library

        while len(queue) != 0:
            currDir      = queue.pop()
            currDirMTime = os.stat(currDir).st_mtime

            # Retrieve previous information on the current directory, if any
            if currDir in oldLibrary: oldDirMTime, oldDirectories, oldFiles = oldLibrary[currDir]
            else:                     oldDirMTime, oldDirectories, oldFiles = -1, [], {}

            # If the directory has not been modified, keep old information
            if currDirMTime == oldDirMTime:
                files, directories = oldFiles, oldDirectories
            else:
                files, directories = {}, []
                for (filename, fullPath) in tools.listDir(currDir):
                    if isdir(fullPath):
                        directories.append(fullPath)
                    elif isfile(fullPath) and media.isSupported(filename):
                        if filename in oldFiles: files[filename] = oldFiles[filename]
                        else:                    files[filename] = [-1, [0, '', '', '', 0, fullPath]]

            # Determine which files need to be updated
            for filename, (oldMTime, tags) in files.iteritems():
                mTime = os.stat(tags[media.NFO_FIL]).st_mtime
                if mTime != oldMTime:
                    files[filename] = [mTime, media.readInfo((tags[media.NFO_FIL],))[0]]

            newLibrary[currDir] = (currDirMTime, directories, files)
            mediaFiles.extend([file for mTime, file in files.itervalues()])
            queue.extend(directories)

            # Update the progress dialog
            try:
                progress.pulse(_('Scanning directories (%u tracks found)' % len(mediaFiles)))
                yield True
            except progressDlg.CancelledException:
                progress.destroy()
                if creation:
                    shutil.rmtree(libPath)
                yield False

        # From now on, the process should not be cancelled
        progress.setCancellable(False)
        if creation: progress.pulse(_('Creating library...'))
        else:        progress.pulse(_('Refreshing library...'))
        yield True

        # Create the database
        for track in mediaFiles:
            album, artist = track[media.NFO_ALB], track[media.NFO_ART]

            if len(track) > 6 and track[media.NFO_AAR] != consts.UNKNOWN_ALBUM_ARTIST:
                artist = track[media.NFO_AAR]

            if artist in db:
                allAlbums = db[artist]
                if album in allAlbums: allAlbums[album].append(track)
                else:                  allAlbums[album] = [track]
            else:
                db[artist] = {album: [track]}

        progress.pulse()
        yield True

        # If an artist name begins with a known prefix, put it at the end (e.g., Future Sound of London (The))
        prefixes = prefs.get(__name__, 'prefixes', PREFS_DEFAULT_PREFIXES)
        for artist in db.keys():
            artistLower = artist.lower()
            for prefix in prefixes:
                if artistLower.startswith(prefix):
                    db[artist[len(prefix):] + ' (%s)' % artist[:len(prefix)-1]] = db[artist]
                    del db[artist]

        progress.pulse()
        yield True

        # Re-create the library structure on the disk
        if isdir(libPath):
            shutil.rmtree(libPath)
            os.mkdir(libPath)

        overallNbAlbums  = 0
        overallNbTracks  = 0
        overallNbArtists = len(db)

        # The 'artists' file contains all known artists with their index, the 'files' file contains the file structure of the root path
        allArtists = sorted([(artist, str(indexArtist), len(db[artist])) for indexArtist, artist in enumerate(db)], key = lambda a: a[0].lower())
        pickleSave(os.path.join(libPath, 'files'),   newLibrary)
        pickleSave(os.path.join(libPath, 'artists'), allArtists)

        for (artist, indexArtist, nbAlbums) in allArtists:
            artistPath       = os.path.join(libPath, indexArtist)
            overallNbAlbums += nbAlbums
            os.mkdir(artistPath)

            albums = []
            for index, (name, tracks) in enumerate(db[artist].iteritems()):
                length           = sum([track[media.NFO_LEN] for track in tracks])
                overallNbTracks += len(tracks)

                albums.append((name, str(index), len(tracks), length))
                pickleSave(os.path.join(artistPath, str(index)), sorted(tracks, key = lambda track: track[media.NFO_NUM]))

            albums.sort(key = lambda album: album[0].lower())
            pickleSave(os.path.join(artistPath, 'albums'), albums)
            progress.pulse()
            yield True

        self.libraries[libName] = (path, overallNbArtists, overallNbAlbums, overallNbTracks)
        self.fillLibraryList()
        if creation:
            modules.postMsg(consts.MSG_CMD_EXPLORER_ADD, {'modName': MOD_L10N, 'expName': libName, 'icon': None, 'widget': self.scrolled})
        progress.destroy()

        # If the refreshed library is currently displayed, refresh the treeview as well
        if self.currLib == libName:
            treeState = self.tree.saveState(ROW_NAME)
            self.loadLibrary(self.tree, self.currLib)
            self.tree.restoreState(treeState, ROW_NAME)

        yield False


    def __getTracksFromSelectedRows(self, tree):
        """ Return a list of tracks with all the associated tags, based on selected rows """
        tracks = []
        for row in tree.getSelectedRows():
            if row[ROW_TYPE] == TYPE_TRACK:
                tracks.append(row[ROW_TAGS])
            elif row[ROW_TYPE] == TYPE_ALBUM:
                tracks.extend(pickleLoad(row[ROW_FULLPATH]))
            elif row[ROW_TYPE] == TYPE_ARTIST:
                for album in pickleLoad(os.path.join(row[ROW_FULLPATH], 'albums')):
                    tracks.extend(pickleLoad(os.path.join(row[ROW_FULLPATH], album[ALB_INDEX])))

        for track in tracks:
            track[media.NFO_FIL] = 'file://' + track[media.NFO_FIL]

        return tracks


    def playSelection(self, tree, replace):
        """ Replace/extend the tracklist """
        tracks = self.__getTracksFromSelectedRows(tree)

        if replace: modules.postMsg(consts.MSG_CMD_TRACKLIST_SET_EXT, {'files': tracks, 'playNow': True})
        else:       modules.postMsg(consts.MSG_CMD_TRACKLIST_ADD_EXT, {'files': tracks})


    def showPopupMenu(self, tree, button, time, path):
        """ Show a popup menu """
        if self.popup is None:
            self.popup = tools.loadGladeFile('LibraryMenu.glade')
            self.popup.get_widget('menu-popup').show_all()
            # Connect handlers
            self.popup.get_widget('item-collapse').connect('activate', lambda widget: self.tree.collapse_all())
            self.popup.get_widget('item-play').connect('activate',     lambda widget: self.playSelection(tree, True))
            self.popup.get_widget('item-add').connect('activate',      lambda widget: self.playSelection(tree, False))
            self.popup.get_widget('item-refresh').connect('activate',  lambda widget: idle_add(self.refreshLibrary(None, self.currLib, self.libraries[self.currLib][LIB_PATH]).next))

        # Determine whether at least one selected item is playable
        playable = False
        for row in self.tree.iterSelectedRows():
            if row[ROW_TYPE] != TYPE_HEADER:
                playable = True
                break

        # Show the popup menu
        self.popup.get_widget('item-add').set_sensitive(playable)
        self.popup.get_widget('item-play').set_sensitive(playable)
        self.popup.get_widget('menu-popup').popup(None, None, None, button, time)


    def loadLibrary(self, tree, name):
        """ Load the given library """
        rows     = []
        path     = os.path.join(ROOT_PATH, name)
        prevChar = None

        # Create the rows, with alphabetical header is needed
        for artist in pickleLoad(os.path.join(path, 'artists')):
            currChar = artist[ART_NAME][0]
            if prevChar is None or (prevChar != currChar and not (prevChar.isdigit() and currChar.isdigit())):
                prevChar = currChar
                if currChar.isdigit(): rows.append((None, '<b>0 - 9</b>',         TYPE_HEADER, None, None))
                else:                  rows.append((None, '<b>%s</b>' % currChar, TYPE_HEADER, None, None))

            rows.append((consts.icoDir, cgi.escape(artist[ART_NAME]), TYPE_ARTIST, os.path.join(path, artist[ART_INDEX]), None))

        # Insert all rows, and then add a fake child to each artist
        tree.replaceContent(rows)
        for node in tree.iterChildren(None):
            if tree.getItem(node, ROW_TYPE) == TYPE_ARTIST:
                tree.appendRow((None, '', TYPE_NONE, '', None), node)


    def loadAlbums(self, tree, node, fakeChild):
        """ Initial load of all albums of the given node, assuming it is of type TYPE_ARTIST """
        allAlbums = pickleLoad(os.path.join(tree.getItem(node, ROW_FULLPATH), 'albums'))
        path      = tree.getItem(node, ROW_FULLPATH)
        rows      = [(consts.icoMediaDir, '<span size="smaller" foreground="#909090">[%s]</span>  %s' % (tools.sec2str(album[ALB_LENGTH], True), cgi.escape(album[ALB_NAME])), TYPE_ALBUM, os.path.join(path, album[ALB_INDEX]), None) for album in allAlbums]

        # Add all the rows, and then add a fake child to each of them
        tree.freeze_child_notify()
        tree.appendRows(rows, node)
        tree.removeRow(fakeChild)
        for child in tree.iterChildren(node):
            tree.appendRow((None, '', TYPE_NONE, '', None), child)
        tree.thaw_child_notify()


    def loadTracks(self, tree, node, fakeChild):
        """ Initial load of all tracks of the given node, assuming it is of type TYPE_ALBUM """
        allTracks = pickleLoad(tree.getItem(node, ROW_FULLPATH))
        rows      = [(consts.icoMediaFile, '%02u. %s' % (track[media.NFO_NUM], cgi.escape(track[media.NFO_TIT])), TYPE_TRACK, track[media.NFO_FIL], track) for track in allTracks]

        tree.appendRows(rows, node)
        tree.removeRow(fakeChild)


    # --== GTK handlers ==--


    def onRowExpanded(self, tree, node):
        """ Populate the expanded row if needed (e.g., it still has only a fake child) """
        child = tree.getChild(node, 0)
        if tree.getItem(child, ROW_TYPE) == TYPE_NONE:
            if tree.getItem(node, ROW_TYPE) == TYPE_ARTIST: self.loadAlbums(tree, node, child)
            else:                                           self.loadTracks(tree, node, child)


    def onButtonPressed(self, tree, event, path):
        """ A mouse button has been pressed """
        if event.button == 3:
            self.showPopupMenu(tree, event.button, event.time, path)
        elif event.button == 1 and event.type == gtk.gdk._2BUTTON_PRESS and path is not None:
            if   tree.getItem(path, ROW_PIXBUF) != consts.icoDir: self.playSelection(tree, True)
            elif tree.row_expanded(path):                         tree.collapse_row(path)
            else:                                                 tree.expand_row(path, False)


    def onKeyPressed(self, tree, event):
        """ A key has been pressed """
        if gtk.gdk.keyval_name(event.keyval) == 'F5':
            idle_add(self.refreshLibrary(None, self.currLib, self.libraries[self.currLib][LIB_PATH]).next)


    def onDragDataGet(self, tree, context, selection, info, time):
        """ Provide information about the data being dragged """
        tracks    = self.__getTracksFromSelectedRows(tree)
        allFields = '\n'.join([' '.join([urllib.pathname2url(str(field)) for field in track]) for track in tracks])

        selection.set(consts.DND_TARGETS[consts.DND_DAP_TRACKS][0], 8, allFields)


    def addAllExplorers(self):
        """ Add all libraries to the Explorer module """
        for (name, (path, nbArtists, nbAlbums, nbTracks)) in self.libraries.iteritems():
            modules.postMsg(consts.MSG_CMD_EXPLORER_ADD, {'modName': MOD_L10N, 'expName': name, 'icon': None, 'widget': self.scrolled})


    def removeAllExplorers(self):
        """ Remove all libraries from the Explorer module """
        for (name, (path, nbArtists, nbAlbums, nbTracks)) in self.libraries.iteritems():
            modules.postMsg(consts.MSG_CMD_EXPLORER_REMOVE, {'modName': MOD_L10N, 'expName': name})


   # --== Message handler ==--


    def handleMsg(self, msg, params):
        """ Handle messages sent to this module """
        if msg == consts.MSG_EVT_APP_STARTED or msg == consts.MSG_EVT_MOD_LOADED:
            self.onAppStarted()
            idle_add(self.addAllExplorers)

        elif msg == consts.MSG_EVT_EXPLORER_CHANGED and params['modName'] == MOD_L10N and params['expName'] != self.currLib:
            # Save the state of the current library
            if self.currLib is not None:
                self.treeState[self.currLib] = self.tree.saveState(ROW_NAME)

            # Switch to the new one
            self.currLib = params['expName']
            self.loadLibrary(self.tree, self.currLib)

            # Restore the state of the new library
            if len(self.tree) != 0 and self.currLib in self.treeState:
                self.tree.restoreState(self.treeState[self.currLib], ROW_NAME)

        elif msg == consts.MSG_EVT_APP_QUIT or msg == consts.MSG_EVT_MOD_UNLOADED:
            self.treeState[self.currLib] = self.tree.saveState(ROW_NAME)
            prefs.set(__name__, 'tree-state', self.treeState)
            prefs.set(__name__, 'libraries',  self.libraries)
            self.removeAllExplorers()


    # --== Configuration ==--


    def configure(self, parent):
        """ Show the configuration dialog """
        if self.cfgWindow is None:
            self.cfgWindow = gui.window.Window('Library.glade', 'vbox1', __name__, MOD_L10N, 370, 400)
            # Create the list of libraries
            txtRdr  = gtk.CellRendererText()
            pixRdr  = gtk.CellRendererPixbuf()
            columns = ((None, [(txtRdr, TYPE_STRING)],                           0, False),
                       ('',   [(pixRdr, gtk.gdk.Pixbuf), (txtRdr, TYPE_STRING)], 2, False))

            self.cfgList = extListview.ExtListView(columns, sortable=False, useMarkup=True)
            self.cfgList.set_headers_visible(False)
            self.cfgWindow.getWidget('scrolledwindow1').add(self.cfgList)
            # Connect handlers
            self.cfgList.connect('key-press-event', self.onCfgKeyboard)
            self.cfgList.get_selection().connect('changed', self.onCfgSelectionChanged)
            self.cfgWindow.getWidget('btn-add').connect('clicked', self.onAddLibrary)
            self.cfgWindow.getWidget('btn-rename').connect('clicked', self.onRenameLibrary)
            self.cfgWindow.getWidget('btn-remove').connect('clicked', lambda btn: self.removeSelectedLibraries(self.cfgList))
            self.cfgWindow.getWidget('btn-refresh').connect('clicked', self.onRefresh)
            self.cfgWindow.getWidget('btn-ok').connect('clicked', lambda btn: self.cfgWindow.hide())
            self.cfgWindow.getWidget('btn-cancel').connect('clicked', lambda btn: self.cfgWindow.hide())
            self.cfgWindow.getWidget('btn-help').connect('clicked', self.onHelp)

        if not self.cfgWindow.isVisible():
            self.fillLibraryList()
            self.cfgWindow.getWidget('btn-ok').grab_focus()

        self.cfgWindow.show()


    def onRefresh(self, btn):
        """ Refresh the first selected library """
        name = self.cfgList.getSelectedRows()[0][0]
        idle_add(self.refreshLibrary(self.cfgWindow, name, self.libraries[name][LIB_PATH]).next)


    def onAddLibrary(self, btn):
        """ Let the user create a new library """
        result = selectPath.SelectPath(MOD_L10N, self.cfgWindow, self.libraries.keys(), ['/']).run()

        if result is not None:
            name, path = result
            # Make sure that the root directory of all libraries exists
            if not isdir(ROOT_PATH):
                os.mkdir(ROOT_PATH)
            # Start from an empty library
            libPath = os.path.join(ROOT_PATH, name)
            if isdir(libPath):
                shutil.rmtree(libPath)
            os.mkdir(libPath)
            pickleSave(os.path.join(libPath, 'files'), {})
            # And 'refresh' it
            idle_add(self.refreshLibrary(self.cfgWindow, name, path, True).next)


    def renameLibrary(self, oldName, newName):
        """ Rename a library """
        self.libraries[newName] = self.libraries[oldName]
        del self.libraries[oldName]

        oldPath = os.path.join(ROOT_PATH, oldName)
        newPath = os.path.join(ROOT_PATH, newName)
        shutil.move(oldPath, newPath)

        modules.postMsg(consts.MSG_CMD_EXPLORER_REMOVE, {'modName': MOD_L10N, 'expName': oldName})
        modules.postMsg(consts.MSG_CMD_EXPLORER_ADD,    {'modName': MOD_L10N, 'expName': newName, 'icon': None, 'widget': self.scrolled})


    def onRenameLibrary(self, btn):
        """ Let the user rename a library """
        name         = self.cfgList.getSelectedRows()[0][0]
        forbidden    = [libName for libName in self.libraries if libName != name]
        pathSelector = selectPath.SelectPath(MOD_L10N, self.cfgWindow, forbidden, ['/'])

        pathSelector.setPathSelectionEnabled(False)
        result = pathSelector.run(name, self.libraries[name][LIB_PATH])

        if result is not None and result[0] != name:
            self.renameLibrary(name, result[0])
            self.fillLibraryList()


    def fillLibraryList(self):
        """ Fill the list of libraries """
        if self.cfgWindow is not None:
            rows = [(name, consts.icoBtnDir, '<b>%s</b>\n<small>%s - %u %s</small>' % (cgi.escape(name), cgi.escape(path), nbTracks, cgi.escape(_('tracks'))))
                    for name, (path, nbArtists, nbAlbums, nbTracks) in sorted(self.libraries.iteritems())]
            self.cfgList.replaceContent(rows)


    def removeSelectedLibraries(self, list):
        """ Remove all selected libraries """
        if list.getSelectedRowsCount() == 1:
            remark   = _('You will be able to recreate this library later on if you wish so.')
            question = _('Remove the selected library?')
        else:
            remark   = _('You will be able to recreate these libraries later on if you wish so.')
            question = _('Remove all selected libraries?')

        if questionMsgBox(self.cfgWindow, question, '%s %s' % (_('Your media files will not be removed.'), remark)) == gtk.RESPONSE_YES:
            for row in list.getSelectedRows():
                # Remove the library from the disk
                libPath = os.path.join(ROOT_PATH, row[0])
                if isdir(libPath):
                    shutil.rmtree(libPath)
                # Remove the corresponding explorer
                modules.postMsg(consts.MSG_CMD_EXPLORER_REMOVE, {'modName': MOD_L10N, 'expName': row[0]})
                del self.libraries[row[0]]
            # Clean up the listview
            list.removeSelectedRows()


    def onCfgKeyboard(self, list, event):
        """ Remove the selection if possible """
        if gtk.gdk.keyval_name(event.keyval) == 'Delete':
            self.removeSelectedLibraries(list)


    def onCfgSelectionChanged(self, selection):
        """ The selection has changed, update the status of the buttons """
        self.cfgWindow.getWidget('btn-remove').set_sensitive(selection.count_selected_rows() != 0)
        self.cfgWindow.getWidget('btn-rename').set_sensitive(selection.count_selected_rows() == 1)
        self.cfgWindow.getWidget('btn-refresh').set_sensitive(selection.count_selected_rows() == 1)


    def onHelp(self, btn):
        """ Display a small help message box """
        helpDlg = help.HelpDlg(MOD_L10N)
        helpDlg.addSection(_('Description'),
                           _('This module organizes your media files by tags instead of using the file structure of your drive. '
                             'Loading tracks is also faster because their tags are already known and do not have to be read again.'))
        helpDlg.addSection(_('Usage'),
                           _('When you add a new library, you have to give the full path to the root directory of that library. '
                             'Then, all directories under this root path are recursively scanned for media files whose tags are read '
                             'and stored in a database.') + '\n\n' + _('Upon refreshing a library, the file structure under the root '
                             'directory and all media files are scanned for changes, to update the database accordingly.'))
        helpDlg.show(self.cfgWindow)
