/*
 * This file is licensed under the terms of the GNU General Public License,
 * version 2. See the file COPYING in the main directory for details.
 *
 *  Copyright (C) 2003  Thiemo Seufer <seufer@csv.ica.uni-stuttgart.de>
 *
 * The ECOFF handling is partially from linux/arch/mips/boot/elf2ecoff.c,
 *  Copyright (C) 1995 Ted Lemon
 *
 * t-rex is a small program which brings a tftp bootloader for
 * DECstations with REX firmware. It tacks an ELF kernel and an initial
 * ramdisk image on this bootloader binary. The resulting image can be
 * loaded and booted from the DECstations firmware.
 * 
 * The image has those sections (in this order):
 *
 *	ecoff:	A ecoff header as it it accepted by the firmware, which
 *		contains everything else in its .text section, all other
 *		sections are empty.
 *
 *	loader: The ELF loader. It relocates the kernel's ELF segments,
 *		presets its command line with the ramdisk parameters and
 *		jumps to the kernel entry.
 *
 *	kernel:	An ELF32 or ELF64 kernel.
 *
 *	ramdisk: A (gzipped) filesystem which needs to be understood by
 *		the bare kernel without modules.
 *
 * Memory map:
 *		   ---------------------
 *		   |One page of padding|
 *		   ---------------------
 *		   |   Ramdisk image   |
 *		^  |-------------------|
 *		|  |   Kernel image    |
 *		|  |-------------------|
 *		   |   Loader image    |
 *	0x80044000 ---------------------
 * 		   |                   |
 *		   ---------------------
 * 		   | relocated kernel  |
 * 	0x80004000 ---------------------
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <unistd.h>
#include <sys/stat.h>
#include <netinet/in.h>

#include <ecoff.h>
#include <elf.h>

#include "t-rex.h"

#define MIN(a, b) ((a) < (b)) ? (a) : (b)
#define MAX(a, b) ((a) > (b)) ? (a) : (b)

#define error(fmt, ...) do { fprintf(stderr, fmt "\n" , ##__VA_ARGS__); } while (0)

/*
 * We pad everything to 4 KiB pages, some boot ROMs need this. We also
 * align the ramdisk to a sector boundary (usually at 512 byte).
 */
#define LOADER_PAGE_SIZE 0x1000

#define LOADER_FILE "/usr/lib/delo/t-rex-loader"

static struct loader_params {
	struct stat st;
	struct loader_data ldata;
	off_t len;
	off_t padded_len;
	uint32_t load_address;
	uint32_t seg_offset;
	uint32_t entry;
} lparam;

static struct kernel_params {
	struct stat st;
	off_t len;
	off_t padded_len;
} kparam;

static struct ramdisk_params {
	struct stat st;
	off_t len;
	off_t padded_len;
} rparam;

static uint16_t u16_to_le(uint16_t value)
{
	union conv {
		uint16_t i;
		uint8_t b[2];
	} c, d;

	c.i = htons(value);
	d.b[0] = c.b[1];
	d.b[1] = c.b[0];

	return d.i;
}

static uint32_t u32_to_le(uint32_t value)
{
	union conv {
		uint32_t i;
		uint8_t b[4];
	} c, d;

	c.i = htonl(value);
	d.b[0] = c.b[3];
	d.b[1] = c.b[2];
	d.b[2] = c.b[1];
	d.b[3] = c.b[0];

	return d.i;
}

static uint16_t le_to_u16(uint16_t value)
{
	union conv {
		uint16_t i;
		uint8_t b[2];
	} c, d;

	c.i = value;
	d.b[0] = c.b[1];
	d.b[1] = c.b[0];

	return ntohs(d.i);
}

static uint32_t le_to_u32(uint32_t value)
{
	union conv {
		uint32_t i;
		uint8_t b[4];
	} c, d;

	c.i = value;
	d.b[0] = c.b[3];
	d.b[1] = c.b[2];
	d.b[2] = c.b[1];
	d.b[3] = c.b[0];

	return ntohl(d.i);
}

static uint64_t le_to_u64(uint64_t value)
{
	uint64_t ret;
	union conv {
		uint32_t i;
		uint8_t b[4];
	} c, d, e, f;

	c.i = (value >> 32) & 0xffffffff;
	d.b[0] = c.b[3];
	d.b[1] = c.b[2];
	d.b[2] = c.b[1];
	d.b[3] = c.b[0];

	e.i = value & 0xffffffff;
	f.b[0] = e.b[3];
	f.b[1] = e.b[2];
	f.b[2] = e.b[1];
	f.b[3] = e.b[0];

	ret = ntohl(d.i);
	ret <<= 32;
	ret |= ntohl(f.i);

	return ret;
}

static void swap_out_ecoff_filehdr(struct filehdr *f)
{
	f->f_magic = u16_to_le(f->f_magic);
	f->f_nscns = u16_to_le(f->f_nscns);
	f->f_timdat = u32_to_le(f->f_timdat);
	f->f_symptr = u32_to_le(f->f_symptr);
	f->f_nsyms = u32_to_le(f->f_nsyms);
	f->f_opthdr = u16_to_le(f->f_opthdr);
	f->f_flags = u16_to_le(f->f_flags);
}

static void swap_out_ecoff_aouthdr(struct aouthdr *a)
{
	a->magic = u16_to_le(a->magic);
	a->vstamp = u16_to_le(a->vstamp);
	a->tsize = u32_to_le(a->tsize);
	a->dsize = u32_to_le(a->dsize);
	a->bsize = u32_to_le(a->bsize);
	a->entry = u32_to_le(a->entry);
	a->text_start = u32_to_le(a->text_start);
	a->data_start = u32_to_le(a->data_start);
	a->bss_start = u32_to_le(a->bss_start);
	a->gprmask = u32_to_le(a->gprmask);
	a->cprmask[0] = u32_to_le(a->cprmask[0]);
	a->cprmask[1] = u32_to_le(a->cprmask[1]);
	a->cprmask[2] = u32_to_le(a->cprmask[2]);
	a->cprmask[3] = u32_to_le(a->cprmask[3]);
	a->gp_value = u32_to_le(a->gp_value);
}

static void swap_out_ecoff_esecs(struct scnhdr *s, int num)
{
	int i;

	for (i = 0; i < num; i++, s++) {
		s->s_paddr = u32_to_le(s->s_paddr);
		s->s_vaddr = u32_to_le(s->s_vaddr);
		s->s_size = u32_to_le(s->s_size);
		s->s_scnptr = u32_to_le(s->s_scnptr);
		s->s_relptr = u32_to_le(s->s_relptr);
		s->s_lnnoptr = u32_to_le(s->s_lnnoptr);
		s->s_nreloc = u16_to_le(s->s_nreloc);
		s->s_nlnno = u16_to_le(s->s_nlnno);
		s->s_flags = u32_to_le(s->s_flags);
	}
}

static void swap_in_elf32_ehdr(Elf32_Ehdr *ehdr)
{
	ehdr->e_type = le_to_u16(ehdr->e_type);
	ehdr->e_machine = le_to_u16(ehdr->e_machine);
	ehdr->e_version = le_to_u32(ehdr->e_version);
	ehdr->e_entry = le_to_u32(ehdr->e_entry);
	ehdr->e_phoff = le_to_u32(ehdr->e_phoff);
	ehdr->e_shoff = le_to_u32(ehdr->e_shoff);
	ehdr->e_flags = le_to_u32(ehdr->e_flags);
	ehdr->e_ehsize = le_to_u16(ehdr->e_ehsize);
	ehdr->e_phentsize = le_to_u16(ehdr->e_phentsize);
	ehdr->e_phnum = le_to_u16(ehdr->e_phnum);
	ehdr->e_shentsize = le_to_u16(ehdr->e_shentsize);
	ehdr->e_shnum = le_to_u16(ehdr->e_shnum);
	ehdr->e_shstrndx = le_to_u16(ehdr->e_shstrndx);
}

static void swap_in_elf64_ehdr(Elf64_Ehdr *ehdr)
{
	ehdr->e_type = le_to_u16(ehdr->e_type);
	ehdr->e_machine = le_to_u16(ehdr->e_machine);
	ehdr->e_version = le_to_u32(ehdr->e_version);
	ehdr->e_entry = le_to_u64(ehdr->e_entry);
	ehdr->e_phoff = le_to_u64(ehdr->e_phoff);
	ehdr->e_shoff = le_to_u64(ehdr->e_shoff);
	ehdr->e_flags = le_to_u32(ehdr->e_flags);
	ehdr->e_ehsize = le_to_u16(ehdr->e_ehsize);
	ehdr->e_phentsize = le_to_u16(ehdr->e_phentsize);
	ehdr->e_phnum = le_to_u16(ehdr->e_phnum);
	ehdr->e_shentsize = le_to_u16(ehdr->e_shentsize);
	ehdr->e_shnum = le_to_u16(ehdr->e_shnum);
	ehdr->e_shstrndx = le_to_u16(ehdr->e_shstrndx);
}

static void swap_in_elf32_phdr(Elf32_Phdr *phdr)
{
	phdr->p_type = le_to_u32(phdr->p_type);
	phdr->p_offset = le_to_u32(phdr->p_offset);
	phdr->p_vaddr = le_to_u32(phdr->p_vaddr);
	phdr->p_paddr = le_to_u32(phdr->p_paddr);
	phdr->p_filesz = le_to_u32(phdr->p_filesz);
	phdr->p_memsz = le_to_u32(phdr->p_memsz);
	phdr->p_flags = le_to_u32(phdr->p_flags);
	phdr->p_align = le_to_u32(phdr->p_align);
}

static void swap_in_elf64_phdr(Elf64_Phdr *phdr)
{
	phdr->p_type = le_to_u64(phdr->p_type);
	phdr->p_offset = le_to_u64(phdr->p_offset);
	phdr->p_vaddr = le_to_u64(phdr->p_vaddr);
	phdr->p_paddr = le_to_u64(phdr->p_paddr);
	phdr->p_filesz = le_to_u64(phdr->p_filesz);
	phdr->p_memsz = le_to_u64(phdr->p_memsz);
	phdr->p_flags = le_to_u64(phdr->p_flags);
	phdr->p_align = le_to_u64(phdr->p_align);
}

static int get_elf32_params(FILE *file, Elf32_Ehdr *ehdr,
			    uint32_t *load_address, uint32_t *seg_offset,
			    uint32_t *entry)
{
	int i;
	int found = 0;
	uint32_t soff = 0;
	uint32_t laddr = 0;

	if (!(ehdr->e_type == ET_EXEC
	      && ehdr->e_machine == EM_MIPS
	      && ehdr->e_version == EV_CURRENT))
		return 1;

	if (fseek(file, ehdr->e_phoff, SEEK_SET)) {
		error("failed to seek in ELF file");
		return 1;
	}

	for(i = 0; i < ehdr->e_phnum; i++) {
		Elf32_Phdr phdr;

		if (!(fread(&phdr, ehdr->e_phentsize, 1, file))) {
			error("failed to read ELF file");
			return 1;
		}

		swap_in_elf32_phdr(&phdr);

		/* Just use the address of the first loadable segment. */
		if (phdr.p_type == PT_LOAD) {
			soff = phdr.p_offset;
			laddr = phdr.p_vaddr;
			found = 1;
			break;
		}
	}
	if (!found)
		return 1;

	*seg_offset = soff;
	*load_address = laddr;
	*entry = ehdr->e_entry;
	return 0;
}

static int get_elf64_params(FILE *file, Elf64_Ehdr *ehdr,
			    uint32_t *load_address, uint32_t *seg_offset,
			    uint32_t *entry)
{
	int i;
	int found = 0;
	uint64_t soff = 0;
	uint64_t laddr = 0;

	if (!(ehdr->e_type == ET_EXEC
	      && ehdr->e_machine == EM_MIPS
	      && ehdr->e_version == EV_CURRENT))
		return 1;

	if (fseek(file, ehdr->e_phoff, SEEK_SET)) {
		error("failed to seek in ELF file");
		return 1;
	}

	for(i = 0; i < ehdr->e_phnum; i++) {
		Elf64_Phdr phdr;

		if (!(fread(&phdr, ehdr->e_phentsize, 1, file))) {
			error("failed to read ELF file");
			return 1;
		}

		swap_in_elf64_phdr(&phdr);

		/* Just use the address of the first loadable segment. */
		if (phdr.p_type == PT_LOAD) {
			soff = phdr.p_offset;
			laddr = phdr.p_vaddr;
			found = 1;
			break;
		}
	}
	if (!found)
		return 1;

	/*
	 * t-rex-loader can't load 64bit kernels outside the 32bit
	 * compatibility space.
	 */
	if ((laddr & ~0x7fffffffUL) != ~0x7fffffffUL) {
		error("kernel not in 32bit address space");
		return 1;
	}

	*seg_offset = (uint32_t)soff;
	*load_address = (uint32_t)laddr;
	*entry = (uint32_t)ehdr->e_entry;
	return 0;
}

static int get_elf_params(FILE *file, uint32_t *load_address,
			  uint32_t *seg_offset, uint32_t *entry)
{
	union {
		Elf32_Ehdr s;
		Elf64_Ehdr b;
	} ehdr;

	if (!(fread(&ehdr, sizeof(ehdr), 1, file))) {
		error("failed to read ELF file");
		return 1;
	}

	if (!(ehdr.s.e_ident[EI_MAG0] == ELFMAG0
	      && ehdr.s.e_ident[EI_MAG1] == ELFMAG1
	      && ehdr.s.e_ident[EI_MAG2] == ELFMAG2
	      && ehdr.s.e_ident[EI_MAG3] == ELFMAG3
	      && (ehdr.s.e_ident[EI_CLASS] == ELFCLASS32
		  || ehdr.s.e_ident[EI_CLASS] == ELFCLASS64)
	      && ehdr.s.e_ident[EI_DATA] == ELFDATA2LSB
	      && ehdr.s.e_ident[EI_VERSION] == EV_CURRENT)) {
		error("MIPS little endian ELF file expected");
		return 1;
	}

	if (ehdr.s.e_ident[EI_CLASS] == ELFCLASS32) {
		swap_in_elf32_ehdr(&ehdr.s);
		return get_elf32_params(file, &ehdr.s, load_address,
					seg_offset, entry);
	} else if (ehdr.s.e_ident[EI_CLASS] == ELFCLASS64) {
		swap_in_elf64_ehdr(&ehdr.b);
		return get_elf64_params(file, &ehdr.b, load_address,
					seg_offset, entry);
	} else
		return 1;
}

static int get_loader_params(FILE *li)
{
	if (fstat(fileno(li), &lparam.st)) {
		error("failed to stat loader file");
		return 1;
	}
	if (fseek(li, 0, SEEK_SET)) {
		error("failed to seek in loader file");
		return 1;
	}
	if (get_elf_params(li, &lparam.load_address, &lparam.seg_offset,
			   &lparam.entry)) {
		error("failed to find ELF data in loader file");
		return 1;
	}
	lparam.len = lparam.st.st_size;
	lparam.padded_len = lparam.len + ((~lparam.len + 1)
					  & (LOADER_PAGE_SIZE - 1));

	return 0;
}

static int get_kernel_params(FILE *ki)
{
	uint32_t load_address;
	uint32_t seg_offset;
	uint32_t entry;

	if (fstat(fileno(ki), &kparam.st)) {
		error("failed to stat kernel file");
		return 1;
	}
	if (fseek(ki, 0, SEEK_SET)) {
		error("failed to seek in kernel file");
		return 1;
	}
	/* Sanity check ELF file. */
	if (get_elf_params(ki, &load_address, &seg_offset, &entry)) {
		error("failed to find ELF data in kernel file");
		return 1;
	}
	kparam.len = kparam.st.st_size;
	kparam.padded_len = kparam.len + ((~kparam.len + 1)
					  & (LOADER_PAGE_SIZE - 1));

	return 0;
}

static int get_ramdisk_params(FILE *ri)
{
	if (fstat(fileno(ri), &rparam.st)) {
		error("failed to stat ramdisk file");
		return 1;
	}
	rparam.len = rparam.st.st_size;
	rparam.padded_len = rparam.len + ((~rparam.len + 1)
					  & (LOADER_PAGE_SIZE - 1));

	return 0;
}

static int create_ecoff(FILE *out)
{
	struct ecoff_file_start ef;
	off_t length;
	int sec_num = 6;

	length = lparam.padded_len;
	lparam.ldata.kernel_off = u32_to_le(length);
	length += kparam.padded_len;
	lparam.ldata.ramdisk_off = u32_to_le(length);
	lparam.ldata.ramdisk_len = u32_to_le(rparam.len);
	length += rparam.padded_len;

	memset(&ef, 0, sizeof(ef));

	ef.efh.f_magic = MIPSELMAGIC;
	ef.efh.f_nscns = sec_num;
	ef.efh.f_opthdr = sizeof(ef.eah);
	ef.efh.f_flags = 0x100f; /* Stripped, not sharable. */

	ef.eah.magic = OMAGIC;
	ef.eah.vstamp = 200;
	ef.eah.tsize = length;

	/*
	 * Fake one page of .bss, some dumb firmware wants to read
	 * entire pages.
	 */
	ef.eah.bsize = LOADER_PAGE_SIZE;

	ef.eah.entry = lparam.entry;
	ef.eah.text_start = lparam.load_address - lparam.seg_offset;
	ef.eah.data_start = ef.eah.text_start + ef.eah.tsize;
	ef.eah.bss_start = ef.eah.data_start + ef.eah.dsize;
	ef.eah.gprmask = 0xf3fffffe;

	strcpy(ef.esecs[0].s_name, ".text");
	ef.esecs[0].s_paddr = ef.eah.text_start;
	ef.esecs[0].s_vaddr = ef.eah.text_start;
	ef.esecs[0].s_size = ef.eah.tsize;
	ef.esecs[0].s_scnptr = N_TXTOFF(ef.efh, ef.eah);
	ef.esecs[0].s_flags = 0x20;

	strcpy(ef.esecs[1].s_name, ".data");
	ef.esecs[1].s_paddr = ef.eah.data_start;
	ef.esecs[1].s_vaddr = ef.eah.data_start;
	ef.esecs[1].s_size = ef.eah.dsize;
	ef.esecs[1].s_scnptr = N_DATOFF(ef.efh, ef.eah);
	ef.esecs[1].s_flags = 0x40;

	strcpy(ef.esecs[2].s_name, ".bss");
	ef.esecs[2].s_paddr = ef.eah.bss_start;
	ef.esecs[2].s_vaddr = ef.eah.bss_start;
	ef.esecs[2].s_size = ef.eah.bsize;
	ef.esecs[2].s_scnptr = N_BSSOFF(ef.efh, ef.eah);
	ef.esecs[2].s_flags = 0x82;

	strcpy(ef.esecs[3].s_name, ".rdata");
	ef.esecs[3].s_flags = 0x100;
	strcpy(ef.esecs[4].s_name, ".sdata");
	ef.esecs[4].s_flags = 0x200;
	strcpy(ef.esecs[5].s_name, ".sbss");
	ef.esecs[5].s_flags = 0x400;

	swap_out_ecoff_filehdr(&ef.efh);
	swap_out_ecoff_aouthdr(&ef.eah);
	swap_out_ecoff_esecs(ef.esecs, sec_num);

	if (fwrite(&ef, sizeof(ef), 1, out) != 1) {
		error("failed to write image file");
		return 1;
	}

	return 0;
}

static int copy_loader(FILE *li, FILE *out)
{
	uint8_t *buf;

	if (!(buf = (uint8_t *)calloc(1, lparam.padded_len))) {
		error("out of memory");
		return 1;
	}
	if (fseek(li, 0, SEEK_SET)) {
		error("failed to seek in loader file");
		free(buf);
		return 1;
	}
	if (!fread(buf, sizeof(uint8_t), lparam.len, li)) {
		error("failed to read loader file");
		free(buf);
		return 1;
	}

	/* FIXME: Gross hack! ldata has to reside at the start of the segment. */
	memcpy(buf + lparam.seg_offset, &lparam.ldata, sizeof(lparam.ldata));

	if ((ssize_t)fwrite(buf, sizeof(uint8_t), lparam.padded_len, out)
	    != lparam.padded_len) {
		error("failed to write image file");
		free(buf);
		return 1;
	}

	free(buf);
	return 0;
}

static int copy_kernel(FILE *ki, FILE *out)
{
	uint8_t *buf;
	off_t count = kparam.len;
	size_t pad;

	if (!(buf = (uint8_t *)malloc(MAX(kparam.st.st_blksize,
					  LOADER_PAGE_SIZE)))) {
		error("out of memory");
		return 1;
	}
	if (fseek(ki, 0, SEEK_SET)) {
		error("failed to seek in kernel file");
		free(buf);
		return 1;
	}

	while (count) {
		ssize_t sz;
		ssize_t msz;

		if (!(sz = fread(buf, sizeof(uint8_t),
				 kparam.st.st_blksize, ki))) {
			error("failed to read kernel file");
			free(buf);
			return 1;
		}
		msz = MIN(count, sz);
		if ((ssize_t)fwrite(buf, sizeof(uint8_t), msz, out) != msz) {
			error("failed to write image file");
			free(buf);
			return 1;
		}
		count -= msz;
	}

	pad = kparam.padded_len - kparam.len;
	memset(buf, 0, pad);
	if (fwrite(buf, sizeof(uint8_t), pad, out) != pad) {
		error("failed to write image file");
		free(buf);
		return 1;
	}

	free(buf);
	return 0;
}

static int copy_ramdisk(FILE *ri, FILE *out)
{
	uint8_t *buf;
	off_t count = rparam.len;
	size_t pad;

	if (!(buf = (uint8_t *)malloc(MAX(rparam.st.st_blksize,
					  LOADER_PAGE_SIZE)))) {
		error("out of memory");
		return 1;
	}
	if (fseek(ri, 0, SEEK_SET)) {
		error("failed to seek in ramdisk file");
		free(buf);
		return 1;
	}

	while (count) {
		ssize_t sz;
		ssize_t msz;

		if (!(sz = fread(buf, sizeof(uint8_t),
				 rparam.st.st_blksize, ri))) {
			error("failed to read ramdisk file");
			free(buf);
			return 1;
		}
		msz = MIN(count, sz);
		if ((ssize_t)fwrite(buf, sizeof(uint8_t), msz, out) != msz) {
			error("failed to write image file");
			free(buf);
			return 1;
		}
		count -= msz;
	}

	pad = rparam.padded_len - rparam.len;
	memset(buf, 0, pad);
	if (fwrite(buf, sizeof(uint8_t), pad, out) != pad) {
		error("failed to write image file");
		free(buf);
		return 1;
	}

	free(buf);
	return 0;
}

static int write_imagefile(FILE *li, FILE *ki, FILE *ri, FILE *out)
{
	uint8_t buf[LOADER_PAGE_SIZE];

	if (get_loader_params(li))
		return 1;
	if (get_kernel_params(ki))
		return 1;
	if (get_ramdisk_params(ri))
		return 1;

	if (create_ecoff(out))
		return 1;

	if (copy_loader(li, out))
		return 1;
	if (copy_kernel(ki, out))
		return 1;
	if (copy_ramdisk(ri, out))
		return 1;

	/*
	 * Fake one page of .bss, some dumb firmware wants to read
	 * entire pages.
	 */
	memset(buf, 0, LOADER_PAGE_SIZE);
	if (fwrite(buf, LOADER_PAGE_SIZE, 1, out) != 1) {
		error("failed to write image file");
		return 1;
	}

	return 0;
}

static void usage(void)
{
	error("Usage: t-rex -k kernel -r ramdisk [-l loader] [-o outfile]");
	exit(EXIT_FAILURE);
}

int main(int argc, char *argv[])
{
	int i;
	char *loader = LOADER_FILE;
	char *kernel = NULL;
	char *outfile = NULL;
	char *ramdisk = NULL;
	FILE *lifile = NULL;
	FILE *kifile = NULL;
	FILE *rifile = NULL;
	FILE *ofile = NULL;

	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			switch (argv[i][1]) {
				case 'k':
					if (i + 1 < argc)
						kernel = argv[++i];
					else
						usage();
					break;
				case 'l':
					if (i + 1 < argc)
						loader = argv[++i];
					else
						usage();
					break;
				case 'o':
					if (i + 1 < argc)
						outfile = argv[++i];
					else
						usage();
					break;
				case 'r':
					if (i + 1 < argc)
						ramdisk = argv[++i];
					else
						usage();
					break;
				default:
					usage();
					break;
			}
		}
	}

	if (!kernel || !ramdisk || !loader)
		usage();

	if (!(lifile = fopen(loader, "r"))) {
		error("Failed to open bootloader file %s", loader);
		return EXIT_FAILURE;
	}
	if (!(kifile = fopen(kernel, "r"))) {
		error("Failed to open kernel file %s", kernel);
		return EXIT_FAILURE;
	}
	if (!(rifile = fopen(ramdisk, "r"))) {
		error("Failed to open ramdisk file %s", ramdisk);
		return EXIT_FAILURE;
	}

	if (outfile)
		ofile = fopen(outfile, "w");
	else
		ofile = stdout;

	if (!ofile) {
		error("Failed to open output file %s", outfile);
		return EXIT_FAILURE;
	}

	if (write_imagefile(lifile, kifile, rifile, ofile)) {
		if (outfile)
			unlink(outfile);
		return EXIT_FAILURE;
	}

	return EXIT_SUCCESS;
}
