# Copyright 2010 Canonical Ltd.
#
# This file is part of desktopcouch.
#
#  desktopcouch is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License version 3
# as published by the Free Software Foundation.
#
# desktopcouch is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with desktopcouch.  If not, see <http://www.gnu.org/licenses/>.
#
# Authors: Chad Miller <chad.miller@canonical.com>

import desktopcouch.tests as test_environment
from desktopcouch.records import server, record
from desktopcouch.contacts import record as contact_record, view as contact_view

import testtools
import random
import string


frances_record_data = {
    "record_type": contact_record.CONTACT_RECORD_TYPE,
    "first_name": "Frances",
    "middle_name": None,
    "last_name": "Hicks",
    "title": None,
    "suffix": None,
    "birth_date": "1918-08-23",
    "nick_name": "Tiny",
    "spouse_name": "Louie",
    "wedding_date": "1970-02-27",
    "company": None,
    "department": None,
    "job_title": None,
    "manager_name": None,
    "assistant_name": None,
    "office": None,
    "addresses": (
       {
           "city": "Edwardburgh",
           "description": "summer",
           "street": "15 Libby Lane",
           "state": "MA",
           "country": "USA",
           "postalcode": "04412"
       },
       {
           "city": "Miami",
           "description": "winter",
           "street": "42 Snowbird Crawl",
           "state": "FL",
           "country": "USA",
           "postalcode": "33012"
       },
    ),
    "phone_numbers": (
       {
           "description": "home",
           "number": "313-876-3123"
       },
    ),
    "email_addresses": (
       {
           "description": "home",
           "address": "frances@example.com"
       },
    ),
    "urls": (
       {
           "description": "blog",
           "address": "http://frances.blag.example.com/"
       },
    ),
    "im_addresses": (
        {
            "address": "frances0",
            "description": "Frances",
            "protocol": "jabber"
        },
    )
}


def new_fake_record():
    new_str = lambda size: "".join(random.choice(string.lowercase) for n in range(size))
    return { "record_type": contact_record.CONTACT_RECORD_TYPE, "first_name": "random" + new_str(10), "middle_name": None, "last_name": new_str(10), "title": None, "suffix": None, "birth_date": "1970-02-13", "nick_name": new_str(10), "spouse_name": new_str(10), "wedding_date": "1970-02-27", "company": new_str(10), "department": new_str(10), "job_title": None, "manager_name": None, "assistant_name": None, "office": None, "addresses": ( { "city": new_str(10), "description": new_str(10), "street": new_str(10), "state": new_str(2), "country": "USA", "postalcode": "04412" }, { "city": new_str(10), "description": new_str(10), "street": new_str(10), "state": "FL", "country": "USA", "postalcode": "33012" },), "phone_numbers": ( { "description": new_str(6), "number": "313-876-3123" },), "email_addresses": ( { "description": "home", "address": new_str(5) + "@example.com" },), "urls": ( { "description": "blog", "address": "http://other.blag.example.com/" },), "im_addresses": ( { "address": new_str(10), "description": new_str(10), "protocol": "jabber" },) }


class TestLocalFiles(testtools.TestCase):

    def setUp(self):
        super(TestLocalFiles, self).setUp()
        self.dbname = "contacts"
        self.db = server.CouchDatabase(self.dbname, create=True, ctx=test_environment.test_context)
        self.db.put_record(contact_record.Contact(data=frances_record_data))
        self.db.put_records_batch([contact_record.Contact(data=new_fake_record()) for m in range(100)])

        
    def tearDown(self):
        super(TestLocalFiles, self).tearDown()
        del self.db._server[self.dbname]

    def test_find_contact_starting(self):
        contacts = list(contact_view.find_contacts_starting(self.db, first_name="F"))
        self.assertEqual(len(contacts), 1)
        self.assertEqual(contacts[0].key, "first_name:Frances")
        self.assertEqual(contacts[0].doc, None)

        contacts = list(contact_view.find_contacts_starting(self.db, include_docs=True, first_name="F"))
        self.assertNotEqual(contacts[0].doc, None)
        self.assertEqual(contacts[0].doc["nick_name"], "Tiny")

        self.assertEqual([], list(contact_view.find_contacts_starting(self.db, first_name="Chad")))

        self.assertEqual([], list(contact_view.find_contacts_starting(self.db, first_name="Frank")))

        contacts = list(contact_view.find_contacts_starting(self.db, first_name="Frances"))
        self.assertEqual(len(contacts), 1)

        contacts = list(contact_view.find_contacts_starting(self.db, birth_date="1918"))
        self.assertEqual(len(contacts), 1)

        contacts = list(contact_view.find_contacts_starting(self.db, birth_date="1918-08"))
        self.assertEqual(len(contacts), 1)

        contacts = list(contact_view.find_contacts_starting(self.db, first_name="random"))
        self.assertEqual(len(contacts), 100)

    def test_find_contacts_exact(self):
        self.assertEqual([], list(contact_view.find_contacts_exact(self.db, first_name="Frank")))
        self.assertEqual([], list(contact_view.find_contacts_exact(self.db, first_name="Fran")))

        contacts = list(contact_view.find_contacts_exact(self.db, first_name="Frances"))
        self.assertEqual(len(contacts), 1)

        contacts = list(contact_view.find_contacts_exact(self.db, birth_date="-08-23"))
        self.assertEqual(len(contacts), 1)
