#! /usr/bin/perl -w

# Perl version of Christoph Lameter's build program, renamed debuild.
# Written by Julian Gilbey, December 1998.

# Copyright 1999, Julian Gilbey
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


# We will do simple option processing.  The calling syntax of this
# program is:
#
#   debuild [debuild-options] binary|binary-arch|binary-indep|clean ...
# or
#   debuild [debuild-options] [dpkg-buildpackage-options] [-L lintian-options]
#
# In the first case, debuild will simply run debian/rules with the
# given parameter.  Available options in this case are:
#   --rootcmd=<gain-root-command>
#   -r<gain-root-command>  Which program should be used to gain root
#                          access.  If this is not specified and debuild
#                          is not being run by root or setuid root,
#                          fakeroot will be used.  
#
#   --preserve-envvar=<envvar>
#   -e<envvar>             The environment variable envvar will not
#                          be removed during the environment purge.
#                          If envvar is PATH, then the PATH variable
#                          will not be touched.
#
#   --preserve-env
#   -E                     The environment will not be touched: this is
#                          potentially quite harmful.
#
# In the second case, the behaviour is to run dpkg-buildpackage and
# then to run lintian on the resulting .changes file.  Lintian options
# may be specified after -L; all following options will be passed only
# to lintian.  The available debuild options are:
#   --no-lintian          Lintian will not be run, and all options after
#                         -L (if there is a -L option) will be ignored.
#                         If --no-lintian is not specified, but the
#                         lintian program is not installed, a warning
#                         will be issued.
#
#   -e<envvar>, -E        As above.

# As this may be running setuid, we make sure to clean out the
# environment before we perform the build, subject to any -e or -E
# options.  Also wise for building the packages, anyway.
# We don't put /usr/local/bin in the PATH as Debian
# programs will presumably be built without the use of any locally
# installed programs.  This could be changed, but in which case,
# please add /usr/local/bin at the END so that you don't get any
# unexpected behaviour.

# We will try to preserve the locale variables, but if it turns out that
# this harms the package building process, we will clean them out too.
# Please file a bug report if this is the case!

#use strict;
use Cwd;
use 5.003;
use lib '@pkgdatadir@';
use checkgettext;

($progname=$0) =~ s,.*/,,;

# Predeclare functions
sub fatal($);

setlocale(LC_MESSAGES(), "");
textdomain("devscripts");

sub usage
{
    print gettext(<<EOF);
First usage method:
  $progname [debuild-options] binary|binary-arch|binary-indep|clean ...
    to run debian/rules with given parameter(s).  Options here are
        --rootcmd=<gain-root-command>
	-r<gain-root-command>    Command used to become root if $progname
                                 not setuid root; default=fakeroot
        --preserve-envvar=<envvar>
        -e<envvar>               Preserve environment variable <envvar>
        --preserve-env
        -E                       Preserve all environment variables
                                 
Second usage method:
  $progname [debuild-options] [dpkg-buildpackage-options] [-L lintian-options]
    to run dpkg-buildpackage and then run lintian on the resulting
    .changes file.  Additional debuild option available in this case is:
        --no-lintian             Don't run lintian; ignore all options
                                 after -L
    For available dpkg-buildpackage and lintian options, see their
    respective manpages.

Also, --help or -h displays this message, --version shows version information.
EOF
# '
}

sub version
{
    printf gettext(<<'EOF'), $progname, '@VERSION@';
This is %s, from the Debian devscripts package, version %s
This code is copyright 1999 by Julian Gilbey, all rights reserved.
Based on a shell-script program by Christoph Lameter.
This program comes with ABSOLUTELY NO WARRANTY.
You are free to redistribute this code under the terms of the
GNU General Public License, version 2 or later.
EOF
}

# Check @ARGV for -e and -E options.  Not very efficient, but it will
# do for the time being.
my (@NEW_ARGV, $arg);
my $preserve_env=0;
my (%save_vars);
my $root_command;
my $run_lintian=1;
@save_vars{qw(TERM HOME LOGNAME PGPPASS PGPPATH FAKEROOTKEY LANG
	      DEB_BUILD_OPTIONS)} = (1) x 8;
{
    no locale;
    while ($arg=shift) {
	$arg =~ /^(-e|--preserve-envvar)$/ && ($_=shift) =~ /^\w+$/ and
	    $save_vars{$_}=1, next;
	$arg =~ /^-e(\w+)$/ and $save_vars{$1}=1, next;
	$arg =~ /^--preserve-envvar=(\w+)$/ and $save_vars{$1}=1, next;
	$arg =~ /^-e/ and
	    warn gettext("Ignoring -e option: ") . $arg .
		($arg eq '-e' ? " $_" : '');
	$arg =~ /^--preserve-envvar/ and
	    warn gettext("Ignoring --preserve-envvar option: ") . $arg .
		($arg eq '--preserve-envvar' ? " $_" : '');
	$arg =~ /^(-E|--preserve-env)$/ and $preserve_env=1, next;
	$arg eq '--no-lintian' and $run_lintian=0, next;
	$arg eq '--rootcmd' and $root_command=shift, next;
	$arg =~ /^-rootcmd=(.*)/ and $root_command=$1, next;
	$arg eq '-r' and $root_command=shift, next;
	$arg =~ /^-r(.*)/ and $root_command=$1, next;
	$arg =~ /^(-h|--help)$/ and usage(), exit 0;
	$arg eq '--version' and version(), exit 0;
	# Not a debuild option, so give up.
	push(@NEW_ARGV, $arg), last;
    }
}

unshift(@ARGV,@NEW_ARGV);
$root_command ||= 'fakeroot';

if ($save_vars{'PATH'}) {
    # Untaint PATH.  Very dangerous in general, but anyone running this
    # as root can do anything anyway.
    $ENV{'PATH'} =~ /^(.*)$/;
    $ENV{'PATH'} = $1;
} else {
    $ENV{'PATH'} = "/usr/sbin:/usr/bin:/sbin:/bin:/usr/bin/X11"
}
$save_vars{'PATH'}=1;
$ENV{'TERM'}='dumb' unless exists $ENV{'TERM'};

unless ($preserve_env) {
    foreach my $var (keys %ENV) {
	delete $ENV{$var} unless
	    $save_vars{$var} or $var =~ /^LC_[A-Z]+$/ ;
    }
}

umask 022;

until (-e "debian/rules")
{
    chdir ".." or fatal gettext("Can't chdir:") . " $!";
    fatal gettext("Cannot find debian/rules anywhere!  " .
	    "Are you in the source code tree?")
	if cwd() eq "/";
}

if ( ! -x _ ) {
    print STDERR gettext("Making debian/rules executable!\n");
    chmod 0755, "debian/rules" or
	fatal gettext("Couldn't make debian/rules executable:") . " $!";
}

# Pick up superuser privileges if we are running set[ug]id root
my $uid=$<;
if ( $< != 0 && $> == 0 ) { $< = $> }
my $gid=$(;
if ( $( != 0 && $) == 0 ) { $( = $) }

# Now let's look at our options, if any.  The first task is to decide
# which version of debuild we wish to run.  The rule is as follows: we
# want to run the first version (calling debian/rules explicitly) if
# there is at most one initial -r... argument, and all of the others
# are one of binary, binary-arch, binary-indep or clean.  We run the
# second version otherwise.  Note that the -r option is the only one
# stripped from the argument list.

my $command_version='rules';

if ( @ARGV == 0 ) { $command_version='dpkg'; }
else {
    foreach (@ARGV) {
	if ( ! /^(binary|binary-indep|binary-arch|clean)$/) {
	    $command_version='dpkg';
	    last;
	}
    }
}

if ( $command_version eq 'dpkg') {
    # We're going to run dpkg-buildpackage and possibly lintian.
    # Our first task is to parse the command line options.

    # And before we get too excited, does lintian even exist?
    system("command -v lintian >/dev/null 2>&1") == 0 or $run_lintian=0;

    my $Lopts=0;
    my $opt_a=0;
    my $arch='';
    my $gnutarget='';
    my $signchanges=1;
    my $signdsc=1;
    my @dpkg_args = qw(-us -uc);
    my @debsign_args = ();

    while ($_=shift) {
	/^-r(.*)/ and $root_command=$1, next;
	/^-a(.*)/ and $opt_a=1, $arch=$1;       # Explained below
	/^-t(.*)/ and $gnutarget=$1;            # Ditto
	/^-[mk]/ and push @debsign_args, $_;            # Key selection options
	/^-[sp](pgp|gpg)$/ and push @debsign_args, $_;  # Ditto
	$_ eq '-us' and $signdsc=0, next;
	$_ eq '-uc' and $signchanges=0, next;
	last if /^(--lintian|-L)$/;
	push @dpkg_args, $_;
    }
    if ($< != 0) {
	unshift @dpkg_args, "-r$root_command";
    }

    if ($signchanges==1 and $signdsc==0) {
	warn gettext("I will sign the .dsc file anyway as a signed .changes file was requested\n");
    }

    # We need to figure out what the changes file will be called,
    # so we copy some code from dpkg-buildpackage for this purpose.
    # Note that dpkg-buildpackage looks at any -a... and -t... parameters
    # it is given to determine the architecture, so we need to do the
    # same to determine the .changes filename.

    # The following is based on dpkg-buildpackage
    my ($pkg, $version, $sversion, $dsc, $changes);
    open DPKG, "dpkg-parsechangelog |" or
	fatal gettext("Cannot open dpkg-parsechangelog pipe:") ." $!";
    while (<DPKG>) {
	/^Source: / && chomp($pkg=$');
	/^Version: / && chomp($version=$');
    }
    close DPKG or fatal gettext("dpkg-parsechangelog pipe error:") . " $!";
    fatal gettext("Could not determine Source and/or Version from changelog")
	unless $pkg and $version;
    unless ($opt_a) {
	if (system("command -v dpkg-architecture >/dev/null 2>&1") >> 8 == 0) {
	    $arch=`dpkg-architecture -t$gnutarget -qDEB_HOST_ARCH 2>/dev/null`;
	    chomp($arch);
	}
	$arch ||= `dpkg --print-architecture 2>/dev/null`;
	chomp($arch);
	fatal gettext("Couldn't determine architecture!?") if ! $arch;
    }

    ($sversion=$version) =~ s/^\d+://;
    $dsc="${pkg}_$sversion.dsc";
    $changes="${pkg}_$sversion" . ($arch ? "_$arch" : "") . ".changes";

    # So now we can run dpkg-buildpackage and lintian...

    system('dpkg-buildpackage', @dpkg_args) == 0
	or fatal gettext("dpkg-buildpackage failed:") . " $!";
    chdir '..' or fatal gettext("Can't chdir:") . " $!";
    if ($run_lintian) {
	$<=$>=$uid;  # Give up on root privileges if we can
	$(=$)=$gid;
	print gettext("Now running lintian...\n");
	# The remaining items in @ARGV, if any, are lintian options
	system('lintian', @ARGV, $changes);
	print gettext("Finished running lintian.\n");
    }
    if ($signchanges) {
	print gettext("Now signing changes and dsc files...\n");
	exec 'debsign', @debsign_args, $changes
	    or fatal gettext("Couldn't exec debsign:") . " $!";
    }
    elsif ($signdsc) {
	print gettext("Now signing dsc file...\n");
	exec 'debsign', @debsign_args, $dsc
	    or fatal gettext("Couldn't exec debsign:") . " $!";
    }
    exit 0;
}
else {
    # Running debian/rules
    # Don't try to use the root command if we are already running as root
    if ( $< == 0 ) {
	exec 'debian/rules', @ARGV
	    or fatal gettext("Couldn't exec debian/rules:") . " $!";
    }

    # So we'll use the selected or default root command
    exec "$root_command debian/rules @ARGV"
	or fatal sprintf (gettext("Couldn't exec %s debian/rules:") . " $!",
		$root_command);
}

###### Subroutines

sub fatal($) {
    my ($pack,$file,$line);
    ($pack,$file,$line) = caller();
    (my $msg = sprintf(gettext("%s: fatal error at line %d:\n"),
		$progname, $line) . "@_\n") =~ tr/\0//d;
     $msg =~ s/\n\n$/\n/;
    die $msg;
}
