/*
 *  dmachinemon / a distributed machine monitor by dancer.
 *  Copyright (C) 2001 Junichi Uekawa
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
/* 
 * client handler, please include dmachinemon-master.h for using this.
 * $Id: dmachinemon-master-clienthandler.c,v 1.9 2001/11/20 11:00:51 dancer Exp $
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pthread.h>
#include "dmachinemon/libsocket.h"
#include "dmachinemon/dmachinemon-libdatabase.h"
#include "dmachinemon/dmachinemon-master.h"

static void
output_machines_list(FILE*f, dm_pairstring * p)
{
  if (p->next)
    output_machines_list(f, p->next);
  pthread_mutex_lock(&(p->lock));
  fprintf(f, "%s: %s\n", p->stringtitle, p->stringdata);  
  pthread_mutex_unlock(&(p->lock));
}

/* check if Seen-By is larger
TRUE on max_seen_by_depth is greater than Seen-By length */
static int
check_seenbydepth(int max_seen_by_depth, dm_machinelist *m)
{
  char * t = dm_get_value_text(m, SEEN_BY);
  int numcount = 0 ;
  int i = 0;
  if (t)
    {
      while (t[i])
	{
	  if (','==t[i])
	    numcount ++;
	  i++;
	}
      numcount ++;
      return (max_seen_by_depth > numcount );
    }
  else 
    return 1;			/* if seen-by is not available. */
}


/* 
 * checkseenflag = check the seen flag, if true.
 * max_seen_by_depth - specifies the depth. It seems to be rather strange. 3 machine-depth if 4 is specified. dunno.
 */
static void
output_machines(FILE*f, dm_machinelist* m, const int check_seenflag, int max_seen_by_depth)
{
  dm_machinelist* mnext;
  char * mmachinename;
  dm_pairstring * mdata;

  pthread_mutex_lock(&(m->lock));
  mnext = m->next;
  mmachinename = m->machinename;  
  mdata = m->data;
  pthread_mutex_unlock(&(m->lock));

  if (mnext)
    output_machines(f, mnext, check_seenflag, max_seen_by_depth);
  if (!mmachinename)
    fprintf (stderr, "DEBUG: assert: mmachinename is NULL -- should only happen temporarily.\n");

  pthread_mutex_lock(&(m->lock));
  if (mmachinename && 
      ((m->dataseenflag == 0) || (check_seenflag == 0)) /* if not seen */
      && (m->clientseencount < CLIENTSEENCOUNT_THRESHOLD) /* if not too old */
      ) 
    {
      if (max_seen_by_depth == 0 || 
	  (check_seenbydepth(max_seen_by_depth, m)) /* if seen-by depth is less than specified amount */
	  )
	{			/* output */
	  fprintf(f, ":%s:\n", m->machinename);
	  if (check_seenflag == 1)
	    m->dataseenflag=1;	/* this data is seen */
	  m->clientseencount ++; /* data is seen */
	  output_machines_list(f, mdata);
	}
    }
  pthread_mutex_unlock(&(m->lock));
}

int 
master_send_info_to_client ( FILE * f, dm_machinelist_information * machine, int check_seenflag, int max_seen_by_depth )
{
  dm_machinelist* machinedb_personal;
  pthread_mutex_lock(&machine->machinedb_lock);
  machinedb_personal = machine->machinedb;  
  pthread_mutex_unlock(&machine->machinedb_lock);
  if (!machinedb_personal)
    return 1;  

  output_machines(f, machinedb_personal, check_seenflag, max_seen_by_depth);
  fflush (f);  
  return 0;  
}

