<?xml version='1.0'?>
<!-- vim: sw=2 sta et
-->

<xsl:stylesheet 
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
  xmlns:lx-hash="http://docbook2x.sourceforge.net/xsl/hashtable"
  xmlns:j-hash="http://www.jclark.com/xt/java/java.util.Hashtable"
  xmlns:saxon="http://icl.com/saxon"
  xmlns:doc="http://nwalsh.com/xsl/documentation/1.0"
  exclude-result-prefixes="lx-hash j-hash saxon doc"
  extension-element-prefixes="lx-hash saxon"
  version='1.0'
  xml:lang="en">

<!-- ********************************************************************
     $Id: texinode-hash.xsl,v 1.2 2004/07/16 02:46:35 stevecheng Exp $
     ********************************************************************

     (C) 2000-2004 Steve Cheng <stevecheng@users.sourceforge.net>

     This file is part of the docbook2X XSLT stylesheets for
     converting DocBook to Texinfo.

     See ../../COPYING for the copyright status of this software.

     ******************************************************************** -->


<!-- ==================================================================== -->
<!-- Variables -->

<!-- Texinfo nodename -> ID map -->
<xsl:variable name="t-n-map" saxon:assignable="yes" />
<!-- ID -> Texinfo nodename map -->
<xsl:variable name="n-t-map" saxon:assignable="yes" />
<!-- These are only used for the Java implementation.
     Unfortunately we cannot select="j-hash:new()" to initialize
     these directly, since that would give an error under other 
     processors, so we have to resort to saxon:assign. :( -->

<xsl:variable name="dummy">
  <xsl:if test="function-available('j-hash:contains-key')">
    <saxon:assign name="t-n-map" select="j-hash:new()" />
    <saxon:assign name="n-t-map" select="j-hash:new()" />
  </xsl:if>
</xsl:variable>


<!-- ==================================================================== -->

<doc:template name="compute-texinfo-node-name" xmlns="">
<refpurpose>Compute (part of) the Texinfo nodename</refpurpose>
<refdescription>
<para>
This is a subroutine used by <function>get-texinfo-node-name</function>
to derive a nodename from the given node.  It checks the suggested
name for collisions with existing names.  If there is a collision, 
it prepends the parent's nodename to the suggested name.
</para>
<para>
If the suggested name is not given, it applies the texinfo-node-name
templates to find one for the given node,
</para>
<para>
This function returns the nodename <emphasis>with the filename and colon</emphasis>
prepended to it, simply to make the <function>get-texinfo-node-name</function> process 
more efficient.
</para>
<para>
This function is internal to the Java implementation because it needs to
test if a candidate name collides with an existing one.
</para>
</refdescription>
<refparameter>
<variablelist>
<varlistentry>
<term><parameter>sugname</parameter></term>
<listitem><para>
A string which is the suggested name.  If not given, regular templates
are applied.
</para></listitem>
</varlistentry>
</variablelist>
</refparameter>
</doc:template>

<xsl:template name="compute-texinfo-node-name">
  <xsl:param name="node" select="." />

  <xsl:param name="sugname">
    <xsl:apply-templates select="$node" mode="for-texinfo-node-name" />
  </xsl:param>

  <xsl:variable name="file">
    <xsl:call-template name="get-texinfo-file-name">
      <xsl:with-param name="node" select="$node" />
    </xsl:call-template>
  </xsl:variable>

  <xsl:variable name="qsugname"
                select="concat($file,':',$sugname)" />

  <xsl:choose>
    <xsl:when test="$sugname = ''">
      <xsl:call-template name="user-message">
        <xsl:with-param name="node" select="$node" />
        <xsl:with-param name="key">No readable node name; using generate-id</xsl:with-param>
      </xsl:call-template>

      <xsl:value-of select="concat($file,':',generate-id($node))" />
    </xsl:when>

    <xsl:when test="function-available('lx-hash:contains-key')
                    and not(lx-hash:contains-key('t-n', $qsugname))">
      <xsl:value-of select="$qsugname" />
    </xsl:when>

    <xsl:when test="function-available('j-hash:contains-key')
                    and not(j-hash:contains-key($t-n-map,$qsugname))">
      <xsl:value-of select="$qsugname" />
    </xsl:when>

    <xsl:when test="not( function-available('lx-hash:contains-key')
                         or function-available('j-hash:contains-key') )">
      <xsl:value-of select="$qsugname" />
    </xsl:when>

    <xsl:otherwise>
      <xsl:call-template name="user-message">
        <xsl:with-param name="node" select="$node" />
        <xsl:with-param name="key">Node name collision</xsl:with-param>
      </xsl:call-template>

      <xsl:variable name="parentnodename">
        <xsl:call-template name="get-texinfo-node-name">
          <xsl:with-param name="node" select="$node/.." />
        </xsl:call-template>
      </xsl:variable>

      <xsl:variable name="qpsugname"
                    select="concat($file,':',$parentnodename,' - ',$sugname)" />

      <xsl:choose>
        <xsl:when test="function-available('lx-hash:contains-key')
                        and lx-hash:contains-key('t-n',$qpsugname)">
          <xsl:call-template name="user-message">
            <xsl:with-param name="node" select="$node" />
            <xsl:with-param name="key">No readable node name; using generate-id</xsl:with-param>
          </xsl:call-template>
          
          <xsl:value-of select="concat($file,':',generate-id($node))" />
        </xsl:when>

        <xsl:when test="function-available('j-hash:contains-key')
                        and j-hash:contains-key($t-n-map,$qpsugname)">
          <xsl:call-template name="user-message">
            <xsl:with-param name="node" select="$node" />
            <xsl:with-param name="key">No readable node name; using generate-id</xsl:with-param>
          </xsl:call-template>

          <xsl:value-of select="concat($file,':',generate-id($node))" />
        </xsl:when>

        <xsl:otherwise>
          <xsl:value-of select="$qpsugname" />
        </xsl:otherwise>
      </xsl:choose>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>






<doc:template name="get-texinfo-node-name" xmlns="">
<refpurpose>Find the Texinfo nodename</refpurpose>
<refdescription>
<para>
Returns the Texinfo nodename from the given node.  This nodename
is guaranteed to be unique across the target Texinfo file.
</para>
<para>
This function needs to be reimplemented for each type of extension the
stylesheet uses.
</para>
</refdescription>
<refparameter>
<variablelist>
<varlistentry>
<term><parameter>node</parameter></term>
<listitem><para>
The node to find information for.  Default is the context node.
</para></listitem>
</varlistentry>
<varlistentry>
<term><parameter>sugname</parameter></term>
<listitem><para>
If the template needs to create a new name, try to use the
suggested name instead of the default names.
</para></listitem>
</varlistentry>
</variablelist>
</refparameter>
</doc:template>

<xsl:template name="get-texinfo-node-name">
  <xsl:param name="node" select="." />
  <xsl:param name="sugname" />

  <xsl:variable name="id" select="generate-id($node)" />

  <xsl:choose>
    <xsl:when test="function-available('lx-hash:contains-key')
                    and lx-hash:contains-key('n-t',$id)">
      <xsl:value-of select="substring-after(lx-hash:get('n-t',$id),':')" />
    </xsl:when>

    <xsl:when test="function-available('j-hash:contains-key')
                    and j-hash:contains-key($n-t-map,$id)">
      <xsl:value-of select="substring-after(j-hash:get($n-t-map,$id),':')" />
    </xsl:when>

    <xsl:otherwise>
      <xsl:variable name="newqname">
        <xsl:choose>
          <xsl:when test="$sugname">
            <xsl:call-template name="compute-texinfo-node-name">
              <xsl:with-param name="node" select="$node" />
              <xsl:with-param name="sugname" select="$sugname" />
            </xsl:call-template>
          </xsl:when>
          <xsl:otherwise>
            <xsl:call-template name="compute-texinfo-node-name">
              <xsl:with-param name="node" select="$node" />
            </xsl:call-template>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:variable>

      <xsl:choose>
        <xsl:when test="function-available('lx-hash:contains-key')">
          <lx-hash:put table='n-t' key="{$id}" value="{$newqname}" />
          <lx-hash:put table='t-n' key="{$newqname}" value="{$id}" />
        </xsl:when>

        <xsl:when test="function-available('j-hash:contains-key')">
          <!-- Evaluate side effects.
               Extension elements are clearly the best way to do this but
               there is unfortunately no simple standard to interface
               Java that way. -->
          <xsl:if test="concat(
            j-hash:put($n-t-map,string($id),string($newqname)),
            j-hash:put($t-n-map,string($newqname),string($id)))" />
        </xsl:when>
      </xsl:choose>

      <xsl:value-of select="substring-after($newqname,':')" />
    </xsl:otherwise>

  </xsl:choose>
</xsl:template>
  
</xsl:stylesheet>

