"""
Base classes for error estimators
"""

__author__ = "Marie E. Rognes (meg@simula.no)"
__copyright__ = "Copyright (C) 2009 - Marie E. Rognes"
__license__  = "GNU GPL version 3 or any later version"

# First added:  2009-11-04
# Last changed: 2010-05-09
#
# Modified by Anders Logg, 2010

from dolfin import interpolate, error, info, Function
from dolfin.cpp import Parameters

from dolfin.adaptivity.dual import Dual
from dolfin.adaptivity.formmanipulation import higher_order_problem, increase_order, glue, increase_bc_order

__all__ = ["ErrorEstimatorBase", "GoalEstimatorBase"]

class ErrorEstimatorBase:
    "Base class for defining error estimators."

    def __init__(self, F, parameters=None):
        """ Initialize """

        # Initialize from other error estimator
        if isinstance(F, ErrorEstimatorBase):
            self.F = F.F
            self.parameters = F.parameters
            return

        self.F = F
        self.parameters = parameters or self._default_parameters()

    def estimate_error(self, u_h):
        "To be implemented by sub-class."
        pass

    def _default_parameters(self):
        parameters = Parameters("error_estimation")
        return parameters

class GoalEstimatorBase(ErrorEstimatorBase):
    "Base class for goal-oriented error estimators."

    def __init__(self, F, bcs=None, goal=None, u=None, parameters=None,
                 goal_exterior_facets=None):

        # Initialize estimator from other estimator
        if isinstance(F, GoalEstimatorBase):
            ErrorEstimatorBase.__init__(self, F.F, F.parameters)
            self.goal = F.goal
            self.dual = F.dual
            self.dual_approximation = F.dual_approximation
            self.dual_difference = F.dual_difference
            self.error_indicators = F.error_indicators
            self.goal_exterior_facets = F.goal_exterior_facets
            return

        # Store input
        ErrorEstimatorBase.__init__(self, F, parameters)
        self.goal = goal
        self.goal_exterior_facets = goal_exterior_facets

        # Create dual problem
        self.dual = Dual(F, bcs, self.goal, u, self.goal_exterior_facets)

        # Variables that might be stored in the process
        self.dual_approximation = None
        self.dual_difference = None
        self.error_indicators = None

    def _default_parameters(self):
        """
        Possible options for 'dual_strategy':

        'higher_order',
        'extrapolation', 'extrapolation-boundary',
        'hierarchic'
        """

        parameters = Parameters("error_estimation")
        parameters.add("dual_strategy", "extrapolation")
        return parameters

    def construct_dual_approximation(self, V_h):
        """
        Based on the parameter 'dual_strategy', construct a suitable
        approximation to the exact dual solution z
        """

        # If improved dual approximation already computed, use that one
        if self.dual_approximation:
            return self.dual_approximation

        strategy = self.parameters["dual_strategy"]
        bcs = self.dual.bcs

        if strategy == "higher_order":
            pde = higher_order_problem(self.dual, V_h, self.goal_exterior_facets)
            z = pde.solve()
        elif strategy == "extrapolation":
            z_h = self.dual.solve()

            #W_h = glue(increase_order(z_h.function_space())) ?
            W_h = increase_order(z_h.function_space())
            z = Function(W_h)
            z.extrapolate(z_h)

            # Apply boundary conditions
            if bcs:
                higher_bcs = increase_bc_order(bcs, V_h)
                for bc in higher_bcs:
                    bc.apply(z.vector())

        else:
            error("Unknown dual strategy (%s)" % strategy)

        return z

    def construct_dual_difference(self, V_h=None):
        """
        Based on the parameter"dual_strategy", compute the stuff that
        goes into the residual equation. (Typically z - z_h where z is
        an approximation to the exact solution and z_h is something in
        the subspace
        """

        # If dual difference already computed, use that one
        if self.dual_difference:
            return self.dual_difference

        # Construct improved dual approximation
        z = self.construct_dual_approximation(V_h)

        # Subtract interpolant
        z_h = interpolate(z, V_h)

        return z - z_h

    def estimate_error(self, u_h):
        "To be implemented by sub-class."
        pass
