/* Drip - a transcoder for Unix
 * Copyright (C) 2001-2003 Jarl van Katwijk
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

// ----------------------------------------------------------------------------------
//
// logofilter
//
// 19-07-2002 - Jarl van Katwijk
//
// ----------------------------------------------------------------------------------
//
// This filter will overlay a bitmap image over the video in the upper-right corner
// for a defined amount of seconds
//
//
// ----------------------------------------------------------------------------------


#include "../config.h"
#include "../src/drip.h"
#include "../encoder/plugin-loader.hh"
#include <gmodule.h>
// Custom includes:
#include "../encoder/conversion.hh"
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <gnome.h>

#undef DEBUG

static gboolean init_done = FALSE;                 // Plugin initialised flag
static gint width,height;                          // Input Width\Height sizes
static gint xoff,yoff,xlen,ylen;                   // Offset and lenght values
static gdouble framerate;                          // Framerate (25.00, 29.97, etc)
static gdouble value1,value2,value3,value4,value5; // Plugin defined values - these are loaded\saved by Drip
static guint8 *src_planes[3];                      // Input data planes (I420 colour space)
static guint8 *dst_planes[3];                      // Output data planes
// Custom vari's
static GdkPixbuf* logo;
static gint logo_x,logo_y,logo_b,logo_cs;
static guint8 *logo_I420;
static gint logo_frames = 0;
static gboolean show = TRUE;
static GtkObject  *filter_threshold_adj;

extern "C" {




/* Return type (SPU,AUDIO,VIDEO) of filter */
G_MODULE_EXPORT module_type_e logofilter_type(void) {
    return VIDEO;
}


/* Return phase (PRE,NORMAL,POST) of filter */
G_MODULE_EXPORT module_phase_e logofilter_phase(void) {
    return NORMAL;
}





/* ---------------- */
/* FILTER FUNCTIONS */
/* ---------------- */

static void duration_changed(GtkWidget *w, gpointer data) {
    value1 = GTK_ADJUSTMENT(filter_threshold_adj)->value; // bext_level
}

/* Function which implements a GtkPlug to let user configure this plugin */
G_MODULE_EXPORT GtkWidget* logofilter_gui(void) {
    static GtkWidget *gui = NULL;
    static GtkWidget *label,*hscale;
    static GtkWidget *fileentry;
    static GtkWidget *combo_entry;
    static GtkWidget *hbox;
    GString *text = g_string_new("");

    #ifdef DEBUG
    g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Plugin: logofilter gui");
    #endif
    /* Build the gtk user interface for this plugins here. Also implement the callbacks */
    filter_threshold_adj = gtk_adjustment_new(value1, 0, 20 + 10, 2, 10, 10);

    gui = gtk_vbox_new (FALSE, 0);
    gtk_widget_ref (gui);
    gtk_widget_show (gui);

    g_string_sprintf(text,"\n\n\nLogofilter\n\nThis filter reads %s/.drip/plugins/drip_logofilter.jpg,\n and put it on the upper left during the set amount of seconds\n",getenv("HOME"));
    label = gtk_label_new(text->str);
    gtk_box_pack_start (GTK_BOX (gui), label, TRUE, TRUE, 0);
    gtk_widget_show(label);


    label = gtk_label_new("Duration logo is overlayed in seconds: ");
    gtk_misc_set_alignment(GTK_MISC(label), 1, 0.5);
    gtk_box_pack_start (GTK_BOX (gui), label, TRUE, TRUE, 0);
    gtk_widget_show(label);

    hscale = gtk_hscale_new(GTK_ADJUSTMENT(filter_threshold_adj));
    gtk_widget_set_usize(hscale, 400, 35);
    gtk_scale_set_digits(GTK_SCALE(hscale), 0);
    gtk_box_pack_start (GTK_BOX (gui), hscale, TRUE, TRUE, 0);
    gtk_signal_connect(GTK_OBJECT(hscale), "motion_notify_event", GTK_SIGNAL_FUNC(duration_changed), NULL);
    gtk_widget_show(hscale);

    fileentry = gnome_file_entry_new (NULL, NULL);
    gtk_widget_ref (fileentry);
    gtk_object_set_data_full (GTK_OBJECT (gui), "fileentry", fileentry,
                              (GtkDestroyNotify) gtk_widget_unref);
    gtk_box_pack_start (GTK_BOX (gui), fileentry, TRUE, TRUE, 0);

    combo_entry = gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (fileentry));
    gtk_widget_ref (combo_entry);
    gtk_object_set_data_full (GTK_OBJECT (gui), "combo_entry", combo_entry,
                              (GtkDestroyNotify) gtk_widget_unref);

    /* Clean & Exit */
    g_string_free(text,TRUE);
    return gui;
}


/* This function is called for every (audio\video) frame.
   Here the actual filtering is done. */
G_MODULE_EXPORT guint8** logofilter_apply(guint8** src, glong size, gulong SCR, gulong PTS) {
    static gint x,y;
    static glong y_xlen,y_logo_x;
    static glong xlen_ylen = xlen*ylen;
    static glong xlen_ylen54 = xlen*ylen*5/4;
    static glong y_xlen2,y_logo_x2;
    static glong logo_x_logo_y = logo_x*logo_y;
    static glong logo_x_logo_y54 = logo_x*logo_y*5/4;
    #ifdef DEBUG
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Plugin: logofilter apply (PTS=%lu,SCR=%lu)",PTS,SCR);
    #endif


    if ((show == TRUE) && (PTS > (90000*value1)) && (logo_frames > 10)) {  /* Just 1st 15 seconds */
        show = FALSE;
    }

    if (show == TRUE) {
        for (y=0;y<logo_y;y++) {
            y_xlen = y*xlen;
            y_logo_x = y*logo_x;
            for (x=0;x<logo_x;x++) {
                src[0][x + y_xlen] = logo_I420[x+y_logo_x]; /* LUM plane */
            }
        }
        for (y=0;y<logo_y/2;y++) {
            y_xlen2 = y*(xlen/2);
            y_logo_x2 = y*(logo_x/2);
            for (x=0;x<logo_x/2;x++) {
                src[1][x + y_xlen2] = logo_I420[x+y_logo_x2 + logo_x_logo_y]; /* CB plane */
                src[2][x + y_xlen2] = logo_I420[x+y_logo_x2 + logo_x_logo_y54]; /* CR plane */
            }
        }
        logo_frames++;
    }


    /* Clean & Exit */
    return src; 
}


G_MODULE_EXPORT void logofilter_init(gint w, gint h,
                                     gint xo, gint yo, gint xl, gint yl,
                                     gdouble fr) {
    GString *bitmap = g_string_new("");
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Plugin: logofilter init");
    /* Bitmap file */
    g_string_sprintf(bitmap,"%s/.drip/plugins/drip_logofilter.jpg",getenv("HOME"));
    /* Copy values */
    width = w; height = h;
    xoff = xo; yoff = yo; xlen = xl; ylen = yl;
    framerate = fr;
    /* Filter specific initialisation */
    logo = gdk_pixbuf_new_from_file(bitmap->str);
    if (logo != NULL) {
        logo_x = gdk_pixbuf_get_width(logo);
        logo_y = gdk_pixbuf_get_height(logo);
        logo_cs = gdk_pixbuf_get_colorspace(logo);
        logo_I420 = (guint8*)g_malloc(logo_x * logo_y * 3);
        RGB24toI420(logo_I420,gdk_pixbuf_get_pixels(logo),logo_x,logo_y);
        switch (logo_cs) {
            case GDK_COLORSPACE_RGB: logo_b = 24;
                                     break;
            default:                 logo_b = 0;
                                     break;
        }
        g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Plugin Logo: Loaded bitmap %s (x=%i,y=%i,bits=%i)",bitmap->str,logo_x,logo_y,logo_b);
    } else {
        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Plugin Logo: Cant load logo bitmap |%s|",bitmap->str);
    }
        
    /* Set init flag */
    init_done = TRUE;
    show = TRUE;
    logo_frames = 0;
    /* Clean & Exit */
    g_string_free(bitmap,TRUE);
    return;
}


/* Called when parameters need tobe reset to default, auto called when plugin used for 1st time */
G_MODULE_EXPORT void logofilter_default(void) { 
    g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Plugin: logofilter default");
    /* Set value's to default */
    value1 = 10;
    value2 = 0;
    value3 = 0;
    value4 = 0;
    value5 = 0;
    /* Clean & Exit */
    return;
}


/* Called after encoding has ended. Variable can be reset etc. */
G_MODULE_EXPORT void logofilter_destruct(void) {      
    g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Plugin: logofilter destruct");
    /* Clean & Exit */
    return;
}






/* -------------------------------- */
/* FIXED FUNCTIONS AFTER THIS POINT */
/* -------------------------------- */


/* Return values of plugin (value1,value2,... value5) */
G_MODULE_EXPORT void logofilter_values(gdouble* values) { 
    values[0] = value1;
    values[1] = value2;
    values[2] = value3;
    values[3] = value4;
    values[4] = value5;
    return;
}

/* Set plugins values */
G_MODULE_EXPORT void logofilter_values_function(gdouble *v) { 
    value1 = v[0];
    value2 = v[1];
    value3 = v[2];
    value4 = v[3];
    value5 = v[4];
    return;
}

/* g_module_check_init is automatically executed upon loading */
G_MODULE_EXPORT const gchar* g_module_check_init(GModule *module) {  
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Plugin: logofilter selftest passed");
    return NULL;
}

} //extern "C"


