/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Copyright 2004-2005 Todd Kulesza
 *
 * Authors:
 * 		Todd Kulesza <todd@dropline.net>
 */

#define _XOPEN_SOURCE /* glibc2 needs this */

#include <sys/types.h>
#include <config.h>

#include <time.h>
#include <string.h>

#include "drivel.h"
#include "drivel_request.h"
#include "journal.h"
#include "network.h"
#include "xmlrpc.h"
#include "blog_blogger.h"

/* To 'login' to Blogger, we send the username and password and get the user's
 * blogid */

DrivelRequest*
blog_blogger_build_login_request (const gchar *username, const gchar *password,
		const gchar *uri)
{
	DrivelRequest *dr;
	gchar *packet;
	
	g_return_val_if_fail (username, NULL);
	g_return_val_if_fail (password, NULL);
	
	debug ("blog_blogger_build_login_request()");
	
	packet = xmlrpc_build_packet ("blogger.getUsersBlogs", 
			XMLRPC_TYPE_STRING, DRIVEL_APPKEY,
			XMLRPC_TYPE_STRING, username,
			XMLRPC_TYPE_STRING, password,
			-1);
	dr = drivel_request_new_with_items (REQUEST_TYPE_LOGIN,
			REQUEST_PROTOCOL_XMLRPC, 
			BLOG_API_BLOGGER, 
			uri,
			g_strdup ("xml"), packet,
			NULL);

	return dr;
}

/* Parse the server's response to our getUsersBlogs request */

void
blog_blogger_parse_login_request (DrivelClient *dc, DrivelRequest *dr)
{
	const gchar *mesg;
	gboolean finished;
	gint count;
	
	g_return_if_fail (dc);
	g_return_if_fail (dr);

	debug ("blog_blogger_parse_login_request()");
	
	mesg = drivel_request_value_lookup (dr, "success");
	if (!mesg || !strcmp (mesg, "fault"))
	{
		mesg = drivel_request_value_lookup (dr, "faultstring0");
		display_error_dialog (dc, _("Server error"), mesg);
		return;
	}

	add_account_to_list (dc);

	/* build the list of journals */
	for (count = 0, finished = FALSE; !finished; count++)
	{
		gchar *key;
		DrivelJournal *dj;
		
		key = g_strdup_printf ("blogname%d", count);
		mesg = drivel_request_value_lookup (dr, key);
		g_free (key);
		if (mesg)
		{
			gchar *name;
			const gchar *id, *url;
			
			name = g_strdup (mesg);
			key = g_strdup_printf ("blogid%d", count);
			id = drivel_request_value_lookup (dr, key);
			g_free (key);
			key = g_strdup_printf ("url%d", count);
			url = drivel_request_value_lookup (dr, key);
			g_free (key);
			if (id && url)
			{
				DrivelRequest *dr;
				
				dj = drivel_journal_new ();
				dj->name = name;
				dj->uri_view = g_strdup (url);
				dj->id = g_strdup (id);
				dj->type = JOURNAL_TYPE_USER;
				dc->journal_list = g_slist_prepend (dc->journal_list, dj);
				
				/* get the recent entries for this journal */
				dr = blog_blogger_build_getevents_request (dc->user->username,
						dc->user->password, dc->user->server, dj->id, 
						DRIVEL_N_RECENT_POSTS);
				net_enqueue_request (dc, dr);
			}
			else
				g_warning ("blog_blogger_parse_login_request: could not read blogid");	
		}
		else
			finished = TRUE;
	}
	
	dc->journal_list = g_slist_sort (dc->journal_list, (GCompareFunc)sort_journals);
	dc->journals = count - 1;
	
	gtk_widget_hide (dc->login_window);
	journal_window_build (dc);
	
	return;
}

/* Build an XML-RPC packet with the journal entry content */

DrivelRequest*
blog_blogger_build_postevent_request (const gchar *username, 
		const gchar *password, const gchar *uri, const gchar *blogid, 
		gboolean publish, const gchar *content)
{
	DrivelRequest *dr;
	gchar *packet;
	
	debug ("blog_blogger_build_postevent_request()");
	
	packet = xmlrpc_build_packet ("blogger.newPost", 
			XMLRPC_TYPE_STRING, DRIVEL_APPKEY,
			XMLRPC_TYPE_STRING, blogid,
			XMLRPC_TYPE_STRING, username,
			XMLRPC_TYPE_STRING, password,
			XMLRPC_TYPE_STRING, content,
			XMLRPC_TYPE_BOOL, publish,
			-1);
	
	dr = drivel_request_new_with_items (REQUEST_TYPE_POSTEVENT,
			REQUEST_PROTOCOL_XMLRPC, 
			BLOG_API_BLOGGER, 
			uri,
			g_strdup ("xml"), packet,
			NULL);
	
	return dr;
}

/* Parse the server's response to our post request */

void
blog_blogger_parse_postevent_request (DrivelClient *dc, DrivelRequest *dr)
{
	const gchar *mesg;
	
	g_return_if_fail (dc);
	g_return_if_fail (dr);
	
	debug ("blog_blogger_parse_postevent_request()");
	
	mesg = drivel_request_value_lookup (dr, "success");
	if (!mesg || !strcmp (mesg, "fault"))
	{
		mesg = drivel_request_value_lookup (dr, "faultstring0");
		display_error_dialog (dc, _("Server error"), mesg);
		return;
	}
	
	net_ping_technorati (dc);
	journal_finished_post (dc);
	
	return;
}

DrivelRequest*
blog_blogger_build_editevent_request (const gchar *username,
		const gchar *password, const gchar *uri, const gchar *postid, 
		gboolean publish, const gchar *content)
{
	DrivelRequest *dr;
	gchar *packet;
	
	debug ("blog_blogger_build_editevent_request()");
	
	packet = xmlrpc_build_packet ("blogger.editPost", 
			XMLRPC_TYPE_STRING, DRIVEL_APPKEY,
			XMLRPC_TYPE_STRING, postid,
			XMLRPC_TYPE_STRING, username,
			XMLRPC_TYPE_STRING, password,
			XMLRPC_TYPE_STRING, content,
			XMLRPC_TYPE_BOOL, publish,
			-1);
	
	dr = drivel_request_new_with_items (REQUEST_TYPE_EDITEVENT,
			REQUEST_PROTOCOL_XMLRPC, 
			BLOG_API_BLOGGER, 
			uri,
			g_strdup ("xml"), packet,
			NULL);
	
	return dr;
}

DrivelRequest*
blog_blogger_build_deleteevent_request (const gchar *username,
		const gchar *password, const gchar *uri, const gchar *postid)
{
	DrivelRequest *dr;
	gchar *packet;
	
	debug ("blog_blogger_build_deleteevent_request()");
	
	packet = xmlrpc_build_packet ("blogger.deletePost", 
			XMLRPC_TYPE_STRING, DRIVEL_APPKEY,
			XMLRPC_TYPE_STRING, postid,
			XMLRPC_TYPE_STRING, username,
			XMLRPC_TYPE_STRING, password,
			XMLRPC_TYPE_BOOL, TRUE, 
			-1);
	
	dr = drivel_request_new_with_items (REQUEST_TYPE_DELETEEVENT,
			REQUEST_PROTOCOL_XMLRPC, 
			BLOG_API_BLOGGER, 
			uri,
			g_strdup ("xml"), packet,
			NULL);
	
	return dr;
}

/* Use blogger.getRecentPosts to revtrieve a history of recent entries */

DrivelRequest*
blog_blogger_build_getevents_request (const gchar *username,
		const gchar *password, const gchar *uri, const gchar *blogid, 
		gint n_posts)
{
	DrivelRequest *dr;
	gchar *packet;
	
	debug ("blog_blogger_build_getevents_request()");
	
	packet = xmlrpc_build_packet ("blogger.getRecentPosts", 
			XMLRPC_TYPE_STRING, DRIVEL_APPKEY,
			XMLRPC_TYPE_STRING, blogid,
			XMLRPC_TYPE_STRING, username,
			XMLRPC_TYPE_STRING, password,
			XMLRPC_TYPE_INT, n_posts,
			-1);
	
	dr = drivel_request_new_with_items (REQUEST_TYPE_GETEVENTS,
			REQUEST_PROTOCOL_XMLRPC, 
			BLOG_API_BLOGGER, 
			uri,
			g_strdup ("xml"), packet,
			g_strdup ("blogid"), blogid, /* needed when parsing the reply */
			NULL);
	
	if (n_posts == 1)
	{
		drivel_request_add_items (dr,
				g_strdup ("last_entry"), g_strdup ("true"),
				NULL);
	}
	
	return dr;
}

void
blog_blogger_parse_getevents_request (DrivelClient *dc, DrivelRequest *dr)
{
	const gchar *mesg;
	gint i;
	gchar *key;
	
	g_return_if_fail (dc);
	g_return_if_fail (dr);
	
	debug ("blog_blogger_parse_getevents_request()");
	
	mesg = drivel_request_value_lookup (dr, "success");
	if (!mesg || !strcmp (mesg, "fault"))
	{
		mesg = drivel_request_value_lookup (dr, "faultstring0");
		display_error_dialog (dc, _("Server error"), mesg);
		return;
	}
	
	if (drivel_request_item_lookup (dr, "last_entry"))
	{
		const gchar *postid, *content;
		
		postid = drivel_request_value_lookup (dr, "postid0");
		content = drivel_request_value_lookup (dr, "content0");
		
		if (postid && content)
		{
			journal_edit_entry (dc, postid, content, NULL, NULL, NULL, NULL,
					NULL, NULL, NULL, NULL, NULL, NULL);
		}
	}
	else
	{
		for (i = 0; i < DRIVEL_N_RECENT_POSTS; i++)
		{
			const gchar *postid, *content, *userid, *date;
			
			key = g_strdup_printf ("postid%d", i);
			postid = drivel_request_value_lookup (dr, key);
			g_free (key);
			key = g_strdup_printf ("content%d", i);
			content = drivel_request_value_lookup (dr, key);
			g_free (key);
			userid = drivel_request_item_lookup (dr, "blogid");
			key = g_strdup_printf ("datecreated%d", i);
			date = drivel_request_value_lookup (dr, key);
			g_free (key);
			
			if (postid && content && userid)
			{
				DrivelJournalEntry *entry = journal_entry_new ();
				
				entry->postid = g_strdup (postid);
				entry->content = g_strdup (content);
				entry->userid = g_strdup (userid);
				strptime (date, "%Y%m%dT%H:%M:%S", &entry->date_posted);
				g_array_append_val (dc->recent_entries, entry);
			}
		}
		
		journal_refresh_recent_entries (dc);
	}
	
	return;
}
