/* misc.c - 2000/06/28 */
/*
 *  EasyTAG - Tag editor for MP3 and OGG files
 *  Copyright (C) 2000  Jerome Couderc <j.couderc@ifrance.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <unistd.h>

#include "misc.h"
#include "easytag.h"
#include "msgbox.h"
#include "id3tag.h"
#include "browser.h"
#include "setting.h"
#include "bar.h"
#include "genres.h"
#include "mylocale.h"


/***************
 * Declaration *
 ***************/
// Playlist window
GtkWidget *WritePlaylistWindow = NULL;
GtkWidget *PlayListNameEntry;
GtkWidget *playlist_use_name;
GtkWidget *playlist_use_dir_name;
GtkWidget *playlist_full_path;
GtkWidget *playlist_relative_path;



/**************
 * Prototypes *
 **************/
gboolean Write_Playlist (gchar *play_list_name, gint use_relative_path);
void     Write_Playlist_Window_Key_Press (GtkWidget *window, GdkEvent *event);
void     Destroy_Write_Playlist_Window (void);
void     Playlist_Write_Button_Pressed (void);



/*************
 * Functions *
 *************/

/******************************
 * Functions managing pixmaps *
 ******************************/
/*
 * Buttons creation with pixmap
 */
#include "../pixmaps/apply.xpm"
#include "../pixmaps/cancel.xpm"
#include "../pixmaps/close.xpm"
#include "../pixmaps/execute.xpm"
#include "../pixmaps/ok.xpm"
#include "../pixmaps/no.xpm"
#include "../pixmaps/save.xpm"
#include "../pixmaps/yes.xpm"
GtkWidget *Create_Button_With_Pixmap (guint button_type)
{
    GtkWidget *Button;
    GtkWidget *HBox;
    GtkWidget *Label;
    GtkWidget *PixmapIcon;
    GdkPixmap *pixmap;
    GdkBitmap *mask;

    gtk_widget_realize(MainWindow);
    switch (button_type)
    {
        case BUTTON_OK:
            Label = gtk_label_new(_(" OK "));
            pixmap = gdk_pixmap_create_from_xpm_d(MainWindow->window,&mask,NULL,ok_xpm);
            break;

        case BUTTON_YES:
            Label = gtk_label_new(_(" Yes "));
            pixmap = gdk_pixmap_create_from_xpm_d(MainWindow->window,&mask,NULL,yes_xpm);
            break;

        case BUTTON_NO:
            Label = gtk_label_new(_(" No "));
            pixmap = gdk_pixmap_create_from_xpm_d(MainWindow->window,&mask,NULL,no_xpm);
            break;

        case BUTTON_APPLY:
            Label = gtk_label_new(_(" Apply "));
            pixmap = gdk_pixmap_create_from_xpm_d(MainWindow->window,&mask,NULL,apply_xpm);
            break;

        case BUTTON_SAVE:
            Label = gtk_label_new(_(" Save "));
            pixmap = gdk_pixmap_create_from_xpm_d(MainWindow->window,&mask,NULL,save_xpm);
            break;

        case BUTTON_CANCEL:
            Label = gtk_label_new(_(" Cancel "));
            pixmap = gdk_pixmap_create_from_xpm_d(MainWindow->window,&mask,NULL,cancel_xpm);
            break;

        case BUTTON_CLOSE:
            Label = gtk_label_new(_(" Close "));
            pixmap = gdk_pixmap_create_from_xpm_d(MainWindow->window,&mask,NULL,close_xpm);
            break;

        case BUTTON_WRITE:
            Label = gtk_label_new(_(" Write "));
            pixmap = gdk_pixmap_create_from_xpm_d(MainWindow->window,&mask,NULL,save_xpm);
            break;

        case BUTTON_EXECUTE:
            Label = gtk_label_new(_(" Execute "));
            pixmap = gdk_pixmap_create_from_xpm_d(MainWindow->window,&mask,NULL,execute_xpm);
            break;

        default:
            Button = gtk_button_new_with_label("Unknown button");
            return Button;
            break;
    }

    /* Create Icon */
    PixmapIcon = gtk_pixmap_new(pixmap,mask);
    gdk_pixmap_unref(pixmap);
    gdk_pixmap_unref(mask);

    Button = gtk_button_new();
    HBox = gtk_hbox_new(FALSE,0);
    gtk_container_add(GTK_CONTAINER(Button),HBox);
    /* Add items in button */
    gtk_container_add(GTK_CONTAINER(HBox),PixmapIcon);
    gtk_container_add(GTK_CONTAINER(HBox),Label);
    /* Alignment of items */
    gtk_misc_set_alignment(GTK_MISC(PixmapIcon),1,0.5);
    gtk_misc_set_alignment(GTK_MISC(Label),0,0.5);

    return Button;
}



/*
 * Return a widget with a pixmap
 * Note: for pixmap 'pixmap.xpm', pixmap_name is 'pixmap_xpm'
 */
GtkWidget *Create_Pixmap_Icon (gchar **pixmap_name)
{
    GtkWidget *PixmapIcon;
    GdkPixmap *pixmap;
    GdkBitmap *mask;

    if (!pixmap_name) return NULL;

    gtk_widget_realize(MainWindow);
    pixmap = gdk_pixmap_create_from_xpm_d(MainWindow->window,&mask,NULL,pixmap_name);
    PixmapIcon = gtk_pixmap_new(pixmap,mask);
    gdk_pixmap_unref(pixmap);
    gdk_pixmap_unref(mask);

    return PixmapIcon;
}


/*
 * Create an icon into an event box to allow some events (as to display tooltips).
 */
GtkWidget *Create_Pixmap_Icon_With_Event_Box (gchar **pixmap_name)
{
    GtkWidget *icon;
    GtkWidget *EventBox;

    EventBox = gtk_event_box_new();
    if (pixmap_name)
    {
        icon = Create_Pixmap_Icon(pixmap_name);
        gtk_container_add(GTK_CONTAINER(EventBox),icon);
    }

    return EventBox;
}



/*
 * Return a button with an icon and a label
 */
GtkWidget *Create_Button_With_Icon_And_Label (gchar **pixmap_name, gchar *label)
{
    GtkWidget *Button;
    GtkWidget *HBox;
    GtkWidget *Label;
    GtkWidget *PixmapIcon;
    GdkPixmap *pixmap;
    GdkBitmap *mask;


    Button = gtk_button_new();
    HBox = gtk_hbox_new(FALSE,0);
    gtk_container_add(GTK_CONTAINER(Button),HBox);

    /* Add a pixmap if not null */
    if (pixmap_name != NULL)
    {
        gtk_widget_realize(MainWindow);
        pixmap = gdk_pixmap_create_from_xpm_d(MainWindow->window,&mask,NULL,pixmap_name);
        PixmapIcon = gtk_pixmap_new(pixmap,mask);
        gdk_pixmap_unref(pixmap);
        gdk_pixmap_unref(mask);
        gtk_container_add(GTK_CONTAINER(HBox),PixmapIcon);
    }

    /* Add a label if not null */
    if (label != NULL)
    {
        Label = gtk_label_new(label);
        gtk_container_add(GTK_CONTAINER(HBox),Label);
    }    

    /* Display a warning message if the both parameters are NULL */
    if (pixmap_name==NULL && label==NULL)
        g_warning("Empty button created 'adr=%x' (no icon and no label)!",(gint)Button);

    return Button;
}




/*
 * Attach an history list to a combobox
 */
void Add_To_Combo_Box_History(GtkObject *combobox)
{
    GList *list, *tmplist;
    
    if (!GTK_IS_COMBO(combobox) || strlen(gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(combobox)->entry)))==0)
        return;

    tmplist = list = gtk_object_get_data(GTK_OBJECT(combobox),"History");
    list = Add_String_To_Glist(list,gtk_editable_get_chars(GTK_EDITABLE(GTK_COMBO(combobox)->entry),0,-1)); // PS: gtk_editable... allocate memory
    if (list)
        gtk_combo_set_popdown_strings(GTK_COMBO(combobox),list);
    else
        gtk_list_clear_items(GTK_LIST(GTK_COMBO(combobox)->list),0,-1);

    /* Attach data only if there was no data attached */
    if (!tmplist)
        gtk_object_set_data(GTK_OBJECT(combobox),"History",list);
}


/*
 * Add the 'string' passed in parameter to the 'list'.
 * If this string already exists into list, it doesn't add it.
 */
GList *Add_String_To_Glist (GList *list, gchar *filename)
{
    GList *tmp_list;
    gint HISTORY_MAX_LENGTH = 15;

    if (!filename || strlen(filename)<=0 )
        return list;

    /* We search and remove duplicated items (matching with filename) */    
    tmp_list = list = g_list_first(list);
    while (tmp_list)
    {
        if ( strcmp(tmp_list->data,filename)==0 )
        {
            list = g_list_remove(list,tmp_list->data);
            tmp_list = g_list_first(list);
            continue;
        }
        if (tmp_list->next == NULL) break;
        tmp_list = g_list_next(tmp_list);
    }

    /* We add the filename to the beginning of the list */
    list = g_list_prepend(list,filename);
    
    /* Cut end of list if length exceeds HISTORY_MAX_LENGTH items */
    if (g_list_length(list)>HISTORY_MAX_LENGTH)
    {
        tmp_list = g_list_nth(list,HISTORY_MAX_LENGTH);
        tmp_list->prev->next = NULL;
        tmp_list->prev = NULL;
        g_list_free(tmp_list);
    }

    return list;
}





/*
 * Sort the list of files by ascending filenames.
 */
void Sort_File_list_By_Ascending_Filename (void)
{
    if (!ETFileList) return;
    
    ET_Save_File_Data_From_UI((ET_File *)ETFileList->data);

    /* Go to the first item of the list */
    ET_File_List_First();
    ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Ascending_Filename);
    
    /* Reload files in browser list */
    Browser_List_Load_Files();

    ET_Display_File_Data_To_UI((ET_File *)ETFileList->data);

    Update_Command_Buttons_Sensivity();
}

/*
 * Sort the list of files by descending filenames.
 */
void Sort_File_list_By_Descending_Filename (void)
{
    if (!ETFileList) return;
    
    ET_Save_File_Data_From_UI((ET_File *)ETFileList->data);

    /* Go to the first item of the list */
    ET_File_List_First();
    ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Descending_Filename);
    
    /* Reload files in browser list */
    Browser_List_Load_Files();

    ET_Display_File_Data_To_UI((ET_File *)ETFileList->data);

    Update_Command_Buttons_Sensivity();
}

/*
 * Sort the list of files by ascending track number.
 */
void Sort_File_List_By_Ascending_Track_Number (void)
{
    if (!ETFileList) return;
    
    ET_Save_File_Data_From_UI((ET_File *)ETFileList->data);

    /* Go to the first item of the list */
    ET_File_List_First();
    ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Ascending_Track_Number);
    
    /* Reload files in browser list */
    Browser_List_Load_Files();

    ET_Display_File_Data_To_UI((ET_File *)ETFileList->data);

    Update_Command_Buttons_Sensivity();
}

/*
 * Sort the list of file by descending track number.
 */
void Sort_File_List_By_Descending_Track_Number (void)
{
    if (!ETFileList) return;
    
    ET_Save_File_Data_From_UI((ET_File *)ETFileList->data);

    /* Go to the first item of the list */
    ET_File_List_First();
    ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Descending_Track_Number);
    
    /* Reload files in browser list */
    Browser_List_Load_Files();

    ET_Display_File_Data_To_UI((ET_File *)ETFileList->data);

    Update_Command_Buttons_Sensivity();
}

/*
 * Sort the list of files by ascending creation date.
 */
void Sort_File_List_By_Ascending_Creation_Date (void)
{
    if (!ETFileList) return;
    
    ET_Save_File_Data_From_UI((ET_File *)ETFileList->data);

    /* Go to the first item of the list */
    ET_File_List_First();
    ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Ascending_Creation_Date);
    
    /* Reload files in browser list */
    Browser_List_Load_Files();

    ET_Display_File_Data_To_UI((ET_File *)ETFileList->data);

    Update_Command_Buttons_Sensivity();
}

/*
 * Sort the list of files by descending creation date.
 */
void Sort_File_List_By_Descending_Creation_Date (void)
{
    if (!ETFileList) return;
    
    ET_Save_File_Data_From_UI((ET_File *)ETFileList->data);

    /* Go to the first item of the list */
    ET_File_List_First();
    ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Descending_Creation_Date);
    
    /* Reload files in browser list */
    Browser_List_Load_Files();

    ET_Display_File_Data_To_UI((ET_File *)ETFileList->data);

    Update_Command_Buttons_Sensivity();
}





/*
 * Event attached to an entry to disable an other widget (for example: a button)
 * when the entry is empty
 */
void Entry_Changed_Disable_Object(GtkObject *widget_to_disable, GtkEditable *source_widget)
{
    gchar *text = NULL;
    
    if (!widget_to_disable || !source_widget) return;
    
    text = gtk_editable_get_chars(GTK_EDITABLE(source_widget),0,-1);
    if (!text || strlen(text)<1)
        gtk_widget_set_sensitive(GTK_WIDGET(widget_to_disable),FALSE);
    else
        gtk_widget_set_sensitive(GTK_WIDGET(widget_to_disable),TRUE);
        
    if (text) g_free(text);
}    



/*
 * Personnal function to create tooltips.
 */
GtkTooltips *gtk_tooltips_new_1 (void)
{
    GtkTooltips *tooltips;
    
    tooltips = gtk_tooltips_new();
    Set_Tips_Color(tooltips);
    
    return tooltips;
}

/*
 * Set the color of tooltips to yellow if the option have been activated
 */
void Set_Tips_Color (GtkTooltips *tips)
{
    GtkStyle *style;
    GdkColor YELLOW = {0, 0xffff, 0xffff, 0xcccc};

    if (USE_COLORED_TOOLTIPS)
    {
        /* Tips coloration */
        gtk_tooltips_force_window(tips);
        style = gtk_style_copy(gtk_widget_get_style(tips->tip_window));
        style->bg[GTK_STATE_NORMAL] = YELLOW;
        gtk_widget_set_style(tips->tip_window,style);
    }
}



/*
 * To insert only digits in an entry. If the text contains only digits: returns it,
 * else only first digits.
 */
#include <ctype.h>
void Insert_Only_Digit (GtkEditable *editable, const gchar *text, gint length,
            gint *position, gpointer data)
{
    gint i;
    gchar *result = g_new (gchar, length);

    for (i=0; i<length; i++)
        result[i] = isdigit(text[i])?text[i]:(gchar)NULL;
    gtk_signal_handler_block_by_func(GTK_OBJECT(editable),GTK_SIGNAL_FUNC(Insert_Only_Digit),data);
    gtk_editable_insert_text (editable, result, length, position);
    gtk_signal_handler_unblock_by_func(GTK_OBJECT(editable),GTK_SIGNAL_FUNC(Insert_Only_Digit),data);
    gtk_signal_emit_stop_by_name(GTK_OBJECT(editable),"insert_text");
    g_free (result);
}


/*
 * Help the user to select the genre: search and propose the genre which 
 * may correspond to the entered first characters.
 *
 * Function needs improvments...
 */
void Parse_Genre (GtkEntry *entry, GdkEvent *event, GtkCombo *combo)
{
    GdkEventKey *kevent;
    gint cursor_position;
    gchar *typed_string = NULL; // Characters entered by the user
    GtkList *list;
    GList *glist;
    GtkListItem *list_item;
    GtkWidget *label;
    gchar *label_string;
    gchar *string_to_add = NULL;


    // Exits if not a key pressed...
    if (!event || event->type != GDK_KEY_PRESS)
        return;

    kevent = (GdkEventKey *)event;
    switch(kevent->keyval)
    {
        case GDK_Left:
        case GDK_Right:
        case GDK_Delete:
        case GDK_BackSpace:
        case GDK_Shift_L:
        case GDK_Shift_R:
        case GDK_Control_L:
        case GDK_Control_R:
        case GDK_Alt_L:
        case GDK_Alt_R:
            return;
        case GDK_Tab:
            gtk_widget_grab_focus(GTK_WIDGET(GenreMButton)); // Give focus to the next widget
        case GDK_KP_Enter:
        case GDK_Return:
            gtk_editable_select_region(GTK_EDITABLE(entry),strlen(gtk_entry_get_text(entry)),-1);
            gtk_editable_set_position(GTK_EDITABLE(entry),strlen(gtk_entry_get_text(entry)));
            //gtk_widget_event(MainWindow,(GdkEvent *)event);
            return;
    }

    // Current position of the cursor
    cursor_position = gtk_editable_get_position(GTK_EDITABLE(entry));

    // Don't use autocompletion if cursor is not at the end.
    if ( (cursor_position < strlen(gtk_entry_get_text(entry))) )
        return;

    typed_string = gtk_editable_get_chars(GTK_EDITABLE(entry),0,cursor_position);

    // Get adress of the combo list
    list = GTK_LIST(combo->list);
    glist = list->children;
    // Search if a genre may correspong in the list
    while (glist)
    {
        list_item = (GtkListItem *)glist->data;
        label = GTK_BIN(list_item)->child;
        gtk_label_get(GTK_LABEL(label),&label_string);
        if (strncasecmp(label_string,typed_string,strlen(typed_string))==0)
        {
            string_to_add = ( strlen(label_string)>strlen(typed_string) ) ? (label_string+strlen(typed_string)) : NULL;
            if (string_to_add)
            {
                gint pos = cursor_position;
                
                gtk_entry_set_text(entry,label_string); // To select item on the list
                gtk_editable_delete_text(GTK_EDITABLE(entry),cursor_position,-1);
                gtk_editable_insert_text(GTK_EDITABLE(entry),string_to_add,strlen(string_to_add),&pos);
                gtk_editable_set_position(GTK_EDITABLE(entry),cursor_position);
                gtk_editable_select_region(GTK_EDITABLE(entry),cursor_position,-1);
            }
            break;    
        }
        glist = g_list_next(glist);
    }

    if (typed_string) g_free(typed_string);
}




/*
 * Parse and auto complete date entry if you don't type the 4 digits.
 */
#include <time.h>
#include <stdlib.h>
#include <math.h>
void Parse_Date (void)
{
    gchar *year;
    gchar *tmp, *tmp1;
    gchar current_year[5];
    time_t t;
    struct tm t0;

    if (!DATE_AUTO_COMPLETION) return;

    /* Get the info entered by user */
    year = gtk_entry_get_text(GTK_ENTRY(YearEntry));

    if ( strcmp(year,"")!=0 && strlen(year)<4 )
    {
        t = time(NULL);
        /* Get the current date */
        memcpy(&t0, localtime(&t), sizeof(struct tm));
        /* Put the current year in 'current_year' tab */
        sprintf(current_year,"%d",1900+t0.tm_year);

        tmp = &current_year[4-strlen(year)];
        if ( atoi(year) <= atoi(tmp) )
        {
            sprintf(current_year,"%d",atoi(current_year)-atoi(tmp));
            tmp1 = g_strdup_printf("%d",atoi(current_year)+atoi(year));
            gtk_entry_set_text(GTK_ENTRY(YearEntry),tmp1);
            g_free(tmp1);

        }else
        {
            sprintf(current_year,"%d",atoi(current_year)-atoi(tmp)-(gint)pow(10,strlen(year)));
            tmp1 = g_strdup_printf("%d",atoi(current_year)+atoi(year));
            gtk_entry_set_text(GTK_ENTRY(YearEntry),tmp1);
            g_free(tmp1);
        }
    }
}




/*
 * Load the genres list to the combo, and sorts it
 */
int Compare_Two_Genres (gchar *genre1,gchar *genre2)
{
    return strcmp(genre1,genre2);
}
void Load_Genres_List_To_UI(void)
{
    GList *genrelist = NULL;
    gint i;

    if (!GenreEntry) return;

    /* Create GList */
    if (!genrelist)
        {
        /* Add these elements for: no genre / unknown */
        genrelist = g_list_append(genrelist,"");
        genrelist = g_list_append(genrelist,"Unknown");
        /* Load others genres */
        for (i=0; i<GENRE_MAX; i++)
            genrelist = g_list_insert_sorted(genrelist,id3_genres[i],
                                             (GCompareFunc)Compare_Two_Genres);
        }
    /* Connect the combo and the glist */
    gtk_combo_set_popdown_strings(GTK_COMBO(GenreEntry),genrelist);
}




/*
 * Load the track numbers (minimum 30) into the track combo list
 */
void Load_Track_List_To_UI (void)
{
    GList *tracklist = NULL;
    gint len;
    gint i;

    if (!ETFileList || !TrackEntry) return;

    if ((len=ETFileList_Length) < 30)
        len = 30;
    
    /* Create GList */
    if (!tracklist)
    {
        tracklist = g_list_append(tracklist," ");
        for (i=1; i<=len; i++)
            tracklist = g_list_append(tracklist,g_strdup_printf("%.2d",i));
    }

    /* Connect the combo and the glist */
    gtk_combo_set_popdown_strings(GTK_COMBO(TrackEntry),tracklist);
}





/*
 * Change mouse cursor
 */
void Init_Mouse_Cursor (void)
{
    MouseCursor = NULL;
}
void Destroy_Mouse_Cursor (void)
{
    if (MouseCursor)
    {
            gdk_cursor_destroy(MouseCursor);
        MouseCursor = NULL;
    }
}
void Set_Busy_Cursor (void)
{
    /* If still built, destroy it to avoid memory leak */
    Destroy_Mouse_Cursor();
    /* Create the new cursor */
    MouseCursor = gdk_cursor_new (GDK_WATCH);
    gdk_window_set_cursor(MainWindow->window,MouseCursor);
}
void Set_Unbusy_Cursor (void)
{
    /* Back to standard cursor */
    gdk_window_set_cursor(MainWindow->window,NULL);
    Destroy_Mouse_Cursor();
}




/*
 * Write a playlist
 */
gboolean Write_Playlist (gchar *play_list_name, gint use_relative_path)
{
    FILE  *file;
    ET_File *etfile;
    GList *etfilelist;
    gchar *filename;
    gchar *basedir;
    gint   duration;

    if ((file = fopen(play_list_name,"wb")) != NULL)
    {
        /* 'base directory' where is located the playlist. Used also to write file with a
         * relative path for file located in this directory and sub-directories
         */
        basedir = g_dirname(play_list_name);

        // First line of the file
        fprintf(file,"#EXTM3U\n");
        etfilelist = ET_File_List_First();
        while (etfilelist)
        {
            etfile   = (ET_File *)etfilelist->data;
            filename = ((File_Name *)etfile->FileNameCur->data)->value;
            duration = ((ET_File_Info *)etfile->ETFileInfo)->duration;
            if (use_relative_path)
            {
                // Keep only files in this directory and sub-dirs
                if ( strncmp(filename,basedir,strlen(basedir))==0 )
                {
                    fprintf(file,"#EXTINF:%d,%s\n",duration,g_basename(filename));
                    fprintf(file,"%s\n",filename+strlen(basedir)+1);
                }
            }else
            {
                fprintf(file,"#EXTINF:%d,%s\n",duration,g_basename(filename));
                fprintf(file,"%s\n",filename);
            }
            etfilelist = etfilelist->next;
        }
        g_free(basedir);
        fclose(file);

        return TRUE;
    }else
    {
        g_print(_("ERROR while opening file: '%s' (%s)\n\a"),play_list_name,g_strerror(errno));
        return FALSE;
    }
}



/*
 * The window to write playlists.
 */
void Open_Write_Playlist_Window (void)
{
    GtkWidget *Frame;
    GtkWidget *VBox;
    GtkWidget *vbox, *hbox;
    GtkWidget *ButtonBox;
    GtkWidget *Button;
    GtkWidget *Separator;
    GList *History_List;
    gint mw_x, mw_y;


    if (WritePlaylistWindow != NULL) 
    {
        gdk_window_raise(WritePlaylistWindow->window);
        return;
    }
    
    WritePlaylistWindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(WritePlaylistWindow),_("Generate a playlist"));
    gtk_window_set_transient_for(GTK_WINDOW(WritePlaylistWindow),GTK_WINDOW(MainWindow));
    gtk_window_set_policy(GTK_WINDOW(WritePlaylistWindow),FALSE,TRUE,TRUE);
    gtk_signal_connect(GTK_OBJECT(WritePlaylistWindow),"destroy",(GtkSignalFunc)Destroy_Write_Playlist_Window,NULL);
    gtk_signal_connect(GTK_OBJECT(WritePlaylistWindow),"delete_event",(GtkSignalFunc)Destroy_Write_Playlist_Window,NULL);
    gtk_signal_connect(GTK_OBJECT(WritePlaylistWindow),"key_press_event",(GtkSignalFunc)Write_Playlist_Window_Key_Press,NULL);

    Frame = gtk_frame_new(NULL);
    gtk_container_add(GTK_CONTAINER(WritePlaylistWindow),Frame);
    gtk_container_set_border_width(GTK_CONTAINER(Frame),2);

    VBox = gtk_vbox_new(FALSE,2);
    gtk_container_add(GTK_CONTAINER(Frame),VBox);
    gtk_container_border_width(GTK_CONTAINER(VBox),4);

    /* Playlist name */
    Frame = gtk_frame_new(_("M3U Playlist Name"));
    gtk_box_pack_start(GTK_BOX(VBox),Frame,TRUE,TRUE,0);
    vbox = gtk_vbox_new(FALSE,0);
    gtk_container_add(GTK_CONTAINER(Frame),vbox);
    gtk_container_border_width(GTK_CONTAINER(vbox),4);
    hbox = gtk_hbox_new(FALSE,0);
    gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,0);

    playlist_use_name = gtk_radio_button_new_with_label(NULL,":");
    gtk_box_pack_start(GTK_BOX(hbox),playlist_use_name,FALSE,FALSE,0);
    PlayListNameEntry = gtk_combo_new();
    gtk_box_pack_start(GTK_BOX(hbox),PlayListNameEntry,TRUE,TRUE,0);
    playlist_use_dir_name = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(playlist_use_name),
        _("Use directory name"));
    gtk_box_pack_start(GTK_BOX(vbox),playlist_use_dir_name,FALSE,FALSE,0);
    // History list
    History_List = Load_Play_List_Name_List();
    History_List = Add_String_To_Glist(History_List,PLAYLIST_NAME);
    if (History_List)
        gtk_combo_set_popdown_strings(GTK_COMBO(PlayListNameEntry),History_List);
    gtk_object_set_data(GTK_OBJECT(PlayListNameEntry),"History",History_List);
    gtk_signal_connect_object(GTK_OBJECT(GTK_ENTRY(GTK_COMBO(PlayListNameEntry)->entry)),"activate",
        Add_To_Combo_Box_History,GTK_OBJECT(PlayListNameEntry));
    gtk_signal_connect_object(GTK_OBJECT(GTK_ENTRY(GTK_COMBO(PlayListNameEntry)->entry)),"focus_out_event",
        Add_To_Combo_Box_History,GTK_OBJECT(PlayListNameEntry));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(playlist_use_name),PLAYLIST_USE_NAME);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(playlist_use_dir_name),PLAYLIST_USE_DIR_NAME);

    /* Playlist options */
    Frame = gtk_frame_new(_("Playlist Options"));
    gtk_box_pack_start(GTK_BOX(VBox),Frame,TRUE,TRUE,0);
    vbox = gtk_vbox_new(FALSE,0);
    gtk_container_add(GTK_CONTAINER(Frame),vbox);
    gtk_container_border_width(GTK_CONTAINER(vbox),4);
    playlist_full_path = gtk_radio_button_new_with_label(NULL,_("Use full path for files in playlist"));
    gtk_box_pack_start(GTK_BOX(vbox),playlist_full_path,FALSE,FALSE,0);
    playlist_relative_path = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(playlist_full_path),
        _("Use relative path for files in playlist"));
    gtk_box_pack_start(GTK_BOX(vbox),playlist_relative_path,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(playlist_full_path),PLAYLIST_FULL_PATH);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(playlist_relative_path),PLAYLIST_RELATIVE_PATH);


    /* Separator line */
    Separator = gtk_hseparator_new();
    gtk_box_pack_start(GTK_BOX(VBox),Separator,FALSE,FALSE,0);

    ButtonBox = gtk_hbutton_box_new ();
    gtk_box_pack_start(GTK_BOX(VBox),ButtonBox,FALSE,FALSE,0);
    gtk_button_box_set_layout(GTK_BUTTON_BOX(ButtonBox),GTK_BUTTONBOX_END);
    gtk_button_box_set_spacing(GTK_BUTTON_BOX(ButtonBox),10);

    /* Button to write the playlist */
    Button = Create_Button_With_Pixmap(BUTTON_WRITE);
    gtk_container_add(GTK_CONTAINER(ButtonBox),Button);
    GTK_WIDGET_SET_FLAGS(Button,GTK_CAN_DEFAULT);
    gtk_signal_connect_object(GTK_OBJECT(Button),"clicked",(GtkSignalFunc)Playlist_Write_Button_Pressed,NULL);
    
    /* Button to cancel */
    Button = Create_Button_With_Pixmap(BUTTON_CLOSE);
    gtk_container_add(GTK_CONTAINER(ButtonBox),Button);
    GTK_WIDGET_SET_FLAGS(Button, GTK_CAN_DEFAULT);
    gtk_widget_grab_default(Button);
    gtk_signal_connect(GTK_OBJECT(Button),"clicked",(GtkSignalFunc)Destroy_Write_Playlist_Window,NULL);

    gtk_widget_show_all(WritePlaylistWindow);

    gdk_window_get_position(MainWindow->window,&mw_x,&mw_y);
    gtk_window_reposition(GTK_WINDOW(WritePlaylistWindow),
        mw_x + (MainWindow->allocation.width/2)  - (WritePlaylistWindow->allocation.width)/2,
        mw_y + (MainWindow->allocation.height/2) - (WritePlaylistWindow->allocation.height)/2);
}
void Destroy_Write_Playlist_Window (void)
{
    if (WritePlaylistWindow)
    {
        GList *list;
        
        /* Save combobox history lists before exit */
        list = gtk_object_get_data(GTK_OBJECT(PlayListNameEntry),"History");
        Save_Play_List_Name_List(list);
 
        gtk_widget_destroy(WritePlaylistWindow);
        WritePlaylistWindow = (GtkWidget *)NULL;
    }
}
void Write_Playlist_Window_Key_Press (GtkWidget *window, GdkEvent *event)
{
    GdkEventKey *kevent;

    if (event && event->type == GDK_KEY_PRESS)
    {
        kevent = (GdkEventKey *)event;
        switch(kevent->keyval)
        {
            case GDK_Escape:
                Destroy_Write_Playlist_Window();
                break;
        }
    }
}
void Playlist_Write_Button_Pressed (void)
{
    gchar *playlist_name = NULL;
    gchar *playlist_path;
    gint   use_relative_path;
    FILE *file;
    gchar *msg;
    GtkWidget *msgbox;
    gint msgbox_button=0;


    // Check if playlist name was filled
    if ( gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(playlist_use_name)) 
     && strlen(gtk_entry_get_text_1(PlayListNameEntry))<=0 )
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(playlist_use_dir_name),TRUE);

    // End of path contains '/'
    playlist_path = Browser_Get_Current_Path();
    
    // Buid the playlist name
    if ( gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(playlist_use_name)) )
    {
        playlist_name = g_strconcat(playlist_path,gtk_entry_get_text_1(PlayListNameEntry),".m3u",NULL);
    }else
    {
        gchar *extract_dir_name;
        
        if ( strcmp(playlist_path,"/")==0 )
        {
            extract_dir_name = g_strdup("playlist");
        }else
        {
            gchar *tmp_string = g_strdup(playlist_path);
            if (tmp_string[strlen(tmp_string)-1]=='/')
                tmp_string[strlen(tmp_string)-1] = '\0';
            extract_dir_name = g_strdup(g_basename(tmp_string));
            g_free(tmp_string);
        }
        playlist_name = g_strconcat(playlist_path,extract_dir_name,".m3u",NULL);
        g_free(extract_dir_name);
    }

    // Use recursive path?
    use_relative_path = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(playlist_relative_path));

    // Check if file exists
    if (CONFIRM_WRITE_PLAYLIST)
    {
        if ( (file=fopen(playlist_name,"r")) != NULL )
        {
            fclose(file);
            msg = g_strdup_printf(_("Playlist file '%s' already exists!\nOverwrite?"),playlist_name); 
            msgbox = msg_box_new (_("Write Playlist..."),msg,MSG_QUESTION,BUTTON_YES,BUTTON_NO,0);
            msg_box_hide_check_button(MSG_BOX(msgbox));
            msgbox_button = msg_box_run(MSG_BOX(msgbox));
            gtk_widget_destroy(msgbox);
            g_free(msg);
        }
    }

    // Write playlist if ok
    if (msgbox_button==0 || msgbox_button==BUTTON_YES )
    {
        if ( Write_Playlist(playlist_name,use_relative_path) == FALSE )
        {
            // Writing fails...
            msg = g_strdup_printf(_("Can't write playlist file '%s'!\n(%s)"),playlist_name,g_strerror(errno)); 
                msgbox = msg_box_new (_("Error..."),msg,MSG_ERROR,BUTTON_OK,0);
            msg_box_hide_check_button(MSG_BOX(msgbox));
            msg_box_run(MSG_BOX(msgbox));
            gtk_widget_destroy(msgbox);
        }else
        {
            msg = g_strdup_printf(_("Written playlist file '%s'"),playlist_name); 
            //msgbox = msg_box_new (_("Information..."),msg,MSG_INFO,BUTTON_OK,0);
            Statusbar_Message(msg,TRUE);
        }
        g_free(msg);
    }

    if (playlist_name) g_free(playlist_name);
}



/*
 * Run Xmms and load files in the current dir
 */
void Run_Xmms (void)
{
    pid_t pid;

    pid = fork();
    switch(pid)
    {
        case -1:
            g_warning(_("Can't fork another process!\n"));
            break;
        case 0:
        {
            gchar **argv;
            GList *etfilelist;
            ET_File *etfile;
            gchar *filename = NULL;
            gint argv_index = 0;
            gint argv_number = ETFileList_Length + 3;

            argv = g_new0(gchar *,argv_number);
            argv[0] = "xmms";
            argv[1] = "-p"; // Start playing current playlist
            argv_index = 2;
            
            etfilelist = ET_File_List_First();
            while (etfilelist)
            {
                etfile   = (ET_File *)etfilelist->data;
                filename = ((File_Name *)etfile->FileNameCur->data)->value;
                argv[argv_index++] = filename;
                etfilelist = etfilelist->next;
            }
            argv[argv_index] = NULL; // Ends the list of arguments
            execvp(argv[0],argv);
            g_free(argv);
            break;
        }
        default:
            break;
    }
}





gchar *Convert_Size (gfloat size)
{
    gchar *data=NULL;
    /* Units Tab of file size (bytes,kilobytes,...) */
    gchar *Units_Tab[] = { N_("B"), N_("KB"), N_("MB"), N_("GB"), N_("TB")};
    gint i = 0;

    while ( (gint)size/1024 && i<(sizeof(Units_Tab)/sizeof(Units_Tab[0])-1) )
    {
        size = size/1024;
        i++;
    }
    return data = g_strdup_printf("%.1f %s",size,_(Units_Tab[i]));    
}


gchar *Convert_Duration (gulong duration)
{
    guint hour=0;
    guint minute=0;
    guint second=0;
    gchar *data = NULL;

    if (duration<=0)
        return g_strdup_printf("%d:%.2d",minute,second);

    hour = duration/3600;
    minute = (duration%3600)/60;
    second = (duration%3600)%60;

    if(hour)
        data = g_strdup_printf("%d:%.2d:%.2d",hour,minute,second);
    else
        data = g_strdup_printf("%d:%.2d",minute,second);
    
    return data;
}


/*
 * Returns the size of a file in bytes
 */
gulong Get_File_Size (gchar *filename)
{
    struct stat statbuf;

    if (filename)
    {
        stat(filename,&statbuf);
        return statbuf.st_size;
    }else
    {
        return 0;
    }
}




/*
 * Delete spaces at the end and the beginning of the string 
 */
void Strip_String (gchar *string)
{
    if (!string) return;
    string = g_strstrip(string);
}


/*
 * Check if widget is an entry or a combobox before to get text
 */
gchar *gtk_entry_get_text_1 (GtkWidget *widget)
{
    if (GTK_IS_COMBO(widget))
        return gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(widget)->entry));
    else if (GTK_IS_ENTRY(widget))
        return gtk_entry_get_text(GTK_ENTRY(widget));
    else
        return NULL;
}



