/**
 * @file main.c Main file
 *
 * Copyright (C) 2004-2006 Christian Hammond.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA  02111-1307  USA
 */
#include "config.h"

#include <libebook/e-book.h>
#include <libgalago/galago.h>
#include <glib.h>
#include <string.h>

static EBookView *book_view = NULL;
static EBook *book = NULL;
static gulong book_view_tag = 0;
static GalagoService *aim, *groupwise, *jabber, *yahoo, *msn, *icq;

static void
add_accounts(GalagoPerson *person, EContact *contact,
			 GalagoService *service, EContactField field)
{
	GalagoAccount *account;
	GList *im_attr_list;

	im_attr_list = e_contact_get_attributes(contact, field);

	while (im_attr_list != NULL)
	{
		EVCardAttribute *attr = (EVCardAttribute *)im_attr_list->data;
		char *username;

		username = e_vcard_attribute_get_value(attr);

		g_message("Adding account %s on %s",
				  username, galago_service_get_id(service));

		if (username != NULL && *username != '\0')
		{
			account = galago_service_create_account(service, person,
													username);
		}

		g_free(username);

		im_attr_list = g_list_delete_link(im_attr_list, im_attr_list);
		e_vcard_attribute_free(attr);
	}
}

static void
add_remove_accounts(GalagoPerson *person, EContact *contact,
					GalagoService *service, EContactField field)
{
	GalagoAccount *account;
	GList *im_attr_list, *l, *l_next, *l2;

	im_attr_list = e_contact_get_attributes(contact, field);

	for (l = im_attr_list; l != NULL; l = l->next)
	{
		EVCardAttribute *attr = (EVCardAttribute *)l->data;
		char *username;

		username = e_vcard_attribute_get_value(attr);

		if (username != NULL && *username != '\0')
		{
			if (galago_person_get_account(person, service, username, FALSE))
			{
				g_message("Adding account %s on %s",
						  username, galago_service_get_id(service));

				account = galago_service_create_account(service, person,
														username);
			}
		}

		g_free(username);
	}

	for (l = galago_person_get_accounts(person, FALSE);
		 l != NULL;
		 l = l_next)
	{
		gboolean found = FALSE;
		const char *username1;
		char *username2;

		l_next = l->next;

		account = (GalagoAccount *)l->data;

		if (galago_account_get_service(account) != service)
			continue;

		username1 = galago_account_get_username(account);

		for (l2 = im_attr_list; l2 != NULL; l2 = l2->next)
		{
			EVCardAttribute *attr = (EVCardAttribute *)l2->data;

			username2 = e_vcard_attribute_get_value(attr);

			if (!strcmp(username1, username2))
				found = TRUE;

			g_free(username2);

			if (found)
				break;
		}

		if (!found)
		{
			g_message("Removing account %s on %s",
					  galago_account_get_username(account),
					  galago_service_get_id(service));
			g_object_unref(account);
		}
	}

	while (im_attr_list != NULL)
	{
		EVCardAttribute *attr = (EVCardAttribute *)im_attr_list->data;

		im_attr_list = g_list_delete_link (im_attr_list, im_attr_list);
		e_vcard_attribute_free(attr);
	}
}

static void
set_attributes(GalagoPerson *person, EContact *contact)
{
	EContactName *contact_name;
	EContactPhoto *ephoto;
	GalagoImage *photo = NULL;

	galago_object_set_attr_string(GALAGO_OBJECT(person),
								  GALAGO_PERSON_ATTR_UID,
								  galago_person_get_id(person));

	contact_name = e_contact_get(contact, E_CONTACT_NAME);

	if (contact_name == NULL)
		return;

	if (contact_name->given != NULL && *contact_name->given != '\0')
	{
		galago_object_set_attr_string(GALAGO_OBJECT(person),
									  GALAGO_PERSON_ATTR_FIRST_NAME,
									  contact_name->given);
	}

	if (contact_name->family != NULL && *contact_name->family != '\0')
	{
		galago_object_set_attr_string(GALAGO_OBJECT(person),
									  GALAGO_PERSON_ATTR_LAST_NAME,
									  contact_name->family);
	}

	e_contact_name_free(contact_name);

	ephoto = e_contact_get(contact, E_CONTACT_PHOTO);

	if (ephoto != NULL)
	{
#ifdef HAVE_PHOTO_TYPE
		switch (ephoto->type)
		{
			case E_CONTACT_PHOTO_TYPE_INLINED:
				photo = galago_image_new_from_data(
					(guchar *)ephoto->inlined.data, ephoto->inlined.length);
				break;

			case E_CONTACT_PHOTO_TYPE_URI:
				if (g_str_has_prefix(ephoto->uri, "file://"))
					photo = galago_image_new_from_file(ephoto->uri + 7);
				else
				{
					/* TODO: use gnome-vfs? */
					g_warning("Cannot handle remote photos");
				}
				break;

			default:
				g_warning("Unhandled photo type %d\n", ephoto->type);
				break;
		}
#else /* !HAVE_PHOTO_TYPE */
		photo = galago_image_new_from_data((guchar *)ephoto->data,
										   ephoto->length);
#endif

		e_contact_photo_free(ephoto);
	}

	galago_person_set_photo(person, photo);
}

static void
contacts_changed_cb(EBookView *book_view, const GList *contacts)
{
	const GList *l;

	for (l = contacts; l != NULL; l = l->next)
	{
		EContact *contact = (EContact *)l->data;
		GalagoPerson *person;
		const char *uid;

		uid = e_contact_get_const(contact, E_CONTACT_UID);

		person = galago_create_person(uid);

		set_attributes(person, contact);

		add_remove_accounts(person, contact, aim,       E_CONTACT_IM_AIM);
		add_remove_accounts(person, contact, groupwise, E_CONTACT_IM_GROUPWISE);
		add_remove_accounts(person, contact, jabber,    E_CONTACT_IM_JABBER);
		add_remove_accounts(person, contact, yahoo,     E_CONTACT_IM_YAHOO);
		add_remove_accounts(person, contact, msn,       E_CONTACT_IM_MSN);
		add_remove_accounts(person, contact, icq,       E_CONTACT_IM_ICQ);
	}
}

static void
contacts_added_cb(EBookView *book_view, const GList *contacts)
{
	const GList *l;

	for (l = contacts; l != NULL; l = l->next)
	{
		EContact *contact = (EContact *)l->data;
		GalagoPerson *person;
		const char *uid;

		uid = e_contact_get_const(contact, E_CONTACT_UID);

		person = galago_create_person(uid);

		set_attributes(person, contact);

		add_accounts(person, contact, aim,       E_CONTACT_IM_AIM);
		add_accounts(person, contact, groupwise, E_CONTACT_IM_GROUPWISE);
		add_accounts(person, contact, jabber,    E_CONTACT_IM_JABBER);
		add_accounts(person, contact, yahoo,     E_CONTACT_IM_YAHOO);
		add_accounts(person, contact, msn,       E_CONTACT_IM_MSN);
		add_accounts(person, contact, icq,       E_CONTACT_IM_ICQ);
	}
}

static void
got_book_view_cb(EBook *book, EBookStatus status, EBookView *view,
				 gpointer user_data)
{
	book_view_tag = 0;

	if (status != E_BOOK_ERROR_OK)
	{
		g_error("Unable to retrieve book view!");
		exit(1);
	}

	book_view = view;

	g_object_ref(book_view);

	g_signal_connect(G_OBJECT(book_view), "contacts_changed",
					 G_CALLBACK(contacts_changed_cb), book);
	g_signal_connect(G_OBJECT(book_view), "contacts_added",
					 G_CALLBACK(contacts_added_cb), book);

	e_book_view_start(view);
}

static void
setup_book_view(void)
{
	EBookQuery *query;

	book = e_book_new_system_addressbook(NULL);

	if (book == NULL)
	{
		g_error("Failed to create local addressbook");

		exit(1);
	}

	if (!e_book_open(book, FALSE, NULL))
	{
		g_error("Failed to open local addressbook");

		g_object_unref(book);

		exit(1);
	}

	query = e_book_query_any_field_contains("");

	book_view_tag = e_book_async_get_book_view(book, query, NULL, -1,
											   got_book_view_cb, NULL);
	e_book_query_unref(query);
}

static void
setup_services(void)
{
#define ADD_SERVICE(var, id) (var) = galago_create_service((id), NULL, 0)
	ADD_SERVICE(aim,       GALAGO_SERVICE_ID_AIM);
	ADD_SERVICE(groupwise, GALAGO_SERVICE_ID_GROUPWISE);
	ADD_SERVICE(jabber,    GALAGO_SERVICE_ID_JABBER);
	ADD_SERVICE(yahoo,     GALAGO_SERVICE_ID_YAHOO);
	ADD_SERVICE(msn,       GALAGO_SERVICE_ID_MSN);
	ADD_SERVICE(icq,       GALAGO_SERVICE_ID_ICQ);
#undef ADD_SERVICE
}

int
main(int argc, char **argv)
{
	GMainLoop *loop;

	loop = g_main_loop_new(NULL, FALSE);

	g_type_init ();

	if (!galago_init("eds-feed", GALAGO_INIT_FEED))
	{
		g_warning("Unable to initialize Galago.");
		exit(1);
	}

	if (!galago_is_registered())
	{
		g_warning("Unable to register with Galago. Exitting.");
		exit(1);
	}

	g_signal_connect_swapped(G_OBJECT(galago_get_core()), "unregistered",
							 G_CALLBACK(g_main_loop_quit), loop);

	setup_book_view();
	setup_services();

	g_main_loop_run(loop);

	return 0;
}
